﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Printing;
using TPClib.Curve;

namespace TPClib.Forms
{
    /// <summary>
    /// GUI component for plotting 2D data
    /// </summary>
    public partial class Plot2D : Panel
    {
        /// <summary>
        /// Curves that are plotted
        /// </summary>
        protected List<Curve> curves = new List<Curve>();
        /// <summary>
        /// Default constructor
        /// </summary>
        public Plot2D() { 
            Initialize(new FloatAxis(), new FloatAxis()); 
        }
        /// <summary>
        /// Constructor with axis
        /// </summary>
        /// <param name="xAxis"></param>
        /// <param name="yAxis"></param>
        public Plot2D(PlotAxis xAxis, PlotAxis yAxis)
        { 
            Initialize(xAxis, yAxis); 
        }
        /// <summary>
        /// Initializes object with axis defined
        /// </summary>
        /// <param name="xAxis">x-axis</param>
        /// <param name="yAxis">y-axis</param>
        private void Initialize(PlotAxis xAxis, PlotAxis yAxis)
        {
            // Format for aligning text center.
            CenterFormat = new StringFormat();
            CenterFormat.Alignment = StringAlignment.Center;

            // Format for aligning text to right
            RightFormat = new StringFormat();
            RightFormat.Alignment = StringAlignment.Far;

            x_axis = xAxis;
            y_axis = yAxis;

            // Printing event
            printDoc.PrintPage += new PrintPageEventHandler(printDoc_PrintPage);

            base.Paint += new PaintEventHandler(Paint_EventHandler);

            plotBounds = new Rectangle();
            plotBounds.X = 40;
            plotBounds.Y = 20;
        }
        public void Paint_EventHandler(object sender, PaintEventArgs e) {
            Paint((sender as Plot2D).CreateGraphics());
        }
        /// <summary>
        /// Plot drawing bounds
        /// </summary>
        Rectangle plotBounds;
        /// <summary>
        /// String format for center alignment
        /// </summary>
        private StringFormat CenterFormat;
        /// <summary>
        /// String alignment for right alignment
        /// </summary>
        private StringFormat RightFormat;
        /// <summary>
        /// Plot x-axis
        /// </summary>
        public PlotAxis x_axis;
        /// <summary>
        /// Plot y-axis
        /// </summary>
        public PlotAxis y_axis;
        /// <summary>
        /// Document that is send to printer.
        /// </summary>
        private PrintDocument printDoc = new PrintDocument();

        protected override void OnResize(EventArgs eventargs)
        {
            base.OnResize(eventargs);
            UpdateScales();
            UpdateScreen();
        }

        /// <summary>
        /// Calculates all the scale variables for x and y axis.
        /// (the interval, text interval, max zero number for log10)
        /// </summary>
        public void UpdateScales()
        {
            // Calculating the interval
            if (curves == null || curves.Count == 0) return;
            
            // Finding the min and max time of curves
            double minOfXvalues = double.MaxValue;
            double maxOfXvalues = double.MinValue;
            double minOfYvalues = double.MaxValue;
            double maxOfYvalues = double.MinValue;
            for (int c = 0; c < curves.Count; c++)
            {
                for (int i = 0; i < curves[c].Length; i++)
                {
                    if (curves[c].GetX(i) > maxOfXvalues) maxOfXvalues = curves[c].GetX(i);
                    if (curves[c].GetX(i) < minOfXvalues) minOfXvalues = curves[c].GetX(i);
                    if (curves[c].GetY(i) > maxOfYvalues) maxOfYvalues = curves[c].GetY(i);
                    if (curves[c].GetY(i) < minOfYvalues) minOfYvalues = curves[c].GetY(i);
                }
            }
            if (x_axis is FloatAxis)
            {
                if ((x_axis as FloatAxis).Scale == ScaleType.Log)
                {
                    // finding the min. (first not 0 assumes that time cell order is linear from min to max)
                    /*double min = 0.1;
                    for (int i = 0; i < tcells.Length; i++) if (tcells[i] > 0) { min = tcells[i]; break; }
                    (x_axis as Chart.FloatAxis).CalculateScale(min, tcells[tcells.Length - 1]);*/
                }
                else //if ((x_axis as FloatAxis).Scale == _Scale.Fit_To_Screen)
                {
                    // We are not in log. We dont have to worry about negative or zero values
                    (x_axis as FloatAxis).CalculateScale(minOfXvalues, maxOfXvalues);
                }
            }

            if (y_axis is FloatAxis)
            {
                if ((y_axis as FloatAxis).Scale == ScaleType.Log)
                {
                    // finding the min. (first not 0 assumes that time cell order is linear from min to max)
                    /*double min = table.GetMinAboveZero();
                    double max = table.GetMax();
                    (y_axis as FloatAxis).CalculateScale(min, max);*/
                }
                else //if ((y_axis as FloatAxis).Scale == _Scale.Fit_To_Screen)
                {
                    // We are not in log. We dont have to worry about negative or zero values
                    (y_axis as FloatAxis).CalculateScale(minOfYvalues, maxOfYvalues);
                }
            }
        }

        /// <summary>
        /// Updates the screen and width/height
        /// </summary>
        public void UpdateScreen()
        {
            plotBounds.Width = Width - 80;
            plotBounds.Height = Height - 80;
            Refresh();
        }

        public void ClearCurves()
        {
            curves.Clear();
        }
        /// <summary>
        /// Adds curve into panel
        /// </summary>
        /// <param name="curve"></param>
        public void AddCurve(TPClib.Forms.Curve curve)
        {
            if (curve == null) return;
            curves.Add(curve);
        }
        /// <summary>
        /// Adds TAC (Time Activity Curve) into panel
        /// </summary>
        /// <param name="curve"></param>
        public void AddCurve(TPClib.Curve.TAC curve)
        {
            if (curve == null) return;
            curves.Add(new Curve());
            if (curve[0] is FrameTimeCell)
            {
                FrameTimeCell fmcell;
                for (int i = 0; i < curves[curves.Count - 1].Length; i++)
                {
                    fmcell = (curve[i] as FrameTimeCell);
                    switch (fmcell.type)
                    {
                        case TPClib.Curve.FrameTimeCell.FrameTimetype.START:
                            curves[curves.Count - 1][i,0] = fmcell.start_time;
                            break;
                        case TPClib.Curve.FrameTimeCell.FrameTimetype.END:
                            curves[curves.Count - 1][i,0] = fmcell.end_time;
                            break;
                        case TPClib.Curve.FrameTimeCell.FrameTimetype.MID:
                            curves[curves.Count - 1][i, 0] = fmcell.mid_time;
                            break;
                        case TPClib.Curve.FrameTimeCell.FrameTimetype.START_END:
                        default:
                            curves[curves.Count - 1][i, 0] = fmcell.start_time;
                            break;
                    }
                }
            }
            else {
                //set default x-steps
                for (int i = 0; i < curves[curves.Count - 1].Length; i++) {
                    curves[curves.Count - 1][i,0] = i;
                }
            }
        }

        /// <summary>
        /// Paints this component using graphics context
        /// </summary>
        /// <param name="g">graphics for drawing</param>
        public new void Paint(Graphics g)
        {
            UpdateScales();

            System.Drawing.Point[] points = null;

            FontFamily fontFamily = new FontFamily("Arial");
            Font font = new Font(fontFamily, 13, FontStyle.Regular, GraphicsUnit.Pixel);
            Brush brush = new SolidBrush(BackColor);
            Pen pen;

            //fill curve background
            g.FillRectangle(brush, Bounds);

            if (curves == null || curves.Count == 0)
            {
                points = new System.Drawing.Point[] { new System.Drawing.Point(0, 0) };
                Color bcolor = Color.FromArgb(byte.MaxValue - BackColor.R,
                                              byte.MaxValue - BackColor.G,
                                              byte.MaxValue - BackColor.B);
                brush = new SolidBrush(y_axis.style.Color);
                g.DrawString("No data to show...", font, brush, Width / 2 - 60, Height / 2);
            }
            else
            {
                // Displaying the curves
                for (int curve = 0; curve < curves.Count; curve++)
                {
                    //pen for drawing the line
                    Pen linePen = new Pen(curves[curve].lineStyle.Color, (float)curves[curve].lineStyle.LineWidth);
                    //brush for filling bars and dots
                    brush = new SolidBrush(curves[curve].lineStyle.Color);
                
                    points = new System.Drawing.Point[curves[curve].Length];

                    for (int i = 0; i < curves[curve].Length; i++)
                    {
                        // Range checks:
                        double px = plotBounds.X + (x_axis.GetPlace(curves[curve][i,0]) * plotBounds.Width);
                        if (px < -1000000000) px = -1000000000;
                        if (px > 1000000000) px = 1000000000;
                        if (Double.IsNaN(px)) px = 0.0d;

                        double py = plotBounds.Bottom - (y_axis.GetPlace(curves[curve][i, 1]) * plotBounds.Height);
                        if (py < -1000000000) py = -1000000000;
                        if (py > 1000000000) py = 1000000000;
                        if (Double.IsNaN(py)) py = 0.0d;

                        points[i] = new System.Drawing.Point((int)px, (int)py);                       
                    }

                    //draw bars if flag is set
                    if (curves[curve].lineStyle.ShowBars)
                    {
                        // Showing the Columns
                        int wid = 100;
                        if (points.Length > 1) wid = points[1].X - points[0].X + 1;

                        foreach (System.Drawing.Point p in points)
                        {
                            // Curve dots if we are not in histogram
                            g.FillRectangle(brush, p.X, p.Y, wid, plotBounds.Bottom - p.Y);
                            g.DrawLine(linePen, p.X, p.Y, p.X, plotBounds.Bottom);
                        }
                    }
                    //draw lines between dots if flag is set
                    if (curves[curve].lineStyle.ShowLines)
                    {
                        // Drawing the curve
                        if (points.Length > 1)
                        {
                            GraphicsPath path = new GraphicsPath();
                            path.AddCurve(points, 0);
                            g.DrawPath(linePen, path);
                        }
                    }
                    //draw square-shaped dots if flag is set
                    if (curves[curve].lineStyle.ShowDots)
                    {
                        // Showing the curve dots
                        foreach (System.Drawing.Point p in points)
                        {
                            g.FillRectangle(brush, p.X - 1, p.Y - 1, 3, 3);
                        }
                    }

                    // The y-axis:
                    brush = new SolidBrush(y_axis.style.Color);
                    pen = new Pen(brush, y_axis.style.LineWidth);
                    g.DrawLine(pen, plotBounds.Left, plotBounds.Top, plotBounds.Left, plotBounds.Bottom);
                    g.DrawString(y_axis.Caption, font, brush, plotBounds.Left - 50, plotBounds.Top - 30);

                    for (double act = y_axis.ResetInterval(); act < y_axis.max; act = y_axis.NextInterval())
                    {
                        if (double.IsInfinity(act)) return;
                        int y = (int)((-y_axis.GetPlace(act)) * plotBounds.Height + plotBounds.Bottom);
                        if (y < 0) continue;
                        string text = y_axis.GetTextOfCurrentInterval();

                        if (text != null)
                        {
                            g.DrawLine(pen, plotBounds.Left - 20, y, plotBounds.Left, y);
                            g.DrawString(text.Replace(',', '.'), font, brush, plotBounds.Left - 20, y - 8, RightFormat);
                        }
                        else
                        {
                            g.DrawLine(pen, plotBounds.Left - 10, y, plotBounds.Left, y);
                        }
                    }

                    // The x-axis:
                    brush = new SolidBrush(x_axis.style.Color);
                    pen = new Pen(brush, x_axis.style.LineWidth);
                    g.DrawLine(pen, plotBounds.Left, plotBounds.Bottom, plotBounds.Right, plotBounds.Bottom);
                    g.DrawString(x_axis.Caption, font, brush, (plotBounds.Left + plotBounds.Right) / 2 - 20, plotBounds.Bottom + 40);

                    for (double time = x_axis.ResetInterval(); time <= x_axis.max; time = x_axis.NextInterval())
                    {
                        int x = (int)(x_axis.GetPlace(time) * plotBounds.Width) + plotBounds.Left;
                        string text = x_axis.GetTextOfCurrentInterval();

                        if (text != null)
                        {
                            // The tic value will be displayed with text
                            g.DrawLine(pen, x, plotBounds.Bottom, x, plotBounds.Bottom + 20);
                            g.DrawString(text.Replace(',','.'), font, brush, x, plotBounds.Bottom + 22, CenterFormat);
                        }
                        else // small tick without text
                        {
                            g.DrawLine(pen, x, plotBounds.Bottom, x, plotBounds.Bottom + 10);
                        }
                    }

                    //draw name
                    g.DrawString(curves[curve].name, font, brush, Width / 2, font.Height);
                }
            }
        }

        /// <summary>
        /// Prints contents of screen
        /// </summary>
        public void Print()
        {
            PageSettings pgSettings = new PageSettings();
            pgSettings.Landscape = true;

            printDoc.DefaultPageSettings = pgSettings;
            PrinterSettings settings = printDoc.PrinterSettings;

            printDoc.Print();

            // Updating the width and height of a4 to screen width and height.
            // Also screen is updated
            UpdateScreen();
        }

        #region events

        /// <summary>
        /// Printing event
        /// </summary>
        private void printDoc_PrintPage(Object sender, PrintPageEventArgs e)
        {
            int oldwidth = Width;
            int oldheight = Height;
            int header_x_fromright = 420;
            int header_x = 0;

            // size of A4 is (11.69 x 8.27) inc. The point graphics unit is inch/72;
            // width and height are made half inc smaller so the curve will better fit to paper
            Width = (int)((11.69 - 0.5) * 72.0);
            Height = (int)((8.27 - 0.5) * 72.0);

            plotBounds.Width = Width - 140;
            plotBounds.Height = Height - 70;
            header_x = Width - header_x_fromright;

            e.Graphics.PageUnit = GraphicsUnit.Point;
            Paint(e.Graphics);

            Width = oldwidth;
            Height = oldheight;
        }
        /// <summary>
        /// Event handler for size change
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private new void SizeChanged(object sender, EventArgs e)
        {
            UpdateScreen();
            Refresh();
        }

        #endregion

    }
}

