﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    /// <summary>
    /// Provides dimensional, identifying and
    /// some processing history information
    /// </summary>
    /// <remarks>
    /// Based on Mayo/SPM "Analyze" Format Spec Compilation
    /// </remarks>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Analyze75Header
    {
        #region enumerators
        /// <summary>
        /// Analyze75 datatype
        /// </summary>
        public enum Datatype {
            /// <summary>
            /// No datatype is defined (DT_NONE==0)
            /// </summary>
            NONE = 0,
            /// <summary>
            /// Unknown datatype (DT_UNKNOWN==0)
            /// </summary>
            UNKNOWN = 0,
            /// <summary>
            /// Binary data (DT_BINARY==1)
            /// </summary>
            BINARY = 1,
            /// <summary>
            /// Unsigned 8-bit data (DT_UNSIGNED_CHAR == 2)
            /// </summary>
            UNSIGNED_CHAR = 2,
            /// <summary>
            /// Signed 16-bit data (DT_SIGNED_SHORT == 4)
            /// </summary>
            SIGNED_SHORT = 4,
            /// <summary>
            /// Signed 32-bit data (SIGNED_INT == 8)
            /// </summary>
            SIGNED_INT = 8,
            /// <summary>
            /// Float data (32-bit) (FLOAT == 16)
            /// </summary>
            FLOAT = 16,
            /// <summary>
            /// Complex data. Two floats (real,imag) (COMPLEX == 32)
            /// </summary>
            COMPLEX = 32,
            /// <summary>
            /// Double data (64-bit) (DOUBLE == 64)
            /// </summary>
            DOUBLE = 64,
            /// <summary>
            /// RGB data (red,green,blue) (24-bit) (RGB == 128)
            /// </summary>
            RGB = 128,
            /// <summary>
            /// All datatypes (ALL == 255)
            /// </summary>
            ALL = 255
        }
        #endregion
        /// <summary>
        /// Header key
        /// </summary>
        public AnalyzeHeaderKey key;         
        /// <summary>
        /// Image dimensions
        /// </summary>
        public AnalyzeHeaderImageDim dime;
        /// <summary>
        /// Data history
        /// </summary>
        public AnalyzeHeaderHistory hist;

        /// <summary>
        /// A flag for little endian coding.
        /// </summary>
        private Boolean isLittleEndian = true;

        /// <summary>
        /// A flag for little endian coding.
        /// </summary>
        public Boolean IsLittleEndian
        {
            get { return isLittleEndian; }
        }

        /// <summary>
        /// The number of pixels in file.
        /// </summary>
        public Int32 Pixels
        {
            get { return dime.dimx * dime.dimy * dime.dimz; }
        }

        /// <summary>
        /// Time Frames
        /// </summary>
        public Int32 Frames
        {
            get { return dime.T; }
        }

        /// <summary>
        /// SPM: Scale factor
        /// </summary>
        public float ScaleFactor
        {
            get { return dime.ScaleFactor; }
            set { dime.ScaleFactor = value; }
        }

        /// <summary>
        /// Bits per pixel: 1, 8, 16, 32, or 64.
        /// </summary>
        public Int16 BitsPerPixel
        {
            get { return dime.bitpix; }
        }

        /// <summary>
        /// Analyze75 data type of the image file.
        /// </summary>
        public Int16 DataType
        {
            get { return dime.DataType; }
        }

        /// <summary>
        /// Byte offset in the .img file at
        /// which voxels start. This value
        /// can be negative to specify that
        /// the absolute value is applied for
        /// every image in the file.
        /// </summary>
        public Int32 VoxelsOffset
        {
            get { return (Int32)dime.VoxelsOffset; }
        }

        /// <summary>
        /// Number of dimensions in database; usually 4
        /// </summary>
        public Int16 Dimensions
        {
            get { return dime.Dimensions; }
        }

        /// <summary>
        /// Defines the frame length.
        /// </summary>
        public Int32 RawSize
        {
            get { return Pixels * BitsPerPixel / 8; }
        }

        /// <summary>
        /// Indicates if the file has more than one frame.
        /// </summary>
        /// <returns></returns>
        public Boolean IsDynamic
        {
            get { return Frames >= 1; }
        }

        /// <summary>
        /// Initializes Analyze file.
        /// </summary>
        public Analyze75Header():base()
        {
            key = new AnalyzeHeaderKey();
            dime = new AnalyzeHeaderImageDim();
            hist = new AnalyzeHeaderHistory();
        }
        /// <summary>
        /// Writes class contents as a byte array
        /// </summary>
        /// <param name="hdr">related image header</param>
        /// <returns>class as a byte array</returns>
        private byte[] WriteBytes(ImageHeader hdr)
        {
            // get data from base class
            try { key.data_type = (byte[])hdr.GetValue("data_type"); }
            catch (Exception) { }
            try { key.db_name = (byte[])hdr.GetValue("db_name"); }
            catch (Exception) { }
            try { key.extents = (int)hdr.GetValue("extents"); }
            catch (Exception) { }
            try { key.hkey_un0 = (byte)hdr.GetValue("hkey_un0"); }
            catch (Exception) { }
            try { key.regular = (byte)hdr.GetValue("regular"); }
            catch (Exception) { }
            try { key.session_error = (short)hdr.GetValue("session_error"); }
            catch (Exception) { }
            try { key.sizeof_hdr = (int)hdr.GetValue("sizeof_hdr"); }
            catch (Exception) { }

            try { dime.bitpix = (short)hdr.GetValue("bitpix"); }
            catch (Exception) { }
            try { dime.cal_max = (float)hdr.GetValue("cal_max"); }
            catch (Exception) { }
            try { dime.cal_min = (float)hdr.GetValue("cal_min"); }
            catch (Exception) { }
            try { dime.cal_units = (byte[])hdr.GetValue("cal_units"); }
            catch (Exception) { }
            try { dime.compressed = (float)hdr.GetValue("compressed"); }
            catch (Exception) { }
            try { dime.datatype = (short)hdr.GetValue("datatype"); }
            catch (Exception) { }
            try { dime.dim = (short[])hdr.GetValue("dim"); }
            catch (Exception) { }
            try { dime.dim_un0 = (short)hdr.GetValue("dim_un0"); }
            catch (Exception) { }
            //note: funused[0] contains global scaling factor for Analyze75
            try { dime.funused = (float[])hdr.GetValue("funused"); }
            catch (Exception) { }
            try { dime.glmax = (int)hdr.GetValue("glmax"); }
            catch (Exception) { }
            try { dime.glmin = (int)hdr.GetValue("glmin"); }
            catch (Exception) { }
            try { dime.pixdim = (float[])hdr.GetValue("pixdim"); }
            catch (Exception) { }
            try { dime.unused1 = (short)hdr.GetValue("unused1"); }
            catch (Exception) { }
            try { dime.verified = (float)hdr.GetValue("verified"); }
            catch (Exception) { }
            try { dime.vox_offset = (float)hdr.GetValue("vox_offset"); }
            catch (Exception) { }
            try { dime.vox_units = (byte[])hdr.GetValue("vox_units"); }
            catch (Exception) { }

            try { hist.aux_file = (byte[])hdr.GetValue("aux_file"); }
            catch (Exception) { }
            try { hist.descrip = (byte[])hdr.GetValue("descrip"); }
            catch (Exception) { }
            try { hist.exp_date = (byte[])hdr.GetValue("exp_date"); }
            catch (Exception) { }
            try { hist.exp_time = (byte[])hdr.GetValue("exp_time"); }
            catch (Exception) { }
            try { hist.field_skip = (int)hdr.GetValue("field_skip"); }
            catch (Exception) { }
            try { hist.generated = (byte[])hdr.GetValue("generated"); }
            catch (Exception) { }
            try { hist.hist_un0 = (byte[])hdr.GetValue("hist_un0"); }
            catch (Exception) { }
            try { hist.omax = (int)hdr.GetValue("omax"); }
            catch (Exception) { }
            try { hist.omin = (int)hdr.GetValue("omin"); }
            catch (Exception) { }
            try { hist.orient = (byte)hdr.GetValue("orient"); }
            catch (Exception) { }
            try { hist.originator = (short[])hdr.GetValue("originator"); }
            catch (Exception) { }
            try {
                byte[] bytes = (byte[])hdr["patient_id"];
                Array.Resize<byte>(ref bytes, 10);
                hist.patient_id = bytes;
            }
            catch (Exception) { }
            try { hist.scannum = (byte[])hdr.GetValue("scannum"); }
            catch (Exception) { }
            try { hist.smax = (int)hdr.GetValue("smax"); }
            catch (Exception) { }
            try { hist.smin = (int)hdr.GetValue("smin"); }
            catch (Exception) { }
            try { hist.start_field = (int)hdr.GetValue("start_field"); }
            catch (Exception) { }
            try { hist.views = (int)hdr.GetValue("views"); }
            catch (Exception) { }
            try { hist.vols_added = (int)hdr.GetValue("vols_added"); }
            catch (Exception) { }

            //override general header information with imageheader fields 
            int i = 0;
            foreach (char c in hdr.description)
            {
                hist.descrip[i] = (byte)c;
                i++;
            }
            dime.dim[0] = (short)hdr.dim.Length;
            for (i = 1; i < hdr.dim.Length && i < 8; i++)
            {
                dime.dim[i] = (short)hdr.dim[i-1];
            }
            dime.pixdim[0] = 0;
            dime.pixdim[1] = hdr.sizex;
            dime.pixdim[2] = hdr.sizey;
            dime.pixdim[3] = hdr.sizez;
            
            List<Byte> list = new List<Byte>();

            list.AddRange(key.GetBytes());
            if(40 != list.Count)
                throw new TPCAnalyzeFileException("Could not read key.");

            list.AddRange(dime.GetBytes());
            if(148 != list.Count)
                throw new TPCAnalyzeFileException("Could not read dimensions.");

            list.AddRange(hist.GetBytes());
            if(348 != list.Count)
                throw new TPCAnalyzeFileException("Could not read history.");

            return list.ToArray();
        }

        /// <summary>
        /// Reads analyze header contents.
        /// </summary>
        /// <param name="filename">header file that is read</param>
        /// <param name="hdr">image header where data is saved</param>
        public void Read(string filename, ref ImageHeader hdr)
        {
            // gets the data in binary format.
            byte[] data = File.ReadAllBytes(filename);

            // Tests for big endian and converts if necessary.
            int test = BitConverter.ToInt32(data, 0);
            
            // Analyze header always starts with number 348
            if (test == 348)
            {
                this.isLittleEndian = true;
            }
            // if it doesn't, we try to swap data
            // in hope that it is coded as big endian.
            else
            {
                try
                {
                    AnalyzeHeaderSwapper.Swap(data);
                }
                catch (Exception e)
                {
                    throw new TPCAnalyzeFileException("Unable to swap: Wrong file format:"+e);
                }

                test = BitConverter.ToInt32(data, 0);

                if (test == 348)
                {
                    this.isLittleEndian = false;
                }
                else
                {
                    throw new TPCAnalyzeFileException("Wrong file format:"+test);
                }
            }

            // reads the data structure
            // delegates the task to member variables.
            key.Read(data);
            
            dime.Read(data);

            hist.Read(data);

            // Place all data into general header structure
            #region place_all_data_into_general_header
            hdr.Add("data_type"    , key.data_type);
            hdr.Add("db_name"      , key.db_name);
            hdr.Add("extents"    , key.extents);
            hdr.Add("hkey_un0"    , key.hkey_un0);
            hdr.Add("regular"      , key.regular);
            hdr.Add("session_error", key.session_error);
            hdr.Add("sizeof_hdr"   , key.sizeof_hdr);

            hdr.Add("bitpix"       , dime.bitpix);
            hdr.Add("cal_max"      , dime.cal_max);
            hdr.Add("cal_min"      , dime.cal_min);
            hdr.Add("cal_units"    , dime.cal_units);
            hdr.Add("compressed"   , dime.compressed);
            hdr.Add("datatype"     , dime.datatype);
            hdr.Add("dim"          , dime.dim);
            hdr.Add("dim_un0"      , dime.dim_un0);
            //note: funused[0] contains global scaling factor for Analyze75 data
            hdr.Add("funused"      , dime.funused);
            hdr.Add("glmax"        , dime.glmax);
            hdr.Add("glmin"        , dime.glmin);
            hdr.Add("pixdim"       , dime.pixdim);
            hdr.Add("unused1"      , dime.unused1);
            hdr.Add("verified"     , dime.verified);
            hdr.Add("vox_offset"   , dime.vox_offset);
            hdr.Add("vox_units"    , dime.vox_units);
            
            hdr.Add("aux_file"     , hist.aux_file);
            hdr.Add("descrip"      , hist.descrip);
            hdr.Add("exp_date"     , hist.exp_date);
            hdr.Add("exp_time"     , hist.exp_time);
            hdr.Add("field_skip"   , hist.field_skip);
            hdr.Add("generated"    , hist.generated);
            hdr.Add("hist_un0"     , hist.hist_un0);
            hdr.Add("omax"         , hist.omax);
            hdr.Add("omin"         , hist.omin);
            hdr.Add("orient"       , hist.orient);
            hdr.Add("originator"   , hist.originator);
            hdr.Add("patient_id"   , hist.patient_id);
            hdr.Add("scannum"      , hist.scannum);
            hdr.Add("smax"         , hist.smax);
            hdr.Add("smin"         , hist.smin);
            hdr.Add("start_field"  , hist.start_field);
            hdr.Add("views"        , hist.views);
            hdr.Add("vols_added"   , hist.vols_added);
            #endregion
            hdr.patientID = BytesAsChars(hist.patient_id);
            hdr.description = BytesAsChars(hist.descrip).Trim();
            
            if (IsDynamic || Frames == 1)
            {
                hdr.dim = new IntLimits(new int[4] { dime.dimx, dime.dimy, dime.dimz, dime.T });
                hdr = new DynamicImageHeader(hdr);
            }
            else
            {
                hdr.dim = new IntLimits(new int[3] { dime.dimx, dime.dimy, dime.dimz });
            }

            //resolve datatype
            switch(dime.DataType) {
                case (short)Datatype.BINARY:
                    hdr.datatype = ImageFile.DataType.BIT1;
                    break;
                case (short)Datatype.UNSIGNED_CHAR:
                    hdr.datatype = ImageFile.DataType.BIT8_U;
                    break;
                case (short)Datatype.SIGNED_SHORT:
                    hdr.datatype = ImageFile.DataType.BIT16_S;
                    break;
                case (short)Datatype.SIGNED_INT:
                    hdr.datatype = ImageFile.DataType.BIT32_S;
                    break;
                case (short)Datatype.FLOAT:
                    hdr.datatype = ImageFile.DataType.FLT32;
                    break;
                case (short)Datatype.COMPLEX:
                    //this is for invalid header data that has normal 
                    //32-bit float even though the datatype is COMPLEX
                    if (BitsPerPixel == 32) hdr.datatype = ImageFile.DataType.FLT32;
                    //complex data is read as float, real and imaginary parts need 
                    //to be resolved out later
                    else hdr.datatype = ImageFile.DataType.FLT64;
                    break;
                case (short)Datatype.DOUBLE:
                    hdr.datatype = ImageFile.DataType.FLT64;
                    break;
                case (short)Datatype.RGB:
                    hdr.datatype = ImageFile.DataType.COLRGB;
                    break;
                case (short)Datatype.ALL:
                case (short)Datatype.NONE:
                default:
                    throw new TPCAnalyzeFileException("Unsupported data type:"+DataType);
            }
            hdr.siz = new Voxel(dime.pixdim[1], dime.pixdim[2], dime.pixdim[3]);
            hdr.dataunit = Data_unit.Undefined;
            hdr.patient_name = "<no name>";
            hdr.modality = ImageModality.Unknown;
        }

        /// <summary>
        /// Writes analyze header contents.
        /// </summary>
        /// <param name="filename">header file where fields are written</param>
        /// <param name="hdr">header data that is written</param>
        public void Write(string filename, ImageHeader hdr)
        {
            // delegates the reading to member variable.
            byte[] data = WriteBytes(hdr);

            if (!isLittleEndian)
            {
                try
                {
                    AnalyzeHeaderSwapper.Swap(data);
                }
                catch (Exception)
                {
                    throw new TPCAnalyzeFileException("Unable to swap: Wrong file format.");
                }
            }
            // writes to the disk
            File.WriteAllBytes(filename, data);
        }

        /// <summary>
        /// Gives the string representation of this object.
        /// </summary>
        /// <returns>header object as a string</returns>
        public override string ToString()
        {
            return "Analyze75Header[dime="+dime+" hist="+hist+" key="+key+"]\n";
        }
        /// <summary>
        /// Gives the string representation of 
        /// header file.
        /// </summary>
        /// <returns></returns>
        public void PrintInfo()
        {
            Console.WriteLine("Analyze 7.5 Header File");
            this.dime.PrintInfo();
            this.key.PrintInfo();
            this.hist.PrintInfo();
        }
        /// <summary>
        /// Converts byte table to character table
        /// </summary>
        /// <param name="table">bytes</param>
        /// <returns>string</returns>
        private static string BytesAsChars(byte[] table)
        {
            string ret = "";
            foreach (byte b in table)
            {
                // Add to string if bit isn't null.
                if (b != 0)
                {
                    ret += (char)b;
                }

            }
            return ret;
        }

        /// <summary>
        /// Swaps from little endian to big endian.
        /// </summary>
        private class AnalyzeHeaderSwapper : BitSwapper
        {
            /// <summary>
            /// Swaps from little endian to big endian.
            /// </summary>
            /// <param name="data"></param>
            public static void Swap(byte[] data)
            {
                //   data, start, bytes
                Swap(data, 0,     4);
                Swap(data, 4, 10);
                Swap(data, 14, 18);
                Swap(data, 32, 4);
                Swap(data, 36, 2);
                // skip 38
                // skip 39
                SwapLoop(data, 40, 74, 2);
                SwapLoop(data, 76, 144, 4);
                Swap(data, 148, 80);
                Swap(data, 228, 24);
                Swap(data, 252, 10);
                Swap(data, 262, 10);
                Swap(data, 272, 11);
                Swap(data, 283, 9);
                Swap(data, 292, 10);
                Swap(data, 302, 10);
                Swap(data, 312, 3);
                SwapLoop(data, 316, 344, 4);
            }
        }

        /// <summary>
        /// Analyze Header Key
        /// </summary>
        public class AnalyzeHeaderKey
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public AnalyzeHeaderKey() {
                regular = Convert.ToByte('r'); 
            }
            /// <summary>
            /// Must indicate the byte size of
            /// the header file.
            /// Non SPM2: Always 348, used
            /// to test big/little-endian).
            /// SPM2: If >348 indicates
            /// extended header
            /// </summary>
            public Int32 sizeof_hdr = 348;

            /// <summary>
            /// Not specified.
            /// </summary>
            public byte[] data_type = new byte[10]; // 8-bit

            /// <summary>
            /// Not specified.
            /// </summary>
            public byte[] db_name = new byte[18]; // 8-bit

            /// <summary>
            /// Should be 16384, the image
            /// file is created as contiguous
            /// with a minimum extent size.
            /// </summary>
            public Int32 extents = 16384;

            /// <summary>
            /// No specified.
            /// </summary>
            public Int16 session_error;

            /// <summary>
            /// Must be "r" to indicate that all
            /// images and volumes are the
            /// same size.
            /// </summary>
            public byte regular; // 8-bit

            /// <summary>
            /// Not specified.
            /// </summary>
            public byte hkey_un0; // 8-bit

            /// <summary>
            /// Converts bytes to class data.
            /// </summary>
            /// <param name="data">bytes to be read</param>
            public void Read(byte[] data)
            {
                // row 1
                Int32 size = BitConverter.ToInt32(data, 0);
                if (size != sizeof_hdr)
                    throw new TPCAnalyzeFileException("Wrong header size: " + size);

                // row 2
                for (int i = 0; i < data_type.Length; i++)
                {
                    data_type[i] = data[4 + i];
                }    

                // row 3
                for (int i = 0; i < db_name.Length; i++)
                {
                    db_name[i] = data[14 + i];
                }
                
                // row 4
                Int32 e = BitConverter.ToInt32(data, 32);
                //removed this so that the library is not too picky about wrong information
                //SPM is suspected of producing this kind of invalid header files
                //if (extents != e)
                //    throw new TPCAnalyzeFileException("Wrong extents: should be "+extents+" but was "+e);

                // row 5
                session_error = BitConverter.ToInt16(data, 36);

                // row 6
                regular = data[39 - 1];
                char reg = BitConverter.ToChar(data, 38);
                if (reg.CompareTo('r') != 0)
                    throw new TPCAnalyzeFileException("reg: " + reg);

                // row 7
                hkey_un0 = data[40 - 1];
            }

            /// <summary>
            /// Converts class data to bytes.
            /// </summary>
            /// <returns>class data in bytes</returns>
            public byte[] GetBytes()
            {
                List<Byte> list = new List<Byte>();

                string errorMsg = "Couldn't convert header to bytes: ";

                list.AddRange(BitConverter.GetBytes(sizeof_hdr));
                if(4 != list.Count || sizeof_hdr != 348)
                    throw new TPCAnalyzeFileException(errorMsg + "sizeof_hdr");

                list.AddRange(data_type);
                if(14 != list.Count || data_type.Length != 10)
                    throw new TPCAnalyzeFileException(errorMsg + "data_type");

                list.AddRange(db_name);
                if (32 != list.Count || db_name.Length != 18)
                    throw new TPCAnalyzeFileException(errorMsg + "db_name");

                list.AddRange(BitConverter.GetBytes(extents));
                if(36 != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "extents");

                list.AddRange(BitConverter.GetBytes(session_error));
                if(38 != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "session_error");

                list.Add(regular);
                char reg = (char)(regular);
                if (reg.CompareTo('r') != 0)
                    throw new TPCAnalyzeFileException(errorMsg + "regular="+regular);
                if(39 != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "regular");

                list.Add(hkey_un0);
                if(40 != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "hkey_un0");

                return list.ToArray();
            }

            /// <summary>
            /// Print this header block as a string.
            /// </summary>
            /// <returns>string</returns>
            public void PrintInfo()
            {
                string s = "Analyze Header Key\n";
                s += "\theader size: " + sizeof_hdr + " bytes\n";
                s += "\tdatatype: `" + BytesAsChars(data_type) + "´\n";
                s += "\tdb_name: `" + BytesAsChars(db_name) + "´\n";
                s += "\textents: " + extents + "\n";
                s += "\tsession_errpr: " + session_error + "\n";
                s += "\tregular: " + regular + "\n";
                s += "\thkey_un0: " + hkey_un0 + "\n";
                Console.WriteLine(s);
            }
        }

        /// <summary>
        /// Analyze Image Dimensions
        /// </summary>
        public class AnalyzeHeaderImageDim
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public AnalyzeHeaderImageDim() {
                //16-bit signed short
                datatype = 4;
                bitpix = 16;
            }
            /// <summary>
            /// dim[0] Number of dimensions in database; usually 4
            /// dim[1] DIM Image X dimension; number of
            /// pixels in an image row
            /// dim[2] DIM Image Y dimension; number of
            /// pixel rows in slice
            /// dim[3] DIM Volume Z dimension; number
            /// of slices in a volume
            /// dim[4] Time points, number of
            /// volumes in database.
            /// dim[5] Gates
            /// dim[6] Bed positions
            /// dim[7] undoc
            /// </summary>
            /// <remarks>
            /// This definitions differs from the base class dim!!!!
            /// </remarks>
            public Int16[] dim = new Int16[8];

            /// <summary>
            /// spatial units of measure for a voxel
            /// </summary>
            public byte[] vox_units = new byte[4];

            /// <summary>
            /// name of the calibration unit
            /// </summary>
            public byte[] cal_units = new byte[8];

            /// <summary>
            /// undoc
            /// </summary>
            public Int16 unused1;

            /// <summary>
            /// DT_NONE 0
            /// DT_UNKNOWN 0
            /// DT_BINARY 1 1 bit per voxel
            /// DT_UNSIGNED_CHAR 2 8 bits per voxel
            /// DT_SIGNED_SHORT 4 16 bits per voxel
            /// DT_SIGNED_INT 8 32 bits per voxel
            /// </summary>
            public Int16 datatype;

            /// <summary>
            /// bits per pixel; 1, 8, 16, 32, or 64.
            /// </summary>
            public Int16 bitpix;

            /// <summary>
            /// unused
            /// </summary>
            public Int16 dim_un0;

            /// <summary>
            /// pixdim[0] undoc
            /// pixdim[1] float voxel width in mm.
            /// pixdim[2] voxel height in mm.
            /// pixdim[3] slice thickness in mm.
            /// pixdim[4] undoc
            /// pixdim[5] undoc
            /// pixdim[6] undoc
            /// pixdim[7] undoc
            /// </summary>
            public float[] pixdim = new float[8];

            /// <summary>
            /// Byte offset in the .img file at
            /// which voxels start. This value
            /// can be negative to specify that
            /// the absolute value is applied for
            /// every image in the file.
            /// </summary>
            public float vox_offset;

            /// <summary>
            /// funused[0] SPM: Scale factor
            /// funused[1] SPM2: image intensity zero-intercept
            /// funused[2] undoc
            /// </summary>
            public float[] funused = new float[3];

            /// <summary>
            /// max calibration value
            /// </summary>
            public float cal_max;

            /// <summary>
            /// min calibration value
            /// </summary>
            public float cal_min;

            /// <summary>
            /// undoc
            /// </summary>
            public float compressed;

            /// <summary>
            /// undoc
            /// </summary>
            public float verified;

            /// <summary>
            /// max pixel value for entire database
            /// </summary>
            public Int32 glmax;

            /// <summary>
            /// min pixel value for entire database
            /// </summary>
            public Int32 glmin;

            /// <summary>
            /// Converts bytes to class data.
            /// </summary>
            /// <param name="data">bytes to be read</param>
            public void Read(byte[] data)
            {
                if (data.Length != 348)
                    throw new TPCAnalyzeFileException("");
                if (vox_units.Length != 4)
                    throw new TPCAnalyzeFileException("");
                if (cal_units.Length != 8)
                    throw new TPCAnalyzeFileException("");
                if (dim.Length != 8)
                    throw new TPCAnalyzeFileException("");                            
                if (pixdim.Length != 8)
                    throw new TPCAnalyzeFileException("");                                
                if(funused.Length != 3)
                    throw new TPCAnalyzeFileException("");
                for (int i = 0; i < dim.Length; i++)
                {
                    int index = 40 + 2 * i; 
                    dim[i] = BitConverter.ToInt16(data, index);
                }
                for (int i = 0; i < vox_units.Length; i++)
                {
                    vox_units[i] = data[56 + i];
                }
                for (int i = 0; i < cal_units.Length; i++)
                {
                    cal_units[i] = data[60 + i];
                }
                unused1 = BitConverter.ToInt16(data, 68);
                datatype = BitConverter.ToInt16(data, 70);
                bitpix = BitConverter.ToInt16(data, 72);
                dim_un0 = BitConverter.ToInt16(data, 74);
                for (int i = 0; i < pixdim.Length; i++)
                {
                    int index = 76 + 4 * i;
                    pixdim[i] = BitConverter.ToSingle(data, index);
                }
                vox_offset = BitConverter.ToSingle(data, 108);
                for (int i = 0; i < funused.Length; i++)
                {
                    int index = 112 + 4 * i;
                    funused[i] = BitConverter.ToSingle(data, index);
                }                
                cal_max = BitConverter.ToSingle(data, 124);
                cal_min = BitConverter.ToSingle(data, 128);
                compressed = BitConverter.ToSingle(data, 132);
                verified = BitConverter.ToSingle(data, 136);
                glmax = BitConverter.ToInt32(data, 140);
                glmin = BitConverter.ToInt32(data, 144);
            }

            /// <summary>
            /// Converts class data to bytes.
            /// </summary>
            /// <returns>class data in bytes</returns>
            public byte[] GetBytes()
            {
                // an index offset
                int e = 40;

                List<Byte> list = new List<Byte>();

                foreach (Int16 i in dim)
                {
                    list.AddRange(BitConverter.GetBytes(i));
                }
                if( (56 - e) != list.Count)
                    throw new TPCAnalyzeFileException("");

                foreach (byte b in vox_units)
                {
                    list.Add(b);
                }
                if (60 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                foreach (byte b in cal_units)
                {
                    list.Add(b);
                }
                if (68 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(unused1));
                if(70 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(datatype));
                if(72 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(bitpix));
                if(74 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(dim_un0));
                if(76 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                foreach (float i in pixdim)
                {
                    list.AddRange(BitConverter.GetBytes(i));
                }
                if(108 - e != list.Count)
                    throw new TPCAnalyzeFileException((108-e)+"-"+list.Count);

                list.AddRange(BitConverter.GetBytes(vox_offset));
                if(112 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                foreach (float i in funused)
                {
                    list.AddRange(BitConverter.GetBytes(i));
                }
                if(124 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(cal_max));
                if(128 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(cal_min));
                if(132 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(compressed));
                if(136 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(verified));
                if(140 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(glmax));
                if(144 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                list.AddRange(BitConverter.GetBytes(glmin));
                if(148 - e != list.Count)
                    throw new TPCAnalyzeFileException("");

                return list.ToArray();
            }

            /// <summary>
            /// Number of dimensions in database; usually 4
            /// </summary>
            public Int16 Dimensions
            {
                get { return dim[0]; }
            }

            /// <summary>
            /// Image X dimension; number of pixels in an image row
            /// </summary>
            public int dimx
            {
                get { return dim[1]; }
            }

            /// <summary>
            /// Image Y dimension; number of pixel rows in slice
            /// </summary>
            public Int16 dimy
            {
                get { return dim[2]; }
            }

            /// <summary>
            /// Volume Z dimension; number of slices in a volume
            /// </summary>
            public Int16 dimz
            {
                get { return dim[3]; }
            }

            /// <summary>
            /// Time points, number of volumes in database.
            /// </summary>
            public Int16 T
            {
                get { return dim[4]; }
            }

            /// <summary>
            /// SPM: Scale factor
            /// </summary>
            public float ScaleFactor
            {
                get { return funused[0]; }
                set { funused[0] = value; }
            }

            /// <summary>
            /// DT_NONE 0
            /// DT_UNKNOWN 0
            /// DT_BINARY 1 1 bit per voxel
            /// DT_UNSIGNED_CHAR 2 8 bits per voxel
            /// DT_SIGNED_SHORT 4 16 bits per voxel
            /// DT_SIGNED_INT 8 32 bits per voxel
            /// </summary>
            public Int16 DataType
            {
                get { return datatype; }
            }

            /// <summary>
            /// Byte offset in the .img file at
            /// which voxels start. This value
            /// can be negative to specify that
            /// the absolute value is applied for
            /// every image in the file.
            /// </summary>
            public float VoxelsOffset
            {
                get { return vox_offset; }
            }

            /// <summary>
            /// Print this header block as a string.
            /// </summary>
            /// <returns>string</returns>
            public string PrintInfo()
            {
                string s = "Image Dimensions: \n";
                s += "\tDim: " + Dimensions + "\n";
                s += "\tX: " + dimx + "\n";
                s += "\tY: " + dimy + "\n";
                s += "\tZ: " + dimz + "\n";
                s += "\tT: " + T + "\n";
                s += "\tBitpix: " + bitpix + "\n";
                s += "\tdim_un0: " + dim_un0 + "\n";
                s += "\tdatatype: " + datatype + "\n";
                s += "\tScale Factor: " + ScaleFactor + "\n";
                s += "\tVoxel offset: " + VoxelsOffset + "\n";
                s += "\tcal_max: " + cal_max + "\n";
                s += "\tcal_min: " + cal_min + "\n";
                s += "\tcompressed: " + compressed + "\n";
                s += "\tverified: " + verified + "\n";
                s += "\tglmax: " + glmax + "\n";
                s += "\tglmin: " + glmin + "\n";
                return s;
            }
        }
        
        /// <summary>
        /// Analyze Header History
        /// </summary>
        public class AnalyzeHeaderHistory
        {
            #region Data
            /// <summary>
            /// Free description field
            /// </summary>
            public byte[] descrip = new byte[80];

            /// <summary>
            /// Auxilary file name
            /// </summary>
            public byte[] aux_file = new byte[24];

            /// <summary>
            /// slice orientation for this dataset
            /// </summary>
            public byte orient;

            /// <summary>
            /// origin[0] SPM99: X near Anterior Commissure
            /// origin[1] SPM99: Y near Anterior Commissure
            /// origin[2] SPM99: Z near Anterior Commissure
            /// origin[3] undoc
            /// origin[4] undoc
            /// </summary>
            public Int16[] originator = new Int16[5];

            /// <summary>
            /// 
            /// </summary>
            public byte[] generated = new byte[10];

            /// <summary>
            /// Scan number
            /// </summary>
            public byte[] scannum = new byte[10];

            /// <summary>
            /// Patiend ID
            /// </summary>
            public byte[] patient_id = new byte[10];

            /// <summary>
            /// Experiment date
            /// </summary>
            public byte[] exp_date = new byte[10];

            /// <summary>
            /// Experiment time
            /// </summary>
            public byte[] exp_time = new byte[10];

            /// <summary>
            /// undoc
            /// </summary>
            public byte[] hist_un0 = new byte[3];

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 views;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 vols_added;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 start_field;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 field_skip;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 omax;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 omin;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 smax;

            /// <summary>
            /// undoc
            /// </summary>
            public Int32 smin;
            #endregion

            /// <summary>
            /// Converts bytes to class data.
            /// </summary>
            /// <param name="data">bytes to be read</param>
            public void Read(byte[] data)
            {
                string errorMsg = "Could not read: ";

                if( descrip.Length != 80)
                    throw new TPCAnalyzeFileException(errorMsg+"descrip");
                if( aux_file.Length != 24 )
                    throw new TPCAnalyzeFileException(errorMsg + "aux_file");
                if( originator.Length != 5)
                    throw new TPCAnalyzeFileException(errorMsg + "originator");
                if( scannum.Length != 10)
                    throw new TPCAnalyzeFileException(errorMsg + "scannum");
                if( patient_id.Length != 10)
                    throw new TPCAnalyzeFileException(errorMsg + "patient_id");
                if( exp_date.Length != 10)
                    throw new TPCAnalyzeFileException(errorMsg + "exp_date");
                if( hist_un0.Length != 3)
                    throw new TPCAnalyzeFileException(errorMsg + "hist_un0");

                if( data.Length != 348 )
                    throw new TPCAnalyzeFileException("Header Data size is wrong.");

                try
                {
                    for (int i = 0; i < descrip.Length; i++)
                        descrip[i] = data[148 + i];
                    for (int i = 0; i < aux_file.Length; i++)
                        aux_file[i] = data[228 + i];
                    orient = data[252 - 1];
                    for (int i = 0; i < originator.Length; i++)
                    {
                        originator[i] = BitConverter.ToInt16(data, 253 - 1 + 2 * i);
                    }
                    for (int i = 0; i < generated.Length; i++)
                        generated[i] = data[263 - 1 + i];
                    for (int i = 0; i < scannum.Length; i++)
                        scannum[i] = data[273 - 1 + i];
                    for (int i = 0; i < patient_id.Length; i++)
                        patient_id[i] = data[283 + i];
                    for (int i = 0; i < exp_date.Length; i++)
                        exp_date[i] = data[293 - 1 + i];
                    for (int i = 0; i < exp_time.Length; i++)
                        exp_time[i] = data[303 - 1 + i];
                    for (int i = 0; i < hist_un0.Length; i++)
                        hist_un0[i] = data[313 - 1 + i];
                    views = BitConverter.ToInt32(data, 316);
                    vols_added = BitConverter.ToInt32(data, 320);
                    start_field = BitConverter.ToInt32(data, 324);
                    field_skip = BitConverter.ToInt32(data, 328);
                    omax = BitConverter.ToInt32(data, 332);
                    omin = BitConverter.ToInt32(data, 336);
                    smax = BitConverter.ToInt32(data, 340);
                    smin = BitConverter.ToInt32(data, 344);
                }
                catch (Exception)
                {
                    throw new TPCAnalyzeFileException("Could not read AnalyzeHeaderHistory.");
                }
            }

            /// <summary>
            /// Converts class data to bytes.
            /// </summary>
            /// <returns>class data in bytes</returns>
            public byte[] GetBytes()
            {
                // an index offset
                int offset = 148;
                string errorMsg = "Could not convert: ";
                List<Byte> list = new List<Byte>();

                list.AddRange(descrip);
                if ((228 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg+"descrip");

                list.AddRange(aux_file);
                if ((252 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "aux_file");

                list.Add(orient);
                if ((253 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "orient");

                foreach (Int16 i in originator)
                {
                    list.AddRange(BitConverter.GetBytes(i));
                }
                if ((263 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "originator");

                list.AddRange(generated);
                if ((273 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "generated");

                list.AddRange(scannum);
                if ((283 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "scannum");

                if(patient_id.Length != 10)
                    throw new TPCAnalyzeFileException("patient_id length:" + patient_id.Length);
                list.AddRange(patient_id);
                if ((293 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "patient_id");

                list.AddRange(exp_date);
                if ((303 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "exp_date");

                list.AddRange(exp_time);
                if ((313 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "exp_time");

                list.AddRange(hist_un0);
                if ((316 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "hist_un0");

                list.AddRange(BitConverter.GetBytes(views));
                if ((320 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "views");

                list.AddRange(BitConverter.GetBytes(vols_added));
                if ((324 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "vols_added");

                list.AddRange(BitConverter.GetBytes(start_field));
                if ((328 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "start_field");

                list.AddRange(BitConverter.GetBytes(field_skip));
                if ((332 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "field_skip");

                list.AddRange(BitConverter.GetBytes(omax));
                if ((336 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "omax");

                list.AddRange(BitConverter.GetBytes(omin));
                if ((340 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "omin");

                list.AddRange(BitConverter.GetBytes(smax));
                if ((344 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "smax");

                list.AddRange(BitConverter.GetBytes(smin));
                if ((348 - offset) != list.Count)
                    throw new TPCAnalyzeFileException(errorMsg + "smin");

                return list.ToArray();
            }

            /// <summary>
            /// Print this header block as a string.
            /// </summary>
            /// <returns>string</returns>
            public void PrintInfo()
            {
                string s = "History: \n";
                s += "\tdescrip: " + BytesAsChars(descrip) + "\n";
                s += "\taux_file: " + BytesAsChars(aux_file) + "\n";
                Console.WriteLine(s);
            }
        }
    }
}
