﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Collections;

using NUnit.Framework;

namespace TPClib.Image
{
    /// <summary>
    /// Applies a strategy pattern
    /// </summary>
    public abstract class Analyze75Image : Image
    {
        /// <summary>
        /// 
        /// </summary>
        protected string filename;

        /// <summary>
        /// 
        /// </summary>
        protected Analyze75Header header;

        /// <summary>
        /// 
        /// </summary>
        private Int64 fsize = 0;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="header"></param>
        public Analyze75Image(string filename,
            Analyze75Header header)
        {
            this.filename = filename;
            this.header = header;

            // Calculates the number of frames.
            FileInfo nfo = new FileInfo(filename);

            if ((nfo.Length % header.RawSize != 0))
                throw new Exception();

            fsize = nfo.Length;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="filename"></param>
        public void Read(string filename)
        {
            BinaryReader br = new BinaryReader(
                File.Open(filename, FileMode.Open));

            this.filename = filename;

            Read(br);

            br.Close();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="filename"></param>
        public void Write(string filename)
        {
            BinaryWriter writer = new BinaryWriter(
            File.Open(filename, FileMode.CreateNew));

            Write(writer);

            writer.Close();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public virtual int Frames() {
            throw new NotImplementedException();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="bw"></param>
        protected virtual void Read(BinaryReader bw)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="bw"></param>
        protected virtual void Write(BinaryWriter bw)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Returns the file size.
        /// </summary>
        protected Int64 FileSize
        {
            get { return fsize; }
        }

        /// <summary>
        /// An explicit concept for frame.
        /// </summary>
        protected class Analyze75Frame<T>
        {
            /// <summary>
            ///
            /// </summary>
            private T[] data;

            /// <summary>
            /// 
            /// </summary>
            /// <param name="data"></param>
            public Analyze75Frame(T[] data)
            {
                this.data = data;
            }

            /// <summary>
            /// 
            /// </summary>
            public T[] Data
            {
                get { return data; }
                set { data = value; }
            }
        }

        /// <summary>
        /// Creates the right data structure according to 
        /// given header.
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        public static Analyze75Image CreateDataStructure(
            string filename,
            Analyze75Header header) 
        {
            
            if( header.DataType == 4 )
            {
                if (header.BitsPerPixel == 16)
                {
                    return new AnalyzeData16(filename, header);
                }
            }

            if (header.DataType == 16)
            {
                
            }

            if (header.BitsPerPixel == 32)
            {
                return new AnalyzeData32(filename, header);
            }

            throw new NotImplementedException(
                "Datatype: "+header.DataType +"\n"+
                "BPP: " + header.BitsPerPixel);
        }
    }






    /// <summary>
    /// 
    /// </summary>
    internal class AnalyzeData16 : Analyze75Image
    {
        /// <summary>
        /// 
        /// </summary>
        private List<Analyze75Frame<float>> frames = null;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="f"></param>
        /// <param name="h"></param>
        public AnalyzeData16(string f, Analyze75Header h)
            : base(f, h)
        {
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="br"></param>
        override protected void Read(BinaryReader br)
        {
            frames = new List<Analyze75Frame<float>>();

            // Calculates the number of total frames and foats
            long totalFrames = (FileSize / header.RawSize);
            Assert.AreEqual(totalFrames, header.Frames);

            if (FileSize % header.DataType != 0)
                throw new Exception();
            long totalFloats = FileSize / header.DataType;
            long floatsPerFrame = totalFloats / totalFrames;

            // Counts the number of floats read.
            int counter = 0;
            for (int i = 0; i < totalFrames; i++)
            {
                // Allocates the needed memory for frame.
                float[] data = new float[floatsPerFrame];

                // Reads every fourth byte.
                for (int j = 0; j < floatsPerFrame; j++)
                {
                    data[j] = br.ReadSingle();
                    counter++;
                }

                frames.Add(new Analyze75Frame<float>(data));
            }

            if (counter != totalFloats)
                throw new Exception("ARead");
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="bw"></param>
        override protected void Write(BinaryWriter bw)
        {
            double sum = 0;
            foreach (Analyze75Frame<float> f in frames)
            {
                foreach (float d in f.Data)
                {
                    bw.Write(d);
                    sum += d;
                }
            }
            Console.WriteLine("Write Float: " + sum);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override int Frames()
        {
            return frames.Count;
        }

    }

    /// <summary>
    /// 
    /// </summary>
    internal class AnalyzeData32 : Analyze75Image
    {
        /// <summary>
        /// 
        /// </summary>
        private List<Analyze75Frame<double>> frames = null;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="f"></param>
        /// <param name="h"></param>
        public AnalyzeData32(string f, Analyze75Header h)
            : base(f, h)
        {
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="br"></param>
        override protected void Read(BinaryReader br)
        {
            Console.WriteLine("In double");
            frames = new List<Analyze75Frame<double>>();

            // Calculates the number of frames.
            if ((FileSize % header.RawSize != 0))
                throw new Exception();
            long totalFrames = (FileSize / header.RawSize);
            if (FileSize % 8 != 0)
                throw new Exception();
            long totalDoubles = FileSize / 8;
            long doublesPerFrame = totalDoubles / totalFrames;

            // Counts the number of floats read.
            int counter = 0;

            // Check sum
            double sum = 0;

            for (int i = 0; i < totalFrames; i++)
            {
                // Allocates the needed memory for frame.
                double[] data = new double[doublesPerFrame];

                // Reads every fourth byte.
                for (int j = 0; j < doublesPerFrame; j++)
                {
                    data[j] = br.ReadDouble();
                    counter++;
                    sum += data[j];
                }

                frames.Add(new Analyze75Frame<double>(data));
            }

            if (counter != doublesPerFrame*totalFrames)
                throw new Exception("\ncounter:\t"+
                    counter+"\ndoubles:\t"+doublesPerFrame+"\n");

            Console.WriteLine("Read Double: " + sum);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="bw"></param>
        override protected void Write(BinaryWriter bw)
        {
            double checkSum = 0;
            foreach (Analyze75Frame<double> f in frames)
            {
                foreach (double d in f.Data)
                {
                    bw.Write(d);
                    checkSum += d;
                }
            }
            //Console.WriteLine("Write Float: " + luku);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override int Frames()
        {
            return frames.Count;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    internal class AnalyzeData16Int : Analyze75Image
    {
        /// <summary>
        /// 
        /// </summary>
        private Analyze75Frame<Int16> frames;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="fname"></param>
        /// <param name="h"></param>
        public AnalyzeData16Int(string fname, Analyze75Header h)
            : base(fname, h)
        {
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="br"></param>
        override protected void Read(BinaryReader br)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="bw"></param>
        override protected void Write(BinaryWriter bw)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override int Frames()
        {
            throw new NotImplementedException();
        }
    }


}