/*****************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 *****************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;
using openDicom.DataStructure;
using openDicom.DataStructure.DataSet;
using openDicom.Registry;
using openDicom.Encoding;

namespace TPClib.Image
{
    /// <summary>
    /// DICOM file.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class DicomFile : ImageFile
    {
        /// <summary>
        /// Image index while writing.
        /// </summary>
        public static int imageindex = 0;
        /// <summary>
        /// DICOM related information
        /// </summary>
        public DicomHeader dicomheader = new DicomHeader();
        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event IOProcessEventHandler IOProgress;
        /// <summary>
        /// generic  info GN
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct GN_INFO
        {
            /// <summary>
            /// study date as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] study_date;
            /// <summary>
            /// study time as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] study_time;
            /// <summary>
            /// series date as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] series_date;
            /// <summary>
            /// series time as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] series_time;
            /// <summary>
            /// acquisition date as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] acquisition_date;
            /// <summary>
            /// acquisition time as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] acquisition_time;
            /// <summary>
            /// imaging date as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] image_date;
            /// <summary>
            /// imaging time as string
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] image_time;
        }
        /// <summary>
        /// specific info MR modality
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct MR_INFO
        {
            /// <summary>
            /// RF sequance repetition time ?
            /// </summary>
            public double repetition_time;
            /// <summary>
            /// unknown
            /// </summary>
            public double echo_time;
            /// <summary>
            /// unknown
            /// </summary>
            public double inversion_time;
            /// <summary>
            /// unknown
            /// </summary>
            public double num_averages;
            /// <summary>
            /// imaging frequency
            /// </summary>
            public double imaging_freq;
            /// <summary>
            /// unknown
            /// </summary>
            public double pixel_bandwidth;
            /// <summary>
            /// unknown
            /// </summary>
            public double flip_angle;
            /// <summary>
            /// unknown
            /// </summary>
            public double dbdt;
            /// <summary>
            /// unknown
            /// </summary>
            public uint transducer_freq;
            /// <summary>
            /// unknown
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] transducer_type;
            /// <summary>
            /// RG pulse repetition frequency
            /// </summary>
            public uint pulse_repetition_freq;
            /// <summary>
            /// unknown
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] pulse_seq_name;
            /// <summary>
            /// unknown
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] steady_state_pulse_seq;
            /// <summary>
            /// unknown
            /// </summary>
            public double slab_thickness;
            /// <summary>
            /// sampling frequency
            /// </summary>
            public double sampling_freq;
        }
        /// <summary>
        /// Modality info structure
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct MOD_INFO
        {
            /// <summary>
            /// PET modality info
            /// </summary>
            public GN_INFO gn_info;
            // XA_INFO xa_info; 
            /// <summary>
            /// MR modality info
            /// </summary>
            public MR_INFO mr_info;
        }
        /// <summary>
        /// Static image related data
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct STATIC_DATA
        {
            /// <summary>
            /// label name of image (Anterior/Posterior)
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = DicomConstants.MAXSTR)]
            public char[] label;       
            /// <summary>
            /// total counts in image
            /// </summary>
            public float total_counts;            //   
            /// <summary>
            /// duration of image (ms)
            /// </summary>
            public float image_duration;        
            /// <summary>
            /// start time hour
            /// </summary>
            public short start_time_hour;        
            /// <summary>
            /// start time minute
            /// </summary>
            public short start_time_minute;       
            /// <summary>
            /// start time second
            /// </summary>
            public short start_time_second;       
        }
        /// <summary>
        /// Gated image related data
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct GATED_DATA
        {
            /// <summary>
            /// gated spect nesting
            /// </summary>
            public sbyte gspect_nesting;       
            /// <summary>
            /// number of projections
            /// </summary>
            public float nr_projections;        
            /// <summary>
            /// extent of rotation
            /// </summary>
            public float extent_rotation;        
            /// <summary>
            /// study duration (ms)
            /// </summary>
            public float study_duration;         
            /// <summary>
            /// image duration (ms)
            /// </summary>
            public float image_duration;          
            /// <summary>
            /// time per proj  (ms)
            /// </summary>
            public float time_per_proj;          
            /// <summary>
            /// lower  limit   (ms)
            /// </summary>
            public float window_low;              
            /// <summary>
            /// higher limit   (ms)
            /// </summary>
            public float window_high;           
            /// <summary>
            /// cardiac cycles observed
            /// </summary>
            public float cycles_observed;        
            /// <summary>
            /// cardiac cycles acquired
            /// </summary>
            public float cycles_acquired;         
        }
        /// <summary>
        /// Acquisition data
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct ACQ_DATA
        {
            /// <summary>
            /// direction of rotation
            /// </summary>
            public short rotation_direction;      //   
            /// <summary>
            /// type detector motion 
            /// </summary>
            public short detector_motion;      
            /// <summary>
            /// centre rotation offset
            /// </summary>
            public float rotation_offset;        
            /// <summary>
            /// radial position
            /// </summary>
            public float radial_position;       
            /// <summary>
            /// start angle (interfile) 180 - dicom
            /// </summary>
            public float angle_start;         
            /// <summary>
            /// angular step
            /// </summary>
            public float angle_step;             
            /// <summary>
            /// angular range
            /// </summary>
            public float scan_arc;               
        }
        /// <summary>
        /// dynamic data
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct DYNAMIC_DATA
        {
            /// <summary>
            /// images in time frame
            /// </summary>
            public uint nr_of_slices;           
            /// <summary>
            /// start time frame (ms)
            /// </summary>
            public float time_frame_start;        
            /// <summary>
            /// delay this frame (ms)
            /// </summary>
            public float time_frame_delay;        
            /// <summary>
            /// duration   frame (ms)
            /// </summary>
            public float time_frame_duration;     
            /// <summary>
            /// delay each slice (ms)
            /// </summary>
            public float delay_slices;            
        }
        /// <summary>
        /// bed data
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Size = 8)]
        protected struct BED_DATA
        {
            /// <summary>
            /// horizon. position (mm)
            /// </summary>
            public float hoffset;                
            /// <summary>
            /// vertical position (mm)
            /// </summary>
            public float voffset;                 
        }
        /// <summary>
        /// images related data
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        protected struct IMG_DATA
        {
            //             **   general data   **                    
            /// <summary>
            /// image dimension
            /// </summary>
            public uint width;
            /// <summary>
            /// image dimension
            /// </summary>
            public uint height;             
            /// <summary>
            /// bits/pixel
            /// </summary>
            public short bits;
            /// <summary>
            /// datatype
            /// </summary>
            public short type;            
            /// <summary>
            /// extra flag
            /// </summary>
            public ushort flags;		        
            /// <summary>
            /// min pixelvalue
            /// </summary>
            public double min;
            /// <summary>
            /// max pixelvalue
            /// </summary>
            public double max;                
            /// <summary>
            /// quantified min
            /// </summary>
            public double qmin;
            /// <summary>
            /// quantified max
            /// </summary>
            public double qmax;               
            /// <summary>
            /// min in whole frame
            /// </summary>
            public double fmin;
            /// <summary>
            /// max in whole frame
            /// </summary>
            public double fmax;               
            /// <summary>
            /// in whole frame (quantified) min
            /// </summary>
            public double qfmin;
            /// <summary>
            /// in whole frame (quantified) max
            /// </summary>
            public double qfmax;         
            /// <summary>
            /// rescale slope. auto filled
            /// </summary>
            public float rescale_slope;        
            /// <summary>
            ///  rescale intercept. auto filled
            /// </summary>
            public float rescale_intercept;    
                        /// <summary>
            /// part of frame (1-based) (auto filled)
            /// </summary>
            public uint frame_number;
            /// <summary>
            /// start    of slice (ms) (auto filled)
            /// </summary>
            public float slice_start;
            /// <summary>
            /// Slice location (mm).
            /// </summary>
            public float slice_location;
            /// <summary>
            /// pointer to raw image, uint8
            /// </summary>
            public IntPtr buf;    
            /// <summary>
            /// load start in file
            /// </summary>
            public int load_location;           
            /// <summary>
            /// internal item: rescaled image?
            /// </summary>
            public sbyte rescaled;             
            /// <summary>
            /// new rescaled max
            /// </summary>
            public double rescaled_min;          
            /// <summary>
            /// new rescaled min
            /// </summary>
            public double rescaled_max;        
            /// <summary>
            /// new rescaled fctr
            /// </summary>
            public double rescaled_fctr;        
            /// <summary>
            /// new rescaled slope
            /// </summary>
            public double rescaled_slope;       
            /// <summary>
            /// new rescaled intercept
            /// </summary>
            public double rescaled_intercept;    
            //             **   ecat64 items   **                    
            /// <summary>
            /// quantification units
            /// </summary>
            public short quant_units;             
            /// <summary>
            /// calibration units
            /// </summary>
            public short calibr_units;           
            /// <summary>
            /// quantification scale
            /// </summary>
            public float quant_scale; 
            /// <summary>
            /// calibration factor
            /// </summary>
            public float calibr_fctr;    
            /// <summary>
            /// scale intercept
            /// </summary>
            public float intercept;               
            /// <summary>
            /// pixel size X      (mm)
            /// </summary>
            public float pixel_xsize;             
            /// <summary>
            /// pixel size Y      (mm)
            /// </summary>
            public float pixel_ysize;             
            /// <summary>
            /// slice width       (mm)
            /// </summary>
            public float slice_width;   
            /// <summary>
            /// recon magnification
            /// </summary>
            public float recon_scale;           
            //            **  Acr/Nema items   **                    
            /// <summary>
            /// image posit  dev  (mm)
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 3)]
            public float[] image_pos_dev;        
            /// <summary>
            /// image orient dev  (mm)
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 6)]
            public float[] image_orient_dev;     
            /// <summary>
            /// image posit  pat  (mm)
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 3)]
            public float[] image_pos_pat;        
            /// <summary>
            /// image orient pat  (mm)
            /// </summary>
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 6)]
            public float[] image_orient_pat;     
            /// <summary>
            /// space btw centres (mm)
            /// </summary>
            public float slice_spacing;       
            /// <summary>
            /// CT image zoom factor
            /// </summary>
            public float ct_zoom_fctr;            
            //            **  Miscellaneous    **                    
            /// <summary>
            /// extra static entries
            /// </summary>
            public IntPtr sdata;            
            /// <summary>
            /// like to attach here?
            /// </summary>
            public IntPtr plugb;          
        }
        /// <summary>
        /// Dictionary containing all recognized DICOM data elements for reading and writing.
        /// </summary>
        public static DataElementDictionary dataElementDictionary;
        /// <summary>
        /// Dictionary containing all recognized DICOM unique identifiers (Uids) for reading and writing.
        /// </summary>
        public static UidDictionary uidDictionary;
        /// <summary>
        /// Static constructor that reads dictionary files for elements and uids, when this 
        /// class is used first time.
        /// </summary>
        static DicomFile() {
            dataElementDictionary = new DataElementDictionary();
            uidDictionary = new UidDictionary();
            try
            {
                dataElementDictionary.LoadFromString(TPClib.Properties.Resources.TPC_elems3, 
                                               DictionaryFileFormat.PropertyFile);
                uidDictionary.LoadFromString(TPClib.Properties.Resources.TPC_uids2,  
                                       DictionaryFileFormat.PropertyFile);
                //dataElementDictionary.LoadFrom("TPC_elems2.dic", DictionaryFileFormat.PropertyFile);
                //uidDictionary.LoadFrom("TPC_uids.dic", DictionaryFileFormat.PropertyFile);
            }
            catch (Exception dictionaryException)
            {
                throw new TPCDicomFileError("Problems processing dictionaries:\n" + dictionaryException);
            }
        }

        /// <summary>
        /// Constructs DICOM file with filename.
        /// </summary>
        /// <param name="filename">full path to a single file</param>
        public DicomFile(string filename) {
            this.filename = filename;
            this.filetype = FileType.DICOM;
            header = new ImageHeader();
        }
        /// <summary>
        /// Constructs DICOM file.
        /// </summary>
        /// <param name="filename">basename of DICOM file</param>
        /// <param name="image">image data</param>
        /// <param name="header">image header</param>
        public DicomFile(string filename, Image image, ImageHeader header)
            : this(filename)
        {
            this.filetype = FileType.DICOM;
            this.image = image;
            if (image is DynamicImage)
            {
                dicomheader.frames = (image as DynamicImage).frames;
                if (image.dim.Length > IntLimits.GATES) dicomheader.gate_number = (image as DynamicImage).gates;
            }
            dicomheader.planes = (short)image.dim.GetDimension(IntLimits.PLANES);
            this.header = header;
            this.header.datatype = DataType.BIT16_S;
            this.header.dim = image.dim;
            this.header.dataunit = header.dataunit;
        }
        /// <summary>
        /// Constructs an empty (all-zero) DICOM file with filename and dimensions.
        /// </summary>
        /// <param name="filename">filename of DICOM file</param>
        /// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
        public DicomFile(string filename, params int[] dim)
        {
            this.filename = filename;
            this.filetype = FileType.DICOM;
            int[] d = new int[4];
            int i = 0;
            for(; i < dim.Length; i++)
                d[i] = dim[i];
            for(; i < d.Length; i++)
                d[i] = 1;
            if (dim.Length > IntLimits.FRAMES)
            {
                header = new DynamicImageHeader();
                image = new DynamicImage(new IntLimits(dim));
            }
            else
            {
                header = new ImageHeader();
                image = new Image(dim);
            }
            header.dim = new TPClib.IntLimits(dim);
            header.datatype = TPClib.Image.ImageFile.DataType.BIT16_S;
        }

        /// <summary>
        /// Converts data object to byte array
        /// </summary>
        /// <param name="data">object that has DICOM tag data</param>
        /// <returns>DICOM data as byte array</returns>
        public byte[] ToBytesArray(Object data) {

            if (data is Array) {
                Array a = ((Array)data);
                List<byte[]> values = new List<byte[]>();
                foreach(Object obj in a)
                {
                    values.Add(ToBytesArray(obj));
                }
                if(values.Count == 0) return new byte[0];
                byte[] r = new byte[values.Count*values[0].Length];
                int r_i = 0;
                for (int i = 0; i < values.Count; i++)
                {
                    for (int j = 0; j < values[i].Length; j++)
                    {
                        r[r_i++] = values[i][j];
                    }
                }
                return r;
            }
            if (data is byte)
            {
                return new byte[1] {(byte)data};
            }
            if (data is char)
            {
                return new byte[1] {(byte)((char)data)};
            }
            else if (data is UInt32)
            {
                return BitConverter.GetBytes((UInt32)data);
            }
            else if (data is Int32)
            {
                return BitConverter.GetBytes((Int32)data);
            }
            else if (data is UInt16)
            {
                return BitConverter.GetBytes((UInt16)data);
            }
            else if (data is Int16)
            {
                return BitConverter.GetBytes((Int16)data);
            }
            else if (data is Int64)
            {
                return BitConverter.GetBytes((Int64)data);
            }
            else if (data is Single)
            {
                return BitConverter.GetBytes((Single)data);
            }
            else if (data is Double)
            {
                return BitConverter.GetBytes((Double)data);
            }
            else if (data is string)
            {
                return ASCIIEncoding.ASCII.GetBytes((string)data);
            }
            else if (data is Uid)
            {
                return ASCIIEncoding.ASCII.GetBytes(((Uid)data).ToString());
            }
            else if (data is openDicom.Encoding.Type.Age)
            {
                openDicom.Encoding.Type.Age age = (openDicom.Encoding.Type.Age)data;
                if(age.IsDays)
                    return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "D");
                else if(age.IsMonths)
                    return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "M");
                else if(age.IsWeeks)
                    return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "W");
                else
                    return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "Y");
            }
            else if (data is System.DateTime)
            {
                return ASCIIEncoding.ASCII.GetBytes(((System.DateTime)data).Year.ToString("0000") +
                    ((System.DateTime)data).Month.ToString("00") +
                    ((System.DateTime)data).Day.ToString("00"));
            }
            else if (data is System.TimeSpan)
            {
                string str = ((System.TimeSpan)data).Hours.ToString("00") +
                    ((System.TimeSpan)data).Minutes.ToString("00") +
                    ((System.TimeSpan)data).Seconds.ToString("00");
                if (((System.TimeSpan)data).Milliseconds > 0)
                    str += ((System.TimeSpan)data).Milliseconds.ToString("000000");
                return ASCIIEncoding.ASCII.GetBytes(str);
            }
            else if (data is openDicom.Encoding.Type.PersonName)
            {
                return ASCIIEncoding.ASCII.GetBytes(((openDicom.Encoding.Type.PersonName)data).ToString());
            }
            else if (data is Decimal)
            {
                return ASCIIEncoding.ASCII.GetBytes(data.ToString());
            }
            else if (data is openDicom.DataStructure.DataSet.Sequence)
            {
                //sequence delimeter has no data
                return new byte[0];
            }
            else
            {
                throw new TPCDicomFileException("Failed to read DICOM tag in file (" + data.GetType() + ") data");
            }
        }

        /// <summary>
        /// Reads all DICOM tags and returns pointer to their data. File must be closed 
        /// before reading, since file is opened and closed.
        /// </summary>
        /// <remarks>This public method is unsafe. Improper usage may result memory corruption.</remarks>
        /// <returns>tag structure array</returns>
        /// <exception cref="TPCDicomFileException">if tag could not be read</exception>
        public DataElement[] ReadTAGs()
        {
            openDicom.File.AcrNemaFile file = null;
            try
            {
                if (openDicom.File.DicomFile.IsDicomFile(filename))
                    file = new openDicom.File.DicomFile(filename, false);
                else if (openDicom.File.AcrNemaFile.IsAcrNemaFile(filename))
                    file = new openDicom.File.AcrNemaFile(filename, false);
                else
                    Console.Error.WriteLine("Selected file is wether a " +
                        "DICOM nor an ACR-NEMA file.");
            }
            catch (Exception dicomFileException)
            {
                throw new TPCDicomFileException("Failed to read DICOM tags in file " + filename + ":" + dicomFileException);
            }
            Sequence s = file.GetJointDataSets().GetJointSubsequences();
            return s.ToArray();
        }

        /// <summary>
        /// Reads DICOM tag. 
        /// <remarks>Note that all DICOM fields are run through. Use readTAGs if reading multiple tags.</remarks>
        /// </summary>
        /// <param name="group">group number</param>
        /// <param name="element">element number</param>
        /// <returns>data element from file</returns>
        /// <exception cref="TPCDicomFileException">if tag could not be read</exception>
        public DataElement ReadTAG(UInt16 group, UInt16 element)
        {
            DataElement[] elements = ReadTAGs();
            foreach (DataElement d in elements)
            {
                if (d.Tag.Equals("("+group.ToString("0000")+","+element.ToString("0000")+")"))
                {
                    return d;
                }
            }
            throw new TPCDicomFileException("Tag (" + group.ToString("0000") + "," + element.ToString("0000") + ") not found");
        }

        /// <summary>
        /// Resolves names of available DICOM files in path including subdirectories
        /// </summary>
        /// <param name="path">full path to the files</param>
        /// <returns>array of files</returns>
        public static FileInfo[] ResolveDicomFilenames(string path) {
            char[] chars;
            string pathroot = Path.GetDirectoryName(path);
            string filename = Path.GetFileName(path);
            chars = Path.GetInvalidFileNameChars();
            for (int i = 0; i < chars.Length; i++)
            {
                if (chars[i].CompareTo('*') == 0) continue;
                if (filename.Contains(chars[i].ToString())) throw new TPCDicomFileException("Invalid character '" + chars[i] + "' in filename");
            }
            chars = Path.GetInvalidPathChars();
            for (int i = 0; i < chars.Length; i++)
                if (pathroot.Contains(chars[i].ToString())) throw new TPCDicomFileException("Invalid character '" + chars[i] + "' in path");
            try
            {
                DirectoryInfo dir = new DirectoryInfo(pathroot);
                return dir.GetFiles(filename, SearchOption.AllDirectories);
            }
            catch (ArgumentException e)
            {
                throw new TPCDicomFileException("Path contains invalid characters:"+e.Message);
            }
            catch (Exception e)
            {
                 throw new TPCDicomFileException("While creating DirectoryInfo from ["+pathroot+"]:" + e.Message);
            }
        }

        /// <summary>
        ///     Returns the entire DICOM pixel data as array of byte arrays.
        /// </summary>
        /// <remarks>
        ///     All non-byte arrays are transcoded into byte arrays. If a DICOM
        ///     pixel data element is not a DICOM sequence of items, an array
        ///     with a single byte array entry will be returned.
        /// </remarks>
        protected static byte[] ToBytesArray(DataElement data)
        {
            byte[] bytesArray;
            if (data.Value[0] is ushort[])
                bytesArray = ByteConvert.ToBytes(
                    (ushort[])data.Value[0]);
            else
                bytesArray = (byte[])data.Value[0];
            return bytesArray;
        }
        /// <summary>
        /// Reads all DICOM files that match search criteria into single file. 
        /// All dimensions higher than or equal to 4th are stacked together as frames.
        /// </summary>
        /// <param name="file">full path to the files</param>
        /// <returns>array of DICOM files</returns>
        public static DicomFile ReadMultiple(string file)
        {
            //all filenames found in path
            FileInfo[] filenames = ResolveDicomFilenames(file);
            //1st read DICOM holding header information
            DicomFile dcmfile_1st = null;
            //currently read DICOM holding header information
            DicomFile dcmfile_current;
            //DICOM files holding header information
            List<DicomFile> dcmfiles = new List<DicomFile>();
            //Event handler where progress events are sent to
            IOProcessEventHandler pevent = null;
            //gather recognized files
            for (int i = 0; i < filenames.Length; i++)
            {
                try
                {
                    if (!DicomFile.CheckFormat(filenames[i].FullName))
                    {
                        //just skip the file if format does not match
                        continue;
                    }
                    //read all header fields except the actual image data
                    dcmfile_current = new DicomFile(filenames[i].FullName);
                    dcmfile_current.ReadHeader();
                    //set dimensions to 1 if information was not found or was zero in the file
                    if (dcmfile_current.dicomheader.planes == 0)
                        dcmfile_current.dicomheader.planes = 1;
                    pevent = IOProgress;
                    if (i % 5 == 0 && pevent != null)
                        pevent.Invoke(dcmfile_current, new IOProgressEventArgs(100.0f * (float)i / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
                    if (dcmfile_1st != null)
                    {
                        //ensure consistency of data
                        if (!dcmfile_1st.header.dim.Equals(dcmfile_current.header.dim))
                            throw new TPCDicomFileException("Inconsistent dimensions in read DICOM files. Cannot combine into single file.");
                    }
                    else
                    {
                        dcmfile_1st = dcmfile_current;
                    }
                    dcmfiles.Add(dcmfile_current);
                }
                catch (TPCUnrecognizedFileFormatException)
                {
                    // just skip the file 
                }
                catch (TPCException e)
                {
                    // print error and skip the file 
                    Console.WriteLine("Cannot read DICOM file " + filenames[i].FullName + ":" + e.Message);
                }
            }
            
            if (dcmfiles.Count == 0) throw new TPCDicomFileException("No DICOM files was found.");
            int framelength = dcmfile_1st.header.dimx*dcmfile_1st.header.dimy*dcmfile_1st.dicomheader.planes;
            if (framelength == 0) throw new TPCDicomFileException("Zero-dimensioned data");
            if ((dcmfiles.Count * dcmfile_1st.dicomheader.planes) / framelength > dcmfile_1st.dicomheader.frames)
                dcmfile_1st.dicomheader.frames = (dcmfiles.Count * dcmfile_1st.dicomheader.planes) / framelength;
            //fix number of planes if planes and frames does not multiply to number of found images
            if (dcmfile_1st.dicomheader.frames * dcmfile_1st.dicomheader.planes < dcmfiles.Count)
            {
                List<float> slice_locations = new List<float>();
                for (int i = 0; i < dcmfiles.Count; i++) {
                    if(!slice_locations.Contains(dcmfiles[i].dicomheader.slice_location)) {
                        slice_locations.Add(dcmfiles[i].dicomheader.slice_location);
                    }
                }
                if(slice_locations.Count > 1) {
                    dcmfile_1st.dicomheader.planes = slice_locations.Count;
                }
                else {
                    if (dcmfile_1st.dicomheader.frames > 0)
                        dcmfile_1st.dicomheader.planes = dcmfiles.Count / dcmfile_1st.dicomheader.frames;
                    else
                        dcmfile_1st.dicomheader.planes = dcmfiles.Count;
                }
            }
            //resolve file size and allocate memory
            DicomFile dcmfile;
            if (dcmfile_1st.dicomheader.frames == 0)
            {
                dcmfile = new DicomFile(file, dcmfile_1st.header.dimx, dcmfile_1st.header.dimy, dcmfiles.Count);
                dcmfile.header = dcmfile_1st.header;
            }
            else
            {
                dcmfile = new DicomFile(file, dcmfile_1st.header.dimx, dcmfile_1st.header.dimy, dcmfile_1st.dicomheader.planes, dcmfile_1st.dicomheader.frames);
                if(dcmfile_1st.header is DynamicImageHeader)
                    dcmfile.header = dcmfile_1st.header;
                else
                    dcmfile.header = new DynamicImageHeader(dcmfile_1st.header);
                for (int i = 0; i < dcmfile_1st.dicomheader.frames; i++)
                {
                    (dcmfile.image as DynamicImage).SetFrameStartTime(i, dcmfiles[i * dcmfile_1st.dicomheader.planes].dicomheader.frame_start_time);
                    (dcmfile.image as DynamicImage).SetFrameDuration(i, dcmfiles[i * dcmfile_1st.dicomheader.planes].dicomheader.frame_duration);
                }
            }
            dcmfile.header.dim = dcmfile.image.dim;
            dcmfile.header.datatype = dcmfile_1st.header.datatype;
            dcmfile.header.dataunit = dcmfile_1st.header.dataunit;

            //read data from previously acquired positions
            int location = 0;
            for (int i = 0; i < dcmfiles.Count; i++)
            {
                dcmfiles[i].GetPixelData(ref dcmfile.image, location, dcmfiles[i].filename, new IntLimits(dcmfile_1st.header.dimx, dcmfile_1st.header.dimy), dcmfile.header.datatype, dcmfiles[i].dicomheader.stream_data_position, dcmfiles[i].dicomheader.scalefactor);
                location+=dcmfile_1st.header.dimx*dcmfile_1st.header.dimy;
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(dcmfiles[i], new IOProgressEventArgs(100.0f * (float)i / dcmfiles.Count, System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel data."));
            }
            return dcmfile;
        }
        /// <summary>
        /// Reads all DICOM files from path.
        /// </summary>
        /// <param name="path">full path to the files</param>
        /// <returns>array of DICOM files</returns>
        public static DicomFile[] ReadDicomPath(string path) {
            FileInfo[] filenames = ResolveDicomFilenames(path);
            List<DicomFile> dicomlist = new List<DicomFile>();
            DicomFile[] r;
            DicomFile dfile = null;
            int current_instance_number = 0;
            IOProcessEventHandler pevent = null;
            //gather data into array and return it
            for(int i = 0; i < filenames.Length; i++)
            {
                try
                {
                    if (!DicomFile.CheckFormat(filenames[i].FullName))
                    {
                        //just skip the file if format does not match
                        continue;
                    }
                    dfile = new DicomFile(filenames[i].FullName);
                    dfile.ReadFile();
                    //start new instance numbering if series is changed
                    if (dfile.dicomheader.instance_nr == 0 &&
                        (dicomlist.Count == 0 || dicomlist[dicomlist.Count - 1].dicomheader.series_nr != dfile.dicomheader.series_nr))
                        current_instance_number = 0;
                    dfile.dicomheader.instance_nr = current_instance_number++;
                    dicomlist.Add(dfile);
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(dfile, new IOProgressEventArgs(100.0f * (float)i / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
                }
                catch (TPCUnrecognizedFileFormatException) { 
                    // just skip the file 
                }
                catch (TPCException e) { 
                    // print error and skip the file 
                    Console.WriteLine("Cannot read file " + filenames[i].FullName + ":" + e.Message);
                }
            }
            r = dicomlist.ToArray();
            if (pevent != null)
                pevent.Invoke(dfile, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));      
            return r;
        }

        /// <summary>
        /// Implementation of ImageFile interface. Reads a single DICOM file.
        /// </summary>
        /// <exception cref="TPCDicomFileException">if file was not opened properly</exception>
        /// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
        public override void ReadFile()
        {
            //read header information
            ReadHeader();

            //read plane image
            if(header is DynamicImageHeader)
                image = new DynamicImage(header.dimx, header.dimy, 1);
            else
                image = new Image(header.dimx, header.dimy, 1);      
            GetPixelData(ref image, 0, filename, new IntLimits(header.dimx, header.dimy), header.datatype, dicomheader.stream_data_position, dicomheader.scalefactor);
            if (image is DynamicImage) {
                (image as DynamicImage).SetFrameStartTime(0, dicomheader.frame_start_time);
                (image as DynamicImage).SetFrameDuration(0, dicomheader.frame_duration);
            }
        }

        /// <summary>
        /// Resolves number of gates in dicom image. 
        /// </summary>
        /// <param name="files">planes of dicom file</param>
        /// <returns>number of gates in image (allways >= 1)</returns>
        /// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
        public static int ResolveNoGates(DicomFile[] files) {
            int j = files[0].dicomheader.slice_start;
            int k = 1;
            DataElement d;

            if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

            //try to read value straight from DICOM field (0054,0101) 'NumberOfTimeSlices'
            try
            {
                d = files[0].ReadTAG(54, 71);
                k = (UInt16)(d.Value.ToArray()[0]);
                if (k > 0) return k;
            }
            catch (TPCDicomFileException)
            {
                //do nothing, failed to resolve gates
            }
            if (k > 1) return k;
            return 1;
        }

        /// <summary>
        /// Resolves number of frames in dicom image. 
        /// </summary>
        /// <param name="files">planes of dicom file</param>
        /// <returns>number of frames in image (allways >= 1)</returns>
        /// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
        public static int ResolveNoFrames(DicomFile[] files) {
            int i = 0;
            int j = files[0].dicomheader.slice_start;
            int k = 1;
            DataElement d;

            if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

            //try to read value straight from DICOM field (0054,0101) 'NumberOfTimeSlices'
            try
            {
                d = files[0].ReadTAG(54, 101);
                k = (UInt16)(d.Value.ToArray()[0]);
                if (k > 0) return k;
            }
            catch (Exception)
            {
                //do nothing, determine value with slice_start fields
                //Console.WriteLine("Field (0054,0101) not found for frame number information:"+e.Message);
            }
            k = 0;
            for (; i < files.Length; i++) {
                // use frame start times 
                if (j != files[i].dicomheader.slice_start)
                {
                    j = files[i].dicomheader.slice_start;
                    k++;
                }
            }
            if (k > 1) return k;
            return 1;
        }
        /// <summary>
        /// Resolves number of planes in dicom image. 
        /// </summary>
        /// <param name="files">planes of dicom file</param>
        /// <returns>number of planes in image (allways >= 1)</returns>
        /// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
        public static int ResolveNoPlanes(DicomFile[] files)
        {
            int j = files[0].dicomheader.slice_start;
            int k = 1;
            DataElement d;

            if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

            //try to read value straight from DICOM field (0054,0081) 'NumberOfSlices'
            try
            {
                d = files[0].ReadTAG(54, 81);
                k = (UInt16)(d.Value.ToArray()[0]);
                if (k > 0) return k;
            }
            catch (Exception)
            {
                //do nothing, determine using number of frames and number of images
            }
            k = files.Length/ResolveNoFrames(files);
            if (k > 1) return k;
            return 1;
        }

        /// <summary>
        /// Calculates optimal scale factor for curent datatype.
        /// </summary>
        public float ResolveScaleFactor() {
            float[] minmax = minmax = image.GetMinMax();
            float r = 1.0f;
            switch (header.datatype)
            {
                //no scaling for byte data 
                case DataType.BIT8_U:
                case DataType.BIT8_S:
                    r = 1.0f;
                    break;
                case DataType.BIT16_U:
                    r = (float)(minmax[1] / (double)ushort.MaxValue);
                    break;
                case DataType.BIT16_S:
                    r = (float)(Math.Max(Math.Abs(minmax[0]),Math.Abs(minmax[1]))/(double)short.MaxValue);
                    break;
                case DataType.BIT32_U:
                    r = (float)(minmax[1] / (double)int.MaxValue);
                    break;
                case DataType.BIT32_S:
                    r = (float)(Math.Max(Math.Abs(minmax[0]),Math.Abs(minmax[1]))/(double)(Math.Pow(2.0, 32)));
                    break;
                case DataType.BIT64_U:
                    r = (float)Math.Abs(minmax[1] / (double)Int64.MaxValue);
                    break;
                case DataType.BIT64_S:
                    r = (float)(Math.Max(Math.Abs(minmax[0]), Math.Abs(minmax[1]))/(double)(Math.Pow(2.0, 64)));
                    break;
                case DataType.FLT32:
                    //no conversion for float data
                    break;
                default:
                    break;
            }
            if (r == 0) r = 1.0f;
            return r;
        }
        /// <summary>
        /// Implementation of ImageFile interface. Writes a single DICOM file. 
        /// </summary>
        /// <param name="image">image data that is written</param>
        public override void WriteFile(ref Image image)
        {
            //write higher dimension images recursively
            int i = 0;
            int j = 0;
            IOProcessEventHandler pevent = null;
            if (image.dim.Length > IntLimits.PLANES)
            {
                for (i = image.dim.Length-1; i >= IntLimits.PLANES; i--)
                {
                    if (image.dim.GetDimension(i) > 1) break;
                }
                if (i >= IntLimits.PLANES)
                {
                    DicomFile[] splitted = SplitDimension(i);
                    for (j = 0; j < splitted.Length; j++)
                    {
                        splitted[j].header.dim.SetLimits(i, 0, splitted.Length);
                        splitted[j].dicomheader.instance_nr = j + 1;
                        splitted[j].WriteFile();
                        pevent = IOProgress;
                        if (pevent != null)
                            pevent.Invoke(this, new IOProgressEventArgs((float)Math.Round(100.0f * (float)j / (float)splitted.Length, 0), System.Reflection.MethodBase.GetCurrentMethod()));
                    }
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
                    return;
                }
            }
            dicomheader.instance_nr = imageindex++;
            try
            {
                FileStream stream = new FileStream(filename, FileMode.Create);
                dicomheader.scalefactor = ResolveScaleFactor();
                Write0002(stream);
                Write0008(stream);
                //no effect to CortexID
                Write0009(stream);
                Write0010(stream);
                //no effect to CortexID
                Write0011(stream);
                //no effect to CortexID
                Write0013(stream);
                //no effect to CortexID
                Write0018(stream);
                Write0020(stream);
                Write0028(stream);
                //no effect to CortexID
                Write0054(stream);
                Write7FE0(stream);
                stream.Close();
            }
            catch (Exception dicomFileException)
            {
                Console.Error.WriteLine("Problems processing DICOM file:\n" + dicomFileException);
                return;
            }
        }
        /// <summary>
        /// Class for writing of recursive fields
        /// </summary>
        private class NestedDataSetWriteData {
            protected List<Object> elements = new List<object>();
            public void Clear()
            {
                elements.Clear();
            }
            public void Add(Object obj)
            {
                elements.Add(obj);

            }
            public List<Object>.Enumerator GetEnumerator()
            {
                return elements.GetEnumerator();
            }
            public int Count {
                get { return elements.Count; }
            }
        }
        /// <summary>
        /// Class for writing of recursive fields
        /// </summary>
        private class SequenceWriteData
        {
            public Tag tag = new Tag(0,0);
            protected List<Object> elements = new List<object>();
            public void SetTag(UInt16 group, UInt16 element) {
                tag = new Tag(group, element);
            }
            public void Clear() {
                elements.Clear();
            }
            public void Add(Object obj) {
                elements.Add(obj);
                
            }
            public List<Object>.Enumerator GetEnumerator() {
                return elements.GetEnumerator();
            }
            public int Count {
                get { return elements.Count; }
            }
        }

        /// <summary>
        /// Creates DICOM dataelement for writing
        /// </summary>
        /// <param name="element">element number</param>
        /// <param name="group">group number</param>
        /// <param name="data">data array</param>
        DataElement CreateDataElement(UInt16 group, UInt16 element, byte[] data)
        {
            Tag tag = new Tag(group, element);
            tag.TransferSyntax = TransferSyntax.Default;
            if (data.Length % 2 != 0) { 
                byte[] even = new byte[data.Length+1];
                data.CopyTo(even, 0);
                data = even;
            }
            if (dataElementDictionary[tag] == null)
                throw new TPCDicomFileError("Tag "+tag+" not found in dictionary, cannot write element.");
            Value value = new Value(dataElementDictionary[tag].VR, new ValueLength(null, data.Length));
            if(!BitConverter.IsLittleEndian)
                swapArrayBytes(data, 0, data.Length);
            value.Add(data);
            return new DataElement(tag, value);
        }

        /// <summary>
        /// Writes DICOM tag element into stream
        /// </summary>
        /// <param name="stream">writable stream</param>
        /// <param name="element">element number</param>
        /// <param name="group">group number</param>
        /// <param name="data">data array</param>
        public static void WriteElement(Stream stream, UInt16 group, UInt16 element, byte[] data)
        {
            Tag tag = new Tag(group, element);
            tag.TransferSyntax = TransferSyntax.Default;
            if (data.Length % 2 != 0) { 
                byte[] even = new byte[data.Length+1];
                data.CopyTo(even, 0);
                data = even;
            }
            if (dataElementDictionary[tag] == null)
                throw new TPCDicomFileError("Tag "+tag+" not found in dictionary, cannot write element.");
            Value value = new Value(dataElementDictionary[tag].VR, new ValueLength(null, data.Length));
            if(!BitConverter.IsLittleEndian)
                swapArrayBytes(data, 0, data.Length);
            value.Add(data);
            DataElement dataelement = new DataElement(tag, value);
            dataelement.SaveTo(stream);
        }

        /// <summary>
        /// Writes DICOM info sequence. Data element length is set to 
        /// 0xFFFFFFFF (undefined).
        /// </summary>
        /// <param name="stream">writable stream</param>
        /// <param name="group">group number</param>
        /// <param name="element">element number</param>
        /// <returns>position at data element length writing</returns>
        long WriteInfoSeq(Stream stream, UInt16 group, UInt16 element)
        {
            Tag tag = new Tag(group, element);
            tag.TransferSyntax = TransferSyntax.Default;
            tag.SaveTo(stream);
            BinaryWriter binarystream = new BinaryWriter(stream);
            binarystream.Write(ToBytesArray("SQ".ToCharArray()));
            binarystream.Write(new byte[2]);
            long pos = stream.Position;
            binarystream.Write(new byte[]{0xFF, 0xFF, 0xFF, 0xFF});
            return pos;
        }
        /// <summary>
        /// Writes info sequene recursively. Writing mode is EXPLICIT, while still
        /// defining sequence length, and without ending tags (mixed representation understood by 
        /// Vinci, DCMTK).
        /// </summary>
        /// <param name="stream">output stream</param>
        /// <param name="dataset">dataset that is written</param>
        void WriteDataSetRecursive(Stream stream, NestedDataSetWriteData dataset)
        {
            long length_pos = WriteItemHeader(stream, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            long start_pos = stream.Position;
            foreach (Object d in dataset)
            {
                if (d is NestedDataSetWriteData)
                {
                    WriteDataSetRecursive(stream, (NestedDataSetWriteData)d);
                }
                else if (d is SequenceWriteData)
                {
                    WriteSequenceRecursive(stream,(SequenceWriteData)d);
                }
                else if (d is DataElement)
                { 
                    //write field
                    ((DataElement)d).SaveTo(stream);
                }
            }
            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes info sequene recursively. Writing mode is EXPLICIT, while still
        /// defining sequence length, and without ending tags (mixed representation understood by 
        /// Vinci, DCMTK).
        /// </summary>
        /// <param name="stream">output stream</param>
        /// <param name="sequence">sequence that is written</param>
        void WriteSequenceRecursive(Stream stream, SequenceWriteData sequence)
        {
            
            long length_pos = WriteInfoSeq(stream,
                UInt16.Parse(sequence.tag.Group, System.Globalization.NumberStyles.HexNumber),
                UInt16.Parse(sequence.tag.Element, System.Globalization.NumberStyles.HexNumber));
            long start_pos = stream.Position;
            foreach (Object d in sequence)
            {
                if (d is NestedDataSetWriteData)
                {
                    WriteDataSetRecursive(stream, (NestedDataSetWriteData)d);
                }
                else if (d is SequenceWriteData)
                {
                    WriteSequenceRecursive(stream, (SequenceWriteData)d);
                }
                else if (d is DataElement)
                {
                    //write field
                    ((DataElement)d).SaveTo(stream);
                }
            }
            WriteLengthToStream(stream, length_pos, (uint)(stream.Position-start_pos));
        }

        /// <summary>
        /// Writes length of item to stream into defined positions. 
        /// The stream is set back original position after writing.
        /// Used for writing item length after items has been written.
        /// </summary>
        /// <param name="stream">stream</param>
        /// <param name="position"></param>
        /// <param name="length"></param>
        public static void WriteLengthToStream(Stream stream, long position, uint length)
        {
            long pos = stream.Position;
            BinaryWriter binarystream = new BinaryWriter(stream);
            stream.Seek(position, SeekOrigin.Begin);
            binarystream.Write(TransferSyntax.Default.CorrectByteOrdering(BitConverter.GetBytes(length)));
            binarystream.Flush();
            stream.Seek(pos, SeekOrigin.Begin);
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="stream">output stream</param>
        /// <param name="data">element length</param>
        /// <returns>position at data element length writing</returns>
        public static long WriteItemHeader(Stream stream, byte[] data)
        {
            Tag tag = new Tag(0xFFFE, 0xE000);
            tag.TransferSyntax = TransferSyntax.Default;
            tag.SaveTo(stream);
            BinaryWriter binarystream = new BinaryWriter(stream);
            long pos = stream.Position;
            binarystream.Write(data);
            binarystream.Flush();
            return pos;
        }
        /// <summary>
        /// Writes group 0x0002 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0002(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            // write the empty preamable 
            buf = new byte[128];
            stream.Write(buf, 0, buf.Length);
            
            // write the signature 
            buf = ToBytesArray("DICM".ToCharArray());
            stream.Write(buf, 0, 4);   

            // group 0x0002 
            WriteElement(stream, 0x0002, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = new byte[] { 0x01, 0x00};
            WriteElement(stream, 0x0002, 0x0001, buf);

            switch (header.modality)
            {
                case ImageModality.M_PT:
                    //Positron Emission Tomography Image Storage
                    buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.128".ToCharArray());
                    break;
                case ImageModality.M_CT:
                    //CT Image Storage
                    buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.2".ToCharArray());
                    break;
                case ImageModality.M_MR:
                    //CT Image Storage
                    buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.4".ToCharArray());
                    break;              
                default:
                    //Nuclear Medicine Image Storage
                    buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.20".ToCharArray());
                    break;
            }
            WriteElement(stream, 0x0002, 0x0002, buf);

            buf = ToBytesArray(("1.2.840.113619.2.99.120." + (Math.Abs(header.patientID.GetHashCode()) % 100) + "." + dicomheader.instance_nr.ToString()).ToCharArray());
            WriteElement(stream, 0x0002, 0x0003, buf);

            // transfer syntax 
            if (BitConverter.IsLittleEndian)
            {
                // Raw data, Implicit VR, Little Endian 
                buf = ToBytesArray("1.2.840.10008.1.2.1".ToCharArray());
                // JPEG compression, Lossy JPEG
                // 1.2.840.10008.1.2.4.50 
                // 1.2.840.10008.1.2.4.64 
                // JPEG compression, Lossless JPEG
                // 1.2.840.10008.1.2.4.65 
                // 1.2.840.10008.1.2.4.70 
                // Lossless Run Length Encoding 
                // 1.2.840.10008.1.2.5 
            }
            else
            {
                // Raw data, Eplicit VR, Big Endian 
                buf = ToBytesArray("1.2.840.10008.1.2.2".ToCharArray());
            }
            WriteElement(stream, 0x0002, 0x0010, buf);

            buf = ToBytesArray("1.2.840.113619.6.218".ToCharArray());
            //WriteElement(stream, 0x0002, 0x0012, buf);
            string str = "";
            while (str.Length+stream.Position-start_pos < 192-8)
                str += " ";
            buf = ToBytesArray(str.ToCharArray());
            //WriteElement(stream, 0x0002, 0x0013, buf);
            //buf = toBytesArray("ADW3".ToCharArray());
            //WriteElement(stream, 0x0002, 0x0016, buf);  

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0008 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0008(Stream stream)
        {
            byte[] buf;
            string str;
            long length_pos = 0;
            long start_pos = 0;

            //generate artificial studyID from patient name 
            int studyID = (int)(Math.Abs(header.patient_name.GetHashCode()) % 100000);

            WriteElement(stream, 0x0008, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = ToBytesArray(string.Format("ISO_IR 100").ToCharArray());
            WriteElement(stream, 0x0008, 0x0005, buf);
            buf = ToBytesArray(string.Format(@"DERIVED\PRIMARY").ToCharArray());
            WriteElement(stream, 0x0008, 0x0008, buf);  
            buf = ToBytesArray(string.Format(@"20080326").ToCharArray());
            WriteElement(stream, 0x0008, 0x0012, buf);
            buf = ToBytesArray(string.Format(@"092143.000").ToCharArray());
            WriteElement(stream, 0x0008, 0x0013, buf);
            buf = ToBytesArray(string.Format(@"1.2.840.113619.1.99.120").ToCharArray());
            WriteElement(stream, 0x0008, 0x0014, buf);

            switch (header.modality)
            {
                case ImageModality.M_PT:
                    buf = ToBytesArray(string.Format("1.2.840.10008.5.1.4.1.1.128").ToCharArray());
                    WriteElement(stream, 0x0008, 0x0016, buf);
                    break;
                default: // default to NM modality 
                    buf = ToBytesArray(string.Format("1.2.840.10008.5.1.4.1.1.20").ToCharArray());
                    WriteElement(stream, 0x0008, 0x0016, buf);
                    break;
            }

            buf = ToBytesArray(("0.0.0.0.3.0." + dicomheader.series_nr + ".0." + dicomheader.instance_nr.ToString()).ToCharArray());
            WriteElement(stream, 0x0008, 0x0018, buf);

            buf = ToBytesArray("20000101".ToCharArray());
            WriteElement(stream, 0x0008, 0x0020, buf);
            WriteElement(stream, 0x0008, 0x0021, buf);
            WriteElement(stream, 0x0008, 0x0022, buf);
            WriteElement(stream, 0x0008, 0x0023, buf);
            buf = ToBytesArray(("100000.00 ").ToCharArray());
            WriteElement(stream, 0x0008, 0x0030, buf);
            buf = ToBytesArray(("100000.00 ").ToCharArray());
            WriteElement(stream, 0x0008, 0x0031, buf);
            buf = ToBytesArray(("100000.00 ").ToCharArray());
            WriteElement(stream, 0x0008, 0x0032, buf);
            buf = ToBytesArray(("100000.00 ").ToCharArray());
            WriteElement(stream, 0x0008, 0x0033, buf);
            buf = ToBytesArray((filename.GetHashCode() % 1000));
            WriteElement(stream, 0x0008, 0x0050, buf);
            switch (header.modality)
            {
                case ImageModality.M_PT:
                    buf = ToBytesArray("PT".ToCharArray());
                    WriteElement(stream, 0x0008, 0x0060, buf);
                    break;
                case ImageModality.M_MR:
                    buf = ToBytesArray("MR".ToCharArray());
                    WriteElement(stream, 0x0008, 0x0060, buf);
                    break;
                case ImageModality.M_CT:
                    buf = ToBytesArray("CT".ToCharArray());
                    WriteElement(stream, 0x0008, 0x0060, buf);
                    break;
                default:
                    buf = ToBytesArray("NM".ToCharArray());
                    WriteElement(stream, 0x0008, 0x0060, buf);
                    break;
            }

            buf = ToBytesArray("Unknown".ToCharArray());
            WriteElement(stream, 0x0008, 0x0070, buf);
            WriteElement(stream, 0x0008, 0x0080, buf);
            WriteElement(stream, 0x0008, 0x0081, buf);
            WriteElement(stream, 0x0008, 0x0090, buf);
            
            if(header is DynamicImageHeader) {
                str = string.Format("{0:g}",Isotope.GetHalflife((header as DynamicImageHeader).isotope));
            } else {
                str = "0.00";
            }
            buf = ToBytesArray(str.ToCharArray());
            //WriteElement(stream, 0x0008, 0x0100, buf);           

            //station name
            buf = ToBytesArray("Unknown".ToCharArray());
            WriteElement(stream, 0x0008, 0x1010, buf);

            buf = ToBytesArray(header.description.ToCharArray());
            WriteElement(stream, 0x0008, 0x1030, buf);

            buf = ToBytesArray(header.description.ToCharArray());
            WriteElement(stream, 0x0008, 0x103E, buf);

            //institutional department name
            buf = ToBytesArray("Unknown".ToCharArray());
            WriteElement(stream, 0x0008, 0x1040, buf);
            WriteElement(stream, 0x0008, 0x1060, buf);
            WriteElement(stream, 0x0008, 0x1070, buf);
            WriteElement(stream, 0x0008, 0x1090, buf);

            buf = ToBytesArray(Version.GetVersionInformation().ToCharArray());
            WriteElement(stream, 0x0008, 0x2111, buf);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }

        /// <summary>
        /// Writes group 0x0009 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0009(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            WriteElement(stream, 0x0009, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = ToBytesArray(string.Format("GEMS_PETD_01").ToCharArray());
            WriteElement(stream, 0x0009, 0x0010, buf);
            buf = ToBytesArray(string.Format("GE Advance").ToCharArray());
            WriteElement(stream, 0x0009, 0x1001, buf);
            buf = ToBytesArray(string.Format("TurkuUni250339-0828").ToCharArray());
            WriteElement(stream, 0x0009, 0x1002, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x1003, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x1004, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x1005, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1006, buf);
            buf = ToBytesArray(string.Format("1.2.124.113532.193.143.211.18.20080318.160454.8870403").ToCharArray());
            WriteElement(stream, 0x0009, 0x1007, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x1008, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x1009, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205923105.104407").ToCharArray());
            WriteElement(stream, 0x0009, 0x100a, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x100b, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009, 0x100c, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x100d, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x100e, buf);
            buf = ToBytesArray(string.Format("3D AIVOT").ToCharArray());
            WriteElement(stream, 0x0009, 0x100f, buf);
            buf = ToBytesArray(string.Format("Turku PET Centre").ToCharArray());
            WriteElement(stream, 0x0009, 0x1010, buf);
            buf = ToBytesArray(string.Format("Advance").ToCharArray());
            WriteElement(stream, 0x0009, 0x1011, buf);
            buf = ToBytesArray(string.Format("GEMS").ToCharArray());
            WriteElement(stream, 0x0009, 0x1012, buf);
            buf = ToBytesArray(string.Format("Orbital Meatal Line").ToCharArray());
            WriteElement(stream, 0x0009, 0x1014, buf);
            buf = ToBytesArray(string.Format("OM").ToCharArray());
            WriteElement(stream, 0x0009, 0x1015, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1016, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1017, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1018, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x1019, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x101a, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x101b, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x101c, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x101d, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x101e, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x101f, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1020, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x1021, buf);
            buf = ToBytesArray((Int32)(2));
            WriteElement(stream, 0x0009, 0x1022, buf);
            buf = ToBytesArray((Int32)(34));
            WriteElement(stream, 0x0009, 0x1023, buf);
            buf = ToBytesArray((Int32)(1));
            WriteElement(stream, 0x0009, 0x1024, buf);
            buf = ToBytesArray((Int32)(2));
            WriteElement(stream, 0x0009, 0x1025, buf);
            buf = ToBytesArray((Int32)(11));
            WriteElement(stream, 0x0009, 0x1026, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1027, buf);
            buf = ToBytesArray((Int32)(2));
            WriteElement(stream, 0x0009, 0x1028, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1029, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x102a, buf);
            buf = ToBytesArray((Int32)(55));
            WriteElement(stream, 0x0009, 0x102b, buf);
            buf = ToBytesArray((Int32)(153));
            WriteElement(stream, 0x0009, 0x102c, buf);
            buf = ToBytesArray((Int32)(2));
            WriteElement(stream, 0x0009, 0x102d, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x102e, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1034, buf);
            buf = ToBytesArray((Int32)(0));
            WriteElement(stream, 0x0009, 0x1035, buf);
            buf = ToBytesArray(string.Format("PIB").ToCharArray());
            WriteElement(stream, 0x0009, 0x1036, buf);
            buf = ToBytesArray(string.Format("11C-PIB").ToCharArray());
            WriteElement(stream, 0x0009, 0x1037, buf);
            buf = ToBytesArray(575.0f);
            WriteElement(stream, 0x0009, 0x1038, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x1039, buf);
            buf = ToBytesArray(0.0f);
            WriteElement(stream, 0x0009, 0x103a, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x103b, buf);
            buf = ToBytesArray(3.0f);
            WriteElement(stream, 0x0009, 0x103c, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009, 0x103d, buf);

            buf = ToBytesArray(string.Format("11C ").ToCharArray());
            WriteElement(stream, 0x0009,0x103e, buf);
            buf = ToBytesArray(1218.0f);
            WriteElement(stream, 0x0009,0x103f, buf);                              
            buf = ToBytesArray((float)1);
            WriteElement(stream, 0x0009,0x1040, buf);
            buf = ToBytesArray((UInt32)0);
            WriteElement(stream, 0x0009,0x104d, buf);
            buf = ToBytesArray((Int32)(-12));
            WriteElement(stream, 0x0009,0x104e, buf);
            buf = ToBytesArray((Int32)11);
            WriteElement(stream, 0x0009,0x104f, buf);
            buf = ToBytesArray((Int32)(-6));
            WriteElement(stream, 0x0009,0x1050, buf);
            buf = ToBytesArray((Int32)6);
            WriteElement(stream, 0x0009,0x1051, buf);
            buf = ToBytesArray((Int32)56);
            WriteElement(stream, 0x0009,0x1052, buf);
            buf = ToBytesArray((Int32)800);
            WriteElement(stream, 0x0009,0x1053, buf);
            //upper energy limit
            buf = ToBytesArray((Int32)900);
            WriteElement(stream, 0x0009,0x1054, buf);
            //low energy limit
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1055, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199808979.324282").ToCharArray());
            WriteElement(stream, 0x0009,0x1056, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199777303.246822").ToCharArray());
            WriteElement(stream, 0x0009,0x1057, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205907722.526955").ToCharArray());
            WriteElement(stream, 0x0009,0x1058, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199862108.530411").ToCharArray());
            WriteElement(stream, 0x0009,0x1059, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x105a, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205924350.876255").ToCharArray());
            WriteElement(stream, 0x0009,0x105c, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205923105.104407").ToCharArray());
            WriteElement(stream, 0x0009,0x105d, buf);
            buf = ToBytesArray(string.Format("1.2.124.113532.193.143.211.18.20080318.160454.8870403").ToCharArray());
            WriteElement(stream, 0x0009,0x105e, buf);
            buf = ToBytesArray(string.Format("TurkuUni250339-0828").ToCharArray());
            WriteElement(stream, 0x0009,0x105f, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009,0x1060, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009,0x1061, buf);
            buf = ToBytesArray(string.Format("/dos/RMIIVLKO/SKKWAAQQ/NZMZQYSN/AAAAAAAA").ToCharArray());
            WriteElement(stream, 0x0009,0x1062, buf);
            buf = ToBytesArray((Int32)25212928);
            WriteElement(stream, 0x0009,0x1063, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1064, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1065, buf);
            buf = ToBytesArray((float)179);
            WriteElement(stream, 0x0009,0x1066, buf);
            buf = ToBytesArray((float)-129.5);
            WriteElement(stream, 0x0009,0x1067, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009,0x1068, buf);
            buf = ToBytesArray((Int32)265);
            WriteElement(stream, 0x0009,0x1069, buf);
            buf = ToBytesArray((float)-11);
            WriteElement(stream, 0x0009,0x106a, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x106b, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009,0x106c, buf);
            buf = ToBytesArray((Int32)600);
            WriteElement(stream, 0x0009,0x106d, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1070, buf);
            buf = ToBytesArray(double.Parse("1,68615e+07"));
            WriteElement(stream, 0x0009,0x1071, buf);
            buf = ToBytesArray(double.Parse("1,66362e+06"));
            WriteElement(stream, 0x0009,0x1072, buf);
            buf = ToBytesArray((Int32)1);
            WriteElement(stream, 0x0009,0x1073, buf);
            buf = ToBytesArray((Int32)207);
            WriteElement(stream, 0x0009,0x1074, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1075, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1076, buf);
            buf = ToBytesArray((Int32)25212928);
            WriteElement(stream, 0x0009,0x1077, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009,0x1079, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009,0x107a, buf);
            buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
            WriteElement(stream, 0x0009,0x107b, buf);
            buf = ToBytesArray((Int32)4);
            WriteElement(stream, 0x0009,0x107c, buf);
            buf = ToBytesArray((Int32)2);
            WriteElement(stream, 0x0009,0x107d, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x107e, buf);
            buf = ToBytesArray(string.Format(@"0\-0\-0 ").ToCharArray());
            WriteElement(stream, 0x0009,0x107f, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1080, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0009,0x1082, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1083, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x1084, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1085, buf);
            buf = ToBytesArray((float)1);
            WriteElement(stream, 0x0009,0x1086, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1087, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1088, buf);
            buf = ToBytesArray((Int32)1);
            WriteElement(stream, 0x0009,0x108b, buf);
            buf = ToBytesArray((Int32)7);
            WriteElement(stream, 0x0009,0x108c, buf);
            buf = ToBytesArray((float)0.096);
            WriteElement(stream, 0x0009,0x108d, buf);
            buf = ToBytesArray((Int32)3);
            WriteElement(stream, 0x0009,0x108e, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x1090, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x1091, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x1092, buf);
            buf = ToBytesArray((Int32)5);
            WriteElement(stream, 0x0009,0x1093, buf);
            buf = ToBytesArray((Int32)8);
            WriteElement(stream, 0x0009,0x1094, buf);
            buf = ToBytesArray((Int32)3);
            WriteElement(stream, 0x0009,0x1095, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199862108.530411").ToCharArray());
            WriteElement(stream, 0x0009,0x1096, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926124.324699").ToCharArray());
            WriteElement(stream, 0x0009,0x1097, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199808979.324282").ToCharArray());
            WriteElement(stream, 0x0009,0x1098, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205907722.526955").ToCharArray());
            WriteElement(stream, 0x0009,0x1099, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x109a, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x109b, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205925720.436393").ToCharArray());
            WriteElement(stream, 0x0009,0x109c, buf);
            buf = ToBytesArray((Int32)3);
            WriteElement(stream, 0x0009,0x109d, buf);
            buf = ToBytesArray((float)4.6875);
            WriteElement(stream, 0x0009,0x109e, buf);
            buf = ToBytesArray((Int32)1);
            WriteElement(stream, 0x0009,0x109f, buf);
            buf = ToBytesArray((float)6.4);
            WriteElement(stream, 0x0009,0x10a0, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x10a1, buf);
            //axial spacing
            buf = ToBytesArray((float)header.siz.sizez);
            WriteElement(stream, 0x0009,0x10a2, buf);
            buf = ToBytesArray((Int32)11);
            WriteElement(stream, 0x0009,0x10a3, buf);
            buf = ToBytesArray(string.Format("02.00").ToCharArray());
            WriteElement(stream, 0x0009,0x10a4, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0009,0x10a5, buf);
            buf = ToBytesArray((Int32)(header.dimz + 1 - dicomheader.instance_nr));
            WriteElement(stream, 0x0009,0x10a6, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x10a9, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x10aa, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x10ab, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x10ac, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926481.123992").ToCharArray());
            WriteElement(stream, 0x0009,0x10ad, buf);
            buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926032.808059").ToCharArray());
            WriteElement(stream, 0x0009,0x10ae, buf);
            buf = ToBytesArray((float)0);
            WriteElement(stream, 0x0009,0x10b9, buf);
            buf = ToBytesArray((Int32)3);
            WriteElement(stream, 0x0009,0x10be, buf);
            buf = ToBytesArray((float)8.6);
            WriteElement(stream, 0x0009,0x10bf, buf);
            buf = ToBytesArray((Int32)1);
            WriteElement(stream, 0x0009,0x10c1, buf);
            buf = ToBytesArray((float)4);
            WriteElement(stream, 0x0009,0x10c2, buf);
            buf = ToBytesArray((float)1.054591);
            WriteElement(stream, 0x0009,0x10c6, buf);
            buf = ToBytesArray((Int32)0);
            WriteElement(stream, 0x0009,0x10c7, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0009,0x10c8, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0009,0x10c9, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0009,0x10ca, buf);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0010 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        public virtual void Write0010(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            WriteElement(stream, 0x0010, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = ToBytesArray(header.patient_name.ToCharArray());
            WriteElement(stream, 0x0010, 0x0010, buf);
            buf = ToBytesArray(header.patientID.ToCharArray());
            WriteElement(stream, 0x0010, 0x0020, buf);
            buf = ToBytesArray("20000101".ToCharArray());
            WriteElement(stream, 0x0010, 0x0030, buf);
            buf = ToBytesArray("000000".ToCharArray());
            WriteElement(stream, 0x0010, 0x0032, buf);
            buf = ToBytesArray("M".ToCharArray());
            WriteElement(stream, 0x0010, 0x0040, buf);
            buf = ToBytesArray("100Y".ToCharArray());
            WriteElement(stream, 0x0010, 0x1010, buf);
            buf = ToBytesArray("1.00".ToCharArray());
            WriteElement(stream, 0x0010, 0x1020, buf);
            buf = ToBytesArray("100 ".ToCharArray());
            WriteElement(stream, 0x0010, 0x1030, buf);
            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0011 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0011(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            WriteElement(stream, 0x0011, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = ToBytesArray("GEMS_PETD_01".ToCharArray());
            WriteElement(stream, 0x0011, 0x0010, buf);            
            SequenceWriteData sequence = new SequenceWriteData();
            sequence.SetTag(0x0011, 0x1001);
                NestedDataSetWriteData nested = new NestedDataSetWriteData();
                sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0013 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0013(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            WriteElement(stream, 0x0013, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = ToBytesArray("GEMS_PETD_01".ToCharArray());
            WriteElement(stream, 0x0013, 0x0010, buf);            
            SequenceWriteData sequence = new SequenceWriteData();
            sequence.SetTag(0x0013, 0x1001);
                NestedDataSetWriteData nested = new NestedDataSetWriteData();
                sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }

        /// <summary>
        /// Writes group 0x0018 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        public virtual void Write0018(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;
            WriteElement(stream, 0x0018, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;
            //radiopharmaceutical name
            if ((header is DynamicImageHeader))
            {
                buf = ToBytesArray((header as DynamicImageHeader).radiopharma.ToCharArray());
                WriteElement(stream, 0x0018, 0x0031, buf);
            }
            buf = ToBytesArray(header.sizez.ToString().Replace(',','.').ToCharArray());
            WriteElement(stream, 0x0018, 0x0050, buf);
            buf = ToBytesArray(string.Format("TIME").ToCharArray());
            WriteElement(stream, 0x0018, 0x0071, buf);
            buf = ToBytesArray(string.Format("MANU").ToCharArray());
            WriteElement(stream, 0x0018, 0x0073, buf);
            buf = ToBytesArray(string.Format("0").ToCharArray());
            WriteElement(stream, 0x0018, 0x0074, buf);
            buf = ToBytesArray(string.Format("0").ToCharArray());
            WriteElement(stream, 0x0018, 0x0075, buf);
            buf = ToBytesArray(string.Format("06.00").ToCharArray());
            WriteElement(stream, 0x0018, 0x1020, buf);
            buf = ToBytesArray(string.Format("").ToCharArray());
            WriteElement(stream, 0x0018, 0x1063, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0018, 0x1081, buf);
            buf = new byte[0];
            WriteElement(stream, 0x0018, 0x1082, buf);
            buf = ToBytesArray(string.Format("0").ToCharArray());
            WriteElement(stream, 0x0018, 0x1083, buf);
            buf = ToBytesArray(string.Format("0").ToCharArray());
            WriteElement(stream, 0x0018, 0x1084, buf);
            //calculate correct value from voxel size and dimension
            buf = ToBytesArray((((int)Math.Round(header.dimx*header.sizex)).ToString()+" ").ToCharArray());
            WriteElement(stream, 0x0018, 0x1100, buf);
            buf = ToBytesArray(string.Format("-0").ToCharArray());
            WriteElement(stream, 0x0018, 0x1120, buf);
            buf = ToBytesArray(string.Format("CYLINDRICAL RING").ToCharArray());
            WriteElement(stream, 0x0018, 0x1147, buf);
            //calculate value from header dimensions
            buf = ToBytesArray(( ((int)Math.Round(header.dimx * header.sizex)).ToString() + "\\"+
                                 ((int)Math.Round(header.dimz * header.sizez)).ToString() ).ToCharArray());
            WriteElement(stream, 0x0018, 0x1149, buf);
            buf = ToBytesArray(string.Format("NONE").ToCharArray());
            WriteElement(stream, 0x0018, 0x1181, buf);
            //some default string because it is unlikely that this string is rea by software
            buf = ToBytesArray(string.Format(@"Rad:\ hanning\  0.000000 mm\ Ax:\ rectangle\  0.000000 mm").ToCharArray());
            WriteElement(stream, 0x0018, 0x1210, buf);
            if(image is DynamicImage) {
                buf = ToBytesArray(((int)(image as DynamicImage).GetFrameDuration(0)).ToString().ToCharArray());
            } else {
                buf = ToBytesArray(string.Format("0").ToCharArray());
            }
            WriteElement(stream, 0x0018, 0x1242, buf);
            buf = ToBytesArray(string.Format(dicomheader.orientation).ToCharArray());
            WriteElement(stream, 0x0018, 0x5100, buf);
            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }

        /// <summary>
        /// Writes group 0x0020 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0020(Stream stream)
        {
            byte[] buf;
            string str;
            long length_pos = 0;
            long start_pos = 0;
            WriteElement(stream, 0x0020, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            //generate artificial studyID from patient name 
            int studyID = (int)(Math.Abs(header.patient_name.GetHashCode()) % 100000);

            // Study instance UID 
            buf = ToBytesArray(("0.0.0.0.3." + studyID + "." + dicomheader.series_nr + ".0.0").ToCharArray());
            WriteElement(stream, 0x0020, 0x000D, buf);
            // Series instance UID, effects Vinci 
            WriteElement(stream, 0x0020, 0x000E, buf);

            buf = ToBytesArray("1.9." + dicomheader.series_nr.ToString());
            WriteElement(stream, 0x0020, 0x0010, buf);
            //calculate own short hascode, may not be unique within filenames
            buf = ToBytesArray(studyID.ToString().ToCharArray());
            WriteElement(stream, 0x0020, 0x0011, buf);

            //modified because Vinci does not convert properly:
            //Vinci sorts files according to ASCII codes, which results
            //missorted slices 
            str = dicomheader.instance_nr.ToString();
            str.PadLeft(2, '0');
            if (str.Length > 2) str = str.Remove(0, str.Length - 2);
            buf = ToBytesArray(str.ToCharArray());
            WriteElement(stream, 0x0020, 0x0013, buf);

            // image pos patient 
            buf = ToBytesArray((
                  (-header.siz.sizex / 2.0f).ToString("0.00").Replace(',', '.') + "\\" +
                  (-header.siz.sizey / 2.0f).ToString("0.00").Replace(',', '.') + "\\" +
                  (header.siz.sizez * dicomheader.instance_nr).ToString("0.00").Replace(',', '.')
                ).ToCharArray());
            WriteElement(stream, 0x0020, 0x0032, buf);
            
            // image orient patient 
            buf = ToBytesArray("1.00\\0.00\\0.00\\0.00\\1.00\\0.00 ".ToCharArray());
            WriteElement(stream, 0x0020, 0x0037, buf);

            buf = ToBytesArray(string.Format("777.777.0.0.0.{0:D}.{1:D}", dicomheader.instance_nr, dicomheader.series_nr).ToCharArray());
            WriteElement(stream, 0x0020, 0x0052, buf);

            buf = ToBytesArray("Orbital Meatal Line".ToCharArray());
            WriteElement(stream, 0x0020, 0x1040, buf);

            buf = ToBytesArray((dicomheader.instance_nr * header.siz.sizez).ToString("0.00").Replace(',', '.').ToCharArray());
            WriteElement(stream, 0x0020, 0x1041, buf);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0028 of DICOM header
        /// </summary>
        /// <param name="stream">stream</param>
        protected virtual void Write0028(Stream stream)
        {
            byte[] buf;
            string str;

            long length_pos = 0;
            long start_pos = 0;
            WriteElement(stream, 0x0028, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;


            //samples per pixel
            buf = BitConverter.GetBytes((UInt16)1);
            WriteElement(stream, 0x0028, 0x0002, buf);

            buf = ToBytesArray("MONOCHROME2".ToCharArray());
            WriteElement(stream, 0x0028, 0x0004, buf);

            //number of frames
            if (image is DynamicImage) {
                str = (image as DynamicImage).frames.ToString();
                str = str.PadLeft(2, '0');
                if (str.Length > 2) str.Substring(str.Length - 2, 2); 
                buf = ToBytesArray(str.ToCharArray());
            } else {
                buf = ToBytesArray("01".ToCharArray());
            }
            WriteElement(stream, 0x0028, 0x0008, buf);

            switch (header.modality)
            {
                case ImageModality.M_PT:
                    switch (this.dicomheader.acquisitiontype)
                    {
                        case AcquisitionType.ACQUISITION_DYNAMIC:
                            buf = new byte[]{0x00, 0x54, 0x00, 0x80, 0x00, 0x54, 0x01, 0x00};
                            break;
                        case AcquisitionType.ACQUISITION_GATED:
                        case AcquisitionType.ACQUISITION_GSPECT:
                        case AcquisitionType.ACQUISITION_STATIC:
                        case AcquisitionType.ACQUISITION_TOMO:
                        case AcquisitionType.ACQUISITION_UNKNOWN:
                        default:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00 };
                            break;
                    }
                    break;
                case ImageModality.M_MR:
                default: // default to NM modality 
                    switch (this.dicomheader.acquisitiontype)
                    {
                        case AcquisitionType.ACQUISITION_DYNAMIC:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00 };
                            break;
                        case AcquisitionType.ACQUISITION_TOMO:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 
                                               0x00, 0x54, 0x00, 0x20, 
                                               0x00, 0x54, 0x00, 0x30, 
                                               0x00, 0x54, 0x01, 0x00 }; 
                            break;
                        case AcquisitionType.ACQUISITION_GATED:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 
                                               0x00, 0x54, 0x00, 0x20, 
                                               0x00, 0x54, 0x00, 0x60, 
                                               0x00, 0x54, 0x00, 0x70 }; 
                            break;
                        case AcquisitionType.ACQUISITION_GSPECT:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x60, 
                                               0x00, 0x54, 0x00, 0x70, 
                                               0x00, 0x54, 0x00, 0x80 };
                            break;
                        case AcquisitionType.ACQUISITION_UNKNOWN:
                        case AcquisitionType.ACQUISITION_STATIC:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 0x00, 0x54, 0x00, 0x20 };
                            break;
                    }
                    break;
            }
            //WriteElement(stream, 0x0028, 0x0009, buf);

            //rows
            buf = BitConverter.GetBytes((UInt16)image.height);
            WriteElement(stream, 0x0028, 0x0010, buf);
            //columns
            buf = BitConverter.GetBytes((UInt16)image.width);
            WriteElement(stream, 0x0028, 0x0011, buf);
            //planes
            buf = BitConverter.GetBytes((UInt16)image.planes);
            WriteElement(stream, 0x0028, 0x0012, buf);
            //pixel spacing, (required by Vinci 2.37.0)
            str = (header.sizex.ToString().Replace(',', '.') + "\\" + header.sizey.ToString().Replace(',', '.'));
            if(str.Length % 2 != 0) str += " ";
            buf = ToBytesArray(str.ToCharArray());
            WriteElement(stream, 0x0028, 0x0030, buf);

            buf = ToBytesArray(@"DECY\ATTN\SCAT\DTIM\RAN\RADL\DCAL\SLSENS\NORM\BLANK\NLOG".ToCharArray());
            WriteElement(stream, 0x0028, 0x0051, buf);
            
            switch (header.datatype)
            {
                case DataType.BIT8_U:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)8));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)8));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)7));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
                    break;
                case DataType.BIT8_S:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)8));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)8));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)7));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
                    break;
                case DataType.BIT16_S:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)16));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)16));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)15));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
                    break;
                case DataType.BIT16_U:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)16));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)16));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)15));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
                    break;
                case DataType.BIT32_S:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)31));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
                    break;
                case DataType.BIT32_U:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)31));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
                    break;
                case DataType.BIT64_S:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)64));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)64));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)63));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
                    break;
                case DataType.BIT64_U:
                    //bits allocated, bits stored, high bit
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)64));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)64));
                    WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)63));
                    WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
                    break;
                case DataType.FLT32:
                    //bits allocated, bits stored
                    WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
                    WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
                    break;
                default:
                    throw new TPCDicomFileException("Unsupported write data type:" + header.datatype);
            }
            
            buf = ToBytesArray((Int16)(-5724));
            //WriteElement(stream, 0x0028, 0x0106, buf);
            buf = ToBytesArray((Int16)(29669));
            //WriteElement(stream, 0x0028, 0x0107, buf);

            //one or more of these fields are critical for CortexID in ADW workstation
            buf = ToBytesArray("730.30639648438 ".ToCharArray());
            WriteElement(stream, 0x0028, 0x1050, buf);
            buf = ToBytesArray("2104.6850585938 ".ToCharArray());
            WriteElement(stream, 0x0028, 0x1051, buf);
            buf = ToBytesArray("0 ".ToCharArray());
            WriteElement(stream, 0x0028, 0x1052, buf);
            str = dicomheader.scalefactor.ToString("0.0000000").Replace(',', '.');
            if(str.Length % 2 != 0) str += " ";
            buf = ToBytesArray(str.ToCharArray());
            WriteElement(stream, 0x0028, 0x1053, buf);
            buf = ToBytesArray("00".ToCharArray());
            WriteElement(stream, 0x0028, 0x2110, buf);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x0054 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected virtual void Write0054(Stream stream)
        {    

            byte[] buf; 
            string str;
            long length_pos = 0;
            long start_pos = 0;
            SequenceWriteData sequence;
            NestedDataSetWriteData nested;
            SequenceWriteData sequence2;
            NestedDataSetWriteData nested2;

            WriteElement(stream, 0x0054, 0x0000, new byte[] {0x00, 0x00, 0x00, 0x00});
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            sequence = new SequenceWriteData();
            sequence.SetTag(0x0054, 0x0013);
                nested = new NestedDataSetWriteData();
                    buf = ToBytesArray("000000000000300".ToCharArray());
                    nested.Add(CreateDataElement(0x0054, 0x0014, buf));
                    buf = ToBytesArray("000000000000650".ToCharArray());
                    nested.Add(CreateDataElement(0x0054, 0x0015, buf));
                sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);

            sequence = new SequenceWriteData();
            sequence.SetTag(0x0054, 0x0016);
                nested = new NestedDataSetWriteData();
                    buf = ToBytesArray("UNKNOWN".ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x0031, buf));
                    buf = ToBytesArray("0".ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x1071, buf));
                    if(header is DynamicImageHeader) {
                        System.DateTime dosetime = (header as DynamicImageHeader).dose_start_time;
                        str = dosetime.ToString("HHmmss.fff");
                    } else {
                        str = "000000.000";
                    }
                    buf = ToBytesArray(str.ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x1072, buf));
                    if(header is DynamicImageHeader) {
                        try
                        {
                            str = ((ulong)((header as DynamicImageHeader).injected_dose * 1000000)).ToString("000000000");
                        } catch(ArithmeticException) {
                            str = "100000000";
                        }
                    } else {
                        str = "100000000";
                    }
                    buf = ToBytesArray(str.ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x1074, buf));
                    buf = ToBytesArray("1218".ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x1075, buf));
                    buf = ToBytesArray("1".ToCharArray());
                    nested.Add(CreateDataElement(0x0018, 0x1076, buf));
                    sequence2 = new SequenceWriteData();
                    sequence2.SetTag(0x0054, 0x0300);
                        nested2 = new NestedDataSetWriteData();
                            buf = ToBytesArray("Y-X0000".ToCharArray());
                            nested2.Add(CreateDataElement(0x0008, 0x0100, buf));
                            buf = ToBytesArray("00SDM".ToCharArray());
                            nested2.Add(CreateDataElement(0x0008, 0x0102, buf));
                            if(header is DynamicImageHeader)
                                buf = ToBytesArray((header as DynamicImageHeader).isotope.ToString());
                            else
                                buf = ToBytesArray("11C".ToCharArray());
                            nested2.Add(CreateDataElement(0x0008, 0x0104, buf));
                    sequence2.Add(nested2);
                    nested.Add(sequence2);
                    sequence2 = new SequenceWriteData();
                    sequence2.SetTag(0x0054, 0x0304);
                        nested2 = new NestedDataSetWriteData();
                            buf = ToBytesArray("00SDM".ToCharArray());
                            nested2.Add(CreateDataElement(0x0008, 0x0102, buf));
                            buf = ToBytesArray("UNKNOWN".ToCharArray());
                            nested2.Add(CreateDataElement(0x0008, 0x0104, buf));
                    sequence2.Add(nested2);
                    nested.Add(sequence2);
                sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);

            buf = ToBytesArray((UInt16)header.dimz);
            WriteElement(stream, 0x0054, 0x0081, buf);
            if(header.dim.Length > IntLimits.FRAMES) {
                buf = ToBytesArray((UInt16)header.dim.GetDimension(IntLimits.FRAMES));
            } else {
                buf = ToBytesArray((UInt16)1);
            }
            WriteElement(stream, 0x0054, 0x0101, buf);
            buf = ToBytesArray("NONE".ToCharArray());
            WriteElement(stream, 0x0054, 0x0202, buf);
            sequence = new SequenceWriteData();
            sequence.SetTag(0x0054, 0x0410);
                nested = new NestedDataSetWriteData();
            sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);
            sequence = new SequenceWriteData();
            sequence.SetTag(0x0054, 0x0414);
            nested = new NestedDataSetWriteData();
            sequence.Add(nested);
            WriteSequenceRecursive(stream, sequence);
            buf = ToBytesArray(@"DYNAMIC\IMAGE".ToCharArray());
            WriteElement(stream, 0x0054, 0x1000, buf);
            buf = ToBytesArray(UnitConverter.ConvertToString(header.dataunit).ToCharArray());
            WriteElement(stream, 0x0054, 0x1001, buf);
            buf = ToBytesArray("EMISSION".ToCharArray());
            WriteElement(stream, 0x0054, 0x1002, buf);
            buf = ToBytesArray("RTSUB".ToCharArray());
            WriteElement(stream, 0x0054, 0x1100, buf);
            buf = ToBytesArray("START".ToCharArray());
            WriteElement(stream, 0x0054, 0x1102, buf);    
            buf = ToBytesArray("3D Kinahan - Rogers".ToCharArray());
            WriteElement(stream, 0x0054, 0x1103, buf);
            buf = ToBytesArray("11".ToCharArray());
            WriteElement(stream, 0x0054, 0x1104, buf);
            buf = ToBytesArray("Gaussian Fit".ToCharArray());
            WriteElement(stream, 0x0054, 0x1105, buf);
            buf = ToBytesArray(@"1\2".ToCharArray());
            WriteElement(stream, 0x0054, 0x1201, buf);
            buf = ToBytesArray("1".ToCharArray());
            WriteElement(stream, 0x0054, 0x1202, buf);
            buf = ToBytesArray("12".ToCharArray());
            WriteElement(stream, 0x0054, 0x1210, buf);
            if (image is DynamicImage)
            {
                buf = ToBytesArray((image as DynamicImage).GetFrameStartTime(0).ToString().ToCharArray());
            }
            else
            {
                buf = ToBytesArray("0".ToCharArray());
            }
            WriteElement(stream, 0x0054, 0x1300, buf);
            buf = ToBytesArray("1".ToCharArray());
            WriteElement(stream, 0x0054, 0x1320, buf);
            buf = ToBytesArray("1.18042".ToCharArray());
            WriteElement(stream, 0x0054, 0x1321, buf);
            buf = ToBytesArray("         12518".ToCharArray());
            WriteElement(stream, 0x0054, 0x1322, buf);
            buf = ToBytesArray("1.04029".ToCharArray());
            WriteElement(stream, 0x0054, 0x1324, buf);
            buf = ToBytesArray((UInt16)dicomheader.instance_nr);
            WriteElement(stream, 0x0054, 0x1330, buf);

            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }
        /// <summary>
        /// Writes group 0x7FE0 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected void Write7FE0(Stream stream)
        {

            byte[] buf;
            int j = 0;
            Tag tag;

            long length_pos = 0;
            long start_pos = 0;

            WriteElement(stream, 0x7FE0, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            //data
            tag = new Tag(0x7FE0, 0x0010);
            tag.TransferSyntax = TransferSyntax.Default;
            BinaryWriter binarystream = new BinaryWriter(stream);
            int bytes = image.dataLength * ImageFile.Type2Bytes(header.datatype);
            //write tag
            tag.SaveTo(stream);
            // "OW"
            binarystream.Write("OW".ToCharArray(), 0, 2);
            binarystream.Write(new byte[2] { 0x00, 0x00 }, 0, 2);
            //data element length
            binarystream.Write(BitConverter.GetBytes((UInt32)bytes));

            if (dicomheader.scalefactor == 0) throw new TPCDicomFileException("Scale factor is zero.");
            int size = 0;
            for (j = 0; j < image.dataLength; j++)
            {
                switch (header.datatype)
                {
                    case DataType.BIT8_U:
                        byte uint8value = (byte)(image[j]);
                        buf = new byte[] { uint8value };
                        size += buf.Length;
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT8_S:
                        sbyte int8value = (sbyte)(image[j]);
                        buf = new byte[] { (byte)int8value };
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT16_S:
                        Int16 int16value = (Int16)(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(int16value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT16_U:
                        UInt16 uint16value = (UInt16)Math.Round(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(uint16value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT32_S:
                        Int32 int32value = (Int32)Math.Round(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(int32value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT32_U:
                        UInt32 uint32value = (UInt32)Math.Round(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(uint32value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT64_S:
                        Int64 int64value = (Int64)Math.Round(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(int64value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.BIT64_U:
                        UInt64 uint64value = (UInt64)Math.Round(image[j] / dicomheader.scalefactor);
                        buf = BitConverter.GetBytes(uint64value);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    case DataType.FLT32:
                        buf = BitConverter.GetBytes(image[j] / dicomheader.scalefactor);
                        binarystream.Write(tag.TransferSyntax.CorrectByteOrdering(buf));
                        break;
                    default:
                        throw new TPCDicomFileException("Unsupported write data type:" + header.datatype);
                }
            }
            WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
            binarystream.Flush();
        }

        /// <summary>
        /// Stacks dicomfiles containing planes, frames and/or gates into one DICOM image. Planes are ordered.
        /// Filename is set to "stacked.dcm".
        /// </summary>
        /// <param name="dicomfiles">array of images</param>
        /// <returns>new DicomFile containing all files</returns>
        /// <exception cref="TPCInvalidArgumentsException">if data is inconsistent</exception>
        public static DicomFile Stack(DicomFile[] dicomfiles) {
            int g = 0;
            int f = 0;
            int p = 0;
            int file_i = 0;
            DicomFile r;
            int frames_total;
            IOProcessEventHandler pevent = null;

            if (!(dicomfiles is DicomFile[])) throw new TPCInvalidArgumentsException("No input array.");
            if (dicomfiles.Length == 0) throw new TPCInvalidArgumentsException("No files.");

            Array.Sort<DicomFile>(dicomfiles, new DicomFile.DicomFileComparer());
            DicomFile[] gates = new DicomFile[DicomFile.ResolveNoGates(dicomfiles)];
            DicomFile[] frames = new DicomFile[DicomFile.ResolveNoFrames(dicomfiles)];
            DicomFile[] planes = new DicomFile[DicomFile.ResolveNoPlanes(dicomfiles)];

            frames_total = gates.Length * frames.Length;
            if (gates.Length * frames.Length * planes.Length != dicomfiles.Length)
                throw new TPCInvalidArgumentsException("Dimension information [gates="+gates.Length+" frames="+frames.Length+" planes="+planes.Length+" is inconsistent with number of input files " + dicomfiles.Length + ".");
            for (g = 0; g < gates.Length; g++)
            {
                for (f = 0; f < frames.Length; f++)
                {
                    for (p = 0; p < planes.Length; p++)
                    {
                        planes[p] = dicomfiles[file_i++];
                    }
                    frames[f] = DicomFile.StackDimension(planes, Limits.PLANES);
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(frames[f], new IOProgressEventArgs(100.0f * (float)((g + 1) * (f + 1)) / frames_total, System.Reflection.MethodBase.GetCurrentMethod()));
                }
                gates[g] = DicomFile.StackDimension(frames, Limits.FRAMES);
            }
            r = DicomFile.StackDimension(gates, Limits.GATES);

            //copy all general header data
            r.header.Clear();
            System.ComponentModel.TypeConverter t = new System.ComponentModel.TypeConverter();
            for (int i = 0; i < dicomfiles[0].header.Count; i++)
            {
                r.header.Add(dicomfiles[0].header[i].Key, dicomfiles[0].header[i].Value);
            }
            
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(r, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
            return r;
        }
        /// <summary>
        /// Comparison class for dicom header sorting. 
        /// </summary>
        public class DicomHeaderComparer : IComparer<DicomHeader>
        {
            /// <summary>
            /// Sorting according to:
            /// group name
            /// frame start times if files have dynamic images
            /// plane z positions
            /// <see cref="DynamicImage"/>
            /// </summary>
            /// <param name="x">first evaluated file</param>
            /// <param name="y">second evaluated file</param>
            /// <returns></returns>
            public int Compare(DicomHeader x, DicomHeader y)
            {
                //sort null objects as 'smaller' than existing ones
                if (x == null)
                {
                    if (y == null) return 0;
                    else return 1;
                }
                else if (y == null) return -1;
                //group with series number
                if (x.series_nr < y.series_nr) return -1;
                if (x.series_nr > y.series_nr) return 1;
                //use gate numbers if available
                if (x.gate_number < y.gate_number) return -1;
                if (x.gate_number > y.gate_number) return 1;

                //use frame start times if available
                if (x.slice_start < y.slice_start) return -1;
                if (x.slice_start > y.slice_start) return 1;

                if (x.slice_location < y.slice_location) return -1;
                if (x.slice_location > y.slice_location) return 1;

                if (x.instance_nr < y.instance_nr) return -1;
                if (x.instance_nr > y.instance_nr) return 1;

                //couldn't find any sorting difference
                return 0;
            }
        }
        /// <summary>
        /// Comparison class for dicom file sorting. 
        /// </summary>
        public class DicomFileComparer : IComparer<DicomFile> {
            /// <summary>
            /// Sorting according to:
            /// group name
            /// frame start times if files have dynamic images
            /// plane z positions
            /// <see cref="DynamicImage"/>
            /// </summary>
            /// <param name="x">first evaluated file</param>
            /// <param name="y">second evaluated file</param>
            /// <returns></returns>
            public int Compare(DicomFile x, DicomFile y)
            {
                //sort null objects as 'smaller' than existing ones
                if (x == null)
                {
                    if (y == null) return 0;
                    else return 1;
                }
                else if (y == null) return -1;
                //group with series number
                if (x.dicomheader.series_nr < y.dicomheader.series_nr) return -1;
                if (x.dicomheader.series_nr > y.dicomheader.series_nr) return 1;
                //use gate numbers if available
                if (x.dicomheader.gate_number < y.dicomheader.gate_number) return -1;
                if (x.dicomheader.gate_number > y.dicomheader.gate_number) return 1;

                //use frame start times if available
                if (x.image is DynamicImage && y.image is DynamicImage)
                {
                    if ((x.image as DynamicImage).GetFrameStartTimes()[0] < (y.image as DynamicImage).GetFrameStartTimes()[0]) return -1;
                    else if ((x.image as DynamicImage).GetFrameStartTimes()[0] > (y.image as DynamicImage).GetFrameStartTimes()[0]) return 1; 
                } else {
                    if (x.dicomheader.slice_start < y.dicomheader.slice_start) return -1;
                    if (x.dicomheader.slice_start > y.dicomheader.slice_start) return 1;
                }

                if (x.dicomheader.slice_location < y.dicomheader.slice_location) return -1;
                if (x.dicomheader.slice_location > y.dicomheader.slice_location) return 1;

                if (x.dicomheader.instance_nr < y.dicomheader.instance_nr) return -1;
                if (x.dicomheader.instance_nr > y.dicomheader.instance_nr) return 1;

                //couldn't find any sorting difference
                return 0;
            }
        }
        /// <summary>
        /// Stacks dicomfiles into one DICOM image according to dimension. Images are sorted.
        /// Filename is set to "stacked.dcm". Result header information is copied from first 
        /// file.
        /// </summary>
        /// <see cref="DicomFile"/>
        /// <param name="dicomfiles">array of images (non-null)</param>
        /// <param name="dimension">dimension that is stacked (0-based)</param>
        /// <returns>new DicomFile object containing stacked data</returns>
        /// <exception cref="TPCInvalidArgumentsException">if data is inconsistent, unsupported stack dimension</exception>
        public static DicomFile StackDimension(DicomFile[] dicomfiles, int dimension)
        {
            DicomFile r;
            Voxel siz;
            IntLimits dim;
            float[] mm = new float[2];
            
            if (dicomfiles.Length < 1) throw new TPCInvalidArgumentsException("No DICOM files to stack.");
            if (dimension < IntLimits.PLANES || dimension > IntLimits.GATES) 
                throw new TPCInvalidArgumentsException("Unsupported dimension for stacking:"+dimension);

            //ensure consistensy
            for (int i = 0; i < dicomfiles.Length; i++)
            {
                if (dicomfiles[i] == null) throw new TPCInvalidArgumentsException("null found at index " + i + ".");
                if (dicomfiles[i].header == null) throw new TPCInvalidArgumentsException("no header at index " + i + ".");
                if (dicomfiles[i].header.siz != dicomfiles[0].header.siz) throw new TPCInvalidArgumentsException("Inconsistent gate voxel sizes.");
                if (dicomfiles[i].image == null) throw new TPCInvalidArgumentsException("no image at index " + i + ".");
                if (dicomfiles[i].image.dim != dicomfiles[0].image.dim) throw new TPCInvalidArgumentsException("Inconsistent dimensions.");
                if ((dicomfiles[i].image is DynamicImage) != (dicomfiles[0].image is DynamicImage)) 
                    throw new TPCInvalidArgumentsException("Tried to stack dynamic and static images together.");
            }

            //sort dicomfiles
            Array.Sort<DicomFile>(dicomfiles, new TPClib.Image.DicomFile.DicomFileComparer());

            r = new DicomFile("stacked.dcm");
            siz = new Voxel(dicomfiles[0].header.siz);
            dim = new IntLimits(dicomfiles[0].image.dim);
            switch(dimension) {
                case Limits.PLANES: r.dicomheader.slice_location = dicomfiles[0].dicomheader.slice_location; break;
                case Limits.FRAMES: r.dicomheader.slice_start = dicomfiles[0].dicomheader.slice_start; break;
                case Limits.GATES: r.dicomheader.gate_number = dicomfiles[0].dicomheader.gate_number; break;
            }

            //set result dimensions to comply with stacking dimension
            while(dim.Length <= dimension) dim.AddLimit(0, 1);
            dim.SetLimit(dimension, Limits.Limit.LOW, 0);
            dim.SetLimit(dimension, Limits.Limit.HIGH, dicomfiles.Length);
            if (dicomfiles[0].header is DynamicImageHeader)
            {
                r.image = new DynamicImage(dim);
                if (!(r.header is DynamicImageHeader)) r.header = new DynamicImageHeader(r.header);
                (r.header as DynamicImageHeader).isotope = (dicomfiles[0].header as DynamicImageHeader).isotope;
                (r.header as DynamicImageHeader).radiopharma = (dicomfiles[0].header as DynamicImageHeader).radiopharma;
            } else {
                r.image = new Image(dim);
            }
            r.header.dim = new IntLimits(dim);
            r.header.siz = new Voxel(siz);
            r.header.datatype = dicomfiles[0].header.datatype;
            r.header.description = dicomfiles[0].header.description;
            r.header.patient_name = dicomfiles[0].header.patient_name;
            r.header.patientID = dicomfiles[0].header.patientID;

            //gather image data
            if (dicomfiles.Length > 1)
            {
                for (int i = 0; i < dicomfiles.Length; i++)
                {
                    dim = r.image.GetDimensionLimits(i, dimension);
                    //upgrade set region to target dimensions
                    while (dim.Length < r.image.dim.Length) dim.AddLimit(0, 1);
                    r.image.SetSubImage(dim, ref dicomfiles[i].image);
                }
                //take only first frame information even if files contain multiple frames, this may need update
                if (r.image is DynamicImage)
                {
                    for (int i = 0; i < dicomfiles.Length; i++)
                    {
                        if (dimension == Limits.FRAMES)
                        {
                            (r.image as DynamicImage).SetFrameStartTime(i, (dicomfiles[i].image as DynamicImage).GetFrameStartTime(0));
                            (r.image as DynamicImage).SetFrameDuration(i, (dicomfiles[i].image as DynamicImage).GetFrameDuration(0));
                        }
                        else
                        {
                            (r.image as DynamicImage).SetFrameStartTime(0, (dicomfiles[i].image as DynamicImage).GetFrameStartTime(0));
                            (r.image as DynamicImage).SetFrameDuration(0, (dicomfiles[i].image as DynamicImage).GetFrameDuration(0));
                        }
                    }
                }
            }
            else {
                r.image = dicomfiles[0].image;
            }
            //copy fileinfo
            return r;
        }
        /// <summary>
        /// Splits dicomfile into DICOM images trough to dimension.
        /// Filename is set to filename _ number .dcm. Result header information is copied from first 
        /// file.
        /// </summary>
        /// <see cref="DicomFile"/>
        /// <param name="dimension">dimension that is splitted (0-based)</param>
        /// <returns>new DicomFile array</returns>
        /// <exception cref="TPCInvalidArgumentsException">if input data is inconsistent</exception>
        public DicomFile[] SplitDimension(int dimension)
        {
            DicomFile[] r;

            if (!(image is Image)) throw new TPCInvalidArgumentsException("Image is null.");
            if (!(header is ImageHeader)) throw new TPCInvalidArgumentsException("Header is null.");
            switch (dimension)
            {
                case Limits.PLANES:
                    r = new DicomFile[image.dim.GetDimension(IntLimits.PLANES)];
                    for (int i = 0; i < r.Length; i++) {
                        r[i] = new DicomFile(filename+"_"+i.ToString("00000"));
                        if (image is DynamicImage)
                        {
                            r[i].image = new DynamicImage(image.GetSubImage(image.GetPlaneLimits(i)));
                            (r[i].image as DynamicImage).SetFrameStartTime(0, (image as DynamicImage).GetFrameStartTime(0));
                            (r[i].image as DynamicImage).SetFrameDuration(0, (image as DynamicImage).GetFrameDuration(0));
                        }
                        else
                        {
                            r[i].image = image.GetSubImage(image.GetPlaneLimits(i));
                        }
                        r[i].dicomheader.slice_location = i;
                        r[i].header = header;
                        //copy fileinfo
                        r[i].dicomheader.instance_nr = i;
                        r[i].dicomheader.series_nr = dicomheader.series_nr;
                        r[i].header.modality = header.modality;
                    }
                    break;
                case Limits.FRAMES:
                    r = new DicomFile[image.dim.GetDimension(IntLimits.FRAMES)];
                    for (int i = 0; i < r.Length; i++) {
                        r[i] = new DicomFile(filename + "_" + i.ToString("000"));
                        if (image is DynamicImage)
                        {
                            r[i].image = new DynamicImage(image.GetSubImage(image.GetDimensionLimits(i, IntLimits.FRAMES)));
                            (r[i].image as DynamicImage).SetFrameStartTime(0, (image as DynamicImage).GetFrameStartTime(i));
                            (r[i].image as DynamicImage).SetFrameDuration(0, (image as DynamicImage).GetFrameDuration(i));
                        }
                        else {
                            r[i].image = image.GetSubImage(image.GetDimensionLimits(i, IntLimits.FRAMES));
                        }
                        r[i].dicomheader.slice_start = 0;
                        r[i].header = header;
                        //copy fileinfo
                        r[i].dicomheader.instance_nr = i * image.planes;
                        r[i].dicomheader.series_nr = dicomheader.series_nr;
                        r[i].header.modality = header.modality;
                    }
                    break;
                case Limits.GATES:
                    r = new DicomFile[image.dim.GetDimension(IntLimits.GATES)];
                    for (int i = 0; i < r.Length; i++)
                    {
                        r[i] = new DicomFile(filename + "_" + i.ToString("000"));
                        if (image is DynamicImage)
                        {
                            r[i].image = new DynamicImage(image.GetSubImage(image.GetDimensionLimits(i, IntLimits.GATES)));
                            (r[i].image as DynamicImage).SetFrameStartTime(0, (image as DynamicImage).GetFrameStartTime(i));
                            (r[i].image as DynamicImage).SetFrameDuration(0, (image as DynamicImage).GetFrameDuration(i));
                        }
                        else
                        {
                            r[i].image = new DynamicImage(image.GetSubImage(image.GetDimensionLimits(i, IntLimits.GATES)));
                        }
                        r[i].dicomheader.gate_number = 0;
                        r[i].header = header;
                        //copy fileinfo
                        r[i].dicomheader.instance_nr = i * image.planes;
                        r[i].dicomheader.series_nr = dicomheader.series_nr;
                        r[i].header.modality = header.modality;
                    }
                    break;
                default:
                    throw new TPCInvalidArgumentsException("Unsupported dimension for splitting:" + dimension);
            }
            return r;
        }
        /// <summary>
        /// Checks file format.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DICOM file</returns>
        /// <exception cref="TPCDicomFileException">if there was a read problem</exception>
        public static bool CheckFormat(string filename) {
            try
            {
                return openDicom.File.DicomFile.IsDicomFile(filename);
            }
            catch (Exception)
            {
                //not a single DICOM file? try first file of many multiple files
                FileInfo[] filenames;
                try
                {
                    filenames = DicomFile.ResolveDicomFilenames(filename);
                }
                catch (Exception) {
                    return false;
                }
                if (filenames.Length == 0)
                    return false;
                for (int i = 0; i < filenames.Length; i++)
                {
                    try
                    {
                        if (openDicom.File.DicomFile.IsDicomFile(filenames[i].FullName))
                            return true;
                    }
                    catch (Exception) {}
                }
                return false;
            }
        }
        /// <summary>
        /// Writes single frame into file. If frame number is larger than 
        /// current number of frames in file, then file size is grown.
        /// </summary>
        /// <param name="image">image data that is written</param>
        /// <param name="frame_No">frame number that is written [1..no frames+1]</param>
        public override void WriteFrame(ref Image image, int frame_No)
        {
            throw new NotImplementedException();
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started (at beginning of tag (7FE0,0010))</param>
        /// <param name="scale">scale factor</param>
        public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {

            if (offset + dim.GetProduct() > data.dataLength)
                throw new TPCInvalidArgumentsException("Invalid parameters: offset(" + offset + ")+length(" + dim.GetProduct() + ")>= data.Length(" + data.dataLength + ")");
            //read data from previously acquired positions
            byte[] bytes = null;
            int location = offset;
            try
            {
                stream.Seek(dicomheader.stream_data_position, SeekOrigin.Begin);
                // tag       (4 bytes)
                Tag tag = new Tag(stream, TransferSyntax.Default);
                // VR        (2 bytes)
                ValueRepresentation vr = ValueRepresentation.LoadFrom(stream, tag);
                // VR        (2 bytes)
                // reserved (2 bytes)
                // length    (4 bytes)
                stream.Seek((long)8, SeekOrigin.Current);
                //read rest of stream as data
                bytes = new byte[stream.Length-stream.Position];
                stream.Read(bytes, 0, bytes.Length);
            }
            catch (Exception e)
            {
                throw new TPCDicomFileException("Exception [" + e + "] while reading image data of [" + filename + "]");
            }

            //store data into image
            if (bytes.Length == 0) throw new TPCDicomFileException("0 bytes in pixel data");
            int bytesPerPixel = ImageFile.BytesPerPixel(datatype);
            byte[] value = new byte[bytesPerPixel];

            switch (datatype)
            {
                case DataType.BIT8_U:
                    for (int j = 0; j < bytes.Length && location < data.dataLength; j++)
                        data[location++] = bytes[j] *(float)dicomheader.scalefactor;
                    break;
                case DataType.BIT8_S:
                    for (int j = 0; j < bytes.Length && location < data.dataLength; j++)
                        data[location++] = (sbyte)bytes[j] *(float)dicomheader.scalefactor;
                    break;
                case DataType.BIT16_U:
                    for (int j = 0; j < bytes.Length && location < data.dataLength; j += bytesPerPixel)
                    {
                        Array.Copy(bytes, j, value, 0, bytesPerPixel);
                        data[location++] = (float)BitConverter.ToUInt16(value, 0)*(float)dicomheader.scalefactor;
                    }
                    break;
                case DataType.BIT16_S:
                    for (int j = 0; j < bytes.Length && location < data.dataLength; j += bytesPerPixel)
                    {
                        Array.Copy(bytes, j, value, 0, bytesPerPixel);
                        data[location++] = (float)BitConverter.ToInt16(value, 0)*(float)dicomheader.scalefactor;
                    }
                    break;
                default:
                    throw new TPCDicomFileException("Unsupported data format:" + header.datatype);
            }
        }
        /// <summary>
        /// Reads only header data from file
        /// </summary>
        /// <returns>Header information in file</returns>
        public override ImageHeader ReadHeader()
        {
            openDicom.File.AcrNemaFile file = null;
            //set end condition tag as pixel data to skip reading it
            Sequence.EndConditionTag = new Tag("(7FE0,0010)");
            try
            {
                if (openDicom.File.DicomFile.IsDicomFile(filename))
                    file = new openDicom.File.DicomFile(filename, false);
                else if (openDicom.File.AcrNemaFile.IsAcrNemaFile(filename))
                    file = new openDicom.File.AcrNemaFile(filename, false);
                else
                    Console.Error.WriteLine("File "+filename+" is wether a DICOM nor an ACR-NEMA file.");
            }
            catch (Exception dicomFileException)
            {
                throw new TPCDicomFileException("Failed to read DICOM file " + filename + ":" + dicomFileException);
            }
            finally {
                //take of end condition
                Sequence.EndConditionTag = null;
            }

            Sequence sequence = file.GetJointDataSets().GetJointSubsequences();
            header.siz.sizez = 0.0f;
            //group tag used to track last grouping tag in 0054
            int nested = 0;
            Tag groupTag = new Tag("(0000,0000)");
            foreach (DataElement d in sequence)
            {
                //add field into general header
                try
                {
                    if (d is DataElement && d.Value is Value && d.Tag is Tag && DicomFile.dataElementDictionary.GetDictionaryEntry(d.Tag) is DataElementDictionaryEntry)
                        header.Add(DicomFile.dataElementDictionary.GetDictionaryEntry(d.Tag).Description, (d as DataElement));
                }
                catch (openDicom.DicomException)
                {
                    //do not add null reference if it does exist
                    //throw new TPCDicomFileError("openDicom.DicomException");
                }
                catch (TPCGeneralHeaderException)
                {
                    //ignore multiple fields, just saving the first occurrence
                }
                if (d.Tag.Equals("(FFFE,E000)"))
                {
                    nested++;
                }
                else if (d.Tag.Equals("(FFFE,E0DD)"))
                {
                    nested--;
                }
                if (d.Tag.Equals("(0054,0300)") || d.Tag.Equals("(0054,0304)"))
                    groupTag = d.Tag;
                //fill header data fields
                if (d.Tag.Equals("(0054,1001)"))
                {
                    header.dataunit = UnitConverter.CreateDataUnit(d.Value.ToArray()[0].ToString());
                }
                //resolve SOP instanceUID for sorting planes
                else if (d.Tag.Equals("(0008,0018)"))
                    header.description = d.Value.ToArray()[0].ToString();
                else if (d.Tag.Equals("(0008,0060)"))
                {
                    try
                    {
                        header.modality = (ImageModality)System.Enum.Parse(typeof(ImageModality), d.Value.ToArray()[0].ToString());
                    }
                    catch (Exception)
                    {
                        if (d.Value.ToArray()[0].ToString().Equals("NUCMED"))
                            header.modality = ImageModality.M_NM;
                        else if (d.Value.ToArray()[0].ToString().StartsWith("SPECT"))
                            header.modality = ImageModality.M_PX;
                        else if (d.Value.ToArray()[0].ToString().StartsWith("PET"))
                            header.modality = ImageModality.M_PT;
                        else if (d.Value.ToArray()[0].ToString().StartsWith("RT"))
                            header.modality = ImageModality.M_RT;
                        else
                            header.modality = ImageModality.Unknown;
                    }
                }
                else if (d.Tag.Equals("(0008,1030)"))
                    header.description = d.Value.ToArray()[0].ToString();
                else if (d.Tag.Equals("(0010,0010)"))
                    header.patient_name = d.Value.ToArray()[0].ToString();
                else if (d.Tag.Equals("(0010,0020)"))
                    header.patientID = d.Value.ToArray()[0].ToString();
                //radiopharmaceutical name
                else if (nested == 0 && d.Tag.Equals("(0018,0031)"))
                {
                    if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                    (header as DynamicImageHeader).radiopharma = d.Value.ToArray()[0].ToString();
                }
                else if (d.Tag.Equals("(0018,0050)"))
                {
                    try
                    {
                        header.siz.sizez = float.Parse(d.Value.ToArray()[0].ToString());
                    }
                    catch (Exception)
                    {
                        //do not change anything
                    }
                }
                //reads field for isotope, affects conversion to DynamicImageHeader if needed
                else if (nested > 0 && d.Tag.Equals("(0008,0104)") && groupTag.Equals("(0054,0300)"))
                {
                    if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                    (header as DynamicImageHeader).isotope = Isotope.CreateIsotope(d.Value.ToArray()[0].ToString());
                }
                //frame duration time, affects image type to be DynamicImage if found
                else if (d.Tag.Equals("(0018,1242)"))
                {
                    dicomheader.frame_duration = Convert.ToDouble((Int64)d.Value.ToArray()[0]);
                }
                //injection time
                else if (d.Tag.Equals("(0018,1072)"))
                {
                    if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                    try
                    {
                        TimeSpan span = (TimeSpan)d.Value.ToArray()[0];
                        (header as DynamicImageHeader).dose_start_time = new System.DateTime(span.Ticks);
                    }
                    catch
                    {
                        (header as DynamicImageHeader).dose_start_time = new System.DateTime(0);
                    }
                }
                else if (d.Tag.Equals("(0020,0013)"))
                {
                    dicomheader.instance_nr = (int)((long)d.Value.ToArray()[0]);
                }
                else if (d.Tag.Equals("(0020,1041)"))
                {
                    dicomheader.slice_location = (float)((decimal)d.Value.ToArray()[0]);
                }
                else if (d.Tag.Equals("(0028,0010)"))
                    dicomheader.height = (UInt16)d.Value.ToArray()[0];
                else if (d.Tag.Equals("(0028,0011)"))
                    dicomheader.width = (UInt16)d.Value.ToArray()[0];
                else if (d.Tag.Equals("(0028,0012)"))
                    dicomheader.planes = (UInt16)d.Value.ToArray()[0];
                else if (d.Tag.Equals("(0028,0100)"))
                {
                    //get number of bytes allocatted for single pixel
                    dicomheader.bytesperpixel = (int)Math.Ceiling(((float)((UInt16)d.Value.ToArray()[0])) / 8.0f);
                }
                else if (d.Tag.Equals("(0028,0103)"))
                {
                    dicomheader.signed = ((UInt16)d.Value.ToArray()[0] != 0);
                }
                else if (d.Tag.Equals("(0028,0030)"))
                {
                    header.siz.sizex = (float)((Decimal)d.Value.ToArray()[0]);
                    header.siz.sizey = (float)((Decimal)d.Value.ToArray()[1]);
                }
                else if (d.Tag.Equals("(0028,1053)"))
                {
                    dicomheader.scalefactor = float.Parse(d.Value.ToArray()[0].ToString());
                }
                //number of planes
                else if (d.Tag.Equals("(0054,0081)"))
                {
                    dicomheader.planes = (UInt16)d.Value.ToArray()[0];
                }
                //number of frames, overwritten when DICOM plane files are stacked into dynamic image
                else if (d.Tag.Equals("(0054,0101)"))
                {
                    dicomheader.frames = (UInt16)d.Value.ToArray()[0];
                }
                //frame reference time, affects image type to be DynamicImage if found
                else if (d.Tag.Equals("(0054,1300)"))
                {
                    dicomheader.frame_start_time = Convert.ToDouble((Decimal)d.Value.ToArray()[0]);
                }
                //resolve bytes per pixel if header fields did not express it already
                else if (d.Tag.Equals("(7FE0,0000)") && dicomheader.bytesperpixel == 0)
                {
                    if (dicomheader.width * dicomheader.height > 0)
                        dicomheader.bytesperpixel = (int)Math.Ceiling((((UInt32)d.Value.ToArray()[0]) - 12) / (decimal)(dicomheader.width * dicomheader.height));
                }
                //data tag position in DICOM file for later fast access
                else if (d.Tag.Equals("(7FE0,0010)"))
                {
                    dicomheader.stream_data_position = d.StreamPosition;
                }
            }
                    
            //set z-dimension 1, change later
            if (header.siz.sizez == 0.0f) header.siz.sizez = 1.0f;

            //copy image size to header as well
            header.dim = new IntLimits(dicomheader.width, dicomheader.height, 1);

            //resolve data type into ImageHeader
            switch (dicomheader.bytesperpixel)
            {
                case 1:
                    if (dicomheader.signed) header.datatype = DataType.BIT8_S;
                    else header.datatype = DataType.BIT8_U;
                    break;
                case 2:
                    if (dicomheader.signed) header.datatype = DataType.BIT16_S;
                    else header.datatype = DataType.BIT16_U;
                    break;
                case 4:
                    if (dicomheader.signed) header.datatype = DataType.BIT32_S;
                    else header.datatype = DataType.BIT32_U;
                    break;
                default:
                    //do not throw exception here so that header can be read even when data is not able to be read
                    //throw new TPCDicomFileException("Unsupported data format: bytes=" + dicomheader.bytesperpixel + ",signed=" + dicomheader.signed);
                    break;
            }

            return header;
        }
        /// <summary>
        /// Gets parameters needed for direct access to pixel data.
        /// </summary>
        /// <returns>parameters to pixel data</returns>
        public override PixelDataParameters GetParametersForPixelData() {
            PixelDataParameters p;
            p.filename = filename;
            p.datatype = header.datatype;
            p.dim = image.dim;
            p.position = dicomheader.stream_data_position;
            p.scale = dicomheader.scalefactor;
            return p;
        }
        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">subregion that is read from file</param>
        public override void ReadSubImage(IntLimits region) {
            ReadFile();
            header.dim = region;
            image = image.GetSubImage(region);
        }
    }
}