﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections.Generic;

namespace TPClib.Image
{
    /// <summary>
    /// Comparison class for dicom file sorting. 
    /// </summary>
    public class DicomHeaderComparer : IComparer<DicomHeader>
    {
        /// <summary>
        /// Sorting according to:
        /// group name
        /// frame start times if files have dynamic images
        /// plane z positions
        /// <see cref="DynamicImage"/>
        /// </summary>
        /// <param name="x">first evaluated file</param>
        /// <param name="y">second evaluated file</param>
        /// <returns></returns>
        public int Compare(DicomHeader x, DicomHeader y)
        {
            //sort null objects as 'smaller' than existing ones
            if (x == null)
            {
                if (y == null) return 0;
                else return 1;
            }
            else if (y == null) return -1;
            //group with series number
            if (x.series_nr < y.series_nr) return -1;
            if (x.series_nr > y.series_nr) return 1;
            //use gate numbers if available
            if (x.gate_number < y.gate_number) return -1;
            if (x.gate_number > y.gate_number) return 1;

            //use frame start times if available
            if (x.slice_start < y.slice_start) return -1;
            if (x.slice_start > y.slice_start) return 1;

            if (x.slice_location < y.slice_location) return -1;
            if (x.slice_location > y.slice_location) return 1;

            if (x.instance_nr < y.instance_nr) return -1;
            if (x.instance_nr > y.instance_nr) return 1;

            //couldn't find any sorting difference
            return 0;
        }
    }
    /// <summary>
    /// DICOM specific header information
    /// </summary>
    public class DicomHeader
    {
        /// <summary>
        /// Patient orientation (0018,5100)
        /// </summary>
        public string orientation = "FFS";
        /// <summary>
        /// Image instance number
        /// </summary>
        public int instance_nr = 0;
        /// <summary>
        /// Slice start time in milliseconds. Used for frame sorting.
        /// </summary>
        public int slice_start = 0;
        /// <summary>
        /// Slice location in z-axis (planes). Used for orientation determination.
        /// </summary>
        public float slice_location = 0;
        /// <summary>
        /// Gate number (1-based). Used fot gate sorting.
        /// </summary>
        public int gate_number = 0;
        /// <summary>
        /// Series number. Used to group one image set.
        /// </summary>
        public int series_nr = 0;
        /// <summary>
        /// Acquisition type
        /// </summary>
        public AcquisitionType acquisitiontype;
        /// <summary>
        /// Scale factor in field (0x0028, 0x1053). This field is used in read and write to scale the intensity 
        /// values to true values. (default == 1.0)
        /// </summary>
        public float scalefactor = 1.0f;
        /// <summary>
        /// Signed/unsigned data
        /// </summary>
        public bool signed = false;
        /// <summary>
        /// Bytes of data in file per single pixel value
        /// </summary>
        public int bytesperpixel = 0;
        /// <summary>
        /// Frame start time.
        /// </summary>
        public double frame_start_time = 0.0;
        /// <summary>
        /// Frame duration.
        /// </summary>
        public double frame_duration = 0.0;
        /// <summary>
        /// image width
        /// </summary>
        public int width = 0;
        /// <summary>
        /// image height
        /// </summary>
        public int height = 0;
        /// <summary>
        /// image planes
        /// </summary>
        public int planes = 0;
        /// <summary>
        /// image frames
        /// </summary>
        public int frames = 0;
        /// <summary>
        /// Stream position for data tag
        /// </summary>
        public long stream_data_position = 0;
        /// <summary>
        /// Returns string representation of this object. 
        /// Most important header fields are listed.
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            return "DicomHeader[instance="+instance_nr+" scale="+scalefactor+" z_location="+slice_location+" width="+width+" height="+height+" planes="+planes+" frames="+frames+" pos="+stream_data_position+"]";
        }
    }
}
