/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Image {
    /// <summary>
    /// Dynamic image that contains frame and radiopharmaceutical information.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class DynamicImage : Image
    {
        /// <summary>
        /// Frame start times in milliseconds
        /// </summary>
        private float[] frameStartTimes;
        /// <summary>
        /// Frame durations in milliseconds
        /// </summary>
        private float[] frameDurations;

        /// <summary>
        /// Construction of dynamic image with zero dimensions is prevented.
        /// </summary>
        protected DynamicImage()
        {
        }

        /// <summary>
        /// Constructs dynamic image from image object
        /// </summary>
        /// <param name="image"></param>
        public DynamicImage(Image image): base(image.dim) {
            //upgrade dimensions to time dimension if it is missing
            //note that the data array size stays the same
            while (dim.Length < 4)
            {
                dim.AddLimit(0, 1);
            }
            if ((int)dim.GetDimension(Limits.FRAMES) > 1)
            {
                frameStartTimes = new float[(int)dim.GetDimension(Limits.FRAMES)];
                frameDurations = new float[(int)dim.GetDimension(Limits.FRAMES)];
            }
            else
            {
                frameStartTimes = new float[1];
                frameDurations = new float[1];
            }
            for (int i = 0; i < image.dataLength; i++)
                data[i] = image[i];
        }
        /// <summary>
        /// Constructs dynamic image with desired dimensions (X x Y x Z x T). Image
        /// has zero at all voxels. Image will allways have time dimension.
        /// </summary>
        /// <param name="dimx">width</param>
        /// <param name="dimy">height</param>
        /// <param name="dimz">planes</param>
        /// <param name="dimt">frames</param>
        /// <exception cref="TPCInvalidArgumentsException">if any parameter is below 1</exception>
        public DynamicImage(int dimx, int dimy, int dimz, int dimt)
            : base(new IntLimits(new int[]{dimx, dimy, dimz, dimt}))
        {
            if (dimx < 1) throw new TPCInvalidArgumentsException("Image width must be positive");
            if (dimy < 1) throw new TPCInvalidArgumentsException("Image height must be positive");
            if (dimz < 1) throw new TPCInvalidArgumentsException("Image planes must be positive");
            if (dimt < 1) throw new TPCInvalidArgumentsException("Image times frames must be positive");
            data = new float[(int)dim.GetProduct()];
            frameStartTimes = new float[(int)dim.GetDimension(Limits.FRAMES)];
            frameDurations = new float[(int)dim.GetDimension(Limits.FRAMES)];
        }
        /// <summary>
        /// Constructs dynamic image with desired dimensions (X x Y x Z x T...). Image
        /// has zero at all voxels. Image will allways have time dimension.
        /// </summary>
        /// <param name="dimensions">dimension values</param>
        public DynamicImage(params int[] dimensions) : this(new IntLimits(dimensions)) {
        }
        /// <summary>
        /// Constructs dynamic image with desired dimensions (X x Y x Z x T...). Image
        /// has zero at all voxels. Image will allways have time dimension.
        /// </summary>
        /// <param name="dimension">dimension object, dimension values will be converted to integer values</param>
        public DynamicImage(IntLimits dimension): base(dimension)
        {
            //upgrade dimensions to time dimension if it is missing
            while (dimension.Length < 4)
            {
                dimension.AddLimit(0, 1);
                dim.AddLimit(0,1);
            }
            if ((int)dimension.GetDimension(Limits.FRAMES) > 1)
            {
                frameStartTimes = new float[(int)dimension.GetDimension(Limits.FRAMES)];
                frameDurations = new float[(int)dimension.GetDimension(Limits.FRAMES)];
            }
            else {
                frameStartTimes = new float[1];
                frameDurations = new float[1];
            }
        }
        
        /// <summary>
        /// Short-cut for number of frames.
        /// </summary>
        public int frames
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.FRAMES);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Short-cut for number of gates.
        /// </summary>
        public int gates
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.GATES);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Short-cut for number of bed positions.
        /// </summary>
        public int beds
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.BEDS);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Short-cut for detectorvectors.
        /// </summary>
        public int detectorvector
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.DETECTORVECTOR);
                if (r % planes * frames * gates * beds != 0)
                    throw new TPCDicomFileException("number of detectorvector must comply with total number of slices");
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Short-cut for energywindows.
        /// </summary>
        public int energywindofilestream
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.ENERGYWINDOW);
                if (r % planes * frames * gates * beds != 0)
                    throw new TPCDicomFileException("number of evergywindofilestream must comply with total number of slices");
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Length of frame in image.
        /// </summary>
        public int frameLength
        {
            get { return (int)dim.GetDimension(Limits.WIDTH) * (int)dim.GetDimension(Limits.HEIGHT) * (int)dim.GetDimension(Limits.PLANES); }
        }
        /// <summary>
        /// Sets frame start times.
        /// </summary>
        /// <param name="start_times">array of non-negative ascending values with same length as number of frames</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameStartTimes(double[] start_times)
        {
            float[] fvalues = new float[start_times.Length];
            Array.Copy(start_times, fvalues, start_times.Length);
            SetFrameStartTimes(fvalues);
        }
        /// <summary>
        /// Sets frame start times.
        /// </summary>
        /// <param name="start_times">array of non-negative ascending values with same length as number of frames</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameStartTimes(float[] start_times)
        {
            if (start_times.Length != frameStartTimes.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start times.");
            if (start_times[0] < 0) throw new TPCInvalidArgumentsException("First start time is negative:" + start_times[0]);
            for (int i = 1; i < start_times.Length; i++)
                if (start_times[i-1] > start_times[i]) throw new TPCInvalidArgumentsException("START times of frames are not in ascending order.");

            frameStartTimes = start_times;
        }
        /// <summary>
        /// Sets a frame start time.
        /// </summary>
        /// <param name="index">time index (0-based)</param>
        /// <param name="start_time">non-negative time value (milliseconds)</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameStartTime(int index, double start_time)
        {
            SetFrameStartTime(index, (float)start_time);
        }
        /// <summary>
        /// Sets a frame start time.
        /// </summary>
        /// <param name="index">time index (0-based)</param>
        /// <param name="start_time">non-negative time value (milliseconds)</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameStartTime(int index, float start_time)
        {
            if (index < 0 || index >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
            if(index > 0 && start_time < frameStartTimes[index-1])
                throw new TPCInvalidArgumentsException("Frame start time "+start_time+" is before previous index's value "+frameStartTimes[index-1]);
            if(index < (frameStartTimes.Length-1) && frameStartTimes[index+1] > 0 && start_time > frameStartTimes[index+1])
                throw new TPCInvalidArgumentsException("Frame start time "+start_time+" is after following index's value "+frameStartTimes[index+1]);
            frameStartTimes[index] = start_time;
        }
        /// <summary>
        /// Sets a frame end time.
        /// </summary>
        /// <param name="index">time index (0-based)</param>
        /// <param name="end_time">non-negative time value (milliseconds)</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameEndTime(int index, double end_time)
        {
            SetFrameEndTime(index, (float)end_time);
        }
        /// <summary>
        /// Sets a frame end time.
        /// </summary>
        /// <param name="index">time index (0-based)</param>
        /// <param name="end_time">non-negative time value (milliseconds)</param>
        /// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
        public void SetFrameEndTime(int index, float end_time)
        {
            if (index < 0 || index >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
            if (index > 0 && end_time < frameStartTimes[index - 1])
                throw new TPCInvalidArgumentsException("Frame end time " + end_time + " is before previous index's value " + frameStartTimes[index - 1]);
            if (index < (frameStartTimes.Length - 1) && end_time > frameStartTimes[index + 1])
                throw new TPCInvalidArgumentsException("Frame end time " + end_time + " is after following index's value " + frameStartTimes[index + 1]);
            if(end_time <= frameStartTimes[index])
                throw new TPCInvalidArgumentsException("Frame end time "+end_time+" is not after frame's start time " + frameStartTimes[index]);
            frameDurations[index] = end_time - frameStartTimes[index];
        }
        /// <summary>
        /// Sets frame duration times.
        /// </summary>
        /// <param name="durations">array of non-negative values with same length as number of frames</param>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public void SetFrameDurations(float[] durations)
        {
            if (durations.Length != frameDurations.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start times.");
            for (int i = 0; i < durations.Length; i++)
                SetFrameDuration(i, durations[i]);
        }
        /// <summary>
        /// Sets frame duration times.
        /// </summary>
        /// <param name="durations">array of non-negative values with same length as number of frames</param>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public void SetFrameDurations(double[] durations)
        {
            if (durations.Length != frameDurations.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start times.");
            for (int i = 0; i < durations.Length; i++)
                SetFrameDuration(i, durations[i]);
        }
        /// <summary>
        /// Sets frame duration time.
        /// </summary>
        /// <param name="index">time index</param>
        /// <param name="duration">a non-negative value</param>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public void SetFrameDuration(int index, double duration)
        {
            SetFrameDuration(index, (float)duration);
        }
        /// <summary>
        /// Sets frame duration time.
        /// </summary>
        /// <param name="index">time index</param>
        /// <param name="duration">a non-negative value in milliseconds</param>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public void SetFrameDuration(int index, float duration)
        {
            if (index < 0 || index >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameDurations.Length - 1) + "]");
            if (duration < 0) throw new TPCInvalidArgumentsException("Tried to set duration " + index + " a negative value:" + duration);
            frameDurations[index] = duration;
        }
        /// <summary>
        /// Returns frame start time in milliseconds.
        /// </summary>
        /// <param name="i">time index (0-based)</param>
        /// <returns>time in milliseconds</returns>
        public float GetFrameStartTime(int i)
        {
            if (i < 0 || i >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
            return frameStartTimes[i];
        }
        /// <summary>
        /// Returns frame start times in milliseconds.
        /// </summary>
        /// <returns>array of times in milliseconds</returns>
        public float[] GetFrameStartTimes()
        {
            return frameStartTimes;
        }
        /// <summary>
        /// Returns frame duration in milliseconds.
        /// </summary>
        /// <param name="i">time index (0-based)</param>
        /// <returns>time in milliseconds</returns>
        public float GetFrameDuration(int i)
        {
            if (i < 0 || i >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameDurations.Length - 1) + "]");
            return frameDurations[i];
        }
        /// <summary>
        /// Returns frame durations in milliseconds.
        /// </summary>
        /// <returns>array of times in milliseconds</returns>
        public float[] GetFrameDurations()
        {
            return frameDurations;
        }
        /// <summary>
        /// Rerturns frame end times.
        /// </summary>
        /// <returns>array of times in milliseconds</returns>
        public float[] GetFrameEndTimes()
        {
            float[] r = new float[frameStartTimes.Length];
            for (int i = 0; i < frameStartTimes.Length; i++)
                r[i] = frameStartTimes[i]+frameDurations[i];
            return r;
        }
        /// <summary>
        /// Returns frame end time.
        /// </summary>
        /// <param name="i">time index (0-based)</param>
        /// <returns>time in milliseconds</returns>
        public float GetFrameEndTime(int i)
        {
            if (i < 0 || i >= frameStartTimes.Length)
                throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
            return frameStartTimes[i] + frameDurations[i];
        }
        /// <summary>
        /// Rerturns frame mid times.
        /// </summary>
        /// <returns>array of times in milliseconds</returns>
        public float[] GetFrameMidTimes()
        {
            float[] r = new float[frameStartTimes.Length];
            for (int i = 0; i < frameStartTimes.Length; i++)
                r[i] = frameStartTimes[i] + frameDurations[i]/2.0f;
            return r;
        }
        /// <summary>
        /// Returns minimum and maximum data value
        /// </summary>
        /// <param name="subregion">subregion for calculation</param>
        /// <returns>array containing [minimum, maximum] value</returns>
        public new float[] GetMinMax(IntLimits subregion)
        {
            if (subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dimz ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dimy ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dimx)
                throw new TPCInvalidArgumentsException("Subregion [" + subregion + "] is not inside image bounds [" + dim + "]");
            if (data.Length == 0) return new float[2] { 0.0f, 0.0f };
            float max = float.MinValue;
            float min = float.MaxValue;
            int i = 0;
            int frames_end = ((this is DynamicImage) ? (this as DynamicImage).frames : 1);
            int frames_start = 0;
            if (subregion.Length > IntLimits.FRAMES)
            {
                frames_start = subregion.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                frames_end = subregion.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
            }
            int planes = (dim.Length > IntLimits.PLANES ? dim.planes : 1);
            int planes_end = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) : 1);
            int planes_start = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) : 0);
            for (int t = frames_start; t < frames_end; t++)
            {
                for (int z = planes_start; z < planes_end; z++)
                {
                    for (int y = subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW); y < subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
                    {
                        for (int x = subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW); x < subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH); x++)
                        {
                            i = t * planes * dimy * dimx + z * dimx * dimy + y * dimx + x;
                            if (data[i] < min) min = data[i];
                            if (data[i] > max) max = data[i];
                        }
                    }
                }
            }
            return new float[2] { min, max };
        }
        /// <summary>
        /// Returns curve at point in dynamic file.
        /// </summary>
        /// <param name="x">x-coordinate location in image</param>
        /// <param name="y">y-coordinate location in image</param>
        /// <param name="z">z-coordinate location in image</param>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public float[] GetCurve(int x, int y, int z)
        {
            if (x < 0 || x >= width)
                throw new TPCInvalidArgumentsException("x-coordinate " + x + " is out of image plane array bounds [0.." + (width  - 1) + "].");
            if (y < 0 || y >= height)
                throw new TPCInvalidArgumentsException("y-coordinate " + y + " is out of image plane array bounds [0.." + (height - 1) + "].");
            if (z < 0 || z >= planes)
                throw new TPCInvalidArgumentsException("z-coordinate " + z + " is out of image plane array bounds [0.." + (planes - 1) + "].");

            int no_high_dimensions = dataLength/this.frameLength;
            float[] r = new float[no_high_dimensions];
            int frameLength = this.frameLength;
            int location_i = z*dimx*dimy + y*dimx + x;

            //go through all higher dimensions in order
            for (int i = 0; i < no_high_dimensions; i++)
            {
                r[i] = data[location_i+i*frameLength];
            }
            return r;
        }
        /// <summary>
        /// Gets curve in 3D-location.
        /// </summary>
        /// <param name="location">location in image 3D space( = x * y * z)</param>
        /// <returns>All data in location (all frames, gates etc.)</returns>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public float[] GetCurve(int location)
        {
            if (location < 0 || location >= this.frameLength)
                throw new TPCInvalidArgumentsException("Index " + location + " is out of image plane array bounds [0.." + (this.frameLength - 1) + "].");
            int no_high_dimensions = dataLength / this.frameLength;
            int frameLength = this.frameLength;
            float[] r = new float[no_high_dimensions];
            //go through all higher dimensions in order
            for (int i = 0; i < no_high_dimensions; i++)
            {
                r[i] = data[i*frameLength+location];
            }
            return r;
        }
        /// <summary>
        /// Returns curve at point in dynamic file.
        /// </summary>
        /// <param name="location">3D-coordinate location in image</param>
        /// <returns>All data in location (all frames, gates etc.)</returns>
        /// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
        public float[] GetCurve(IntPoint location)
        {
            if (location.x < 0 || location.x >= width)
                throw new TPCInvalidArgumentsException("x-coordinate " + location.x + " is out of image plane array bounds [0.." + (width - 1) + "].");
            if (location.y < 0 || location.y >= height)
                throw new TPCInvalidArgumentsException("y-coordinate " + location.y + " is out of image plane array bounds [0.." + (height - 1) + "].");
            if (location.z < 0 || location.z >= planes)
                throw new TPCInvalidArgumentsException("z-coordinate " + location.z + " is out of image plane array bounds [0.." + (planes - 1) + "].");

            int no_high_dimensions = dataLength / this.frameLength;
            float[] r = new float[no_high_dimensions];
            int frameLength = this.frameLength;
            int location_i = location.z * dimx * dimy + location.y * dimx + location.x;

            //go through all higher dimensions in order
            for (int i = 0; i < no_high_dimensions; i++)
            {
                r[i] = data[i * frameLength + location_i];
            }
            return r;
        }
        /// <summary>
        /// Shortcut for getting limits for image's frame subregion.
        /// </summary>
        /// <param name="frame">frame index (0-based)</param>
        /// <returns>limits object that represents frame dimensions</returns>
        public IntLimits GetFrameLimits(int frame)
        {
            return GetDimensionLimits(frame, IntLimits.FRAMES);
        }
        /// <summary>
        /// Shortcut for getting limits for image's gate subregion.
        /// </summary>
        /// <param name="gate">gate index (0-based)</param>
        /// <returns>limits object that represents gate dimensions</returns>
        public IntLimits GetGateLimits(int gate)
        {
            return GetDimensionLimits(gate, IntLimits.GATES);
        }
        /// <summary>
        /// Voxels of image as an multidimensional array.
        /// </summary>
        /// <param name="x">x-coordinate (0-based)</param>
        /// <param name="y">y-coordinate (0-based)</param>
        /// <param name="z">z-coordinate (0-based)</param>
        /// <param name="t">t(time)-coordinate (0-based)</param>
        /// <returns>value at location</returns>
        public float this[int x, int y, int z, int t]
        {
            get
            {
                return GetValue(new int[] { x, y, z, t });
            }
            set
            {
                SetValue(value, new int[] { x, y, z, t });
            }
        }
        /// <summary>
        /// Returns data of one frame from image.
        /// </summary>
        /// <param name="frame">frame index (0-based)</param>
        /// <returns>new image containing frame data</returns>
        public Image GetFrame(int frame)
        {
            Image r = GetSubImage(GetFrameLimits(frame));
            r.dim.RemoveSingletons();
            return r;
        }

        /// <summary>
        /// Returns data of one frame from image.
        /// </summary>
        /// <param name="frame">frame index (0-based)</param>
        /// <param name="image"></param>
        /// <returns>new image containing frame data</returns>
        public void SetFrame(int frame, ref Image image)
        {
            SetSubImage(GetFrameLimits(frame), ref image);
        }

        /// <summary>
        /// Prints image information to stdout.
        /// </summary>
        public new void PrintImage()
        {
            base.PrintImage();
            Console.WriteLine(" Dynamic:");
            Console.Write("   frameStartTimes:");
            for(int i = 0; i < frameStartTimes.Length; i++)
                Console.Write(" " + frameStartTimes[i]);
            Console.WriteLine();
            Console.Write("   frameDurations:");
            for(int i = 0; i < frameDurations.Length; i++)
                Console.Write(" " + frameDurations[i]);
            Console.WriteLine();
        }
        /// <summary>
        /// Evaluates if two dynamic images are equal.
        /// Currently only frame times are evaluated in addition to 
        /// Image evaluation.
        /// </summary>
        /// <see cref="Image"/>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if files have equals data</returns>
        public override bool Equals(object obj)
        {
            if (obj is DynamicImage)
            {
                DynamicImage img = (obj as DynamicImage);
                for (int i = 0; i < frameStartTimes.Length; i++) {
                    if (frameStartTimes[i] != img.frameStartTimes[i]) return false;
                    if (frameDurations[i] != img.frameDurations[i]) return false;
                }
                return base.Equals(img as Image);
            }
            else
            {
                return false;
            }
        }
        /// <summary>
        /// Override that just calls base class method.
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        /// <summary>
        /// Multiply operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">multiplicator</param>
        /// <returns>multiplied image</returns>
        public static DynamicImage operator *(DynamicImage image, float value)
        {
            DynamicImage r = image;
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] * value;
            return r;
        }
        /// <summary>
        /// Addition operator for image.
        /// </summary>
        /// <param name="image">target image.</param>
        /// <param name="value">addition</param>
        /// <returns>resulting image</returns>
        public static DynamicImage operator +(DynamicImage image, float value)
        {
            DynamicImage r = image;
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] + value;
            return r;
        }
        /// <summary>
        /// Decrement operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">increment</param>
        /// <returns>resulting image</returns>
        public static DynamicImage operator +(DynamicImage image, Image value)
        {
            if (image.dim != value.dim)
                throw new TPCException("Cannot apply arithmetic + operation because image sizes are not the same");
            DynamicImage r = image;
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] + value[i];
            return r;
        }
    }
}
