﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    /// <summary>
    /// Common header information to all images.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class DynamicImageHeader : ImageHeader
    {
        /// <summary>
        /// Dose start time in milliseconds.
        /// </summary>
        public DateTime dose_start_time;
        /// <summary>
        /// Injected dose amount in MBq
        /// </summary>
        public float injected_dose;
        /// <summary>
        /// Radiopharmaceutical name (DEFAULT == unknown)
        /// </summary>
        public string radiopharma = "<unknown>";
        /// <summary>
        /// Isotope of image (DEFAULT == no isotope)
        /// </summary>
        public Isotope_enumerator isotope = Isotope_enumerator.No_isotope;
        /// <summary>
        /// frames (time dimension)
        /// </summary>
        public int frames
        {
            get { return dim.GetDimension(IntLimits.FRAMES); }
        }
        /// <summary>
        /// t-dimension. (time dimension)
        /// </summary>
        public int dimt
        {
            get { return dim.GetDimension(IntLimits.FRAMES); }
        }
        /// <summary>
        /// Constructs dynamic header from image header
        /// </summary>
        public DynamicImageHeader(DynamicImageHeader header)
            : base(header.patient_name, header.patientID, header.dim, header.siz, header.description, header.datatype)
        {
            this.dose_start_time = header.dose_start_time;
            this.injected_dose = header.injected_dose;
            this.isotope = header.isotope;
            this.AddRange(header);
            this.modality = header.modality;
            this.AddRange(header);
        }
        /// <summary>
        /// Constructs dynamic header from image header
        /// </summary>
        public DynamicImageHeader(ImageHeader header)
            : base(header.patient_name, header.patientID, header.dim, header.siz, header.description, header.datatype)
        {
            if (header is DynamicImageHeader)
            {
                this.dose_start_time = (header as DynamicImageHeader).dose_start_time;
                this.injected_dose = (header as DynamicImageHeader).injected_dose;
                this.isotope = (header as DynamicImageHeader).isotope;
                this.radiopharma = (header as DynamicImageHeader).radiopharma;
            }
            else {
                this.dose_start_time = new DateTime(0);
                this.injected_dose = 0.0f;
                this.isotope = Isotope_enumerator.No_isotope;
            }
            this.AddRange(header);
            this.modality = header.modality;
        }
        /// <summary>
        /// Default constructor
        /// </summary>
        public DynamicImageHeader()
            : base()
        {
            this.dose_start_time = new DateTime(0);
            this.patient_name = "<no name>";
            this.injected_dose = 0.0f;
            this.isotope = Isotope_enumerator.No_isotope;
            this.dim.AddLimit(0,0);
        }
        /// <summary>
        /// Returns dynamic image header as string
        /// </summary>
        /// <returns>string representation</returns>
        public override string ToString()
        {
            return "DynamicImageHeader[patient_name=" + patient_name +
                ",patientID=" + patientID +
                "," + dim +
                "," + siz +
                ",datatype=" + datatype +
                ",dose start time (ms)=" + dose_start_time +
                ",injected dose(MBq)=" + injected_dose +
                ",isotope=" + isotope + "]";
        }
        /// <summary>
        /// Evaluates if two objects are equal
        /// </summary>
        /// <param name="obj">evaluated obejct</param>
        /// <returns>true if headers are equal</returns>
        public override bool Equals(object obj)
        {
            DynamicImageHeader h;
            if (obj is DynamicImageHeader)
            {
                h = (obj as DynamicImageHeader);
                if (h.isotope != isotope) return false;
                if (h.radiopharma != radiopharma) return false;
                if (h.dose_start_time != dose_start_time) return false;
                if (h.injected_dose != injected_dose) return false;
                return base.Equals(obj as ImageHeader);
            }
            else
            {
                return false;
            }
        }
        /// <summary>
        /// Override that just calls base class method.
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
