/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.IO;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Image
{
    /// <summary>
    /// Class representing Ecat7 file.
    /// </summary>
    public class Ecat7File : ImageFile
    {
        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public event IOProcessEventHandler IOProgress;
        /// <summary>
        /// General upper class for all subheaders.
        /// </summary>
        public class Ecat7_subheader {
            /// <summary>
            /// Total duration of current frame (in msec)
            /// </summary>
            public int frame_duration = 0;
            /// <summary>
            /// Frame start time (offset from first frame, in msec)
            /// </summary>
            public int frame_start_time = 0;
            /// <summary> 
            /// If data type is integer, this factor is used to 
            /// convert to float values
            /// </summary>
            public float scale_factor = 0.0f;
        }
        /// <summary>
        /// The main header is the first block of a matrix file. Since the 
        /// main header always is the same length (1 block = 512 bytes) 
        /// for any matrix file, no entry exists for it in the directory list. 
        /// This structure provides storage space for information common to all matrices in the file.
        /// </summary>
        public class Ecat7_mainheader
        {  /* 512 bytes */
            /// <summary>
            /// Unix file type indentification number
            /// </summary>
            public char[] magic_number = new char[14];
            /// <summary>
            /// Scan file's creation number
            /// </summary>
            public char[] original_file_name = new char[32];
            /// <summary>
            /// Software version number 
            /// </summary>
            public short sw_version;
            /// <summary>
            /// Scanner model 
            /// </summary>
            public short system_type;
            /// <summary>
            /// Matrix file type (short)
            /// </summary>
            public datatype_e file_type;
            /// <summary>
            /// Serial number of the gantry 
            /// </summary>
            public char[] serial_number = new char[10];
            /// <summary>
            /// Date and time when acquisition was started (sec from base time) 
            /// </summary>
            public uint scan_start_time;
            /// <summary>
            /// String representation of the isotope 
            /// </summary>
            public char[] isotope_name = new char[8];
            /// <summary>
            /// Half-life of isotope (sec) 
            /// </summary>
            public float isotope_halflife;
            /// <summary>
            /// String representation of the tracer name 
            /// </summary>
            public char[] radiopharmaceutical = new char[32];
            /// <summary>
            /// Angle (degrees) 
            /// </summary>
            public float gantry_tilt;
            /// <summary>
            /// Angle (degrees) 
            /// </summary>
            public float gantry_rotation;
            /// <summary>
            /// Bed height from lowest point (cm) 
            /// </summary>
            public float bed_elevation;
            /// <summary>
            /// Angle that the first detector of Bucket 0 is offset from top center (in degrees) 
            /// </summary>
            public float intrinsic_tilt;
            /// <summary>
            /// Revolutions/minute (0 if not wobbled) 
            /// </summary>
            public short wobble_speed;
            /// <summary>
            /// Enumerated type (SRC_NONE, _RRING, _RING,_ROD, _RROD) (short)
            /// </summary>
            public Transm_source_type_e transm_source_type;
            /// <summary>
            /// Total distance scanned (cm) 
            /// </summary>
            public float distance_scanned;
            /// <summary>
            /// Diameter of transaxial view (cm) 
            /// </summary>
            public float transaxial_fov;
            /// <summary>
            /// 0=no mash, 1=mash of 2, 2=mash of 4 
            /// </summary>
            public short angular_compression;
            /// <summary>
            /// 0=Net trues, 1=Prompts and Delayed, 3=Prompts, Delayed, and Multiples 
            /// </summary>
            public short coin_samp_mode;
            /// <summary>
            /// 0=Normal, 1=2X, 2=3X 
            /// </summary>
            public short axial_samp_mode;
            /// <summary>
            /// Ecat calibration factor. 
            /// </summary>
            public float ecat_calibration_factor;
            /// <summary>
            /// 0=Uncalibrated; 1=Calibrated; 2=Processed 
            /// </summary>
            public short calibration_units;
            /// <summary>
            /// Whether data_units[] is filled or not? 
            /// </summary>
            public short calibration_units_label;
            /// <summary>
            /// Enumerated type (COMP_NONE, (This is the only value)) (short)
            /// </summary>
            public Compression_code_e compression_code;
            /// <summary>
            /// Study descriptor 
            /// </summary>
            public char[] study_type = new char[12];
            /// <summary>
            /// Patient identification descriptor 
            /// </summary>
            public char[] patient_id = new char[16];
            /// <summary>
            /// Patient name (free format ASCII) 
            /// </summary>
            public char[] patient_name = new char[32];
            /// <summary>
            /// Enumerated type (SEX_MALE, _FEMALE,_UNKNOWN) (char)
            /// </summary>
            public Patient_sex_e patient_sex;
            /// <summary>
            /// Enumerated type (DEXT_RT, _LF, _UNKNOWN) (char)
            /// </summary>
            public char patient_dexterity;
            /// <summary>
            /// Patient age (years) 
            /// </summary>
            public float patient_age;
            /// <summary>
            /// Patient height (cm) 
            /// </summary>
            public float patient_height;
            /// <summary>
            /// Patient weight (kg) 
            /// </summary>
            public float patient_weight;
            /// <summary>
            /// YYYYMMDD (sec from time zero) 
            /// </summary>
            public int patient_birth_date;
            /// <summary>
            /// Attending Physician name (free format) 
            /// </summary>
            public char[] physician_name = new char[32];
            /// <summary>
            /// Operator name (free format) 
            /// </summary>
            public char[] operator_name = new char[32];
            /// <summary>
            /// Free format ASCII 
            /// </summary>
            public char[] study_description = new char[32];
            /// <summary>
            /// Enumerated type (0=Undefined, 1=Blank, 2=Transmission, 3=Static emission, 4=Dynamic
            /// emission, 5=Gated emission, 6=Transmission rectilinear, 7=Emission rectilinear) 
            /// </summary>
            public Acquisition_type_e acquisition_type;
            /// <summary>
            /// Enumerated Type (Bit 0 clear - Feet first, Bit 0 set - Head first, Bit 1-2 00 - Prone, 
            ///  Bit 1-2 01 - Supine, Bit 1-2 10 - Decubitus Right, Bit 1-2 11 - Decubitus Left) (short)
            ///  </summary>
            public Patient_orientation_e patient_orientation;
            /// <summary>
            /// Free format ASCII 
            /// </summary>
            public char[] facility_name = new char[20];
            /// <summary>
            /// Number of planes of data collected 
            /// </summary>
            public short num_planes;
            /// <summary>
            /// Number of frames of data collected OR Highest frame number (in partially reconstructed files) 
            /// </summary>
            public short num_frames;
            /// <summary>
            /// Number of gates of data collected 
            /// </summary>
            public short num_gates;
            /// <summary>
            /// Number of bed positions of data collected 
            /// </summary>
            public short num_bed_pos;
            /// <summary>
            /// Absolute location of initial bed position (in cm.) 
            /// </summary>
            public float init_bed_position;
            /// <summary>
            /// Absolute bed location (in cm.) 
            /// </summary>
            public float[] bed_position = new float[15];
            /// <summary>
            /// Physical distance between adjacent planes (cm) 
            /// </summary>
            public float plane_separation;
            /// <summary>
            /// Lowest threshold setting for scatter (in KeV) 
            /// </summary>
            public short lwr_sctr_thres;
            /// <summary>
            /// Lower threshold setting for trues in (in KeV) 
            /// </summary>
            public short lwr_true_thres;
            /// <summary>
            /// Upper threshold setting for trues (in KeV) 
            /// </summary>
            public short upr_true_thres;
            /// <summary>
            /// Data processing code (defined by user) 
            /// </summary>
            public char[] user_process_code = new char[10];
            /// <summary>
            /// Enumerated Type (0=Normal, 1=Windowed, 2=Windowed and Nonwindowed, 3=Dual energy,
            /// 4=Upper energy, 5=Emission and Transmission) (short)
            /// </summary>
            public Acquisition_mode_e acquisition_mode;
            /// <summary>
            /// Width of view sample (cm) 
            /// </summary>
            public float bin_size;
            /// <summary>
            /// Fraction of decay by positron emission 
            /// </summary>
            public float branching_fraction;
            /// <summary>
            /// Actual time radiopharmaceutical was injected or flow was started (in sec since base time) 
            /// </summary>
            public int dose_start_time;
            /// <summary>
            /// Radiopharmaceutical dosage at time of injection (Bq/cc) 
            /// </summary>
            public float dosage;
            /// <summary>
            /// TBD 
            /// </summary>
            public float well_counter_corr_factor;
            /// <summary>
            /// Free text field; fixed strings: "ECAT counts/sec", "Bq/cc" 
            /// </summary>
            public char[] data_units = new char[32];
            /// <summary>
            /// Septa position during scan (0=septa extended, 1=septa retracted) 
            /// </summary>
            public short septa_state;
            /// <summary>
            /// CTI Reserved space (12 bytes)
            /// </summary>
            public short[] fill = new short[6];
            /// <summary>
            /// Resolves minimum and maximum for dataype
            /// </summary>
            /// <returns></returns>
            public int[] resolveMinMax()
            {
                switch (file_type)
                {
                    case datatype_e.ecat7_IMAGE8:
                    case datatype_e.ecat7_3DSCAN8:
                        return new int[] { -128, 127 };
                    case datatype_e.ecat7_VOLUME8:
                        return new int[] { 0, 256};
                    case datatype_e.ecat7_IMAGE16:
                    case datatype_e.ecat7_PROJ16:
                        return new int[] { -32768, 32767 };
                    case datatype_e.ecat7_VOLUME16:
                        return new int[] { 0, 32767 };
                    case datatype_e.ecat7_POLARMAP:
                    case datatype_e.ecat7_2DSCAN:
                    case datatype_e.ecat7_3DSCANFIT:
                    case datatype_e.ecat7_ATTEN:
                    case datatype_e.ecat7_2DNORM:
                    case datatype_e.ecat7_PROJ:
                    case datatype_e.ecat7_3DSCAN:
                    case datatype_e.ecat7_3DNORM:
                    case datatype_e.ecat7_UNKNOWN:
                    default:
                        return new int[] { 0, 0 };
                }
            }
            /// <summary>
            /// Resolves imagetype from mainheader
            /// </summary>
            /// <returns>imagetype</returns>
            public ecat7_imagetype resolveImageType()
            {
                switch (file_type)
                {
                    case datatype_e.ecat7_IMAGE8:
                    case datatype_e.ecat7_VOLUME8:
                    case datatype_e.ecat7_IMAGE16:
                    case datatype_e.ecat7_VOLUME16:
                        return ecat7_imagetype.IMAGE;
                    case datatype_e.ecat7_POLARMAP:
                        return ecat7_imagetype.POLARMAP;
                    case datatype_e.ecat7_2DSCAN:                        
                    case datatype_e.ecat7_3DSCAN8:
                    case datatype_e.ecat7_3DSCANFIT:
                        return ecat7_imagetype.RAW;
                    case datatype_e.ecat7_ATTEN:
                    case datatype_e.ecat7_2DNORM:
                    case datatype_e.ecat7_PROJ:
                    case datatype_e.ecat7_PROJ16:
                    case datatype_e.ecat7_3DSCAN:
                    case datatype_e.ecat7_3DNORM:
                    case datatype_e.ecat7_UNKNOWN:                
                    default:
                        return ecat7_imagetype.IMAGE;
                }
            }
            /// <summary>
            /// Resolves file type from datatype.
            /// </summary>
            /// <param name="datatype">datatype</param>
            /// <returns>filetype enumeration value</returns>
            public datatype_e resolveFileType(ImageFile.DataType datatype)
            {
                switch (datatype)
                {
                    case ImageFile.DataType.BIT1:
                        return datatype_e.ecat7_UNKNOWN;
                    case ImageFile.DataType.BIT8_S:
                    case ImageFile.DataType.BIT8_U:
                        return datatype_e.ecat7_VOLUME8;
                    case ImageFile.DataType.BIT16_S:
                    case ImageFile.DataType.BIT16_U:
                        return datatype_e.ecat7_VOLUME16;
                    case ImageFile.DataType.BIT32_S:
                    case ImageFile.DataType.BIT32_U:
                    case ImageFile.DataType.BIT64_S:
                    case ImageFile.DataType.BIT64_U:
                    case ImageFile.DataType.FLT32:
                    case ImageFile.DataType.FLT64:
                    case ImageFile.DataType.ASCII:
                    case ImageFile.DataType.VAXI16:
                    case ImageFile.DataType.VAXI32:
                    case ImageFile.DataType.VAXFL32:
                    case ImageFile.DataType.COLRGB:
                    case ImageFile.DataType.SUNI2:
                    case ImageFile.DataType.SUNI4:    
                        return datatype_e.ecat7_UNKNOWN;                
                }
                return datatype_e.ecat7_UNKNOWN;
            }
            /// <summary>
            /// Resolves general datatype from mainheader
            /// </summary>
            /// <returns>datatype</returns>
            public DataType resolveDataType() {
                switch(file_type) {
                    case datatype_e.ecat7_VOLUME8:
                    case datatype_e.ecat7_IMAGE8:
                    case datatype_e.ecat7_3DSCAN8:
                        return DataType.BIT8_S;
                    case datatype_e.ecat7_UNKNOWN:
                    case datatype_e.ecat7_2DSCAN:
                    case datatype_e.ecat7_IMAGE16:
                    case datatype_e.ecat7_ATTEN:
                    case datatype_e.ecat7_2DNORM:
                    case datatype_e.ecat7_POLARMAP:
                    case datatype_e.ecat7_VOLUME16:
                    case datatype_e.ecat7_PROJ:
                    case datatype_e.ecat7_PROJ16:
                    case datatype_e.ecat7_3DSCAN:
                    case datatype_e.ecat7_3DNORM:
                    case datatype_e.ecat7_3DSCANFIT:
                        return DataType.BIT16_S;
                }
                return DataType.BIT16_S;
            }
            /// <summary>
            /// Ecat7 datatype
            /// </summary>
            public enum datatype_e
            {
                /// <summary>
                /// Unknown type
                /// </summary>
                ecat7_UNKNOWN = 0,
                /// <summary>
                /// 2D scan
                /// </summary>
                ecat7_2DSCAN = 1,
                /// <summary>
                /// 16-bit image
                /// </summary>
                ecat7_IMAGE16 = 2,
                /// <summary>
                /// Attenuation image
                /// </summary>
                ecat7_ATTEN = 3,
                /// <summary>
                /// 2D normalization
                /// </summary>
                ecat7_2DNORM = 4,
                /// <summary>
                /// Polarmap
                /// </summary>
                ecat7_POLARMAP = 5,
                /// <summary>
                /// 8-bit volume
                /// </summary>
                ecat7_VOLUME8 = 6,
                /// <summary>
                /// 16-bit volume
                /// </summary>
                ecat7_VOLUME16 = 7,
                /// <summary>
                /// Projection
                /// </summary>
                ecat7_PROJ = 8,
                /// <summary>
                /// 16-bit Projection
                /// </summary>
                ecat7_PROJ16 = 9,
                /// <summary>
                /// 8-bit image
                /// </summary>
                ecat7_IMAGE8 = 10,
                /// <summary>
                /// 3D scan
                /// </summary>
                ecat7_3DSCAN = 11,
                /// <summary>
                /// 8-bit 3D scan
                /// </summary>
                ecat7_3DSCAN8 = 12,
                /// <summary>
                /// 3D normalization
                /// </summary>
                ecat7_3DNORM = 13,
                /// <summary>
                /// 3D scan (fit)
                /// </summary>
                ecat7_3DSCANFIT = 14
            }
            /// <summary>
            /// Transmission source type
            /// </summary>
            public enum Transm_source_type_e {
                /// <summary>
                /// No source
                /// </summary>
                SRC_NONE,
                /// <summary>
                /// R-ring
                /// </summary>
                RRING,
                /// <summary>
                /// ring
                /// </summary>
                _RING,
                /// <summary>
                /// rod
                /// </summary>
                _ROD,
                /// <summary>
                /// prod
                /// </summary>
                _RROD
            }
            /// <summary>
            /// Compression code type (allways = COMP_NONE)
            /// </summary>
            public enum Compression_code_e {
                /// <summary>
                /// no compression
                /// </summary>
                COMP_NONE
            }
            /// <summary>
            /// Patient sex type
            /// </summary>
            public enum Patient_sex_e {
                /// <summary>
                /// Male
                /// </summary>
                SEX_MALE, 
                /// <summary>
                /// Female
                /// </summary>
                _FEMALE, 
                /// <summary>
                /// Unknown
                /// </summary>
                _UNKNOWN
            }
            /// <summary>
            /// Acquisition type
            /// </summary>
            public enum Acquisition_type_e {
                /// <summary>
                /// Unknown
                /// </summary>
                Undefined = 0, 
                /// <summary>
                /// Blank
                /// </summary>
                Blank, 
                /// <summary>
                /// Transmission scan
                /// </summary>
                Transmission, 
                /// <summary>
                /// Static emission
                /// </summary>
                Static_emission, 
                /// <summary>
                /// Dynamic emission
                /// </summary>
                Dynamic_emission, 
                /// <summary>
                /// Gated acquisition
                /// </summary>
                Gated_emission, 
                /// <summary>
                /// Rectilinear transmission
                /// </summary>
                Transmission_rectilinear, 
                /// <summary>
                /// Rectilinear emission
                /// </summary>
                Emission_rectilinear
            }
            /// <summary>
            /// Patient orientation
            /// </summary>
            public enum Patient_orientation_e { 
                /// <summary>
                /// Feet first prone
                /// </summary>
                Feet_first_Prone = 0x000,
                /// <summary>
                /// Feet first supine
                /// </summary>
                Feet_first_Supine = 0x010,
                /// <summary>
                /// Feet first decubitus right
                /// </summary>
                Feet_first_Decubitus_Right = 0x100,
                /// <summary>
                /// Feet first decubitus right
                /// </summary>
                Feet_first_Decubitus_Left = 0x110,
                /// <summary>
                /// Head first prone
                /// </summary>
                Head_first_Prone = 0x001,
                /// <summary>
                /// Head first supine
                /// </summary>
                Head_first_Supine = 0x011,
                /// <summary>
                /// Head first decubitus right
                /// </summary>
                Head_first_Decubitus_Right = 0x101,
                /// <summary>
                /// Head first decubitus left
                /// </summary>
                Head_first_Decubitus_Left = 0x111,
            }
            /// <summary>
            /// Acquisition mode
            /// </summary>
            public enum Acquisition_mode_e { 
                /// <summary>
                /// Normal mode
                /// </summary>
                Normal = 0,
                /// <summary>
                /// Windowed mode
                /// </summary>
                Windowed, 
                /// <summary>
                /// Windowed or non-windowed mode
                /// </summary>
                Windowed_and_Nonwindowed, 
                /// <summary>
                /// Dual energy mode
                /// </summary>
                Dual_energy,
                /// <summary>
                /// Upper energy mode
                /// </summary>
                Upper_energy, 
                /// <summary>
                /// Emission or transmission mode
                /// </summary>
                Emission_and_Transmission
            }
            /// <summary>
            /// Ecat7 image type (raw, image or polarmap)
            /// </summary>
            public enum ecat7_imagetype
            {
                /// <summary>
                /// Raw data image
                /// </summary>
                RAW,
                /// <summary>
                /// Intensity image
                /// </summary>
                IMAGE,
                /// <summary>
                /// Polar map
                /// </summary>
                POLARMAP
            }
            /// <summary>
            /// Default constructor.
            /// </summary>
            public Ecat7_mainheader() {
                "MATRIX70v".CopyTo(0, magic_number, 0, 9);
                "Unknown".CopyTo(0, study_type, 0, 7);
                "Unknown".CopyTo(0, patient_id, 0, 7);
                "Unknown".CopyTo(0, patient_name, 0, 7);
                file_type = datatype_e.ecat7_UNKNOWN;
                sw_version = 70;
                ecat_calibration_factor = 1.0f;
                calibration_units = 2;
                patient_orientation = Patient_orientation_e.Head_first_Prone;
                "<undefined>".CopyTo(0, radiopharmaceutical, 0, 11);
                "<no original filename>".CopyTo(0, original_file_name , 0, 22);                
            }
        }
        /// <summary>
        /// Subheader for Matrix Image Files
        /// </summary>
        public class Ecat7_imageheader : Ecat7_subheader
        {
            /// <summary>
            /// Enumerated type (short)
            /// </summary>
            public data_type_e data_type = data_type_e.Unknown_Matrix_Data_Type;
            /// <summary>
            /// Number of dimensions
            /// </summary>
            public short num_dimensions = 0;
            /// <summary>
            /// Dimension along x axis
            /// </summary>
            public short x_dimension = 0;
            /// <summary>
            /// Dimension along y axis 
            /// </summary>
            public short y_dimension = 0;
            /// <summary>
            /// Dimension along z axis 
            /// </summary>
            public short z_dimension = 0;
            /// <summary>
            /// Offset in x axis for recon target (in cm) 
            /// </summary>
            public float x_offset = 0.0f;
            /// <summary>
            /// Offset in y axis for recon target (in cm.) 
            /// </summary>
            public float y_offset = 0.0f;
            /// <summary>
            /// Offset in z axis for recon target (in cm.) 
            /// </summary>
            public float z_offset = 0.0f;
            /// <summary>
            /// Reconstruction magnification factor 
            /// </summary>
            public float recon_zoom = 0.0f;
            /// <summary>
            /// Image minimum pixel value 
            /// </summary>
            public short image_min = 0;
            /// <summary>
            /// Image maximum pixel value 
            /// </summary>
            public short image_max = 0;
            /// <summary>
            /// X dimension pixel size (cm) 
            /// </summary>
            public float x_pixel_size = 0.0f;
            /// <summary>
            /// Y dimension pixel size (cm) 
            /// </summary>
            public float y_pixel_size = 0.0f;
            /// <summary>
            /// Z dimension pixel size (cm) 
            /// </summary>
            public float z_pixel_size = 0.0f;
            /// <summary>
            /// filter code (short)
            /// </summary>
            public filter_code_e filter_code = filter_code_e.all_pass;
            /// <summary>
            /// Resolution in the x dimension (in cm) 
            /// </summary>
            public float x_resolution = 0.0f;
            /// <summary>
            /// Resolution in the y dimension (in cm) 
            /// </summary>
            public float y_resolution = 0.0f;
            /// <summary>
            /// Resolution in the z dimension (in cm) 
            /// </summary>
            public float z_resolution = 0.0f;
            /// <summary>
            /// Number R elements from sinogram 
            /// </summary>
            public float num_r_elements = 0.0f;
            /// <summary>
            /// Nr of angles from sinogram 
            /// </summary>
            public float num_angles = 0.0f;
            /// <summary>
            /// Rotation in the xy plane (in degrees). Use righthand coordinate system for rotation angle sign. 
            /// </summary>
            public float z_rotation_angle = 0.0f;
            /// <summary>
            /// Isotope decay compensation applied to data 
            /// </summary>
            public float decay_corr_fctr = 0.0f;
            /// <summary>
            /// Bit mask (0=Not Processed, 1=Normalized, 2=Measured Attenuation Correction,
            /// 4=Calculated Attenuation Correction, 8=X smoothing, 16=Y smoothing, 32=Z smoothing,
            /// 64=2D scatter correction, 128=3D scatter correction, 256=Arc correction, 512=Decay
            /// correction, 1024=Online compression) 
            /// </summary>
            public uint processing_code = 0;
            /// <summary>
            /// Gate duration (in msec) 
            /// </summary>
            public int gate_duration = 0;
            /// <summary>
            /// R wave offset (For phase sliced studies, average, in msec) 
            /// </summary>
            public int r_wave_offset = 0;
            /// <summary>
            /// Number of accepted beats for this gate 
            /// </summary>
            public int num_accepted_beats = 0;
            /// <summary>
            /// Cutoff frequency 
            /// </summary>
            public float filter_cutoff_frequency = 0.0f;
            /// <summary>
            /// Do not use 
            /// </summary>
            public float filter_resolution = 0.0f;
            /// <summary>
            /// Do not use 
            /// </summary>
            public float filter_ramp_slope = 0.0f;
            /// <summary>
            /// Do not use 
            /// </summary>
            public short filter_order = 0;
            /// <summary>
            /// Do not use 
            /// </summary>
            public float filter_scatter_fraction = 0.0f;
            /// <summary>
            /// Do not use 
            /// </summary>
            public float filter_scatter_slope = 0.0f;
            /// <summary>
            /// Free format ASCII 
            /// </summary>
            public char[] annotation = new char[40];
            /// <summary>
            /// Matrix transformation element (1,1) 
            /// </summary>
            public float mt_1_1 = 0.0f;
            /// <summary>
            /// Matrix transformation element (1,2) 
            /// </summary>
            public float mt_1_2 = 0.0f;
            /// <summary>
            /// Matrix transformation element (1,3) 
            /// </summary>
            public float mt_1_3 = 0.0f;
            /// <summary>
            /// Matrix transformation element (2,1) 
            /// </summary>
            public float mt_2_1 = 0.0f;
            /// <summary>
            /// Matrix transformation element (2,2) 
            /// </summary>
            public float mt_2_2 = 0.0f;
            /// <summary>
            /// Matrix transformation element (2,3) 
            /// </summary>
            public float mt_2_3 = 0.0f;
            /// <summary>
            /// Matrix transformation element (3,1) 
            /// </summary>
            public float mt_3_1 = 0.0f;
            /// <summary>
            /// Matrix transformation element (3,2) 
            /// </summary>
            public float mt_3_2 = 0.0f;
            /// <summary>
            /// Matrix transformation element (3,3) 
            /// </summary>
            public float mt_3_3 = 0.0f;
            /// <summary>
            /// Reconstruction filter cutoff 
            /// </summary>
            public float rfilter_cutoff = 0.0f;
            /// <summary>
            /// Reconstruction filter resolution  
            /// </summary>
            public float rfilter_resolution = 0.0f;
            /// <summary>
            /// Reconstruction filter code 
            /// </summary>
            public short rfilter_code = 0;
            /// <summary>
            /// Reconstruction filter order 
            /// </summary>
            public short rfilter_order = 0;
            /// <summary>
            /// 
            /// </summary>
            public float zfilter_cutoff = 0.0f;
            /// <summary>
            /// 
            /// </summary>
            public float zfilter_resolution = 0.0f;
            /// <summary>
            /// 
            /// </summary>
            public short zfilter_code = 0;
            /// <summary>
            /// 
            /// </summary>
            public short zfilter_order = 0;
            /// <summary>
            /// Matrix transformation element (1,4) 
            /// </summary>
            public float mt_1_4 = 0.0f;
            /// <summary>
            /// Matrix transformation element (2,4) 
            /// </summary>
            public float mt_2_4 = 0.0f;
            /// <summary>
            /// Matrix transformation element (3,4) 
            /// </summary>
            public float mt_3_4 = 0.0f;
            /// <summary>
            /// Scatter type (short)
            /// </summary>
            public scatter_type_e scatter_type = scatter_type_e.None;
            /// <summary>
            /// Reconstruction type (short)
            /// </summary>
            public recon_type_e recon_type = recon_type_e.Filtered_backprojection;
            /// <summary>
            /// Number of views used to reconstruct the data 
            /// </summary>
            public short recon_views = 0;
            /// <summary>
            /// CTI Reserved space (174 bytes) 
            /// </summary>
            public short[] fill_cti = new short[87];
            /// <summary>
            /// User Reserved space (100 bytes) Note: Use highest bytes first 
            /// </summary>
            public short[] fill_user = new short[48];
            /// <summary>
            /// Filtering type code
            /// </summary>
            public enum filter_code_e {
                /// <summary>
                /// No filtering
                /// </summary>
                all_pass = 0, 
                /// <summary>
                /// Ramp filtering
                /// </summary>
                ramp, 
                /// <summary>
                /// Butterworth filtering
                /// </summary>
                Butterworth, 
                /// <summary>
                /// Hanning filter
                /// </summary>
                Hanning, 
                /// <summary>
                /// Hamming filter
                /// </summary>
                Hamming,
                /// <summary>
                /// Perzen filter
                /// </summary>
                Parzen,
                /// <summary>
                /// Shepp filter
                /// </summary>
                Shepp,
                /// <summary>
                /// Butterworth filter of order 2
                /// </summary>
                Butterworth_order_2, 
                /// <summary>
                /// Gaussian filter
                /// </summary>
                Gaussian, 
                /// <summary>
                /// Median filter
                /// </summary>
                Median,
                /// <summary>
                /// Boxcar filter
                /// </summary>
                Boxcar
            }
            /// <summary>
            /// Scattering type
            /// </summary>
            public enum scatter_type_e {
                /// <summary>
                /// No scatter
                /// </summary>
                None = 0,
                /// <summary>
                /// Deconvolution scatter
                /// </summary>
                Deconvolution,
                /// <summary>
                /// Simulated scatter
                /// </summary>
                Simulated,
                /// <summary>
                /// Dual energy scatter
                /// </summary>
                Dual_Energy
            }
            /// <summary>
            /// Reconstruction type
            /// </summary>
            public enum recon_type_e {
                /// <summary>
                /// Filtered backprojection 
                /// </summary>
                Filtered_backprojection = 0,
                /// <summary>
                /// Forward projection
                /// </summary>
                Forward_projection_3D_PROMIS,
                /// <summary>
                /// 3D Ramp
                /// </summary>
                Ramp_3D,
                /// <summary>
                /// 3D FAVOR
                /// </summary>
                FAVOR_3D,
                /// <summary>
                /// SSRB
                /// </summary>
                SSRB,
                /// <summary>
                /// Multi-slice rebinning
                /// </summary>
                Multi_slice_rebinning,
                /// <summary>
                /// FORE
                /// </summary>
                FORE
            }
            /// <summary>
            /// Converts data interpretation type into string
            /// </summary>
            /// <param name="type">data type</param>
            /// <returns>string representation of type</returns>
            public static string ToString(data_type_e type) {
                switch (type)
                {
                    case data_type_e.Unknown_Matrix_Data_Type: return "unknown";
                    case data_type_e.Byte_Data: return "byte";
                    case data_type_e.VAX_Ix2: return "VAX 2 byte integer";
                    case data_type_e.VAX_Ix4: return "VAX 4 byte integer";
                    case data_type_e.VAX_Rx4: return "VAX 4 byte float";
                    case data_type_e.IEEE_Float: return "IEEE 4 byte float";
                    case data_type_e.Sun_short: return "SUN 2 byte integer";
                    case data_type_e.Sun_long: return "SUN 4 byte integer";
                }
                throw new TPCEcat7FileException("internal error: unknown enumerate");
            }
            /// <summary>
            /// Data interpretation type.
            /// </summary>
            public enum data_type_e { 
                /// <summary>
                /// Uknown
                /// </summary>
                Unknown_Matrix_Data_Type = 0,
                /// <summary>
                /// Byte data
                /// </summary>
                Byte_Data = 1,
                /// <summary>
                /// 2-byte VAX
                /// </summary>
                VAX_Ix2 = 2,
                /// <summary>
                /// 4-byte VAX
                /// </summary>
                VAX_Ix4 = 3,
                /// <summary>
                /// 4-byte realvalue VAX
                /// </summary>
                VAX_Rx4 = 4, 
                /// <summary>
                /// 32-bit IEEE float
                /// </summary>
                IEEE_Float = 5,
                /// <summary>
                /// 16-bit Sun
                /// </summary>
                Sun_short = 6,
                /// <summary>
                /// 32-bit Sun
                /// </summary>
                Sun_long = 7
            }
            /// <summary>
            /// Converts ecat7 imageheader datatype to general data type
            /// </summary>
            /// <param name="type"></param>
            /// <returns></returns>
            public static data_type_e convertFrom(DataType type)
            {
                switch (type)
                {
                    case DataType.BIT16_S:
                    case DataType.BIT16_U:
                        return data_type_e.Sun_short;
                    case DataType.BIT8_S:
                    case DataType.BIT8_U:
                        return data_type_e.Byte_Data;
                    case DataType.VAXI16:
                        return data_type_e.VAX_Ix2;
                    case DataType.VAXI32:
                        return data_type_e.VAX_Ix4;
                    case DataType.VAXFL32:
                        return data_type_e.VAX_Rx4;
                    case DataType.FLT32:
                        return data_type_e.IEEE_Float;
                    case DataType.SUNI2:
                        return data_type_e.Sun_short;
                    case DataType.BIT32_S:
                    case DataType.BIT32_U:
                    case DataType.SUNI4:
                        return data_type_e.Sun_long;
                }
                return data_type_e.Unknown_Matrix_Data_Type;
            }
            /// <summary>
            /// Converts ecat7 imageheader datatype to general data type
            /// </summary>
            /// <param name="type"></param>
            /// <returns></returns>
            public static DataType convertFrom(data_type_e type) {
                switch (type) {
                    case data_type_e.Unknown_Matrix_Data_Type:
                        //guessing..
                        return DataType.BIT16_S;
                    case data_type_e.Byte_Data:
                        return DataType.BIT8_S;
                    case data_type_e.VAX_Ix2:
                        return DataType.VAXI16;
                    case data_type_e.VAX_Ix4:
                        return DataType.VAXI32;
                    case data_type_e.VAX_Rx4:
                        return DataType.VAXFL32;
                    case data_type_e.IEEE_Float:
                        return DataType.FLT32;
                    case data_type_e.Sun_short:
                        return DataType.SUNI2;
                    case data_type_e.Sun_long:
                        return DataType.SUNI4;
                }
                return DataType.BIT16_S;
            }
            /// <summary>
            /// Constructs imageheader with default values
            /// </summary>
            public Ecat7_imageheader() {
                scale_factor = 1.0f;
                x_resolution = 0.0f;
                y_resolution = 0.0f;
                z_resolution = 0.0f;
                recon_zoom = 0.0f;
                frame_start_time = 1000;
                frame_duration = 0;
            }
        }
        /// <summary>
        /// Subheader for 3D Matrix Scan Files
        /// </summary>
        public sealed class Ecat7_scanheader : Ecat7_subheader
        {
            /// <summary>
            /// Resolves data type
            /// </summary>
            /// <returns>datatype</returns>
            public DataType resolveDataType()
            {
                switch (data_type)
                {
                    case data_type_e.ByteData: return DataType.BIT8_S;
                    case data_type_e.SunShortt: return DataType.BIT16_S;
                }
                return DataType.BIT16_S;
            }
            /// <summary>
            /// Data type.
            /// </summary>
            public enum data_type_e {
                /// <summary>
                /// Byte data
                /// </summary>
                ByteData, 
                /// <summary>
                /// Sun short
                /// </summary>
                SunShortt
            }
            /// <summary>
            /// Enumerated type.
            /// </summary>
            public data_type_e data_type;
            /// <summary>
            /// Number of Dimensions
            /// </summary>
            public short num_dimensions;
            /// <summary>
            /// Total elemenst collected (r dimension )
            /// </summary>
            public short num_r_elements;
            /// <summary>
            /// Total views collected (theta dimension)
            /// </summary>
            public short num_angles;
            /// <summary> 
            /// Designates processing applied to scan data 
            /// (Bit encoded, Bit 0 - Norm, Bit 1 - Atten, Bit 2 -Smooth)
            /// </summary>
            public short corrections_applied;
            /// <summary> 
            /// Total elements collected (z dimension) For 3D scans
            /// </summary>
            public short[] num_z_elements = new short[64];
            /// <summary> 
            /// Max ring difference (d dimension) in this frame 
            /// </summary>
            public short ring_difference;
            /// <summary> 
            /// Data storage order (r theta zd or rz theta d)
            /// </summary>
            public short storage_order;
            /// <summary> 
            /// Axial compression code or factor, generally 
            /// referred to as SPAN
            /// </summary>
            public short axial_compression;
            /// <summary> 
            /// Resolution in r dimension (in cm) 
            /// </summary>
            public float x_resolution;
            /// <summary> 
            /// Resolution in Theta dimension (in radians) 
            /// </summary>
            public float v_resolution;
            /// <summary> 
            /// Resolution in z dimension (in cm) 
            /// </summary>
            public float z_resolution;
            /// <summary> 
            /// Not Used
            /// </summary>
            public float w_resolution;
            /// <summary>
            /// RESERVED for gating 
            /// </summary>
            public short[] fill_gate = new short[6];
            /// <summary> 
            /// Gating segment length (msec, Average time if 
            /// phased gates are used)
            /// </summary>
            public int gate_duration;
            /// <summary> 
            /// Time from start of first gate (Average, in msec.)
            /// </summary>
            public int r_wave_offset;
            /// <summary> 
            /// Number of accepted beats for this gate
            /// </summary>
            public int num_accepted_beats;
            /// <summary> 
            /// Minimum value in sinogram if data is in integer form
            /// </summary>
            public short scan_min;
            /// <summary> 
            /// Maximum value in sinogram if data is in integer form
            /// </summary>
            public short scan_max;
            /// <summary> 
            /// Total prompts collected in this frame/gate
            /// </summary>
            public int prompts;
            /// <summary> 
            /// Total delays collected in this frame/gate
            /// </summary>
            public int delayed;
            /// <summary> 
            /// Total multiples collected in this frame/gate
            /// </summary>
            public int multiples;
            /// <summary> 
            /// Total net trues (promptsrandoms)
            /// </summary>
            public int net_trues;
            /// <summary> 
            /// Mean value of loss-corrected singles
            /// </summary>
            public float tot_avg_cor;
            /// <summary> 
            /// Mean value of singles (not loss corrected)
            /// </summary>
            public float tot_avg_uncor;
            /// <summary> 
            /// Measured coincidence rate (from IPCP)
            /// </summary>
            public int total_coin_rate;
            /// <summary> 
            /// Dead-time correction factor applied to the 
            /// sinogram
            /// </summary>
            public float deadtime_correction_factor;
            /// <summary>
            /// CTI Reserved space (180 bytes)
            /// </summary>
            public short[] fill_cti = new short[90];
            /// <summary> 
            /// User Reserved space (100 bytes) Note: Use 
            /// highest bytes first
            /// </summary>
            public short[] fill_user = new short[50];
            /// <summary> 
            /// Total uncorrected singles from each bucket
            /// </summary>
            public float[] uncor_singles = new float[128];
        }
        /// <summary>
        /// Subheader for Matrix Polar Map Files
        /// </summary>
        public class Ecat7_polmapheader : Ecat7_subheader
        {
            /// <summary>
            /// Resolves data type
            /// </summary>
            /// <returns>datatype</returns>
            public DataType resolveDataType()
            {
                switch (data_type)
                {
                    case data_type_e.DTYPE_BYTES: return DataType.BIT8_S;
                    case data_type_e._I2: return DataType.BIT16_S;
                    case data_type_e._I4: return DataType.FLT32;
                }
                return DataType.BIT16_S;
            }
            /// <summary>
            /// Enumerated type (DTYPE_BYTES, _I2,_I4)
            /// </summary>
            public enum data_type_e {
                /// <summary>
                /// Byte data
                /// </summary>
                DTYPE_BYTES, 
                /// <summary>
                /// 2-byte data
                /// </summary>
                _I2,
                /// <summary>
                /// 4-byte data
                /// </summary>
                _I4
            }
            /// <summary>
            /// Data type (short)
            /// </summary>
            public data_type_e data_type;
            /// <summary>
            /// Enumerated Type (Always 0 for now; denotes the version of the PM structure)
            /// </summary>
            public short polar_map_type;       
            /// <summary>
            /// Number of rings in this polar map
            /// </summary>
            public short num_rings;
            /// <summary>
            /// Number of sectors in each ring for up to 32 rings 
            /// (1, 9, 18, or 32 sectors normally)
            /// </summary>
            public short[] sectors_per_ring = new short[32];   
            /// <summary>
            /// Fractional distance along the long axis from base to apex
            /// </summary>
            public float[] ring_position = new float[32];     
            /// <summary>
            /// Ring angle relative to long axis(90 degrees along 
            /// cylinder, decreasing to 0 at the apex)
            /// </summary>
            public short[] ring_angle = new short[32];   
            /// <summary>
            /// START angle for rings (Always 258 degrees,
            /// defines Polar Maps 0)
            /// </summary>
            public short start_angle;   
            /// <summary>
            /// x, y, z location of long axis base end (in pixels)
            /// </summary>
            public short[] long_axis_left = new short[3];       
            /// <summary>
            /// x, y, z location of long axis apex end (in pixels)
            /// </summary>
            public short[] long_axis_right = new short[3];   
            /// <summary>
            /// Enumerated type (0 - Not available, 1 - Present)
            /// </summary>
            public short position_data;    
            /// <summary>
            /// Minimum pixel value in this polar map
            /// </summary>
            public short image_min; 
            /// <summary>
            /// Maximum pixel value in this polar map
            /// </summary>
            public short image_max;
            /// <summary>
            /// Pixel size (in cubic cm, represents voxels)
            /// </summary>
            public float pixel_size;   
            /// <summary>
            /// Bit Encoded (1- Map type (0 = Sector Analysis, 1 = Volumetric), 
            /// 2 - Threshold Applied, 
            /// 3 - Summed Map, 
            /// 4 - Subtracted Map, 
            /// 5 - Product of two maps, 
            /// 6 - Ratio of two maps, 
            /// 7 - Bias, 
            /// 8 - Multiplier, 
            /// 9 - Transform, 
            /// 10 - Polar Map calculational protocol used)
            /// </summary>
            public short processing_code;
            /// <summary>
            /// Enumerated Type (0 - Default (see main header),
            /// 1 - Normalized, 2 - Mean, 3 - Std. Deviation from Mean)
            /// </summary>
            public enum quant_units_e {
                /// <summary>
                /// Default
                /// </summary>
                Default = 0,
                /// <summary>
                /// Normalized
                /// </summary>
                Normalized = 1,
                /// <summary>
                /// Mean
                /// </summary>
                Mean = 2, 
                /// <summary>
                /// Standard deviation from mean
                /// </summary>
                Std_Deviation_from_Mean = 3
            }
            /// <summary>
            /// Quantitation units (short)
            /// </summary>
            public quant_units_e quant_units;
            /// <summary>
            /// Label for polar map display
            /// </summary>
            public char[] annotation = new char[40];
            /// <summary>
            /// Gate duration (in msec)
            /// </summary>
            public int gate_duration;
            /// <summary>
            /// R wave offset (Average, in msec)
            /// </summary>
            public int r_wave_offset;
            /// <summary>
            /// Number of accepted beats for this gate
            /// </summary>
            public int num_accepted_beats;
            /// <summary>
            /// Polar Map protocol used to generate this polar map
            /// </summary>
            public char[] polar_map_protocol = new char[20];
            /// <summary>
            /// Database name used for polar map comparison
            /// </summary>
            public char[] database_name = new char[30];  
            /// <summary>
            /// Reserved for future CTI use (54 bytes)
            /// </summary>
            public short[] fill_cti = new short[27];   
            /// <summary>
            /// User reserved space (54 bytes) Note: Use highest bytes first
            /// </summary>
            public short[] fill_user = new short[27];
        }
        /// <summary>
        /// Ecat7 matrix directory header. Entry 1 in matrix directory list. A double-linked list of 
        /// directory list blocks.
        /// </summary>
        protected struct ecat7_MatHdr
        {
            /// <summary>
            /// Number of free entries in list [0..31]
            /// </summary>
            public int nr_free;
            /// <summary>
            /// block number (1-based) of next directory list
            /// </summary>
            public int next_blk;
            /// <summary>
            /// block number (1-based) of previous directory list
            /// </summary>
            public int prev_blk;
            /// <summary>
            /// Number of used entries in list [0..31]
            /// </summary>
            public int nr_used;
            /// <summary>
            /// Constructs matrix header. 
            /// <remarks>nr_free + nr_used should be 31</remarks>
            /// </summary>
            /// <param name="nr_free">number of free items</param>
            /// <param name="next_blk">next block number</param>
            /// <param name="prev_blk">previous block number</param>
            /// <param name="nr_used">number of used items</param>
            public ecat7_MatHdr(int nr_free, int next_blk, int prev_blk, int nr_used)
            {                
                this.nr_free = nr_free;
                this.next_blk = next_blk;
                this.prev_blk = prev_blk;
                this.nr_used = nr_used;
            }
            /// <summary>
            /// MAtrix header as string
            /// </summary>
            /// <returns>string representation</returns>
            public override string ToString()
            {
                return "ecat7_MatHdr[nr_free=" + nr_free + " next_blk=" + next_blk + " prev_blk=" + prev_blk + " nr_used=" + nr_used + "]";
            }
        }
        /// <summary>
        /// Ecat7 matrix directory entry. Entries 1-31 in directory list.
        /// </summary>
        public struct Ecat7_MatDir {
            /// <summary>
            /// Matrix identifier (int)
            /// </summary>
            public Ecat7_Matval id;
            /// <summary>
            /// Matrix subheader record's block number.
            /// <see cref="MatBLKSIZE"/>
            /// </summary>
            public int strtblk;
            /// <summary>
            /// Last record's block number for matrix data block.
            /// <see cref="MatBLKSIZE"/>
            /// </summary>
            public int endblk;
            /// <summary>
            /// Matrix status (int)
            /// </summary>
            public status_e status;
            /// <summary>
            /// Constructs matrix directory item
            /// </summary>
            /// <param name="matval">matrix value</param>
            /// <param name="strtblk">data start block number</param>
            /// <param name="endblk">data end block number</param>
            /// <param name="status">data status</param>
            public Ecat7_MatDir(Ecat7_Matval matval, int strtblk, int endblk, status_e status)
            { 
                this.id = matval;
                this.strtblk = strtblk;
                this.endblk = endblk;
                this.status = status; 
            }
            /// <summary>
            /// Constructs matrix directory item
            /// </summary>
            /// <param name="id">matrix value code number</param>
            /// <param name="strtblk">data start block number</param>
            /// <param name="endblk">data end block number</param>
            /// <param name="status">data status</param>
            public Ecat7_MatDir(int id, int strtblk, int endblk, status_e status)
            { 
                this.id = new Ecat7_Matval(id);
                this.strtblk = strtblk;
                this.endblk = endblk;
                this.status = status; 
            }
            /// <summary>
            /// Status enumerator. EXITS = access read/write, DELETED = no access
            /// </summary>
            public enum status_e {
                /// <summary>
                /// Dat does not exist
                /// </summary>
                DELETED = 0,
                /// <summary>
                /// Data is not yet written. No need to use unless
                /// in multithreaded program.
                /// </summary>
                DATE_NOT_YET_WRITTEN = 2, 
                /// <summary>
                /// Data exists
                /// </summary>
                EXISTS = 1
            }
            /// <summary>
            /// Matrix entry as string
            /// </summary>
            /// <returns>string representation</returns>
            public override string ToString()
            {
                return "ecat7_MatDir[id=" + id + " strtblk=" + strtblk + " endblk=" + endblk + " status=" + status + "]";
            }
        }
        /// <summary>
        /// Class representing coded matrix id value that holds various matrix information.
        /// </summary>
        public class Ecat7_Matval {
            /// <summary>
            /// matrix id value containing various coded matrix information
            /// </summary>
            public int matrix_id;
            /// <summary>
            /// frame number [0..65536]
            /// </summary>
            public int frame { get { return (int)(matrix_id & 0x1FF); } }
            /// <summary>
            /// plane number [0..65536]
            /// </summary>
            public int plane { get { return (int)(((matrix_id >> 16) & 0xFF) + ((matrix_id >> 1) & 0x300)); } }
            /// <summary>
            /// gate number [0..64]
            /// </summary>
            public int gate { get { return (int)((matrix_id >> 24) & 0x3F); } }
            /// <summary>
            /// data number [0..1]
            /// </summary>
            public int data { get { return (int)(((matrix_id >> 30) & 0x3) + ((matrix_id >> 9) & 0x4)); } }
            /// <summary>
            /// bed number [0..16]
            /// </summary>
            public int bed { get { return (int)((matrix_id >> 12) & 0xF); } }
            /// <summary>
            /// Converts matrix-value.
            /// </summary>
            /// <param name="frame">frame number [0..65536]</param>
            /// <param name="plane">plane number [0..65536]</param>
            /// <param name="gate">[0..64]</param>
            /// <param name="data">[0..1]</param>
            /// <param name="bed">[0..16]</param>
            /// <returns>matrix id value</returns>
            public static int ecat7_val_to_id(int frame, int plane, int gate, int data, int bed) {
                return  ((bed & 0xF) << 12) |    /* bed */
                        (frame & 0x1FF) |        /* frame */
                        ((gate & 0x3F) << 24) |  /* gate */
                        ((plane & 0xFF) << 16) | /* plane low */
                        ((plane & 0x300) << 1) | /* plane high */
                        ((data & 0x3) << 30) |   /* data low */
                        ((data & 0x4) << 9);      /* data high */
            }
            /// <summary>
            /// Constructs matrix-value.
            /// </summary>
            /// <param name="matrix_id">matrix id value</param>
            /// <exception cref="TPCInvalidArgumentsException">if parameter would result out of bounds values</exception>
            public Ecat7_Matval(int matrix_id) {
                this.matrix_id = matrix_id;
                if (frame < 0 || frame > 65536) throw new TPCInvalidArgumentsException("frame value would be out of bounds");
                if (plane < 0 || plane > 65536) throw new TPCInvalidArgumentsException("plane value would be out of bounds");
                if (gate < 0 || gate > 65536) throw new TPCInvalidArgumentsException("gate value would be out of bounds");
                if (data < 0 || data > 65536) throw new TPCInvalidArgumentsException("data value would be out of bounds");
                if (bed < 0 || bed > 65536) throw new TPCInvalidArgumentsException("bed value would be out of bounds");
            }
            /// <summary>
            /// Constructs matrix-value.
            /// </summary>
            /// <param name="frame">frame number [0..65536]</param>
            /// <param name="plane">plane number [0..65536]</param>
            /// <param name="gate">[0..64]</param>
            /// <param name="data">[0..1]</param>
            /// <param name="bed">[0..16]</param>
            /// <exception cref="TPCInvalidArgumentsException">if parameter is out of bounds</exception>
            public Ecat7_Matval(int frame, int plane, int gate, int data, int bed) {
                if (frame < 0 || frame > 65536) throw new TPCInvalidArgumentsException("frame value out of bounds");
                if (plane < 0 || plane > 65536) throw new TPCInvalidArgumentsException("plane value out of bounds");
                if (gate < 0 || gate > 65536) throw new TPCInvalidArgumentsException("gate value out of bounds");
                if (data < 0 || data > 65536) throw new TPCInvalidArgumentsException("data value out of bounds");
                if (bed < 0 || bed > 65536) throw new TPCInvalidArgumentsException("bed value out of bounds");
                matrix_id = ecat7_val_to_id(frame, plane, gate, data, bed);
            }
            /// <summary>
            /// Matrix directory entry as string
            /// </summary>
            /// <returns>string representation</returns>
            public override string ToString()
            {
                return "Matval[frame="+frame+" plane="+plane+" gate="+gate+" data="+data+" bed="+bed+"]";
            }
        }
        /// <summary>
        /// Static matrix block size
        /// </summary>
        public static readonly int MatBLKSIZE = 512;
        /// <summary>
        /// Ecat 7 magic number
        /// </summary>
        public static readonly string ecat7V_MAGICNR = "MATRIX72v";
        /// <summary>
        /// Ecat 7 magic number
        /// </summary>
        public static readonly string ecat7S_MAGICNR = "MATRIX7011";
        /// <summary>
        /// Main header of file.
        /// </summary>
        public Ecat7_mainheader mainheader;
        /// <summary>
        /// One or more subheaders. Header type depends on image type. Note that this 
        /// array may or may not contain null references, depending image matrix data in last read Ecat7 
        /// file.
        /// </summary>
        public Ecat7_subheader[] subheaders;
        /// <summary>
        /// Matrix list that was last read from file.
        /// </summary>
        protected Ecat7_MatDir[] mlist;
        /// <summary>
        /// Matrix list that was last read from file. Read-only from outside of class.
        /// </summary>
        public Ecat7_MatDir[] matrixlist {
            get { return mlist; }
        }
        /// <summary>
        /// Shortcut for sample distance. Applicable only if there is mainheader or proper subheader.
        /// </summary>
        public float sampledistance {
            get
            {
                if (subheaders.Length > 0 && 
                    (subheaders[0] is Ecat7_scanheader) && 
                    (subheaders[0] as Ecat7_scanheader).x_resolution > 0.0)
                {
                    return 10.0f * (subheaders[0] as Ecat7_scanheader).x_resolution;
                }
                else
                {
                    return 10.0f * mainheader.bin_size;
                }
            }
        }
        /// <summary>
        /// Constructs Ecat7 file with filename.
        /// </summary>
        /// <param name="filename"></param>
        /// 
        public Ecat7File(string filename) {
            this.filename = filename;
            mainheader = new Ecat7_mainheader();
            header = new DynamicImageHeader();
        }
        /// <summary>
        /// Constructs Ecat7 file.
        /// </summary>
        /// <remarks>default filetype is 'ecat7_IMAGE16'</remarks>
        /// <param name="filename"></param>
        /// <param name="image">image data</param>
        /// <param name="header">image header</param>
        public Ecat7File(string filename, Image image, ImageHeader header): this(filename)
        {
            this.image = image;
            if (image is DynamicImage)
            {
                mainheader.num_frames = (short)(image as DynamicImage).frames;
                if(image.dim.Length > IntLimits.GATES) mainheader.num_gates = (short)(image as DynamicImage).gates;
                if (image.dim.Length > IntLimits.BEDS) mainheader.num_bed_pos = (short)(image as DynamicImage).beds;
            }
            else {
                mainheader.num_frames = 1;
                mainheader.num_gates = 1;
                mainheader.num_bed_pos = 1;
            }
            mainheader.num_planes = (short)image.dim.getDimension(IntLimits.PLANES);
            mainheader.file_type = mainheader.resolveFileType(header.datatype);
            this.header = new DynamicImageHeader(header);
            //create subheaders for gates and frames
            if (image.dim.Length > IntLimits.FRAMES && image.dim.getDimension(IntLimits.FRAMES) > 1)
            {
                if (image.dim.Length > IntLimits.GATES && image.dim.getDimension(IntLimits.GATES) > 1)
                    subheaders = new Ecat7_subheader[image.dim.getDimension(IntLimits.GATES) * image.dim.getDimension(IntLimits.FRAMES)];
                else
                    subheaders = new Ecat7_subheader[image.dim.getDimension(IntLimits.FRAMES)];
            } 
            else
            {
                if (image.dim.Length > IntLimits.GATES && image.dim.getDimension(IntLimits.GATES) > 1)
                    subheaders = new Ecat7_subheader[image.dim.getDimension(IntLimits.GATES)];
                else
                    subheaders = new Ecat7_subheader[1];
            }
            double[] start_times = null;
            double[] durations = null;
            if (image is DynamicImage)
            {
                start_times = (image as DynamicImage).getFrameStartTimes();
                durations = (image as DynamicImage).getFrameDurations();
            }
            for (int i = 0; i < subheaders.Length; i++) {
                subheaders[i] = new Ecat7_subheader();
                subheaders[i].scale_factor = 1.0f;
                if (image is DynamicImage && i < start_times.Length)
                {
                    subheaders[i].frame_start_time = (int)start_times[i] * 1000;
                    subheaders[i].frame_duration = (int)durations[i] * 1000;
                }
                else {
                    subheaders[i].frame_start_time = 1000;
                    subheaders[i].frame_duration = 0;
                }
            }
        }
        /// <summary>
        /// Read ECAT 7.x main header.
        /// </summary>
        /// <param name="filename">full path to file that is read</param>
        /// <returns>Ecat7 main header</returns>
        /// <exception cref="TPCEcat7FileException">if error occurred while reading</exception>
        public static Ecat7_mainheader ecat7ReadMainheader(string filename)
        {
            if(!checkFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");

            byte[] buf = new byte[MatBLKSIZE];
            string stringbuffer = "";
            Ecat7_mainheader h = new Ecat7_mainheader();
            FileStream filestream = new FileStream(filename, FileMode.Open);
            
            /* Seek the first block */
            filestream.Seek(0, SeekOrigin.Begin);
            /* Read the header block */
            filestream.Read(buf, 0, MatBLKSIZE);
            for (int i = 0; i < buf.Length; i++) {
                stringbuffer += (char)buf[i];
            }   
            
            /* Copy the header fields and swap if necessary */
            h.magic_number = stringbuffer.ToCharArray(0, 14);
            h.original_file_name = stringbuffer.ToCharArray(14, 32);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 46, 2); h.sw_version = BitConverter.ToInt16(buf, 46);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 48, 2); h.system_type = BitConverter.ToInt16(buf, 48);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 50, 2); h.file_type = (Ecat7_mainheader.datatype_e)BitConverter.ToUInt16(buf, 50);
            h.serial_number = toCharArray(buf, 52, 10);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 62, 4); h.scan_start_time = BitConverter.ToUInt32(buf, 62);
            h.isotope_name = toCharArray(buf, 66, 8);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 74, 4); h.isotope_halflife = BitConverter.ToSingle(buf, 74);
            h.radiopharmaceutical = toCharArray(buf, 78, 32);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 110, 4); h.gantry_tilt = BitConverter.ToSingle(buf, 110);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 114, 4); h.gantry_rotation = BitConverter.ToSingle(buf, 114);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 118, 4); h.bed_elevation = BitConverter.ToSingle(buf, 118);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 122, 4); h.intrinsic_tilt = BitConverter.ToSingle(buf, 122);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 126, 4); h.wobble_speed = BitConverter.ToInt16(buf, 126);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 128, 2); h.transm_source_type = (Ecat7_mainheader.Transm_source_type_e)BitConverter.ToInt16(buf, 128);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 130, 4); h.distance_scanned = BitConverter.ToSingle(buf, 130);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 134, 4); h.transaxial_fov = BitConverter.ToSingle(buf, 134);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 138, 2); h.angular_compression = BitConverter.ToInt16(buf, 138);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 140, 2); h.coin_samp_mode = BitConverter.ToInt16(buf, 140);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 142, 2); h.axial_samp_mode = BitConverter.ToInt16(buf, 142);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 144, 4); h.ecat_calibration_factor = BitConverter.ToSingle(buf, 144);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 148, 2); h.calibration_units = BitConverter.ToInt16(buf, 148);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 150, 2); h.calibration_units_label = BitConverter.ToInt16(buf, 150);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 152, 2); h.compression_code = (Ecat7_mainheader.Compression_code_e)BitConverter.ToInt16(buf, 152);
            h.study_type = toCharArray(buf, 154, 12);
            h.patient_id = toCharArray(buf, 166, 16);
            h.patient_name = toCharArray(buf, 182, 32);
            h.patient_sex = (Ecat7_mainheader.Patient_sex_e)BitConverter.ToChar(buf, 214);
            h.patient_dexterity = BitConverter.ToChar(buf, 215);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 216, 4); h.patient_age = BitConverter.ToSingle(buf, 216);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 220, 4); h.patient_height = BitConverter.ToSingle(buf, 220);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 224, 4); h.patient_weight = BitConverter.ToSingle(buf, 224);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 228, 4); h.patient_birth_date = BitConverter.ToInt32(buf, 228);
            h.physician_name = toCharArray(buf, 232, 32);
            h.operator_name = toCharArray(buf, 264, 32);
            h.study_description = toCharArray(buf, 296, 32);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 328, 2); h.acquisition_type = (Ecat7_mainheader.Acquisition_type_e)BitConverter.ToInt16(buf, 328);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 330, 2); h.patient_orientation = (Ecat7_mainheader.Patient_orientation_e)BitConverter.ToInt16(buf, 330);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 354, 2); h.num_planes = BitConverter.ToInt16(buf, 352);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 354, 2); h.num_frames = BitConverter.ToInt16(buf, 354);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 356, 2); h.num_gates = BitConverter.ToInt16(buf, 356);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 358, 2); h.num_bed_pos = BitConverter.ToInt16(buf, 358);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 360, 4); h.init_bed_position = BitConverter.ToSingle(buf, 360);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 364, 15*4); h.bed_position = toFloatArray(buf, 364, 15);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 424, 4); h.plane_separation = BitConverter.ToSingle(buf, 424);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 428, 2); h.lwr_sctr_thres = BitConverter.ToInt16(buf, 428);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 430, 2); h.lwr_true_thres = BitConverter.ToInt16(buf, 430);
            h.user_process_code = toCharArray(buf, 434, 10);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 444, 2); h.acquisition_mode = (Ecat7_mainheader.Acquisition_mode_e)BitConverter.ToInt16(buf, 444);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 446, 2); h.bin_size = BitConverter.ToSingle(buf, 446);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 450, 2); h.branching_fraction = BitConverter.ToSingle(buf, 450);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 454, 2); h.dose_start_time = BitConverter.ToInt32(buf, 454);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 458, 2); h.dosage = BitConverter.ToSingle(buf, 458);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 462, 2); h.well_counter_corr_factor = BitConverter.ToSingle(buf, 462);
            h.data_units = toCharArray(buf, 466, 32);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 498, 2); h.septa_state = BitConverter.ToInt16(buf, 498);
            h.fill = toShortArray(buf, 500, 6);
            filestream.Close();
            return h;
        }
        /// <summary>
        /// Write ECAT 7.2 main header.
        /// </summary>
        /// <param name="filename">full path to file that is read</param>
        /// <param name="h">main header to write</param>
        /// <exception cref="TPCEcat7FileException">if error occurred while writing</exception>
        public static void ecat7WriteMainheader(string filename, Ecat7_mainheader h)
        {
            byte[] buf = new byte[MatBLKSIZE];
            if (h == null) throw new TPCEcat7FileException("header data is null");
            FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate);

            /* Seek the first block */
            filestream.Seek(0, SeekOrigin.Begin);

            /* Copy the header fields and swap if necessary */
            toBytesArray(h.magic_number).CopyTo(buf, 0);
            toBytesArray(h.original_file_name).CopyTo(buf, 14);
            BitConverter.GetBytes(h.sw_version).CopyTo(buf, 46); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 46, 2);
            BitConverter.GetBytes(h.system_type).CopyTo(buf, 48); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 48, 2);
            BitConverter.GetBytes((short)h.file_type).CopyTo(buf, 50); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 50, 2); 
            toBytesArray(h.serial_number).CopyTo(buf, 52);
            BitConverter.GetBytes(h.scan_start_time).CopyTo(buf, 62); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 62, 4); 
            toBytesArray(h.isotope_name).CopyTo(buf, 66);
            BitConverter.GetBytes(h.isotope_halflife).CopyTo(buf, 74); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 74, 4);
            toBytesArray(h.radiopharmaceutical).CopyTo(buf, 78);
            BitConverter.GetBytes(h.gantry_tilt).CopyTo(buf, 110); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 110, 4);
            BitConverter.GetBytes(h.gantry_rotation).CopyTo(buf, 114); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 114, 4);
            BitConverter.GetBytes(h.bed_elevation).CopyTo(buf, 118); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 118, 4);
            BitConverter.GetBytes(h.intrinsic_tilt).CopyTo(buf, 122); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 122, 4); 
            BitConverter.GetBytes(h.wobble_speed).CopyTo(buf, 126); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 126, 2);
            BitConverter.GetBytes((short)h.transm_source_type).CopyTo(buf, 128); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 128, 2);
            BitConverter.GetBytes(h.distance_scanned).CopyTo(buf, 130); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 130, 4);
            BitConverter.GetBytes(h.transaxial_fov).CopyTo(buf, 134); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 134, 4);
            BitConverter.GetBytes(h.angular_compression).CopyTo(buf, 138); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 138, 2);
            BitConverter.GetBytes(h.coin_samp_mode).CopyTo(buf, 140); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 140, 2);
            BitConverter.GetBytes(h.axial_samp_mode).CopyTo(buf, 142); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 142, 2);
            BitConverter.GetBytes(h.ecat_calibration_factor).CopyTo(buf, 144); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 144, 4);
            BitConverter.GetBytes(h.calibration_units).CopyTo(buf, 148); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 148, 2);
            BitConverter.GetBytes(h.calibration_units_label).CopyTo(buf, 150); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 150, 2);
            BitConverter.GetBytes((short)h.compression_code).CopyTo(buf, 152); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 152, 2);
            toBytesArray(h.study_type).CopyTo(buf, 154);
            toBytesArray(h.patient_id).CopyTo(buf, 166);
            toBytesArray(h.patient_name).CopyTo(buf, 182);
            BitConverter.GetBytes((char)h.patient_sex).CopyTo(buf, 214);
            BitConverter.GetBytes(h.patient_dexterity).CopyTo(buf, 215);
            BitConverter.GetBytes(h.patient_age).CopyTo(buf, 216); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 216, 4);
            BitConverter.GetBytes(h.patient_height).CopyTo(buf, 220); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 220, 4);
            BitConverter.GetBytes(h.patient_weight).CopyTo(buf, 224); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 224, 4);
            BitConverter.GetBytes(h.patient_birth_date).CopyTo(buf, 228); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 228, 4); 
            toBytesArray(h.physician_name).CopyTo(buf, 232);
            toBytesArray(h.operator_name).CopyTo(buf, 264);
            toBytesArray(h.study_description).CopyTo(buf, 296);
            BitConverter.GetBytes((short)h.acquisition_type).CopyTo(buf, 328); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 328, 2);
            BitConverter.GetBytes((short)h.patient_orientation).CopyTo(buf, 330); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 330, 2);
            BitConverter.GetBytes(h.num_planes).CopyTo(buf, 352); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 352, 2);
            BitConverter.GetBytes(h.num_frames).CopyTo(buf, 354); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 354, 2);
            BitConverter.GetBytes(h.num_gates).CopyTo(buf, 356); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 356, 2);
            BitConverter.GetBytes(h.num_bed_pos).CopyTo(buf, 358); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 358, 2);
            BitConverter.GetBytes(h.init_bed_position).CopyTo(buf, 360); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 360, 4); 
            toBytesArray(h.bed_position).CopyTo(buf, 364); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 364, 15 * 4);
            BitConverter.GetBytes(h.plane_separation).CopyTo(buf, 424); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 424, 4);
            BitConverter.GetBytes(h.lwr_sctr_thres).CopyTo(buf, 428); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 428, 2);
            BitConverter.GetBytes(h.lwr_true_thres).CopyTo(buf, 430); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 430, 2); 
            toBytesArray(h.user_process_code).CopyTo(buf, 434);
            BitConverter.GetBytes((short)h.acquisition_mode).CopyTo(buf, 444); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 444, 2);
            BitConverter.GetBytes(h.bin_size).CopyTo(buf, 446); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 446, 4);
            BitConverter.GetBytes(h.branching_fraction).CopyTo(buf, 450); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 450, 2);
            BitConverter.GetBytes(h.dose_start_time).CopyTo(buf, 454); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 454, 4);
            BitConverter.GetBytes(h.dosage).CopyTo(buf, 458); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 458, 4);
            BitConverter.GetBytes(h.well_counter_corr_factor).CopyTo(buf, 462); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 462, 4);
            toBytesArray(h.data_units).CopyTo(buf, 466);
            BitConverter.GetBytes(h.septa_state).CopyTo(buf, 498); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 498, 2);
            toBytesArray(h.fill).CopyTo(buf, 500);
            filestream.Write(buf, 0, buf.Length);
            filestream.Close();
        }
        /// <summary>
        /// Read ECAT 7.x 3D scan headers. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="filename">file that is read</param>
        /// <returns>array of subheaders</returns>
        /// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
        public static Ecat7_scanheader[] ecat7ReadScanheaders(string filename)
        {
            if (!checkFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
            Ecat7_MatDir[] mlist = ecat7ReadMatlist(filename);
            return ecat7ReadScanheaders(mlist, filename);
        }
        /// <summary>
        /// Read ECAT 7.x 3D scan headers. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="matdir">array of matrix directory entries</param>
        /// <param name="filename">Ecat7 file that is read</param>
        /// <returns>array of subheaders</returns>
        protected static Ecat7_scanheader[] ecat7ReadScanheaders(Ecat7_MatDir[] matdir, string filename)
        {
            List<Ecat7_scanheader> scanheader_list = new List<Ecat7_scanheader>();
            for (int i = 0; i < matdir.Length; i++)
            {
                if (matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
                    scanheader_list.Add(ecat7ReadScanheader(matdir[i], filename));
                else
                    scanheader_list.Add(null);
                
            }
            return scanheader_list.ToArray();
        }
        /// <summary>
        /// Read ECAT 7.x 3D scan header
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="filename">full path to file that is read</param>
        /// <returns></returns>
        protected static Ecat7_scanheader ecat7ReadScanheader(Ecat7_MatDir matdir, string filename) {
            byte[] buf = new byte[2*MatBLKSIZE];
            int blk = matdir.strtblk;
            FileStream filestream = new FileStream(filename, FileMode.Open);
            Ecat7_scanheader h = new Ecat7_scanheader();

            /* Seek the subheader block */
            filestream.Seek((blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
            if (filestream.Position != (blk - 1) * MatBLKSIZE)
            {
                throw new TPCEcat7FileException("Invalid stream position after seek with " + matdir + "," + filename + ".");
            }
            /* Read the header block */
            if (filestream.Read(buf, 0, 2*MatBLKSIZE) < 2*MatBLKSIZE)
            {
                throw new TPCEcat7FileException("Failed to read all scan header data with "+matdir+","+filename+".");
            }

            /* Copy the header fields and swap if necessary */
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_scanheader.data_type_e)BitConverter.ToInt16(buf, 0);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 2, 2); h.num_dimensions = BitConverter.ToInt16(buf, 2);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 4, 2); h.num_r_elements = BitConverter.ToInt16(buf, 4);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 6, 2); h.num_angles = BitConverter.ToInt16(buf, 6);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 8, 2); h.corrections_applied = BitConverter.ToInt16(buf, 8);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 10, 64*2); h.num_z_elements = toShortArray(buf, 10, 64);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 138, 2); h.ring_difference = BitConverter.ToInt16(buf, 138);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 140, 2); h.storage_order = BitConverter.ToInt16(buf, 140);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 142, 2); h.axial_compression = BitConverter.ToInt16(buf, 142);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 144, 4); h.x_resolution = BitConverter.ToSingle(buf, 144);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 148, 4); h.v_resolution = BitConverter.ToSingle(buf, 148);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 152, 4); h.z_resolution = BitConverter.ToSingle(buf, 152);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 156, 4); h.w_resolution = BitConverter.ToSingle(buf, 156);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 160, 6*2); h.fill_gate = toShortArray(buf, 160, 6);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 172, 4); h.gate_duration = BitConverter.ToInt32(buf, 172);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 176, 4); h.r_wave_offset = BitConverter.ToInt32(buf, 176);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 180, 4); h.num_accepted_beats = BitConverter.ToInt32(buf, 180);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 184, 4); h.scale_factor = BitConverter.ToSingle(buf, 184);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 188, 2); h.scan_min = BitConverter.ToInt16(buf, 188);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 190, 2); h.scan_max = BitConverter.ToInt16(buf, 190);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 192, 4); h.prompts = BitConverter.ToInt32(buf, 192);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 196, 4); h.delayed = BitConverter.ToInt32(buf, 196);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 200, 4); h.multiples = BitConverter.ToInt32(buf, 200);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 204, 4); h.net_trues = BitConverter.ToInt32(buf, 204);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 208, 4); h.tot_avg_cor = BitConverter.ToSingle(buf, 208);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 212, 4); h.tot_avg_uncor = BitConverter.ToSingle(buf, 212);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 216, 4); h.total_coin_rate = BitConverter.ToInt32(buf, 216);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 220, 4); h.frame_start_time = BitConverter.ToInt32(buf, 220);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf , 224, 4); h.frame_duration = BitConverter.ToInt32(buf, 224);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 228, 4); h.deadtime_correction_factor = BitConverter.ToSingle(buf, 228);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 232, 90*2); h.fill_cti = toShortArray(buf, 232, 90);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 412, 50*2); h.fill_user = toShortArray(buf, 412, 50);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 512, 128*4); h.uncor_singles = toFloatArray(buf, 512, 128);
            filestream.Close();
            return h;
        }
        /// <summary>
        /// Checks that all matrixlist entries have read/write status.
        /// </summary>
        /// <param name="mlist">list to be checked</param>
        /// <returns>true if ok, false otherwise</returns>
        protected bool ecat7CheckMatlist(Ecat7_MatDir[] mlist)
        {
            if(mlist == null) return false;
            for (int i = 0; i < mlist.Length; i++) 
                if(mlist[i].status != Ecat7_MatDir.status_e.EXISTS) return false;
            return true;
        }
        /// <summary>
        /// Read ECAT matrix list.
        /// </summary>
        /// <returns>matrix list from file</returns>
        protected static Ecat7_MatDir[] ecat7ReadMatlist(string filename) {
            int i;
            //buffer having directory list data
            byte[] buf = new byte[MatBLKSIZE];
            //matrix entry list
            List<Ecat7_MatDir> mlist = new List<Ecat7_MatDir>();
            //filestream
            FileStream filestream = new FileStream(filename, FileMode.Open);
            //start from block 1
            int blk = 1;
            //directory list header
            ecat7_MatHdr dirhdr;

            // seek the first directory list block
            try {
                filestream.Seek(blk*MatBLKSIZE, SeekOrigin.Begin);
            } catch(Exception e) {
                throw new TPCEcat7FileException("Failed to seek matrix list beginning:"+e.Message);
            }

            // read until first directory list is going to be the next directory list
            do {
                //read the directory list into buffer
                if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE)
                    throw new TPCEcat7FileException("Failed to read full matrix block.");
                if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, buf.Length);
                /* 'directory header' */
                dirhdr = new ecat7_MatHdr(  BitConverter.ToInt32(buf, 0), 
                                            BitConverter.ToInt32(buf, 4), 
                                            BitConverter.ToInt32(buf, 8), 
                                            BitConverter.ToInt32(buf, 12));
                /* matrix lists */
                for(i=16; i < MatBLKSIZE; i+=16) {
                    if(((Ecat7_MatDir.status_e)BitConverter.ToInt32(buf, i+12)) == Ecat7_MatDir.status_e.EXISTS) {
                        mlist.Add(new Ecat7_MatDir( BitConverter.ToInt32(buf, i),
                                                    BitConverter.ToInt32(buf, i+4),
                                                    BitConverter.ToInt32(buf, i+8),
                                                    (Ecat7_MatDir.status_e)BitConverter.ToInt32(buf, i + 12)));
                    } else {
                        mlist.Add(new Ecat7_MatDir(0, 0, 0, Ecat7_MatDir.status_e.DELETED));
                    }
                }
                //seek next directory list block
                try {
                    filestream.Seek((dirhdr.next_blk-1)*MatBLKSIZE, SeekOrigin.Begin);
                } catch(Exception e) {
                    //this line is to get rid of warning which cannot be turned off because it is used other place
                    e.ToString();
                    break;
                }
            } while(dirhdr.next_blk != 2);
            filestream.Close();
            return mlist.ToArray();
        }
        /// <summary>
        /// Writes ECAT matrix list data into file. 
        /// </summary>
        /// <param name="filename">file name</param>
        /// <param name="list">matrix list, length must be power of 31 (prefilled with possible empty lines)</param>
        /// <returns>matrix list from file</returns>
        protected static void ecat7WriteMatlist(string filename, Ecat7_MatDir[] list)
        {
            //byte location in block
            int i = 16;
            //matrix list index number
            int j = 0;
            //buffer having directory list data
            byte[] buf = new byte[MatBLKSIZE];
            //current block number
            int blk = 1;
            
            if (list.Length % 31 != 0)
            {
                throw new TPCEcat7FileException("Number of matrixes must be power of 31.");
            }

            //filestream
            FileStream filestream = new FileStream(filename, FileMode.Open);
            //directory list header
            ecat7_MatHdr dirhdr = new ecat7_MatHdr();
            dirhdr.nr_used = 0;
            dirhdr.nr_free = 31;
            dirhdr.next_blk = 1;
            dirhdr.prev_blk = 0;          

            //write directory list entries
            for (j = 0; j < list.Length; j++)
            {
                /* add matrix item */
                Array.Copy(BitConverter.GetBytes(list[j].id.matrix_id), 0, buf, i, 4);
                Array.Copy(BitConverter.GetBytes(list[j].strtblk), 0, buf, i + 4, 4);
                Array.Copy(BitConverter.GetBytes(list[j].endblk), 0, buf, i + 8, 4);
                Array.Copy(BitConverter.GetBytes((int)list[j].status), 0, buf, i + 12, 4);
                if (list[j].status == Ecat7_MatDir.status_e.EXISTS)
                {
                    dirhdr.nr_free--;
                    dirhdr.nr_used++;
                }
                i += 16;
                /* add 'directory header' and write block into file when buffer is full */
                if (i == MatBLKSIZE)
                {
                    //set location of next buffer to current buffer if this is the end of matrix list
                    if (j == list.Length - 1)
                    {
                        dirhdr.next_blk = 2;
                        dirhdr.prev_blk = 2;
                    }
                    else
                    {
                        dirhdr.next_blk = list[j].endblk + 1;
                    }
                    //set matrixlist header into start of buffer
                    Array.Copy(BitConverter.GetBytes(dirhdr.nr_free), 0, buf,  0, 4);
                    Array.Copy(BitConverter.GetBytes(dirhdr.next_blk), 0, buf,  4, 4);
                    Array.Copy(BitConverter.GetBytes(dirhdr.prev_blk), 0, buf,  8, 4);
                    Array.Copy(BitConverter.GetBytes(dirhdr.nr_used), 0, buf, 12, 4);
                    //write the directory list from buffer
                    if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, buf.Length);
                    try
                    {
                        filestream.Seek(blk * MatBLKSIZE, SeekOrigin.Begin);
                        filestream.Write(buf, 0, MatBLKSIZE);
                        //reset nunber of used and unused list indexes and set previsous and current block number
                        dirhdr.nr_used = 0;
                        dirhdr.prev_blk = blk;
                        blk = dirhdr.next_blk-1;
                        dirhdr.nr_free = 31;
                    }
                    catch (Exception e)
                    {
                        throw new TPCEcat7FileException("Failed to write matrix block:"+e.Message);
                    }
                    i = 16;
                }
            }
            /* commented because making linking according to standard might cause endless loops 
             * in some programs reading Ecat7
            //write linkage between first and last matrix list
            try
            {
                //write previous block for first item (== 0)
                Array.Copy(BitConverter.GetBytes(blk), 0, buf, 0, 4);
                filestream.Seek(1 * MatBLKSIZE + 8, SeekOrigin.Begin);
                if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, 4);
                filestream.Write(buf, 0, 4);
                //write next block for last item (== 2)
                Array.Copy(BitConverter.GetBytes(2), 0, buf, 0, 4);
                filestream.Seek(blk * MatBLKSIZE + 4, SeekOrigin.Begin);
                if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, 4);
                filestream.Write(buf, 0, 4);

                filestream.Seek(1 * MatBLKSIZE, SeekOrigin.Begin);
                filestream.Read(buf, 0, 16);
                if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, 16);
                // 'directory header'
                dirhdr = new ecat7_MatHdr(BitConverter.ToInt32(buf, 0),
                                            BitConverter.ToInt32(buf, 4),
                                            BitConverter.ToInt32(buf, 8),
                                            BitConverter.ToInt32(buf, 12));
            }
            catch (Exception e)
            {
                throw new TPCEcat7FileException("Failed to seek matrix list beginning:" + e.Message);
            }
            */
            filestream.Close();
        }
        /// <summary>
        /// Comparison class for matrix lists. 
        /// </summary>
        private class MatrixComparer : IComparer<Ecat7_MatDir>
        {
            /// <summary>
            /// Sorts according to plane and then according to frame.
            /// </summary>
            /// <param name="x"></param>
            /// <param name="y"></param>
            /// <returns></returns>
            public int Compare(Ecat7_MatDir x, Ecat7_MatDir y)
            {
                if (x.id.plane < y.id.plane) return -1;
                if (x.id.plane > y.id.plane) return 1;
                if (x.id.gate < y.id.gate) return -1;
                if (x.id.gate > y.id.gate) return 1;
                if (x.id.frame < y.id.frame) return -1;
                if (x.id.frame > y.id.frame) return 1;
                else return 0;
            }
        }
        /// <summary>
        /// Read ECAT 7.x polar map header. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="filename">file that is read</param>
        /// <returns>array of subheaders</returns>
        /// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
        public static Ecat7_polmapheader[] ecat7ReadPolmapheaders(string filename)
        {
            if (!checkFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
            Ecat7_MatDir[] mlist = ecat7ReadMatlist(filename);
            return ecat7ReadPolmapheaders(mlist, filename);
        }        
        /// <summary>
        /// Read ECAT 7.x polar map headers. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="matdir">array of matrix directory entries</param>
        /// <param name="filename">Ecat7 file that is read</param>
        /// <returns>array of subheaders</returns>
        protected static Ecat7_polmapheader[] ecat7ReadPolmapheaders(Ecat7_MatDir[] matdir, string filename)
        {
            List<Ecat7_polmapheader> polmapheader_list = new List<Ecat7_polmapheader>();
            for (int i = 0; i < matdir.Length; i++)
            {
                if(matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
                    polmapheader_list.Add(ecat7ReadPolmapheader(matdir[i], filename));
                else
                    polmapheader_list.Add(null);
            }
            return polmapheader_list.ToArray();
        }
        /// <summary>
        /// Reads ECAT 7.x polar map header
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="filename">Ecat7 file that is read</param>
        /// <see cref="MatBLKSIZE"/>
        /// <returns>polar map header</returns>
        protected static Ecat7_polmapheader ecat7ReadPolmapheader(Ecat7_MatDir matdir, string filename)
        {
            byte[] buf = new byte[MatBLKSIZE];
            int blk = matdir.strtblk;
            FileStream filestream = new FileStream(filename, FileMode.Open);
            Ecat7_polmapheader h = new Ecat7_polmapheader();
            /* Seek the subheader block */
            filestream.Seek((blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
            if (filestream.Position != (blk - 1) * MatBLKSIZE)
            {
                throw new TPCEcat7FileException("Invalid stream position after seek.");
            }
            /* Read the header block */
            if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE)
            {
                throw new TPCEcat7FileException("Failed to read all scan header data.");
            }

            /* Copy the header fields and swap if necessary */
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_polmapheader.data_type_e)BitConverter.ToInt16(buf, 0);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 2, 2); h.polar_map_type = BitConverter.ToInt16(buf, 2);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 4, 2); h.num_rings = BitConverter.ToInt16(buf, 4);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 6, 32*2); h.sectors_per_ring = toShortArray(buf, 6, 32);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 70, 32*4); h.ring_position = toFloatArray(buf, 70, 32);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 198, 32*2); h.ring_angle = toShortArray(buf, 198, 32);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 262, 2); h.start_angle = BitConverter.ToInt16(buf, 262);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 264, 3*2); h.long_axis_left = toShortArray(buf, 264, 3);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 270, 3*2); h.long_axis_right = toShortArray(buf, 270, 3);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 276, 2); h.position_data = BitConverter.ToInt16(buf, 276);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 278, 2); h.image_min = BitConverter.ToInt16(buf, 278);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 280, 2); h.image_max = BitConverter.ToInt16(buf, 280);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 282, 4); h.scale_factor = BitConverter.ToSingle(buf, 282);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 286, 4); h.pixel_size= BitConverter.ToSingle(buf, 286);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 290, 4); h.frame_duration = BitConverter.ToInt32(buf, 290);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 294, 4); h.frame_start_time = BitConverter.ToInt32(buf, 294);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 298, 2); h.processing_code = BitConverter.ToInt16(buf, 298);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 300, 2); h.quant_units = (Ecat7_polmapheader.quant_units_e)BitConverter.ToInt16(buf, 300);
            h.annotation = toCharArray(buf, 302, 40);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 342, 4); h.gate_duration = BitConverter.ToInt32(buf, 342);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 346, 4); h.r_wave_offset = BitConverter.ToInt32(buf, 346);
            if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 350, 4); h.num_accepted_beats = BitConverter.ToInt32(buf, 350);
            h.polar_map_protocol = toCharArray(buf, 354, 20);
            h.database_name = toCharArray(buf, 374, 30);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 404, 27*2); h.fill_cti = toShortArray(buf, 404, 27);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 464, 27*2); h.fill_cti = toShortArray(buf, 464, 27);
            filestream.Close();
            return h;
        }
        /// <summary>
        /// Read ECAT 7.x image header. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="filename">file that is read</param>
        /// <returns>array of subheaders</returns>
        /// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
        public static Ecat7_imageheader[] ecat7ReadImageheaders(string filename)
        {
            if (!checkFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
            Ecat7_MatDir[] mlist = ecat7ReadMatlist(filename);
            return ecat7ReadImageheaders(mlist, filename);
        }        
        /// <summary>
        /// Read ECAT 7.x image headers. Null is placed where matrix does not exist.
        /// </summary>
        /// <param name="matdir">array of matrix directory entries</param>
        /// <param name="filename">Ecat7 file that is read</param>
        /// <returns>array of subheaders</returns>
        protected static Ecat7_imageheader[] ecat7ReadImageheaders(Ecat7_MatDir[] matdir, string filename) {
            List<Ecat7_imageheader> imageheader_list = new List<Ecat7_imageheader>();
            for (int i = 0; i < matdir.Length; i++)
            {
                if(matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
                    imageheader_list.Add(ecat7ReadImageheader(matdir[i], filename));
                else
                    imageheader_list.Add(null);
            }
            return imageheader_list.ToArray();
        }
        /// <summary>
        /// Reads ECAT 7.x image header
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="filename">Ecat7 file that is read</param>
        /// <see cref="MatBLKSIZE"/>
        /// <returns></returns>
        protected static Ecat7_imageheader ecat7ReadImageheader(Ecat7_MatDir matdir, string filename)
        {
            byte[] buf = new byte[MatBLKSIZE];
            Ecat7_imageheader h = new Ecat7_imageheader();
            FileStream filestream = new FileStream(filename, FileMode.Open);
            int blk = matdir.strtblk;

            /* Seek the subheader block */
            filestream.Seek((blk-1)*MatBLKSIZE, SeekOrigin.Begin);
            if (filestream.Position != (blk - 1) * MatBLKSIZE)
            {
                throw new TPCEcat7FileException("Invalid stream position after seek.");
            }
            /* Read the header block */
            if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE)
            {
                throw new TPCEcat7FileException(filestream.Position+" Failed to read all image header data.");
            }

            /* Copy the header fields and swap if necessary */
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_imageheader.data_type_e)BitConverter.ToInt16(buf,0);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 2, 2); h.num_dimensions = BitConverter.ToInt16(buf,2);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 4, 2); h.x_dimension = BitConverter.ToInt16(buf,4);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 6, 2); h.y_dimension = BitConverter.ToInt16(buf,6);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 8, 2); h.z_dimension = BitConverter.ToInt16(buf,8);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 10, 4); h.x_offset = BitConverter.ToSingle(buf,10);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 14, 4); h.y_offset = BitConverter.ToSingle(buf,14);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 18, 4); h.z_offset = BitConverter.ToSingle(buf,18);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 22, 4); h.recon_zoom = BitConverter.ToSingle(buf,22);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 26, 4); h.scale_factor = BitConverter.ToSingle(buf,26);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 30, 2); h.image_min = BitConverter.ToInt16(buf,30);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 32, 2); h.image_max = BitConverter.ToInt16(buf,32);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 34, 4); h.x_pixel_size = BitConverter.ToSingle(buf,34);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 38, 4); h.y_pixel_size = BitConverter.ToSingle(buf,38);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 42, 4); h.z_pixel_size = BitConverter.ToSingle(buf,42);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 46, 4); h.frame_duration = BitConverter.ToInt32(buf,46);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 50, 4); h.frame_start_time = BitConverter.ToInt32(buf,50);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 54, 2); h.filter_code = (Ecat7_imageheader.filter_code_e)BitConverter.ToInt16(buf, 54);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 56, 4); h.x_resolution = BitConverter.ToSingle(buf,56);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 60, 4); h.y_resolution = BitConverter.ToSingle(buf,60);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 64, 4); h.z_resolution = BitConverter.ToSingle(buf,64);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 68, 4); h.num_r_elements = BitConverter.ToSingle(buf,68);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 72, 4); h.num_angles = BitConverter.ToSingle(buf,72);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 76, 4); h.z_rotation_angle = BitConverter.ToSingle(buf,76);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 80, 4); h.decay_corr_fctr = BitConverter.ToSingle(buf,80);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 84, 4); h.processing_code = BitConverter.ToUInt32(buf,84);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 88, 4); h.gate_duration = BitConverter.ToInt32(buf,88);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 92, 4); h.r_wave_offset = BitConverter.ToInt32(buf,92);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 96, 4); h.num_accepted_beats = BitConverter.ToInt32(buf,96);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 100, 4); h.filter_cutoff_frequency = BitConverter.ToSingle(buf,100);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 104, 4); h.filter_resolution = BitConverter.ToSingle(buf,104);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 108, 4); h.filter_ramp_slope = BitConverter.ToSingle(buf,108);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 112, 2); h.filter_order = BitConverter.ToInt16(buf,112);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 114, 4); h.filter_scatter_fraction = BitConverter.ToSingle(buf,114);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 118, 4); h.filter_scatter_slope = BitConverter.ToSingle(buf,118);
            h.annotation = toCharArray(buf, 122, 40);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 162, 4); h.mt_1_1 = BitConverter.ToSingle(buf,162);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 166, 4); h.mt_1_2 = BitConverter.ToSingle(buf,166);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 170, 4); h.mt_1_3 = BitConverter.ToSingle(buf,170);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 174, 4); h.mt_2_1 = BitConverter.ToSingle(buf,174);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 178, 4); h.mt_2_2 = BitConverter.ToSingle(buf,178);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 182, 4); h.mt_2_3 = BitConverter.ToSingle(buf,182);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 186, 4); h.mt_3_1 = BitConverter.ToSingle(buf,186);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 190, 4); h.mt_3_2 = BitConverter.ToSingle(buf,190);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 194, 4); h.mt_3_3 = BitConverter.ToSingle(buf,194);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 198, 4); h.rfilter_cutoff = BitConverter.ToSingle(buf,198);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 202, 4); h.rfilter_resolution = BitConverter.ToSingle(buf,202);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 206, 2); h.rfilter_code = BitConverter.ToInt16(buf,206);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 208, 2); h.rfilter_order = BitConverter.ToInt16(buf,208);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 210, 4); h.zfilter_cutoff = BitConverter.ToSingle(buf,210);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 214, 4); h.zfilter_resolution = BitConverter.ToSingle(buf,214);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 218, 2); h.zfilter_code = BitConverter.ToInt16(buf,218);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 220, 2); h.zfilter_order = BitConverter.ToInt16(buf,220);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 222, 4); h.mt_1_4 = BitConverter.ToSingle(buf,222);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 226, 4); h.mt_2_4 = BitConverter.ToSingle(buf,226);
            if (BitConverter.IsLittleEndian) swapWordBytes(buf, 230, 4); h.mt_3_4 = BitConverter.ToSingle(buf,230);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 234, 2); h.scatter_type = (Ecat7_imageheader.scatter_type_e)BitConverter.ToInt16(buf, 234);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 236, 2); h.recon_type = (Ecat7_imageheader.recon_type_e)BitConverter.ToInt16(buf, 236);
            if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 238, 2); h.recon_views = BitConverter.ToInt16(buf,238);
            h.fill_cti = toShortArray(buf, 240, 87);
            h.fill_user = toShortArray(buf, 414, 48);
            filestream.Close();
            return h;
        }
        /// <summary>
        /// Writes ECAT 7.x image header
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="h">image header</param>
        /// <param name="filename">Ecat7 file where header is written</param>
        protected static void ecat7WriteImageheader(Ecat7_MatDir matdir, Ecat7_imageheader h, string filename)
        {
            byte[] buf = new byte[MatBLKSIZE];
            FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate);

            /* Seek the proper block */
            try
            {
                filestream.Seek((matdir.strtblk-1) * MatBLKSIZE, SeekOrigin.Begin);
            } catch(Exception e) {
                throw new TPCEcat7FileException("Failed to seek position " + (matdir.strtblk - 1) * MatBLKSIZE + " for imageheader writing:" + e.Message);
            }

            /* Copy the header fields and swap if necessary */
            BitConverter.GetBytes((short)h.data_type).CopyTo(buf, 0); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, 2);
            BitConverter.GetBytes(h.num_dimensions).CopyTo(buf, 2); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 2, 2);
            BitConverter.GetBytes(h.x_dimension).CopyTo(buf, 4); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 4, 2);
            BitConverter.GetBytes(h.y_dimension).CopyTo(buf, 6); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 6, 2);
            BitConverter.GetBytes(h.z_dimension).CopyTo(buf, 8); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 8, 2);
            BitConverter.GetBytes(h.x_offset).CopyTo(buf, 10); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 10, 4);
            BitConverter.GetBytes(h.y_offset).CopyTo(buf, 14); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 14, 4);
            BitConverter.GetBytes(h.z_offset).CopyTo(buf, 18); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 18, 4);
            BitConverter.GetBytes(h.recon_zoom).CopyTo(buf, 22); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 22, 4);
            BitConverter.GetBytes(h.scale_factor).CopyTo(buf, 26); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 26, 4);
            BitConverter.GetBytes(h.image_min).CopyTo(buf, 30); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 30, 2);
            BitConverter.GetBytes(h.image_max).CopyTo(buf, 32); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 32, 2);
            BitConverter.GetBytes(h.x_pixel_size).CopyTo(buf, 34); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 34, 4);
            BitConverter.GetBytes(h.y_pixel_size).CopyTo(buf, 38); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 38, 4);
            BitConverter.GetBytes(h.z_pixel_size).CopyTo(buf, 42); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 42, 4);
            BitConverter.GetBytes(h.frame_duration).CopyTo(buf, 46); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 46, 4);
            BitConverter.GetBytes(h.frame_start_time).CopyTo(buf, 50); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 50, 4);
            BitConverter.GetBytes((short)h.filter_code).CopyTo(buf, 54); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 54, 2);
            BitConverter.GetBytes(h.x_resolution).CopyTo(buf, 56); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 56, 4);
            BitConverter.GetBytes(h.y_resolution).CopyTo(buf, 60); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 60, 4);
            BitConverter.GetBytes(h.z_resolution).CopyTo(buf, 64); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 64, 4);
            BitConverter.GetBytes(h.num_r_elements).CopyTo(buf, 68); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 68, 4);
            BitConverter.GetBytes(h.num_angles).CopyTo(buf, 72); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 72, 4);
            BitConverter.GetBytes(h.z_rotation_angle).CopyTo(buf, 76); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 76, 4);
            BitConverter.GetBytes(h.decay_corr_fctr).CopyTo(buf, 80); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 80, 4);
            BitConverter.GetBytes(h.processing_code).CopyTo(buf, 84); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 84, 4);
            BitConverter.GetBytes(h.gate_duration).CopyTo(buf, 88); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 88, 4);
            BitConverter.GetBytes(h.r_wave_offset).CopyTo(buf, 92); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 92, 4);
            BitConverter.GetBytes(h.num_accepted_beats).CopyTo(buf, 96); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 96, 4);
            BitConverter.GetBytes(h.filter_cutoff_frequency).CopyTo(buf, 100); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 100, 4);
            BitConverter.GetBytes(h.filter_resolution).CopyTo(buf, 104); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 104, 4);
            BitConverter.GetBytes(h.filter_ramp_slope).CopyTo(buf, 108); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 108, 4);
            BitConverter.GetBytes(h.filter_order).CopyTo(buf, 112); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 112, 2);
            BitConverter.GetBytes(h.filter_scatter_fraction).CopyTo(buf, 114); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 114, 4);
            BitConverter.GetBytes(h.filter_scatter_slope).CopyTo(buf, 118); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 118, 4); 
            toBytesArray(h.annotation).CopyTo(buf, 122);
            BitConverter.GetBytes(h.mt_1_1).CopyTo(buf, 162); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 162, 4);
            BitConverter.GetBytes(h.mt_1_2).CopyTo(buf, 166); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 166, 4);
            BitConverter.GetBytes(h.mt_1_3).CopyTo(buf, 170); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 170, 4);
            BitConverter.GetBytes(h.mt_2_1).CopyTo(buf, 174); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 174, 4);
            BitConverter.GetBytes(h.mt_2_2).CopyTo(buf, 178); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 178, 4);
            BitConverter.GetBytes(h.mt_2_3).CopyTo(buf, 182); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 182, 4);
            BitConverter.GetBytes(h.mt_3_1).CopyTo(buf, 186); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 186, 4);
            BitConverter.GetBytes(h.mt_3_2).CopyTo(buf, 190); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 190, 4);
            BitConverter.GetBytes(h.mt_3_3).CopyTo(buf, 194); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 194, 4);
            BitConverter.GetBytes(h.rfilter_cutoff).CopyTo(buf, 198); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 198, 4);
            BitConverter.GetBytes(h.rfilter_resolution).CopyTo(buf, 202); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 202, 4);
            BitConverter.GetBytes(h.rfilter_code).CopyTo(buf, 206); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 206, 2);
            BitConverter.GetBytes(h.rfilter_order).CopyTo(buf, 208); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 208, 2);
            BitConverter.GetBytes(h.zfilter_cutoff).CopyTo(buf, 210); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 210, 4);
            BitConverter.GetBytes(h.zfilter_resolution).CopyTo(buf, 214); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 214, 4);
            BitConverter.GetBytes(h.zfilter_code).CopyTo(buf, 218); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 218, 2);
            BitConverter.GetBytes(h.zfilter_order).CopyTo(buf, 220); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 220, 2);
            BitConverter.GetBytes(h.mt_1_4).CopyTo(buf, 222); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 222, 4);
            BitConverter.GetBytes(h.mt_2_4).CopyTo(buf, 226); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 226, 4);
            BitConverter.GetBytes(h.mt_3_4).CopyTo(buf, 230); if (BitConverter.IsLittleEndian) swapWordBytes(buf, 230, 4);
            BitConverter.GetBytes((short)h.scatter_type).CopyTo(buf, 234); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 234, 2);
            BitConverter.GetBytes((short)h.recon_type).CopyTo(buf, 236); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 236, 2);
            BitConverter.GetBytes(h.recon_views).CopyTo(buf, 238); if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 238, 2); 
            toBytesArray(h.fill_cti).CopyTo(buf, 240);
            toBytesArray(h.fill_user).CopyTo(buf, 414);
            Console.WriteLine("writing imageheader: "+buf.Length+" bytes to "+filestream.Position);
            filestream.Write(buf, 0, buf.Length);            
            filestream.Close();
        }
        /// <summary>
        /// Write ECAT 7.x matrix data to a specified file position.
        /// Data must be represented in current machines byte order, and it is
        /// always saved in big endian byte order.
        /// </summary>
        /// <param name="start_block">block number from where to start writing</param>
        /// <param name="data">data in byte array</param>
        /// <see cref="MatBLKSIZE"/>
        protected void ecat7WriteMatrixdata(int start_block, byte[] data) {
            byte[] buf = new byte[MatBLKSIZE];
            int i, blkNr, dataSize, byteNr;
            if(start_block < 1) throw new TPCInvalidArgumentsException("Cannot write matrix data before 1st block.");
            if(data.Length < 1) throw new TPCInvalidArgumentsException("No data in input array.");
            FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate);

            /* block nr taken by all pixels */
            blkNr = (data.Length+MatBLKSIZE-1)/MatBLKSIZE;
            /* Search the place for writing */
            filestream.Seek((start_block) * MatBLKSIZE, SeekOrigin.Begin);
            if (filestream.Position != (start_block) * MatBLKSIZE)
                throw new TPCEcat7FileException("Failed to seek correct point in file.");
            /* Write blocks one at a time */
            dataSize = data.Length;
            Console.WriteLine("writing data starting from "+filestream.Position);
            for(i = 0; i < blkNr; i++) {
                byteNr = (dataSize<MatBLKSIZE) ? dataSize : MatBLKSIZE;
                Array.Copy(data, i * MatBLKSIZE, buf, 0, byteNr);
                /* Change matrix byte order in little endian platforms */
                swabBytes(buf, header.datatype);
                /* Write block */
                filestream.Write(buf, 0, MatBLKSIZE);
                /* Prepare for the next block */
                dataSize -= byteNr;
                if (dataSize < 0) break;
            } /* next block */
            Console.WriteLine("writing data ends to " + filestream.Position);
            filestream.Close();
        }
        /// <summary>
        /// Read ECAT7 matrix data and convert byte order if necessary. Datatype 
        /// field is used to determine if byte swpping is necessary.
        /// </summary>
        /// <param name="start_block">Matrix subheader record number</param>
        /// <param name="end_block">last block to read</param>
        /// <returns></returns>
        protected byte[] ecat7ReadMatrixdata(int start_block, int end_block)
        {
            if (end_block <= start_block) throw new TPCEcat7FileException("end block is not after start block");
            byte[] buf = new byte[MatBLKSIZE * (end_block - start_block)];
            FileStream filestream = new FileStream(filename, FileMode.Open);
            int ret = 0;

            /* Check the arguments */
            if (start_block < 1) throw new TPCInvalidArgumentsException("Matrix subheader record number.");
            /* Seek the first data block */

            filestream.Seek((start_block) * MatBLKSIZE, SeekOrigin.Begin);
            if (filestream.Position != (start_block) * MatBLKSIZE) throw new TPCEcat7FileException("Failed to seek correct point in file.");
            /* Read the data blocks */
            try {
                ret = filestream.Read(buf, 0, MatBLKSIZE * (end_block - start_block));
            } catch(Exception e) {
                throw new TPCEcat7FileException("Error while readint image data:"+e.Message);
            }
            if (ret < MatBLKSIZE * (end_block - start_block))
                throw new TPCEcat7FileException("Failed to read all of " + MatBLKSIZE * (end_block - start_block) + " matrix data. Memory usage:"+System.GC.GetTotalMemory(false)+" bytes.");
            swabBytes(buf, header.datatype);
            filestream.Close();
            return buf;
        }
        /// <summary>
        /// Read ECAT7 image matrix data.
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="h">imageheader of matrix</param>
        /// <returns>matrix data that is read, allways dynamic</returns>
        protected DynamicImage ecat7ReadImageMatrix(Ecat7_MatDir matdir, Ecat7_imageheader h) {
            int i, pxlNr;
            byte[] buf;
            DynamicImage volume;
            IntLimits dim;

            /* Read subheader */
            pxlNr=h.x_dimension*h.y_dimension;
            if(h.num_dimensions > 2) pxlNr *= h.z_dimension;
            if(pxlNr<=0) {
                throw new TPCEcat7FileException("Invalid matrix dimensions: ["+h.x_dimension+", "+h.y_dimension+", "+h.z_dimension+"]");
            }

            /* Read matrix data */
            if(matdir.endblk < matdir.strtblk) throw new TPCInvalidArgumentsException("last_block number is smaller than first_block number");

            //read matrix data into buffer
            buf = ecat7ReadMatrixdata(matdir.strtblk, matdir.endblk);
            
            //create image
            dim = new IntLimits(h.x_dimension );
            if (h.y_dimension >= 1)
            {
                dim.addLimit(0, h.y_dimension);
                if (h.z_dimension >= 1)
                {
                    dim.addLimit(0, h.z_dimension);
                }
            }
            volume = new DynamicImage(dim);
            float value;
            
            /* Convert matrix data to floats */            
            switch(h.data_type) {
                case Ecat7_imageheader.data_type_e.Byte_Data:
                    for(i = 0; i < (int)volume.dim.getProduct(); i++) {
                        volume.setValue(i, (float)buf[i]);
                    }
                    break;
                case Ecat7_imageheader.data_type_e.VAX_Ix2:
                case Ecat7_imageheader.data_type_e.Sun_short:
                    for (i = 0; i < (int)volume.dim.getProduct(); i++)
                    {
                        value = (float)BitConverter.ToInt16(buf, i * 2);
                        volume.setValue(i, value);                        
                    }
                    break;
                case Ecat7_imageheader.data_type_e.VAX_Ix4:
                case Ecat7_imageheader.data_type_e.Sun_long:
                    for (i = 0; i < ((int)volume.dim.getProduct()) / 4; i++)
                    {
                        volume.setValue(i, (float)BitConverter.ToUInt32(buf, i * 4));
                    }
                    break;
                case Ecat7_imageheader.data_type_e.IEEE_Float:
                case Ecat7_imageheader.data_type_e.VAX_Rx4:
                    for (i = 0; i < ((int)volume.dim.getProduct()) / 4; i++)
                    {
                        volume.setValue(i, (float)BitConverter.ToSingle(buf, i * 4));
                    }
                    break;
                case Ecat7_imageheader.data_type_e.Unknown_Matrix_Data_Type:
                    break;
                default:
                    throw new TPCEcat7FileException("Unsupported image datatype:"+h.data_type);
            }
            volume.Multiply(h.scale_factor);
            return volume;
        }
        /// <summary>
        /// Reads a frame from ecat7 file and fills header and image variables.
        /// </summary>
        /// <param name="frame">frame index (0-based)</param>
        /// <exception cref="TPCEcat7FileException">if file was not opened properly</exception>
        public void readFrame(int frame) {
            throw new NotImplementedException();
        }
        /// <summary>
        /// Implementation of ImageFile interface. Reads a ecat7 file and fills header and image variables.
        /// </summary>
        /// <exception cref="TPCEcat7FileException">if file was not opened properly</exception>
        /// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
        public override void ReadFile() {
            int bed = 0;
            int gate = 0;
            int frame = 0;
            int i = 0;
            IntLimits dim = new IntLimits(new int[]{1, 1, 1, 1});
            DynamicImage volume;
            IOProcessEventHandler pevent = null;
            
            if(!checkFormat(filename)) throw new TPCEcat7FileException("Fileformat not recognized.");

            // Read main header
            try {
                mainheader = ecat7ReadMainheader(filename);
            } catch(Exception e) {         
                throw new TPCEcat7FileException("Failed to read main header:"+e.Message);
            }

            // Set fileFormat
            header.datatype = mainheader.resolveDataType();
            
            // Read matrix list 
            mlist = ecat7ReadMatlist(filename);
            //Array.Sort<Ecat7_MatDir>(mlist, new MatrixComparer());
            // Read subheaders, depending on image type
            switch(mainheader.resolveImageType()) {
                case Ecat7_mainheader.ecat7_imagetype.IMAGE:
                    subheaders = ecat7ReadImageheaders(mlist, filename);
                    if (subheaders.Length == 0) throw new TPCEcat7FileException("No image subheaders.");
                    i = 0;
                    while (i < subheaders.Length && subheaders[i] == null) i++;
                    if(i == subheaders.Length || subheaders[i] == null) throw new TPCEcat7FileException("Couldn't find any imageheaders.");
                    
                    if ((subheaders[i] as Ecat7_imageheader).x_dimension == 0) throw new TPCEcat7FileException("Image header x-dimension is zero");
                    if ((subheaders[i] as Ecat7_imageheader).y_dimension == 0) throw new TPCEcat7FileException("Image header y-dimension is zero");
                    if ((subheaders[i] as Ecat7_imageheader).z_dimension == 0) throw new TPCEcat7FileException("Image header z-dimension is zero");
                    //resolve image dimensions
                    dim = new IntLimits((subheaders[i] as Ecat7_imageheader).x_dimension);
                    if ((subheaders[i] as Ecat7_imageheader).y_dimension >= 1)
                    {
                        dim.addLimit(0, (subheaders[i] as Ecat7_imageheader).y_dimension);
                        if ((subheaders[i] as Ecat7_imageheader).z_dimension >= 1)
                        {
                            dim.addLimit(0, (subheaders[i] as Ecat7_imageheader).z_dimension);
                            frame = int.MinValue;
                            gate = int.MinValue;
                            bed = int.MinValue;
                            for (i = 0; i < mlist.Length; i++)
                            {
                                if (mlist[i].id.frame > frame) frame = mlist[i].id.frame;
                                if (mlist[i].id.gate > gate) gate = mlist[i].id.gate;
                                if (mlist[i].id.bed > bed) bed = mlist[i].id.bed;
                            }
                            if (frame > 1)
                            {
                                dim.addLimit(0, frame);
                                if (gate > 1)
                                {
                                    dim.addLimit(0, gate);
                                    if (bed > 1)
                                    {
                                        dim.addLimit(0, bed);
                                    }
                                }
                            }
                        }
                    }
                    header.dim = dim;
                    break;
                case Ecat7_mainheader.ecat7_imagetype.POLARMAP:
                    subheaders = ecat7ReadPolmapheaders(mlist, filename);
                    i = 0;
                    while (subheaders[i] == null) i++;
                    if (subheaders[i] == null) throw new TPCEcat7FileException("Couldn't find any polmapheader.");
                    if((subheaders[i] as Ecat7_polmapheader).num_rings == 0)
                        throw new TPCEcat7FileException("No rings in polar map");
                    break;
                case Ecat7_mainheader.ecat7_imagetype.RAW:
                    subheaders = ecat7ReadScanheaders(mlist, filename);
                    i = 0;
                    while (subheaders[i] == null) i++;
                    if (subheaders[i] == null) throw new TPCEcat7FileException("Couldn't find any scanheader.");
                    if((subheaders[i] as Ecat7_scanheader).num_r_elements == 0)
                        throw new TPCEcat7FileException("Scanheader r-dimension is zero");
                    if((subheaders[i] as Ecat7_scanheader).num_angles == 0)
                        throw new TPCEcat7FileException("Scanheader angles-dimension is zero");
                    break;
            }
            if (dim.getProduct() == 0) throw new TPCEcat7FileException("Image volume is zero");
            //create image
            image = new DynamicImage(dim);
            // Copy information from mainheader
            (image as DynamicImage).isotope = Isotope.createIsotope(new string(mainheader.isotope_name));
            (image as DynamicImage).dose_start_time = mainheader.dose_start_time;
            (image as DynamicImage).radiopharma = new string(mainheader.radiopharmaceutical);
            (header as DynamicImageHeader).dose_start_time = mainheader.dose_start_time;
            (header as DynamicImageHeader).injected_dose = mainheader.dosage;
            (header as DynamicImageHeader).isotope = Isotope.createIsotope(new string(mainheader.isotope_name));
            header.description = new string(mainheader.study_description);
            header.patient_name = new string(mainheader.patient_name);
            header.patientID = new string(mainheader.patient_id);
            header.dataunit = Unit.createUnit(new string(mainheader.data_units));
            /* Read matrices */
            for(i = 0; i < mlist.Length; i++) {
                
                //skip non-existing matrixes
                if (mlist[i].status != Ecat7_MatDir.status_e.EXISTS)
                {
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)i / mlist.Length));
                    continue;
                }
                //ecat7 is expected to have frame durations in milliseconds
                (image as DynamicImage).setFrameStartTime((int)mlist[i].id.frame - 1, subheaders[i].frame_start_time);
                (image as DynamicImage).setFrameDuration((int)mlist[i].id.frame-1, subheaders[i].frame_duration);
                /* Read subheader and data */
                switch(mainheader.resolveImageType()) {
                    case Ecat7_mainheader.ecat7_imagetype.IMAGE:
                        //ecat7PrintImageheader((subheaders[i] as Ecat7_imageheader)); 
                        if (dim.Length > IntLimits.FRAMES) dim.setLimits(IntLimits.FRAMES, (int)mlist[i].id.frame - 1, (int)mlist[i].id.frame - 1);
                        if (dim.Length > IntLimits.GATES) dim.setLimits(IntLimits.GATES, (int)mlist[i].id.gate - 1, (int)mlist[i].id.gate - 1);
                        if (dim.Length > IntLimits.BEDS) dim.setLimits(IntLimits.BEDS, (int)mlist[i].id.bed - 1, (int)mlist[i].id.bed - 1);
                        volume = ecat7ReadImageMatrix(mlist[i], (subheaders[i] as Ecat7_imageheader));
                        while (volume.dim.Length < image.dim.Length) volume.dim.addLimit(0, 1);
                        while (dim.Length < image.dim.Length) volume.dim.addLimit(0, 1);
                        image.setSubImage(dim, volume);
                        //convert from cm to mm
                        header.siz = new Voxel(10.0 * (subheaders[i] as Ecat7_imageheader).x_pixel_size,
                                               10.0 * (subheaders[i] as Ecat7_imageheader).y_pixel_size,
                                               10.0 * (subheaders[i] as Ecat7_imageheader).z_pixel_size);
                        break;
                    case Ecat7_mainheader.ecat7_imagetype.POLARMAP:
                        dim = new IntLimits(new int[] { 0, 0, (int)mlist[i].id.plane, (int)mlist[i].id.frame, (int)mlist[i].id.gate, (int)mlist[i].id.bed });
                        image.setSubImage(dim, ecat7ReadPolmapMatrix(mlist[i], subheaders[i] as Ecat7_polmapheader));
                        //set voxel so that it has desired size
                        header.siz = new Voxel(0.001 * (subheaders[i] as Ecat7_polmapheader).pixel_size, 1.0f, 1.0f);
                        break;
                    case Ecat7_mainheader.ecat7_imagetype.RAW:
                        dim = new IntLimits(new int[] { 0, 0, (int)mlist[i].id.plane, (int)mlist[i].id.frame, (int)mlist[i].id.gate, (int)mlist[i].id.bed });
                        image.setSubImage(dim, ecat7ReadScanMatrix(mlist[i], subheaders[i] as Ecat7_scanheader));
                        // also, correct for dead-time
                        if ((subheaders[i] as Ecat7_scanheader).deadtime_correction_factor > 0.0)
                        {
                            image.Multiply((subheaders[i] as Ecat7_scanheader).deadtime_correction_factor);
                        }
                        break;
                }
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)i / mlist.Length));
            }
            header.dim = image.dim;
            /* Calibrate */
            if(mainheader.ecat_calibration_factor > 0.0) {
                image.Multiply(mainheader.ecat_calibration_factor);
                header.dataunit = Unit.createUnit(new string(mainheader.data_units));
            }
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f));
        }
        /// <summary>
        /// Returns string representation of ecat7 file type.
        /// </summary>
        /// <param name="type">ecat7 file type</param>
        /// <returns>string representation of type</returns>
        public static string ecat7filetype(Ecat7_mainheader.datatype_e type)
        { 
            switch(type) {
                case Ecat7_mainheader.datatype_e.ecat7_UNKNOWN:
                    return "unknown";
                case Ecat7_mainheader.datatype_e.ecat7_2DSCAN:
                    return "2D sinogram";
                case Ecat7_mainheader.datatype_e.ecat7_IMAGE16:
                    return "image-16-bits";
                case Ecat7_mainheader.datatype_e.ecat7_ATTEN:
                    return "attenuation correction";
                case Ecat7_mainheader.datatype_e.ecat7_2DNORM:
                    return "2D normalization";
                case Ecat7_mainheader.datatype_e.ecat7_POLARMAP:
                    return "polar map";
                case Ecat7_mainheader.datatype_e.ecat7_VOLUME8:
                    return "volume 8-bits";
                case Ecat7_mainheader.datatype_e.ecat7_VOLUME16:
                    return "volume 16-bits";
                case Ecat7_mainheader.datatype_e.ecat7_PROJ:
                    return "projection 8-bits";
                case Ecat7_mainheader.datatype_e.ecat7_PROJ16:
                    return "projection 16-bits";
                case Ecat7_mainheader.datatype_e.ecat7_IMAGE8:
                    return "image 8-bits";
                case Ecat7_mainheader.datatype_e.ecat7_3DSCAN:
                    return "3D sinogram 16-bits";
                case Ecat7_mainheader.datatype_e.ecat7_3DSCAN8:
                    return "3D sinogram 8-bits";
                case Ecat7_mainheader.datatype_e.ecat7_3DNORM:
                    return "3D normalization";
                case Ecat7_mainheader.datatype_e.ecat7_3DSCANFIT:
                    return "3D sinogram fit";
            }
            return "Internal error: unrecognized enumerator value:"+type;
        }
        /// <summary>
        /// Returns string describing the ECAT7 acquisition_type
        /// </summary>
        /// <param name="acquisition_type">acquisition type</param>
        /// <returns>string representation of type</returns>
        public static string ecat7acquisitiontype(Ecat7_mainheader.Acquisition_type_e acquisition_type) {
            switch(acquisition_type) {
                case Ecat7_mainheader.Acquisition_type_e.Blank: return "blank";
                case Ecat7_mainheader.Acquisition_type_e.Dynamic_emission: return "dynamic emission";
                case Ecat7_mainheader.Acquisition_type_e.Emission_rectilinear: return "emission rectilinear";
                case Ecat7_mainheader.Acquisition_type_e.Gated_emission: return "gated emission";
                case Ecat7_mainheader.Acquisition_type_e.Static_emission: return "static emission";
                case Ecat7_mainheader.Acquisition_type_e.Transmission: return "transmission";
                case Ecat7_mainheader.Acquisition_type_e.Transmission_rectilinear: return "transmission rectilinear";
                case Ecat7_mainheader.Acquisition_type_e.Undefined: return "undefined";
            }
            return "Internal error: unrecognized enumerator value:" + acquisition_type;
        }
        /// <summary>
        /// Prints character array to standard output.
        /// </summary>
        /// <param name="title">title of line</param>
        /// <param name="array">char data in array</param>
        private static void printCharArray(string title, char[] array) {
            Console.Write(title);
            for (int i = 0; i < array.Length; i++) Console.Write("{0:s}", array[i]);
            Console.WriteLine();
        }
        /// <summary>
        /// Print ECAT 7.x image header contents
        /// </summary>
        /// <param name="h">image header object</param>
        public static void ecat7PrintImageheader(Ecat7_imageheader h)
        {
            Console.WriteLine("data_type := {0:g} ({1:s})", h.data_type, Ecat7_imageheader.ToString(h.data_type));
            Console.WriteLine("num_dimensions := {0:n}", h.num_dimensions);
            Console.WriteLine("x_dimension := {0:n}", h.x_dimension);
            Console.WriteLine("y_dimension := {0:n}", h.y_dimension);
            Console.WriteLine("z_dimension := {0:n}", h.z_dimension);
            Console.WriteLine("x_offset := {0:g}", h.x_offset);
            Console.WriteLine("y_offset := {0:g}", h.y_offset);
            Console.WriteLine("z_offset := {0:g}", h.z_offset);
            Console.WriteLine("recon_zoom := {0:g}", h.recon_zoom);
            Console.WriteLine("scale_factor := {0:E}", h.scale_factor);
            Console.WriteLine("image_min := {0:g}", h.image_min);
            Console.WriteLine("image_max := {0:g}", h.image_max);
            Console.WriteLine("x_pixel_size := {0:g}", h.x_pixel_size);
            Console.WriteLine("y_pixel_size := {0:g}", h.y_pixel_size);
            Console.WriteLine("z_pixel_size := {0:g}", h.z_pixel_size);
            Console.WriteLine("frame_duration := {0:g}", h.frame_duration);
            Console.WriteLine("frame_start_time := {0:g}", h.frame_start_time);
            Console.WriteLine("filter_code := {0:g}", h.filter_code);
            Console.WriteLine("x_resolution := {0:g}", h.x_resolution);
            Console.WriteLine("y_resolution := {0:g}", h.y_resolution);
            Console.WriteLine("z_resolution := {0:g}", h.z_resolution);
            Console.WriteLine("num_r_elements := {0:g}", h.num_r_elements);
            Console.WriteLine("num_angles := {0:g}", h.num_angles);
            Console.WriteLine("z_rotation_angle := {0:g}", h.z_rotation_angle);
            Console.WriteLine("decay_corr_fctr := {0:g}", h.decay_corr_fctr);
            Console.WriteLine("processing_code := {0:n}", h.processing_code);
            Console.WriteLine("gate_duration := {0:n}", h.gate_duration);
            Console.WriteLine("r_wave_offset := {0:n}", h.r_wave_offset);
            Console.WriteLine("num_accepted_beats := {0:n}", h.num_accepted_beats);
            Console.WriteLine("filter_cutoff_frequency := {0:E}", h.filter_cutoff_frequency);
            Console.WriteLine("filter_resolution := {0:E}", h.filter_resolution);
            Console.WriteLine("filter_ramp_slope := {0:E}", h.filter_ramp_slope);
            Console.WriteLine("filter_order := {0:n}", h.filter_order);
            Console.WriteLine("filter_scatter_fraction := {0:E}", h.filter_scatter_fraction);
            Console.WriteLine("filter_scatter_slope := {0:E}", h.filter_scatter_slope);
            Console.WriteLine("annotation := {0:40:s}", new string(h.annotation));
            Console.WriteLine("mt_1_1 := {0:g}", h.mt_1_1);
            Console.WriteLine("mt_1_2 := {0:g}", h.mt_1_2);
            Console.WriteLine("mt_1_3 := {0:g}", h.mt_1_3);
            Console.WriteLine("mt_2_1 := {0:g}", h.mt_2_1);
            Console.WriteLine("mt_2_2 := {0:g}", h.mt_2_2);
            Console.WriteLine("mt_2_3 := {0:g}", h.mt_2_3);
            Console.WriteLine("mt_3_1 := {0:g}", h.mt_3_1);
            Console.WriteLine("mt_3_2 := {0:g}", h.mt_3_2);
            Console.WriteLine("mt_3_3 := {0:g}", h.mt_3_3);
            Console.WriteLine("rfilter_cutoff := {0:g}", h.rfilter_cutoff);
            Console.WriteLine("rfilter_resolution := {0:g}", h.rfilter_resolution);
            Console.WriteLine("rfilter_code := {0:n}", h.rfilter_code);
            Console.WriteLine("rfilter_order := {0:n}", h.rfilter_order);
            Console.WriteLine("zfilter_cutoff := {0:g}", h.zfilter_cutoff);
            Console.WriteLine("zfilter_resolution := {0:g}", h.zfilter_resolution);
            Console.WriteLine("zfilter_code := {0:n}", h.zfilter_code);
            Console.WriteLine("zfilter_order := {0:n}", h.zfilter_order);
            Console.WriteLine("mt_1_4 := {0:g}", h.mt_1_4);
            Console.WriteLine("mt_2_4 := {0:g}", h.mt_2_4);
            Console.WriteLine("mt_3_4 := {0:g}", h.mt_3_4);
            Console.WriteLine("scatter_type := {0:g}", h.scatter_type);
            Console.WriteLine("recon_type := {0:g}", h.recon_type);
            Console.WriteLine("recon_views := {0:n}", h.recon_views);
            Console.Write("fill_cti :=");
            for (int i = 0; i < 87; i++) Console.Write(" {0:n}", h.fill_cti[i]); 
            Console.WriteLine();
            Console.Write("fill_user :=");
            for (int i = 0; i < 48; i++) Console.Write(" {0:n}", h.fill_user[i]); 
            Console.WriteLine();
        }
        /// <summary>
        /// Print ECAT 7.x main header contents
        /// </summary>
        /// <param name="h">main header object</param>
        public static void ecat7PrintMainheader(Ecat7_mainheader h) {
            printCharArray("magic_number := ", h.magic_number);
            printCharArray("original_file_name := ", h.original_file_name);
            Console.WriteLine("sw_version := {0:g}", h.sw_version);
            Console.WriteLine("system_type := {0:g}", h.system_type);
            Console.WriteLine("file_type := {0:g} ({1:s})", h.file_type, ecat7filetype(h.file_type));
            printCharArray("serial_number := ", h.serial_number);
            DateTime scan_start_time = new DateTime((long)h.scan_start_time);
            scan_start_time.ToLocalTime();
            Console.WriteLine("scan_start_time := {0:r}", scan_start_time);
            printCharArray("isotope_name := ", h.isotope_name);
            Console.WriteLine("isotope_halflife := {0:e} sec", h.isotope_halflife);
            printCharArray("radiopharmaceutical := ", h.radiopharmaceutical);
            Console.WriteLine("gantry_tilt := {0:g}", h.gantry_tilt);
            Console.WriteLine("gantry_rotation := {0:g}", h.gantry_rotation);
            Console.WriteLine("bed_elevation := {0:g}", h.bed_elevation);
            Console.WriteLine("intrinsic_tilt := {0:g}", h.intrinsic_tilt);
            Console.WriteLine("wobble_speed := {0:g}", h.wobble_speed);
            Console.WriteLine("transm_source_type := {0:d}", h.transm_source_type);
            Console.WriteLine("distance_scanned := {0:g}", h.distance_scanned);
            Console.WriteLine("transaxial_fov := {0:g}", h.transaxial_fov);
            Console.WriteLine("angular_compression := {0:g}", h.angular_compression);
            Console.WriteLine("coin_samp_mode := {0:g}", h.coin_samp_mode);
            Console.WriteLine("axial_samp_mode := {0:g}", h.axial_samp_mode);
            Console.WriteLine("ecat_calibration_factor := {0:e}", h.ecat_calibration_factor);
            Console.WriteLine("calibration_units := {0:g}", h.calibration_units);
            Console.WriteLine("calibration_units_label := {0:g}", h.calibration_units_label);
            Console.WriteLine("compression_code := {0:g}", h.compression_code);
            printCharArray("study_type := ", h.study_type);
            printCharArray("patient_id := ", h.patient_id);
            printCharArray("patient_name := ", h.patient_name);
            Console.WriteLine("patient_sex := {0:g}", h.patient_sex);
            Console.WriteLine("patient_dexterity := {0:s}", (h.patient_dexterity != 0) ? h.patient_dexterity : (char)32);
            Console.WriteLine("patient_age := {0:g}", h.patient_age);
            Console.WriteLine("patient_height := {0:g}", h.patient_height);
            Console.WriteLine("patient_weight := {0:g}", h.patient_weight);
            Console.WriteLine("patient_birth_date := {0:g}", h.patient_birth_date);
            printCharArray("physician_name := ", h.physician_name);
            printCharArray("operator_name := ", h.operator_name);
            printCharArray("study_description := ", h.study_description);
            Console.WriteLine("acquisition_type := {0:g} ({1:s})", h.acquisition_type, ecat7acquisitiontype(h.acquisition_type));
            Console.WriteLine("patient_orientation := {0:g}", h.patient_orientation);
            printCharArray("facility_name := ", h.facility_name);
            Console.WriteLine("num_planes := {0:g}", h.num_planes);
            Console.WriteLine("num_frames := {0:g}", h.num_frames);
            Console.WriteLine("num_gates := {0:g}", h.num_gates);
            Console.WriteLine("num_bed_pos := {0:g}", h.num_bed_pos);
            Console.WriteLine("init_bed_position := {0:g}", h.init_bed_position);
            Console.Write("bed_position :=");
            for (int i = 0; i < 15; i++) Console.Write(" {0:g}", h.bed_position[i]);
            Console.WriteLine();
            Console.WriteLine("plane_separation := {0:g} cm", h.plane_separation);
            Console.WriteLine("lwr_sctr_thres := {0:g}", h.lwr_sctr_thres);
            Console.WriteLine("lwr_true_thres := {0:g}", h.lwr_true_thres);
            Console.WriteLine("upr_true_thres := {0:g}", h.upr_true_thres);
            printCharArray("user_process_code := ", h.user_process_code);
            Console.WriteLine("acquisition_mode := {0:g}", h.acquisition_mode);
            Console.WriteLine("bin_size := {0:g} cm", h.bin_size);
            Console.WriteLine("branching_fraction := {0:g}", h.branching_fraction);
            DateTime dose_start_time = new DateTime((long)h.dose_start_time);
            dose_start_time.ToLocalTime();
            Console.WriteLine("dose_start_time := {0:r}", dose_start_time);
            Console.WriteLine("dosage := {0:g}", h.dosage);
            Console.WriteLine("well_counter_corr_factor := {0:e}", h.well_counter_corr_factor);
            printCharArray("data_units := ", h.data_units);
            Console.WriteLine("septa_state := {0:g}", h.septa_state);
            Console.Write("fill_cti :=");
            for(int i=0; i < 6; i++) Console.Write(" {0:g}", h.fill[i]);
            Console.WriteLine();
        }
        /// <summary>
        /// Implementation of ImageFile interface. Writes a ecat7 file.
        /// </summary>
        /// <exception cref="TPCEcat7FileException">if file was not opened properly</exception>
        /// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
        public override void WriteFile()
        {
            //exception because other than 16-bit signed data is not tested.
            if (header.datatype != DataType.BIT16_S) throw new TPCEcat7FileException("Writing of other than 16-bit signed data is not supported.");

            Ecat7_Matval matrixId;
            Ecat7_imageheader h = new Ecat7_imageheader();
            double[] frame_starttimes = null;
            double[] frame_durations = null; 
            int frame = 0;
            int gate = 0;
            int bed = 0;
            int frames;
            int gates;
            int beds;
            int i = 0;
            int si = 0;
            int j = 0;
            int k = 0;
            int datatype_bytes = type2Bytes(header.datatype);
            byte[] data;
            byte[] valuedata = new byte[datatype_bytes];
            int[] minmax = new int[2];
            float[] minmaxf = new float[2];
            Image frameImage = null;
            IntLimits frameLimits;
            IOProcessEventHandler pevent = null;
            float[] mm;

            /* Write main header */
            if (header.dim.Length > IntLimits.BEDS) mainheader.num_bed_pos = (short)header.dim.getDimension(IntLimits.BEDS);
            else mainheader.num_bed_pos = 0;
            if (header.dim.Length > IntLimits.GATES) mainheader.num_gates = (short)header.dim.getDimension(IntLimits.GATES);
            else mainheader.num_gates = 1;
            if (header.dim.Length > IntLimits.FRAMES) mainheader.num_frames = (short)header.dim.getDimension(IntLimits.FRAMES);
            else mainheader.num_frames = 1;
            if (header.dim.Length > IntLimits.PLANES) mainheader.num_planes = (short)header.dim.getDimension(IntLimits.PLANES);
            else mainheader.num_planes = 1;
            if (mainheader.num_gates == 0) mainheader.num_gates = 1;
            if (mainheader.num_frames == 0) mainheader.num_frames = 1;
            frames = mainheader.num_frames;
            gates = mainheader.num_gates;
            beds = mainheader.num_bed_pos;
            if (beds == 0) beds = 1;
            if (image is DynamicImage) {
                mainheader.radiopharmaceutical = (image as DynamicImage).radiopharma.ToCharArray();
                mainheader.isotope_halflife = (float)(image as DynamicImage).isotope.halflife;
                mainheader.isotope_name = (image as DynamicImage).isotope.isotopeCode.ToCharArray();
            }
            if (header is DynamicImageHeader)
            {
                mainheader.dosage = (header as DynamicImageHeader).injected_dose;
                mainheader.dose_start_time = (int)((header as DynamicImageHeader).dose_start_time % int.MaxValue);
            }
            mainheader.data_units = header.dataunit.unitname.ToCharArray();
            mainheader.study_description = header.description.ToCharArray();
            
            header.patient_name.CopyTo(0, mainheader.patient_name, 0, (32 < header.patient_name.Length ? 32 : header.patient_name.Length));
            header.patientID.CopyTo(0, mainheader.patient_id, 0, (16 < header.patientID.Length ? 16 : header.patientID.Length));
            header.description.CopyTo(0, mainheader.study_description, 0, (32 < header.description.Length ? 32 : header.description.Length));
            header.dataunit.unitname.CopyTo(0, mainheader.data_units, 0, (32 < header.dataunit.unitname.Length ? 32 : header.dataunit.unitname.Length));
            if(mainheader.file_type == Ecat7_mainheader.datatype_e.ecat7_UNKNOWN)
                mainheader.file_type = mainheader.resolveFileType(header.datatype);
            ecat7WriteMainheader(filename, mainheader);
          
            /* Resolve matrix lists */
            updateMatrixList();
            ecat7WriteMatlist(filename, matrixlist);
            
            if (image is DynamicImage) {
                frame_starttimes = (image as DynamicImage).getFrameStartTimes();
                frame_durations = (image as DynamicImage).getFrameDurations();
            }     

            /* write image data */
            frameLimits = new IntLimits(image.dim.getDimension(Limits.WIDTH),
                                        image.dim.getDimension(Limits.HEIGHT),
                                        image.dim.getDimension(Limits.PLANES));
            while(frameLimits.Length < image.dim.Length)
                frameLimits.addLimit(0, 1);
            data = new byte[image.width * image.height * image.planes * datatype_bytes];
            i = 0;

            for (bed = 0; bed < beds; bed++)
            {
                for (gate = 1; gate <= gates; gate++)
                {
                    for (frame = 1; frame <= frames; frame++)
                    {
                        while (si < subheaders.Length && subheaders[si] == null) si++;
                        if (si == subheaders.Length || subheaders[si] == null) 
                            throw new TPCEcat7FileException("Couldn't find subheader (bed="+bed+" gate="+gate+" frame="+frame+" subheader index="+si+").");
                        h.data_type = Ecat7_imageheader.convertFrom(header.datatype);
                        h.num_dimensions = 3;                        
                        h.x_dimension = (short)header.dimx;
                        h.y_dimension = (short)header.dimy;
                        h.z_dimension = (short)header.dimz;
                        h.x_pixel_size = header.sizex / 10.0f;
                        h.y_pixel_size = header.sizey / 10.0f;
                        h.z_pixel_size = header.sizez / 10.0f;
                        if (image is DynamicImage)
                        {
                            if (frame > frame_starttimes.Length) throw new TPCEcat7FileException("Number of frame times is exceeded.");
                            h.frame_start_time = (int)frame_starttimes[frame - 1];
                            h.frame_duration = (int)frame_durations[frame - 1];
                        }
                        minmax = mainheader.resolveMinMax();
                        h.image_min = (short)minmax[0];
                        h.image_max = (short)minmax[1];

                        if (image.dim.Length > Limits.FRAMES)
                        {
                            frameLimits.setLimit(Limits.FRAMES, Limits.Limit.LOW, frame-1);
                            frameLimits.setLimit(Limits.FRAMES, Limits.Limit.HIGH, frame);
                        } 
                        
                        mm = image.getMinMax();
                        frameImage = image.getSubImage(frameLimits);
                        mm = frameImage.getMinMax();
                        //resolve scale factor
                        switch (header.datatype)
                        {
                            case DataType.BIT8_U:
                                h.scale_factor = 255.0f/mm[1];
                                h.image_min = 0;
                                h.image_max = 255;
                                break;
                            case DataType.BIT64_S:
                            case DataType.BIT32_S:
                            case DataType.BIT16_S:
                                if (Math.Abs(mm[1]) > Math.Abs(mm[0]))
                                {
                                    h.scale_factor =  mm[1]/Int16.MaxValue;
                                    h.image_min = Int16.MinValue;
                                    h.image_max = Int16.MaxValue;
                                }
                                else {
                                    h.scale_factor = Math.Abs(mm[0] / Int16.MaxValue);
                                    h.image_min = Int16.MinValue;
                                    h.image_max = Int16.MaxValue;
                                }
                                break;
                            case DataType.BIT64_U:
                                h.scale_factor = mm[1] / Int64.MaxValue;
                                h.image_min = 0;
                                h.image_max = Int16.MaxValue;
                                break;
                            case DataType.BIT32_U:
                                h.scale_factor = mm[1] / Int32.MaxValue;
                                h.image_min = 0;
                                h.image_max = Int16.MaxValue;
                                break;
                            case DataType.BIT16_U:
                                h.scale_factor = mm[1] / UInt16.MaxValue;
                                h.image_min = 0;
                                h.image_max = Int16.MaxValue;
                                break;
                            default:
                                h.scale_factor = subheaders[si].scale_factor;
                                break;
                        }
                        if (h.scale_factor == 0) h.scale_factor = 1.0f;
                        frameImage.Multiply(1.0f/h.scale_factor);
                        for (j = 0; j < image.width * image.height * image.planes; j++)
                        {
                            switch(header.datatype) {
                                case DataType.BIT16_S: 
                                    Int16 int16value = (Int16)(frameImage[j]);
                                    valuedata = BitConverter.GetBytes(int16value);
                                    break;
                                case DataType.BIT16_U:
                                    UInt16 uint16value = (UInt16)Math.Round(frameImage[j]);
                                    valuedata = BitConverter.GetBytes(uint16value);
                                    break;
                                case DataType.BIT32_S:
                                    Int32 int32value = (Int32)Math.Round(frameImage[j]);
                                    valuedata = BitConverter.GetBytes(int32value);
                                     break;
                                case DataType.BIT32_U:
                                     UInt32 uint32value = (UInt32)Math.Round(frameImage[j]);
                                    valuedata = BitConverter.GetBytes(uint32value);
                                    break;
                                case DataType.BIT64_S:
                                    Int64 int64value = (Int64)Math.Round(frameImage[j]);
                                    valuedata = BitConverter.GetBytes(int64value);
                                    break;
                                case DataType.BIT64_U:
                                    UInt64 uint64value = (UInt64)Math.Round(frameImage.getValue(j));
                                    valuedata = BitConverter.GetBytes(uint64value);
                                    break;
                                case DataType.FLT32:
                                    valuedata = BitConverter.GetBytes(frameImage.getValue(j)); break;
                                default:
                                    throw new TPCEcat7FileException("Unsupported write data type:"+header.datatype);
                            }
                            for (k = 0; k < valuedata.Length; k++)
                            {
                                data[j * valuedata.Length + k] = valuedata[k];                                
                            }
                        }
                        //write subheader 
                        ecat7WriteImageheader(matrixlist[i], h, filename);
                        //write data
                        ecat7WriteMatrixdata(matrixlist[i].strtblk, data);                       
                        i++;
                        si++;
                        pevent = IOProgress;
                        if (pevent != null)
                            pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)i / matrixlist.Length));
                    }
                }
            }
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f));          
        }
        /// <summary>
        /// Checks file format. Tries to read magic number from file.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DICOM file</returns>
        /// <exception cref="TPCEcat7FileException">if there was a read problem</exception>
        public static bool checkFormat(string filename)
        {
            char[] bytes = new char[7];
            string stringbuffer = "";
            StreamReader fs = new StreamReader(filename, true);
            fs.Read(bytes, 0, 7);
            fs.Close();
            stringbuffer = new string(bytes);
            //compare only 7 first characters
            return stringbuffer.Equals(Ecat7File.ecat7V_MAGICNR.Substring(0,7));
        }
        /// <summary>
        /// Returns image object if it is present.
        /// </summary>
        /// <returns>image data</returns>
        public DynamicImage getImage()
        {
            if (image == null) throw new TPCException("Tried to get image which is null.");
            if(image is DynamicImage) return (image as DynamicImage);
            else return new DynamicImage(image);
        }
        /// <summary>
        /// Read ECAT7 3D sinogram matrix data.
        /// Note: data is converted to floats with scale_factor in the scan matrix header.
        /// Note: data is not calibrated with ecat_calibration_factor in main header.
        /// Note: data is not multiplied with deadtime_correction_factor.
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="h">scanheader of matrix</param>
        /// <returns>matrix data that is read, allways dynamic</returns>
        protected DynamicImage ecat7ReadScanMatrix(Ecat7_MatDir matdir, Ecat7_scanheader h)
        {
            int i, blockNr, trueblockNr, pxlNr, dimz;
            byte[] buf;
            DynamicImage volume;

            /* Read subheader */
            pxlNr=h.num_r_elements*h.num_angles;
            for(i=dimz=0; i<64; i++)
                dimz+=h.num_z_elements[i]; 
            pxlNr*=dimz;
            if(pxlNr<=0) {
                throw new TPCEcat7FileException("Invalid matrix dimensions: ["+h.num_r_elements+", "+h.num_angles+"]");
            }

            /* Read matrix data */
            blockNr = matdir.endblk-(matdir.strtblk+2);
            if(blockNr < 0) throw new TPCInvalidArgumentsException("not enough space between last_block and first_block");
            trueblockNr=pxlNr*type2Bytes(h.resolveDataType());
            trueblockNr=(trueblockNr+MatBLKSIZE-1)/MatBLKSIZE;
            if(blockNr < trueblockNr) trueblockNr=blockNr;

            //read matrix data into buffer
            buf = ecat7ReadMatrixdata(matdir.strtblk + 1, matdir.strtblk + 1+trueblockNr);

            volume = new DynamicImage(new IntLimits(new int[] { h.num_r_elements-1, h.num_angles-1, 1, 1, 1, 1 }));

            /* Convert matrix data to floats */
            switch (h.resolveDataType())
            {
                case DataType.BIT16_S:
                    throw new System.NotImplementedException();
                case DataType.VAXI16:
                case DataType.SUNI2:
                    throw new System.NotImplementedException();
                case DataType.VAXI32:
                case DataType.SUNI4:
                    throw new System.NotImplementedException();
                case DataType.VAXFL32:
                case DataType.FLT32:
                    throw new System.NotImplementedException();
            }
            return volume;
        }
        /// <summary>
        /// Read ECAT7 3D polarmap matrix data.
        /// </summary>
        /// <param name="matdir">matrix directory entry</param>
        /// <param name="h">polarmapheader of matrix</param>
        /// <returns>matrix data that is read, allways dynamic</returns>
        protected DynamicImage ecat7ReadPolmapMatrix(Ecat7_MatDir matdir, Ecat7_polmapheader h)
        {
            int i, blockNr, trueblockNr, pxlNr;
            byte[] buf;
            DynamicImage volume;

            /* Read subheader */
            for (i = pxlNr = 0; i < h.num_rings; i++) pxlNr += h.sectors_per_ring[i];
            if (pxlNr <= 0)
            {
                throw new TPCEcat7FileException("Invalid matrix dimensions: no sectors found in rings");
            }

            /* Read matrix data */
            blockNr = matdir.endblk - (matdir.strtblk + 1);
            if (blockNr < 0) throw new TPCInvalidArgumentsException("not enough space between last_block and first_block");
            trueblockNr = pxlNr * type2Bytes(h.resolveDataType());
            trueblockNr = (trueblockNr + MatBLKSIZE - 1) / MatBLKSIZE;
            if (blockNr < trueblockNr) trueblockNr = blockNr;

            //read matrix data into buffer
            buf = ecat7ReadMatrixdata(matdir.strtblk + 1, matdir.strtblk + 1+trueblockNr);

            volume = new DynamicImage(new IntLimits(new int[] { pxlNr-1, 1, 1, 1, 1, 1 }));

            /* Convert matrix data to floats */
            switch (h.resolveDataType())
            {
                case DataType.BIT16_S:
                    throw new System.NotImplementedException();
                case DataType.VAXI16:
                case DataType.SUNI2:
                    throw new System.NotImplementedException();
                case DataType.VAXI32:
                case DataType.SUNI4:
                    throw new System.NotImplementedException();
                case DataType.VAXFL32:
                case DataType.FLT32:
                    throw new System.NotImplementedException();
            }
            return volume;
        }
        /// <summary>
        /// Updates Ecat7 matrix list to apply current image data. 'data' field in matrix indentifier 
        /// is not supported in matrix list items (it is allways set to zero).
        /// </summary>
        /// <returns>matrix list that is consistent with image data</returns>
        public void updateMatrixList() {
            int product = 1;
            if(image.dim.Length < 2) throw new TPCEcat7FileException("1-dimensional data is not supported by this method.");
            //calculate number of voxels in single image frame (or gate etc.)
            if (image.dim.Length < 3)
            {
                product = image.dim[0] * image.dim[1];
            }
            else {
                product = image.dim[0] * image.dim[1] * image.dim[2];
            }
            Console.WriteLine(image.dim);
            //number of frames
            int frames = 1;
            if (image.dim.Length > IntLimits.FRAMES) frames = image.dim.getDimension(IntLimits.FRAMES);
            //number of gates
            int gates = 1;
            if (image.dim.Length > IntLimits.GATES) gates = image.dim.getDimension(IntLimits.GATES);
            //number of bed positions
            int beds = 1;
            if (image.dim.Length > IntLimits.BEDS) beds = image.dim.getDimension(IntLimits.BEDS);
            //number of blocks in item, resized to match 512 blocks 
            int blocks = (int)Math.Ceiling((double)(product*ImageFile.type2Bytes(header.datatype)) / 512.0);
            //create matrix list, resized to match 31 item blocks
            int list_size = (int)(Math.Ceiling((double)(frames*gates*beds)/31.0)*31.0);
            Console.WriteLine(list_size + "=" + frames + "*" + gates + "*" + beds);
            mlist = new Ecat7_MatDir[list_size];
            //set data in matrix list
            int blk = 3;
            int list_i = 0;
            for (int frame_i = 1; frame_i <= frames; frame_i++) {
            for (int gate_i = 1; gate_i <= gates; gate_i++) {
            for (int bed_i = 0; bed_i < beds; bed_i++) {
                //set list item
                mlist[list_i].id = new Ecat7_Matval(frame_i, 1, gate_i, 0, bed_i);
                mlist[list_i].strtblk = blk;
                mlist[list_i].endblk = blk + blocks;
                mlist[list_i].status = Ecat7_MatDir.status_e.EXISTS;
                //set data for next round
                blk = mlist[list_i].endblk + 1;
                //add space for matrix list header block when item block of 31 items becomes full
                if ((list_i+1) % 31 == 0) blk++;
                list_i++;
            }
            }
            }
            //fill rest of the list with empty entries
            while (list_i < mlist.Length)
            {
                //set list item
                mlist[list_i].id = new Ecat7_Matval(0, 0, 0, 0, 0);
                mlist[list_i].strtblk = 0;
                mlist[list_i].endblk = 0;
                mlist[list_i].status = Ecat7_MatDir.status_e.DELETED;
                list_i++;
            }
            Console.WriteLine(image.dim);
            Console.WriteLine(header.dim);
            for (int i = 0; i < mlist.Length; i++) {
                Console.WriteLine(mlist[i]);
            }
        }
    }   
}
