﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    /// <summary>
    /// General header field list class containing key-data item pairs. Only one value 
    /// can exist for each field name, case-insensitive.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class HeaderFieldList: List<KeyValuePair<string, Object>>
    {
        /// <summary>
        /// Default constructor
        /// </summary>
        public HeaderFieldList()
        {
        }
        /// <summary>
        /// Field name that is used in next search for matching pairs from list.
        /// </summary>
        private static string searchedFieldName;
        /// <summary>
        /// Search operator fo field name.
        /// </summary>
        /// <param name="p">key pair that is evaluated</param>
        /// <returns></returns>
        private static bool NameMatch(KeyValuePair<string, Object> p)
        {
            //case-insensitive search
            return (string.Compare(p.Key, searchedFieldName, false) == 0);
        }
        /// <summary>
        /// Returns true if name is found from header field list. 
        /// </summary>
        /// <param name="name">name that is searched for</param>
        /// <returns>true if name was found from list</returns>
        public bool Contains(string name) {
            searchedFieldName = name;
            int i = base.FindIndex(NameMatch);
            return (i > -1);
        }
        /// <summary>
        /// Gets value defined by field.
        /// </summary>
        /// <typeparam name="T">Type to return</typeparam>
        /// <param name="name">field name</param>
        /// <returns>field data as instance of T</returns>
        public T Get<T>(string name)
        {
            return (T)Convert.ChangeType(this[name], typeof(T));
        }
        /// <summary>
        /// Gets value defined by field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <returns>field data object</returns>
        /// <exception cref="TPCInvalidArgumentsException">if field is not found from header</exception>
        public Object GetValue(string fieldName)
        {
            searchedFieldName = fieldName;
            int i = base.FindIndex(NameMatch);
            if (i == -1)
                throw new TPCInvalidArgumentsException("Field [" + fieldName + "] was not found from header.");
            return this[i].Value;
        }
        /// <summary>
        /// Gets or sets value defined by field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <exception cref="TPCInvalidArgumentsException">if field was not found from header field list</exception>
        /// <returns>field data object</returns>
        public object this[string fieldName]
        {
            get
            {
                searchedFieldName = fieldName;
                int i = base.FindIndex(NameMatch);
                if (i == -1)
                    throw new TPCInvalidArgumentsException("Field [" + fieldName + "] was not found from header.");
                return this[i].Value;
            }
            set {
                searchedFieldName = fieldName;
                int i = base.FindIndex(NameMatch);
                if (i != -1)
                {
                    this[i] = new KeyValuePair<string, object>(fieldName, value);
                }
                else
                {
                    this.Add(fieldName, value);
                }
            }
        }
        /// <summary>
        /// Sets value of a field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <param name="value">new field value</param>
        public void SetValue(string fieldName, Object value)
        {
            searchedFieldName = fieldName;
            int i = base.FindIndex(NameMatch);
            this[i] = new KeyValuePair<string,object>(fieldName, value);
        }
        /// <summary>
        /// Adds new item into list.
        /// </summary>
        /// <param name="fieldname">field name</param>
        /// <param name="o">field data</param>
        /// <exception cref="TPCException">if the same name already exists in list</exception>
        public void Add(string fieldname, object o)
        {
            base.Add(new KeyValuePair<string, object>(fieldname, o));
        }
        /// <summary>
        /// Adds new item into list.
        /// </summary>
        /// <param name="p">new item to add</param>
        /// <exception cref="TPCException">if the same name already exists in list</exception>
        public new void Add(KeyValuePair<string,object> p) {
            searchedFieldName = p.Key;
            int i = this.FindIndex(NameMatch);
            if (i != -1)
                throw new TPCGeneralHeaderException("Name ["+p.Key+"] already exists in the list.",this[i]);
            base.Add(p);
        }
        /// <summary>
        /// Removes the occurrence with specific field name from the list.
        /// </summary>
        /// <param name="fieldname">name of the field</param>
        public void Remove(string fieldname) {
            searchedFieldName = fieldname;
            int i = base.FindIndex(NameMatch);
            if (i != -1) base.Remove(base[i]);
        }
        /// <summary>
        /// Returns all field names. Names are in the order they 
        /// appear in list.
        /// </summary>
        /// <returns>field names in list</returns>
        public string[] GetFieldnames() {
            string[] r = new string[this.Count];
            for(int i = 0; i < this.Count; i++) {
                r[i] = this[i].Key;                 
            }
            return r;
        }
        /// <summary>
        /// Returns all values. Values are in the order they 
        /// appear in list.
        /// </summary>
        /// <returns>field names in list</returns>
        public object[] GetValues()
        {
            object[] r = new object[this.Count];
            for (int i = 0; i < this.Count; i++)
            {
                r[i] = this[i].Value;
            }
            return r;
        }
        /// <summary>
        /// Comparator for sorting fields by their field names.
        /// </summary>
        /// <param name="x">list item</param>
        /// <param name="y">list item</param>
        /// <returns>-1 (y greater),0 (equal) or 1 (x greater) depending on order</returns>
        private static int CompareFieldsByName(KeyValuePair<string, Object> x, KeyValuePair<string, Object> y) {
            return string.Compare(x.Key, y.Key);
        }
        /// <summary>
        /// Sorts name-value pairs by names in ascending alphapetical order.
        /// </summary>
        public void SortByNames() {
            this.Sort(CompareFieldsByName);
        }
    }
}
