/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections;
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    /// <summary>
    /// Represents an image in N-dimensional space.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Image
    {
        /// <summary>
        /// Data of image in one array. The data represents
        /// actual true values (already scaled if it was necessary).
        /// </summary>
        protected float[] data;
        /// <summary>
        /// Gets data of image.
        /// </summary>
        protected virtual float[] Data {
            get { return data; }
        }

        /// <summary>
        /// Image matrix dimensions. Limits object has allways
        /// zeros at its low limits. Dimensions
        /// cannot be changed after image has been created.
        /// </summary>
        public readonly IntLimits dim = new IntLimits();
        /// <summary>
        /// short-cut for image time dimensions (frames)
        /// </summary>
        public int dimt
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.FRAMES);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Constructs image with zero dimensions.
        /// </summary>
        public Image()
        {
        }

        /// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension values</param>
        public Image(params int[] dimension)
        {
            if (dimension.Length > 6)
                throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            int product = 1;
            for (int i = 0; i < dimension.Length; i++)
            {
                if (dimension[i] < 0)
                    throw new TPCInvalidArgumentsException("Negative dimension value for creation of image.");
                product *= dimension[i];
            }
            GC.Collect();
            try
            {
                data = new float[product];
            }
            catch (OutOfMemoryException) {
                GC.Collect();
                throw new TPCException("Cannot create image because there is not enough memory.");
            }
            for (int i = 0; i < dimension.Length; i++)
                dim.AddLimit(0, dimension[i]);
        }
        /// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension object, dimension values will be converted to integer values</param>
        public Image(IntLimits dimension)
        {
            if (dimension.Length > 6) throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            for (int i = 0; i < dimension.Length; i++)
                if (dimension.GetLimit(i, Limits.Limit.LOW) != 0)
                    throw new TPCInvalidArgumentsException(i + "'s low limit was non-zero while creating an image:" + dimension.GetLimit(i, Limits.Limit.LOW));
            GC.Collect();
            data = new float[(int)dimension.GetProduct()];
            for (int i = 0; i < dimension.Length; i++)
                dim.AddLimit(dimension.GetLimit(i, Limits.Limit.LOW), dimension.GetLimit(i, Limits.Limit.HIGH));
        }
        /// <summary>
        /// short-cut for image width
        /// </summary>
        public int dimx
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.WIDTH);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// short-cut for image width
        /// </summary>
        public int width
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.WIDTH);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// short-cut for image height
        /// </summary>
        public int dimy
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.HEIGHT);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// short-cut for image height
        /// </summary>
        public int height
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.HEIGHT);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// short-cut for image number of planes
        /// </summary>
        public int dimz
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.PLANES);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// short-cut for image number of planes
        /// </summary>
        public int planes
        {
            get
            {
                int r = (int)dim.GetDimension(Limits.PLANES);
                if (r > 0) return r;
                else return 1;
            }
        }
        /// <summary>
        /// Length of plane in image.
        /// </summary>
        public int planeLength
        {
            get { return (int)dim.GetDimension(Limits.WIDTH) * (int)dim.GetDimension(Limits.HEIGHT); }
        }
        /// <summary>
        /// Length of data in image.
        /// </summary>
        public int dataLength {
            get { return data.Length; }
        }
        /// <summary>
        /// Gets all image float data as 1-dimensional array.
        /// </summary>
        /// <returns>image data as an array</returns>
        public float[] GetArray() { 
            return data; 
        }
        /// <summary>
        /// Returns an System.Collections.IEnumerator for the System.Array.
        /// </summary>
        /// <returns>enumerator over all image data</returns>
        public IEnumerator GetEnumerator()
        {
            return data.GetEnumerator();
        }
        /// <summary>
        /// Voxels of image as an array.
        /// </summary>
        /// <param name="location">index number (0-based)</param>
        /// <returns>value at index</returns>
        public virtual float this[int location]
        {
            get
            {
                if (location < 0 || location >= data.Length)
                    throw new TPCInvalidArgumentsException("Index " + location + " is out of image array bounds [0.." + (data.Length - 1) + "].");
                return data[location];
            }
            set
            {
                if (location < 0 || location >= data.Length)
                    throw new TPCInvalidArgumentsException("Index " + location + " is out of image array bounds [0.." + (data.Length - 1) + "].");
                data[location] = value;
            }
        }
        /// <summary>
        /// Gets value at given index in image.
        /// </summary>
        /// <param name="index">index where value is</param>
        /// <exception cref="TPCInvalidArgumentsException">index is outside of array bounds</exception>
        /// <returns>value at index</returns>
        public float GetValue(int index)
        {
            if (index < 0 || index >= data.Length)
                throw new TPCInvalidArgumentsException("Index " + index + " is out of image array bounds [0.." + (data.Length - 1) + "].");
            return data[index];
        }
        /// <summary>
        /// Sets value at given index in image.
        /// </summary>
        /// <param name="index">index where value is set</param>
        /// <param name="value">new value</param>
        /// <exception cref="TPCInvalidArgumentsException">index is outside of array bounds</exception>
        public void SetValue(int index, float value)
        {
            if (index < 0 || index >= data.Length)
                throw new TPCInvalidArgumentsException("Index " + index + " is out of image array bounds [0.." + (data.Length - 1) + "].");
            data[index] = value;
        }
        /// <summary>
        /// Voxels of image as an multidimensional array.
        /// </summary>
        /// <param name="x">x-coordinate (0-based)</param>
        /// <param name="y">y-coordinate (0-based)</param>
        /// <param name="z">z-coordinate (0-based)</param>
        /// <returns>value at location</returns>
        public virtual float this[int x, int y, int z]
        {
            get {
                return data[z*dim.dimx*dim.dimy+y*dim.dimx+x];
            }
            set {
                data[z*dim.dimx*dim.dimy+y*dim.dimx+x] = value;
            }
        }
        /// <summary>
        /// Voxels of image located by point.
        /// </summary>
        /// <param name="p">location in image</param>
        /// <returns>image voxel value at location</returns>
        public virtual float this[Point p]
        {
            get
            {
                return data[(int)p.z * dim.dimx * dim.dimy + (int)p.y * dim.dimx + (int)p.x];
            }
            set
            {
                data[(int)p.z * dim.dimx * dim.dimy + (int)p.y * dim.dimx + (int)p.x] = value;
            }
        }
        /// <summary>
        /// Voxels of image located by point.
        /// </summary>
        /// <param name="p">location in image</param>
        /// <returns>image voxel value at location</returns>
        public virtual float this[IntPoint p]
        {
            get
            {
                return data[p.z * dim.dimx * dim.dimy + p.y * dim.dimx + p.x];
            }
            set
            {
                data[p.z * dim.dimx * dim.dimy + p.y * dim.dimx + p.x] = value;
            }
        }
        /// <summary>
        /// Returns value at given point in image.
        /// </summary>
        /// <param name="point">point where value is retrieved (0-based coordinates)</param>
        /// <returns>data value</returns>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public virtual float GetValue(IntPoint point)
        {
            if (point.x < 0 || point.x >= dim.GetDimension(Limits.WIDTH) ||
               point.y < 0 || point.y >= dim.GetDimension(Limits.HEIGHT) ||
               point.z < 0 || point.z >= dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("Point (" + point.x + "," + point.y + "," + point.z + ") is out image limits.");
            return data[point.x + point.y * dim.GetDimension(Limits.WIDTH) + point.z * dim.GetDimension(Limits.WIDTH) * dim.GetDimension(Limits.HEIGHT)];
        }
        /// <summary>
        /// Returns value at given location in image.
        /// </summary>
        /// <param name="location">location where value is retrieved (0-based coordinates), representing dimensions [x, y, z, ..]</param>
        /// <returns>data value</returns>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public virtual float GetValue(params int[] location)
        {
            int i = 0;
            int j = 0;
            int arrayindex = 0;
            int prod = 0;
            for (i = 0; i < location.Length; i++)
            {
                if (i >= dim.Length)
                    throw new TPCInvalidArgumentsException("Tried to get value from dimension index " + i + " outside maximum " + dim.Length + ".");
                if (location[i] < 0 || location[i] >= dim.GetDimension(i))
                    throw new TPCInvalidArgumentsException("" + i + "'s coordinate value " + location[i] + " is out of image dimensions ["+0+".."+(dim.GetDimension(i)-1)+"].");
                prod = 1;
                for (j = i - 1; j >= 0; j--)
                    prod *= dim.GetDimension(j);
                arrayindex += location[i] * prod;
            }
            if (arrayindex < 0 || arrayindex >= data.Length)
                throw new TPCError("Internal error: index " + arrayindex + " is out of bounds [0.." + (data.Length - 1) + "].");
            return data[arrayindex];
        }
        /// <summary>
        /// Returns value at given location in image while skipping location. Used for dimension 
        /// removal.
        /// </summary>
        /// <param name="skip">skip dimension</param>
        /// <param name="location">location where value is retrieved (0-based coordinates), representing dimensions [x, y, z, ..]</param>
        /// <returns>data value</returns>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        protected int GetIndex(int skip, params int[] location)
        {
            int i = 0;
            int j = 0;
            int arrayindex = 0;
            int prod = 0;
            for (i = 0; i < location.Length; i++)
            {
                if (i >= dim.Length)
                    throw new TPCInvalidArgumentsException("Tried to get value from dimension index " + i + " outside maximum " + dim.Length + ".");
                if (location[i] < 0 || location[i] >= dim.GetDimension(i))
                    throw new TPCInvalidArgumentsException("" + i + "'s coordinate value " + location[i] + " is out of image dimensions.");
                prod = 1;
                for (j = i - 1; j >= 0; j--)
                    if(j != skip) prod *= dim.GetDimension(j);
                arrayindex += location[i] * prod;
            }
            return arrayindex;
        }
        /// <summary>
        /// Sets value at given point in image.
        /// </summary>
        /// <param name="point">point where value is set (0-based coordinates)</param>
        /// <param name="value">new value</param>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public virtual void SetValue(IntPoint point, float value)
        {
            if (point.x < 0 || point.x >= dim.GetDimension(Limits.WIDTH) ||
               point.y < 0 || point.y >= dim.GetDimension(Limits.HEIGHT) ||
               point.z < 0 || point.z >= dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("Point (" + point.x + "," + point.y + "," + point.z + ") is out of bounds (" + dim + ").");
            data[point.x + point.y * dim.GetDimension(Limits.WIDTH) + point.z * dim.GetDimension(Limits.WIDTH) * dim.GetDimension(Limits.HEIGHT)] = value;
        }
        /// <summary>
        /// Sets value at given location in image.
        /// </summary>
        /// <param name="value">new value</param>
        /// <param name="location">location where value is set (0-based coordinates)</param>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public virtual void SetValue(float value, params int[] location)
        {
            int i = 0;
            int j = 0;
            int arrayindex = 0;
            int prod = 0;
            for (i = 0; i < location.Length; i++)
            {
                if (location[i] < 0 || location[i] > dim.GetDimension(i))
                    throw new TPCInvalidArgumentsException("index " + i + " coordinate value " + location[i] + " is out of image dimensions [0.." + dim.GetLimit(i, Limits.Limit.HIGH) + "].");
                prod = 1;
                for (j = i - 1; j >= 0; j--)
                {
                    prod *= dim.GetDimension(j);
                }
                arrayindex += location[i] * prod;
            }
            if (arrayindex < 0 || arrayindex >= data.Length)
            {
                throw new TPCError("Internal error: index " + arrayindex + " is out of bounds [0.." + (data.Length - 1) + "].");
            }
            data[arrayindex] = value;
        }
        /// <summary>
        /// Creates new subimage from current image.
        /// </summary>
        /// <param name="limits">subregion</param>
        /// <returns>image that has dimensions as subregion and contains corresponding voxel values</returns>
        /// <exception cref="TPCInvalidArgumentsException">invalid image region is accesses</exception>
        public virtual Image GetSubImage(IntLimits limits)
        {
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length)
                throw new TPCInvalidArgumentsException("Limits " + limits.Length + " does not match source image number of dimensions " + dim.Length + ".");
            if ((limits.Length > Limits.WIDTH) && limits.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dim.GetDimension(Limits.WIDTH))
                throw new TPCInvalidArgumentsException("High x limit must be inside image.");
            if ((limits.Length > Limits.HEIGHT) && limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dim.GetDimension(Limits.HEIGHT))
                throw new TPCInvalidArgumentsException("High y limit must be inside image.");
            if ((limits.Length > Limits.PLANES) && limits.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("High z limit must be inside image.");
            if ((limits.Length > Limits.FRAMES) && limits.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) > dim.GetDimension(Limits.FRAMES))
                throw new TPCInvalidArgumentsException("High t (4th dim) limit must be inside image.");
            if ((limits.Length > Limits.GATES) && limits.GetLimit(Limits.GATES, Limits.Limit.HIGH) > dim.GetDimension(Limits.GATES))
                throw new TPCInvalidArgumentsException("High (5th dim) limit must be inside image.");
            if ((limits.Length > Limits.BEDS) && limits.GetLimit(Limits.BEDS, Limits.Limit.HIGH) > dim.GetDimension(Limits.BEDS))
                throw new TPCInvalidArgumentsException("High (6th dim) limit must be inside image.");
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException("Limits does not match target image number of dimensions.");
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int planes = 1;
            int height = 1;
            int width = 1;
            int[] p = new int[limits.Length];
            int[] q = new int[limits.Length];
            Image r = new Image(limits.GetDimensions());
            if (limits.Length > Limits.BEDS) beds = r.dim.GetDimension(IntLimits.BEDS);
            if (limits.Length > Limits.GATES) gates = r.dim.GetDimension(IntLimits.GATES);
            if (limits.Length > Limits.FRAMES) frames = r.dim.GetDimension(IntLimits.FRAMES);
            if (limits.Length > Limits.PLANES) planes = r.dim.GetDimension(IntLimits.PLANES);
            if (limits.Length > Limits.HEIGHT) height = r.dim.GetDimension(IntLimits.HEIGHT);
            width = r.dim.GetDimension(IntLimits.WIDTH);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (limits.Length > Limits.BEDS)
                {
                    p[5] = b;
                    q[5] = b + limits.GetLimit(Limits.BEDS, Limits.Limit.LOW);
                }
                for (g = 0; g < gates; g++)
                {
                    if (limits.Length > Limits.GATES)
                    {
                        q[4] = g;
                        q[4] = g + limits.GetLimit(Limits.GATES, Limits.Limit.LOW);
                    }
                    for (t = 0; t < frames; t++)
                    {
                        if (limits.Length > Limits.FRAMES)
                        {
                            p[3] = t;
                            q[3] = t + limits.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                        }
                        for (z = 0; z < planes; z++)
                        {
                            if (limits.Length > Limits.PLANES)
                            {
                                p[2] = z;
                                q[2] = z + limits.GetLimit(Limits.PLANES, Limits.Limit.LOW);
                            }
                            for (y = 0; y < height && y<limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
                            {
                                if (limits.Length > Limits.HEIGHT)
                                {
                                    p[1] = y;
                                    q[1] = y + limits.GetLimit(Limits.HEIGHT, Limits.Limit.LOW);
                                }
                                for (x = 0; x < width; x++)
                                {
                                    p[0] = x;
                                    q[0] = x + limits.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
                                    r.SetValue(GetValue(q), p);
                                }
                            }
                        }
                    }
                }
            }
            //set fields
            return r;
        }
        /// <summary>
        /// Sets value to subimage of current image.
        /// </summary>
        /// <param name="limits">subregion</param>
        /// <param name="image">values that are put into image</param>
        /// <exception cref="TPCInvalidArgumentsException">invalid image region is accesses</exception>
        public virtual void SetSubImage(IntLimits limits, ref Image image)
        {
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException(limits.Length + " limits does not match target image number of dimensions " + dim.Length + ".");
            if (limits.Length < image.dim.Length)
                throw new TPCInvalidArgumentsException("Limits " + limits.Length + " cannot be smaller than source image number of dimensions " + image.dim.Length + ".");
            if ((limits.Length > Limits.WIDTH) && limits.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dim.GetDimension(Limits.WIDTH))
                throw new TPCInvalidArgumentsException("High x limit must be inside image.");
            if ((limits.Length > Limits.HEIGHT) && limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dim.GetDimension(Limits.HEIGHT))
                throw new TPCInvalidArgumentsException("High y limit must be inside image.");
            if ((limits.Length > Limits.PLANES) && limits.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("High z limit must be inside image.");
            if ((limits.Length > Limits.FRAMES) && limits.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) > dim.GetDimension(Limits.FRAMES))
            {
                throw new TPCInvalidArgumentsException("High t (4th dim) limit must be inside image.");
            }
            if ((limits.Length > Limits.GATES) && limits.GetLimit(Limits.GATES, Limits.Limit.HIGH) > dim.GetDimension(Limits.GATES))
                throw new TPCInvalidArgumentsException("High (5th dim) limit must be inside image.");
            if ((limits.Length > Limits.BEDS) && limits.GetLimit(Limits.BEDS, Limits.Limit.HIGH) > dim.GetDimension(Limits.BEDS))
                throw new TPCInvalidArgumentsException("High (6th dim) limit must be inside image.");
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException("Limits does not match target image number of dimensions.");
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int planes = 1;
            int height = 1;
            int[] p = new int[image.dim.Length];
            int[] q = new int[limits.Length];
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            if (image.dim.Length > Limits.PLANES) planes = image.dim.GetDimension(IntLimits.PLANES);
            if (image.dim.Length > Limits.HEIGHT) height = image.dim.GetDimension(IntLimits.HEIGHT);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (image.dim.Length > Limits.BEDS)
                    p[5] = b;
                if (limits.Length > Limits.BEDS)
                    q[5] = b + limits.GetLimit(Limits.BEDS, Limits.Limit.LOW);
                for (g = 0; g < gates; g++)
                {
                    if (image.dim.Length > Limits.GATES)
                        q[4] = g;
                    if (limits.Length > Limits.GATES)
                        q[4] = g + limits.GetLimit(Limits.GATES, Limits.Limit.LOW);                        
                    for (t = 0; t < frames; t++)
                    {
                        if (image.dim.Length > Limits.FRAMES)
                            p[3] = t;
                        if (limits.Length > Limits.FRAMES)
                            q[3] = t + limits.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                        for (z = 0; z < planes; z++)
                        {
                            if (image.dim.Length > Limits.PLANES)
                                p[2] = z;
                            if (limits.Length > Limits.PLANES)
                                q[2] = z + limits.GetLimit(Limits.PLANES, Limits.Limit.LOW);
                            for (y = 0; y < height; y++)
                            {
                                if (image.dim.Length > Limits.HEIGHT)
                                    p[1] = y;
                                if (limits.Length > Limits.HEIGHT)
                                    q[1] = y + limits.GetLimit(Limits.HEIGHT, Limits.Limit.LOW);
                                for (x = 0; x < image.dim.GetDimension(IntLimits.WIDTH); x++)
                                {
                                    p[0] = x;
                                    q[0] = x + limits.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
                                    SetValue(image.GetValue(p), q);
                                }
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Multiplies all data with value
        /// </summary>
        /// <param name="value">multiplication value</param>
        public virtual void Multiply(float value)
        {
            for (int i = 0; i < data.Length; i++)
            {
                data[i] *= value;
            }
        }
        /// <summary>
        /// Returns minimum data value
        /// </summary>
        /// <returns>minimum value</returns>
        public virtual float GetMin()
        {
            if (data.Length == 0) return 0;
            float m = float.MaxValue;
            for (int i = 0; i < data.Length; i++)
                if (data[i] < m) m = data[i];
            return m;
        }
        /// <summary>
        /// Returns minimum and maximum data value
        /// </summary>
        /// <param name="subregion">subregion for calculation</param>
        /// <returns>array containing [minimum, maximum] value</returns>
        public virtual float[] GetMinMax(IntLimits subregion)
        {
            int frames;
            if (subregion.Length > IntLimits.GATES)
                throw new TPCInvalidArgumentsException("Cannot use subregion "+subregion+" because it has too much dimensions");
            if (subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dimz ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dimy ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dimx)
                throw new TPCInvalidArgumentsException("Subregion ["+subregion+"] is not inside image bounds ["+dim+"]");
            frames = (dim.Length > IntLimits.FRAMES ? dim.frames : 1);
            if (subregion.Length > IntLimits.FRAMES) {
                if (subregion.GetLimit(Limits.FRAMES, Limits.Limit.LOW) < 0 ||
                    subregion.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) > frames)
                    throw new TPCInvalidArgumentsException("Subregion [" + subregion + "] is not inside image bounds [" + dim + "]");
            }
            if (data.Length == 0) return new float[2] { 0.0f, 0.0f };
            float max = float.MinValue;
            float min = float.MaxValue;
            int i = 0;
            int planes = (dim.Length > IntLimits.PLANES ? dim.planes : 1);
            int planes_end = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) : 1);
            int planes_start = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) : 0);
            int frames_end = (dim.Length > IntLimits.FRAMES ? subregion.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) : 1);
            int frames_start = (dim.Length > IntLimits.FRAMES ? subregion.GetLimit(Limits.FRAMES, Limits.Limit.LOW) : 0);
            for (int t = frames_start; t < frames_end; t++)
            {
                for (int z = planes_start; z < planes_end; z++)
                {
                    for (int y = subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW); y < subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
                    {
                        for (int x = subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW); x < subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH); x++)
                        {
                            i = t * dimx * dimy * dimz + z * dimx * dimy + y * dimx + x;
                            if (data[i] < min) min = data[i];
                            if (data[i] > max) max = data[i];
                        }
                    }
                }
            }
            return new float[2] { min, max };
        }
        /// <summary>
        /// Returns minimum and maximum data value
        /// </summary>
        /// <returns>array containing [minimum, maximum] value</returns>
        public virtual float[] GetMinMax()
        {
            if (data.Length == 0) return new float[2] { 0.0f, 0.0f };

            float max = float.MinValue;
            float min = float.MaxValue;
            for (int i = 0; i < data.Length; i++)
            {
                if (data[i] < min) min = data[i];
                if (data[i] > max) max = data[i];
            }
            return new float[2] { min, max };
        }
        /// <summary>
        /// Returns maximum data value
        /// </summary>
        /// <returns>maximum value</returns>
        public virtual float GetMax()
        {
            if (data.Length == 0) return 0;
            float m = float.MinValue;
            for (int i = 0; i < data.Length; i++)
                if (data[i] > m) m = data[i];
            return m;
        }
        /// <summary>
        /// Returns the mean data value
        /// </summary>
        /// <returns>mean value</returns>
        public virtual float GetMean()
        {
            if (data.Length == 0)
                throw new TPCException("Cannot calculate mean of image because there is not voxels.");
            double m = 0.0f;
            for (int i = 0; i < data.Length; i++)
                m += (double)data[i];
            return (float)(m / (double)data.Length);
        }
        /// <summary>
        /// Returns the mean data value
        /// </summary>
        /// <param name="subregion">subregion for calculation</param>
        /// <returns>mean value</returns>
        public virtual float GetMean(IntLimits subregion)
        {
            if (subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW) < 0 ||
                subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dimz ||
                subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dimy ||
                subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dimx)
                throw new TPCInvalidArgumentsException("Subregion [" + subregion + "] is not inside image bounds [" + dim + "]");
            if (data.Length == 0) return 0.0f;
            double mean = 0.0f;
            int i = 0;
            int frames_end = ((this is DynamicImage) ? (this as DynamicImage).frames : 1);
            int frames_start = 0;
            if (subregion.Length > IntLimits.FRAMES)
            {
                frames_start = subregion.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                frames_end = subregion.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
            }
            int planes = (dim.Length > IntLimits.PLANES ? dim.planes : 1);
            int planes_end = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) : 1);
            int planes_start = (dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) : 0);
            for (int t = frames_start; t < frames_end; t++)
            {
                for (int z = planes_start; z < planes_end; z++)
                {
                    for (int y = subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW); y < subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
                    {
                        for (int x = subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW); x < subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH); x++)
                        {
                            i = t * planes * dimy * dimx + z * dimx * dimy + y * dimx + x;
                            mean += (double)data[i];
                        }
                    }
                }
            }
            return (float)mean/(float)subregion.GetProduct();
        }
        /// <summary>
        /// Converts array index into coordinate location using image
        /// as reference image. Higher dimensions are ignored.
        /// </summary>
        /// <param name="arrayindex">index value in 1-dimensional array</param>
        /// <returns>3D-coordinate value that corresponds the index value</returns>
        public IntPoint To3DCoordinate(int arrayindex)
        {
            int x, y, z;
            z = arrayindex / (width * height);
            arrayindex -= z * width * height;
            y = arrayindex / width;
            arrayindex -= y * width;
            x = arrayindex;
            return new IntPoint(x, y, z);
        }
        /// <summary>
        /// Shortcut for getting limits for image's subregion.
        /// </summary>
        /// <see cref="IntLimits"/>
        /// <param name="index">index (0-based)</param>
        /// <param name="dimension">dimension number</param>
        /// <returns>limits object that representd frame dimensions</returns>
        /// <exception cref="TPCInvalidArgumentsException">in case of invalid parameters</exception>
        public IntLimits GetDimensionLimits(int index, int dimension)
        {
            return dim.GetDimensionLimits(index, dimension);
        }
        /// <summary>
        /// Shortcut for getting limits for image's plane subregion.
        /// </summary>
        /// <param name="plane">plane index (0-based)</param>
        public IntLimits GetPlaneLimits(int plane)
        {
            return GetDimensionLimits(plane, IntLimits.PLANES);
        }
        /// <summary>
        /// Returns one plane from image.
        /// </summary>
        /// <param name="plane">plane index (0-based)</param>
        /// <returns>new image containing plane data</returns>
        public virtual Image GetPlane(int plane) {
            return GetSubImage(GetPlaneLimits(plane));
        }
        /// <summary>
        /// Prints image information to stdout.
        /// </summary>
        public virtual void PrintImage()
        {
            Console.WriteLine("Image:");
            Console.WriteLine(" dim:" + dim);
            Console.WriteLine(" min:" + GetMin() + " max:" + GetMax());
        }
        /// <summary>
        /// Returns image dimensions and min/max-values as string
        /// </summary>
        /// <returns>string representation object</returns>
        public override string ToString()
        {
            return "Image[" + dim + ",min=" + GetMin() + ",max=" + GetMax() + "]";
        }
        /// <summary>
        /// Multiply operator of image.
        /// </summary>
        /// <param name="value">multiplicator</param>
        /// <param name="image">target image</param>
        /// <returns>multiplied image</returns>
        public static Image operator *(float value, Image image)
        {
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] * value;
            return r;
        }
        /// <summary>
        /// Multiply operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">multiplicator</param>
        /// <returns>multiplied image</returns>
        public static Image operator *(Image image, float value)
        {
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] * value;
            return r;
        }
        /// <summary>
        /// Multiply operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">multiplicator image</param>
        /// <returns>multiplied image</returns>
        public static Image operator *(Image image, Image value)
        {
            if (image.dim != value.dim)
                throw new TPCException("Cannot apply arithmetic - operation because image sizes are not the same");
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] * value.data[i];
            return r;
        }
        /// <summary>
        /// Divide operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">divisor</param>
        /// <returns>resulting image</returns>
        public static Image operator /(Image image, float value)
        {
            if (value == 0) throw new TPCException("Tried to divide image by zero.");
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] / value;
            return r;
        }
        /// <summary>
        /// Addition operator for image.
        /// </summary>
        /// <param name="value">addition</param>
        /// <param name="image">target image.</param>
        /// <returns>resulting image</returns>
        public static Image operator +(float value, Image image)
        {
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] + value;
            return r;
        }
        /// <summary>
        /// Addition operator for image.
        /// </summary>
        /// <param name="image">target image.</param>
        /// <param name="value">addition</param>
        /// <returns>resulting image</returns>
        public static Image operator +(Image image, float value)
        {
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] + value;
            return r;
        }
        /// <summary>
        /// Decrement operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">decrement</param>
        /// <returns>resulting image</returns>
        public static Image operator -(Image image, float value)
        {
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] - value;
            return r;
        }
        /// <summary>
        /// Decrement operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">decrement</param>
        /// <returns>resulting image</returns>
        public static Image operator -(Image image, Image value)
        {
            if (image.dataLength != value.dataLength)
                throw new TPCException("Cannot apply arithmetic - operation because image data lengths [" + image.dataLength + "] and [" + value.dataLength + "] are not the same");
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] - value.data[i];
            return r;
        }
        /// <summary>
        /// Decrement operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">increment</param>
        /// <returns>resulting image</returns>
        public static Image operator +(Image image, Image value)
        {
            if (image.dataLength != value.dataLength)
                throw new TPCException("Cannot apply arithmetic + operation because image data lengths [" + image.dataLength + "] and [" + value.dataLength + "] are not the same");
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
                r.data[i] = image.data[i] + value.data[i];
            return r;
        }
        /// <summary>
        /// Decrement operator of image.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">divider</param>
        /// <returns>resulting image</returns>
        public static Image operator /(Image image, Image value)
        {
            if (image.dataLength != value.dataLength)
                throw new TPCException("Cannot apply arithmetic / operation because image data lengths [" + image.dataLength + "] and [" + value.dataLength + "] are not the same");
            Image r = new Image(image.dim);
            for (int i = 0; i < r.data.Length; i++)
            {
                if (value.data[i] == 0) throw new TPCException("Tried to divide with zero intensity value.");
                r.data[i] = image.data[i] / value.data[i];
            }
            return r;
        }
        /// <summary>
        /// Implicit conversion from 3-dimensional array. 
        /// </summary>
        /// <param name="data">converted data</param>
        /// <returns>image holding the data</returns>
        public static explicit operator Image(float[,,] data) {
            Image r = new Image(data.GetLength(0), data.GetLength(1), data.GetLength(2));
            for (int z = 0; z < data.GetLength(0); z++)
                for (int y = 0; y < data.GetLength(1); y++)
                    for (int x = 0; x < data.GetLength(2); x++)
                        r[x,y,z] = data[x,y,z];
            return r;
        }
        /// <summary>
        /// Evaluates if images equals. Currently only image dimensions and data is evaluated.
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>false if files differ</returns>
        public override bool Equals(object obj)
        {
            if (obj is Image)
            {
                Image img = (obj as Image);
                if (dim != img.dim) return false;
                for (int i = 0; i < data.Length; i++)
                    if (data[i] != img.data[i]) return false;
                return true;
            }
            else
            {
                return false;
            }
        }
        /// <summary>
        /// Override that just calls base class method.
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
