/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.IO;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib.Image {
    /// <summary>
    /// Image file. Parent class for all image file format classes.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public abstract class ImageFile: TPCFile {
        /// <summary>
        /// Contains parameters for GetPixelData method. This 
        /// structure is for easier usability of GetPixelData method from this class.
        /// </summary>
        public struct PixelDataParameters {
            /// <summary>
            /// Target array
            /// </summary>
            public string filename;
            /// <summary>
            /// Input data dimensions
            /// </summary>
            public IntLimits dim;
            /// <summary>
            /// Data type to be read
            /// </summary>
            public DataType datatype;
            /// <summary>
            /// file position where reading is started
            /// </summary>
            public long position;
            /// <summary>
            /// scale factor
            /// </summary>
            public float scale;
        }
        /// <summary>
        /// Event handler for I/O progress notifications.
        /// </summary>
        /// <param name="sender">object that sent the event</param>
        /// <param name="e">event arguments</param>
        public delegate void IOProcessEventHandler(object sender, IOProgressEventArgs e);
        /// <summary>
        /// Image data type (byte to value representation)
        /// </summary>
        public enum DataType
        {
            /// <summary>
            /// 1-bit data
            /// </summary>
            BIT1 = 1,    /*  1-bit           */
            /// <summary>
            /// 8-bit signed [-128..127]
            /// </summary>
            BIT8_S = 2,    /*  8-bit signed    */
            /// <summary>
            /// 8-bit unsigned [0..255]
            /// </summary>
            BIT8_U = 3,    /*  8-bit unsigned  */
            /// <summary>
            /// 16-bit signed [-32768..32767] (==4)
            /// </summary>
            BIT16_S = 4,    /* 16-bit signed    */
            /// <summary>
            /// 16-bit unsigned [0..65535]
            /// </summary>
            BIT16_U = 5,    /* 16-bit unsigned  */
            /// <summary>
            /// 32-bit signed [-2147483648..2147483647]
            /// </summary>
            BIT32_S = 6,    /* 32-bit signed    */
            /// <summary>
            /// 32-bit unsigned [0..4294967295]
            /// </summary>
            BIT32_U = 7,    /* 32-bit unsigned  */
            /// <summary>
            /// 64-bit signed
            /// </summary>
            BIT64_S = 8,    /* 64-bit signed    */
            /// <summary>
            /// 64-bit unsigned
            /// </summary>
            BIT64_U = 9,    /* 64-bit unsigned  */
            /// <summary>
            /// 32-bit float
            /// </summary>
            FLT32 = 10,   /* 32-bit float     */
            /// <summary>
            /// 64-bit float
            /// </summary>
            FLT64 = 11,    /* 64-bit double    */
            /// <summary>
            /// 64-bit ASCII
            /// </summary>
            ASCII = 12,    /* ascii            */
            /// <summary>
            /// 16-bit VAX integer
            /// </summary>
            VAXI16 = 13,
            /// <summary>
            /// 32-bit VAX integer
            /// </summary>
            VAXI32 = 14,
            /// <summary>
            /// 32-bit VAX float
            /// </summary>
            VAXFL32 = 15,    /* 32-bit vaxfloat  */
            /// <summary>
            /// 24-bit RGB (8-bits each [0..255])
            /// </summary>
            COLRGB = 16,    /* RGB triplets     */
            /// <summary>
            /// SUN 2-byte integer
            /// </summary>
            SUNI2 = 17,    
            /// <summary>
            /// SUN 4-byte integer 
            /// </summary>
            SUNI4 = 18    
        }
        /// <summary>
        /// String representation of DataType
        /// </summary>
        /// <param name="type">converted value</param>
        /// <returns>enumeration value as string</returns>
        public static string ToString(DataType type)
        {
            switch (type)
            {
                case DataType.BIT1: return "1-bit data";
                case DataType.BIT8_S: return "8-bit signed [-128..127]";
                case DataType.BIT8_U: return "8-bit unsigned [0..255]";
                case DataType.BIT16_S: return "16-bit signed";
                case DataType.BIT16_U: return "16-bit unsigned [0..65535]";
                case DataType.BIT32_S: return "32-bit signed [-2147483648..2147483647]";
                case DataType.BIT32_U: return "32-bit unsigned [0..4294967295]";
                case DataType.BIT64_S: return "64-bit signed";
                case DataType.BIT64_U: return "64-bit unsigned";
                case DataType.FLT32: return "32-bit float";
                case DataType.FLT64: return "64-bit double";
                case DataType.ASCII: return "64-bit ASCII";
                case DataType.VAXI16: return "16-bit VAX integer";
                case DataType.VAXI32: return "32-bit VAX integer";
                case DataType.VAXFL32: return "32-bit VAX float";
                case DataType.COLRGB: return "24-bit RGB (8-bits each [0..255])";
                case DataType.SUNI2: return "SUN 2-byte integer";
                case DataType.SUNI4: return "SUN 4-byte integer";
            }
            throw new Exception("internal error: unknown enumeration instance");
        }
        /// <summary>
        /// Resolves number of bytes per pixel from datatype.
        /// </summary>
        /// <param name="type">converted value</param>
        /// <returns>enumeration value as string</returns>
        public static int BytesPerPixel(DataType type)
        {
            switch (type)
            {
                case DataType.BIT1: return 0;
                case DataType.BIT8_S: 
                case DataType.BIT8_U: return 1;
                case DataType.BIT16_S: 
                case DataType.BIT16_U: return 2;
                case DataType.BIT32_S: 
                case DataType.BIT32_U: return 4;
                case DataType.BIT64_S:
                case DataType.BIT64_U: return 8;
                case DataType.FLT32: return 4;
                case DataType.FLT64: return 8;
                case DataType.ASCII: return 8;
                case DataType.VAXI16: return 2;
                case DataType.VAXI32: 
                case DataType.VAXFL32: return 4;
                case DataType.COLRGB: return 3;
                case DataType.SUNI2: return 2;
                case DataType.SUNI4: return 4;
            }
            throw new Exception("internal error: unknown enumeration instance");
        }
        /// <summary>
        /// Filetype of file.
        /// </summary>
        public FileType filetype;
        /// <summary>
        /// Image data.
        /// </summary>
        public Image image = null;
        /// <summary>
        /// Image header data.
        /// </summary>
        public ImageHeader header = null;
        /// <summary>
        /// Resolves file format from available file formats in library.
        /// </summary>
        /// <param name="filename">full path to tested file(s)</param>
        /// <returns>resolved image file type</returns>
        public static FileType ResolveFileFormat(string filename) {
            if (Analyze75File.CheckFormat(filename)) return FileType.Analyze75;
            else if (Ecat7File.CheckFormat(filename)) return FileType.Ecat7;
            else if (DicomFile.CheckFormat(filename)) return FileType.DICOM;
            else if (MicroPETFile.CheckFormat(filename)) return FileType.MicroPET;
            else if (InterfileFile.CheckFormat(filename)) return FileType.Interfile;
            throw new TPCException("Failed to recognize file ["+filename+"] format.");
        }
        /// <summary>
        /// Resolves number of significant bits in filetype.
        /// </summary>
        /// <param name="type">datatype</param>
        /// <returns>number of bits</returns>
        public static int Type2Bits(ImageFile.DataType type)
        {
            int bits = 0;
            switch (type)
            {
                case DataType.BIT1: bits = 1; break;
                case DataType.BIT8_S:
                case DataType.BIT8_U: bits = 8; break;
                case DataType.BIT16_S:
                case DataType.BIT16_U: bits = 16; break;
                case DataType.COLRGB: bits = 24; break;
                case DataType.BIT32_S:
                case DataType.BIT32_U:
                case DataType.FLT32:
                case DataType.VAXFL32: bits = 32; break;
                case DataType.ASCII: /* read as double */
                case DataType.BIT64_S:
                case DataType.BIT64_U:
                case DataType.FLT64: bits = 64; break;
            }
            return bits;
        }
        /// <summary>
        /// Returns number of bytes used by a datatype.
        /// </summary>
        /// <param name="type">datatype</param>
        /// <returns>number of bytes</returns>
        public static int Type2Bytes(ImageFile.DataType type)
        {
            int bytes = 0;
            switch (type)
            {
                case DataType.BIT1:
                case DataType.BIT8_S:
                case DataType.BIT8_U: 
                    bytes = 1; break;
                case DataType.BIT16_S:
                case DataType.BIT16_U:
                case DataType.SUNI2: 
                    bytes = 2; break;
                case DataType.COLRGB: 
                    bytes = 3; break;
                case DataType.BIT32_S:
                case DataType.BIT32_U:
                case DataType.SUNI4:
                case DataType.FLT32:
                case DataType.VAXFL32: 
                    bytes = 4; break;
                case DataType.ASCII:
                case DataType.BIT64_S:
                case DataType.BIT64_U:
                case DataType.FLT64: 
                    bytes = 8; break;
            }
            return (bytes);
        }
        /// <summary>
        /// Swaps 4-byte words.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">starting index</param>
        /// <param name="length">number of bytes to swap</param>
        protected static byte[] swapWordBytes(byte[] buf, int startindex, int length)
        {
            byte b;
            //swap inside words
            if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length-1) + "] goes out of buffer bounds [0.."+(buf.Length-1)+"]");
            for (int i = startindex; i < startindex+length; i += 4)
            {
                b = buf[i];
                buf[i] = buf[i + 3];
                buf[i + 3] = b;
                b = buf[i + 1];
                buf[i + 1] = buf[i + 2];
                buf[i + 2] = b;
            }
            return buf;
        }
        /// <summary>
        /// Swaps bytes and 4-byte words.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">starting index</param>
        /// <param name="length">number of bytes to swap</param>
        protected static byte[] swapWordAndArrayBytes(byte[] buf, int startindex, int length)
        {
            byte b;
            if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length - 1) + "] goes out of buffer bounds [0.." + (buf.Length - 1) + "]");
            //swap bytes in array
            buf = swapArrayBytes(buf, startindex, length);
            //swap inside words
            for (int i = startindex; i < startindex + length; i += 4)
            {
                b = buf[i];
                buf[i] = buf[i + 3];
                buf[i + 3] = b;
                b = buf[i + 1];
                buf[i + 1] = buf[i + 2];
                buf[i + 2] = b;
            }
            return buf;
        }
        /// <summary>
        /// Swaps bytes of array
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">first byte to swap</param>
        /// <param name="length">number of bytes to swap</param>
        /// <returns>swapped byte array</returns>
        protected static byte[] swapArrayBytes(byte[] buf, int startindex, int length) {
            byte b;
            if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length - 1) + "] goes out of buffer bounds [0.." + (buf.Length - 1) + "]");
            for (int i = 1; i < length; i += 2)
            {
                b = buf[startindex + i];
                buf[startindex + i] = buf[startindex + i - 1];
                buf[startindex + i - 1] = b;
            }
            return buf;
        }
        /// <summary>
        /// Read 32-bit VAX floats
        /// </summary>
        /// <param name="buf">array of 32-bit float data</param>
        /// <param name="isvax">true if VAX format</param>
        /// <param name="islittle">true if little endian</param>
        /// <returns>converted byte array</returns>
        protected byte[] VAX32toFloat(byte[] buf, bool isvax, bool islittle)
        {
            int ul;
            for (int i = 0; i < buf.Length; i += 4)
            {
                ul = BitConverter.ToInt32(buf, i);
                if (ul == 0) continue;
                if (isvax)
                {
                    /* if input is in VAX format */
                    /* Swap words on i386 and bytes on SUN */
                    if (islittle)
                        swapWordBytes(buf, i, 4);
                    else
                        swapArrayBytes(buf, i, 4);
                    ul -= (2 << 23); /* subtract 2 from exp */
                }
                else
                {
                    /* input is in i386 format */
                    if (!islittle) swapWordAndArrayBytes(buf, i, 4); /* Switch words and bytes on SUN */
                }
            }
            return buf;
        }
        /// <summary>
        /// Swabs bytes according to data type if necessary.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="datatype">data type</param>
        protected void swabBytes(byte[] buf, DataType datatype)
        {
            switch (datatype)
            {
                case DataType.BIT16_S:
                case DataType.VAXI16:
                    // byte conversion necessary on big endian platform
                    if (BitConverter.IsLittleEndian)
                    {
                        swapArrayBytes(buf, 0, buf.Length);
                    }
                    break;
                case DataType.VAXI32:
                    // Swap both words and bytes on SUN 
                    if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 0, buf.Length);
                    break;
                case DataType.VAXFL32: //ecat7rInt
                    buf = VAX32toFloat(buf, true, BitConverter.IsLittleEndian);
                    break;
                case DataType.FLT32:
                // IEEE float ; byte conversion necessary on little endian platforms
                case DataType.SUNI4:                    
                    // SUN int ; byte conversion necessary on little endian platforms
                    if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, buf.Length);
                    break;
                case DataType.SUNI2:
                    // SUN short ; byte conversion necessary on little endian platforms
                    if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, buf.Length);
                    break;
                default:
                    break;
            }
        }
        /// <summary>
        /// Converts byte values to char array.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">starting index</param>
        /// <param name="length">nunber of bytes to convert</param>
        /// <returns>character array</returns>
        protected static char[] toCharArray(byte[] buf, int startindex, int length)
        {
            char[] r = new char[length];
            for (int i = 0; i < r.Length; i++)
                r[i] = (char)buf[startindex + i];
            return r;
        }
        /// <summary>
        /// Converts char array to byte values.
        /// </summary>
        /// <param name="buf">char array</param>
        /// <returns>byte array</returns>
        protected static byte[] ToBytesArray(char[] buf)
        {
            byte[] r = new byte[buf.Length];
            for (int i = 0; i < buf.Length; i++)
                r[i] = (byte)buf[i];
            return r;
        }
        /// <summary>
        /// Converts float array to byte values.
        /// </summary>
        /// <param name="buf">float array</param>
        /// <returns>byte array</returns>
        protected static byte[] ToBytesArray(float[] buf)
        {
            byte[] r = new byte[buf.Length * 4];
            for (int i = 0; i < buf.Length; i++)
                BitConverter.GetBytes(buf[i]).CopyTo(r, i * 4);
            return r;
        }
        /// <summary>
        /// Converts short array to byte values.
        /// </summary>
        /// <param name="buf">short array</param>
        /// <returns>byte array</returns>
        protected static byte[] ToBytesArray(short[] buf)
        {
            byte[] r = new byte[buf.Length * 2];
            for (int i = 0; i < buf.Length; i++)
                BitConverter.GetBytes(buf[i]).CopyTo(r, i * 2);
            return r;
        }
        /// <summary>
        /// Converts byte values to float array.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">starting index</param>
        /// <param name="length">number of floats to convert</param>
        /// <returns>float array</returns>
        protected static float[] ToFloatArray(byte[] buf, int startindex, int length)
        {
            float[] r = new float[length];
            for (int i = 0; i < r.Length; i++)
                r[i] = BitConverter.ToSingle(buf, startindex + i * 4);
            return r;
        }
        /// <summary>
        /// Converts byte values to short array.
        /// </summary>
        /// <param name="buf">byte array</param>
        /// <param name="startindex">starting index</param>
        /// <param name="length">nunber of shorts to convert</param>
        /// <returns>short array</returns>
        protected static short[] ToShortArray(byte[] buf, int startindex, int length)
        {
            short[] r = new short[length];
            for (int i = 0; i < r.Length; i++)
                r[i] = BitConverter.ToInt16(buf, startindex + i * 2);
            return r;
        }
        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">even length with parameters {low_x, hi_x, low_y, hi_y ..}</param>
        public void ReadSubImage(int[] region) {
            if(region.Length % 2 != 0)
                throw new TPCInvalidArgumentsException("Subregion array must have even length.");
            IntLimits limits = new IntLimits();
            for(int i = 0; i < region.Length; i+=2) {
                limits.AddLimit(region[i], region[i+1]);
            }
            ReadSubImage(limits);
        }
        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">subregion that is read from file</param>
        public abstract void ReadSubImage(IntLimits region);
        /// <summary>
        /// Implementation of TPCFile interface. Writes this file.
        /// </summary>
        public override void WriteFile()
        {
            WriteFile(ref image);
        }
        /// <summary>
        /// Writes a file.
        /// </summary>
        /// <param name="image">image data that is written</param>
        public abstract void WriteFile(ref Image image);
        /// <summary>
        /// Reads image file.
        /// </summary>
        /// <param name="filename">filename of image data</param>
        /// <returns>image file that has all data in image</returns>
        /// <exception cref="TPCException">if reading of file failed</exception>
        public static ImageFile ReadFile(string filename) {

            FileType filetype;
            try
            {
                filetype = ImageFile.ResolveFileFormat(filename);
            }
            catch(Exception)
            {
                throw new TPCException("Failed to recognize file "+filename);
            }
            switch (filetype) {
                case FileType.Analyze75:
                    Analyze75File file_Analyze75 = new Analyze75File(filename);
                    file_Analyze75.ReadFile();
                    return file_Analyze75;
                case FileType.DICOM:
                    DicomFile file_Dicom;
                    //read one or multiple DICOM files 
                    if (filename.EndsWith("*"))
                    {
                        file_Dicom = DicomFile.ReadMultiple(filename);
                    }
                    else
                    {
                        file_Dicom = new DicomFile(filename);
                        file_Dicom.ReadFile();
                    }
                    return file_Dicom;
                case FileType.Ecat7:
                    Ecat7File file_Ecat7 = new Ecat7File(filename);
                    file_Ecat7.ReadFile();
                    return file_Ecat7;
                case FileType.MicroPET:
                    MicroPETFile file_MicroPET = new MicroPETFile(filename);
                    file_MicroPET.ReadFile();
                    return file_MicroPET;
                case FileType.Interfile:
                    InterfileFile file_Interfile;
                    if (filename.EndsWith("*"))
                    {
                        file_Interfile = InterfileFile.ReadMultiple(filename);
                    }
                    else
                    {
                        file_Interfile = new InterfileFile(filename);
                        file_Interfile.ReadFile();
                    }
                    return file_Interfile;
                default:
                    throw new TPCException("Cannot read data with this method since type " + filetype + " is not currently supported.");
            }
        }
        /// <summary>
        /// Writes image file. Note that actual image file type is read from 
        /// image data's filetype field.
        /// </summary>
        /// <seealso cref="ImageFile.filetype"/>
        /// <param name="imagefile">image data that is written</param>
        /// <returns>image file that has all data in image</returns>
        /// <exception cref="Exception">if writing of file failed</exception>
        public static void WriteFile(ImageFile imagefile)
        {
            //write image data depending on data type
            switch (imagefile.filetype) {
                case FileType.Analyze75:
                    //change some datatypes to supported ones
                    switch (imagefile.header.datatype)
                    {
                        case DataType.BIT16_U:
                            imagefile.header.datatype = DataType.BIT16_S;
                            break;
                        default:
                            break;
                    }
                    //use input object straight if possible
                    if(imagefile is Analyze75File) {
                        (imagefile as Analyze75File).WriteFile();
                    } else {
                        Analyze75File file = new Analyze75File(imagefile.filename);
                        file.image = imagefile.image;
                        file.header = imagefile.header;
                        file.WriteFile();
                    }
                    break;
                case FileType.DICOM:
                    //use input object straight if possible
                    if(imagefile is DicomFile) {
                        (imagefile as DicomFile).WriteFile();
                    } else {
                        DicomFile file = new DicomFile(imagefile.filename, imagefile.image, imagefile.header);
                        file.WriteFile();
                    }
                    break;
                case FileType.Ecat7:
                    //use input object straight if possible
                    if(imagefile is Ecat7File) {
                        (imagefile as Ecat7File).WriteFile();
                    } else {
                        Ecat7File file = new Ecat7File(imagefile.filename, imagefile.image, imagefile.header);
                        file.WriteFile();
                    }
                    break;
                case FileType.Interfile:
                    //use input object straight if possible
                    if (imagefile is InterfileFile)
                    {
                        (imagefile as InterfileFile).WriteFile();
                    }
                    else
                    {
                        InterfileFile file = new InterfileFile(imagefile.filename, imagefile.image, imagefile.header);
                        file.WriteFile();
                    }
                    break;
                case FileType.MicroPET:
                    //use input object straight if possible
                    if (imagefile is MicroPETFile)
                    {
                        (imagefile as MicroPETFile).WriteFile();
                    }
                    else
                    {
                        MicroPETFile file = new MicroPETFile(imagefile.filename, imagefile.image, imagefile.header);
                        file.WriteFile();
                    }
                    break;
                default:
                    throw new TPCException("Cannot write data with this method since type "+imagefile.filetype+ " is not currently supported.");
            }
        }
        /// <summary>
        /// Reads single frame from file. The header and image dimensions 
        /// are changed.
        /// </summary>
        /// <param name="frame_No">frame number that is read [1..no frames]</param>
        /// <returns>image data in frame</returns>
        public void ReadFrame(int frame_No) {
            ReadSubImage(image.dim.GetDimensionLimits(frame_No, IntLimits.FRAMES));
        }
        /// <summary>
        /// Reads single frame from file. The header and image dimensions 
        /// are changed.
        /// </summary>
        /// <param name="plane_No">plane number that is read [1..no planes]</param>
        /// <returns>image data in frame</returns>
        public void ReadPlane(int plane_No)
        {
            ReadSubImage(image.dim.GetDimensionLimits(plane_No, IntLimits.PLANES));
        }
        /// <summary>
        /// Writes single frame into file. If frame number is larger than 
        /// current number of frames in file, then file size is grown.
        /// </summary>
        /// <param name="image">image data that is written</param>
        /// <param name="frame_No">frame number that is written [1..no frames+1]</param>
        public abstract void WriteFrame(ref Image image, int frame_No);
        /// <summary>
        /// Reads only header data from file
        /// </summary>
        public abstract ImageHeader ReadHeader();
        /// <summary>
        /// Reads only header data from file
        /// </summary>
        /// <param name="filename">file that is read</param>
        public static ImageHeader ReadHeader(string filename) {
            FileType filetype;
            try
            {
                filetype = ImageFile.ResolveFileFormat(filename);
            }
            catch (Exception)
            {
                throw new TPCException("Failed to recognize file " + filename);
            }
            switch (filetype)
            {
                case FileType.Analyze75:
                    Analyze75File file_Analyze75 = new Analyze75File(filename);
                    return file_Analyze75.ReadHeader();
                case FileType.DICOM:
                    DicomFile file_Dicom = new DicomFile(filename);
                    return file_Dicom.ReadHeader();
                case FileType.Ecat7:
                    Ecat7File file_Ecat7 = new Ecat7File(filename);
                    return file_Ecat7.ReadHeader();
                case FileType.MicroPET:
                    MicroPETFile file_MicroPET = new MicroPETFile(filename);
                    return file_MicroPET.ReadHeader();
                case FileType.Interfile:
                    InterfileFile file_Interfile = new InterfileFile(filename);
                    return file_Interfile.ReadHeader();
                default:
                    throw new TPCException("Cannot read header with this method since type " + filetype + " is not currently supported.");
            }        
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// Scaling factor 1.0 is used by this method. The data is read from the start of the file. 
        /// The data is written into the beginning of the image array.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="filename">input filename</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        public void GetPixelData(ref Image data, string filename, IntLimits dim, DataType datatype)
        {
            GetPixelData(ref data, 0, filename, dim, datatype, 0, 1.0f);
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// Scaling factor 1.0 is used by this method. The data is read from the start of the file.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="filename">input filename</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        public void GetPixelData(ref Image data, int offset, string filename, IntLimits dim, DataType datatype)
        {
            GetPixelData(ref data, offset, filename, dim, datatype, 0, 1.0f);
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// Scaling factor 1.0 is used by this method.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="filename">input filename</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started</param>
        public void GetPixelData(ref Image data, int offset, string filename, IntLimits dim, DataType datatype, long position)
        {
            GetPixelData(ref data, offset, filename, dim, datatype, position, 1.0f);
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="parameters">structure containing parameters for data access</param>
        public void GetPixelData(ref Image data, int offset, PixelDataParameters parameters) {
            GetPixelData(ref data, offset, parameters.filename, parameters.dim, parameters.datatype, parameters.position, parameters.scale);
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="filename">input filename, full path</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started</param>
        /// <param name="scale">scale factor</param>
        public void GetPixelData(ref Image data, int offset, string filename, IntLimits dim, DataType datatype, long position, float scale) {
            FileStream reader;
            try {
                reader = new FileStream(filename, FileMode.Open, FileAccess.Read);
            }
            catch(Exception) {
                throw new TPCException("Could not open file ["+filename+"] for reading.");
            }
            GetPixelData(ref data, offset, (Stream)reader, dim, datatype, position, scale);
            try {
                reader.Close();
            }
            catch(Exception) {
                throw new TPCException("Could not close file ["+filename+"] after reading.");
            }
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started</param>
        /// <param name="scale">scale factor</param>
        public abstract void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale);
        /// <summary>
        /// Gets parameters needed for direct access to pixel data.
        /// </summary>
        /// <returns>parameters to pixel data</returns>
        public abstract PixelDataParameters GetParametersForPixelData();
    }
}
