/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    #region public_enumerators
    /// <summary>
    /// Isotope name
    /// </summary>
    public enum Isotope_enumerator {
        /// <summary>
        /// Coal-11
        /// </summary>
        C11,
        /// <summary>
        /// Oxygen-14
        /// </summary>
        O14,
        /// <summary>
        /// Oxygen-15
        /// </summary>
        O15,
        /// <summary>
        /// Fluor-18
        /// </summary>
        F18,
        /// <summary>
        /// Germanium-68
        /// </summary>
        Ge68,
        /// <summary>
        /// Gallium-68
        /// </summary>
        Ga68,
        /// <summary>
        /// Bromine-75
        /// </summary>
        Br75,
        /// <summary>
        /// Bromine-76
        /// </summary>
        Br76,
        /// <summary>
        /// Copper-62
        /// </summary>
        Cu62,
        /// <summary>
        /// Copper-64
        /// </summary>
        Cu64,
        /// <summary>
        /// Iron-52
        /// </summary>
        Fe52,
        /// <summary>
        /// Nitrogen-13
        /// </summary>
        N13,
        /// <summary>
        /// Natrium-22
        /// </summary>
        Na22,
        /// <summary>
        /// Bromine-82
        /// </summary>
        Br82,
        /// <summary>
        /// Zinc-62
        /// </summary>
        Zn62,
        /// <summary>
        /// Iodine-124
        /// </summary>
        I124,
        /// <summary>
        /// No isotope
        /// </summary>
        No_isotope   
    }
    /// <summary>
    /// PET image acquisition type
    /// </summary>
    public enum AcquisitionType
    {
        /// <summary>
        /// unknown = static
        /// </summary>
        ACQUISITION_UNKNOWN = 0,
        /// <summary>
        /// 
        /// </summary>
        ACQUISITION_STATIC = 1,
        /// <summary>
        /// dynamic
        /// </summary>
        ACQUISITION_DYNAMIC = 2,
        /// <summary>
        /// tomographic
        /// </summary>
        ACQUISITION_TOMO = 3,
        /// <summary>
        /// gated
        /// </summary>
        ACQUISITION_GATED = 4,
        /// <summary>
        /// gated spect
        /// </summary>
        ACQUISITION_GSPECT = 5
    }
    /// <summary>
    /// Patient orientation
    /// </summary>
    public enum Patient_orientation
    {
        /// <summary>
        /// Feet first prone (0000b)
        /// </summary>
        Feet_first_Prone = 0x0,
        /// <summary>
        /// Feet first supine (0010b)
        /// </summary>
        Feet_first_Supine = 0x2,
        /// <summary>
        /// Feet first decubitus right (0100b)
        /// </summary>
        Feet_first_Decubitus_Right = 0x4,
        /// <summary>
        /// Feet first decubitus left (0110b)
        /// </summary>
        Feet_first_Decubitus_Left = 0x6,
        /// <summary>
        /// Head first prone (0001b)
        /// </summary>
        Head_first_Prone = 0x1,
        /// <summary>
        /// Head first supine (0011b)
        /// </summary>
        Head_first_Supine = 0x3,
        /// <summary>
        /// Head first decubitus right (0101b)
        /// </summary>
        Head_first_Decubitus_Right = 0x5,
        /// <summary>
        /// Head first decubitus left (0111b)
        /// </summary>
        Head_first_Decubitus_Left = 0x7,
        /// <summary>
        /// Unknown orientation
        /// </summary>
        Unknown = 0x8
    }
    /// <summary>
    /// Weight unit
    /// </summary>
    public enum Weight_unit
    {
        /// <summary>
        /// Unknown weight units
        /// </summary>
        Unknown = 0,
        /// <summary>
        /// Grams
        /// </summary>
        g = 1,
        /// <summary>
        /// Ounces
        /// </summary>
        oz = 2,
        /// <summary>
        /// Kilograms
        /// </summary>
        kg = 3,
        /// <summary>
        /// Pounds
        /// </summary>
        lb = 4
    }
    /// <summary>
    /// Length unit
    /// </summary>
    public enum Length_unit
    {
        /// <summary>
        /// Unknown length units
        /// </summary>
        Unknown = 0,
        /// <summary>
        /// millimeters
        /// </summary>
        mm = 1,
        /// <summary>
        /// centimeters 
        /// </summary>
        cm = 2,
        /// <summary>
        /// meters
        /// </summary>
        m = 3,
        /// <summary>
        /// inches
        /// </summary>
        inch = 4
    }
    /// <summary>
    /// Dicom modality type, for MDC library
    /// </summary>
    public enum ImageModality
    {
        /// <summary>
        /// Unknown modality
        /// </summary>
        Unknown = 0,
        /// <summary>
        /// Angioscopy
        /// </summary>
        M_AS = ('A' << 8) | 'S',
        /// <summary>
        /// Audio
        /// </summary>
        M_AU = ('A' << 8) | 'U',
        /// <summary>
        /// Biomagnetic Imaging
        /// </summary>
        M_BI = ('B' << 8) | 'I',
        /// <summary>
        /// Color Flow Doppler
        /// </summary>
        M_CD = ('C' << 8) | 'D',
        /// <summary>
        /// Cinefluorography
        /// </summary>
        M_CF = ('C' << 8) | 'F',
        /// <summary>
        /// Culposcopy
        /// </summary>
        M_CP = ('C' << 8) | 'P',
        /// <summary>
        /// Computed Radiography
        /// </summary>
        M_CR = ('C' << 8) | 'R',
        /// <summary>
        /// Cystoscopy
        /// </summary>
        M_CS = ('C' << 8) | 'S',
        /// <summary>
        /// Computed Tomography
        /// </summary>
        M_CT = ('C' << 8) | 'T',
        /// <summary>
        /// Duplex Doppler
        /// </summary>
        M_DD = ('D' << 8) | 'D',
        /// <summary>
        /// Digital Fluoroscopy
        /// </summary>
        M_DF = ('D' << 8) | 'F',
        /// <summary>
        /// Diaphanography
        /// </summary>
        M_DG = ('D' << 8) | 'G',
        /// <summary>
        /// Digital Microscopy
        /// </summary>
        M_DM = ('D' << 8) | 'M',
        /// <summary>
        /// Digital Substraction Angiography
        /// </summary>
        M_DS = ('D' << 8) | 'S',
        /// <summary>
        /// Digital Radiography
        /// </summary>
        M_DX = ('D' << 8) | 'X',
        /// <summary>
        /// Echocardiography
        /// </summary>
        M_EC = ('E' << 8) | 'C',
        /// <summary>
        /// Endoscopy
        /// </summary>
        M_ES = ('E' << 8) | 'S',
        /// <summary>
        /// Fluorescein Angiography
        /// </summary>
        M_FA = ('F' << 8) | 'A',
        /// <summary>
        /// Fundoscopy
        /// </summary>
        M_FS = ('F' << 8) | 'S',
        /// <summary>
        /// General Microscopy
        /// </summary>
        M_GM = ('G' << 8) | 'M',
        /// <summary>
        /// Hemodynamic Waveform
        /// </summary>
        M_HD = ('H' << 8) | 'D',
        /// <summary>
        /// Intra-Oral Radiography
        /// </summary>
        M_IO = ('I' << 8) | 'O',
        /// <summary>
        /// Hardcopy
        /// </summary>
        M_HC = ('H' << 8) | 'C',
        /// <summary>
        /// Laparoscopy
        /// </summary>
        M_LP = ('L' << 8) | 'P',
        /// <summary>
        /// Magnetic Resonance Angiography
        /// </summary>
        M_MA = ('M' << 8) | 'A',
        /// <summary>
        /// Mammography
        /// </summary>
        M_MG = ('M' << 8) | 'G',
        /// <summary>
        /// Magnetic Resonance
        /// </summary>
        M_MR = ('M' << 8) | 'R',
        /// <summary>
        /// Magnetic Resonance Spectroscopy
        /// </summary>
        M_MS = ('M' << 8) | 'S',
        /// <summary>
        /// Nuclear Medicine
        /// </summary>
        M_NM = ('N' << 8) | 'M',
        /// <summary>
        /// Other
        /// </summary>
        M_OT = ('O' << 8) | 'T',
        /// <summary>
        /// Positron Emission Tomography
        /// </summary>
        M_PT = ('P' << 8) | 'T',
        /// <summary>
        /// Panoramic X-Ray
        /// </summary>
        M_PX = ('P' << 8) | 'X',
        /// <summary>
        /// Radio Fluoroscopy
        /// </summary>
        M_RF = ('R' << 8) | 'F',
        /// <summary>
        /// Radiographic Imaging
        /// </summary>
        M_RG = ('R' << 8) | 'G',
        /// <summary>
        /// Radiotherapy
        /// </summary>
        M_RT = ('R' << 8) | 'T',
        /// <summary>
        /// Slide Microscopy
        /// </summary>
        M_SM = ('S' << 8) | 'M',
        /// <summary>
        /// SR Document
        /// </summary>
        M_SR = ('S' << 8) | 'R',
        /// <summary>
        /// Single-Photon Emission Computed Tomography
        /// </summary>
        M_ST = ('S' << 8) | 'T',
        /// <summary>
        /// Thermography
        /// </summary>
        M_TG = ('T' << 8) | 'G',
        /// <summary>
        /// Ultrasound
        /// </summary>
        M_US = ('U' << 8) | 'S',
        /// <summary>
        /// Videofluorography
        /// </summary>
        M_VF = ('V' << 8) | 'F',
        /// <summary>
        /// X-Ray Angiography
        /// </summary>
        M_XA = ('X' << 8) | 'A',
        /// <summary>
        /// External-Camera Photography
        /// </summary>
        M_XC = ('X' << 8) | 'C'
    }
    #endregion
    /// <summary>
    /// Common header information to all images.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class ImageHeader : HeaderFieldList
    {
        /// <summary>
        /// image width in voxels.
        /// </summary>
        public int width
        {
            get { return dim.GetDimension(IntLimits.WIDTH); }
        }
        /// <summary>
        /// image height in voxels.
        /// </summary>
        public int height
        {
            get { return dim.GetDimension(IntLimits.HEIGHT); }
        }
        /// <summary>
        /// number of planes.
        /// </summary>
        public int planes
        {
            get { return dim.GetDimension(IntLimits.PLANES); }
        }
        /// <summary>
        /// x-dimension.
        /// </summary>
        public int dimx {
            get { return dim.GetDimension(IntLimits.WIDTH); }
        }
        /// <summary>
        /// y-dimension.
        /// </summary>
        public int dimy
        {
            get { return dim.GetDimension(IntLimits.HEIGHT); }
        }
        /// <summary>
        /// z-dimension.
        /// </summary>
        public int dimz
        {
            get { return dim.GetDimension(IntLimits.PLANES); }
        }
        /// <summary>
        /// Read-only short-cut for voxel size (mm).
        /// </summary>
        public float sizex
        {
            get { return siz.sizex; }
        }
        /// <summary>
        /// Read-only short-cut for voxel size (mm).
        /// </summary>
        public float sizey
        {
            get { return siz.sizey; }
        }
        /// <summary>
        /// Read-only short-cut for voxel size (mm).
        /// </summary>
        public float sizez
        {
            get { return siz.sizez; }
        }
        /// <summary>
        /// Patient name.
        /// </summary>
        public string patient_name;
        /// <summary>
        /// Dimensions. Low values are allways zero, high value is number of indexes
        /// in axis.
        /// </summary>
        public IntLimits dim;
        /// <summary>
        /// Voxel size
        /// </summary>
        public Voxel siz;
        /// <summary>
        /// patient identification string
        /// </summary>
        public string patientID;
        /// <summary>
        /// description field of image
        /// </summary>
        public string description;
        /// <summary>
        /// Data binary representation type.
        /// </summary>
        public ImageFile.DataType datatype;
        /// <summary>
        /// Data representation unit.
        /// </summary>
        public Data_unit dataunit;
        /// <summary>
        /// Image modality type
        /// </summary>
        public ImageModality modality;
        /// <summary>
        /// Default constructor
        /// </summary>
        public ImageHeader()
        {
            this.patient_name = "";
            this.patientID = "";
            this.dim = new IntLimits(0,0,0);
            this.siz = new Voxel();
            this.description = "";
            this.datatype = ImageFile.DataType.BIT16_U;
            this.dataunit = Data_unit.Undefined;
            this.modality = ImageModality.Unknown;
        }
        /// <summary>
        /// Constructor from superclass
        /// </summary>
        public ImageHeader(HeaderFieldList headerlist)
        {
            this.patient_name = "";
            this.patientID = "";
            this.dim = new IntLimits(0, 0, 0);
            this.siz = new Voxel();
            this.description = "";
            this.datatype = ImageFile.DataType.BIT16_U;
            this.dataunit = Data_unit.Undefined;
            this.modality = ImageModality.Unknown;
            this.AddRange(headerlist);
        }
        /// <summary>
        /// Constructor with parameters.
        /// </summary>
        /// <param name="patient_name">patient name</param>
        /// <param name="patient_ID">patient ID</param>
        /// <param name="dim">image dimensions</param>
        /// <param name="siz">voxel size</param>
        /// <param name="description">description of image</param>
        /// <param name="datatype">datatype of file</param>
        public ImageHeader(string patient_name, string patient_ID, IntLimits dim, Voxel siz, string description, ImageFile.DataType datatype) {
            this.patient_name = patient_name;
            this.patientID = patient_ID;
            this.dim = dim;
            this.siz = siz;
            this.description = description;
            this.datatype = datatype;
            this.dataunit = Data_unit.Unknown;
            this.modality = ImageModality.Unknown;
        }
        /// <summary>
        /// Returns image header as string
        /// </summary>
        /// <returns>string representation</returns>
        public override string  ToString() {
 	        return "ImageHeader[patient_name="+patient_name+", patientID="+patientID+", "+dim+", "+siz+", datatype="+datatype+"]";
        }
        /// <summary>
        /// Evaluates if two objects are equal
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if header data is equal to this object's data</returns>
        public override bool Equals(object obj)
        {
            ImageHeader h;
            if (obj is ImageHeader)
            {
                h = (obj as ImageHeader);
                if(h.datatype != datatype) return false;
                if(h.dataunit != dataunit) return false;
                if(h.description != description) return false;
                if(h.dim != dim) return false;
                return true;
            }
            else {
                return false;
            }
        }
        /// <summary>
        /// Override that just calls base class method.
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
