/******************************************************************************
 *
 * Copyright (c) 2008,2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using MathNet.SignalProcessing.Filter.FIR;
using MathNet.Numerics.Distributions;

namespace TPClib.Image
{
    /// <summary>
    /// Generals utility functions for Image objects.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public sealed class ImageUtils
    {
        /// <summary>
        /// Creates a Gaussian filter in 1D.
        /// </summary>
        /// <param name="length"></param>
        /// <returns></returns>
        protected static double[] CreateGaussianFilter(int length) {
            if (length % 2 == 0)
                samples = new double[length - 1];
            else
                samples = new double[length];
            double[] samples = new double[length]; 
            samples[samples.Length / 2] = 1.0;
            for (x = 1; x <= samples.Length / 2; x++)
            {
                samples[samples.Length / 2 + x] = ndist.ProbabilityDensity(X * x);
                samples[samples.Length / 2 - x] = ndist.ProbabilityDensity(X * x);
            }
        } 
        /// <summary>
        /// Gaussian filtering of image
        /// </summary>
        /// <param name="image">filtered image</param>
        /// <param name="X">convolution kernel X in units</param>
        /// <param name="Y">convolution kernel Y in units</param>
        /// <param name="Z">convolution kernel Z in units</param>
        public static void GaussianFilter(ref Image image, double X, double Y, double Z) {
            OnlineFirFilter filter;
            NormalDistribution ndist = new NormalDistribution();
            m.SetRow(row_i, FIRfilter.ProcessSamples(X.GetRow(row_i)));
            double[] samples;
            int loc;
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            //filter each direction individually
            //gaussian FIR filter is generated for each direction 
            for (b = 0; b < beds; b++)
            {
                for (g = 0; g < gates; g++)
                {
                    for (t = 0; t < frames; t++)
                    {
                        ndist.SetDistributionParameters(0.0, X);
                        if(image.width % 2 == 0)
                            samples = new double[image.width-1];
                        else
                            samples = new double[image.width];
                        samples[samples.Length/2] = 1.0;
                        for (x = 1; x <= samples.Length/2; x++) {
                            samples[samples.Length / 2 + x] = ndist.ProbabilityDensity(X * x);
                            samples[samples.Length / 2 - x] = ndist.ProbabilityDensity(X * x);
                        }
                        filter = new OnlineFirFilter();
                        for (z = 0; z < image.planes; z++)
                        {
                            for (y = 0; y < image.height; y++)
                            {
                                loc = z * image.height * image.width + y * image.width;
                                for (x = 0; x < image.width; x++) samples[x] = image[loc + x];
                                samples = filter.ProcessSamples(samples);
                                for (x = 0; x < image.width; x++) image[loc + x] = samples[x];
                            }
                        }
                        filter = new OnlineFirFilter(FirCoefficients.LowPass(fir_samplingrate, fir_cutoff, halforder));
                        for (z = 0; z < image.planes; z++)
                        {
                            for (x = 0; x < image.width; x++)
                            {
                                loc = z * image.height * image.width + x;
                                for (y = 0; y < image.width; y++) samples[y] = image[loc + y*image.width];
                                samples = filter.ProcessSamples(samples);
                                for (y = 0; y < image.width; y++) image[loc + y * image.width] = samples[y];
                            }
                        }
                        filter = new OnlineFirFilter(FirCoefficients.LowPass(fir_samplingrate, fir_cutoff, halforder));
                        for (x = 0; x < image.width; x++)
                        {
                            for (y = 0; y < image.height; y++)
                            {
                                loc = y * image.width + x;
                                for (z = 0; z < image.width; z++) samples[z] = image[loc + z * image.height * image.width];
                                samples = filter.ProcessSamples(samples);
                                for (z = 0; z < image.width; z++) image[loc + z * image.height * image.width] = samples[z];
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Flips image around Z-axis.
        /// <param name="image">flipped image</param>
        /// </summary>
        public static void flipZ(ref Image image) {
            if (image.dim.Length <= IntLimits.PLANES) throw new TPCException("Not enough dimensions for Z-flip:" + image.dim.Length);
            float[] tmp_data = new float[image.planes];
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int[] q = new int[image.dim.Length];
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (image.dim.Length > Limits.BEDS) q[5] = b;
                for (g = 0; g < gates; g++)
                {
                    if (image.dim.Length > Limits.GATES) q[4] = g;
                    for (t = 0; t < frames; t++)
                    {
                        if (image.dim.Length > Limits.FRAMES) q[3] = t;
                        for (y = 0; y < image.height; y++)
                        {
                            q[1] = y;
                            for (x = 0; x < image.width; x++)
                            {
                                q[0] = x;
                                for (z = 0; z < image.planes; z++)
                                {
                                    q[2] = z;
                                    tmp_data[image.planes - z - 1] = image.GetValue(q);
                                }
                                for (z = 0; z < image.planes; z++)
                                {
                                    q[2] = z;
                                    image.SetValue(tmp_data[z], q);
                                }
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Flips image around Y-axis.
        /// <param name="image">flipped image</param>
        /// </summary>
        public static void flipY(ref Image image)
        {
            if (image.dim.Length <= IntLimits.HEIGHT) throw new TPCException("Not enough dimensions for Y-flip:" + image.dim.Length);
            float[] tmp_data = new float[image.height];
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int[] q = new int[image.dim.Length];
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (image.dim.Length > Limits.BEDS) q[5] = b;
                for (g = 0; g < gates; g++)
                {
                    if (image.dim.Length > Limits.GATES) q[4] = g;
                    for (t = 0; t < frames; t++)
                    {
                        if (image.dim.Length > Limits.FRAMES) q[3] = t;
                        for (z = 0; z < image.planes; z++)
                        {
                            q[2] = z;
                            for (x = 0; x < image.width; x++)
                            {
                                q[0] = x; 
                                for (y = 0; y < image.height; y++)
                                {
                                    q[1] = y;
                                    tmp_data[image.width - y - 1] = image.GetValue(q);
                                }
                                for (y = 0; y < image.height; y++)
                                {
                                    q[1] = y;
                                    image.SetValue(tmp_data[y], q);
                                }
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Flips image around X-axis.
        /// <param name="image">flipped image</param>
        /// </summary>
        public static void flipX(ref Image image)
        {
            if (image.dim.Length <= IntLimits.WIDTH) throw new TPCException("Not enough dimensions for X-flip:" + image.dim.Length);
            float[] tmp_data = new float[image.width];
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int[] q = new int[image.dim.Length];
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (image.dim.Length > Limits.BEDS) q[5] = b;
                for (g = 0; g < gates; g++)
                {
                    if (image.dim.Length > Limits.GATES) q[4] = g;
                    for (t = 0; t < frames; t++)
                    {
                        if (image.dim.Length > Limits.FRAMES) q[3] = t;
                        for (z = 0; z < image.planes; z++)
                        {
                            q[2] = z;
                            for (y = 0; y < image.height; y++)
                            {
                                q[1] = y;
                                for (x = 0; x < image.width; x++)
                                {
                                    q[0] = x;
                                    tmp_data[image.width - x - 1] = image.GetValue(q);
                                }
                                for (x = 0; x < image.width; x++)
                                {
                                    q[0] = x;
                                    image.SetValue(tmp_data[x], q);
                                }
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Implements erosion operation.
        /// </summary>
        /// <param name="image">source image</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="parameters">ignored</param>
        /// <returns>original value or 0</returns>
        public static float SpatialErosionOperation(ref Image image, IntPoint targetlocation, int kernel, float[] parameters)
        {
            //return dilation value if value is found n kernel neighbourhood
            for (int z = targetlocation.z - kernel; z <= targetlocation.z + kernel; z++)
            {
                for (int y = targetlocation.y - kernel; y <= targetlocation.y + kernel; y++)
                {
                    for (int x = targetlocation.x - kernel; x <= targetlocation.x + kernel; x++)
                    {
                        if (image[x, y, z] == 0) return 0.0f;
                    }
                }
            }
            return image[targetlocation];
        }
        /// <summary>
        /// Implements dilation operation.
        /// </summary>
        /// <param name="image">source image</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="dilation">dilation value at index 0</param>
        /// <returns>dilation value or 0</returns>
        public static float SpatialDilationOperation(ref Image image, IntPoint targetlocation, int kernel, float[] dilation)
        {
            //return dilation value if value is found n kernel neighbourhood
            for (int z = targetlocation.z - kernel; z <= targetlocation.z + kernel; z++)
            {
                for (int y = targetlocation.y - kernel; y <= targetlocation.y + kernel; y++)
                {
                    for (int x = targetlocation.x - kernel; x <= targetlocation.x + kernel; x++)
                    {
                        if (image[x, y, z] != 0) return dilation[0];
                    }
                }
            }
            return 0.0f;
        }
        /// <summary>
        /// Implements image value inversion operation. Note that number of parameters 
        /// is not checked for performance reasons.
        /// </summary>
        /// <param name="value">image value that is inverted</param>
        /// <param name="minmax">array containing image min and max values {min, max}</param>
        /// <returns>inverted value</returns>
        public static float InverseOperation(float value, params float[] minmax)
        {
            //1. move values so that overall minimum is at zero
            //2. multiply with -1 so that the data is inverted
            //3. move values up so that old maximum location has zero value
            //4. move values so that old minimum is preserved
            return (((value - minmax[0]) * (-1f)) + minmax[1]) + minmax[0];
        }
        /// <summary>
        /// Inverses images intensity values. If minimum value 
        /// is negative, then that negative value will be placed at locations 
        /// where was maximum value. 
        /// </summary>
        /// <param name="image">target image</param>
        public static void Inverse(ref Image image)
        {
            //calculate minimum and maximum
            float[] minmax = image.GetMinMax();

            //perform inversion operation for all image data
            performOperation(ref image, InverseOperation, minmax);
        }
        /// <summary>
        /// Implements image value normalization operation. Note that number of parameters 
        /// is not checked for performance reasons. The parameters p are interpreted as follows:
        /// min = min_new-min_old
        /// C = (max_old-min_old)/(max_new-min_new)
        /// </summary>
        /// <param name="value">image value that is inverted</param>
        /// <param name="p">array containing move towards minimum and data range multiplication factor {min, C}</param>
        /// <returns>inverted value</returns>
        public static float NormalizationOperation(float value, params float[] p)
        {
            //1. multiply with C to new range 
            //2. move values so that old minimum is preserved
            return (value + p[0]) * p[1];
        }
        /// <summary>
        /// Inverses images intensity values. If minimum value 
        /// is negative, then that negative value will be placed at locations 
        /// where was maximum value. 
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="limits">minimum and maximum, 0 and 1 values are used values are not given</param>
        public static void Normalization(ref Image image, params float[] limits)
        {
            //calculate minimum and maximum
            float[] minmax = image.GetMinMax();
            float[] new_minmax = new float[]{0,1};
            //use limit values if they are given
            if(limits.Length > 0) new_minmax[0] = limits[0];
            if(limits.Length > 1) new_minmax[1] = limits[1];
            if (minmax[1] - minmax[0] == 0) return;
            if (new_minmax[1] - new_minmax[0] == 0) throw new TPCInvalidArgumentsException("New data range cannot be zero.");
            float C = (new_minmax[1] - new_minmax[0]) / (minmax[1] - minmax[0]);
            float min_d = new_minmax[0] - new_minmax[0];
            //perform inversion operation for all image data
            performOperation(ref image, NormalizationOperation, new float[] { min_d, C });
        }
        /// <summary>
        /// Performs user-defined operation to image.
        /// </summary>
        /// <param name="image"></param>
        /// <param name="operation">operation that is applied to all image data</param>
        /// <param name="parameters">additional parameters for operation, if any</param>
        public static void performOperation(ref Image image, ImageOperation operation, params float[] parameters)
        {
            for (int i = 0; i < image.dataLength; i++)
                image[i] = operation(image[i], parameters);
        }
        /// <summary>
        /// Performs user-defined spatial operation to image.
        /// </summary>
        /// <param name="image"></param>
        /// <param name="operation">operation that is applied to all image data</param>
        /// <param name="kernel">spatial 3D-kernel size [-1..1 etc.]</param>
        /// <param name="parameters">additional parameters for operation, if any</param>
        public static void perform3DSpatialOperation(ref Image image, 
                                                   SpatialImageOperation operation, 
                                                   int kernel, 
                                                   params float[] parameters)
        {
            int x = 0;
            int y = 0;
            int z = 0;
            Image temp = new Image(image.dim);
            for (z = kernel; z < image.planes - kernel; z++)
            {
                for (y = kernel; y < image.height - kernel; y++)
                {
                    for (x = kernel; x < image.width - kernel; x++)
                    {
                        temp[x, y, z] = operation(ref image, new IntPoint(x,y,z), kernel, parameters);
                    }
                }
            }
            image = temp;
        }
        /// <summary>
        /// Delegate for doing arbitrary operation to each image value
        /// </summary>
        /// <param name="value">image value</param>
        /// <param name="parameters">operation specific parameters</param>
        /// <returns>value after operation</returns>
        public delegate float ImageOperation(float value, params float[] parameters);
        /// <summary>
        /// Delegate for doing arbitrary operation to each image value
        /// </summary>
        /// <param name="image">image value</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="parameters">operation specific parameters</param>
        /// <returns>value after operation</returns>
        public delegate float SpatialImageOperation(ref Image image,
                                                    IntPoint targetlocation,
                                                    int kernel, 
                                                    params float[] parameters);

        /// <summary>
        /// 3D Bresenham's algorithm for line voxelization.
        /// 
        /// Ported from Matlab code (which was ported from another code):
        /// 
        /// http://www.mathworks.com/matlabcentral/fx_files/21057/1/bresenham_line3d.m
        /// 
        /// Original articles:
        /// 
        /// B.Pendleton.  line3d - 3D Bresenham's (a 3D line drawing algorithm)
        /// ftp://ftp.isc.org/pub/usenet/comp.sources.unix/volume26/line3d, 1992
        ///
        /// Fischer, J., A. del Rio (2004).  A Fast Method for Applying Rigid
        /// Transformations to Volume Data, WSCG2004 Conference.
        /// http://wscg.zcu.cz/wscg2004/Papers_2004_Short/M19.pdf
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="a">starting point </param>
        /// <param name="b">ending point</param>
        /// <returns>list of masked coordinate points</returns>
        public static List<Point> maskBresenhamLine(ref Image mask, Point a, Point b)
        {
            int precision = 0;
            Point P1 = new Point((int)a.x, (int)a.y, (int)a.z);
            Point P2 = new Point((int)b.x, (int)b.y, (int)b.z);

            Point P2P1 = new Point();
            Point.Sub(ref P2P1, P2, P1);
            double d = Math.Max(Math.Max(Math.Abs(P2P1.x) + 1.0, Math.Abs(P2P1.y) + 1.0), Math.Abs(P2P1.z) + 1.0);

            double[] X = new double[(int)d];
            double[] Y = new double[(int)d];
            double[] Z = new double[(int)d];

            double x1 = P1[0];
            double y1 = P1[1];
            double z1 = P1[2];

            double x2 = P2[0];
            double y2 = P2[1];
            double z2 = P2[2];

            double dx = x2 - x1;
            double dy = y2 - y1;
            double dz = z2 - z1;

            double ax = Math.Abs(dx) * 2;
            double ay = Math.Abs(dy) * 2;
            double az = Math.Abs(dz) * 2;

            double sx = Math.Sign(dx);
            double sy = Math.Sign(dy);
            double sz = Math.Sign(dz);

            double x = x1;
            double y = y1;
            double z = z1;
            int idx = 1;

            // x dominant
            if (ax >= Math.Max(ay, az))
            {
                double yd = ay - ax / 2;
                double zd = az - ax / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    //end
                    if (x == x2) break;
                    // move along y
                    if (yd >= 0)
                    {
                        y = y + sy;
                        yd = yd - ax;
                    }
                    // move along z
                    if (zd >= 0)
                    {
                        z = z + sz;
                        zd = zd - ax;
                    }
                    // move along x
                    x = x + sx;
                    yd = yd + ay;
                    zd = zd + az;
                }
            }
            // y dominant
            else if (ay >= Math.Max(ax, az))
            {
                double xd = ax - ay / 2;
                double zd = az - ay / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    // end
                    if (y == y2) break;
                    // move along x
                    if (xd >= 0)
                    {
                        x = x + sx;
                        xd = xd - ay;
                    }
                    // move along z
                    if (zd >= 0)
                    {
                        z = z + sz;
                        zd = zd - ay;
                    }
                    // move along y
                    y = y + sy;
                    xd = xd + ax;
                    zd = zd + az;
                }
            }
            // z dominant
            else if (az >= Math.Max(ax, ay))
            {
                double xd = ax - az / 2;
                double yd = ay - az / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    // end
                    if (z == z2) break;
                    // move along x
                    if (xd >= 0)
                    {
                        x = x + sx;
                        xd = xd - az;
                    }
                    // move along y
                    if (yd >= 0)
                    {
                        y = y + sy;
                        yd = yd - az;
                    }
                    // move along z
                    z = z + sz;
                    xd = xd + ax;
                    yd = yd + ay;
                }
            }

            //this dividation is used for tighter distance between 
            //mask points, currently not in use
            if (precision != 0)
            {
                for (int i = 0; i < X.Length; i++)
                {
                    X[i] /= Math.Pow(10, precision);
                    Y[i] /= Math.Pow(10, precision);
                    Z[i] /= Math.Pow(10, precision);
                }
            }

            //mask into image and gather masked points
            List<Point> points = new List<Point>();
            for (int i = 0; i < X.Length; i++)
            {
                mask[(int)X[i], (int)Y[i], (int)Z[i]] = 1;
                points.Add(new Point(X[i], Y[i], Z[i]));
            }
            return points;
        }
    }
}
