﻿/*****************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 *****************************************************************************/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using openDicom.Registry;

namespace TPClib.Image
{
    /// <summary>
    /// Class representing imagefile produced by Micro PET scanner
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class InterfileFile: ImageFile
    {
        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event IOProcessEventHandler IOProgress;
        /// <summary>
        /// List of frame subheaders or null. Subheaders are only generated for 
        /// frames in dynamic data.
        /// </summary>
        public HeaderFieldList[] subheaders;
        /// <summary>
        /// Creates and assings the basename to this class.
        /// </summary>
        /// <param name="filename">full path to filename</param>
        public InterfileFile(string filename)
        {
            if (filename.EndsWith(".i"))
                this.filename = filename.Remove(filename.Length - 2);
            else
                this.filename = filename;
            base.filetype = FileType.Interfile;
            base.header = new ImageHeader();
            base.image = null;
            header.dataunit = Data_unit.Undefined;
            header.datatype = TPClib.Image.ImageFile.DataType.FLT32;
        }
        /// <summary>
        /// Constructs Interfile file.
        /// </summary>
        /// <param name="filename">full name of Interfile file</param>
        /// <param name="image">image data</param>
        /// <param name="header">image header</param>
        public InterfileFile(string filename, Image image, ImageHeader header)
            : this(filename)
        {
            this.image = image;
            this.header = new DynamicImageHeader(header);
            this.header.datatype = DataType.FLT32;
            this.header.dim = image.dim;
            this.header.dataunit = header.dataunit;
            //create subheaders for gates and frames
            if (image.dim.Length > IntLimits.FRAMES && image.dim.GetDimension(IntLimits.FRAMES) > 1)
            {
                if (image.dim.Length > IntLimits.GATES && image.dim.GetDimension(IntLimits.GATES) > 1)
                    subheaders = new HeaderFieldList[image.dim.GetDimension(IntLimits.GATES) * image.dim.GetDimension(IntLimits.FRAMES)];
                else
                    subheaders = new HeaderFieldList[image.dim.GetDimension(IntLimits.FRAMES)];
            }
            else {
                subheaders = new HeaderFieldList[1];
            }
            float[] start_times = null;
            float[] durations = null;
            if (image is DynamicImage)
            {
                start_times = (image as DynamicImage).GetFrameStartTimes();
                durations = (image as DynamicImage).GetFrameDurations();
            }
            for (int i = 0; i < subheaders.Length; i++) {
                subheaders[i] = new HeaderFieldList();
                if (image is DynamicImage && i < start_times.Length)
                {
                    subheaders[i].Add("image relative start time", (int)start_times[i]);
                    subheaders[i].Add("image duration", (int)durations[i]);
                }
                else {
                    subheaders[i].Add("image relative start time", (int)0);
                    subheaders[i].Add("image duration", (int)0);
                }
            }
        }
        /// <summary>
        /// Constructs an empty (all-zero) Interfile file with filename and dimensions.
        /// </summary>
        /// <param name="filename">filename (*.img)</param>
        /// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
        public InterfileFile(string filename, params int[] dim)
        {
            if (filename.EndsWith(".i"))
                this.filename = filename.Remove(filename.Length - 2);
            else
                this.filename = filename;
            filetype = FileType.Interfile;
            if (dim.Length > IntLimits.FRAMES)
            {
                header = new DynamicImageHeader();
                (header as DynamicImageHeader).dim = new TPClib.IntLimits(dim);
                image = new DynamicImage(dim);
            }
            else
            {
                header = new ImageHeader();
                header.dim = new TPClib.IntLimits(dim);
                image = new Image(dim);
            }
            header.datatype = TPClib.Image.ImageFile.DataType.FLT32;
            header.dataunit = Data_unit.Undefined;
        }
        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">subregion that is read from file</param>
        public override void ReadSubImage(IntLimits region)
        {
            ReadFile();
            header.dim = region;
            image = image.GetSubImage(region);
        }
        /// <summary>
        /// Checks file format. 
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a Interfile file</returns>
        /// <exception cref="TPCInterfileFileException">if there was a read problem</exception>
        public static bool CheckFormat(string filename)
        {
            //files filename and filename.hdr must exist
            string hdrname;
            //try to resolve exsiting header file name
            try
            {
                hdrname = ResolveHdrName(filename);
            } catch(Exception) {
                return false;
            }
            string line;
            StreamReader reader;
            try
            {
                FileStream stream = new FileStream(hdrname, FileMode.Open);
                reader = new StreamReader(stream);
                line = reader.ReadLine();
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Cannot open file " + filename + ".hdr for reading:" + e);
            }
            if (line.StartsWith("!INTERFILE"))
            {
                reader.Close();
                return true;
            }
            else
            {
                return false;
            }
        }
        /// <summary>
        /// Help routine for ReadFile
        /// </summary>
        /// <returns></returns>
        private delegate float GetValue();
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started (ignored)</param>
        /// <param name="scale">scale factor</param>
        public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {
            int datalength = dim.GetProduct();
            if (offset + datalength > data.dataLength)
                throw new TPCInterfileFileException("Input data of length " + datalength + " does not fit into image of length " + data.dataLength);
            //initialization for event sending
            IOProcessEventHandler pevent = null;
            int voxels_in_plane = 0;
            //seek position
            try
            {
                stream.Seek(position, SeekOrigin.Begin);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Failed to seek starting position " + position + " from " + filename + ":" + e);
            }

            //read data from position
            BinaryReader br = new BinaryReader(stream);
            GetValue getValue;
            switch (datatype)
            {
                case DataType.BIT16_U:
                    getValue = delegate()
                    {
                        return br.ReadUInt16();
                    };
                    break;
                case DataType.BIT16_S:
                    getValue = delegate()
                    {
                        return br.ReadInt16();
                    };
                    break;
                case DataType.FLT32:
                    getValue = delegate()
                    {
                        return br.ReadSingle();
                    };
                    break;
                default:
                    throw new TPCInterfileFileException("Datatype is unsupported for reading with this method type: " + datatype);
            }
            // start reading data.
            voxels_in_plane = dim.dimx * dim.dimy;
            for (int i = 0; i < datalength; i++)
            {
                try
                {
                    data[offset + i] = getValue()*scale;
                }
                catch (System.IO.EndOfStreamException) {
                    throw new TPCException("End of file reached before reading all data dim="+data.dim+" datatype="+header.datatype.ToString());
                }
                //send event after each plane
                if (i > 0 && i % voxels_in_plane == 0)
                {
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)datalength), System.Reflection.MethodBase.GetCurrentMethod()));
                }
            }
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
        }
        /// <summary>
        /// Gets parameters needed for direct access to pixel data.
        /// </summary>
        /// <returns>parameters to pixel data</returns>
        public override PixelDataParameters GetParametersForPixelData()
        {
            PixelDataParameters p;
            p.datatype = header.datatype;
            p.dim = image.dim;
            p.filename = filename+".i";
            p.position = 0;
            p.scale = 1.0f;
            return p;
        }
        /// <summary>
        /// Resolves basenames of available Interfile files in path
        /// </summary>
        /// <param name="path">full path to the files</param>
        /// <returns>array of files</returns>
        public static FileInfo[] ResolveInterfileBasenames(string path)
        {
            char[] chars;
            string pathroot = Path.GetDirectoryName(path);
            string filename = Path.GetFileName(path);
            chars = Path.GetInvalidFileNameChars();
            for (int i = 0; i < chars.Length; i++)
            {
                if (chars[i].CompareTo('*') == 0) continue;
                if (filename.Contains(chars[i].ToString())) throw new TPCInterfileFileException("Invalid character '" + chars[i] + "' in filename");
            }
            chars = Path.GetInvalidPathChars();
            for (int i = 0; i < chars.Length; i++)
                if (pathroot.Contains(chars[i].ToString())) throw new TPCInterfileFileException("Invalid character '" + chars[i] + "' in path");
            try
            {
                DirectoryInfo dir = new DirectoryInfo(pathroot);
                FileInfo[] filenames = dir.GetFiles(filename, SearchOption.TopDirectoryOnly);
                //resolve individual basenames
                List<FileInfo> fileinfos = new List<FileInfo>();
                for (int i = 0; i < filenames.Length; i++)
                {
                    string basename = Path.GetFileNameWithoutExtension(filenames[i].Name);
                    basename = Path.GetFileNameWithoutExtension(basename);
                    //skip if same basename is already in the list
                    if (fileinfos.FindIndex(new Predicate<FileInfo>(
                        delegate(FileInfo a)
                        {
                            if (a.Name.Equals(basename)) return true;
                            return false;
                        })) > -1)
                        continue;
                    filenames[i] = new FileInfo(filenames[i].DirectoryName +Path.DirectorySeparatorChar+basename);
                    fileinfos.Add(filenames[i]);
                }
                return fileinfos.ToArray();
            }
            catch (ArgumentException e)
            {
                throw new TPCInterfileFileException("Path contains invalid characters:" + e.Message);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("While creating DirectoryInfo:" + e.Message);
            }
        }
        /// <summary>
        /// Reads all interfiles files that match criteria into single file. 
        /// All dimensions higher than or equal to 4th are stacked together as frames.
        /// </summary>
        /// <param name="file">full path to the files</param>
        /// <returns>array of DICOM files</returns>
        public static InterfileFile ReadMultiple(string file)
        {
            //all filenames found in path
            FileInfo[] filenames = ResolveInterfileBasenames(file);
            //1st read DICOM holding header information
            InterfileFile file_1st = null;
            //currently read DICOM holding header information
            InterfileFile file_current;
            //DICOM files holding header information
            List<InterfileFile> files = new List<InterfileFile>();
            //Event handler where progress events are sent to
            IOProcessEventHandler pevent = null;
            //gather recognized files
            for (int i = 0; i < filenames.Length; i++)
            {
                try
                {
                    if (!InterfileFile.CheckFormat(filenames[i].FullName))
                    {
                        //just skip the file if format does not match
                        continue;
                    }
                    //read all header fields except the actual image data
                    file_current = new InterfileFile(filenames[i].FullName);
                    file_current.ReadHeader();
                    pevent = IOProgress;
                    if (i % 5 == 0 && pevent != null)
                        pevent.Invoke(file_current, new IOProgressEventArgs(100.0f * (float)i / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
                    if (file_1st != null)
                    {
                        //ensure consistency of data
                        if (!file_1st.header.dim.Equals(file_current.header.dim))
                            throw new TPCInterfileFileException("Inconsistent dimensions in read Interfile files. Cannot combine into single file.");
                    }
                    else
                    {
                        file_1st = file_current;
                    }
                    files.Add(file_current);
                }
                catch (TPCUnrecognizedFileFormatException)
                {
                    // just skip the file 
                }
                catch (TPCException e)
                {
                    // print error and skip the file 
                    Console.WriteLine("Cannot read DICOM file " + filenames[i].FullName + ":" + e.Message);
                }
            }
            if (files.Count == 0) throw new TPCInterfileFileException("No Interfile files was found.");
            int framelength = file_1st.header.dimx * file_1st.header.dimy * file_1st.header.dimz;
            if (framelength == 0) throw new TPCInterfileFileException("Zero-dimensioned data");

            //resolve file size and allocate memory
            InterfileFile interfile;
            if (files.Count == 1)
            {
                interfile = new InterfileFile(file, file_1st.header.dimx, file_1st.header.dimy, file_1st.header.dimz);
                interfile.header = file_1st.header;
            }
            else
            {
                //sort files according to frame field 
                //if the field does not exist in some of the files, then those files are not sorted
                files.Sort(new Comparison<InterfileFile>(delegate(InterfileFile a, InterfileFile b) {
                    try {
                        int frame_a = (int)a.header["frame"];
                        int frame_b = (int)b.header["frame"];
                        if(frame_a < frame_b) return -1;
                        else if(frame_a > frame_b) return 1;
                        return 0;
                    }
                    catch(Exception) {}
                    return 0;
                }));
                
                interfile = new InterfileFile(file, file_1st.header.dimx, file_1st.header.dimy, file_1st.header.dimz, files.Count);
                interfile.header = new DynamicImageHeader(file_1st.header);
                //set dynamic header information
                try {
                    (interfile.header as DynamicImageHeader).isotope = (Isotope_enumerator)file_1st.header["Dose type"];
                } catch(Exception) {}
                try
                {
                    if (((UnitConverter)file_1st.header["Dose Strength (unit)"]).Equals(Data_unit.MBQ_per_ML))
                        (interfile.header as DynamicImageHeader).injected_dose = (float)file_1st.header["Dose Strength (value)"];
                }
                catch (Exception) { }
                List<HeaderFieldList> subheaderlist = new List<HeaderFieldList>();
                for (int i = 0; i < files.Count; i++)
                {
                    try {
                        (interfile.image as DynamicImage).SetFrameStartTime(i, (double)((int)files[i].header["image relative start time"]));
                    } catch (Exception) { }
                    try {
                        (interfile.image as DynamicImage).SetFrameDuration(i, (double)((int)files[i].header["image duration"]));
                    } catch (Exception) { }
                    HeaderFieldList hdr = new HeaderFieldList();
                    hdr.AddRange(files[i].header);
                    subheaderlist.Add(hdr);
                }
                interfile.subheaders = subheaderlist.ToArray();
            }
            interfile.header.dim = interfile.image.dim;
            interfile.header.datatype = file_1st.header.datatype;

            //read data from previously acquired positions
            int location = 0;
            float scale = 1.0f;
            for (int i = 0; i < files.Count; i++)
            {
                if(files[i].header.Contains("quantification factor"))
                    scale = (float)files[i].header["quantification factor"];
                else
                    scale = 1.0f;
                files[i].GetPixelData(ref interfile.image, location, files[i].filename+".i", 
                                      file_1st.header.dim, interfile.header.datatype, 0, scale);
                location += framelength;
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(files[i], new IOProgressEventArgs(100.0f * (float)i / files.Count, System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel data."));
            }
            return interfile;
        }
        /// <summary>
        /// Writes single frame into file. If frame number is larger than 
        /// current number of frames in file, then file size is grown.
        /// </summary>
        /// <remarks>The header information including scaling factor is not updated by this method.</remarks>
        /// <param name="image">image data that is written</param>
        /// <param name="frame_No">frame number that is written [1..no frames+1]</param>
        public override void WriteFrame(ref Image image, int frame_No)
        {
            if((!(image is DynamicImage) && frame_No != 1) || (frame_No < 1 || frame_No > (image as DynamicImage).frames))
                throw new TPCInvalidArgumentsException("Frame number "+frame_No+" is out of bounds "+image.dim);
            //try to open file for writing
            FileStream stream;
            try
            {
                stream = File.Open(filename+".i", FileMode.CreateNew);
            }
            catch (Exception e)
            {
                throw new TPCMicroPETFileException("Could not open file [" + filename + ".i" + "] for writing:" + e);
            }
            IntLimits framedim = new IntLimits(image.dim.dimx, image.dim.dimy, image.dim.dimz);
            int framelength = framedim.GetProduct();
            float scale = 1.0f;
            try
            {
                scale = (float)header["quantification factor"];
            }
            catch {}
            try
            {
                WritePixelData(ref image, framelength * (frame_No - 1), stream, framedim, header.datatype, 0, scale);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Exception while writing frame data:" + e);
            }
            finally {
                stream.Close();
            }
        }
        /// <summary>
        /// Writes pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position for writing</param>
        /// <param name="scale">scale factor</param>
        public void WritePixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {
            int write_end = offset+dim.GetProduct();
            if (scale == 0)
                throw new TPCInvalidArgumentsException("Scale factor cannot be zero.");
            if (offset < 0 || write_end > data.dataLength)
                throw new TPCInvalidArgumentsException("Write region [" + offset + ".." + write_end + "] is out of bounds " + data.dim);
            stream.Seek(position, SeekOrigin.Begin);
            BinaryWriter br = new BinaryWriter(stream);
            switch (datatype)
            {
                //16-bit unsigned int
                case DataType.BIT16_U:
                    for (int i = offset; i < write_end; i++)
                    {
                        br.Write((UInt16)(data[i]/scale));
                    }
                    break;
                //16-bit signed int
                case DataType.BIT16_S:
                    for (int i = offset; i < write_end; i++)
                    {
                        br.Write((Int16)(data[i] / scale));
                    }
                    break;
                case ImageFile.DataType.FLT32:
                    for (int i = offset; i < write_end; i++)
                    {
                        br.Write(data[i] / scale);
                    }
                    break;
                default:
                    throw new TPCInterfileFileException("Unsupported data type in Interfile for writing: " + header.datatype);
            }
            br.Flush();
        }
        /// <summary>
        /// Resolves header filename that corresponds to basename. For dynamic image returns 
        /// first header file name.
        /// </summary>
        /// <param name="filename">Interfile basename or path for dynamic file</param>
        /// <returns>header file name</returns>
        protected static string ResolveHdrName(string filename) {
            if (File.Exists(filename + ".i.hdr")) return filename + ".i.hdr";
            if (File.Exists(filename + ".h33")) return filename + ".h33";
            if (File.Exists(filename + ".hdr")) return filename + ".hdr";
            if (filename.EndsWith("*"))
            {
                FileInfo[] fileinfos = ResolveInterfileBasenames(filename);
                if (fileinfos.Length > 0) {
                    string filename_1st = Path.ChangeExtension(fileinfos[0].FullName,"");
                    if(filename_1st.EndsWith(".i"))
                        filename_1st = Path.ChangeExtension(filename_1st, "");
                    filename_1st = filename_1st.TrimEnd('.');
                    return ResolveHdrName(filename_1st);
                }
            }
            throw new TPCInterfileFileException("Could not resolve header file for basename [" + filename + "]");
        }
        /// <summary>
        /// Reads only header information from file. 
        /// </summary>
        /// <returns>Header information in file</returns>
        public override ImageHeader ReadHeader()
        {
            StreamReader reader;
            try
            {
                string hdrname = ResolveHdrName(filename);
                FileStream stream = new FileStream(hdrname, FileMode.Open);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Cannot open file " + filename + ".hdr for reading:" + e);
            }
            try
            {
                string line;
                string[] splitted;
                while(!reader.EndOfStream) {
                    line = reader.ReadLine();
                    //anythign that has assignment operator is not a comment
                    if (!line.Contains(":=")) continue;
                    if (line.StartsWith("!end of interfile")) break;
                    splitted = line.Split(new string[]{":="}, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0)
                        splitted[0] = splitted[0].Trim();
                    #region resolve_data_type
                    if (splitted.Length > 1) {
                        try
                        {
                            splitted[1] = splitted[1].Trim();
                            if (splitted[0].Equals("!study date (dd:mm:yryr)"))
                            {
                                header.Add(splitted[0], System.DateTime.Parse(splitted[1].Replace(':','/')));
                            }
                            else if (splitted[0].Equals("!study time (hh:mm:ss)"))
                            {
                                System.DateTime study_time = System.DateTime.Parse("01/01/1970 "+splitted[1]);
                                header.Add(splitted[0], study_time);
                            }
                            else if (splitted[0].Equals("Dose type"))
                            {
                                header.Add(splitted[0], Isotope.CreateIsotope(splitted[1]));
                            }
                            else if (splitted[0].Equals("dose_start_time"))
                            {
                                header.Add(splitted[0], System.DateTime.Parse(splitted[1]));
                            }
                            else if (splitted[0].Equals("Dosage Strength"))
                            {
                                splitted = splitted[1].Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                                if (splitted.Length == 1)
                                    header.Add("Dosage Strength", float.Parse(splitted[0]));
                                if (splitted.Length == 2)
                                {
                                    //add additional field containing dosage unit
                                    header.Add("Dosage Strength (value)", float.Parse(splitted[0].Replace('.',',')));
                                    header.Add("Dosage Strength (unit)", UnitConverter.CreateActivityUnit(splitted[1]));
                                }
                            }
                            //NOTE:this header field is handled here so that is taken as a string even if it would be 
                            //possible to parse it sometimes into integer
                            else if (splitted[0].Equals("Patient ID") ||
                                     splitted[0].Equals("Patient name") ||
                                     splitted[0].Equals("data description"))
                            {
                                header.Add(splitted[0], splitted[1]);
                            }
                            // try to parse field
                            // 1.: as int
                            // 1.: as long
                            // 2.: as float
                            else {
                                try
                                {
                                    int intvalue = int.Parse(splitted[1]);
                                    header.Add(splitted[0], intvalue);
                                }
                                catch (Exception)
                                {
                                    try
                                    {
                                        long longvalue = long.Parse(splitted[1]);
                                        header.Add(splitted[0], longvalue);
                                    }
                                    catch (Exception)
                                    {
                                        try
                                        {
                                            float floatvalue = float.Parse(splitted[1].Replace('.',','));
                                            header.Add(splitted[0], floatvalue);
                                        }
                                        catch (Exception)
                                        {
                                            header.Add(splitted[0], splitted[1]);
                                        }
                                    }
                                }
                            }
                        }
                        //do nothing, just skip to next header field
                        catch (Exception) { }
                    }
                    #endregion
                }
            }
            catch (Exception e) {
                throw new TPCInterfileFileException("Exception while reading header file "+filename+".hdr:"+e);
            }
            reader.Close();

            //fill ImageHeader structure
            #region fill_ImageHeader_fields
            try { header.patient_name = header["Patient name"].ToString(); }
            catch (Exception) { }
            try { header.patientID = header["Patient ID"].ToString(); }
            catch (Exception) { }
            try { header.description = header["data description"].ToString(); }
            catch (Exception) { }
            try {
                int dimx = (int)header["matrix size [1]"];
                int dimy = (int)header["matrix size [2]"];
                int dimz = (int)header["matrix size [3]"];
                header.dim = new IntLimits(dimx, dimy, dimz);
            }
            catch (Exception) { }
            try
            {
                float sizx = (float)header["scaling factor (mm/pixel) [1]"];
                float sizy = (float)header["scaling factor (mm/pixel) [2]"];
                float sizz = (float)header["scaling factor (mm/pixel) [3]"];
                header.siz = new Voxel(sizx, sizy, sizz);
            }
            catch (Exception) { }
            try { header.dataunit = (Data_unit)header["Dose type"]; }
            catch (Exception) { }
            try
            {
                string format = header["number format"].ToString();
                int bytesperpixel = (int)header["number of bytes per pixel"];
                switch (bytesperpixel) {
                    case 2:
                        if (format.Contains("unsigned"))
                            header.datatype = DataType.BIT16_U;
                        else if (format.Contains("signed"))
                            header.datatype = DataType.BIT16_S;
                        break;
                    case 4:
                        if (format.Contains("float"))
                            header.datatype = DataType.FLT32;
                        break;
                    default:
                        header.datatype = DataType.FLT32;
                        break;
                }
            }
            catch (Exception) {
                //default datatype
                header.datatype = DataType.FLT32;
            }
            if (header.Contains("dose_start_time"))
            {
                if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                (header as DynamicImageHeader).dose_start_time = (System.DateTime)header["dose_start_time"];
            }
            if (header.Contains("radiopharmaceutical"))
            {
                if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                (header as DynamicImageHeader).radiopharma = (string)header["radiopharmaceutical"];
                Console.WriteLine("READ[" + (header as DynamicImageHeader).radiopharma + "]");
            }
            else if (header.Contains("Radiopharmaceutical"))
            {
                if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                (header as DynamicImageHeader).radiopharma = (string)header["Radiopharmaceutical"];
                Console.WriteLine("READ[" + (header as DynamicImageHeader).radiopharma + "]");
            }
            if (header.Contains("isotope name"))
            {
                if (!(header is DynamicImageHeader)) header = new DynamicImageHeader(header);
                (header as DynamicImageHeader).isotope = Isotope.CreateIsotope((string)header["isotope name"]);
            }
            //try to resolve data units
            if(header.Contains("Units[1]"))
                header.dataunit = UnitConverter.CreateDataUnit(header["Units[1]"].ToString());
            else if(header.Contains("units[1]"))
                header.dataunit = UnitConverter.CreateDataUnit(header["units[1]"].ToString());
            else
                header.dataunit = Data_unit.Unknown;
            //try to quess modality from existence of some fields
            //the find results are 
            if (header.Contains("!imaging modality"))
            {
                try {
                    string str = ((string)header["!imaging modality"]).Trim().ToUpperInvariant();
                    if (str.Equals("NUCMED"))
                        header.modality = ImageModality.M_NM;
                    else if(str.StartsWith("CT"))
                        header.modality = ImageModality.M_CT;
                    else if (str.StartsWith("MR"))
                        header.modality = ImageModality.M_MR;
                    else if (str.StartsWith("SPECT"))
                        header.modality = ImageModality.M_PX;
                    else if (str.StartsWith("PET"))
                        header.modality = ImageModality.M_PT;
                    else if (str.StartsWith("RT"))
                        header.modality = ImageModality.M_RT;
                }
                catch
                {
                    //do nothing if modality was not a string
                }
            }
            else
            {
                foreach (KeyValuePair<string, object> a in header)
                {
                    if (a.Key.Contains("PET data type"))
                    {
                        header.modality = ImageModality.M_PT;
                        break;
                    }
                    else if (a.Key.Contains("SPECT STUDY"))
                    {
                        header.modality = ImageModality.M_PX;
                        break;
                    }
                }
            }
            #endregion

            return header;
        }
        /// <summary>
        /// Reads the .img and .hdr from 
        /// the file to the data structures.
        /// </summary>
        public override void ReadFile()
        {
            //read header
            header = ReadHeader();
            //create image according to header information
            image = new Image(header.dim);
            //read pixel data
            float scale = 1.0f;
            if (header.Contains("quantification factor"))
            {
                try {
                    scale = (float)header["quantification factor"];
                }
                catch {}
            }
            string imagefilename = filename;
            if(File.Exists(filename + ".i")) {   imagefilename = filename + ".i";    }
            else if(File.Exists(filename + ".img")) {   imagefilename = filename + ".img";    }
            GetPixelData(ref image, 0, imagefilename, header.dim, header.datatype, 0, scale);
        }
        /// <summary>
        /// Writes Interfile header data into curretly selected header file.
        /// </summary>
        protected void WriteHeader()
        {
            System.IO.StreamWriter hdrfile;
            try
            {
                FileStream stream = File.Open(filename + ".i.hdr", FileMode.CreateNew, FileAccess.Write);
                hdrfile = new System.IO.StreamWriter(stream);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Failed to open header file " + filename + ".i.hdr for writing:" + e);
            }
            //copy from general header structure
            #region set_general_header_data
            header["Patient name"] = header.patient_name;
            header["Patient ID"] = header.patientID;
            header["data description"] = header.description;
            header["matrix size [1]"] = header.dimx;
            header["matrix size [2]"] = header.dimy;
            header["matrix size [3]"] = header.dimz;
            header["scaling factor (mm/pixel) [1]"] = header.sizex;
            header["scaling factor (mm/pixel) [2]"] = header.sizey;
            header["scaling factor (mm/pixel) [3]"] = header.sizez;
            switch(header.modality) {
                case ImageModality.M_NM:
                    header["!imaging modality"] = "NUCMED";
                    break;
                default:
                    header["!imaging modality"] = header.modality.ToString();
                    break;
            }
            if (header is DynamicImageHeader)
            {
                header["Dose type"] = (header as DynamicImageHeader).isotope;
                header["dose_start_time"] = (header as DynamicImageHeader).dose_start_time;
            }
            header["number of bytes per pixel"] = ImageFile.BytesPerPixel(header.datatype);
            switch(header.datatype) {
                case DataType.FLT32:
                    header["number format"] = "float";
                    break;
                case DataType.BIT16_U:
                    header["number format"] = "unsigned short";
                    break;
                case DataType.BIT16_S:
                    header["number format"] = "signed short";
                    break;
                default:
                    header["number format"] = "unknown";
                    break;
            }
            #endregion
            header["!data starting block"] = 0;
            for (int i = 0; i < header.dim.Length; i++) {
                header["Units[" + (i + 1) + "]"] = UnitConverter.ConvertToString(header.dataunit);
            }

            //write header data
            try
            {
                hdrfile.WriteLine("!INTERFILE");
                for (int i = 0; i < header.Count; i++)
                {
                    #region special_rules_for_writing
                    if (header[i].Key.Equals("isotope halflife"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0000.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("Dosage Strength (value)"))
                    {
                        string value = ((float)header[i].Value).ToString("0.000000").Replace(',', '.');
                        string unit = "";
                        try
                        {
                            unit = (string)header["Dosage Strength (unit)"];
                        }
                        catch (Exception) { }
                        hdrfile.WriteLine("Dosage Strength := " + value + " " + unit);
                    }
                    else if (header[i].Key.Equals("Dosage Strength (unit)"))
                    {
                        //do nothing here because the field is written with (value)
                    }
                    else if (header[i].Key.Equals("!study date (dd:mm:yryr)"))
                    {
                        System.DateTime study_date = (System.DateTime)header[i].Value;
                        hdrfile.WriteLine(header[i].Key + " := " + study_date.ToString("dd:MM:yyyy"));
                    }
                    else if (header[i].Key.Equals("!study time (hh:mm:ss)"))
                    {
                        System.DateTime study_time = (System.DateTime)header[i].Value;
                        hdrfile.WriteLine(header[i].Key + " := " + study_time.ToString("HH:mm:ss"));
                    }
                    else if (header[i].Key.Equals("branching factor"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("Dead time correction factor"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("!histogrammer revision"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.0").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("scaling factor (mm/pixel) [1]"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("scaling factor (mm/pixel) [2]"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("scaling factor (mm/pixel) [3]"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("decay correction factor"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("decay correction factor2"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((float)header[i].Value).ToString("0.000000").Replace(',', '.'));
                    }
                    else if (header[i].Key.Equals("Dose type"))
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((Isotope_enumerator)header[i].Value).ToString());
                    }
                    #endregion
                    else if (header[i].Value is Char[]) {
                        hdrfile.WriteLine(header[i].Key + " := " + (new string((Char[])header[i].Value)));
                    }
                    else if (header[i].Value is Single) {
                        hdrfile.WriteLine(header[i].Key + " := " + (((Single)header[i].Value).ToString("0.000000").Replace(',', '.')));
                    }
                    else if (header[i].Value is Double)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + (((Double)header[i].Value).ToString("0.000000").Replace(',', '.')));
                    }
                    else if (header[i].Value is UInt32)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((UInt32)header[i].Value).ToString());
                    }
                    else if (header[i].Value is Int32)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((Int32)header[i].Value).ToString());
                    }
                    else if (header[i].Value is System.DateTime)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((System.DateTime)header[i].Value));
                    }
                    else if (header[i].Value is System.TimeSpan)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((System.TimeSpan)header[i].Value).ToString());
                    }
                    else if (header[i].Value is Uid)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((Uid)header[i].Value).GetDictionaryEntry().Name);
                    }
                    else if (header[i].Value is String)
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + ((String)header[i].Value));
                    }
                    else if (header[i].Value is Int16[])
                    {
                        Int16[] integers = (Int16[])header[i].Value;
                        hdrfile.Write(header[i].Key + " :=");
                        foreach (Int16 s in integers)
                            hdrfile.Write(" " + s.ToString());
                        hdrfile.WriteLine();
                    }
                    else if (header[i].Value is Single[])
                    {
                        Single[] singles = (Single[])header[i].Value;
                        hdrfile.Write(header[i].Key + " :=");
                        foreach (Single s in singles)
                            hdrfile.Write(" " + s.ToString("0.000000").Replace(',', '.'));
                        hdrfile.WriteLine();
                    }
                    else if (header[i].Key == "PixelData")
                    {
                        //do nothing for pixel data since it is not header information
                    }
                    else
                    {
                        hdrfile.WriteLine(header[i].Key + " := " + header[i].Value);
                    }
                }
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Failed to write header data into file " + filename + ".hdr:" + e);
            }
            hdrfile.Close();
        }
        /// <summary>
        /// Writes both the header and image data to the disk.
        /// </summary>
        /// <param name="image">image data that is written</param>
        public override void WriteFile(ref Image image)
        {
            //resolve scaling factor according to datatype
            #region resolve_scaling_factor
            float scale = 1.0f;
            float min = image.GetMin();
            float max = image.GetMax();
            switch (header.datatype)
            {
                case DataType.BIT8_S:
                    if (Math.Abs(min) <= Math.Abs(max))
                        scale = max / (SByte.MaxValue);
                    else
                        scale = min / (SByte.MinValue);
                    header["!maximum pixel count"] = SByte.MaxValue;
                    break;
                case DataType.BIT8_U:
                    scale = max / Byte.MaxValue;
                    header["!maximum pixel count"] = Byte.MaxValue;
                    break;
                case DataType.BIT16_S:
                    if (Math.Abs(min) <= Math.Abs(max))
                        scale = max / (Int16.MaxValue);
                    else
                        scale = min / (Int16.MinValue);
                    header["!maximum pixel count"] = Int16.MaxValue;
                    break;
                case DataType.BIT16_U:
                    scale = max / UInt16.MaxValue;
                    header["!maximum pixel count"] = UInt16.MaxValue;
                    break;
                case DataType.FLT32:
                case DataType.FLT64:
                case DataType.VAXFL32:
                    break;
                default:
                    throw new TPCInterfileFileException("Unsupported datatype " + header.datatype.ToString() + " for resolving scale factor.");
            }
            #endregion
            header["quantification factor"] = scale;

            if (image is DynamicImage) {
                string basename = filename;
                basename = basename.TrimEnd('*');
                for(int i = 0; i < (image as DynamicImage).frames; i++) {
                    InterfileFile file;
                    if((image as DynamicImage).frames > 1)
                        file = new InterfileFile(basename + "_frame" + i);
                    else
                        file = new InterfileFile(basename);
                    string datafile = Path.GetFileName(file.filename);
                    //take header from subheader list if it exists
                    //otherwise use header
                    if (subheaders != null && subheaders.Length > i)
                    {
                        file.header = new DynamicImageHeader(new ImageHeader(subheaders[i]));
                        if (header is DynamicImageHeader) {
                            (file.header as DynamicImageHeader).dose_start_time = (header as DynamicImageHeader).dose_start_time;
                            (file.header as DynamicImageHeader).injected_dose = (header as DynamicImageHeader).injected_dose;
                            (file.header as DynamicImageHeader).isotope = (header as DynamicImageHeader).isotope;
                            (file.header as DynamicImageHeader).radiopharma = (header as DynamicImageHeader).radiopharma;
                        }
                        file.header.datatype = header.datatype;
                        file.header.dataunit = header.dataunit;
                        file.header.description = header.description;
                        file.header.dim = image.dim;
                        file.header.patient_name = header.patient_name;
                        file.header.patientID = header.patientID;
                        file.header.siz = header.siz;
                        file.header.modality = header.modality;
                        file.header.AddRange(header);
                    }
                    else
                    {
                        if (header is DynamicImageHeader)
                            file.header = header;
                        else
                        {
                            file.header = new DynamicImageHeader(header);
                            (file.header as DynamicImageHeader).dose_start_time = (header as DynamicImageHeader).dose_start_time;
                            (file.header as DynamicImageHeader).injected_dose = (header as DynamicImageHeader).injected_dose;
                            (file.header as DynamicImageHeader).isotope = (header as DynamicImageHeader).isotope;
                            (file.header as DynamicImageHeader).radiopharma = (header as DynamicImageHeader).radiopharma;
                        }
                    }
                    //add or rename some fields
                    if (!file.header.Contains("name of data file"))
                        file.header.Add(new KeyValuePair<string, object>("name of data file", datafile + ".i"));
                    else
                        file.header["name of data file"] = datafile + ".i";
                    //set frame number
                    try
                    {
                        file.header["frame"] = i;
                    } catch(TPCInvalidArgumentsException) {
                        file.header.Add(new KeyValuePair<string,object>("frame", i));
                    }
                    file.WriteHeader();
                    file.WriteFrame(ref image, i+1);
                }
                return;
            }

            //write header data
            WriteHeader();
            //initialization for event sending
            IOProcessEventHandler pevent = null;
            //write image data
            string fname = filename;
            //resolve frame length in voxels
            int framelength = (int)image.dim.GetProduct(IntLimits.PLANES);

            //try to open file for writing
            FileStream stream;
            try
            {
                stream = File.Open(fname+".i", FileMode.CreateNew);
            }
            catch (Exception e)
            {
                throw new TPCInterfileFileException("Could not open file [" + fname + "] for writing:" + e);
            }
            //resolve voxels in plane for sending event after writing each plane
            int voxels_in_plane = image.dimx * image.dimy;

            //write data according to data type
            long position = 0;
            for (int i = 0; i < image.dimz; i++) {
                WritePixelData(ref image, i * voxels_in_plane, stream, new IntLimits(image.dimx, image.dimy), header.datatype, position, scale);
                position += voxels_in_plane*ImageFile.BytesPerPixel(header.datatype);
                //send event after each plane
                if (i > 0 && i % voxels_in_plane == 0)
                {
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)image.dataLength), System.Reflection.MethodBase.GetCurrentMethod()));
                }
            }
            stream.Close();
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
        }
   }
}
