﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections;
using System;

namespace TPClib.Image
{
    /// <summary>
    /// Binary image containing only logical values in its voxels.
    /// </summary>
    public class MaskImage: Image
    {
        /// <summary>
        /// Binary data in image
        /// </summary>
        protected new BitArray data;
        /// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension values</param>
        public MaskImage(params int[] dimension)
        {
            if (dimension.Length > 6)
                throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            int product = 1;
            for (int i = 0; i < dimension.Length; i++)
            {
                if (dimension[i] < 0)
                    throw new TPCInvalidArgumentsException("Negative dimension value for creation of image.");
                product *= dimension[i];
            }
            GC.Collect();
            try
            {
                data = new BitArray(product);
            }
            catch (OutOfMemoryException) {
                GC.Collect();
                throw new TPCException("Cannot create image because there is not enough memory.");
            }
            for (int i = 0; i < dimension.Length; i++)
                dim.AddLimit(0, dimension[i]);
        }
        /// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension object, dimension values will be converted to integer values</param>
        public MaskImage(IntLimits dimension)
        {
            if (dimension.Length > 6) throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            for (int i = 0; i < dimension.Length; i++)
                if (dimension.GetLimit(i, Limits.Limit.LOW) != 0)
                    throw new TPCInvalidArgumentsException(i + "'s low limit was non-zero while creating an image:" + dimension.GetLimit(i, Limits.Limit.LOW));
            GC.Collect();
            base.data = null;
            data = new BitArray((int)dimension.GetProduct());
            for (int i = 0; i < dimension.Length; i++)
                dim.AddLimit(dimension.GetLimit(i, Limits.Limit.LOW), dimension.GetLimit(i, Limits.Limit.HIGH));
        }
        /// <summary>
        /// Voxels of image as an array.
        /// </summary>
        /// <param name="location">index number (0-based)</param>
        /// <returns>value at index </returns>
        public override float this[int location]
        {
            get
            {
                if (location < 0 || location >= data.Length)
                    throw new TPCInvalidArgumentsException("Index " + location + " is out of image array bounds [0.." + (data.Length - 1) + "].");
                return (data[location] ? 1.0f : 0.0f);
            }
            set
            {
                if (location < 0 || location >= data.Length)
                    throw new TPCInvalidArgumentsException("Index " + location + " is out of image array bounds [0.." + (data.Length - 1) + "].");
                data[location] = (value != 0 ? true : false);
            }
        }
        /// <summary>
        /// Voxels of image as an multidimensional array.
        /// </summary>
        /// <param name="x">x-coordinate (0-based)</param>
        /// <param name="y">y-coordinate (0-based)</param>
        /// <param name="z">z-coordinate (0-based)</param>
        /// <returns>value at location</returns>
        public override float this[int x, int y, int z]
        {
            get
            {
                return (data[z * dim.dimx * dim.dimy + y * dim.dimx + x] ? 1.0f : 0.0f);
            }
            set
            {
                data[z * dim.dimx * dim.dimy + y * dim.dimx + x] = (value != 0 ? true : false);
            }
        }
        /// <summary>
        /// Voxels of image located by point.
        /// </summary>
        /// <param name="p">location in image</param>
        /// <returns>image voxel value at location</returns>
        public override float this[Point p]
        {
            get
            {
                return (data[(int)p.z * dim.dimx * dim.dimy + (int)p.y * dim.dimx + (int)p.x] ? 1.0f : 0.0f);
            }
            set
            {
                data[(int)p.z * dim.dimx * dim.dimy + (int)p.y * dim.dimx + (int)p.x] = (value != 0 ? true : false);
            }
        }
        /// <summary>
        /// Bitwise AND operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>multiplied image</returns>
        public static MaskImage operator &(MaskImage image, MaskImage value)
        {
            if (image.dim != value.dim)
                throw new TPCException("Cannot apply arithmetic - operation because image sizes are not the same");
            MaskImage r = new MaskImage(image.dim);
            r.data = image.data.And(value.data);
            return r;
        }
        /// <summary>
        /// Bitwise OR operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>resulting image</returns>
        public static MaskImage operator |(MaskImage image, MaskImage value)
        {
            if (image.dataLength != value.dataLength)
                throw new TPCException("Cannot apply arithmetic - operation because image data lengths [" + image.dataLength + "] and [" + value.dataLength + "] are not the same");
            MaskImage r = new MaskImage(image.dim);
            r.data = image.data.Or(value.data);
            return r;
        }
        /// <summary>
        /// Bitwise XOR (exclusive or) operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>resulting image</returns>
        public static MaskImage operator ^(MaskImage image, MaskImage value)
        {
            if (image.dataLength != value.dataLength)
                throw new TPCException("Cannot apply arithmetic - operation because image data lengths [" + image.dataLength + "] and [" + value.dataLength + "] are not the same");
            MaskImage r = new MaskImage(image.dim);
            r.data = image.data.Xor(value.data);
            return r;
        }
        /// <summary>
        /// Multiplies all data with value
        /// </summary>
        /// <param name="value">multiplication value</param>
        public override void Multiply(float value)
        {
            if (value == 0) data.SetAll(false);
        }
        /// <summary>
        /// Returns minimum and maximum data value
        /// </summary>
        /// <returns>array containing [minimum, maximum] value</returns>
        public override float[] GetMinMax()
        {
            if (data.Length == 0) 
                throw new TPCException("Cannot get minimum and maximum because there are no values in the image.");

            float max = 0.0f;
            float min = 1.0f;
            foreach (bool b in data) {
                if (b) max = 1.0f;
                else min = 0.0f;
            }
            return new float[2] { min, max };
        }
        /// <summary>
        /// Returns maximum data value
        /// </summary>
        /// <returns>1.0 if there are one or more ones in the image</returns>
        public override float GetMax()
        {
            foreach (bool b in data) if (b) return 1.0f;
            return 0.0f;
        }
        /// <summary>
        /// Returns minimum data value
        /// </summary>
        /// <returns>0.0 if there are one or more zeros in the image</returns>
        public override float GetMin()
        {
            foreach (bool b in data) if (!b) return 0.0f;
            return 1.0f;
        }
        /// <summary>
        /// Returns the mean data value.
        /// </summary>
        /// <returns>proportion of ones to total number of voxels</returns>
        public override float GetMean()
        {
            if (data.Length == 0)
                throw new TPCException("Cannot calculate mean of image because there is not voxels.");
            float m = 0.0f;
            for (int i = 0; i < data.Length; i++)
                if (data[i]) m++;
            return m / data.Length;
        }
        /// <summary>
        /// Returns value at given location in image.
        /// </summary>
        /// <param name="location">location where value is retrieved (0-based coordinates), representing dimensions [x, y, z, ..]</param>
        /// <returns>data value</returns>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public override float GetValue(params int[] location)
        {
            int i = 0;
            int j = 0;
            int arrayindex = 0;
            int prod = 0;
            for (i = 0; i < location.Length; i++)
            {
                if (i >= dim.Length)
                    throw new TPCInvalidArgumentsException("Tried to get value from dimension index " + i + " outside maximum " + dim.Length + ".");
                if (location[i] < 0 || location[i] >= dim.GetDimension(i))
                    throw new TPCInvalidArgumentsException("" + i + "'s coordinate value " + location[i] + " is out of image dimensions [" + 0 + ".." + (dim.GetDimension(i) - 1) + "].");
                prod = 1;
                for (j = i - 1; j >= 0; j--)
                    prod *= dim.GetDimension(j);
                arrayindex += location[i] * prod;
            }
            if (arrayindex < 0 || arrayindex >= data.Length)
                throw new TPCError("Internal error: index " + arrayindex + " is out of bounds [0.." + (data.Length - 1) + "].");
            return (data[arrayindex] ? 1.0f : 0.0f); ;
        }
        /// <summary>
        /// Sets value at given point in image.
        /// </summary>
        /// <param name="point">point where value is set (0-based coordinates)</param>
        /// <param name="value">new value</param>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public override void SetValue(IntPoint point, float value)
        {
            if (point.x < 0 || point.x >= dim.GetDimension(Limits.WIDTH) ||
               point.y < 0 || point.y >= dim.GetDimension(Limits.HEIGHT) ||
               point.z < 0 || point.z >= dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("Point (" + point.x + "," + point.y + "," + point.z + ") is out of bounds (" + dim + ").");
            data[point.x + point.y * dim.GetDimension(Limits.WIDTH) + point.z * dim.GetDimension(Limits.WIDTH) * dim.GetDimension(Limits.HEIGHT)] = (value != 0 ? true : false);
        }
        /// <summary>
        /// Sets value at given location in image.
        /// </summary>
        /// <param name="value">new value</param>
        /// <param name="location">location where value is set (0-based coordinates)</param>
        /// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
        public override void SetValue(float value, params int[] location)
        {
            int i = 0;
            int j = 0;
            int arrayindex = 0;
            int prod = 0;
            for (i = 0; i < location.Length; i++)
            {
                if (location[i] < 0 || location[i] > dim.GetDimension(i))
                    throw new TPCInvalidArgumentsException("index " + i + " coordinate value " + location[i] + " is out of image dimensions [0.." + dim.GetLimit(i, Limits.Limit.HIGH) + "].");
                prod = 1;
                for (j = i - 1; j >= 0; j--)
                {
                    prod *= dim.GetDimension(j);
                }
                arrayindex += location[i] * prod;
            }
            if (arrayindex < 0 || arrayindex >= data.Length)
            {
                throw new TPCError("Internal error: index " + arrayindex + " is out of bounds [0.." + (data.Length - 1) + "].");
            }
            data[arrayindex] = (value != 0 ? true : false);
        }
        /// <summary>
        /// Creates new subimage from current image.
        /// </summary>
        /// <param name="limits">subregion</param>
        /// <returns>image that has dimensions as subregion and contains corresponding voxel values</returns>
        /// <exception cref="TPCInvalidArgumentsException">invalid image region is accesses</exception>
        public override Image GetSubImage(IntLimits limits)
        {
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length)
                throw new TPCInvalidArgumentsException("Limits " + limits.Length + " does not match source image number of dimensions " + dim.Length + ".");
            if ((limits.Length > Limits.WIDTH) && limits.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dim.GetDimension(Limits.WIDTH))
                throw new TPCInvalidArgumentsException("High x limit must be inside image.");
            if ((limits.Length > Limits.HEIGHT) && limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dim.GetDimension(Limits.HEIGHT))
                throw new TPCInvalidArgumentsException("High y limit must be inside image.");
            if ((limits.Length > Limits.PLANES) && limits.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("High z limit must be inside image.");
            if ((limits.Length > Limits.FRAMES) && limits.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) > dim.GetDimension(Limits.FRAMES))
                throw new TPCInvalidArgumentsException("High t (4th dim) limit must be inside image.");
            if ((limits.Length > Limits.GATES) && limits.GetLimit(Limits.GATES, Limits.Limit.HIGH) > dim.GetDimension(Limits.GATES))
                throw new TPCInvalidArgumentsException("High (5th dim) limit must be inside image.");
            if ((limits.Length > Limits.BEDS) && limits.GetLimit(Limits.BEDS, Limits.Limit.HIGH) > dim.GetDimension(Limits.BEDS))
                throw new TPCInvalidArgumentsException("High (6th dim) limit must be inside image.");
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException("Limits does not match target image number of dimensions.");
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int planes = 1;
            int height = 1;
            int width = 1;
            int[] p = new int[limits.Length];
            int[] q = new int[limits.Length];
            Image r = new Image(limits.GetDimensions());
            if (limits.Length > Limits.BEDS) beds = r.dim.GetDimension(IntLimits.BEDS);
            if (limits.Length > Limits.GATES) gates = r.dim.GetDimension(IntLimits.GATES);
            if (limits.Length > Limits.FRAMES) frames = r.dim.GetDimension(IntLimits.FRAMES);
            if (limits.Length > Limits.PLANES) planes = r.dim.GetDimension(IntLimits.PLANES);
            if (limits.Length > Limits.HEIGHT) height = r.dim.GetDimension(IntLimits.HEIGHT);
            width = r.dim.GetDimension(IntLimits.WIDTH);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (limits.Length > Limits.BEDS)
                {
                    p[5] = b;
                    q[5] = b + limits.GetLimit(Limits.BEDS, Limits.Limit.LOW);
                }
                for (g = 0; g < gates; g++)
                {
                    if (limits.Length > Limits.GATES)
                    {
                        q[4] = g;
                        q[4] = g + limits.GetLimit(Limits.GATES, Limits.Limit.LOW);
                    }
                    for (t = 0; t < frames; t++)
                    {
                        if (limits.Length > Limits.FRAMES)
                        {
                            p[3] = t;
                            q[3] = t + limits.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                        }
                        for (z = 0; z < planes; z++)
                        {
                            if (limits.Length > Limits.PLANES)
                            {
                                p[2] = z;
                                q[2] = z + limits.GetLimit(Limits.PLANES, Limits.Limit.LOW);
                            }
                            for (y = 0; y < height && y < limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
                            {
                                if (limits.Length > Limits.HEIGHT)
                                {
                                    p[1] = y;
                                    q[1] = y + limits.GetLimit(Limits.HEIGHT, Limits.Limit.LOW);
                                }
                                for (x = 0; x < width; x++)
                                {
                                    p[0] = x;
                                    q[0] = x + limits.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
                                    r.SetValue(GetValue(q), p);
                                }
                            }
                        }
                    }
                }
            }
            //set fields
            return r;
        }
        /// <summary>
        /// Sets value to subimage of current image.
        /// </summary>
        /// <param name="limits">subregion</param>
        /// <param name="image">values that are put into image</param>
        /// <exception cref="TPCInvalidArgumentsException">invalid image region is accesses</exception>
        public override void SetSubImage(IntLimits limits, ref Image image)
        {
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException(limits.Length + " limits does not match target image number of dimensions " + dim.Length + ".");
            if (limits.Length < image.dim.Length)
                throw new TPCInvalidArgumentsException("Limits " + limits.Length + " cannot be smaller than source image number of dimensions " + image.dim.Length + ".");
            if ((limits.Length > Limits.WIDTH) && limits.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > dim.GetDimension(Limits.WIDTH))
                throw new TPCInvalidArgumentsException("High x limit must be inside image.");
            if ((limits.Length > Limits.HEIGHT) && limits.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > dim.GetDimension(Limits.HEIGHT))
                throw new TPCInvalidArgumentsException("High y limit must be inside image.");
            if ((limits.Length > Limits.PLANES) && limits.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > dim.GetDimension(Limits.PLANES))
                throw new TPCInvalidArgumentsException("High z limit must be inside image.");
            if ((limits.Length > Limits.FRAMES) && limits.GetLimit(Limits.FRAMES, Limits.Limit.HIGH) > dim.GetDimension(Limits.FRAMES))
            {
                throw new TPCInvalidArgumentsException("High t (4th dim) limit must be inside image.");
            }
            if ((limits.Length > Limits.GATES) && limits.GetLimit(Limits.GATES, Limits.Limit.HIGH) > dim.GetDimension(Limits.GATES))
                throw new TPCInvalidArgumentsException("High (5th dim) limit must be inside image.");
            if ((limits.Length > Limits.BEDS) && limits.GetLimit(Limits.BEDS, Limits.Limit.HIGH) > dim.GetDimension(Limits.BEDS))
                throw new TPCInvalidArgumentsException("High (6th dim) limit must be inside image.");
            if (limits.Length > 6) throw new TPCInvalidArgumentsException("This method is not supported with images having bigger than 6 dimensions.");
            if (limits.Length != dim.Length) throw new TPCInvalidArgumentsException("Limits does not match target image number of dimensions.");
            int x = 0;
            int y = 0;
            int z = 0;
            int t = 0;
            int g = 0;
            int b = 0;
            int beds = 1;
            int gates = 1;
            int frames = 1;
            int planes = 1;
            int height = 1;
            int[] p = new int[image.dim.Length];
            int[] q = new int[limits.Length];
            if (image.dim.Length > Limits.BEDS) beds = image.dim.GetDimension(IntLimits.BEDS);
            if (image.dim.Length > Limits.GATES) gates = image.dim.GetDimension(IntLimits.GATES);
            if (image.dim.Length > Limits.FRAMES) frames = image.dim.GetDimension(IntLimits.FRAMES);
            if (image.dim.Length > Limits.PLANES) planes = image.dim.GetDimension(IntLimits.PLANES);
            if (image.dim.Length > Limits.HEIGHT) height = image.dim.GetDimension(IntLimits.HEIGHT);
            //set data
            for (b = 0; b < beds; b++)
            {
                if (image.dim.Length > Limits.BEDS)
                    p[5] = b;
                if (limits.Length > Limits.BEDS)
                    q[5] = b + limits.GetLimit(Limits.BEDS, Limits.Limit.LOW);
                for (g = 0; g < gates; g++)
                {
                    if (image.dim.Length > Limits.GATES)
                        q[4] = g;
                    if (limits.Length > Limits.GATES)
                        q[4] = g + limits.GetLimit(Limits.GATES, Limits.Limit.LOW);
                    for (t = 0; t < frames; t++)
                    {
                        if (image.dim.Length > Limits.FRAMES)
                            p[3] = t;
                        if (limits.Length > Limits.FRAMES)
                            q[3] = t + limits.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
                        for (z = 0; z < planes; z++)
                        {
                            if (image.dim.Length > Limits.PLANES)
                                p[2] = z;
                            if (limits.Length > Limits.PLANES)
                                q[2] = z + limits.GetLimit(Limits.PLANES, Limits.Limit.LOW);
                            for (y = 0; y < height; y++)
                            {
                                if (image.dim.Length > Limits.HEIGHT)
                                    p[1] = y;
                                if (limits.Length > Limits.HEIGHT)
                                    q[1] = y + limits.GetLimit(Limits.HEIGHT, Limits.Limit.LOW);
                                for (x = 0; x < image.dim.GetDimension(IntLimits.WIDTH); x++)
                                {
                                    p[0] = x;
                                    q[0] = x + limits.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
                                    SetValue(image.GetValue(p), q);
                                }
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Returns one plane from image.
        /// </summary>
        /// <param name="plane">plane index (0-based)</param>
        /// <returns>new image containing plane data</returns>
        public override Image GetPlane(int plane)
        {
            return GetSubImage(GetPlaneLimits(plane));
        }
        /// <summary>
        /// Returns image dimensions and min/max-values as string
        /// </summary>
        /// <returns>string representation object</returns>
        public override string ToString()
        {
            return "MaskImage[" + dim + ",min=" + GetMin() + ",max=" + GetMax() + "]";
        }
        /// <summary>
        /// Evaluates if images equals. Currently only image dimensions and data is evaluated.
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>false if files differ</returns>
        public override bool Equals(object obj)
        {
            if (obj is MaskImage)
            {
                MaskImage img = (obj as MaskImage);
                if (dim != img.dim) return false;
                for (int i = 0; i < data.Length; i++)
                    if (data[i] != img.data[i]) return false;
                return true;
            }
            else
            {
                return false;
            }
        }
    }
}
