﻿/*****************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 *****************************************************************************/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;

namespace TPClib.Image
{
    /// <summary>
    /// Class representing imagefile produced by Micro PET scanner
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class MicroPETFile: ImageFile
    {
        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event IOProcessEventHandler IOProgress;
        /// <summary>
        /// Micro PET main Header
        /// </summary>
        public MicroPETHeader microPETHeader;
        /// <summary>
        /// One or more subheaders.
        /// </summary>
        public MicroPETSubHeader[] microPETSubheaders;
        /// <summary>
        /// Creates and assings the basename to this class.
        /// </summary>
        /// <param name="filename">full path to filename</param>
        public MicroPETFile(string filename)
        {
            if (filename.EndsWith(".img"))
                this.filename = filename.Remove(filename.Length - 4);
            else
                this.filename = filename;
            base.filetype = FileType.MicroPET;
            base.header = new ImageHeader();
            base.image = null;
            microPETHeader = new MicroPETHeader();
            microPETSubheaders = new MicroPETSubHeader[1];
        }
        /// <summary>
        /// Constructs MicroPET file.
        /// </summary>
        /// <param name="filename">full name of MicroPET file</param>
        /// <param name="image">image data</param>
        /// <param name="header">image header</param>
        public MicroPETFile(string filename, Image image, ImageHeader header)
            : this(filename)
        {
            this.image = image;
            this.header = new DynamicImageHeader(header);
            this.header.datatype = DataType.FLT32;
            this.header.dim = image.dim;
            this.header.dataunit = header.dataunit;
            //create subheaders for gates and frames
            if (image.dim.Length > IntLimits.FRAMES && image.dim.GetDimension(IntLimits.FRAMES) > 1)
            {
                if (image.dim.Length > IntLimits.GATES && image.dim.GetDimension(IntLimits.GATES) > 1)
                    microPETSubheaders = new MicroPETSubHeader[image.dim.GetDimension(IntLimits.GATES) * image.dim.GetDimension(IntLimits.FRAMES)];
                else
                    microPETSubheaders = new MicroPETSubHeader[image.dim.GetDimension(IntLimits.FRAMES)];
            }
            else {
                microPETSubheaders = new MicroPETSubHeader[1];
            }
            float[] start_times = null;
            float[] durations = null;
            if (image is DynamicImage)
            {
                start_times = (image as DynamicImage).GetFrameStartTimes();
                durations = (image as DynamicImage).GetFrameDurations();
            }
            for (int i = 0; i < microPETSubheaders.Length; i++)
            {
                microPETSubheaders[i] = new MicroPETSubHeader();
                if (image is DynamicImage && i < start_times.Length)
                {
                    microPETSubheaders[i].frame_start = (start_times[i]/1000.0f);
                    microPETSubheaders[i].frame_duration = (durations[i]/1000.0f);
                }
                else {
                    microPETSubheaders[i].frame_start = 0;
                    microPETSubheaders[i].frame_duration = 0;
                }
            }
        }
        /// <summary>
        /// Constructs an empty (all-zero) Micro PET file with filename and dimensions.
        /// </summary>
        /// <param name="filename">filename (*.img)</param>
        /// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
        public MicroPETFile(string filename, params int[] dim)
        {
            if (filename.EndsWith(".img"))
                this.filename = filename.Remove(filename.Length - 4);
            else
                this.filename = filename;
            this.filetype = FileType.MicroPET;
            microPETHeader = new MicroPETHeader();
            if (dim.Length > IntLimits.FRAMES)
            {
                header = new DynamicImageHeader();
                (header as DynamicImageHeader).dim = new TPClib.IntLimits(dim);
                image = new DynamicImage(dim);
                microPETSubheaders = new MicroPETSubHeader[dim[IntLimits.FRAMES]];
            }
            else
            {
                header = new ImageHeader();
                header.dim = new TPClib.IntLimits(dim);
                image = new Image(dim);
                microPETSubheaders = new MicroPETSubHeader[1];
            }
            //micro PET data type default is 32-bit float
            header.datatype = TPClib.Image.ImageFile.DataType.FLT32;
        }
        /// <summary>
        /// Help routine for ReadFile
        /// </summary>
        /// <returns></returns>
        private delegate float GetValue();

        /// <summary>
        /// Reads the .img and .hdr from 
        /// the file to the data structures.
        /// </summary>
        public override void ReadFile()
        {
            //read header
            header = ReadHeader();
            //create image according to header information
            if (header is DynamicImageHeader)
            {
                image = new DynamicImage(header.dim);
                for (int i = 0; i < (header as DynamicImageHeader).dimt; i++)
                {
                    (image as DynamicImage).SetFrameStartTime(i, microPETSubheaders[i].frame_start * 1000.0f);
                    (image as DynamicImage).SetFrameDuration(i, microPETSubheaders[i].frame_duration * 1000.0f);
                }
            }
            else
            {
                image = new Image(header.dim);
            }
            //read pixel data
            GetPixelData(ref image, 0, filename + ".img", header.dim, header.datatype, 0, this.microPETHeader.calibration_factor);
        }

        /// <summary>
        /// Writes both the header and image data to the disk.
        /// </summary>
        /// <param name="image">image data that is written</param>
        public override void WriteFile(ref Image image)
        {
            //initialization for event sending
            IOProcessEventHandler pevent = null;
            //resolve frame length in voxels
            int framelength = (int)image.dim.GetProduct(IntLimits.PLANES);
            //index in subheaders array that currently has scale factor used
            int frame_i = 0;
            //currently used scale factor
            float scale = 1.0f;
            //resolve scaling factor according to datatype
            //NOTE: CT modality does not have scaling factor, so default 1.0 is used instead
            if (header.modality != ImageModality.M_CT)
            {
                #region resolve_scaling_factor(
                float min = image.GetMin();
                float max = image.GetMax();
                switch (header.datatype)
                {
                    case DataType.BIT8_S:
                        if (Math.Abs(min) <= Math.Abs(max))
                            scale = max / (SByte.MaxValue);
                        else
                            scale = min / (SByte.MinValue);
                        header["!maximum pixel count"] = SByte.MaxValue;
                        break;
                    case DataType.BIT8_U:
                        scale = max / Byte.MaxValue;
                        header["!maximum pixel count"] = Byte.MaxValue;
                        break;
                    case DataType.BIT16_S:
                        if (Math.Abs(min) <= Math.Abs(max))
                            scale = max / (Int16.MaxValue);
                        else
                            scale = min / (Int16.MinValue);
                        header["!maximum pixel count"] = Int16.MaxValue;
                        break;
                    case DataType.BIT16_U:
                        scale = max / UInt16.MaxValue;
                        header["!maximum pixel count"] = UInt16.MaxValue;
                        break;
                    case DataType.FLT32:
                    case DataType.FLT64:
                    case DataType.VAXFL32:
                        break;
                    default:
                        throw new TPCInterfileFileException("Unsupported datatype " + header.datatype.ToString() + " for resolving scale factor.");
                }
                #endregion
            }
            microPETHeader.calibration_factor = scale;
            if (microPETSubheaders.Length > 0)
            {
                for (int i = 0; i < microPETSubheaders.Length; i++) {
                    microPETSubheaders[i].scale_factor = scale;
                }
            }
            //write header data
            WriteHeader();
            
            //try to open file for writing
            BinaryWriter writer;
            try
            {
                FileStream stream = File.Open(filename+".img", FileMode.Create);
                writer = new BinaryWriter(stream);
            }
            catch (Exception e)
            {
                throw new TPCMicroPETFileException("Could not open file [" + filename + ".img" + "] for writing:" + e);
            }
            //resolve voxels in plane for sending event after writing each plane
            int voxels_in_plane = image.dim.dimx * image.dim.dimy;
            //write data according to data type
            switch (this.microPETHeader.data_type)
            {
                //32-bit IEEE float
                case MicroPETHeader.Data_type.Intel_4byte_Float:
                    {
                        if (this.microPETHeader.data_order == MicroPETHeader.Data_order.View)
                            throw new TPCMicroPETFileException("Only XYZW data order is currently supported for writing.");
                        for (int i = 0; i < image.dataLength; i++)
                        {
                            //switch to use next scaling factor if framelength met
                            if (i % framelength == 0)
                            {
                                scale = microPETSubheaders[frame_i++].scale_factor;
                            }
                            writer.Write(image[i]);
                            //send event after each plane
                            if (i > 0 && i % voxels_in_plane == 0)
                            {
                                pevent = IOProgress;
                                if (pevent != null)
                                    pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)image.dataLength), System.Reflection.MethodBase.GetCurrentMethod()));
                            }
                        }
                        break;
                    }
                //16-bit signed integer
                case MicroPETHeader.Data_type.Intel_2byte_Int:
                    {
                        if (this.microPETHeader.data_order == MicroPETHeader.Data_order.View)
                            throw new TPCMicroPETFileException("Only XYZW data order is currently supported for writing.");
                        //frame minimum and maximum
                        float[] mm;
                        for (int i = 0; i < image.dataLength; i++)
                        {
                            //switch to use next scaling factor if framelength met
                            if (i % framelength == 0)
                            {
                                if (image is DynamicImage)
                                    mm = image.GetMinMax((image as DynamicImage).GetFrameLimits(frame_i));
                                scale = microPETSubheaders[frame_i++].scale_factor;
                            }
                            writer.Write((Int16)(image[i]/scale));
                            //send event after each plane
                            if (i > 0 && i % voxels_in_plane == 0)
                            {
                                pevent = IOProgress;
                                if (pevent != null)
                                    pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)image.dataLength), System.Reflection.MethodBase.GetCurrentMethod()));
                            }
                        }
                        break;
                    }
                default:
                    throw new TPCMicroPETFileException("Unsupported data type in MicroPET for writing: " + microPETHeader.data_type);
            }
            writer.Close();
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
        }
        /// <summary>
        /// Writes main header into stream. The written data is read from microPETheader variable.
        /// </summary>
        /// <param name="writer">stream where the data is written</param>
        /// <param name="microPETHeader">header that is written</param>
        public static void WriteMicroPETMainheader(TextWriter writer, MicroPETHeader microPETHeader)
        {
            #region writing_of_data_line_by_line
            writer.WriteLine("#");
            writer.WriteLine("# Header file for data file " + microPETHeader.file_name);
            writer.WriteLine("#    with "+microPETHeader.time_frames+" frames.");
            writer.WriteLine("#");
            writer.WriteLine("#");
            writer.WriteLine("#  Following parameters are used for ALL modalities.");
            writer.WriteLine("#");
            writer.WriteLine("#");
            writer.WriteLine("# Version of header parameters (float)");
            writer.WriteLine("#");
            writer.WriteLine("version "+microPETHeader.version.ToString("000.000").Replace(',','.'));
            writer.WriteLine("#");
            writer.WriteLine("# Manufacturer's name (string)");
            writer.WriteLine("#");
            writer.WriteLine("manufacturer "+microPETHeader.manufacturer);
            writer.WriteLine("#");
            writer.WriteLine("# Scanner model number (integer)");
            writer.WriteLine("#");
            writer.WriteLine("#   0 - Unknown");
            writer.WriteLine("#   2000 - Primate");
            writer.WriteLine("#   2001 - Rodent");
            writer.WriteLine("#   2002 - microPET2");
            writer.WriteLine("#   2500 - Focus_220");
            writer.WriteLine("#   2501 - Focus_120");
            writer.WriteLine("#   3000 - mCAT");
            writer.WriteLine("#   3500 - mCATII");
            writer.WriteLine("#   4000 - mSPECT");
            writer.WriteLine("#   5000 - Inveon_Dedicated_PET");
            writer.WriteLine("#   5001 - Inveon_MM_Platform");
            writer.WriteLine("#   6000 - MR_PET_Head_Insert");
            writer.WriteLine("#   8000 - Tuebingen_PET_MR");
            writer.WriteLine("#");
            writer.WriteLine("model "+(int)microPETHeader.model);
            writer.WriteLine("#");
            writer.WriteLine("# Acquisition scanner modality (integer)");
            writer.WriteLine("#   -1 - Unknown acquisition modality");
            writer.WriteLine("#   0 - PET acquisition");
            writer.WriteLine("#   1 - CT acquisition");
            writer.WriteLine("#   2 - SPECT acquisition");
            writer.WriteLine("#");
            writer.WriteLine("modality " + (int)microPETHeader.modality);
            writer.WriteLine("#");
            writer.WriteLine("# Scanner modality configuration number (integer)");
            writer.WriteLine("#");
            writer.WriteLine("#   0 - Unknown");
            writer.WriteLine("#   2000 - Primate");
            writer.WriteLine("#   2001 - Rodent");
            writer.WriteLine("#   2002 - microPET2");
            writer.WriteLine("#   2500 - Focus_220");
            writer.WriteLine("#   2501 - Focus_120");
            writer.WriteLine("#   5000 - Inveon_Dedicated_PET");
            writer.WriteLine("#   6000 - MR_PET_Head_Insert");
            writer.WriteLine("#   8000 - Tuebingen_PET_MR");
            writer.WriteLine("#   5500 - Inveon_MM_PET");
            writer.WriteLine("#");
            writer.WriteLine("modality_configuration " + (int)microPETHeader.modality_configuration);
            writer.WriteLine("#");
            writer.WriteLine("# Institution identification (string)");
            writer.WriteLine("#");
            writer.WriteLine("institution "+microPETHeader.institution);
            writer.WriteLine("#");
            writer.WriteLine("# Study type/description (string)");
            writer.WriteLine("#");
            writer.WriteLine("study "+microPETHeader.study);
            writer.WriteLine("#");
            writer.WriteLine("# Data filename, possibly including path (string)");
            writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
            writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
            writer.WriteLine("#");
            writer.WriteLine("file_name "+microPETHeader.file_name);
            writer.WriteLine("#");
            writer.WriteLine("# Data file type (integer)");
            writer.WriteLine("#   0 - Unknown data file type");
            writer.WriteLine("#   1 - List mode data file");
            writer.WriteLine("#   2 - Sinogram data file");
            writer.WriteLine("#   3 - Normalization data file");
            writer.WriteLine("#   4 - Attenuation correction data file");
            writer.WriteLine("#   5 - Image data file");
            writer.WriteLine("#   6 - Blank data file");
            writer.WriteLine("#   8 - Mu map data file");
            writer.WriteLine("#   9 - Scatter correction data file");
            writer.WriteLine("#   10 - Crystal efficiency data");
            writer.WriteLine("#   11 - Crystal interference correction");
            writer.WriteLine("#   12 - Transaxial geometric correction");
            writer.WriteLine("#   13 - Axial geometric correction");
            writer.WriteLine("#   14 - CT projection data");
            writer.WriteLine("#   15 - SPECT raw projection data");
            writer.WriteLine("#   16 - SPECT energy data from projections");
            writer.WriteLine("#   17 - SPECT normalization data");
            writer.WriteLine("#");
            writer.WriteLine("file_type " + (int)microPETHeader.file_type);
            writer.WriteLine("#");
            writer.WriteLine("# Acquisition mode (integer)");
            writer.WriteLine("#   0 - Unknown acquisition mode");
            writer.WriteLine("#   1 - Blank acquisition");
            writer.WriteLine("#   2 - Emission acquisition");
            writer.WriteLine("#   3 - Dynamic acquisition");
            writer.WriteLine("#   4 - Gated acquisition");
            writer.WriteLine("#   5 - Continuous bed motion acquisition");
            writer.WriteLine("#   6 - Singles transmission acquisition");
            writer.WriteLine("#   7 - Windowed coincidence transmission acquisition");
            writer.WriteLine("#   8 - Non-windowed coincidence transmission acquisition");
            writer.WriteLine("#   9 - CT projection acquisition");
            writer.WriteLine("#   10 - CT calibration acquisition");
            writer.WriteLine("#   11 - SPECT planar projection acquisition");
            writer.WriteLine("#   12 - SPECT multi-projection acquisition");
            writer.WriteLine("#   13 - SPECT calibration acquisition");
            writer.WriteLine("#   14 - SPECT normalization acquisition");
            writer.WriteLine("#   15 - SPECT detector setup acquisition");
            writer.WriteLine("#   16 - SPECT scout view acquisition");
            writer.WriteLine("#");
            writer.WriteLine("acquisition_mode " + (int)microPETHeader.acquisition_mode);
            writer.WriteLine("#");
            writer.WriteLine("# Number of frames in data file (integer)");
            writer.WriteLine("#");
            writer.WriteLine("total_frames "+microPETHeader.total_frames);
            if (microPETHeader.modality != MicroPETHeader.scanner_modality.CT)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Number of time frames in data file (integer)");
                writer.WriteLine("#");
                writer.WriteLine("time_frames " + microPETHeader.time_frames);
            }
            writer.WriteLine("#");
            writer.WriteLine("# Bed control (integer)");
            writer.WriteLine("#   0 - Unknown bed control");
            writer.WriteLine("#   1 - Dedicated PET");
            writer.WriteLine("#   2 - microCAT II");
            writer.WriteLine("#   3 - Multimodality bed control");
            writer.WriteLine("#   4 - microPET bed control");
            writer.WriteLine("#");
            writer.WriteLine("bed_control " + (int)microPETHeader.bed_control);
            writer.WriteLine("#");
            writer.WriteLine("# Bed motion (integer)");
            writer.WriteLine("#   0 - Static or unknown bed motion");
            writer.WriteLine("#   1 - Continuous bed motion");
            writer.WriteLine("#   2 - Multiple bed positions, i.e. step and shoot");
            writer.WriteLine("#");
            writer.WriteLine("bed_motion " + (int)microPETHeader.bed_motion);
            writer.WriteLine("#");
            writer.WriteLine("# Number of bed positions in data file (integer)");
            writer.WriteLine("#");
            writer.WriteLine("number_of_bed_positions "+microPETHeader.number_of_bed_positions);
            writer.WriteLine("#");
            writer.WriteLine("# Horizontal bed calibration, in microns (float)");
            writer.WriteLine("#");
            writer.WriteLine("horizontal_bed_calibration " + microPETHeader.horizontal_bed_calibration.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Vertical bed calibration, in microns (float)");
            writer.WriteLine("#");
            writer.WriteLine("vertical_bed_calibration " + microPETHeader.vertical_bed_calibration.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Transaxial projection bin size, in cm (float)");
            writer.WriteLine("#");
            writer.WriteLine("transaxial_bin_size "+microPETHeader.transaxial_bin_size.ToString().Replace(',','.'));
            writer.WriteLine("#");
            writer.WriteLine("# Axial plane size, in cm (float)");
            writer.WriteLine("#");
            writer.WriteLine("axial_plane_size " + microPETHeader.axial_plane_size.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Number of detector panels (\"rings/CT\" are \"1\") (integer)");
            writer.WriteLine("#");
            writer.WriteLine("number_detector_panels "+microPETHeader.number_detector_panels);
            writer.WriteLine("#");
            writer.WriteLine("# Data type (integer)");
            writer.WriteLine("#   0 - Unknown data type");
            writer.WriteLine("#   1 - Byte (8-bits) data type");
            writer.WriteLine("#   2 - 2-byte integer - Intel style");
            writer.WriteLine("#   3 - 4-byte integer - Intel style");
            writer.WriteLine("#   4 - 4-byte float - Intel style");
            writer.WriteLine("#   5 - 4-byte float - Sun style");
            writer.WriteLine("#   6 - 2-byte integer - Sun style");
            writer.WriteLine("#   7 - 4-byte integer - Sun style");
            writer.WriteLine("#");
            writer.WriteLine("data_type " + (int)microPETHeader.data_type);
            writer.WriteLine("#");
            writer.WriteLine("# Number of dimensions in data set (integer)");
            writer.WriteLine("#     Order from fastest to slowest is XYZW");
            writer.WriteLine("#");
            writer.WriteLine("number_of_dimensions "+microPETHeader.number_of_dimensions);
            writer.WriteLine("#");
            writer.WriteLine("# Size of X dimension in data set (integer)");
            writer.WriteLine("#");
            writer.WriteLine("x_dimension "+microPETHeader.x_dimension);
            writer.WriteLine("#");
            writer.WriteLine("# Size of Y dimension in data set (integer)");
            writer.WriteLine("#");
            writer.WriteLine("y_dimension " + microPETHeader.y_dimension);
            writer.WriteLine("#");
            writer.WriteLine("# Size of Z dimension in data set (integer)");
            writer.WriteLine("#");
            writer.WriteLine("z_dimension " + microPETHeader.z_dimension);
            writer.WriteLine("#");
            writer.WriteLine("# Acquisition user ID (string)");
            writer.WriteLine("acquisition_user_id "+microPETHeader.acquisition_user_id);
            writer.WriteLine("#");
            writer.WriteLine("# Histogram user ID (string)");
            writer.WriteLine("histogram_user_id "+microPETHeader.histogram_user_id);
            writer.WriteLine("#");
            writer.WriteLine("# Reconstruction user ID (string)");
            writer.WriteLine("reconstruction_user_id "+microPETHeader.reconstruction_user_id);
            writer.WriteLine("#");
            writer.WriteLine("# Scatter correction user ID (string)");
            writer.WriteLine("scatter_correction_user_id "+microPETHeader.scatter_correction_user_id);
            writer.WriteLine("#");
            writer.WriteLine("# Acquisition notes (string)");
            writer.WriteLine("acquisition_notes "+microPETHeader.acquisition_notes);
            writer.WriteLine("#");
            writer.WriteLine("# Scan start date and time (string)");
            writer.WriteLine("#    Format is: Sun Sep 16 01:03:52 1973");
            writer.WriteLine("#");
            writer.WriteLine("scan_time "+microPETHeader.scan_time);
            writer.WriteLine("#");
            writer.WriteLine("# Scan start date and time - GMT-based (string)");
            writer.WriteLine("#    Format is: Sun Sep 16 01:03:52 1973");
            writer.WriteLine("#");
            writer.WriteLine("gmt_scan_time "+microPETHeader.gmt_scan_time);
            writer.WriteLine("#");
            writer.WriteLine("# X origin of volume, in voxels (integer)");
            writer.WriteLine("#");
            writer.WriteLine("volume_origin_x "+microPETHeader.volume_origin_x);
            writer.WriteLine("#");
            writer.WriteLine("# Y origin of volume, in voxels (integer)");
            writer.WriteLine("#");
            writer.WriteLine("volume_origin_y "+microPETHeader.volume_origin_y);
            writer.WriteLine("#");
            writer.WriteLine("# Z origin of volume, in voxels (integer)");
            writer.WriteLine("#");
            writer.WriteLine("volume_origin_z "+microPETHeader.volume_origin_z);
            writer.WriteLine("#");
            writer.WriteLine("# Registration data available (integer)");
            writer.WriteLine("#   0 - No registration data available");
            writer.WriteLine("#   1 - CT registration data available");
            writer.WriteLine("#   2 - PET registration data available");
            writer.WriteLine("#");
            writer.WriteLine("registration_available " + (int)microPETHeader.registration_available);
            writer.WriteLine("#");
            writer.WriteLine("# Transformation matrix filename, possibly including path (string)");
            writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
            writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
            writer.WriteLine("#");
            writer.WriteLine("transformation_matrix "+microPETHeader.transformation_matrix);
            writer.WriteLine("#");
            writer.WriteLine("# Spatial identification for registration (string)");
            writer.WriteLine("#");
            writer.WriteLine("spatial_identifier "+microPETHeader.spatial_identifier);
            writer.WriteLine("#");
            writer.WriteLine("# Normalization applied to data set (integer)");
            writer.WriteLine("#   0 - No normalization applied");
            writer.WriteLine("#   1 - Point source inversion");
            writer.WriteLine("#   2 - Point source component based");
            writer.WriteLine("#   3 - Cylinder source inversion");
            writer.WriteLine("#   4 - Cylinder source component based");
            writer.WriteLine("#   5 - Dark/bright field log normalization (CT)");
            writer.WriteLine("#   6 - SPECT flood inversion based");
            writer.WriteLine("#");
            writer.WriteLine("normalization_applied " + (int)microPETHeader.normalization_applied);
            if (microPETHeader.modality != MicroPETHeader.scanner_modality.CT)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Normalization filename, possibly including path (string)");
                writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
                writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
                writer.WriteLine("#");
                writer.WriteLine("normalization_filename " + microPETHeader.normalization_filename);
            }
            writer.WriteLine("#");
            writer.WriteLine("# Reconstruction type (integer)");
            writer.WriteLine("#   0 - Unknown, or no, algorithm type");
            writer.WriteLine("#   1 - Filtered Backprojection");
            writer.WriteLine("#   2 - OSEM2d");
            writer.WriteLine("#   3 - OSEM3d");
            writer.WriteLine("#   6 - OSEM3D followed by MAP or FastMAP");
            writer.WriteLine("#   7 - MAPTR for transmission image");
            writer.WriteLine("#   8 - MAP 3D reconstruction");
            writer.WriteLine("#   9 - Feldkamp cone beam");
            writer.WriteLine("#");
            writer.WriteLine("recon_algorithm " + (int)microPETHeader.recon_algorithm);
            writer.WriteLine("#");
            writer.WriteLine("# Version of reconstruction program used (float)");
            writer.WriteLine("#");
            writer.WriteLine("recon_version " + microPETHeader.recon_version.ToString("000.000").Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# X filter and/or apodizing windows type (integer)");
            writer.WriteLine("#       and cutoff (float)");
            writer.WriteLine("#   0 - No filter");
            writer.WriteLine("#   1 - Ramp filter (backprojection) or no filter");
            writer.WriteLine("#   2 - First-order Butterworth window");
            writer.WriteLine("#   3 - Hanning window");
            writer.WriteLine("#   4 - Hamming window");
            writer.WriteLine("#   5 - Parzen window");
            writer.WriteLine("#   6 - Shepp filter");
            writer.WriteLine("#   7 - Second-order Butterworth window");
            writer.WriteLine("#   NOTE that a cutoff of 0.5 is the Nyquist point");
            writer.WriteLine("#     i.e 1.0 / (2.0 * sampling).");
            writer.WriteLine("#     Also, the Ramp and Shepp should ONLY be used");
            writer.WriteLine("#     for backprojection");
            writer.WriteLine("#");
            writer.WriteLine("x_filter " + (int)microPETHeader.x_filter_type + " " + microPETHeader.x_filter_cutoff.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Y apodizing filter type (integer) and cutoff (float)");
            writer.WriteLine("#   0 - No filter");
            writer.WriteLine("#   2 - First-order Butterworth window");
            writer.WriteLine("#   3 - Hanning window");
            writer.WriteLine("#   4 - Hamming window");
            writer.WriteLine("#   5 - Parzen window");
            writer.WriteLine("#   7 - Second-order Butterworth window");
            writer.WriteLine("#   NOTE that a cutoff of 0.5 is the Nyquist point");
            writer.WriteLine("#     i.e 1.0 / (2.0 * sampling).");
            writer.WriteLine("#");
            writer.WriteLine("y_filter " + (int)microPETHeader.y_filter_type + " " + microPETHeader.y_filter_cutoff.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Z apodizing filter type (integer) and cutoff (float)");
            writer.WriteLine("#   0 - No filter");
            writer.WriteLine("#   2 - First-order Butterworth window");
            writer.WriteLine("#   3 - Hanning window");
            writer.WriteLine("#   4 - Hamming window");
            writer.WriteLine("#   5 - Parzen window");
            writer.WriteLine("#   7 - Second-order Butterworth window");
            writer.WriteLine("#   NOTE that a cutoff of 0.5 is the Nyquist point");
            writer.WriteLine("#     i.e 1.0 / (2.0 * sampling).");
            writer.WriteLine("#");
            writer.WriteLine("z_filter " + (int)microPETHeader.z_filter_type + " " + microPETHeader.z_filter_cutoff.ToString().Replace(',', '.'));
            if (microPETHeader is MicroPETHeader_CTspecific) {
                writer.WriteLine("#");
                writer.WriteLine("# Rotation, in degrees, applied to data set (not applied to OSED2d reconstructions) (float)");
                writer.WriteLine("#");
                writer.WriteLine("rotation " + (microPETHeader as MicroPETHeader_CTspecific).rotation);
            }
            writer.WriteLine("#");
            writer.WriteLine("# X offset, in cm, applied to data set (float)");
            writer.WriteLine("#");
            writer.WriteLine("x_offset " + microPETHeader.x_offset.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Y offset, in cm, applied to data set (float)");
            writer.WriteLine("#");
            writer.WriteLine("y_offset " + microPETHeader.y_offset.ToString().Replace(',', '.'));
            if (microPETHeader is MicroPETHeader_CTspecific)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Z offset, in cm, applied to data set (float)");
                writer.WriteLine("#");
                writer.WriteLine("z_offset " + (microPETHeader as MicroPETHeader_CTspecific).z_offset.ToString().Replace(',', '.'));
            }
            writer.WriteLine("#");
            writer.WriteLine("# Zoom applied to data set (float)");
            writer.WriteLine("#");
            writer.WriteLine("zoom " + microPETHeader.zoom.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Reconstructed pixel size, in cm (float)");
            writer.WriteLine("#     NOTE: pixel_size = (((X_crystal_pitch / 2.0) * X_dim) /");
            writer.WriteLine("#           (image_size * zoom)) * (effective_radius / radius)");
            writer.WriteLine("#");
            writer.WriteLine("pixel_size " + microPETHeader.pixel_size.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Reconstructed pixel size in X, in mm (float)");
            writer.WriteLine("pixel_size_x " + microPETHeader.pixel_size_x.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Reconstructed pixel size in Y, in mm (float)");
            writer.WriteLine("pixel_size_y " + microPETHeader.pixel_size_y.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Reconstructed pixel size in Z, in mm (float)");
            writer.WriteLine("pixel_size_z " + microPETHeader.pixel_size_z.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Investigator identification (string)");
            writer.WriteLine("#");
            writer.WriteLine("investigator "+microPETHeader.investigator_id);
            writer.WriteLine("#");
            writer.WriteLine("# Operator identification (string)");
            writer.WriteLine("#");
            writer.WriteLine("operator "+microPETHeader.operator_id);
            writer.WriteLine("#");
            writer.WriteLine("# Study identification (string)");
            writer.WriteLine("#");
            writer.WriteLine("study_identifier "+microPETHeader.study_identifier);
            writer.WriteLine("#");
            writer.WriteLine("# Injected compound (string)");
            writer.WriteLine("#");
            writer.WriteLine("injected_compound "+microPETHeader.injected_compound);
            writer.WriteLine("#");
            writer.WriteLine("# Subject identifier (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_identifier "+microPETHeader.subject_identifier);
            writer.WriteLine("#");
            writer.WriteLine("# Subject genus (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_genus "+microPETHeader.subject_genus);
            writer.WriteLine("#");
            writer.WriteLine("# Subject orientation (integer)");
            writer.WriteLine("#   0 - Unknown subject orientation");
            writer.WriteLine("#   1 - Feet first, prone");
            writer.WriteLine("#   2 - Head first, prone");
            writer.WriteLine("#   3 - Feet first, supine");
            writer.WriteLine("#   4 - Head first, supine");
            writer.WriteLine("#   5 - Feet first, right");
            writer.WriteLine("#   6 - Head first, right");
            writer.WriteLine("#   7 - Feet first, left");
            writer.WriteLine("#   8 - Head first, left");
            writer.WriteLine("#");
            writer.WriteLine("subject_orientation " + (int)microPETHeader.subject_orientation);
            writer.WriteLine("#");
            writer.WriteLine("# Length units (integer)");
            writer.WriteLine("#   0 - Unknown length units");
            writer.WriteLine("#   1 - millimeters");
            writer.WriteLine("#   2 - centimeters");
            writer.WriteLine("#   3 - inches");
            writer.WriteLine("#");
            writer.WriteLine("subject_length_units " + (int)microPETHeader.subject_length_units);
            writer.WriteLine("#");
            writer.WriteLine("# Subject weight (float)");
            writer.WriteLine("#");
            writer.WriteLine("subject_length " + microPETHeader.subject_length.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Weight units (integer)");
            writer.WriteLine("#   0 - Unknown weight units");
            writer.WriteLine("#   1 - grams");
            writer.WriteLine("#   2 - ounces");
            writer.WriteLine("#   3 - kilograms");
            writer.WriteLine("#   4 - pounds");
            writer.WriteLine("#");
            writer.WriteLine("subject_weight_units " + (int)microPETHeader.subject_weight_units);
            writer.WriteLine("#");
            writer.WriteLine("# Subject weight (float)");
            writer.WriteLine("#");
            writer.WriteLine("subject_weight " + microPETHeader.subject_weight.ToString().Replace(',', '.'));
            writer.WriteLine("#");
            writer.WriteLine("# Subject phenotype (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_phenotype "+microPETHeader.subject_phenotype);
            writer.WriteLine("#");
            writer.WriteLine("# Study model (string)");
            writer.WriteLine("#");
            writer.WriteLine("study_model "+microPETHeader.study_model);
            writer.WriteLine("#");
            writer.WriteLine("# Subject anesthesia (string)");
            writer.WriteLine("#");
            writer.WriteLine("anesthesia "+microPETHeader.anesthesia);
            writer.WriteLine("#");
            writer.WriteLine("# Subject analgesia (string)");
            writer.WriteLine("#");
            writer.WriteLine("analgesia "+microPETHeader.analgesia);
            writer.WriteLine("#");
            writer.WriteLine("# Other drugs (string)");
            writer.WriteLine("#");
            writer.WriteLine("other_drugs "+microPETHeader.other_drugs);
            writer.WriteLine("#");
            writer.WriteLine("# Food access (string)");
            writer.WriteLine("#");
            writer.WriteLine("food_access "+microPETHeader.food_access);
            writer.WriteLine("#");
            writer.WriteLine("# Water access (string)");
            writer.WriteLine("#");
            writer.WriteLine("water_access "+microPETHeader.water_access);
            writer.WriteLine("#");
            writer.WriteLine("# Subject date of birth (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_date_of_birth "+microPETHeader.subject_date_of_birth);
            writer.WriteLine("#");
            writer.WriteLine("# Subject age (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_age "+microPETHeader.subject_age);
            writer.WriteLine("#");
            writer.WriteLine("# Subject sex (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_sex "+microPETHeader.subject_sex);
            writer.WriteLine("#");
            writer.WriteLine("# Subject scan region (string)");
            writer.WriteLine("#");
            writer.WriteLine("subject_scan_region "+microPETHeader.subject_scan_region);
            writer.WriteLine("#");
            writer.WriteLine("#  Following parameters are used for PET or CT modalities.");
            writer.WriteLine("#");
            writer.WriteLine("#");
            writer.WriteLine("# Original acquisition filename, DOES NOT INCLUDE path (string)");
            writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
            writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
            writer.WriteLine("#");
            writer.WriteLine("acquisition_file_name " + microPETHeader.acquisition_file_name);
            #region non_CT_headers
            if (microPETHeader.modality != MicroPETHeader.scanner_modality.CT)
            {
                writer.WriteLine("#");
                writer.WriteLine("#  Following parameters are used for PET or SPECT modalities.");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("# Transaxial crystal pitch, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("transaxial_crystal_pitch " + microPETHeader.transaxial_crystal_pitch.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Axial crystal pitch, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("axial_crystal_pitch " + microPETHeader.axial_crystal_pitch.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Crystal thickness, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("crystal_thickness " + microPETHeader.crystal_thickness.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Depth of interaction, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("depth_of_interaction " + microPETHeader.depth_of_interaction.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Isotope description (string)");
                writer.WriteLine("#");
                writer.WriteLine("isotope " + microPETHeader.isotope);
                writer.WriteLine("#");
                writer.WriteLine("# Isotope half-life, in secs (float)");
                writer.WriteLine("#");
                writer.WriteLine("isotope_half_life " + microPETHeader.isotope_half_life.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Isotope branching fraction (float)");
                writer.WriteLine("#   NOTE: Frame scale factor DOES NOT include");
                writer.WriteLine("#         isotope branching fraction.");
                writer.WriteLine("#");
                writer.WriteLine("isotope_branching_fraction " + microPETHeader.isotope_branching_fraction.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Deadtime correction applied to data set (integer)");
                writer.WriteLine("#   0 - No deadtime correction applied");
                writer.WriteLine("#   1 - Global estimate based on singles");
                writer.WriteLine("#   2 - CMS estimate based on singles");
                writer.WriteLine("#   3 - Global estimate based on running deadtime average");
                writer.WriteLine("#   4 - Blank/TX singles estimate (block based)");
                writer.WriteLine("#");
                writer.WriteLine("deadtime_correction_applied " + (int)microPETHeader.deadtime_correction_applied);
                writer.WriteLine("#");
                writer.WriteLine("# Decay correction applied to data set (integer)");
                writer.WriteLine("#     0 (FALSE) - Decay correction has NOT been applied");
                writer.WriteLine("#     !0 (TRUE) - Decay correction has been applied");
                writer.WriteLine("#");
                writer.WriteLine("decay_correction_applied " + (microPETHeader.decay_correction_applied ? 1 : 0));
                writer.WriteLine("#");
                writer.WriteLine("# Attenuation applied to data set (integer)");
                writer.WriteLine("#   0 - No attenuation applied");
                writer.WriteLine("#   1 - Point source in windowed TX coincidence");
                writer.WriteLine("#   2 - Point source singles based TX");
                writer.WriteLine("#   3 - Segmented point source in TX coincidence");
                writer.WriteLine("#   4 - Segmented point source singles based TX");
                writer.WriteLine("#   5 - Calculated by geometry");
                writer.WriteLine("#   6 - Non-positron source singles based TX");
                writer.WriteLine("#   7 - Point source in non-windowed TX coincidence");
                writer.WriteLine("#");
                writer.WriteLine("attenuation_applied " + (int)microPETHeader.attenuation_applied);
                writer.WriteLine("#");
                writer.WriteLine("# Attenuation correction filename, possibly including path (string)");
                writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
                writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
                writer.WriteLine("#");
                writer.WriteLine("attenuation_filename " + microPETHeader.attenuation_filename);
                writer.WriteLine("#");
                writer.WriteLine("# Scatter correction applied to data set (integer)");
                writer.WriteLine("#   0 - No scatter correction applied");
                writer.WriteLine("#   1 - Fit of emission tail");
                writer.WriteLine("#   2 - Monte Carlo of emission and transmission data");
                writer.WriteLine("#   3 - Direct calculation from analytical formulas");
                writer.WriteLine("#   4 - Model-based scatter for singles TX");
                writer.WriteLine("#   5 - TX off-window windowed coincidence subtraction");
                writer.WriteLine("#   6 - Singles TX scaled scatter from attenuation subtraction");
                writer.WriteLine("#");
                writer.WriteLine("scatter_correction " + (int)microPETHeader.scatter_correction);
                writer.WriteLine("#");
                writer.WriteLine("# Calibration units (integer)");
                writer.WriteLine("#   0 - Unknown calibration units");
                writer.WriteLine("#   1 - nCi/cc");
                writer.WriteLine("#   2 - Bq/cc");
                writer.WriteLine("#");
                writer.WriteLine("calibration_units " + (int)microPETHeader.calibration_units);
                writer.WriteLine("#");
                writer.WriteLine("# Calibration factor (float)");
                writer.WriteLine("#   NOTE: Frame scale factor DOES NOT include");
                writer.WriteLine("#         calibration factor.");
                writer.WriteLine("#");
                writer.WriteLine("calibration_factor " + microPETHeader.calibration_factor.ToString("0.00000e+000").Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Calibration source branching fraction (float)");
                writer.WriteLine("#   NOTE: Frame scale factor DOES NOT include");
                writer.WriteLine("#         calibration source branching fraction.");
                writer.WriteLine("#");
                writer.WriteLine("calibration_branching_fraction " + microPETHeader.calibration_branching_fraction.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Dose units (integer)");
                writer.WriteLine("#   0 - Unknown dose units");
                writer.WriteLine("#   1 - mCi");
                writer.WriteLine("#   2 - MBq");
                writer.WriteLine("#");
                writer.WriteLine("dose_units " + (int)microPETHeader.dose_units);
                writer.WriteLine("#");
                writer.WriteLine("# Injected dose (float)");
                writer.WriteLine("dose " + microPETHeader.dose.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Injection date and time (string)");
                writer.WriteLine("#    Format is: Sun Sep 16 01:03:52 1973");
                writer.WriteLine("#");
                writer.WriteLine("injection_time " + microPETHeader.injection_time);
                writer.WriteLine("#");
                writer.WriteLine("# Injection decay correction factor (float)");
                writer.WriteLine("#   NOTE: Frame scale factor and decay correction factor");
                writer.WriteLine("#   DO NOT include injection decay correction factor.");
                writer.WriteLine("#");
                writer.WriteLine("injection_decay_correction " + microPETHeader.injection_decay_correction.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Pre- and residual activity units (integer)");
                writer.WriteLine("#   0 - Unknown dose units");
                writer.WriteLine("#   1 - mCi");
                writer.WriteLine("#   2 - MBq");
                writer.WriteLine("#");
                writer.WriteLine("activity_units " + (int)microPETHeader.activity_units);
                writer.WriteLine("#");
                writer.WriteLine("# Activity before injection (float)");
                writer.WriteLine("activity_before_injection 27.1");
                writer.WriteLine("#");
                writer.WriteLine("# Activity before injection measurement date and time (string)");
                writer.WriteLine("#    Format is: Sun Sep 16 01:03:52 1973");
                writer.WriteLine("#");
                writer.WriteLine("activity_before_injection_time " + microPETHeader.activity_before_injection_time);
                writer.WriteLine("#");
                writer.WriteLine("# Residual activity (float)");
                writer.WriteLine("residual_activity " + microPETHeader.residual_activity.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Residual activity measurement date and time (string)");
                writer.WriteLine("#    Format is: Sun Sep 16 01:03:52 1973");
                writer.WriteLine("#");
                writer.WriteLine("residual_activity_time " + microPETHeader.residual_activity_time);
                writer.WriteLine("#");
                writer.WriteLine("#  Following parameters are used for PET modality only.");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("# Transaxial crystals per block (integer)");
                writer.WriteLine("#");
                writer.WriteLine("transaxial_crystals_per_block " + microPETHeader.transaxial_crystals_per_block);
                writer.WriteLine("#");
                writer.WriteLine("# Axial crystals per block (integer)");
                writer.WriteLine("#");
                writer.WriteLine("axial_crystals_per_block " + microPETHeader.axial_crystals_per_block);
                writer.WriteLine("#");
                writer.WriteLine("# Crystal offset for intrinsic rotation (integer)");
                writer.WriteLine("#");
                writer.WriteLine("intrinsic_crystal_offset " + microPETHeader.intrinsic_crystal_offset);
                writer.WriteLine("#");
                writer.WriteLine("# Number of transaxial blocks (integer)");
                writer.WriteLine("#");
                writer.WriteLine("transaxial_blocks " + microPETHeader.transaxial_blocks);
                writer.WriteLine("#");
                writer.WriteLine("# Number of axial blocks (integer)");
                writer.WriteLine("#");
                writer.WriteLine("axial_blocks " + microPETHeader.axial_blocks);
                writer.WriteLine("#");
                writer.WriteLine("# Ring radius to crystal face, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("radius " + microPETHeader.radius.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Radial field-of-view, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("radial_fov " + microPETHeader.radial_fov.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Source radius, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("src_radius " + microPETHeader.src_radius.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Source axial cm per revolution, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("src_cm_per_rev " + microPETHeader.src_cm_per_rev.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Source encoder steps per revolution (integer)");
                writer.WriteLine("#");
                writer.WriteLine("src_steps_per_rev " + microPETHeader.src_steps_per_rev);
                writer.WriteLine("#");
                writer.WriteLine("# Default number of projections (integer)");
                writer.WriteLine("#");
                writer.WriteLine("default_projections " + microPETHeader.default_projections);
                writer.WriteLine("#");
                writer.WriteLine("# Default number of transaxial angles (integer)");
                writer.WriteLine("#");
                writer.WriteLine("default_transaxial_angles " + microPETHeader.default_transaxial_angles);
                writer.WriteLine("#");
                writer.WriteLine("# Lower level energy threshold, in KeV (float)");
                writer.WriteLine("#");
                writer.WriteLine("lld " + microPETHeader.lld.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Upper level energy threshold, in KeV (float)");
                writer.WriteLine("#");
                writer.WriteLine("uld " + microPETHeader.uld.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Coincidence timing window, in nsecs (int or float)");
                writer.WriteLine("#");
                writer.WriteLine("timing_window " + microPETHeader.timing_window.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Span of data set (integer)");
                writer.WriteLine("#");
                writer.WriteLine("span " + microPETHeader.span);
                writer.WriteLine("#");
                writer.WriteLine("# Maximum ring difference of data set (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ring_difference " + microPETHeader.ring_difference);
                writer.WriteLine("#");
                writer.WriteLine("# Size of W dimension in data set (integer)");
                writer.WriteLine("#");
                writer.WriteLine("w_dimension " + microPETHeader.w_dimension);
                writer.WriteLine("#");
                writer.WriteLine("# Version of histogram program used (float)");
                writer.WriteLine("#");
                writer.WriteLine("histogram_version " + microPETHeader.histogram_version.ToString("000.000").Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Rebinning type (integer)");
                writer.WriteLine("#   0 - Unknown, or no, algorithm type");
                writer.WriteLine("#   1 - Full 3D binning (span and ring difference)");
                writer.WriteLine("#   2 - Single-Slice Rebinning");
                writer.WriteLine("#   3 - Fourier Rebinning");
                writer.WriteLine("#");
                writer.WriteLine("rebinning_type " + (int)microPETHeader.rebinning_type);
                writer.WriteLine("#");
                writer.WriteLine("# Version of rebinning program used (float)");
                writer.WriteLine("#");
                writer.WriteLine("rebinning_version " + microPETHeader.rebinning_version.ToString("000.000").Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Source type (integer)");
                writer.WriteLine("#");
                writer.WriteLine("#   0 - Unknown TX source type");
                writer.WriteLine("#   1 - TX point source");
                writer.WriteLine("#   2 - TX line source");
                writer.WriteLine("#");
                writer.WriteLine("tx_src_type " + (int)microPETHeader.tx_src_type);
                writer.WriteLine("#");
                writer.WriteLine("# Data order (integer)");
                writer.WriteLine("#   0 - Element/Axis/View/Ring_Diff - view mode");
                writer.WriteLine("#   1 - Element/View/Axis/Ring_Diff - sinogram mode");
                writer.WriteLine("#  NOTE that ElVwAxRd (XYZW) is the data order for images.");
                writer.WriteLine("#    ElVwAxRd for images means that Z and Y are flipped.");
                writer.WriteLine("#");
                writer.WriteLine("data_order " + (int)microPETHeader.data_order);
                writer.WriteLine("#");
                writer.WriteLine("# OSEM2D method (integer)");
                writer.WriteLine("#");
                writer.WriteLine("#   0 - Unweighted osem2d reconstruction");
                writer.WriteLine("#   1 - Attenuation weighted osem2d reconstruction");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_method " + (int)microPETHeader.osem2d_method);
                writer.WriteLine("#");
                writer.WriteLine("# Number of osem2d subsets (integer)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_subsets " + microPETHeader.osem2d_subsets);
                writer.WriteLine("#");
                writer.WriteLine("# Number of osem2d iterations (integer)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_iterations " + microPETHeader.osem2d_iterations);
                writer.WriteLine("#");
                writer.WriteLine("# Number of EM iterations after osem2d iterations (integer)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_em_iterations " + microPETHeader.osem2d_em_iterations);
                writer.WriteLine("#");
                writer.WriteLine("# Epsilon and power values for map regularization (float integer)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_map " + microPETHeader.osem2d_map_E.ToString().Replace(',', '.') + " " + microPETHeader.osem2d_map_P);
                writer.WriteLine("#");
                writer.WriteLine("# Large object osem2d x_offset in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_x_offset " + microPETHeader.osem2d_x_offset.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Large object osem2d y_offset in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_y_offset " + microPETHeader.osem2d_y_offset.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Large object osem2d zoom (float)");
                writer.WriteLine("#");
                writer.WriteLine("osem2d_zoom " + microPETHeader.osem2d_zoom.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Arc correction applied to data set (integer)");
                writer.WriteLine("#   0 - Unknown, or no, arc correction");
                writer.WriteLine("#   1 - Radial arc correction (1D)");
                writer.WriteLine("#   2 - Resampled radial/angular arc correction (2D)");
                writer.WriteLine("#");
                writer.WriteLine("arc_correction_applied " + (int)microPETHeader.arc_correction_applied);
                writer.WriteLine("#");
                writer.WriteLine("# Number of singles rates in subheader (integer)");
                writer.WriteLine("#   NOTE: This normally is the number of blocks.");
                writer.WriteLine("#");
                writer.WriteLine("number_of_singles_rates " + microPETHeader.number_of_singles_rates);
                writer.WriteLine("#");
                writer.WriteLine("# Subject glucose level (string)");
                writer.WriteLine("#");
                writer.WriteLine("subject_glucose_level " + microPETHeader.subject_glucose_level);
                writer.WriteLine("#");
                writer.WriteLine("# Subject glucose level measurement time (string)");
                writer.WriteLine("#");
                writer.WriteLine("subject_glucose_level_time " + microPETHeader.subject_glucose_level_time);
            }
            #endregion
            #region CT_header_fields
            else {
                writer.WriteLine("#");
                writer.WriteLine("#  Following parameters are used for CT or SPECT modalities.");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("# Gantry rotation (integer)");
                writer.WriteLine("#");
                writer.WriteLine("#     0 (FALSE) - Gantry does NOT rotate");
                writer.WriteLine("#     !0 (TRUE) - Gantry does rotate");
                writer.WriteLine("#");
                writer.WriteLine("gantry_rotation " + ((microPETHeader as MicroPETHeader_CTspecific).gantry_rotation ? 1 : 0));
                writer.WriteLine("#");
                writer.WriteLine("# Rotation direction (integer)");
                writer.WriteLine("#");
                writer.WriteLine("#     0 - Clockwise");
                writer.WriteLine("#     1 - Counterclockwise");
                writer.WriteLine("#");
                writer.WriteLine("rotation_direction " + (int)(microPETHeader as MicroPETHeader_CTspecific).rotation_direction);
                writer.WriteLine("#");
                writer.WriteLine("# Rotating gantry starting angle, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("rotating_stage_start_position " + (microPETHeader as MicroPETHeader_CTspecific).rotating_stage_start_position.ToString().Replace(',','.'));
                writer.WriteLine("#");
                writer.WriteLine("# Rotating gantry stop angle, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("rotating_stage_stop_position " + (microPETHeader as MicroPETHeader_CTspecific).rotating_stage_stop_position.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Number of rotation projections for rotating gantry (integer)");
                writer.WriteLine("#");
                writer.WriteLine("number_of_projections " + (microPETHeader as MicroPETHeader_CTspecific).number_of_projections);
                writer.WriteLine("#");
                writer.WriteLine("#  Following parameters are used for CT modality only.");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("# CAT file version (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_file_version " + (microPETHeader as MicroPETHeader_CTspecific).ct_file_version);
                writer.WriteLine("#");
                writer.WriteLine("# Header size of CAT files before dark and light projections (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_header_size " + (microPETHeader as MicroPETHeader_CTspecific).ct_header_size);
                writer.WriteLine("#");
                writer.WriteLine("# CT transaxial projection size, in pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_proj_size_transaxial " + (microPETHeader as MicroPETHeader_CTspecific).ct_proj_size_transaxial);
                writer.WriteLine("#");
                writer.WriteLine("# CT axial projection size, in pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_proj_size_axial " + (microPETHeader as MicroPETHeader_CTspecific).ct_proj_size_axial);
                writer.WriteLine("#");
                writer.WriteLine("# Number to average the dark calibration projections (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_average_dark_projections " + (microPETHeader as MicroPETHeader_CTspecific).ct_average_dark_projections);
                writer.WriteLine("#");
                writer.WriteLine("# Number to average the light calibration projections (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_average_light_projections " + (microPETHeader as MicroPETHeader_CTspecific).ct_average_light_projections);
                writer.WriteLine("#");
                writer.WriteLine("# Total positions to acquire the light calibration projections (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_light_calibration_projections " + (microPETHeader as MicroPETHeader_CTspecific).ct_light_calibration_projections);
                writer.WriteLine("#");
                writer.WriteLine("# Indicates if the positions to acquire light projections are same as scan projection positions (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_dependent_light_calibration_projections " + (microPETHeader as MicroPETHeader_CTspecific).ct_dependent_light_calibration_projections);
                writer.WriteLine("#");
                writer.WriteLine("# CT X-ray detector offset, in mm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_xray_detector_offset " + (microPETHeader as MicroPETHeader_CTspecific).ct_xray_detector_offset.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT detector transaxial position, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_detector_transaxial_position " + (microPETHeader as MicroPETHeader_CTspecific).ct_detector_transaxial_position.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT detector uncropped transaxial pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_uncropped_transaxial_pixels " + (microPETHeader as MicroPETHeader_CTspecific).ct_uncropped_transaxial_pixels);
                writer.WriteLine("#");
                writer.WriteLine("# CT detector uncropped axial pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_uncropped_axial_pixels " + (microPETHeader as MicroPETHeader_CTspecific).ct_uncropped_axial_pixels);
                writer.WriteLine("#");
                writer.WriteLine("# CT detector cropped transaxial pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_cropped_transaxial_pixels " + (microPETHeader as MicroPETHeader_CTspecific).ct_cropped_transaxial_pixels);
                writer.WriteLine("#");
                writer.WriteLine("# CT detector cropped axial pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_cropped_axial_pixels " + (microPETHeader as MicroPETHeader_CTspecific).ct_cropped_axial_pixels);
                writer.WriteLine("#");
                writer.WriteLine("# CT X-ray detector pitch, in um (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_xray_detector_pitch " + (microPETHeader as MicroPETHeader_CTspecific).ct_xray_detector_pitch.ToString().Replace(',','.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT horizontal rotation-axis-bed angle, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_horiz_rot_axis_bed_angle " + (microPETHeader as MicroPETHeader_CTspecific).ct_horiz_rot_axis_bed_angle.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT vertical rotation-axis-bed angle, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_vert_rot_axis_bed_angle " + (microPETHeader as MicroPETHeader_CTspecific).ct_vert_rot_axis_bed_angle.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT exposure time, in msecs (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_exposure_time " + (microPETHeader as MicroPETHeader_CTspecific).ct_exposure_time.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT total scan time, in min:sec (integer:integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_scan_time " + (microPETHeader as MicroPETHeader_CTspecific).ct_scan_time[0]+":"+
                                                   (microPETHeader as MicroPETHeader_CTspecific).ct_scan_time[1]);
                writer.WriteLine("#");
                writer.WriteLine("# CT warping type (integer)");
                writer.WriteLine("#   0 - Unknown warping type");
                writer.WriteLine("#   1 - No warping");
                writer.WriteLine("#   2 - Bilinear warping");
                writer.WriteLine("#   3 - Nearest neighbor warping");
                writer.WriteLine("#");
                writer.WriteLine("ct_warping " + (int)(microPETHeader as MicroPETHeader_CTspecific).ct_warping);
                writer.WriteLine("#");
                writer.WriteLine("# CT defect map filename, possibly including path (string)");
                writer.WriteLine("#    NOTE: Filename may contain spaces, therefore the ENTIRE ");
                writer.WriteLine("#          line, up to the EOL, is used after the parameter name.");
                writer.WriteLine("#");
                writer.WriteLine("ct_defect_map_file_name " + (microPETHeader as MicroPETHeader_CTspecific).ct_defect_map_file_name);
                writer.WriteLine("#");
                writer.WriteLine("# CT x-ray voltage, in kVp (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_xray_voltage " + (microPETHeader as MicroPETHeader_CTspecific).ct_xray_voltage.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT anode current, in uA (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_anode_current " + (microPETHeader as MicroPETHeader_CTspecific).ct_anode_current.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# Number of CT calibration exposures (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_calibration_exposures " + (microPETHeader as MicroPETHeader_CTspecific).ct_calibration_exposures);
                writer.WriteLine("#");
                writer.WriteLine("# CT cone angle, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_cone_angle " + (microPETHeader as MicroPETHeader_CTspecific).ct_cone_angle.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection interpolation type (integer)");
                writer.WriteLine("#   0 - Unknown projection interpolation type");
                writer.WriteLine("#   1 - Bilinear projection interpolation");
                writer.WriteLine("#   2 - Nearest neighbor projection interpolation");
                writer.WriteLine("#");
                writer.WriteLine("ct_projection_interpolation " + (int)(microPETHeader as MicroPETHeader_CTspecific).ct_projection_interpolation);
                writer.WriteLine("#");
                writer.WriteLine("# CT source to detector distance, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_source_to_detector " + (microPETHeader as MicroPETHeader_CTspecific).ct_source_to_detector.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT source to center of rotation, in cm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_source_to_crot " + (microPETHeader as MicroPETHeader_CTspecific).ct_source_to_crot.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT vetical detector offset, in pixels (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_detector_vertical_offset " + (microPETHeader as MicroPETHeader_CTspecific).ct_detector_vertical_offset.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT detector tilt relative to horizontal axis, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_detector_horizontal_tilt " + (microPETHeader as MicroPETHeader_CTspecific).ct_detector_horizontal_tilt.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT detector tilt relative to vertical axis, in degrees (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_detector_vertical_tilt " + (microPETHeader as MicroPETHeader_CTspecific).ct_detector_vertical_tilt.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT axial projection bin factor, in pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_transaxial_bin_factor " + (microPETHeader as MicroPETHeader_CTspecific).ct_transaxial_bin_factor);
                writer.WriteLine("#");
                writer.WriteLine("# CT axial projection bin factor, in pixels (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_axial_bin_factor " + (microPETHeader as MicroPETHeader_CTspecific).ct_axial_bin_factor);
                writer.WriteLine("#");
                writer.WriteLine("# CT gating signal used (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_gating " + (microPETHeader as MicroPETHeader_CTspecific).ct_gating);
                writer.WriteLine("#");
                writer.WriteLine("# CT Hounsfield scale (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_hounsfield_scale " + (microPETHeader as MicroPETHeader_CTspecific).ct_hounsfield_scale.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT Hounsfield offset (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_hounsfield_offset " + (microPETHeader as MicroPETHeader_CTspecific).ct_hounsfield_offset.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection downsample factor (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_proj_downsample_factor " + (microPETHeader as MicroPETHeader_CTspecific).ct_proj_downsample_factor);
                writer.WriteLine("#");
                writer.WriteLine("# CT first projection used in reconstruction (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_first_recon_proj " + (microPETHeader as MicroPETHeader_CTspecific).ct_first_recon_proj);
                writer.WriteLine("#");
                writer.WriteLine("# CT last projection used in reconstruction (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_last_recon_proj " + (microPETHeader as MicroPETHeader_CTspecific).ct_last_recon_proj);
                writer.WriteLine("#");
                writer.WriteLine("# CT every Nth projection used for reconstruction (integer)");
                writer.WriteLine("#");
                writer.WriteLine("ct_recon_every_nth_proj " + (microPETHeader as MicroPETHeader_CTspecific).ct_recon_every_nth_proj);
                writer.WriteLine("#");
                writer.WriteLine("# CT attenuation of water, in cm^-1 (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_attenuation_water " + (microPETHeader as MicroPETHeader_CTspecific).ct_attenuation_water.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT TX rotation offsets: X Y Z, in mm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_tx_rotation_offsets " + (microPETHeader as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[0].ToString().Replace(',', '.')+" "+
                                                             (microPETHeader as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[1].ToString().Replace(',', '.')+" "+
                                                             (microPETHeader as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[2].ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT TX transaxial offsets: X Y Z, in mm (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_tx_transaxial_offsets " + (microPETHeader as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[0].ToString().Replace(',', '.')+" "+
                                                               (microPETHeader as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[1].ToString().Replace(',', '.')+" "+
                                                               (microPETHeader as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[2].ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT BH correction applied (int)");
                writer.WriteLine("#");
                writer.WriteLine("#     0 (FALSE) - CT BH correction has NOT been applied");
                writer.WriteLine("#     !0 (TRUE) - CT BH correction has been applied");
                writer.WriteLine("#");
                writer.WriteLine("ct_bh_correction " + (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction.ToString("e5").Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT BH correction coefficients (float  float  float   float)");
                writer.WriteLine("#   NOTE: This is ONLY present when BH correction has been applied.");
                writer.WriteLine("#");
                writer.WriteLine("ct_bh_correction_coefficients " + (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[0].ToString().Replace(',', '.')+" "+
                                                                    (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[1].ToString().Replace(',', '.')+" "+
                                                                    (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[2].ToString().Replace(',', '.')+" "+
                                                                    (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[3].ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT aluminum filter thickness (mm) (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_aluminum_filter_thickness " + (microPETHeader as MicroPETHeader_CTspecific).ct_aluminum_filter_thickness.ToString().Replace(',', '.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection array (integer  float)");
                writer.WriteLine("#       projection_number     acquisition_position (degrees)");
                writer.WriteLine("#");
                for(int i = 0; i < (microPETHeader as MicroPETHeader_CTspecific).projections.Length; i++)
                    writer.WriteLine("projection "+(microPETHeader as MicroPETHeader_CTspecific).projections[i].projection_number+" "+
                                                   (microPETHeader as MicroPETHeader_CTspecific).projections[i].data.ToString().Replace(',','.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection average center_offset (float)");
                writer.WriteLine("#");
                writer.WriteLine("ct_projection_average_center_offset "+(microPETHeader as MicroPETHeader_CTspecific).ct_projection_average_center_offset.ToString().Replace(',','.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection average center_offset array (integer  float)");
                writer.WriteLine("#       projection_number     average center offset (mm)");
                writer.WriteLine("#");
                for(int i = 0; i < (microPETHeader as MicroPETHeader_CTspecific).ct_projection_center_offsets.Length; i++)
                    writer.WriteLine("ct_projection_center_offset "+(microPETHeader as MicroPETHeader_CTspecific).ct_projection_center_offsets[i].projection_number+" "+
                                                   (microPETHeader as MicroPETHeader_CTspecific).ct_projection_center_offsets[i].data.ToString().Replace(',','.'));
                writer.WriteLine("#");
                writer.WriteLine("# CT projection horizontal bed offset array (integer  float)");
                writer.WriteLine("#       projection_number     horizontal bed offset (mm)");
                writer.WriteLine("#");
                for(int i = 0; i < (microPETHeader as MicroPETHeader_CTspecific).ct_projection_horizontal_bed_offsets.Length; i++)
                    writer.WriteLine("ct_projection_horizontal_bed_offset "+(microPETHeader as MicroPETHeader_CTspecific).ct_projection_horizontal_bed_offsets[i].projection_number+" "+
                                                   (microPETHeader as MicroPETHeader_CTspecific).ct_projection_horizontal_bed_offsets[i].data.ToString().Replace(',','.'));
            }
            #endregion
            writer.WriteLine("#");
            writer.WriteLine("# End of Header indicator ");
            writer.WriteLine("#");
            writer.WriteLine("end_of_header");
            #endregion
        } 
        /// <summary>
        /// Writes sub header into stream. Note that only 1st subheader normally has comments. 
        /// </summary>
        /// <param name="writer">stream where the data is written</param>
        /// <param name="subheader">sub header that is written</param>
        /// <param name="comments">true for writing comments as well</param>
        /// <param name="modality">scanner modality</param>
        protected static void WriteMicroPETSubheader(TextWriter writer, MicroPETSubHeader subheader, bool comments, MicroPETHeader.scanner_modality modality) {
            #region writing_of_data_line_by_line
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Header for frame 0, detector_panel " + subheader.detector_panel +
                                 ", event_type " + (int)subheader.event_type +
                                 ", energy_window " + subheader.energy_window +
                                 ", gate " + subheader.gate +
                                 ", bed " + subheader.bed + ".");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("#  Following parameters are used for ALL modalities.");
                writer.WriteLine("#");
                writer.WriteLine("#");
                writer.WriteLine("# Frame number (integer)");
                writer.WriteLine("#");
            }
            writer.WriteLine("frame "+subheader.frame);
            if (modality != MicroPETHeader.scanner_modality.CT)
            {
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Detector panel - NOTE: \"ring\" systems are \"0\" ONLY (integer)");
                    writer.WriteLine("#");
                }
                writer.WriteLine("detector_panel " + subheader.detector_panel);
            }
            if(comments) {
                writer.WriteLine("#");
                writer.WriteLine("# Event type (integer)");
                writer.WriteLine("#   0 - Unknown event type");
                writer.WriteLine("#   1 - Singles");
                writer.WriteLine("#   2 - Prompt events (coincidences)");
                writer.WriteLine("#   3 - Delay events");
                writer.WriteLine("#   4 - Trues (prompts - delays)");
                writer.WriteLine("#   5 - Energy spectrum data");
                writer.WriteLine("#");
            }
            writer.WriteLine("event_type "+(int)subheader.event_type);
            if (modality != MicroPETHeader.scanner_modality.CT)
            {
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Energy window - NOTE PET/CT systems are typically \"0\" ONLY (integer)");
                    writer.WriteLine("#");
                }
                writer.WriteLine("energy_window " + subheader.energy_window);
            }
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Gate number (integer)");
                writer.WriteLine("#");
            }
            writer.WriteLine("gate "+subheader.gate);
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Bed number (integer)");
                writer.WriteLine("#");
            }
            writer.WriteLine("bed "+subheader.bed);
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Bed offset, in cm (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("bed_offset " + subheader.bed_offset.ToString().Replace(',', '.'));
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Ending horizontal bed offset, in cm (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("ending_bed_offset "+subheader.ending_bed_offset.ToString().Replace(',','.'));
            if (modality == MicroPETHeader.scanner_modality.CT)
            {
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Number of bed passes during frame (integer)");
                    writer.WriteLine("#");
                }
                writer.WriteLine("bed_passes " + subheader.bed_passes);
            }
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Vertical bed offset, in cm (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("vertical_bed_offset " + subheader.vertical_bed_offset.ToString().Replace(',', '.'));
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Data file offset to start of data (2 32-bit ints)");
                writer.WriteLine("#     Values are: low_part");
                writer.WriteLine("#             or: high_part low_part");
                writer.WriteLine("#");
            }
            writer.WriteLine("data_file_pointer "+subheader.data_file_pointer[0]+" "+subheader.data_file_pointer[1]);
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Frame start time, in secs (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("frame_start "+subheader.frame_start);
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Frame duration, in secs (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("frame_duration " + subheader.frame_duration.ToString().Replace(',', '.'));
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Scale factor for data set (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("scale_factor " + subheader.scale_factor.ToString().Replace(',', '.'));
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Minimum value in data set (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("minimum " + subheader.minimum.ToString().Replace(',', '.'));
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# Maximum value in data set (float)");
                writer.WriteLine("#");
            }
            writer.WriteLine("maximum " + subheader.maximum.ToString().Replace(',', '.'));
            if (modality == MicroPETHeader.scanner_modality.CT) {
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("#  Following parameters are used for CT modality only.");
                    writer.WriteLine("#");
                }                
            }
            else
            {
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("#  Following parameters are used for PET or SPECT modalities.");
                    writer.WriteLine("#");
                    writer.WriteLine("#");
                    writer.WriteLine("# Deadtime correction for data set (float)");
                    writer.WriteLine("#             NOTE: Scale factor INCLUDES this value.");
                    writer.WriteLine("#");
                }
                writer.WriteLine("deadtime_correction " + subheader.deadtime_correction.ToString().Replace(',', '.'));
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Global decay correction applied to data set (float)");
                    writer.WriteLine("#             NOTE: Scale factor INCLUDES this value.");
                    writer.WriteLine("#");
                }
                writer.WriteLine("decay_correction " + subheader.decay_correction.ToString().Replace(',', '.'));
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Prompts count for data set (64-bit ints)");
                    writer.WriteLine("#");
                    writer.WriteLine("#   NOTE: The 3 values represent histogrammed prompts,");
                    writer.WriteLine("#         unhistogrammed prompts, and prompts at coincidence detection.");
                    writer.WriteLine("#");
                }
                writer.WriteLine("prompts " + subheader.prompts[0] + "   " + subheader.prompts[1] + "   " + subheader.prompts[2]);
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Prompts countrate per sec before histogramming (int)");
                    writer.WriteLine("#");
                }
                writer.WriteLine("prompts_rate " + subheader.prompts_rate);
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("#  Following parameters are used for PET modality only.");
                    writer.WriteLine("#");
                    writer.WriteLine("#");
                    writer.WriteLine("# Delays count for data set (64-bit ints)");
                    writer.WriteLine("#");
                    writer.WriteLine("#   NOTE: The 3 values represent histogrammed delays, ");
                    writer.WriteLine("#         unhistogrammed delays, and delays at coincidence detection.");
                    writer.WriteLine("#");
                }
                writer.WriteLine("delays " + subheader.delays[0] + "   " + subheader.delays[1] + "   " + subheader.delays[2]);
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Trues count for data set (64-bit ints)");
                    writer.WriteLine("#");
                    writer.WriteLine("#   NOTE: The 3 values represent histogrammed trues, ");
                    writer.WriteLine("#         unhistogrammed trues, and trues at coincidence detection.");
                    writer.WriteLine("#");
                }
                writer.WriteLine("trues " + subheader.trues[0] + "   " + subheader.trues[1] + "   " + subheader.trues[2]);
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Delays countrate per sec before histogramming (int)");
                    writer.WriteLine("#");
                }
                writer.WriteLine("delays_rate " + subheader.delays_rate);
                if (comments)
                {
                    writer.WriteLine("#");
                    writer.WriteLine("# Singles rates array (integer  float  float)");
                    writer.WriteLine("#       block_number    singles/sec    raw_singles/sec    xy_stored_singles/sec    block_singles_output/sec    raw_block_prompts/sec    block_prompts/sec    raw_block_delays/sec    block_delays/sec");
                    writer.WriteLine("#");
                }
                if (subheader.singles != null)
                {
                    for (int i = 0; i < subheader.singles.Length; i++)
                    {
                        writer.WriteLine("singles " + (subheader.singles[i].block_number + " " +
                            subheader.singles[i].singles_per_sec + " " +
                            subheader.singles[i].raw_singles_per_sec + " " +
                            subheader.singles[i].xy_stored_singles_per_sec + " " +
                            subheader.singles[i].block_singles_output_per_sec + " " +
                            subheader.singles[i].raw_block_prompts_per_sec + " " +
                            subheader.singles[i].block_prompts_per_sec + " " +
                            subheader.singles[i].raw_block_delays_per_sec + " " +
                            subheader.singles[i].block_delays_per_sec).Replace(',', '.'));
                    }
                }
            }
            if (comments)
            {
                writer.WriteLine("#");
                writer.WriteLine("# End of Header indicator ");
                writer.WriteLine("#");
            }
            writer.WriteLine("end_of_header");
            writer.Flush();
            #endregion
        }
        /// <summary>
        /// Parses field value from line 
        /// </summary>
        /// <param name="line">line in file</param>
        /// <param name="field">field name</param>
        /// <returns>rest of the line after field if the field was found, null otherwise</returns>
        protected string ParseField(string line, string field) {
            int index = 0;
            if ((index = line.IndexOf(field)) == 0)
            {
                return line.Substring(index + field.Length);
            } else {
                return null;
            }
        }
        /// <summary>
        /// Reads main header from stream.
        /// </summary>
        /// <param name="reader">open stream for reading</param>
        /// <returns>header data</returns>
        protected MicroPETHeader ReadMicroPETMainheader(StreamReader reader) {
            MicroPETHeader hdr = null;
            string line;
            string strippedline;
            long pos = reader.BaseStream.Position;
            List<MicroPETHeader_CTspecific.Projection> projections = new List<MicroPETHeader_CTspecific.Projection>();
            List<MicroPETHeader_CTspecific.Projection> ct_projection_center_offsets = new List<MicroPETHeader_CTspecific.Projection>();
            List<MicroPETHeader_CTspecific.Projection> ct_projection_horizontal_bed_offsets = new List<MicroPETHeader_CTspecific.Projection>();

            //resolve modality first
            #region resolve_modality
            while (!reader.EndOfStream)
            {
                line = reader.ReadLine();
                if (line.StartsWith("#")) continue;
                if ((strippedline = ParseField(line, "modality ")) != null)
                {
                    MicroPETHeader.scanner_modality modality = (MicroPETHeader.scanner_modality)int.Parse(strippedline);
                    switch (modality) {
                        case MicroPETHeader.scanner_modality.CT:
                            hdr = new MicroPETHeader_CTspecific();
                            break;
                        case MicroPETHeader.scanner_modality.PET:
                        case MicroPETHeader.scanner_modality.SPECT:
                        case MicroPETHeader.scanner_modality.Unknown:
                        default:
                            hdr = new MicroPETHeader();
                            break;
                    }
                    hdr.modality = modality;
                    break;
                } else if(line.StartsWith("end_of_header")) {
                    break;
                }
            }
            //if modality was not found then the header cannot be read
            if (hdr == null)
                throw new TPCMicroPETFileException("Failed to read modality from main header.");
            //go back to start reading the rest of the fields 
            reader.BaseStream.Seek(pos, SeekOrigin.Begin);
            #endregion

            //read until end of header tag is found or file ends
            while(!reader.EndOfStream) {
                line = reader.ReadLine();
                if (line.StartsWith("#")) continue;
                #region try_to_match_each_header_field
                if((strippedline = ParseField(line, "version ")) != null) {
                    hdr.version = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "manufacturer ")) != null) {
                    hdr.manufacturer = strippedline.Trim();
                } else if((strippedline = ParseField(line, "model ")) != null) {
                    hdr.model = (MicroPETHeader.model_number)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "modality_configuration ")) != null) {
                    hdr.modality_configuration = (MicroPETHeader.Modality_configuration)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "institution ")) != null) {
                    hdr.institution = strippedline.Trim();
                } else if((strippedline = ParseField(line, "study ")) != null) {
                    hdr.study = strippedline.Trim();
                } else if((strippedline = ParseField(line, "file_name ")) != null) {
                    hdr.file_name = strippedline.Trim();
                } else if((strippedline = ParseField(line, "file_type ")) != null) {
                    hdr.file_type = (MicroPETHeader.Data_file_type)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "acquisition_mode ")) != null) {
                    hdr.acquisition_mode = (MicroPETHeader.Acquisition_mode)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "total_frames ")) != null) {
                    hdr.total_frames = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "time_frames ")) != null) {
                    hdr.time_frames = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "bed_control ")) != null) {
                    hdr.bed_control = (MicroPETHeader.Bed_Control)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "bed_motion ")) != null) {
                    hdr.bed_motion = (MicroPETHeader.Bed_motion)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "number_of_bed_positions ")) != null) {
                    hdr.number_of_bed_positions = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "horizontal_bed_calibration ")) != null) {
                    hdr.horizontal_bed_calibration = float.Parse(strippedline);
                } else if((strippedline = ParseField(line, "vertical_bed_calibration ")) != null) {
                    hdr.vertical_bed_calibration = float.Parse(strippedline);
                } else if((strippedline = ParseField(line, "transaxial_bin_size ")) != null) {
                    hdr.transaxial_bin_size = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "axial_plane_size ")) != null) {
                    hdr.axial_plane_size = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "number_detector_panels ")) != null) {
                    hdr.number_detector_panels = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "data_type ")) != null) {
                    hdr.data_type = (MicroPETHeader.Data_type)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "number_of_dimensions ")) != null) {
                    hdr.number_of_dimensions = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "x_dimension ")) != null) {
                    hdr.x_dimension = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "y_dimension ")) != null) {
                    hdr.y_dimension = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "z_dimension ")) != null) {
                    hdr.z_dimension = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "acquisition_user_id ")) != null) {
                    hdr.acquisition_user_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "histogram_user_id ")) != null) {
                    hdr.histogram_user_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "reconstruction_user_id ")) != null) {
                    hdr.reconstruction_user_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "scatter_correction_user_id ")) != null) {
                    hdr.scatter_correction_user_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "acquisition_notes ")) != null) {
                    hdr.acquisition_notes = strippedline.Trim();
                } else if((strippedline = ParseField(line, "scan_time ")) != null) {
                    hdr.scan_time = strippedline.Trim();
                } else if((strippedline = ParseField(line, "gmt_scan_time ")) != null) {
                    hdr.gmt_scan_time = strippedline.Trim();
                } else if((strippedline = ParseField(line, "volume_origin_x ")) != null) {
                    hdr.volume_origin_x = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "volume_origin_y ")) != null) {
                    hdr.volume_origin_y = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "volume_origin_z ")) != null) {
                    hdr.volume_origin_z = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "registration_available ")) != null) {
                    hdr.registration_available = (MicroPETHeader.Registration_data_availability)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "transformation_matrix ")) != null) {
                    hdr.transformation_matrix = strippedline.Trim();
                } else if((strippedline = ParseField(line, "spatial_identifier ")) != null) {
                    hdr.spatial_identifier = strippedline.Trim();
                } else if((strippedline = ParseField(line, "normalization_applied ")) != null) {
                    hdr.normalization_applied = (MicroPETHeader.Normalization_type)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "normalization_filename ")) != null) {
                    hdr.normalization_filename = strippedline.Trim();
                } else if((strippedline = ParseField(line, "recon_algorithm ")) != null) {
                    hdr.recon_algorithm = (MicroPETHeader.Reconstruction_type)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "recon_version ")) != null) {
                    hdr.recon_version = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "x_filter ")) != null) {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.x_filter_type = (MicroPETHeader.Filter_type)int.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.x_filter_cutoff = float.Parse(splitted[1].Replace('.', ','));
                } else if((strippedline = ParseField(line, "y_filter ")) != null) {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.y_filter_type = (MicroPETHeader.Filter_type)int.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.y_filter_cutoff = float.Parse(splitted[1].Replace('.', ','));
                } else if ((strippedline = ParseField(line, "z_filter ")) != null) {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.z_filter_type = (MicroPETHeader.Filter_type)int.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.z_filter_cutoff = float.Parse(splitted[1].Replace('.', ','));
                } else if ((strippedline = ParseField(line, "x_offset ")) != null) {
                    hdr.x_offset = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "y_offset ")) != null) {
                    hdr.y_offset = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "zoom ")) != null) {
                    hdr.zoom = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "pixel_size ")) != null) {
                    hdr.pixel_size = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "pixel_size_x ")) != null) {
                    hdr.pixel_size_x = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "pixel_size_y ")) != null) {
                    hdr.pixel_size_y = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "pixel_size_z ")) != null) {
                    hdr.pixel_size_z = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "investigator ")) != null) {
                    hdr.investigator_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "operator ")) != null) {
                    hdr.operator_id = strippedline.Trim();
                } else if((strippedline = ParseField(line, "study_identifier ")) != null) {
                    hdr.study_identifier = strippedline.Trim();
                } else if((strippedline = ParseField(line, "injected_compound ")) != null) {
                    hdr.injected_compound = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_identifier ")) != null) {
                    hdr.subject_identifier = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_genus ")) != null) {
                    hdr.subject_genus = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_orientation ")) != null) {
                    hdr.subject_orientation = (MicroPETHeader.Subject_orientation)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "subject_length_units ")) != null) {
                    hdr.subject_length_units = (MicroPETHeader.Length_unit)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "subject_length ")) != null) {
                    hdr.subject_length = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "subject_weight_units ")) != null) {
                    hdr.subject_weight_units = (MicroPETHeader.Weight_unit)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "subject_weight ")) != null) {
                    hdr.subject_weight = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "subject_phenotype ")) != null) {
                    hdr.subject_phenotype = strippedline.Trim();
                } else if((strippedline = ParseField(line, "study_model ")) != null) {
                    hdr.study_model = strippedline.Trim();
                } else if((strippedline = ParseField(line, "anesthesia ")) != null) {
                    hdr.anesthesia = strippedline.Trim();
                } else if((strippedline = ParseField(line, "analgesia ")) != null) {
                    hdr.analgesia = strippedline.Trim();
                } else if((strippedline = ParseField(line, "other_drugs ")) != null) {
                    hdr.other_drugs = strippedline.Trim();
                } else if((strippedline = ParseField(line, "food_access ")) != null) {
                    hdr.food_access = strippedline.Trim();
                } else if((strippedline = ParseField(line, "water_access ")) != null) {
                    hdr.water_access = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_date_of_birth ")) != null) {
                    hdr.subject_date_of_birth = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_age ")) != null) {
                    hdr.subject_age = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_sex ")) != null) {
                    hdr.subject_sex = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_scan_region ")) != null) {
                    hdr.subject_scan_region = strippedline.Trim();
                } else if((strippedline = ParseField(line, "acquisition_file_name ")) != null) {
                    hdr.acquisition_file_name = strippedline.Trim();
                } else if((strippedline = ParseField(line, "transaxial_crystal_pitch ")) != null) {
                    hdr.transaxial_crystal_pitch = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "axial_crystal_pitch ")) != null) {
                    hdr.axial_crystal_pitch = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "crystal_thickness ")) != null) {
                    hdr.crystal_thickness = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "depth_of_interaction ")) != null) {
                    hdr.depth_of_interaction = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "isotope ")) != null) {
                    hdr.isotope = strippedline.Trim();
                } else if((strippedline = ParseField(line, "isotope_half_life ")) != null) {
                    hdr.isotope_half_life = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "isotope_branching_fraction ")) != null) {
                    hdr.isotope_branching_fraction = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "deadtime_correction_applied ")) != null) {
                    hdr.deadtime_correction_applied = (MicroPETHeader.Deadtime_correction)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "decay_correction_applied ")) != null) {
                    hdr.decay_correction_applied = (int.Parse(strippedline) == 0 ? false : true);
                } else if((strippedline = ParseField(line, "attenuation_applied ")) != null) {
                    hdr.attenuation_applied = (MicroPETHeader.Attenuation_correction)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "attenuation_filename ")) != null) {
                    hdr.attenuation_filename = strippedline.Trim();
                } else if((strippedline = ParseField(line, "scatter_correction ")) != null) {
                    hdr.scatter_correction = (MicroPETHeader.Scatter_correction)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "calibration_units ")) != null) {
                    hdr.calibration_units = (MicroPETHeader.Calibration_units)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "calibration_factor ")) != null) {
                    hdr.calibration_factor = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "calibration_branching_fraction ")) != null) {
                    hdr.calibration_branching_fraction = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "dose_units ")) != null) {
                    hdr.dose_units = (MicroPETHeader.Radioactivity_units)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "dose ")) != null) {
                    hdr.dose = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "injection_time ")) != null) {
                    hdr.injection_time = strippedline.Trim();
                } else if((strippedline = ParseField(line, "injection_decay_correction ")) != null) {
                    hdr.injection_decay_correction = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "activity_units ")) != null) {
                    hdr.activity_units = (MicroPETHeader.Radioactivity_units)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "activity_before_injection ")) != null) {
                    hdr.activity_before_injection = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "activity_before_injection_time ")) != null) {
                    hdr.activity_before_injection_time = strippedline.Trim();
                } else if((strippedline = ParseField(line, "residual_activity ")) != null) {
                    hdr.residual_activity = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "residual_activity_time ")) != null) {
                    hdr.residual_activity_time = strippedline.Trim();
                } else if((strippedline = ParseField(line, "transaxial_crystals_per_block ")) != null) {
                    hdr.transaxial_crystals_per_block = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "axial_crystals_per_block ")) != null) {
                    hdr.axial_crystals_per_block = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "intrinsic_crystal_offset ")) != null) {
                    hdr.intrinsic_crystal_offset = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "transaxial_blocks ")) != null) {
                    hdr.transaxial_blocks = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "axial_blocks ")) != null) {
                    hdr.axial_blocks = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "radius ")) != null) {
                    hdr.radius = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "radial_fov ")) != null) {
                    hdr.radial_fov = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "src_radius ")) != null) {
                    hdr.src_radius = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "src_cm_per_rev ")) != null) {
                    hdr.src_cm_per_rev = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "src_steps_per_rev ")) != null) {
                    hdr.src_steps_per_rev = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "default_projections ")) != null) {
                    hdr.default_projections = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "default_transaxial_angles ")) != null) {
                    hdr.default_transaxial_angles = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "lld ")) != null) {
                    hdr.lld = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "uld ")) != null) {
                    hdr.uld = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "timing_window ")) != null) {
                    hdr.timing_window = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "span ")) != null) {
                    hdr.span = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "ring_difference ")) != null) {
                    hdr.ring_difference = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "w_dimension ")) != null) {
                    hdr.w_dimension = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "histogram_version ")) != null) {
                    hdr.histogram_version = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "rebinning_type ")) != null) {
                    hdr.rebinning_type = (MicroPETHeader.Rebinning)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "rebinning_version ")) != null) {
                    hdr.rebinning_version = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "tx_src_type ")) != null) {
                    hdr.tx_src_type = (MicroPETHeader.Source_type)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "data_order ")) != null) {
                    hdr.data_order = (MicroPETHeader.Data_order)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_method ")) != null) {
                    hdr.osem2d_method = (MicroPETHeader.OSEM_method)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_subsets ")) != null) {
                    hdr.osem2d_subsets = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_iterations ")) != null) {
                    hdr.osem2d_iterations = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_em_iterations ")) != null) {
                    hdr.osem2d_em_iterations = int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_map ")) != null) {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.osem2d_map_E = float.Parse(splitted[0].Replace('.', ','));
                    if (splitted.Length > 1) hdr.osem2d_map_P = int.Parse(splitted[1]);
                } else if((strippedline = ParseField(line, "osem2d_x_offset ")) != null) {
                    hdr.osem2d_x_offset = float.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_y_offset ")) != null) {
                    hdr.osem2d_y_offset = float.Parse(strippedline);
                } else if((strippedline = ParseField(line, "osem2d_zoom ")) != null) {
                    hdr.osem2d_zoom = float.Parse(strippedline.Replace('.', ','));
                } else if((strippedline = ParseField(line, "arc_correction_applied ")) != null) {
                    hdr.arc_correction_applied = (MicroPETHeader.Arc_correction)int.Parse(strippedline);
                } else if((strippedline = ParseField(line, "number_of_singles_rates ")) != null) {
                    hdr.number_of_singles_rates = int.Parse(strippedline); 
                } else if((strippedline = ParseField(line, "subject_glucose_level ")) != null) {
                    hdr.subject_glucose_level = strippedline.Trim();
                } else if((strippedline = ParseField(line, "subject_glucose_level_time ")) != null) {
                    hdr.subject_glucose_level_time = strippedline.Trim();
                    //CT spcific fields are read if file modality is CT
                    #region CT_specific
                } else if(hdr.modality == MicroPETHeader.scanner_modality.CT) {
                    if ((strippedline = ParseField(line, "rotation ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).rotation = float.Parse(strippedline.Replace('.', ','));
                    }
                    if ((strippedline = ParseField(line, "z_offset ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).z_offset = float.Parse(strippedline.Replace('.', ','));
                    } 
                    else if ((strippedline = ParseField(line, "gantry_rotation ")) != null) {
                        (hdr as MicroPETHeader_CTspecific).gantry_rotation = (int.Parse(strippedline.Replace('.', ',')) == 0 ? false : true);
                    }
                    else if ((strippedline = ParseField(line, "rotation_direction ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).rotation_direction = (MicroPETHeader_CTspecific.Rotation_direction)int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "rotating_stage_start_position ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).rotating_stage_start_position = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "rotating_stage_stop_position ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).rotating_stage_stop_position = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "number_of_projections ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).number_of_projections = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_file_version ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_file_version = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_header_size ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_header_size = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_proj_size_transaxial ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_proj_size_transaxial = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_proj_size_axial ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_proj_size_axial = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_average_dark_projections ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_average_dark_projections = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_average_light_projections ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_average_light_projections = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_light_calibration_projections ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_light_calibration_projections = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_dependent_light_calibration_projections ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_dependent_light_calibration_projections = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_xray_detector_offset ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_xray_detector_offset = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_detector_transaxial_position ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_detector_transaxial_position = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_uncropped_transaxial_pixels ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_uncropped_transaxial_pixels = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_uncropped_axial_pixels ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_uncropped_axial_pixels = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_cropped_transaxial_pixels ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_cropped_transaxial_pixels = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_cropped_axial_pixels ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_cropped_axial_pixels = int.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_xray_detector_pitch ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_xray_detector_pitch = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_horiz_rot_axis_bed_angle ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_horiz_rot_axis_bed_angle = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_vert_rot_axis_bed_angle ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_vert_rot_axis_bed_angle = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_exposure_time ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_exposure_time = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_scan_time ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[]{':'}, StringSplitOptions.RemoveEmptyEntries);
                        if(splitted.Length > 0)
                            (hdr as MicroPETHeader_CTspecific).ct_scan_time[0] = int.Parse(splitted[0]);
                        if (splitted.Length > 1)
                            (hdr as MicroPETHeader_CTspecific).ct_scan_time[1] = int.Parse(splitted[1]);
                    }
                    else if ((strippedline = ParseField(line, "ct_warping ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_warping = (MicroPETHeader_CTspecific.Warping_type)int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_defect_map_file_name ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_defect_map_file_name = strippedline.Trim();
                    }
                    else if ((strippedline = ParseField(line, "ct_xray_voltage ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_xray_voltage = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_anode_current ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_anode_current = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_calibration_exposures ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_calibration_exposures = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_cone_angle ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_cone_angle = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_warping ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_warping = (MicroPETHeader_CTspecific.Warping_type)int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_source_to_detector ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_source_to_detector = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_source_to_crot ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_source_to_crot = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_detector_vertical_offset ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_detector_vertical_offset = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_detector_horizontal_tilt ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_detector_horizontal_tilt = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_detector_vertical_tilt ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_detector_vertical_tilt = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_transaxial_bin_factor ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_transaxial_bin_factor = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_axial_bin_factor ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_axial_bin_factor = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_gating ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_gating = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_hounsfield_scale ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_hounsfield_scale = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_hounsfield_offset ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_hounsfield_offset = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_proj_downsample_factor ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_proj_downsample_factor = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_first_recon_proj ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_first_recon_proj = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_last_recon_proj ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_last_recon_proj = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_recon_every_nth_proj ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_recon_every_nth_proj = int.Parse(strippedline);
                    }
                    else if ((strippedline = ParseField(line, "ct_attenuation_water ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_attenuation_water = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_tx_rotation_offsets ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[]{' '}, StringSplitOptions.RemoveEmptyEntries);
                        if(splitted.Length > 0)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[0] = float.Parse(splitted[0].Replace('.', ','));
                        if (splitted.Length > 1)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[1] = float.Parse(splitted[1].Replace('.', ','));
                        if (splitted.Length > 2)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_rotation_offsets[2] = float.Parse(splitted[2].Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_tx_transaxial_offsets ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                        if (splitted.Length > 0)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[0] = float.Parse(splitted[0].Replace('.', ','));
                        if (splitted.Length > 1)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[1] = float.Parse(splitted[1].Replace('.', ','));
                        if (splitted.Length > 2)
                            (hdr as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets[2] = float.Parse(splitted[2].Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_bh_correction ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_bh_correction = Single.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_bh_correction_coefficients ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                        if (splitted.Length > 0)
                            (hdr as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[0] = float.Parse(splitted[0].Replace('.', ','));
                        if (splitted.Length > 1)
                            (hdr as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[1] = float.Parse(splitted[1].Replace('.', ','));
                        if (splitted.Length > 2)
                            (hdr as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[2] = float.Parse(splitted[2].Replace('.', ','));
                        if (splitted.Length > 2)
                            (hdr as MicroPETHeader_CTspecific).ct_bh_correction_coefficients[3] = float.Parse(splitted[3].Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_aluminum_filter_thickness ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_aluminum_filter_thickness = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "projection ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                        MicroPETHeader_CTspecific.Projection projection = new MicroPETHeader_CTspecific.Projection();
                        if (splitted.Length > 0)
                            projection.projection_number = int.Parse(splitted[0]);
                        if (splitted.Length > 1)
                            projection.data = float.Parse(splitted[1].Replace('.', ','));
                        projections.Add(projection);
                    }
                    else if ((strippedline = ParseField(line, "ct_projection_average_center_offset ")) != null)
                    {
                        (hdr as MicroPETHeader_CTspecific).ct_projection_average_center_offset = float.Parse(strippedline.Replace('.', ','));
                    }
                    else if ((strippedline = ParseField(line, "ct_projection_center_offset ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                        MicroPETHeader_CTspecific.Projection ct_projection_center_offset = new MicroPETHeader_CTspecific.Projection();
                        if (splitted.Length > 0)
                            ct_projection_center_offset.projection_number = int.Parse(splitted[0]);
                        if (splitted.Length > 1)
                            ct_projection_center_offset.data = float.Parse(splitted[1].Replace('.', ','));
                        ct_projection_center_offsets.Add(ct_projection_center_offset);
                    }
                    else if ((strippedline = ParseField(line, "ct_projection_horizontal_bed_offset ")) != null)
                    {
                        string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                        MicroPETHeader_CTspecific.Projection ct_projection_horizontal_bed_offset = new MicroPETHeader_CTspecific.Projection();
                        if (splitted.Length > 0)
                            ct_projection_horizontal_bed_offset.projection_number = int.Parse(splitted[0]);
                        if (splitted.Length > 1)
                            ct_projection_horizontal_bed_offset.data = float.Parse(splitted[1].Replace('.', ','));
                        ct_projection_horizontal_bed_offsets.Add(ct_projection_horizontal_bed_offset);
                    }
                    else if (line.StartsWith("end_of_header")) {
                        break;
                    }
                    #endregion
                } else if(line.StartsWith("end_of_header")) {
                    break;
                }
                #endregion
            }
            if (hdr is MicroPETHeader_CTspecific) {
                (hdr as MicroPETHeader_CTspecific).projections = projections.ToArray();
                (hdr as MicroPETHeader_CTspecific).ct_projection_center_offsets = ct_projection_center_offsets.ToArray();
                (hdr as MicroPETHeader_CTspecific).ct_projection_horizontal_bed_offsets = ct_projection_horizontal_bed_offsets.ToArray();
            }
            return hdr;
        }
        /// <summary>
        /// Reads one MicroPET subheader from stream
        /// </summary>
        /// <param name="reader">open stream for reading</param>
        /// <returns>next subheader in stream</returns>
        protected MicroPETSubHeader ReadMicroPETSubheader(StreamReader reader) {
            MicroPETSubHeader hdr = new MicroPETSubHeader();
            string line;
            string strippedline;
            List<MicroPETSubHeader.Singles_rate> singles = new List<MicroPETSubHeader.Singles_rate>();
            //read until end of header tag is found or file ends
            while (!reader.EndOfStream)
            {
                line = reader.ReadLine();
                if (line.StartsWith("#")) continue;
                if ((strippedline = ParseField(line, "frame ")) != null)
                {
                    hdr.frame = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "detector_panel ")) != null)
                {
                    hdr.detector_panel = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "event_type ")) != null)
                {
                    hdr.event_type = (MicroPETSubHeader.Event_type)int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "energy_window ")) != null)
                {
                    hdr.energy_window = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "gate ")) != null)
                {
                    hdr.gate = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "bed ")) != null)
                {
                    hdr.bed = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "bed_offset ")) != null)
                {
                    hdr.bed_offset = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "ending_bed_offset ")) != null)
                {
                    hdr.ending_bed_offset = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "bed_passes ")) != null)
                {
                    hdr.bed_passes = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "vertical_bed_offset ")) != null)
                {
                    hdr.vertical_bed_offset = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "data_file_pointer ")) != null)
                {
                    string[] splitted = strippedline.Split(new char[] {' '}, StringSplitOptions.RemoveEmptyEntries);
                    if(splitted.Length > 0) hdr.data_file_pointer[0] = int.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.data_file_pointer[1] = int.Parse(splitted[1]);
                }
                else if ((strippedline = ParseField(line, "frame_start ")) != null)
                {
                    hdr.frame_start = int.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "frame_duration ")) != null)
                {
                    hdr.frame_duration = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "scale_factor ")) != null)
                {
                    hdr.scale_factor = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "minimum ")) != null)
                {
                    hdr.minimum = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "maximum ")) != null)
                {
                    hdr.maximum = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "deadtime_correction ")) != null)
                {
                    hdr.deadtime_correction = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "decay_correction ")) != null)
                {
                    hdr.decay_correction = float.Parse(strippedline.Replace('.', ','));
                }
                else if ((strippedline = ParseField(line, "prompts ")) != null)
                {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.prompts[0] = long.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.prompts[1] = long.Parse(splitted[1]);
                    if (splitted.Length > 2) hdr.prompts[2] = long.Parse(splitted[2]);
                }
                else if ((strippedline = ParseField(line, "prompts_rate ")) != null)
                {
                    hdr.prompts_rate = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "delays ")) != null)
                {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.delays[0] = long.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.delays[1] = long.Parse(splitted[1]);
                    if (splitted.Length > 2) hdr.delays[2] = long.Parse(splitted[2]);
                }
                else if ((strippedline = ParseField(line, "trues ")) != null)
                {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    if (splitted.Length > 0) hdr.trues[0] = long.Parse(splitted[0]);
                    if (splitted.Length > 1) hdr.trues[1] = long.Parse(splitted[1]);
                    if (splitted.Length > 2) hdr.trues[2] = long.Parse(splitted[2]);
                }
                else if ((strippedline = ParseField(line, "delays_rate ")) != null)
                {
                    hdr.delays_rate = int.Parse(strippedline);
                }
                else if ((strippedline = ParseField(line, "singles ")) != null)
                {
                    string[] splitted = strippedline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    MicroPETSubHeader.Singles_rate single = new MicroPETSubHeader.Singles_rate();
                    if (splitted.Length > 0) single.block_number = int.Parse(splitted[0]);
                    if (splitted.Length > 1) single.singles_per_sec = float.Parse(splitted[1].Replace('.', ','));
                    if (splitted.Length > 2) single.raw_singles_per_sec = float.Parse(splitted[2].Replace('.', ','));
                    if (splitted.Length > 3) single.xy_stored_singles_per_sec = float.Parse(splitted[3].Replace('.', ','));
                    if (splitted.Length > 4) single.block_singles_output_per_sec = float.Parse(splitted[4].Replace('.', ','));
                    if (splitted.Length > 5) single.raw_block_prompts_per_sec = float.Parse(splitted[5].Replace('.', ','));
                    if (splitted.Length > 6) single.block_prompts_per_sec = float.Parse(splitted[6].Replace('.', ','));
                    if (splitted.Length > 7) single.raw_block_delays_per_sec = float.Parse(splitted[7].Replace('.', ','));
                    if (splitted.Length > 8) single.block_delays_per_sec = float.Parse(splitted[8].Replace('.', ','));
                    singles.Add(single);
                } else if(line.StartsWith("end_of_header")) {
                    break;
                }
            }
            hdr.singles = singles.ToArray();
            return hdr;
        }
        /// <summary>
        /// Writes MicroPET main header data into header file
        /// </summary>
        protected void WriteHeader() {

            // update dimension information
            // from the general header structure
            switch (header.datatype)
            {
                case DataType.BIT8_S:
                case DataType.BIT8_U:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Byte;
                    break;
                case DataType.BIT16_S:
                case DataType.BIT16_U:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Intel_2byte_Int;
                    break;
                case DataType.SUNI2:
                case DataType.VAXI16:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Sun_2byte_Int;
                    break;
                case DataType.BIT32_S:
                case DataType.BIT32_U:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Intel_4byte_Int;
                    break;
                case DataType.FLT32:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Intel_4byte_Float;
                    break;
                case DataType.SUNI4:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Sun_4byte_Int;
                    break;
                case DataType.VAXFL32:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Sun_4byte_Float;
                    break;
                default:
                    microPETHeader.data_type = MicroPETHeader.Data_type.Unknown;
                    break;
            }
            microPETHeader.dose_units = MicroPETHeader.Radioactivity_units.Unknown;
            microPETHeader.subject_identifier = header.patientID;
            microPETHeader.study = header.description;
            microPETHeader.x_dimension = image.dimx;
            microPETHeader.y_dimension = image.dimy;
            microPETHeader.z_dimension = image.dimz;
            if (image is DynamicImage) {
                microPETHeader.time_frames = (image as DynamicImage).frames;
            }
            if (header is DynamicImageHeader)
            {
                microPETHeader.injection_time = (header as DynamicImageHeader).dose_start_time.ToString("ddd MMM d hh:mm:ss.fff yyyy", System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
                microPETHeader.isotope = (header as DynamicImageHeader).radiopharma;
            }
            microPETHeader.pixel_size_x = header.sizex;
            microPETHeader.pixel_size_y = header.sizey;
            microPETHeader.pixel_size_z = header.sizez;
            switch(header.modality) {
                case ImageModality.M_CT:
                    microPETHeader.modality = MicroPETHeader.scanner_modality.CT;
                    break;
                case ImageModality.M_PT:
                    microPETHeader.modality = MicroPETHeader.scanner_modality.PET;
                    break;
                case ImageModality.M_ST:
                    microPETHeader.modality = MicroPETHeader.scanner_modality.SPECT;
                    break;
                default:
                    microPETHeader.modality = MicroPETHeader.scanner_modality.Unknown;
                    break;
            }

            #region try_to_fill_rest_of_the_fields_from_general_field_structure
            try { microPETHeader.version = (float)header["version"]; }
            catch (Exception) {}
            try { microPETHeader.manufacturer = (string)header["manufacturer"]; }
            catch (Exception) {}
            try { microPETHeader.model = (MicroPETHeader.model_number)header["model"]; }
            catch (Exception) {}
            try { microPETHeader.modality = (MicroPETHeader.scanner_modality)header["modality"]; }
            catch (Exception) {}
            try { microPETHeader.modality_configuration = (MicroPETHeader.Modality_configuration)header["modality_configuration"]; }
            catch (Exception) {}
            try { microPETHeader.institution = (string)header["institution"]; }
            catch (Exception) {}
            try { microPETHeader.study = (string)header["study"]; }
            catch (Exception) {}
            try { microPETHeader.file_name = (string)header["file_name"]; }
            catch (Exception) {}
            try { microPETHeader.file_type = (MicroPETHeader.Data_file_type)header["file_type"]; }
            catch (Exception) {}
            try { microPETHeader.acquisition_mode = (MicroPETHeader.Acquisition_mode)header["acquisition_mode"]; }
            catch (Exception) {}
            try { microPETHeader.total_frames = (int)header["total_frames"]; }
            catch (Exception) {}
            try { microPETHeader.time_frames = (int)header["time_frames"]; }
            catch (Exception) {}
            try { microPETHeader.bed_control = (MicroPETHeader.Bed_Control)header["bed_control"]; }
            catch (Exception) {}
            try { microPETHeader.bed_motion = (MicroPETHeader.Bed_motion)header["bed_motion"]; }
            catch (Exception) {}
            try { microPETHeader.number_of_bed_positions = (int)header["number_of_bed_positions"]; }
            catch (Exception) {}
            try { microPETHeader.horizontal_bed_calibration = (float)header["horizontal_bed_calibration"]; }
            catch (Exception) {}
            try { microPETHeader.vertical_bed_calibration = (float)header["vertical_bed_calibration"]; }
            catch (Exception) {}
            try { microPETHeader.transaxial_bin_size = (float)header["transaxial_bin_size"]; }
            catch (Exception) {}
            try { microPETHeader.axial_plane_size = (float)header["axial_plane_size"]; }
            catch (Exception) {}
            try { microPETHeader.number_detector_panels = (int)header["number_detector_panels"]; }
            catch (Exception) {}
            try { microPETHeader.data_type = (MicroPETHeader.Data_type)header["data_type"]; }
            catch (Exception) {}
            try { microPETHeader.number_of_dimensions = (int)header["number_of_dimensions"]; }
            catch (Exception) {}
            try { microPETHeader.x_dimension = (int)header["x_dimension"]; }
            catch (Exception) {}
            try { microPETHeader.y_dimension = (int)header["y_dimension"]; }
            catch (Exception) {}
            try { microPETHeader.z_dimension = (int)header["z_dimension"]; }
            catch (Exception) {}
            try { microPETHeader.acquisition_user_id = (string)header["acquisition_user_id"]; }
            catch (Exception) {}
            try { microPETHeader.histogram_user_id = (string)header["histogram_user_id"]; }
            catch (Exception) {}
            try { microPETHeader.reconstruction_user_id = (string)header["reconstruction_user_id"]; }
            catch (Exception) {}
            try { microPETHeader.scatter_correction_user_id = (string)header["scatter_correction_user_id"]; }
            catch (Exception) {}
            try { microPETHeader.acquisition_notes = (string)header["acquisition_notes"]; }
            catch (Exception) {}
            try { microPETHeader.scan_time = (string)header["scan_time"]; }
            catch (Exception) {}
            try { microPETHeader.gmt_scan_time = (string)header["gmt_scan_time"]; }
            catch (Exception) {}
            try { microPETHeader.volume_origin_x = (int)header["volume_origin_x"]; }
            catch (Exception) {}
            try { microPETHeader.volume_origin_y = (int)header["volume_origin_y"]; }
            catch (Exception) {}
            try { microPETHeader.volume_origin_z = (int)header["volume_origin_z"]; }
            catch (Exception) {}
            try { microPETHeader.registration_available = (MicroPETHeader.Registration_data_availability)header["registration_available"]; }
            catch (Exception) {}
            try { microPETHeader.transformation_matrix = (string)header["transformation_matrix"]; }
            catch (Exception) {}
            try { microPETHeader.spatial_identifier = (string)header["spatial_identifier"]; }
            catch (Exception) {}
            try { microPETHeader.normalization_applied = (MicroPETHeader.Normalization_type)header["normalization_applied"]; }
            catch (Exception) {}
            try { microPETHeader.normalization_filename = (string)header["normalization_filename"]; }
            catch (Exception) {}
            try { microPETHeader.recon_algorithm = (MicroPETHeader.Reconstruction_type)header["recon_algorithm"]; }
            catch (Exception) {}
            try { microPETHeader.recon_version = (float)header["recon_version"]; }
            catch (Exception) {}
            try { microPETHeader.y_filter_type = (MicroPETHeader.Filter_type)header["x_filter"]; }
            catch (Exception) {}
            try { microPETHeader.y_filter_type = (MicroPETHeader.Filter_type)header["y_filter"]; }
            catch (Exception) {}
            try { microPETHeader.y_filter_type = (MicroPETHeader.Filter_type)header["z_filter"]; }
            catch (Exception) {}
            try { microPETHeader.x_offset = (float)header["x_offset"]; }
            catch (Exception) {}
            try { microPETHeader.y_offset = (float)header["y_offset"]; }
            catch (Exception) {}
            try { microPETHeader.zoom = (float)header["zoom"]; }
            catch (Exception) {}
            try { microPETHeader.pixel_size = (float)header["pixel_size"]; }
            catch (Exception) {}
            try { microPETHeader.pixel_size_x = (float)header["pixel_size_x"]; }
            catch (Exception) {}
            try { microPETHeader.pixel_size_y = (float)header["pixel_size_y"]; }
            catch (Exception) {}
            try { microPETHeader.pixel_size_z = (float)header["pixel_size_z"]; }
            catch (Exception) {}
            try { microPETHeader.investigator_id = (string)header["investigator"]; }
            catch (Exception) {}
            try { microPETHeader.operator_id = (string)header["operator"]; }
            catch (Exception) {}
            try { microPETHeader.study_identifier = (string)header["study_identifier"]; }
            catch (Exception) {}
            try { microPETHeader.injected_compound = (string)header["injected_compound"]; }
            catch (Exception) {}
            try { microPETHeader.subject_identifier = (string)header["subject_identifier"]; }
            catch (Exception) {}
            try { microPETHeader.subject_genus = (string)header["subject_genus"]; }
            catch (Exception) {}
            try { microPETHeader.subject_orientation = (MicroPETHeader.Subject_orientation)header["subject_orientation"]; }
            catch (Exception) {}
            try { microPETHeader.subject_length_units = (MicroPETHeader.Length_unit)header["subject_length_units"]; }
            catch (Exception) {}
            try { microPETHeader.subject_length = (float)header["subject_length"]; }
            catch (Exception) {}
            try { microPETHeader.subject_weight_units = (MicroPETHeader.Weight_unit)header["subject_weight_units"]; }
            catch (Exception) {}
            try { microPETHeader.subject_weight = (float)header["subject_weight"]; }
            catch (Exception) {}
            try { microPETHeader.subject_phenotype = (string)header["subject_phenotype"]; }
            catch (Exception) {}
            try { microPETHeader.study_model = (string)header["study_model"]; }
            catch (Exception) {}
            try { microPETHeader.anesthesia = (string)header["anesthesia"]; }
            catch (Exception) {}
            try { microPETHeader.analgesia = (string)header["analgesia"]; }
            catch (Exception) {}
            try { microPETHeader.other_drugs = (string)header["other_drugs"]; }
            catch (Exception) {}
            try { microPETHeader.food_access = (string)header["food_access"]; }
            catch (Exception) {}
            try { microPETHeader.water_access = (string)header["water_access"]; }
            catch (Exception) {}
            try { microPETHeader.subject_date_of_birth = (string)header["subject_date_of_birth"]; }
            catch (Exception) {}
            try { microPETHeader.subject_age = (string)header["subject_age"]; }
            catch (Exception) {}
            try { microPETHeader.subject_sex = (string)header["subject_sex"]; }
            catch (Exception) {}
            try { microPETHeader.subject_scan_region = (string)header["subject_scan_region"]; }
            catch (Exception) {}
            try { microPETHeader.acquisition_file_name = (string)header["acquisition_file_name"]; }
            catch (Exception) {}
            try { microPETHeader.transaxial_crystal_pitch = (float)header["transaxial_crystal_pitch"]; }
            catch (Exception) {}
            try { microPETHeader.axial_crystal_pitch = (float)header["axial_crystal_pitch"]; }
            catch (Exception) {}
            try { microPETHeader.crystal_thickness = (float)header["crystal_thickness"]; }
            catch (Exception) {}
            try { microPETHeader.depth_of_interaction = (float)header["depth_of_interaction"]; }
            catch (Exception) {}
            try { microPETHeader.isotope = (string)header["isotope"]; }
            catch (Exception) {}
            try { microPETHeader.isotope_half_life = (float)header["isotope_half_life"]; }
            catch (Exception) {}
            try { microPETHeader.isotope_branching_fraction = (float)header["isotope_branching_fraction"]; }
            catch (Exception) {}
            try { microPETHeader.deadtime_correction_applied = (MicroPETHeader.Deadtime_correction)header["deadtime_correction_applied"]; }
            catch (Exception) {}
            try { microPETHeader.decay_correction_applied = (bool)header["decay_correction_applied"]; }
            catch (Exception) {}
            try { microPETHeader.attenuation_applied = (MicroPETHeader.Attenuation_correction)header["attenuation_applied"]; }
            catch (Exception) {}
            try { microPETHeader.attenuation_filename = (string)header["attenuation_filename"]; }
            catch (Exception) {}
            try { microPETHeader.scatter_correction = (MicroPETHeader.Scatter_correction)header["scatter_correction"]; }
            catch (Exception) {}
            try { microPETHeader.calibration_units = (MicroPETHeader.Calibration_units)header["calibration_units"]; }
            catch (Exception) {}
            try { microPETHeader.calibration_factor = (float)header["calibration_factor"]; }
            catch (Exception) {}
            try { microPETHeader.calibration_branching_fraction = (float)header["calibration_branching_fraction"]; }
            catch (Exception) {}
            try { microPETHeader.dose_units = (MicroPETHeader.Radioactivity_units)header["dose_units"]; }
            catch (Exception) {}
            try { microPETHeader.dose = (float)header["dose"]; }
            catch (Exception) {}
            try { microPETHeader.injection_decay_correction = (float)header["injection_decay_correction"]; }
            catch (Exception) {}
            try { microPETHeader.activity_units = (MicroPETHeader.Radioactivity_units)header["activity_units"]; }
            catch (Exception) {}
            try { microPETHeader.activity_before_injection = (float)header["activity_before_injection"]; }
            catch (Exception) {}
            try { microPETHeader.activity_before_injection_time = (string)header["activity_before_injection_time"]; }
            catch (Exception) {}
            try { microPETHeader.residual_activity = (float)header["residual_activity"]; }
            catch (Exception) {}
            try { microPETHeader.residual_activity_time = (string)header["residual_activity_time"]; }
            catch (Exception) {}
            try { microPETHeader.transaxial_crystals_per_block = (int)header["transaxial_crystals_per_block"]; }
            catch (Exception) {}
            try { microPETHeader.axial_crystals_per_block = (int)header["axial_crystals_per_block"]; }
            catch (Exception) {}
            try { microPETHeader.intrinsic_crystal_offset = (int)header["intrinsic_crystal_offset"]; }
            catch (Exception) {}
            try { microPETHeader.transaxial_blocks = (int)header["transaxial_blocks"]; }
            catch (Exception) {}
            try { microPETHeader.axial_blocks = (int)header["axial_blocks"]; }
            catch (Exception) {}
            try { microPETHeader.radius = (float)header["radius"]; }
            catch (Exception) {}
            try { microPETHeader.radial_fov = (float)header["radial_fov"]; }
            catch (Exception) {}
            try { microPETHeader.src_radius = (float)header["src_radius"]; }
            catch (Exception) {}
            try { microPETHeader.src_cm_per_rev = (float)header["src_cm_per_rev"]; }
            catch (Exception) {}
            try { microPETHeader.src_steps_per_rev = (int)header["src_steps_per_rev"]; }
            catch (Exception) {}
            try { microPETHeader.default_projections = (int)header["default_projections"]; }
            catch (Exception) {}
            try { microPETHeader.default_transaxial_angles = (int)header["default_transaxial_angles"]; }
            catch (Exception) {}
            try { microPETHeader.lld = (float)header["lld"]; }
            catch (Exception) {}
            try { microPETHeader.uld = (float)header["uld"]; }
            catch (Exception) {}
            try { microPETHeader.timing_window = (float)header["timing_window"]; }
            catch (Exception) {}
            try { microPETHeader.span = (int)header["span"]; }
            catch (Exception) {}
            try { microPETHeader.ring_difference = (int)header["ring_difference"]; }
            catch (Exception) {}
            try { microPETHeader.w_dimension = (int)header["w_dimension"]; }
            catch (Exception) {}
            try { microPETHeader.histogram_version = (float)header["histogram_version"]; }
            catch (Exception) {}
            try { microPETHeader.rebinning_type = (MicroPETHeader.Rebinning)header["rebinning_type"]; }
            catch (Exception) {}
            try { microPETHeader.rebinning_version = (float)header["rebinning_version"]; }
            catch (Exception) {}
            try { microPETHeader.tx_src_type = (MicroPETHeader.Source_type)header["tx_src_type"]; }
            catch (Exception) {}
            try { microPETHeader.data_order = (MicroPETHeader.Data_order)header["data_order"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_method = (MicroPETHeader.OSEM_method)header["osem2d_method"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_subsets = (int)header["osem2d_subsets"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_iterations = (int)header["osem2d_iterations"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_em_iterations = (int)header["osem2d_em_iterations"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_map_E = (float)header["osem2d_map_E"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_map_P = (int)header["osem2d_map_P"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_x_offset = (float)header["osem2d_x_offset"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_y_offset = (float)header["osem2d_y_offset"]; }
            catch (Exception) {}
            try { microPETHeader.osem2d_zoom = (float)header["osem2d_zoom"]; }
            catch (Exception) {}
            try { microPETHeader.arc_correction_applied = (MicroPETHeader.Arc_correction)header["arc_correction_applied"]; }
            catch (Exception) {}
            try { microPETHeader.number_of_singles_rates = (int)header["number_of_singles_rates"]; }
            catch (Exception) {}
            try { microPETHeader.subject_glucose_level = (string)header["subject_glucose_level"]; }
            catch (Exception) {}
            try { microPETHeader.subject_glucose_level_time = (string)header["subject_glucose_level_time"]; }
            catch (Exception) {}            
            #endregion

            System.IO.StreamWriter hdrfile;
            try
            {
                hdrfile = new System.IO.StreamWriter(filename + ".img.hdr", false);
            }
            catch (Exception e) {
                throw new TPCMicroPETFileException("Failed to open header file "+filename+".hdr for writing:"+e);
            }
            MicroPETFile.WriteMicroPETMainheader(hdrfile, microPETHeader);
            //resolve frame data from dynamic image
            if (image is DynamicImage) {
                float[] start_times = (image as DynamicImage).GetFrameStartTimes();
                float[] durations = (image as DynamicImage).GetFrameDurations();
                if ((image as DynamicImage).frames > microPETSubheaders.Length) {
                    List<MicroPETSubHeader> subhdrlist = new List<MicroPETSubHeader>(microPETSubheaders);
                    for (int i = microPETSubheaders.Length; i < (image as DynamicImage).frames; i++)
                        subhdrlist.Add(new MicroPETSubHeader());
                    microPETSubheaders = subhdrlist.ToArray();
                }
                for (int i = 0; i < (image as DynamicImage).frames; i++) {
                    microPETSubheaders[i].frame_start = (int)(start_times[i]/1000.0f);
                    microPETSubheaders[i].frame_duration = (int)(durations[i]/1000.0f);
                    microPETSubheaders[i].frame = i;
                }
            }
            if(microPETSubheaders.Length > 0) {
                for (int i = 0; i < microPETSubheaders.Length; i++) {
                    if(i == 0) MicroPETFile.WriteMicroPETSubheader(hdrfile, microPETSubheaders[i], true, microPETHeader.modality);
                    else MicroPETFile.WriteMicroPETSubheader(hdrfile, microPETSubheaders[i], false, microPETHeader.modality);
                }
            }
            hdrfile.Close();
        }
        /// <summary>
        /// Checks file format. 
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a MicroPET file</returns>
        /// <exception cref="TPCMicroPETFileException">if there was a read problem</exception>
        public static bool CheckFormat(string filename)
        {
            String ext = Path.GetExtension(filename);
            String basename = Path.GetFileNameWithoutExtension(filename);
            if(basename.EndsWith(".img"))
                basename = Path.GetFileNameWithoutExtension(basename);
            String path = Path.GetDirectoryName(filename);
            //files filename and filename.hdr must exist
            if (!File.Exists(path + Path.DirectorySeparatorChar + basename + ".img") ||
                !File.Exists(path + Path.DirectorySeparatorChar + basename + ".img.hdr"))
            {
                return false;
            }
            //try to read MicroPET main header data
            try
            {
                MicroPETFile file = new MicroPETFile(path + Path.DirectorySeparatorChar + basename);
                file.ReadHeader();
            }
            catch (FileNotFoundException)
            {
                throw new TPCAnalyzeFileException("Header file was not found");
            }
            catch (IOException e)
            {
                throw new TPCMicroPETFileException("Read error while determining MicroPET file format:" + e);
            }
            //file format is not recognized if TPCAnalyzeFileException is thrown
            catch (TPCMicroPETFileException)
            {
                return false;
            }
            catch (Exception e)
            {
                throw new TPCMicroPETFileException("Unknown error while determining MicroPET file format:" + e);
            }
            return true;
        }
        /// <summary>
        /// Writes single frame into file. If frame number is larger than 
        /// current number of frames in file, then file size is grown.
        /// </summary>
        /// <remarks>The header information including scaling factor is not updated by this method.</remarks>
        /// <param name="image">image data that is written</param>
        /// <param name="frame_No">frame number that is written [1..no frames+1]</param>
        public override void WriteFrame(ref Image image, int frame_No)
        {
            if (frame_No < 1 || frame_No > this.microPETHeader.time_frames + 1)
                throw new TPCInvalidArgumentsException("Frame number " + frame_No + " out of bounds [1..no frames +1]");

            string fname = filename + ".img";
            FileStream stream = File.Open(fname, FileMode.OpenOrCreate);
            BinaryWriter writer = new BinaryWriter(stream);
            //go to start position of frame
            try
            {
                writer.Seek(ImageFile.BytesPerPixel(header.datatype) * (frame_No - 1), SeekOrigin.Begin);
            }
            catch (Exception e) {
                writer.Close();
                throw new TPCMicroPETFileException("Failed to seek frame start position in data file:"+e);
            }

            switch (this.microPETHeader.data_type)
            {
                case MicroPETHeader.Data_type.Intel_4byte_Float:
                    {
                        if (this.microPETHeader.data_order == MicroPETHeader.Data_order.View)
                            throw new TPCMicroPETFileException("Only XYZW data order is currently supported for writing.");
                        for (int i = 0; i < image.dataLength; i++)
                        {
                            writer.Write(image[i]);
                        }
                        break;
                    }
                default:
                    writer.Close();
                    throw new TPCMicroPETFileException("Unsupported data type in MicroPET for writing: " + microPETHeader.data_type);
            }
            writer.Close();
        }
        /// <summary>
        /// Reads only header information from file. 
        /// </summary>
        /// <returns>Header information in file</returns>
        public override ImageHeader ReadHeader()
        {
            StreamReader reader;
            try {
                FileStream stream = new FileStream(filename+".img.hdr", FileMode.Open);
                reader = new StreamReader(stream);
            } catch(Exception e) {
                throw new TPCMicroPETFileException("Cannot open file "+filename+".img.hdr for reading:"+e);
            }
            try
            {
                microPETHeader = ReadMicroPETMainheader(reader);
            }
            catch (Exception e) {
                reader.Close();
                throw new TPCMicroPETFileException("Failed to read main header:"+e);
            }
            List<MicroPETSubHeader> subheaders = new List<MicroPETSubHeader>();
            while(!reader.EndOfStream) {
                subheaders.Add(ReadMicroPETSubheader(reader));
            }
            reader.Close(); 
            microPETSubheaders = subheaders.ToArray();
            
            //fill header structure
            header = new ImageHeader();
            header.dim = new IntLimits(microPETHeader.x_dimension, microPETHeader.y_dimension, microPETHeader.z_dimension);
            if (microPETHeader.time_frames > 0)
            {
                header = new DynamicImageHeader(header);
                header.dim.AddLimit(0, microPETHeader.time_frames);
                for (int i = 0; i < microPETHeader.time_frames; i++) {
                    switch (microPETHeader.dose_units) {
                        case MicroPETHeader.Radioactivity_units.MBq:
                            (header as DynamicImageHeader).injected_dose = microPETHeader.dose;
                            break;
                        case MicroPETHeader.Radioactivity_units.mCi:
                            (header as DynamicImageHeader).injected_dose = microPETHeader.dose/1000000000.0f;
                            break;
                        //set value as null if unit type is not recognized
                        default:
                            (header as DynamicImageHeader).injected_dose = 0.0f;
                            break;
                    }
                }
                try
                {
                    if (microPETHeader.injection_time.Trim().Length > 0)
                        (header as DynamicImageHeader).dose_start_time = System.DateTime.ParseExact(microPETHeader.injection_time, "ddd MMM d hh:mm:ss.fff yyyy", System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
                    else
                        (header as DynamicImageHeader).dose_start_time = new System.DateTime(0);
                }
                catch (FormatException e) {
                    try
                    {
                        (header as DynamicImageHeader).dose_start_time = System.DateTime.ParseExact(microPETHeader.injection_time, "ddd MMM d hh:mm:ss yyyy", System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
                    
                    }
                    catch
                    {
                        try
                        {
                            (header as DynamicImageHeader).dose_start_time = System.DateTime.Parse(microPETHeader.injection_time, System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
                        }
                        catch
                        {
                            throw new TPCMicroPETFileException("Injection start time field [" + microPETHeader.injection_time + "] was not in correct format:" + e);
                        }
                    }
                }
                (header as DynamicImageHeader).isotope = Isotope.CreateIsotope(microPETHeader.isotope);
                (header as DynamicImageHeader).radiopharma = microPETHeader.isotope;
            }
            
            header.description = microPETHeader.study;
            switch(microPETHeader.data_type) {
                case MicroPETHeader.Data_type.Intel_2byte_Int:
                    header.datatype = DataType.BIT16_S;
                    break;
                case MicroPETHeader.Data_type.Intel_4byte_Float:
                default:
                    header.datatype = DataType.FLT32;
                    break;
            }
            switch(microPETHeader.dose_units) {
                case MicroPETHeader.Radioactivity_units.MBq:
                    header.dataunit = Data_unit.MBQ_per_ML;
                    break;
                case MicroPETHeader.Radioactivity_units.mCi:
                default:
                    header.dataunit = Data_unit.Unknown;
                    break;
            }
            header.patient_name = "";
            header.patientID = microPETHeader.subject_identifier;
            header.siz = new Voxel(microPETHeader.pixel_size_x, microPETHeader.pixel_size_y, microPETHeader.pixel_size_z);
            switch(microPETHeader.modality) {
                case MicroPETHeader.scanner_modality.CT:
                    header.modality = ImageModality.M_CT;
                    break;
                case MicroPETHeader.scanner_modality.PET:
                    header.modality = ImageModality.M_PT;
                    break;
                case MicroPETHeader.scanner_modality.SPECT:
                    header.modality = ImageModality.M_ST;
                    break;
                default:
                    header.modality = ImageModality.Unknown;
                    break;
            }
            
            //add only mainheader and 1st subheader into fields array
            #region fill_field_array
            header.Add("z_dimension", microPETHeader.z_dimension);
            header.Add("acquisition_user_id", microPETHeader.acquisition_user_id);
            header.Add("histogram_user_id", microPETHeader.histogram_user_id);
            header.Add("reconstruction_user_id", microPETHeader.reconstruction_user_id);
            header.Add("scatter_correction_user_id", microPETHeader.scatter_correction_user_id);
            header.Add("acquisition_notes", microPETHeader.acquisition_notes);
            header.Add("scan_time", microPETHeader.scan_time);
            header.Add("gmt_scan_time", microPETHeader.gmt_scan_time);
            header.Add("volume_origin_x", microPETHeader.volume_origin_x);
            header.Add("volume_origin_y", microPETHeader.volume_origin_y);
            header.Add("volume_origin_z", microPETHeader.volume_origin_z);
            header.Add("registration_available", microPETHeader.registration_available);
            header.Add("transformation_matrix", microPETHeader.transformation_matrix);
            header.Add("spatial_identifier", microPETHeader.spatial_identifier);
            header.Add("normalization_applied", microPETHeader.normalization_applied);
            header.Add("normalization_filename", microPETHeader.normalization_filename);
            header.Add("recon_algorithm", microPETHeader.recon_algorithm);
            header.Add("recon_version", microPETHeader.recon_version);
            header.Add("x_filter", microPETHeader.x_filter_type + " " + microPETHeader.x_filter_cutoff);
            header.Add("y_filter", microPETHeader.y_filter_type + " " + microPETHeader.y_filter_cutoff);
            header.Add("z_filter", microPETHeader.z_filter_type + " " + microPETHeader.z_filter_cutoff);
            header.Add("x_offset", microPETHeader.x_offset);
            header.Add("y_offset", microPETHeader.y_offset);
            header.Add("zoom", microPETHeader.zoom);
            header.Add("pixel_size", microPETHeader.pixel_size);
            header.Add("pixel_size_x", microPETHeader.pixel_size_x);
            header.Add("pixel_size_y", microPETHeader.pixel_size_y);
            header.Add("pixel_size_z", microPETHeader.pixel_size_z);
            header.Add("investigator", microPETHeader.investigator_id);
            header.Add("operator", microPETHeader.operator_id);
            header.Add("study_identifier", microPETHeader.study_identifier);
            header.Add("injected_compound", microPETHeader.injected_compound);
            header.Add("subject_identifier", microPETHeader.subject_identifier);
            header.Add("subject_genus", microPETHeader.subject_genus);
            header.Add("subject_orientation", microPETHeader.subject_orientation);
            header.Add("subject_length_units", microPETHeader.subject_length_units);
            header.Add("subject_length", microPETHeader.subject_length);
            header.Add("subject_weight_units", microPETHeader.subject_weight_units);
            header.Add("subject_weight", microPETHeader.subject_weight);
            header.Add("subject_phenotype", microPETHeader.subject_phenotype);
            header.Add("study_model", microPETHeader.study_model);
            header.Add("anesthesia", microPETHeader.anesthesia);
            header.Add("analgesia", microPETHeader.analgesia);
            header.Add("other_drugs", microPETHeader.other_drugs);
            header.Add("food_access", microPETHeader.food_access);
            header.Add("water_access", microPETHeader.water_access);
            header.Add("subject_date_of_birth", microPETHeader.subject_date_of_birth);
            header.Add("subject_age", microPETHeader.subject_age);
            header.Add("subject_sex", microPETHeader.subject_sex);
            header.Add("subject_scan_region", microPETHeader.subject_scan_region);
            header.Add("acquisition_file_name", microPETHeader.acquisition_file_name);
            header.Add("transaxial_crystal_pitch", microPETHeader.transaxial_crystal_pitch);
            header.Add("axial_crystal_pitch", microPETHeader.axial_crystal_pitch);
            header.Add("crystal_thickness", microPETHeader.crystal_thickness);
            header.Add("depth_of_interaction", microPETHeader.depth_of_interaction);
            header.Add("isotope", microPETHeader.isotope);
            header.Add("isotope_half_life", microPETHeader.isotope_half_life);
            header.Add("isotope_branching_fraction", microPETHeader.isotope_branching_fraction);
            header.Add("deadtime_correction_applied", microPETHeader.deadtime_correction_applied);
            header.Add("decay_correction_applied", microPETHeader.decay_correction_applied);
            header.Add("attenuation_applied", microPETHeader.attenuation_applied);
            header.Add("attenuation_filename", microPETHeader.attenuation_filename);
            header.Add("scatter_correction", microPETHeader.scatter_correction);
            header.Add("calibration_units", microPETHeader.calibration_units);
            header.Add("calibration_factor", microPETHeader.calibration_factor);
            header.Add("calibration_branching_fraction", microPETHeader.calibration_branching_fraction);
            header.Add("dose_units", microPETHeader.dose_units);
            header.Add("dose", microPETHeader.dose);
            header.Add("injection_time", microPETHeader.injection_time);
            header.Add("injection_decay_correction", microPETHeader.injection_decay_correction);
            header.Add("activity_units", microPETHeader.activity_units);
            header.Add("activity_before_injection", microPETHeader.activity_before_injection);
            header.Add("activity_before_injection_time", microPETHeader.activity_before_injection_time);
            header.Add("residual_activity", microPETHeader.residual_activity);
            header.Add("residual_activity_time", microPETHeader.residual_activity_time);
            header.Add("transaxial_crystals_per_block", microPETHeader.transaxial_crystals_per_block);
            header.Add("axial_crystals_per_block", microPETHeader.axial_crystals_per_block);
            header.Add("intrinsic_crystal_offset", microPETHeader.intrinsic_crystal_offset);
            header.Add("transaxial_blocks", microPETHeader.transaxial_blocks);
            header.Add("axial_blocks", microPETHeader.axial_blocks);
            header.Add("radius", microPETHeader.radius);
            header.Add("radial_fov", microPETHeader.radial_fov);
            header.Add("src_radius", microPETHeader.src_radius);
            header.Add("src_cm_per_rev", microPETHeader.src_cm_per_rev);
            header.Add("src_steps_per_rev", microPETHeader.src_steps_per_rev);
            header.Add("default_projections", microPETHeader.default_projections);
            header.Add("default_transaxial_angles", microPETHeader.default_transaxial_angles);
            header.Add("lld", microPETHeader.lld);
            header.Add("uld", microPETHeader.uld);
            header.Add("timing_window", microPETHeader.timing_window);
            header.Add("span", microPETHeader.span);
            header.Add("ring_difference", microPETHeader.ring_difference);
            header.Add("w_dimension", microPETHeader.w_dimension);
            header.Add("histogram_version", microPETHeader.histogram_version);
            header.Add("rebinning_type", microPETHeader.rebinning_type);
            header.Add("rebinning_version", microPETHeader.rebinning_version);
            header.Add("tx_src_type", microPETHeader.tx_src_type);
            header.Add("data_order", microPETHeader.data_order);
            header.Add("osem2d_method", microPETHeader.osem2d_method);
            header.Add("osem2d_subsets", microPETHeader.osem2d_subsets);
            header.Add("osem2d_iterations", microPETHeader.osem2d_iterations);
            header.Add("osem2d_em_iterations", microPETHeader.osem2d_em_iterations);
            header.Add("osem2d_map", "e="+microPETHeader.osem2d_map_E+" p="+microPETHeader.osem2d_map_P);
            header.Add("osem2d_x_offset", microPETHeader.osem2d_x_offset);
            header.Add("osem2d_y_offset", microPETHeader.osem2d_y_offset);
            header.Add("osem2d_zoom", microPETHeader.osem2d_zoom);
            header.Add("arc_correction_applied", microPETHeader.arc_correction_applied);
            header.Add("number_of_singles_rates", microPETHeader.number_of_singles_rates);
            header.Add("subject_glucose_level", microPETHeader.subject_glucose_level);
            header.Add("subject_glucose_level_time", microPETHeader.subject_glucose_level_time);
            if (microPETHeader is MicroPETHeader_CTspecific) {

            }
            if (microPETSubheaders.Length > 0 && (microPETHeader is MicroPETHeader_CTspecific))
            {
                header.Add("ct_file_version", (microPETHeader as MicroPETHeader_CTspecific).ct_file_version);
                header.Add("ct_header_size", (microPETHeader as MicroPETHeader_CTspecific).ct_header_size);
                header.Add("ct_proj_size_transaxial", (microPETHeader as MicroPETHeader_CTspecific).ct_proj_size_transaxial);
                header.Add("ct_proj_size_axial", (microPETHeader as MicroPETHeader_CTspecific).ct_proj_size_axial);
                header.Add("ct_average_dark_projections", (microPETHeader as MicroPETHeader_CTspecific).ct_average_dark_projections);
                header.Add("ct_average_light_projections", (microPETHeader as MicroPETHeader_CTspecific).ct_average_light_projections);
                header.Add("ct_light_calibration_projections", (microPETHeader as MicroPETHeader_CTspecific).ct_light_calibration_projections);
                header.Add("ct_dependent_light_calibration_projections", (microPETHeader as MicroPETHeader_CTspecific).ct_dependent_light_calibration_projections);
                header.Add("ct_xray_detector_offset", (microPETHeader as MicroPETHeader_CTspecific).ct_xray_detector_offset);
                header.Add("ct_detector_transaxial_position", (microPETHeader as MicroPETHeader_CTspecific).ct_detector_transaxial_position);
                header.Add("ct_uncropped_transaxial_pixels", (microPETHeader as MicroPETHeader_CTspecific).ct_uncropped_transaxial_pixels);
                header.Add("ct_uncropped_axial_pixels", (microPETHeader as MicroPETHeader_CTspecific).ct_uncropped_axial_pixels);
                header.Add("ct_cropped_transaxial_pixels", (microPETHeader as MicroPETHeader_CTspecific).ct_cropped_transaxial_pixels);
                header.Add("ct_cropped_axial_pixels", (microPETHeader as MicroPETHeader_CTspecific).ct_cropped_axial_pixels);
                header.Add("ct_xray_detector_pitch", (microPETHeader as MicroPETHeader_CTspecific).ct_xray_detector_pitch);
                header.Add("ct_horiz_rot_axis_bed_angle", (microPETHeader as MicroPETHeader_CTspecific).ct_horiz_rot_axis_bed_angle);
                header.Add("ct_vert_rot_axis_bed_angle", (microPETHeader as MicroPETHeader_CTspecific).ct_vert_rot_axis_bed_angle);
                header.Add("ct_exposure_time", (microPETHeader as MicroPETHeader_CTspecific).ct_exposure_time);
                header.Add("ct_scan_time", (microPETHeader as MicroPETHeader_CTspecific).ct_scan_time);
                header.Add("ct_warping", (microPETHeader as MicroPETHeader_CTspecific).ct_warping);
                header.Add("ct_defect_map_file_name", (microPETHeader as MicroPETHeader_CTspecific).ct_defect_map_file_name);
                header.Add("ct_xray_voltage", (microPETHeader as MicroPETHeader_CTspecific).ct_xray_voltage);
                header.Add("ct_anode_current", (microPETHeader as MicroPETHeader_CTspecific).ct_anode_current);
                header.Add("ct_calibration_exposures", (microPETHeader as MicroPETHeader_CTspecific).ct_calibration_exposures);
                header.Add("ct_cone_angle", (microPETHeader as MicroPETHeader_CTspecific).ct_cone_angle);
                header.Add("ct_projection_interpolation", (microPETHeader as MicroPETHeader_CTspecific).ct_projection_interpolation);
                header.Add("ct_source_to_detector", (microPETHeader as MicroPETHeader_CTspecific).ct_source_to_detector);
                header.Add("ct_source_to_crot", (microPETHeader as MicroPETHeader_CTspecific).ct_source_to_crot);
                header.Add("ct_detector_vertical_offset", (microPETHeader as MicroPETHeader_CTspecific).ct_detector_vertical_offset);
                header.Add("ct_detector_horizontal_tilt", (microPETHeader as MicroPETHeader_CTspecific).ct_detector_horizontal_tilt);
                header.Add("ct_detector_vertical_tilt", (microPETHeader as MicroPETHeader_CTspecific).ct_detector_vertical_tilt);
                header.Add("ct_transaxial_bin_factor", (microPETHeader as MicroPETHeader_CTspecific).ct_transaxial_bin_factor);
                header.Add("ct_axial_bin_factor", (microPETHeader as MicroPETHeader_CTspecific).ct_axial_bin_factor);
                header.Add("ct_gating", (microPETHeader as MicroPETHeader_CTspecific).ct_gating);
                header.Add("ct_hounsfield_scale", (microPETHeader as MicroPETHeader_CTspecific).ct_hounsfield_scale);
                header.Add("ct_hounsfield_offset", (microPETHeader as MicroPETHeader_CTspecific).ct_hounsfield_offset);
                header.Add("ct_proj_downsample_factor", (microPETHeader as MicroPETHeader_CTspecific).ct_proj_downsample_factor);
                header.Add("ct_first_recon_proj", (microPETHeader as MicroPETHeader_CTspecific).ct_first_recon_proj);
                header.Add("ct_last_recon_proj", (microPETHeader as MicroPETHeader_CTspecific).ct_last_recon_proj);
                header.Add("ct_recon_every_nth_proj", (microPETHeader as MicroPETHeader_CTspecific).ct_recon_every_nth_proj);
                header.Add("ct_attenuation_water", (microPETHeader as MicroPETHeader_CTspecific).ct_attenuation_water);
                header.Add("ct_tx_rotation_offsets", (microPETHeader as MicroPETHeader_CTspecific).ct_tx_rotation_offsets);
                header.Add("ct_tx_transaxial_offsets", (microPETHeader as MicroPETHeader_CTspecific).ct_tx_transaxial_offsets);
                header.Add("ct_bh_correction", (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction);
                header.Add("ct_bh_correction_coefficients", (microPETHeader as MicroPETHeader_CTspecific).ct_bh_correction_coefficients);
                header.Add("ct_aluminum_filter_thickness", (microPETHeader as MicroPETHeader_CTspecific).ct_aluminum_filter_thickness);
            }
            #endregion
            return header;
        }
        /// <summary>
        /// Gets pixel data from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write data</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input data dimensions</param>
        /// <param name="datatype">data type to be read</param>
        /// <param name="position">file position where reading is started (ignored)</param>
        /// <param name="scale">scale factor</param>
        public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {
            int datalength = dim.GetProduct();
            if (offset + datalength > data.dataLength)
                throw new TPCMicroPETFileException("Input data of length "+datalength+" does not fit into image of length "+data.dataLength);
            //initialization for event sending
            IOProcessEventHandler pevent = null;
            int voxels_in_plane = 0;
            //seek position
            try
            {
                stream.Seek(position, SeekOrigin.Begin);
            }
            catch (Exception e)
            {
                throw new TPCMicroPETFileException("Failed to seek starting position "+position+" from "+filename+":"+e);
            }

            //read data from position
            BinaryReader br = new BinaryReader(stream);
            GetValue getValue;
            switch(datatype) {
                case DataType.BIT16_U:
                    getValue = delegate()
                    {
                        return br.ReadUInt16();
                    };
                    break;
                case DataType.BIT16_S:
                    getValue = delegate()
                    {
                        return br.ReadInt16();
                    };
                    break;
                case DataType.FLT32:
                    getValue = delegate()
                    {
                        return br.ReadSingle();
                    };
                    break;
                default:
                    throw new TPCMicroPETFileException("Datatype is unsupported for reading with this method type: " + datatype);
            }
            // start reading data.
            voxels_in_plane = dim.dimx * dim.dimy;
            for (int i = 0; i < datalength; i++)
            {
                try {
                    data[offset+i] = getValue()*scale;
                }
                catch (System.IO.EndOfStreamException) {
                    throw new TPCException("End of file reached before reading all data dim="+data.dim+" datatype="+header.datatype.ToString());
                }
                //send event after each plane
                if (i > 0 && i % voxels_in_plane == 0)
                {
                    pevent = IOProgress;
                    if (pevent != null)
                        pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)datalength), System.Reflection.MethodBase.GetCurrentMethod()));
                }
            }
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
        }
        /// <summary>
        /// Gets parameters needed for direct access to pixel data.
        /// </summary>
        /// <returns>parameters to pixel data</returns>
        public override PixelDataParameters GetParametersForPixelData()
        {
            PixelDataParameters p;
            p.datatype = header.datatype;
            p.dim = image.dim;
            p.filename = filename;
            p.position = 0;
            p.scale = microPETHeader.calibration_factor;
            return p;
        }
        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">subregion that is read from file</param>
        public override void ReadSubImage(IntLimits region)
        {
            ReadFile();
            header.dim = region;
            image = image.GetSubImage(region);
        }
    }
}