﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.Image
{
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Analyze75 : NUnitTestbench
    {
        /// <summary>
        /// The location of test files.
        /// </summary>
        static string path = @"P:\data\Analyze75\";

        /// <summary>
        /// The basenames of the test files.
        /// </summary>
        string[] fnames = {                             // index
            "DYN_HEART_PET_16bit",                      // 0
            "DYN_HEART_PET_32bit",                      // 1
            "SINGLE_FRAME_HEART_PET_16bit",             // 2
            "SINGLE_FRAME_HEART_PET_32bit",             // 3
            "HEART_CT_32bit",                           // 4
            "SINGLE_FRAME_HEART_PET_16bit_bigendian",   // 5
            "_2D_Norm_429_1d58_se8_rsl16"               // 6
        };

        /// <summary>
        /// 1.0 Tests that header information is read correctly.
        /// Some individual values are tested againts pre-known values from Matlab implementation.
        /// No exception thrown
        /// </summary>
        [Test]
        public void Test1_0_ReadHeader()
        {
            string basename = path + fnames[1];
            string hname = basename + ".hdr";
            Assert.IsTrue(File.Exists(hname), "File ["+hname+"] did not exist. Cannot continue testing.");

            Analyze75File a75 = new Analyze75File(basename);
            a75.ReadHeader();
            Analyze75Header h = a75.analyzeHeader;
            
            //comparison of values against Matlab routine ReadAnalyzeHdr.m
            //name
            Assert.AreEqual(10,h.key.data_type.Length);
            for (int i = 0; i < h.key.data_type.Length; i++)
                Assert.AreEqual(0,h.key.db_name[i]);
            //path
            Assert.AreEqual(h.key.db_name.Length,path.Length);
            for (int i = 0; i < h.key.db_name.Length; i++)
                Assert.AreEqual(0, h.key.db_name[i]);
            //data_type
            Assert.AreEqual(10,h.key.data_type.Length);
            for (int i = 0; i < 10; i++)
                Assert.AreEqual(0,h.key.data_type[i]);
            //pre
            Assert.AreEqual(32,h.dime.bitpix);
            //dim
            Assert.AreEqual(128, a75.header.dimx);
            Assert.AreEqual(128, a75.header.dimy);
            Assert.AreEqual(47,  a75.header.dimz);
            Assert.AreEqual(22 , h.dime.dim[4]);
            //siz
            Assert.LessOrEqual(Math.Abs(2.73438000679016 - a75.header.sizex), 0.000001);
            Assert.LessOrEqual(Math.Abs(2.73438000679016 - a75.header.sizey), 0.000001);
            Assert.LessOrEqual(Math.Abs(3.26999998092651 - a75.header.sizez), 0.000001);
            //lim
            Assert.AreEqual(641279, h.dime.glmax);
            Assert.AreEqual(     0, h.dime.glmin);
            //scale
            Assert.AreEqual(1.0, h.ScaleFactor);
            //funused2
            Assert.AreEqual(0, h.dime.funused[1]);
            //funused3
            Assert.AreEqual(0, h.dime.funused[2]);
            //cal_max
            Assert.AreEqual(0, h.dime.cal_min);
            //cal_min
            Assert.AreEqual(0, h.dime.cal_max);
            //compressed
            Assert.AreEqual(0.0f, h.dime.compressed);
            //verified
            Assert.AreEqual(0.0f, h.dime.verified);
            //offset
            Assert.AreEqual(0.0f, h.dime.vox_offset);
            //orient
            Assert.AreEqual(0, h.hist.orient);
            //origin
            Assert.AreEqual(0, h.hist.originator[0]);
            Assert.AreEqual(0, h.hist.originator[1]);
            Assert.AreEqual(0, h.hist.originator[2]);
            //descr
            Assert.AreEqual(80, h.hist.descrip.Length);
            //endian
            Assert.That(h.IsLittleEndian);
        }

        /// <summary>
        /// 1.1 Tests that header information is read correctly.
        /// Header data matches to the manually copied.
        /// Todetaan kaikki 'Header'-luokan kentät
        /// on luettu oikein Analyze7.5 luokan sisään
        /// </summary>
        [Test]
        public void Test1_1_HeaderIsReadCorrectly()
        {
            Console.WriteLine("Dynamic Files");

            string filename = path + fnames[0];
            Analyze75File a75 = new Analyze75File(filename);
            a75.ReadHeader();
            Analyze75Header h = a75.analyzeHeader;

            Console.WriteLine(h.ToString());

            Assert.AreEqual(4, h.DataType);
            Assert.AreEqual(16, h.BitsPerPixel);
            Assert.AreEqual(4, h.Dimensions, "dim");
            Assert.AreEqual(770048, h.Pixels);
            Assert.AreEqual(770048 * 16 / 8, h.RawSize);
            Assert.AreEqual(0, h.VoxelsOffset, "");
            Assert.AreEqual(50.4896f, h.ScaleFactor, "scalefactor");
            Assert.AreEqual(22, h.Frames);


            filename = path + fnames[1];
            a75 = new Analyze75File(filename);
            a75.ReadHeader();
            h = a75.analyzeHeader;

            Console.WriteLine(h.ToString());

            Assert.AreEqual(32, h.DataType);
            Assert.AreEqual(32, h.BitsPerPixel);
            Assert.AreEqual(4, h.Dimensions, "dim");
            Assert.AreEqual(770048, h.Pixels);
            Assert.AreEqual(770048 * 32 / 8, h.RawSize);
            Assert.AreEqual(0, h.VoxelsOffset, "");
            Assert.AreEqual(1.0f, h.ScaleFactor, "scalefactor");
            Assert.AreEqual(22, h.Frames);


            Console.WriteLine("Single Frame");

            filename = path + fnames[2];
            a75 = new Analyze75File(filename);
            a75.ReadHeader();
            h = a75.analyzeHeader;

            Console.WriteLine(h.ToString());

            Assert.AreEqual(4, h.DataType);
            Assert.AreEqual(16, h.BitsPerPixel);
            Assert.AreEqual(4, h.Dimensions, "dim");
            Assert.AreEqual(770048, h.Pixels);
            Assert.AreEqual(770048 * 16 / 8, h.RawSize);
            Assert.AreEqual(0, h.VoxelsOffset, "");
            Assert.AreEqual(6.96689f, h.ScaleFactor, "scalefactor");
            Assert.AreEqual(1, h.Frames);

            filename = path + fnames[3];
            a75 = new Analyze75File(filename);
            a75.ReadHeader();
            h = a75.analyzeHeader;

            Console.WriteLine(h.ToString());

            Assert.AreEqual(32, h.DataType);
            Assert.AreEqual(32, h.BitsPerPixel);
            Assert.AreEqual(4, h.Dimensions, "dim");
            Assert.AreEqual(770048, h.Pixels);
            Assert.AreEqual(770048 * 32 / 8, h.RawSize);
            Assert.AreEqual(0, h.VoxelsOffset, "");
            Assert.AreEqual(1.0f, h.ScaleFactor, "scalefactor");
            Assert.AreEqual(1, h.Frames);

            filename = path + fnames[4];
            a75 = new Analyze75File(filename);
            a75.ReadHeader();
            h = a75.analyzeHeader;

            Console.WriteLine(h.ToString());

            Assert.AreEqual(16, h.DataType);
            Assert.AreEqual(32, h.BitsPerPixel);
            Assert.AreEqual(4, h.Dimensions, "dim");
            Assert.AreEqual(58720256, h.Pixels);
            Assert.AreEqual(58720256 * 32 / 8, h.RawSize);
            Assert.AreEqual(0, h.VoxelsOffset, "");
            Assert.AreEqual(1.0f, h.ScaleFactor, "scalefactor");
            Assert.AreEqual(1, h.Frames);
        }

        /// <summary>
        /// 1.2 Tests that data is read and written correctly
        /// No exception thrown.
        /// </summary>
        /// <remarks>
        /// Tests all files that are in headers table.
        /// </remarks>
        [Test]
        public void Test1_2_ReadAndWriteHeadersCorrectly()
        {
            foreach (string fname in fnames)
            {
                string filename = path + fname;
                string testfile = "S:\\temp\\" + fname + "_test";

                Console.WriteLine("\nTESTING: " + filename);

                // reads the data
                Analyze75File a75 = new Analyze75File(filename);
                a75.ReadFile();
                Analyze75Header h = a75.analyzeHeader;
                Console.WriteLine("h.dim="+a75.header.dim);

                // ensures that test file doesn't exist
                if (File.Exists(testfile + ".hdr")) File.Delete(testfile + ".hdr");
                if (File.Exists(testfile + ".img")) File.Delete(testfile + ".img");
                a75.filename = testfile;
                a75.WriteFile();
                Assert.IsTrue(Analyze75File.CheckFormat(testfile), "File ["+testfile+"] was not recognized as Analyze75.");

                // Checks that files are the same bit by bit.
                Console.WriteLine("testing file equality ["+filename+"]<->["+testfile+"]");
                FileAssert.AreEqual(filename+".hdr", testfile+".hdr");

                // The test file isn't needed anymore.
                File.Delete(testfile);

                Console.WriteLine(filename + "TEST PASSED");
            }
        }

        /// <summary>
        /// 1.3 Module testing for checkFormat. 
        /// Tests that Analyze75 data format is recognized correctly.
        /// No exception thrown.
        /// </summary>
        /// <remarks>
        /// Tests all files that are in headers table.
        /// </remarks>
        [Test]
        public void Test1_3_checkFormat()
        {
            //test different valid files
            foreach (string fname in fnames)
            {
                string filename_basename = path + fname;
                string filename_hdr = path + fname + ".hdr";
                string filename_img = path + fname + ".img";
                
                Console.Write("Testing [" + filename_basename + "]..");
                Console.Write("basename..");
                Assert.That(Analyze75File.CheckFormat(filename_basename));
                Console.Write("hdr..");
                Assert.That(Analyze75File.CheckFormat(filename_hdr));
                Console.Write("img..");
                Assert.That(Analyze75File.CheckFormat(filename_img));
                Console.WriteLine("PASSED");
            }
            //test non-accepted file formats
            //small ASCII file
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\dft\typical_weights"));
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\dft\typical_weights.dft"));             
            //ecat7 file
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\Ecat7\dynamic_heart_water\003"));
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\Ecat7\dynamic_heart_water\003.v"));
            //DICOM file
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\DICOM\phantom_head_FDG\heart_024_PT047"));
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\DICOM\phantom_head_FDG\heart_024_PT047.dcm"));
            //DICOM directory
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\DICOM\phantom_head_FDG"));
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\DICOM\phantom_head_FDG\"));
            Assert.That(!Analyze75File.CheckFormat(@"P:\data\DICOM\phantom_head_FDG\*"));
        }
        /// <summary>
        /// Read SPM produced file that has invalid extent information (little endian)
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_4_readInvalidExtent()
        {
            ImageFile img = ImageFile.ReadFile(@"P:\data\Analyze75\brain_invalid_ext");
            float mean = img.image.GetMean();
            Assert.AreEqual(0.0, Math.Abs(20020.3114075445 - mean) / mean, 0.005);
        }
        /// <summary>
        /// Template for 3.x -tests.
        /// </summary>
        /// <param name="i"></param>
        private void ReadAndWrite(int i)
        {
            string basename = path + fnames[i];
            string testfile = basename + ".test";

            File.Delete(testfile + ".img");
            Assert.IsFalse(File.Exists(testfile + ".img"));

            Analyze75File d = new Analyze75File(basename);
            Console.WriteLine("Reading data file");
            d.ReadFile();

            Console.WriteLine("Writing data file");
            d.filename = testfile;
            d.WriteFile();

            // The actual criteria
            FileAssert.AreEqual(basename + ".img", testfile + ".img");

            File.Delete(testfile);
            Assert.IsFalse(File.Exists(testfile));
        }

        /// <summary>
        /// 3.0 The filesize 1 Mb.
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_0_WriteData16bit_1Mb()
        {
            ReadAndWrite(2);
        }

        /// <summary>
        /// 3.1 The filesize 3 Mb
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_1_WriteData32bit_3Mb()
        {
            ReadAndWrite(3);
        }

        /// <summary>
        /// 3.0 Read and write The filesize 32 Mb.
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_2_WriteData16bit_32Mb()
        {
            ReadAndWrite(0);
        }

        /// <summary>
        /// 3.3 The filesize 64 Mb.
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_3_WriteData32bit_64Mb()
        {
            ReadAndWrite(1);
        }

        /// <summary>
        /// 3.4 The filesize over 230 Mb
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_4_WriteData32bit_230Mb()
        {
            ReadAndWrite(4);
        }

        /// <summary>
        /// 3.5 The filesize over 230 Mb
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_5_BigEndian()
        {
            ReadAndWrite(5);
        }

        /// <summary>
        /// 3.6 Image is 16-bit scaled integer with negative value.
        /// Input and output files match.
        /// </summary>
        [Test]
        public void Test3_6_16bitHasNegative()
        {
            ReadAndWrite(6);
        }

        /// <summary>
        /// 4.0 File format is incorrect.
        /// Wanted Exception thrown
        /// </summary>
        [Test]
        public void Test4_0_IncorrectFile()
        {
            ImageFile img = new Analyze75File(path + "incorrect");
            try
            {
                img.ReadFile();
            }
            catch (TPCAnalyzeFileException e)
            {
                Assert.IsTrue(e.Message.StartsWith("Unable to swap"));
            }
        }

        /// <summary>
        /// 4.1 Header doesn't exist
        /// Wanted Exception thrown
        /// </summary>
        [Test]
        public void Test4_1_HeaderDoesNotExist()
        {   
            try
            {
                ImageFile img = new Analyze75File(path + "notexist");
            }
            catch (TPCAnalyzeFileException e)
            {
                Assert.IsTrue(e.Message.StartsWith("No header file."), e.Message);
            }
        }

        /// <summary>
        /// 4.2 Header exist but not image file.
        /// Wanted Exception thrown
        /// </summary>
        [Test]
        public void Test4_2_ImageNotExists()
        {
            try
            {
                ImageFile img = new Analyze75File(path + "incorrect2");
            }
            catch (TPCAnalyzeFileException e)
            {
                Assert.IsTrue(e.Message.StartsWith("No image file."), e.Message);
            }
        }

        /// <summary>
        /// Template for 5.x -tests.
        /// Evaluates first and last frame mean and global mean.
        /// </summary>
        /// <param name="fname">the filename</param>
        /// <param name="first">the first image mean value.</param>
        /// <param name="last">the last image mean value.</param>
        /// <param name="full">the mean of all data.</param>
        private void Template5x(string fname, double first, double last, double full)
        {
            string basename = path + fname;
            Analyze75File.IOProgress += EventListener;
            Analyze75File file = new Analyze75File(basename);

            file.ReadFile();

            double expected, delta;
            double errorMargin = 0.05;
            Console.WriteLine("error margin:"+errorMargin+"% of expected value");

            if (file.header.dim.Length > 3 && file.header.dim[3] > 1)
            {
                DynamicImage dyn = (DynamicImage)file.image;
                Image img1 = dyn.GetFrame(0);
                Image img2 = dyn.GetFrame(dyn.frames - 1);
                Image img3 = dyn;

                // 1st part
                expected = first;
                delta = expected * errorMargin;
                Assert.AreEqual(expected, img1.GetMean(), delta);

                // last part
                expected = last;
                delta = expected * errorMargin;
                Assert.AreEqual(expected, img2.GetMean(), delta);

                // full image
                expected = full;
                delta = expected * errorMargin;
                Assert.AreEqual(expected, img3.GetMean(), delta);
            }
            else {
                // full image
                delta = first * errorMargin;
                Assert.AreEqual(first, file.image.GetMean(file.image.GetPlaneLimits(0)), delta);
                delta = last * errorMargin;
                Assert.AreEqual(last, file.image.GetMean(file.image.GetPlaneLimits(file.image.planes - 1)), delta);
                delta = full * errorMargin;
                Assert.AreEqual(full, file.image.GetMean(), delta);
            }
        }

        /// <summary>
        /// 5.0 16-bit Signed Short
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_0_DynamicInt16()
        {
            Template5x("DYN_HEART_PET_16bit",
                1952.52492122947d,
                7459.71300726718d,
                10865.9666987622d);
        }

        /// <summary>
        /// 5.1 32-bit Single Frame
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_1_StaticDouble()
        {
            Template5x("SINGLE_FRAME_HEART_PET_32bit",
                10837.1d,
                5044.21d,
                8805.99797422362d);       
        }

        /// <summary>
        /// 5.2 32-bit Dynamic Image
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_2_Dynamic32bit()
        {
            Template5x("DYN_HEART_PET_32bit",
                1953.18434227295,
                7460.42727088039,
                10866.4615216913);   
        }

        /// <summary>
        /// 5.3 16-bit Static Image
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_3_StaticInt16()
        {
            Template5x("SINGLE_FRAME_HEART_PET_16bit",
                10554.8,
                5044.52,
                8805.99098608007);
        }

        /// <summary>
        /// 5.4 16-bit Single Frame Big Endian coding.
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_4_SINGLE_16bit_bigendian()
        {
            Template5x("SINGLE_FRAME_HEART_PET_16bit_bigendian",
                10554.8,
                5044.52,
                8805.99098608007);
        }

        /// <summary>
        /// 5.5 32-bit Dynamic image.
        /// The mean values match.
        /// </summary>
        [Test]
        public void Test5_5_HEART_CT_32bit()
        {
            Template5x("HEART_CT_32bit",
                393.22,
                107.67,
                250.893130935067);   

        }

        /// <summary>
        /// 6.0 Handle Analyze 7.5 file as a default image file.
        /// 
        /// Test man value after write-read.
        /// </summary>
        [Test]
        public void Test6_0_ChangeData()
        {
            string fname = path + "SINGLE_FRAME_HEART_PET_16bit";
            ImageFile f = new Analyze75File(fname);

            Assert.AreEqual(FileType.Analyze75, f.filetype);

            f.ReadFile();

            Assert.IsTrue(f.header is ImageHeader);
            
            Assert.AreEqual(ImageFile.DataType.BIT16_S, f.header.datatype);
            Assert.AreEqual(Data_unit.Undefined, f.header.dataunit);

            string s = "";
            Assert.AreEqual(s, f.header.description);
            
            Assert.AreEqual(128, f.header.dimx);
            Assert.AreEqual(128, f.header.dimy);
            Assert.AreEqual(47, f.header.dimz);
            Assert.AreEqual("<no name>", f.header.patient_name);
            Assert.AreEqual("", f.header.patientID);

            Assert.LessOrEqual(Math.Abs(2.73438000679016f-f.header.sizex), 0.0001);
            Assert.LessOrEqual(Math.Abs(2.73438000679016f-f.header.sizey), 0.0001);
            Assert.LessOrEqual(Math.Abs(3.26999998092651f-f.header.sizez), 0.0001);

            Assert.IsTrue(f.image != null);
            Assert.IsTrue(f.image is Image);

            Console.WriteLine(f.image.ToString());

            f.filename = path + "TestOutput";
            f.WriteFile();

            File.Delete(f.filename + ".hdr");
            File.Delete(f.filename + ".img");
        }

        /// <summary>
        /// Used when we want to clear all written test data.
        /// </summary>
        [Test]
        public void Test7_0_DeleteTestData()
        {
            foreach (string name in fnames)
            {
                Console.WriteLine(path + name + ".test.hdr");
                File.Delete(path + name + ".test.hdr");
                Console.WriteLine(path + name + ".test.img");
                File.Delete(path + name + ".test.img");
            }
        }

        /// <summary>
        /// Test I/O operations with single frame data.
        /// Test exception throwing with invalid input.
        /// 
        /// Ignored because the method is removed from the library.
        /// </summary>
        //[Test]
        //[Ignore]
        public void Test8_0_frameIO()
        {
            string filename = path + fnames[0] + ".hdr";
            Analyze75File a75 = new Analyze75File(filename);
            Exception exception = null;
            a75.ReadHeader();
            Analyze75File ana75 = new Analyze75File(filename);
            try { ana75.ReadFrame(-1); }
            catch (TPCInvalidArgumentsException e) { exception = e; }
            Assert.That(exception is TPCInvalidArgumentsException);
            exception = null;
            try { ana75.ReadFrame(0); }
            catch (TPCInvalidArgumentsException e) { exception = e; }
            Assert.That(exception is TPCInvalidArgumentsException);
            exception = null;
            try { ana75.ReadFrame(1); }
            catch (TPCInvalidArgumentsException) { Assert.Fail(); }
            exception = null;
            try { ana75.ReadFrame((a75.header as DynamicImageHeader).frames + 1); }
            catch (TPCInvalidArgumentsException e) { exception = e; }
            Assert.That(exception is TPCInvalidArgumentsException);
            exception = null;
        }
    }
}






