﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;
using System.Runtime.InteropServices;
using System.Resources;
using TPClib;
using openDicom.DataStructure;
using openDicom.DataStructure.DataSet;
using openDicom.Registry;
using openDicom.File;
using openDicom.Encoding;

namespace TPClib.Image
{
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_DICOM : NUnitTestbench
    {
        /// <summary>
        /// Test reading DICOM heart file.
        /// 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_2_DICOM_read()
        {
            ImageFile file;
            //DICOM
            file = ImageFile.ReadFile(@"P:\data\DICOM\Heart\*");
            Assert.IsTrue(file is DicomFile);
            Assert.AreEqual(new IntLimits(128,128,47), file.image.dim);
            Assert.AreEqual(new Voxel(2.73438, 2.73438, 3.27), file.header.siz);
            //Compare to volume mean measured with Vinci software
            Assert.AreEqual(238863.41f, file.image.GetMean(), 10.0f);
            Assert.AreEqual(183953.0f, file.image.GetPlane(46).GetMean(), 10.0f);
            Assert.AreEqual(158039.0f, file.image.GetPlane(0).GetMean(), 10.0f);
        }
        /// <summary>
        /// Test reading all DICOM header data with GeneralHeader interface.
        /// 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_3_DICOM_readAllHeaderdata()
        {
            ImageFile file;
            //DICOM
            file = ImageFile.ReadFile(@"P:\data\DICOM\Heart\*");
            Assert.IsTrue(file is DicomFile);
            Console.WriteLine("Mean value:" + file.image.GetMean() + " dimensions:" + file.image.dim);
            Assert.AreEqual(158, file.header.Count);
        }
        /// <summary>
        /// For modification of TAGs file. Reads dictionary source and combines them. 
        /// This method is not for testing, but to create DICOM dictionaries for the library.
        /// </summary>
        //[Test]
        //[Ignore("Disabled because not required")]
        public void modify_TAGlist()
        {
            DataElementDictionary dataElementDictionary = new DataElementDictionary();
            DataElementDictionary dataElementDictionary2 = new DataElementDictionary();
            UidDictionary uidDictionary = new UidDictionary();
            UidDictionary uidDictionary2 = new UidDictionary();
            try
            {
                dataElementDictionary.LoadFrom(@"C:\coding\trunk_TPClib\Resources\TPC_elems2.dic", DictionaryFileFormat.PropertyFile);
                dataElementDictionary2.LoadFrom(@"S:\temp\harri\parsed.dic", DictionaryFileFormat.PropertyFile);
                uidDictionary.LoadFrom(@"C:\coding\trunk_TPClib\Resources\TPC_uids.dic", DictionaryFileFormat.PropertyFile);
                uidDictionary2.LoadFrom(@"S:\temp\harri\parsed_UIDs.dic", DictionaryFileFormat.PropertyFile);
            }
            catch (Exception dictionaryException)
            {
                throw new TPCDicomFileError("Problems processing dictionaries:\n" + dictionaryException);
            }
            UidDictionaryEntry[] a = uidDictionary2.ToArray();
            int new_elements = 0;
            for (int i = 0; i < uidDictionary2.Count; i++)
            {
                if (!uidDictionary.Contains(a[i].Uid))
                {
                    Console.WriteLine("[" + (++new_elements) + "] " + a[i].Uid + " " + a[i].Name);
                    uidDictionary.Add(a[i]);
                }
            }
            uidDictionary.SaveTo(@"C:\coding\trunk_TPClib\Resources\TPC_uids2.dic", DictionaryFileFormat.PropertyFile);
            
            DataElementDictionaryEntry[] b = dataElementDictionary2.ToArray();
            new_elements = 0;
            for (int i = 0; i < dataElementDictionary2.Count; i++) {
                if (!dataElementDictionary.Contains(b[i].Tag))
                {
                    Console.WriteLine("[" + (++new_elements) + "] " + b[i].Tag + " " + b[i].Description);
                    dataElementDictionary.Add(b[i]);
                }
            }
            dataElementDictionary.SaveTo(@"C:\coding\trunk_TPClib\Resources\TPC_elems3.dic", DictionaryFileFormat.PropertyFile);            
        }
        /// <summary>
        /// Read one RGB DICOM file. Not included in testing because data is 
        /// not in required folder yet.
        /// </summary>
        //[Test]
        //[Ignore("Disabled because not required")]
        public void Test1_4_readRGB()
        {
            ImageFile file;
            //DICOM

            ImageHeader hdr = ImageFile.ReadHeader(@"S:\temp\hanc\dcm\i1198771.SCPT.1.dcm");
            file = ImageFile.ReadFile(@"S:\temp\hanc\dcm\*");
            Assert.IsTrue(file is DicomFile);
            Console.WriteLine("Mean value:" + file.image.GetMean() + " dimensions:" + file.image.dim);
        }
        /// <summary>
        /// Read large CT file and write it back. The main purpose of this method 
        /// is not the test library, but rather convert invalid/corrupted DICOM data to
        /// be more readable.
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test_readCT()
        {
            string[] paths = new string[] { @"S:\temp\riku\HOFFMAN_phantom\harri\CT\*" };
            for (int path_i = 0; path_i < 1; path_i++)
            {   
                //create subdirectory
                DirectoryInfo dir = new FileInfo(paths[path_i].TrimEnd('*')).Directory;
                DirectoryInfo[] subdirs = dir.GetDirectories("TPClib_converted", SearchOption.TopDirectoryOnly);
                if (subdirs.Length > 0) subdirs[0].Delete(true);
                dir = dir.CreateSubdirectory("TPClib_converted");

                //read filenames
                FileInfo[] fileinfos = DicomFile.ResolveDicomFilenames(paths[path_i]);
                if (fileinfos.Length == 0) {
                    Console.WriteLine("no files in [" + paths[path_i] + "]");
                    continue;
                }
                Console.WriteLine("path[" + fileinfos[0].Directory.FullName + "]");
                //read headers with index numbers
                List<KeyValuePair<int, DicomHeader>> headers = new List<KeyValuePair<int, DicomHeader>>();
                List<string> filenames = new List<string>();
                DicomFile dcmfile;
                for (int file_i = 0; file_i < fileinfos.Length; file_i++) {
                    if (!DicomFile.CheckFormat(fileinfos[file_i].FullName)) continue;
                    dcmfile = new DicomFile(fileinfos[file_i].FullName);
                    dcmfile.ReadHeader();
                    headers.Add(new KeyValuePair<int, DicomHeader>(headers.Count, dcmfile.dicomheader));
                    filenames.Add(fileinfos[file_i].FullName);
                    Console.WriteLine(file_i+":"+headers[headers.Count-1]);
                }
                //sort headers with dicomheaders
                headers.Sort(new Comparison<KeyValuePair<int, DicomHeader>>(delegate(KeyValuePair<int, DicomHeader> x, KeyValuePair<int, DicomHeader> y)
                {
                    return (new DicomHeaderComparer()).Compare(x.Value, y.Value); 
                }));

                //sort out duplicate slice locations
                List<float> slice_locations = new List<float>();
                List<KeyValuePair<int, DicomHeader>> resolved_headers = new List<KeyValuePair<int, DicomHeader>>();
                for (int i = 0; i < headers.Count; i++)
                {
                    if (!slice_locations.Contains(headers[i].Value.slice_location))
                    {
                        slice_locations.Add(headers[i].Value.slice_location);
                        resolved_headers.Add(headers[i]);
                    }
                }
                for (int file_i = 0; file_i < resolved_headers.Count; file_i++)
                {
                    Console.WriteLine("[" + filenames[headers[file_i].Key] + "]");
                    dcmfile = (DicomFile)DicomFile.ReadFile(filenames[resolved_headers[file_i].Key]);
                    dcmfile.filename = dir.FullName + @"\converted_" + file_i + ".dcm";
                    dcmfile.dicomheader.instance_nr = (resolved_headers.Count-file_i);
                    Console.WriteLine("->[" + dcmfile.filename + "]");
                    dcmfile.WriteFile();
                } 
            }
        }
        /// <summary>
        /// Module testing for ReadPathIntoSingleFile. Reads one dynamic file using 
        /// interface offered by ImageFile.
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_5_ReadPathIntoSingleFile()
        {
            ImageFile file;
            DicomFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            //DICOM
            file = ImageFile.ReadFile(@"P:\data\DICOM\heartwater_rest\*");
            Assert.IsTrue(file is DicomFile);
            Console.WriteLine("Mean value:" + file.image.GetMean() + " dimensions:" + file.image.dim);
            Assert.AreEqual(new IntLimits(128, 128, 47, 24), file.header.dim);
            Assert.AreEqual(new Voxel(2.734375, 2.734375, 3.27), file.header.siz);
            //VOI means from Vinci (DCMTK package)
            float[] VOI_means = {
        297.20f,       2834.51f,       9449.17f,      17789.31f,      24148.35f,      25917.57f,      25316.01f,      21800.59f,
      18293.29f,      15498.34f,      13635.30f,      12711.52f,      12128.65f,      11872.36f,      11416.49f,      11066.19f,
      10702.90f,      10432.16f,      10135.27f,       9823.64f,       9509.38f,       9164.61f,       8937.55f,       8685.22f
                                };
            Image frame;
            float mean;
            for (int i = 0; i < (file.header as DynamicImageHeader).dimt; i++) {
                frame = (file.image as DynamicImage).GetFrame(i);
                mean = frame.GetMean();
                Console.WriteLine("Assert.AreEqual("+VOI_means[i]+", "+mean+", 0.1)");
               // Assert.AreEqual(VOI_means[i], mean, 0.1);
            }
        }
        /// <summary>
        /// Test reading DICOM brain file.
        /// 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_6_DICOM_read()
        {
            ImageFile file;
            //DICOM
            file = ImageFile.ReadFile(@"P:\data\DICOM\CortexID3\*");
            Assert.IsTrue(file is DicomFile);
            Assert.AreEqual(new IntLimits(128, 128, 63), file.image.dim);
            Assert.AreEqual(2.574f, file.header.sizex, 0.001f);
            Assert.AreEqual(2.574f, file.header.sizey, 0.001f);
            Assert.AreEqual(2.425f, file.header.sizez, 0.001f);
            //Compare to volume mean measured with Vinci software
            Assert.AreEqual(1283.3f, file.image.GetMean(), 10.0);
            Assert.AreEqual(2.91037f, file.image.GetPlane(62).GetMean(), 10.0f);
            Assert.AreEqual(1150.62f, file.image.GetPlane(0).GetMean(), 10.0f);
        }
    }
}
