﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System.Runtime.InteropServices;
using System;
using System.Collections.Generic;
using TPClib.Image;

namespace TPClib.Image
{
    /// <summary>
    /// Class for DicomFile unit tests using NUnit
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Ecat7File : NUnitTestbench
    {
        /// <summary>
        /// Test that orientation os correct when reading data
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void Test1_0_Orientation()
        {
            Ecat7File e7file;
            Limits dim = new Limits(32, 32, 32, 1);
            int x = 0;
            int y = 0;
            int z = 0;

            //read path and test intensity value, file from Vinci DICOM->Ecat7 conversion
            e7file = new Ecat7File("P:\\data\\Ecat7\\cell1.00_rsl.v");
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            e7file.ReadFile();
            ImageUtils.flipZ(ref e7file.image);
            Assert.That(e7file.header.datatype == ImageFile.DataType.BIT16_S);
            Assert.AreEqual(dim, e7file.header.dim);
            Assert.AreEqual(dim, e7file.image.dim);
            //ensure orientation after stacking of file
            for (z = 0; z < e7file.image.planes; z++)
            {
                for (y = 0; y < e7file.image.height; y++)
                {
                    for (x = 0; x < e7file.image.width; x++)
                    {
                        if (z == 8 && y == 8 && x == 8)
                        {
                            Assert.That(e7file.image.GetValue(x, y, z) == 127.0);
                        }
                        else if (y == 31 || x == 31)
                        {
                            Assert.That(e7file.image.GetValue(x, y, z) == 0.0);
                        }
                        else
                        {
                            Assert.That(Math.Abs(e7file.image.GetValue(x, y, z) - 23.0) < 0.001);
                        }
                    }
                }
            }
            Assert.That(e7file.image.GetValue(8, 8, 8) == 127);
            //compare mean to analytical value
            Assert.That(Math.Abs(e7file.image.GetMean() - 21.588134765625) < 0.001);
        }
        /// <summary>
        /// Read tests with test data
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void Test1_1_Read()
        {
            Ecat7File e7file;
            Limits dim = new Limits(32, 32, 32, 1);
            string proc_stdout;
            string[] proc_stdout_lines;
            int i = 0;
            //compare header fields to lmhdr_2_2_0.exe results
            e7file = new Ecat7File(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_2\m000-stacks-NA3BG_LANNERANGA_MR_PL001.v");
            e7file.ReadFile();
            proc_stdout = GetStdOut(@"P:/bin/windows/lmhdr_2_2_0.exe", e7file.filename);
            Assert.That(proc_stdout.Contains("patient_name := " + e7file.header.patient_name.Substring(0, 14)));
            Assert.That(proc_stdout.Contains("patient_id := " + e7file.header.patientID.Substring(0, 11)));
            Assert.That(proc_stdout.Contains("num_planes := " + e7file.header.dimz));
            Assert.That(proc_stdout.Contains("num_frames := " + e7file.header.dim.GetDimension(Limits.FRAMES)));

            //compare header fields to lshdr_2_2_0.exe results
            proc_stdout = GetStdOut(@"P:/bin/windows/lshdr_2_2_0.exe", e7file.filename);
            Assert.That(proc_stdout.Contains("x_dimension := " + e7file.header.dimx));
            Assert.That(proc_stdout.Contains("y_dimension := " + e7file.header.dimy));
            Assert.That(proc_stdout.Contains("z_dimension := " + e7file.header.dimz));
            Assert.That(proc_stdout.Contains(("x_pixel_size := " + 0.1 * Math.Round(e7file.header.sizex, 2)).Replace(',', '.')));
            Assert.That(proc_stdout.Contains(("y_pixel_size := " + 0.1 * Math.Round(e7file.header.sizey, 6)).Replace(',', '.')));
            Assert.That(proc_stdout.Contains(("z_pixel_size := " + 0.1 * Math.Round(e7file.header.sizez, 6)).Replace(',', '.')));
            Assert.That(e7file.subheaders.Length > 0);
            i = 0;
            while (i < e7file.subheaders.Length && e7file.subheaders[i] == null) i++;
            Assert.That(proc_stdout.Contains("scale_factor := " + string.Format("{0:E}", e7file.subheaders[i].scale_factor).Replace(',', '.')));

            //compare max to ecatmax, max percentual difference %1
            proc_stdout = GetStdOut(@"P:/bin/windows/ecatmax_0_2_3.exe", e7file.filename + " -C").Replace('.', ',');
            float max_ecatmax = 0.0f;
            Assert.That(float.TryParse(proc_stdout, out max_ecatmax));
            Assert.That(Math.Abs(e7file.image.GetMax() - max_ecatmax) / e7file.image.GetMax() < 0.01);

            //ensure that frame durations and start times are read correctly
            e7file.filename = @"P:/data/dicom2ecat/ecat/Stress.v";
            e7file.header.Clear();
            e7file.ReadFile();
            Assert.That(e7file.image is DynamicImage);
            proc_stdout_lines = GetStdOut(@"P:/bin/windows/lshdr_2_2_0.exe", e7file.filename, "frame_duration");
            Assert.That((e7file.image as DynamicImage).frames == proc_stdout_lines.Length);
            for (i = 0; i < proc_stdout_lines.Length; i++)
            {
                Assert.That(proc_stdout_lines[i].Equals("frame_duration := " +
                    (e7file.image as DynamicImage).GetFrameDuration(i).ToString("0")));
            }
            proc_stdout_lines = GetStdOut(@"P:/bin/windows/lshdr_2_2_0.exe", e7file.filename, "frame_start_time");
            Assert.That((e7file.image as DynamicImage).frames == proc_stdout_lines.Length);
            for (i = 0; i < proc_stdout_lines.Length; i++)
            {
                Assert.That(proc_stdout_lines[i].Equals("frame_start_time := " +
                    (e7file.image as DynamicImage).GetFrameStartTime(i).ToString("0")));
            }

            dim = new Limits(128, 128, 47, 24);
            e7file = new Ecat7File("P:\\data\\dicom2ecat\\ecat\\Stress.v");
            e7file.ReadFile();
            Assert.That(e7file.header.datatype == ImageFile.DataType.BIT16_S);
            Assert.AreEqual(dim, e7file.header.dim);
            Assert.AreEqual(dim, e7file.image.dim);
            //compare to values from epxl2dft_1_1_0.exe Stress.v out.dft 64,64,20
            Assert.AreEqual(e7file.image.GetValue(63, 63, 19, 0), 1771.0, 1.0);
            Assert.AreEqual(e7file.image.GetValue(63, 63, 19, 23), 23690.0, 5.0);
        }
        /// <summary>
        /// Test ecat7 I/O with:
        ///     16-bit signed image
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void Test1_2_WriteRead()
        {
            int i = 0;
            Ecat7File e7file;
            DynamicImage dimage;
            DynamicImageHeader dheader;
            e7file = new Ecat7File("P:\\data\\dicom2ecat\\ecat\\Stress.v");
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            e7file.ReadFile();
            //test write->read, frame information
            dimage = (e7file.image as DynamicImage);
            Assert.AreNotEqual(dimage, null);
            dheader = (e7file.header as DynamicImageHeader);
            Assert.AreNotEqual(dheader, null);
            e7file.filename = "test_readFile_1.v";
            e7file.WriteFile();
            e7file.ReadFile();
            Console.WriteLine("T_ISOTOPE READ[" + new string((char[])e7file.header["isotope_name"]) + "]");
            Assert.AreNotEqual((e7file.image as DynamicImage), null);
            Assert.AreNotEqual((e7file.header as DynamicImageHeader), null);
            for (i = 0; i < dimage.frames; i++)
            {
                Assert.AreEqual(dimage.GetFrameStartTime(i), (e7file.image as DynamicImage).GetFrameStartTime(i));
                Assert.AreEqual(dimage.GetFrameDuration(i), (e7file.image as DynamicImage).GetFrameDuration(i));
            }
            Assert.AreEqual(dimage.dim, (e7file.image as DynamicImage).dim);
            Assert.AreEqual(dheader.radiopharma, (e7file.header as DynamicImageHeader).radiopharma);
            Assert.AreEqual(dheader.isotope, (e7file.header as DynamicImageHeader).isotope);
            Assert.AreEqual(dheader.dose_start_time, (e7file.header as DynamicImageHeader).dose_start_time);
            Assert.AreEqual(dimage.GetMean(),(e7file.image as DynamicImage).GetMean(), 50.0);
            for (i = 0; i < dheader.Count; i++)
            {
                if (i == 5) continue;
                //isotope halflife is here in minutes (according to standard) 
                //rather than in seconds (as in original file)
                if (i == 24) {
                    Assert.AreEqual((float)dheader[i].Value, (float)(e7file.header as DynamicImageHeader)[i].Value * 60.0f, 1.0f);
                    continue;
                }
                if (i == 25)
                {
                    //skip because isotope name comes from library and is not expected to match exactly "o-15" vs "o15"
                    continue;
                }
                if (i == 45)
                {
                    Console.WriteLine("NAME "+i+"=[" + dheader[i].Key + "]");
                }
                Assert.AreEqual(dheader[i].Value, e7file.header[i].Value, "i="+i);
            }
        }
        /// <summary>
        /// Test of static methods
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void Test2_0_StaticMethods()
        {
            Assert.IsFalse(Ecat7File.CheckFormat(@"P:\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_1\NA3BG_LANNERANGA_MR_PL001.dcm"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_2\m000-stacks-NA3BG_LANNERANGA_MR_PL001.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_2\m000-stacks-NA3BG_LANNERANGA_MR_PL001.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_1\m000-stacks-NA3BG_LANNERANGA_MR_PL001.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_2\DICOM_SECOND_LEVEL_2\m000-stacks-TRIMFAT_PALM_PT_FR01_PL01.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"S:\petsoftware\data\DICOM\Incoming\DICOM_FIRST_LEVEL_3\DICOM_SECOND_LEVEL_1\m000-stacks-Image.0_0.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"P:\data\dicom2ecat\ecat\oncology.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"P:\data\dicom2ecat\ecat\rest.v"));
            Assert.IsTrue(Ecat7File.CheckFormat(@"P:\data\dicom2ecat\ecat\Stress.v"));
        }
        /// <summary>
        /// Test fixing of file with corrupted frame numbers (frame number 9 is missing) and writing 
        /// of file with 36 frames (more than 31 fitting into single matrix list).
        /// This error is not automatically fixed while reading so that read header data would be as it is in file.
        /// </summary>
        //[Test]
        //[Ignore("Disabled because not required")]
        public void Test_FixingCorrupted()
        {
            Ecat7File e7file;
            //compare header fields to lmhdr_2_2_0.exe results
            e7file = new Ecat7File(@"P:\data\Ecat7\dynamic_heart_36frames_missing_frame.v");
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            e7file.ReadFile();
            //fix caps in frame numbering
            int frame_No = 0;
            List<Ecat7File.Ecat7_subheader> subheaders = new List<Ecat7File.Ecat7_subheader>();
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
            Console.WriteLine("--------------------------------------------------------------------");
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                if (e7file.matrixlist[i].status == Ecat7File.Ecat7_MatDir.status_e.EXISTS)
                {
                    e7file.matrixlist[i].id = new Ecat7File.Ecat7_Matval(++frame_No,
                                                e7file.matrixlist[i].id.plane,
                                                e7file.matrixlist[i].id.gate,
                                                e7file.matrixlist[i].id.data,
                                                e7file.matrixlist[i].id.bed);
                }
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
            Console.WriteLine("--------------------------------------------------------------------");
            frame_No = 0;
            DynamicImage src_dyn_img = (e7file.image as DynamicImage);
            for (int i = 0; i < e7file.image.dim[3]; i++)
            {
                if (src_dyn_img.GetFrame(i).GetMean() != 0) frame_No++;
                Console.WriteLine(frame_No + ":" + src_dyn_img.GetFrame(i).GetMean() +
                    " " + src_dyn_img.GetFrameStartTime(i) +
                    " " + src_dyn_img.GetFrameDuration(i));
                Console.WriteLine(frame_No + ":" + e7file.subheaders[frame_No - 1].frame_start_time + " " + e7file.subheaders[frame_No - 1].frame_duration);
            }
            //correct calibration factor
            Console.WriteLine("ecat_calibration_factor:" + e7file.mainheader.ecat_calibration_factor);
            e7file.image.Multiply(1.0f / e7file.mainheader.ecat_calibration_factor);
            Console.WriteLine("subheaders " + e7file.subheaders.Length);
            Console.WriteLine("frames:" + frame_No);
            Ecat7File.Ecat7_subheader[] subs = new Ecat7File.Ecat7_subheader[frame_No];
            DynamicImage dyn_img = new DynamicImage(e7file.image.width, e7file.image.height, e7file.image.planes, frame_No);
            frame_No = 0;
            for (int i = 0; i < e7file.image.dim[3]; i++)
            {
                if (src_dyn_img.GetFrame(i).GetMean() != 0)
                {
                    subs[frame_No] = e7file.subheaders[frame_No];
                    Image subimage = src_dyn_img.GetFrame(i);
                    dyn_img.SetSubImage(dyn_img.GetFrameLimits(frame_No), ref subimage);
                    dyn_img.SetFrameDuration(frame_No, src_dyn_img.GetFrameDuration(i));
                    dyn_img.SetFrameStartTime(frame_No, src_dyn_img.GetFrameStartTime(i));
                    Console.WriteLine(dyn_img.GetFrameStartTime(frame_No) + " " + dyn_img.GetFrameDuration(frame_No));
                    subs[frame_No].frame_duration = (int)dyn_img.GetFrameDuration(frame_No);
                    subs[frame_No].frame_start_time = (int)dyn_img.GetFrameStartTime(frame_No);
                    Console.WriteLine(frame_No + ":" + subs[frame_No].frame_start_time + " " + subs[frame_No].frame_duration);
                    frame_No++;
                }
            }
            e7file.image = dyn_img;
            e7file.subheaders = subs;
            e7file.header.dim = e7file.image.dim;
            e7file.mainheader.num_frames = (short)e7file.image.dim[3];
            Console.WriteLine(e7file.header.dim);
            Console.WriteLine(e7file.image.dim);
            Console.WriteLine("subs.Length:" + e7file.subheaders.Length);
            e7file.filename = @"S:\temp\dynamic_heart_36frames_missing_frame_fixed.v";
            e7file.WriteFile();
            e7file.ReadFile();
            Console.WriteLine(e7file.image.dim);
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
        }
        /// <summary>
        /// Test fixing of file with corrupted frame numbers (frame number 9 is missing) and writing 
        /// of file with 36 frames (more than 31 fitting into single matrix list).
        /// This error is not automatically fixed while reading so that read header data would be as it is in file.
        /// 
        /// Ignored because this method is only for doing Ecat7 fixing, not primarily for testing purposes
        /// </summary>
        //[Test]
        //[Ignore("Disabled because not required")]
        public void test_fixing_corrupted2()
        {
            Ecat7File e7file;
            //compare header fields to lmhdr_2_2_0.exe results
            e7file = new Ecat7File(@"S:\temp\harri\is336dy1.v");
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            e7file.ReadFile();
            //fix caps in frame numbering
            int frame_No = 0;
            List<Ecat7File.Ecat7_subheader> subheaders = new List<Ecat7File.Ecat7_subheader>();
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
            Console.WriteLine("--------------------------------------------------------------------");
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                if (e7file.matrixlist[i].status == Ecat7File.Ecat7_MatDir.status_e.EXISTS)
                {
                    e7file.matrixlist[i].id = new Ecat7File.Ecat7_Matval(++frame_No,
                                                e7file.matrixlist[i].id.plane,
                                                e7file.matrixlist[i].id.gate,
                                                e7file.matrixlist[i].id.data,
                                                e7file.matrixlist[i].id.bed);
                }
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
            Console.WriteLine("--------------------------------------------------------------------");
            frame_No = 0;
            DynamicImage src_dyn_img = (e7file.image as DynamicImage);
            for (int i = 0; i < e7file.image.dim[3]; i++)
            {
                if (src_dyn_img.GetFrame(i).GetMean() != 0) frame_No++;
                Console.WriteLine(frame_No + ":" + src_dyn_img.GetFrame(i).GetMean() +
                    " " + src_dyn_img.GetFrameStartTime(i) +
                    " " + src_dyn_img.GetFrameDuration(i));
                Console.WriteLine(frame_No + ":" + e7file.subheaders[frame_No - 1].frame_start_time + " " + e7file.subheaders[frame_No - 1].frame_duration);
            }
            //correct calibration factor
            Console.WriteLine("ecat_calibration_factor:" + e7file.mainheader.ecat_calibration_factor);
            e7file.image.Multiply(1.0f / e7file.mainheader.ecat_calibration_factor);
            Console.WriteLine("subheaders " + e7file.subheaders.Length);
            Console.WriteLine("frames:" + frame_No);
            Ecat7File.Ecat7_subheader[] subs = new Ecat7File.Ecat7_subheader[frame_No];
            DynamicImage dyn_img = new DynamicImage(e7file.image.width, e7file.image.height, e7file.image.planes, frame_No);
            frame_No = 0;
            for (int i = 0; i < e7file.image.dim[3]; i++)
            {
                if (src_dyn_img.GetFrame(i).GetMean() != 0)
                {
                    subs[frame_No] = e7file.subheaders[frame_No];
                    Image subimage = src_dyn_img.GetFrame(i);
                    dyn_img.SetSubImage(dyn_img.GetFrameLimits(frame_No), ref subimage);
                    dyn_img.SetFrameDuration(frame_No, src_dyn_img.GetFrameDuration(i));
                    dyn_img.SetFrameStartTime(frame_No, src_dyn_img.GetFrameStartTime(i));
                    Console.WriteLine(dyn_img.GetFrameStartTime(frame_No) + " " + dyn_img.GetFrameDuration(frame_No));
                    subs[frame_No].frame_duration = (int)dyn_img.GetFrameDuration(frame_No);
                    subs[frame_No].frame_start_time = (int)dyn_img.GetFrameStartTime(frame_No);
                    Console.WriteLine(frame_No + ":" + subs[frame_No].frame_start_time + " " + subs[frame_No].frame_duration);
                    frame_No++;
                }
            }
            e7file.image = dyn_img;
            e7file.subheaders = subs;
            e7file.header.dim = e7file.image.dim;
            e7file.mainheader.num_frames = (short)e7file.image.dim[3];
            Console.WriteLine(e7file.header.dim);
            Console.WriteLine(e7file.image.dim);
            Console.WriteLine("subs.Length:" + e7file.subheaders.Length);
            e7file.filename = @"S:\temp\harri\is336dy1.v";
            e7file.WriteFile();
            e7file.ReadFile();
            Console.WriteLine(e7file.image.dim);
            for (int i = 0; i < e7file.matrixlist.Length; i++)
            {
                Console.WriteLine(i + ":" + e7file.matrixlist[i]);
            }
        }
        /// <summary>
        /// Testing of ecat7 in DICOM to ecat conversion (16-bit unsigned)
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test3_0_Dicom2Ecat()
        {
            DicomFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            DicomFile[] dcmfiles = DicomFile.ReadDicomPath(@"P:\data\DICOM\Incoming\DICOM_FIRST_LEVEL_1\DICOM_SECOND_LEVEL_1\*");
            DicomFile dcmfile = DicomFile.Stack(dcmfiles);

            Ecat7File e7 = new Ecat7File("test_dicom2ecat.v", dcmfile.image, dcmfile.header);
            e7.header.datatype = ImageFile.DataType.BIT16_S;
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            Console.WriteLine("writing:" + e7.filename + " " + e7.header.datatype);
            e7.WriteFile();
            e7.ReadFile();
            Console.WriteLine("percentual error of means for write<>read:" + Math.Abs(dcmfile.image.GetMean() - (e7.image as DynamicImage).GetMean()) / dcmfile.image.GetMean());
            Assert.That(Math.Abs(dcmfile.image.GetMean() - (e7.image as DynamicImage).GetMean()) / dcmfile.image.GetMean() < 0.001);
        }
    }
}
