﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System.Runtime.InteropServices;
using System;
using System.Collections.Generic;
using System.IO;
using TPClib.Image;

namespace TPClib.Image
{
    /// <summary>
    /// Test bench for Interfile file format
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_InterfileFile : NUnitTestbench
    {
        /// <summary>
        /// Test reading static PET image, Shepp-Logan digital phantom. 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_0_ReadWritePET()
        {
            InterfileFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            string testfilename = @"S:\temp\testinterfile";
            string testfilename_orig = @"P:\data\Interfile\modified-shepp-logan-256";
            InterfileFile file = (InterfileFile)ImageFile.ReadFile(testfilename_orig);
            if (File.Exists(testfilename + ".i")) File.Delete(testfilename + ".i");
            if (File.Exists(testfilename + ".i.hdr")) File.Delete(testfilename + ".i.hdr");
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file
            Assert.AreEqual(ImageFile.DataType.FLT32, file.header.datatype);
            //mean value was read from Vinci software
            Assert.AreEqual(0.377541, file.image.GetMean(), 0.00001);
            Assert.AreEqual(new Voxel(1.21875, 1.21875, 1.21875), file.header.siz);

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(testfilename + ".i.hdr");
            StreamReader f2 = new StreamReader(testfilename_orig + ".i.hdr");
            string line1;
            string line2;
            int line_No1 = 0;
            int line_No2 = 0;
            while (!f1.EndOfStream && !f2.EndOfStream)
            {
                line1 = f1.ReadLine(); line_No1++; 
                line2 = f2.ReadLine(); line_No2++;
                //NOTE:not tested because dose type name is changed on purpose ('F-18' to 'F18')
                if(line1.StartsWith("Dose type :=")) continue;
                //trim ending 'Mega-Bq' which is added by the library
                if(line1.StartsWith("Dosage Strength := ")) {
                    line2 = line2.Substring(0, line2.Length - 8);
                }
                Assert.AreEqual(line1.Trim(), line2.Trim(), "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1.Trim() + "][" + line2.Trim() + "]");
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            Console.Write("Data equality assertion (may take some time)..");
            FileAssert.AreEqual(testfilename_orig + ".i", testfilename + ".i");
            Console.WriteLine("done.");
        }
        /// <summary>
        /// Test reading static PET image, Ge68. 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_1_ReadWritePET()
        {
            InterfileFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            string testfilename = @"S:\temp\testinterfile";
            string testfilename_orig = @"P:\data\Interfile\calibrationstudyGe68-osem-i08";
            InterfileFile file = (InterfileFile)ImageFile.ReadFile(testfilename_orig);
            if (File.Exists(testfilename + ".i")) File.Delete(testfilename + ".i");
            if (File.Exists(testfilename + ".i.hdr")) File.Delete(testfilename + ".i.hdr");
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file
            Assert.AreEqual(ImageFile.DataType.FLT32, file.header.datatype);
            //mean value was read from Vinci software
            Assert.AreEqual(0.23451, file.image.GetMean(), 0.00001);
            Assert.AreEqual(new Voxel(1.21875, 1.21875, 1.21875), file.header.siz);

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(testfilename + ".i.hdr");
            StreamReader f2 = new StreamReader(testfilename_orig + ".i.hdr");
            string line1;
            string line2;
            int line_No1 = 0;
            int line_No2 = 0;
            while (!f1.EndOfStream && !f2.EndOfStream)
            {
                line1 = f1.ReadLine(); line_No1++;
                line2 = f2.ReadLine(); line_No2++;
                //NOTE:not tested because dose type name is changed on purpose (e. g. 'F-18' to 'F18')
                if (line1.StartsWith("Dose type :=")) continue;
                //trim ending 'Mega-Bq' which is added by the library
                if (line1.StartsWith("Dosage Strength := "))
                {
                    line2 = line2.Substring(0, line2.Length - 8);
                }
                Assert.AreEqual(line1.Trim(), line2.Trim(), "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1.Trim() + "][" + line2.Trim() + "]");
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            Console.Write("Data equality assertion (may take some time)..");
            FileAssert.AreEqual(testfilename_orig + ".i", testfilename + ".i");
            Console.WriteLine("done.");
        }
        /// <summary>
        /// Test reading static mumap h33 image. 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_2_ReadWriteH33()
        {
            InterfileFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            string testfilename = @"S:\temp\testinterfile";
            string testfilename_orig = @"P:\data\Interfile\h33\ra7f5_mumap";
            InterfileFile file = (InterfileFile)ImageFile.ReadFile(testfilename_orig);
            if (File.Exists(testfilename + ".i")) File.Delete(testfilename + ".i");
            if (File.Exists(testfilename + ".i.hdr")) File.Delete(testfilename + ".i.hdr");
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file
            Assert.AreEqual(ImageFile.DataType.FLT32, file.header.datatype);
            //mean value was read from Vinci software
            Assert.AreEqual(0.0138505, file.image.GetMean(), 0.00001);
            Assert.AreEqual(new Voxel(2.4375, 2.4375, 1.21875), file.header.siz);

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(testfilename + ".i.hdr");
            StreamReader f2 = new StreamReader(testfilename_orig + ".h33");
            string line1;
            string line2;
            int line_No1 = 0;
            int line_No2 = 0;
            while (!f1.EndOfStream && !f2.EndOfStream)
            {
                line1 = f1.ReadLine(); line_No1++;
                line2 = f2.ReadLine(); line_No2++;
                if(line1.StartsWith("scaling factor")) {
                    line1 = line1.Trim().Substring(0, 6);
                    line2 = line2.Trim().Substring(0, 6);
                    Assert.AreEqual(line1, line2, "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1 + "][" + line2 + "]");                    
                } else {
                    Assert.AreEqual(line1.Trim(), line2.Trim(), "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1.Trim() + "][" + line2.Trim() + "]");
                }
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            Console.Write("Data equality assertion (may take some time)..");
            FileAssert.AreEqual(testfilename_orig + ".i", testfilename + ".i");
            Console.WriteLine("done.");
        }
        /// <summary>
        /// Test reading dynamic PET image. 
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_3_ReadWriteDynamic()
        {
            InterfileFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            string testfilename = @"S:\temp\testinterfile*";
            string testfilename_orig = @"P:\data\Interfile\dynamic\ra7n_oposem_sp9*";
            InterfileFile file = (InterfileFile)ImageFile.ReadFile(testfilename_orig);
            for (int i = 0; i < 4; i++)
            {
                if (File.Exists(@"S:\temp\testinterfile_frame" + i + ".i")) File.Delete(@"S:\temp\testinterfile_frame" + i + ".i");
                if (File.Exists(@"S:\temp\testinterfile_frame" + i + ".i.hdr")) File.Delete(@"S:\temp\testinterfile_frame" + i + ".i.hdr");
            }
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file
            Assert.AreEqual(ImageFile.DataType.FLT32, file.header.datatype);
            //mean value was read from Vinci software
            Assert.AreEqual(new IntLimits(256, 256, 207, 4), file.header.dim);
            Assert.AreEqual(new Voxel(1.21875, 1.21875, 1.21875), file.header.siz);
            Assert.That(file.image is DynamicImage);
            float[] means = { 0.000723763f, 0.00135006f, 0.00124472f, 0.00114921f };
            int[] start_times = { 540, 600, 720, 840 };
            int[] durations = { 60, 120, 120, 120 };
            for (int i = 0; i < 4; i++)
            {
                Assert.AreEqual(means[i], file.image.GetMean((file.image as DynamicImage).GetFrameLimits(i)), 0.00001f);
                Assert.AreEqual(start_times[i], (file.image as DynamicImage).GetFrameStartTime(i));
                Assert.AreEqual(durations[i], (file.image as DynamicImage).GetFrameDuration(i));
            }

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(@"S:\temp\testinterfile_frame3.i.hdr");
            StreamReader f2 = new StreamReader(@"P:\data\Interfile\dynamic\ra7n_oposem_sp9_frame13_256i08_2mm.i.hdr");
            string line1;
            string line2;
            int line_No1 = 0;
            int line_No2 = 0;
            while (!f1.EndOfStream && !f2.EndOfStream)
            {
                line1 = f1.ReadLine(); line_No1++;
                line2 = f2.ReadLine(); line_No2++;
                //trim ending 'Mega-Bq' which is added by the library
                if (line1.StartsWith("Dosage Strength := "))
                {
                    line2 = line2.Substring(0, line2.Length - 8);
                }
                //test special lines with specific ways or just compare lines 
                if (line1.StartsWith("name of data file"))
                    continue;
                else if (line1.StartsWith("Dose type"))
                {
                    Assert.AreEqual("Dose type := C11", line1);
                    continue;
                }
                else if (line1.StartsWith("frame :="))
                {
                    Assert.AreEqual("frame := 3", line1);
                    continue;
                }
                else if (line1.StartsWith("scaling factor"))
                {
                    line1 = line1.Trim().Substring(0, 6);
                    line2 = line2.Trim().Substring(0, 6);
                    Assert.AreEqual(line1, line2, "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1 + "][" + line2 + "]");
                }
                else
                {
                    Assert.AreEqual(line1.Trim(), line2.Trim(), "Difference in line " + line_No1 + " and " + line_No2 + "\n" + "[" + line1.Trim() + "][" + line2.Trim() + "]");
                }
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            Console.Write("Data equality assertion (may take some time)..");
            FileAssert.AreEqual(@"P:\data\Interfile\dynamic\ra7n_oposem_sp9_frame13_256i08_2mm.i", @"S:\temp\testinterfile_frame3.i");
            Console.WriteLine("done.");
        }
    }
}
