﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;
using System.Runtime.InteropServices;
using System.Resources;
using TPClib;
using openDicom.DataStructure;
using openDicom.DataStructure.DataSet;
using openDicom.Registry;
using openDicom.File;
using openDicom.Encoding;

namespace TPClib.Image
{
    /// <summary>
    /// Test bench for MicroPET file format
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_MicroPETFile : NUnitTestbench
    {
        /// <summary>
        /// Test reading dynamic PET image, mouse. 
        /// Basic fields are tested after reading, then read-write-read is tested after writing.
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_0_ReadWriteDynamic()
        {
            MicroPETFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            MicroPETFile file = (MicroPETFile)ImageFile.ReadFile(@"P:\data\MicroPETimage\ae72.pet.img");
            float mean_orig = file.image.GetMean();
            string testfilename = @"S:\temp\microPETtestheader";
            if (File.Exists(testfilename)) File.Delete(testfilename);
            if (File.Exists(testfilename + ".img.hdr")) File.Delete(testfilename + ".img.hdr");
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file
            Assert.That(file.image is DynamicImage);
            float[] means = new float[(file.image as DynamicImage).frames];
            for(int i = 0; i < (file.image as DynamicImage).frames; i++)
                means[i] = (file.image as DynamicImage).GetFrame(i).GetMean();
            Assert.AreEqual(0.878789, file.microPETHeader.injection_decay_correction, 0.0000001f);
            Assert.AreEqual(58.5139, file.microPETSubheaders[0].bed_offset, 0.00001f);
            Assert.AreEqual(58.5139, file.microPETSubheaders[0].ending_bed_offset, 0.00001f);
            Assert.AreEqual(2.82177, file.microPETSubheaders[0].vertical_bed_offset, 0.000001f);
            Assert.AreEqual(10, file.microPETSubheaders[0].frame_duration);
            Assert.AreEqual(0.1, file.microPETSubheaders[0].scale_factor, 0.000001f);
            Assert.AreEqual(4.21373, file.microPETSubheaders[0].maximum, 0.0001f);
            Assert.AreEqual(21, (file.image as DynamicImage).frames);
            Assert.AreEqual(ImageFile.DataType.FLT32, file.header.datatype);

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(@"S:\temp\microPETtestheader.img.hdr");
            StreamReader f2 = new StreamReader(@"P:\data\MicroPETimage\ae72.pet.img.hdr");
            string line1;
            string line2;
            int line_No = 1;
            while (!f1.EndOfStream && !f2.EndOfStream) {
                do { line1 = f1.ReadLine(); } while (line1.Length > 0 && line1[0] == '#');
                do { line2 = f2.ReadLine(); } while (line2.Length > 0 && line2[0] == '#');
                line_No++;
                Assert.AreEqual(f1.EndOfStream, f2.EndOfStream);
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            float mean_test = file.image.GetMean();
            Assert.LessOrEqual(Math.Abs(mean_orig - mean_test) / Math.Abs(mean_orig), 0.01);
        }
        /// <summary>
        /// Test reading dynamic PET image, mouse. 
        /// Basic fields are tested after reading, then read-write-read is tested after writing.
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test1_1_ReadWriteCT()
        {
            MicroPETFile.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            string testfilename = @"S:\temp\microPETtestheader";
            string CTfilename = @"P:\data\MicroPETimage\ae7.ct.img";
            MicroPETFile file = (MicroPETFile)ImageFile.ReadFile(CTfilename);
            Console.WriteLine("ORIG IMAGE["+file.image+"]"+file.header.datatype);
            float mean_orig = file.image.GetMean();
            if (File.Exists(testfilename)) File.Delete(testfilename);
            if (File.Exists(testfilename + ".img.hdr")) File.Delete(testfilename + ".img.hdr");
            file.filename = testfilename;

            //test some header field values against pre-known information that is read visually from the file

            //write file 
            file.WriteFile();

            //check that header is equal after writing
            StreamReader f1 = new StreamReader(CTfilename + ".hdr");
            StreamReader f2 = new StreamReader(@"S:\temp\microPETtestheader.img.hdr");
            string line1;
            string line2;
            int line_No1 = 0;
            int line_No2 = 0;
            while (!f1.EndOfStream && !f2.EndOfStream)
            {
                do { line1 = f1.ReadLine(); line_No1++; } while (!f1.EndOfStream && line1.StartsWith("#"));
                do { line2 = f2.ReadLine(); line_No2++; } while (!f2.EndOfStream && line2.StartsWith("#"));

                //NOTE: this value is skipped because original value is close to zero
                if (line1.StartsWith("ct_bh_correction")) continue;
                //NOTE: scale factor may change according to data so it is not tested
                if (line1.StartsWith("scale_factor")) continue;
                Assert.AreEqual(line1.Trim(), line2.Trim(), "Difference in line " + line_No1 + " and " + line_No2);
            }
            f1.Close();
            f2.Close();

            //check that written data equals original
            file.ReadFile();
            Console.WriteLine("TEST IMAGE[" + file.image + "]" + file.header.datatype);
            float mean_test = file.image.GetMean();
            Assert.LessOrEqual(Math.Abs(mean_orig-mean_test)/Math.Abs(mean_orig), 0.01);
        }
    }
}
