﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using NUnit.Framework;
using System.IO;
using TPClib.Model;

namespace TPClib.Image
{
    /// <summary>
    /// Test cases for Trilinear interpolation.
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Trilinear
    {
        string path = @"P:\data\Analyze75\";

        /// <summary>
        /// Check that the input AnalyzeFile is correct before
        /// the next tests.
        /// </summary>
        [Test]
        public void Test1_1_CheckAnalyze75File()
        {
            string filename = path + "ALDERS_rsl";
            Analyze75Header h = new Analyze75Header();
            h.Read(filename + ".hdr");
            
            //data_type
            Assert.AreEqual(10, h.key.data_type.Length);
            for (int i = 0; i < 10; i++)
                Assert.AreEqual(0, h.key.data_type[i]);
            //pre
            Assert.AreEqual(32, h.dime.bitpix);
            //dim
            Assert.AreEqual(32, h.dimx);
            Assert.AreEqual(64, h.dimy);
            Assert.AreEqual(64, h.dimz);
            //siz
            Assert.AreEqual(1, h.sizex);
            Assert.AreEqual(1, h.sizey);
            Assert.AreEqual(1, h.sizez);
            //lim
            Assert.AreEqual(142739232, h.dime.glmax);
            Assert.AreEqual(0, h.dime.glmin);
            //scale
            Assert.AreEqual(1.0f, h.ScaleFactor);
            //funused2
            Assert.AreEqual(0, h.dime.funused[1]);
            //funused3
            Assert.AreEqual(0, h.dime.funused[2]);
            //cal_max
            Assert.AreEqual(0, h.dime.cal_min);
            //cal_min
            Assert.AreEqual(142739232.0f, h.dime.cal_max);
            //compressed
            Assert.AreEqual(0.0f, h.dime.compressed);
            //verified
            Assert.AreEqual(0.0f, h.dime.verified);
            //offset
            Assert.AreEqual(0.0f, h.dime.vox_offset);
            //orient
            Assert.AreEqual(0, h.hist.orient);
            //origin
            Assert.AreEqual(0, h.hist.originator[0]);
            Assert.AreEqual(0, h.hist.originator[1]);
            Assert.AreEqual(0, h.hist.originator[2]);
            //descr
            Assert.AreEqual(80, h.hist.descrip.Length);
            //endian
            Assert.That(h.IsLittleEndian);
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_2_ReadAndWriteAnalyze()
        {
            // read the input file
            string name = path + "ALDERS_rsl";
            Analyze75File f = new Analyze75File(name);
            f.ReadFile();

            // write the output file
            f.filename = name + ".test";
            File.Delete(name + ".test.img");
            f.WriteFile();
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_3_WithoutMatrix()
        {
            string name = path + "ALDERS_rsl";
            IntLimits target_dim = new IntLimits(new int[4] { 64, 100, 100, 1 });
            Voxel v = new Voxel();
            Voxel u = new Voxel();

            Analyze75File f = new Analyze75File(name);
            f.ReadFile();

            TrilinearInterpolation t = new TrilinearInterpolation();
            f.image = t.Interpolate(f.image, v, target_dim, u);

            Console.WriteLine(u);
            f.header.dim.setLimits(0, 0, f.image.dim[0]);
            f.header.dim.setLimits(1, 0, f.image.dim[1]);
            f.header.dim.setLimits(2, 0, f.image.dim[2]);

            // write the output file
            f.filename = name + ".test_generated";
            File.Delete(name + ".test_generated.img");
            f.WriteFile();
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_5_ResliceWithMatrix()
        {
            string name = path + "ALDERS_rsl";
            Analyze75File f = new Analyze75File(name);
            f.ReadFile();

            Matrix scale = Transformation.Scale4x4(2, 2, 2);
            double rott = Math.PI / 8;
            Matrix rot = Transformation.Rotation4x4(0, 0, rott);
            Matrix tran = Transformation.Translation4x4(32, 64, 64);
            Matrix A = tran;

            Console.WriteLine(A);
            Console.WriteLine(Transformation.Inverse4x4(tran));

            Voxel unused1 = new Voxel();
            Voxel unused2 = new Voxel();
            IntLimits target_dim = new IntLimits(new int[4] { 64, 128, 128, 1 });
            
            TrilinearInterpolation sut = new TrilinearInterpolation();
            f.image = sut.Reslice(f.image, unused1, target_dim, unused2, A);

            f.header.dim.setLimits(0, 0, f.image.dim[0]);
            f.header.dim.setLimits(1, 0, f.image.dim[1]);
            f.header.dim.setLimits(2, 0, f.image.dim[2]);

            // write the output file
            f.filename = name + ".test_generated2";
            File.Delete(name + ".test_generated2.img");
            f.WriteFile();
        }

        private string MathematicaForm(Matrix m)
        {
            IFormatProvider form = System.Globalization.CultureInfo.CreateSpecificCulture("en-us");
            string ret = "mat = {";

            for (int i = 0; i < m.Rows; i++)
            {
                ret += "{";
                for (int j = 0; j < m.Columns; j++)
                {
                    ret += m[j, i].ToString(form);
                    if (j != 3)
                        ret += ",";
                    else
                        ret += "},";
                }
            }
            ret += "}";

            return ret;
        }
    }
}
