/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System.Runtime.InteropServices;
using System;
using System.Collections.Generic;
using TPClib.Image;

namespace TPClib.Image {

    /// <summary>
    /// Class for DynamicImage unit tests
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_DynamicImage : NUnitTestbench
    {
        /// <summary>
        /// Maximum error that is tolerated when image is read. Value means
        /// proportion to original value.
        /// </summary>
        double maximumMeanIntensityErrorTolerance = 0.05;
        /// <summary>
        /// Sum image test
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void Test1_0_SumImage()
        {
            string filename = @"P:\data\Ecat7\dynamic_heart_water\003.v";
            Ecat7File e7file = new Ecat7File(filename);
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);

            e7file.ReadFile();
            DynamicImage dimg = (DynamicImage)e7file.image;
            Image sum = new Image(new int[] { dimg.GetFrameLimits(0)[0], dimg.GetFrameLimits(0)[1], dimg.GetFrameLimits(0)[2] });
            for (int i = 0; i < dimg.frames; i++)
            {
                Console.WriteLine(dimg.GetFrameLimits(i));
                sum += dimg.GetFrame(i);
            }
            float mean = dimg.GetMean();
            float max = dimg.GetMax();
            float min = dimg.GetMin();
            Assert.That((sum.GetMean()/24.0f - mean) / mean < 0.0001);
            e7file.image = sum;
            e7file.header.dim = sum.dim;
            e7file.filename = "test.v";
            Console.WriteLine(e7file.header);
            Console.WriteLine(e7file.image);
            e7file.WriteFile();
            e7file = new Ecat7File("test.v", sum, new ImageHeader("test", "id", sum.dim, new Voxel(), "desc", ImageFile.DataType.BIT16_U));
            Console.WriteLine(e7file.header);
            Console.WriteLine(e7file.image);
        }

        /// <summary>
        /// Read DICOM write Ecat7
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void TestDynamic1_2_ReadWrite()
        {
            DicomFile[] dcmfiles;
            string path;
            DicomFile dcmfile;
            DicomFile.IOProgress += EventListener;
            float mean = 0.0f;

            path = @"P:\data\DICOM\Germany\*";
            Console.WriteLine(path);
            dcmfiles = DicomFile.ReadDicomPath(path);
            Assert.IsNotEmpty(dcmfiles, "no files found in " + path);
            Assert.AreEqual(ImageFile.DataType.BIT16_S, dcmfiles[0].header.datatype);
            Assert.AreEqual(17, DicomFile.ResolveNoFrames(dcmfiles));
            Assert.AreEqual(1, DicomFile.ResolveNoGates(dcmfiles));
            Assert.AreEqual(35, DicomFile.ResolveNoPlanes(dcmfiles));
            dcmfile = DicomFile.Stack(dcmfiles);
            Assert.AreEqual(ImageFile.DataType.BIT16_S, dcmfile.header.datatype);

            //test that all frame times are read correctly
            int[] referenceTimes = new int[] {     0,  10000,  20000,  30000,  40000, 
                                             50000,  60000,  70000,  80000,  90000, 100000, 
                                            110000, 120000, 150000, 180000, 210000, 240000};
            int[] durations = new int[] { 10000, 10000, 10000, 10000, 10000,
                                        10000, 10000, 10000, 10000, 10000, 10000,
                                        10000, 30000, 30000, 30000, 30000, 60000};
            Assert.That(dcmfile.image is DynamicImage);
            Assert.AreEqual(17, (dcmfile.image as DynamicImage).frames);
            for (int i = 0; i < (dcmfile.image as DynamicImage).frames; i++)
            {
                Assert.AreEqual(durations[i], (dcmfile.image as DynamicImage).GetFrameDuration(i));
                Assert.AreEqual(referenceTimes[i], (dcmfile.image as DynamicImage).GetFrameStartTime(i));
            }

            //compare to mean value acquired from Matlab version 7.5.0.342
            Console.WriteLine("dcmfile.image.getMean():" + dcmfile.image.GetMean());
            Assert.AreEqual(0.0, Math.Abs(dcmfile.image.GetMean() - 1301.3555915973895000) / 1301.3555915973895000, maximumMeanIntensityErrorTolerance);
            dcmfile.filename = Environment.CurrentDirectory + @"\testDynamic_readwrite";
            Console.WriteLine(dcmfile.filename);
            dcmfile.header.patient_name = "testDynamic_readwrite";
            mean = dcmfile.image.GetMean();
            Ecat7File e7file = new Ecat7File("test5.v", dcmfile.image, dcmfile.header);
            e7file.WriteFile();
            Console.WriteLine(e7file.filename);
            e7file.ReadFile();
            //test that all frame times are correct after writing
            Assert.That(e7file.image is DynamicImage);
            Assert.AreEqual(17, (e7file.image as DynamicImage).frames);
            for (int i = 0; i < (e7file.image as DynamicImage).frames; i++)
                Console.WriteLine((e7file.image as DynamicImage).GetFrameDuration(i) + " " + (e7file.image as DynamicImage).GetFrameStartTime(i));
            for (int i = 0; i < (e7file.image as DynamicImage).frames; i++)
            {
                Assert.AreEqual(durations[i], (e7file.image as DynamicImage).GetFrameDuration(i));
                Assert.AreEqual(referenceTimes[i], (e7file.image as DynamicImage).GetFrameStartTime(i));
            }
            Console.WriteLine(e7file.image.GetMean() + "-" + mean + " = " + (Math.Abs(e7file.image.GetMean() - mean) / mean));
            Assert.AreEqual(0.0, Math.Abs(e7file.image.GetMean() - mean) / mean, maximumMeanIntensityErrorTolerance);
        }
        /// <summary>
        /// Test index operators after conversion to DynamicImage
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void TestDynamic1_2_indexoperators()
        {
            Ecat7File e7file = new Ecat7File("P:\\data\\Ecat7\\cell1.00_rsl.v");
            Ecat7File.IOProgress += new ImageFile.IOProcessEventHandler(EventListener);
            e7file.ReadFile();
            Image img = e7file.image;
            DynamicImage img2 = new DynamicImage(img);
            int x;
            int y;
            int z;
            
            for (z = 0; z < e7file.image.planes; z++)
            {
                for (y = 0; y < e7file.image.height; y++)
                {
                    for (x = 0; x < e7file.image.width; x++)
                    {
                        Assert.AreEqual(img2.GetValue(x, y, z, 0), img[x, y, z]);
                        Assert.AreEqual(img2.GetValue(x, y, z, 0), img.GetValue(x, y, z));
                        Assert.AreEqual(img2.GetValue(x, y, z, 0), img2.GetCurve(x, y, z)[0]);
                        IntPoint p = new IntPoint(x, y, z);
                        Assert.AreEqual(p.x, x);
                        Assert.AreEqual(p.y, y);
                        Assert.AreEqual(p.z, z);
                        Assert.AreEqual(img2.GetValue(x, y, z, 0), img2.GetCurve(new IntPoint(x, y, z))[0]);
                        Assert.AreEqual(img[x, y, z], img2.GetCurve(new IntPoint(x, y, z))[0]);
                    }
                }
            }
        }
    }
}