﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using TPClib.Model;
namespace TPClib.Image
{
    /// <summary>
    /// Method for Trilinear Interpolation.
    /// Trilinear interpolation is the name given to the
    /// process of linearly interpolating points within a
    /// box (3D) given values at the vertices of the box.
    /// In general the box will not be of unit size nor will
    /// it be aligned at the origin. Simple translation and
    /// scaling (possibly of each axis independently) can be
    /// used to transform into then out of this simplified situation.
    /// </summary>
    public class TrilinearInterpolation
    {
        /// <summary>
        /// http://www.grc.nasa.gov/WWW/winddocs/utilities/b4wind_guide/trilinear.html
        /// Precondition: the corner points form a unit cube.
        /// p[i] is [0,1] for i=0,1,2
        /// v[0] = 0 0 0
        /// v[1] = 1 0 0
        /// v[2] = 0 1 0
        /// v[3] = 1 1 0
        /// v[4] = 0 0 1
        /// v[5] = 1 0 1
        /// v[6] = 0 1 1
        /// v[7] = 1 1 1
        /// </summary>
        /// <param name="values">values of eight corner points</param>
        /// <param name="p">point inside cube</param>
        /// <returns>interpolated value</returns>
        public double InterpolateUnitCube(double[] values, Point p)
        {
            return ((1 - p.z) * (1 - p.y) * (1 - p.x) * values[0] +
                    (1 - p.z) * (1 - p.y) * p.x * values[1] +
                    (1 - p.z) * p.y * (1 - p.x) * values[2] +
                    (1 - p.z) * p.y * p.x * values[3] +
                    p.z * (1 - p.y) * (1 - p.x) * values[4] +
                    p.z * (1 - p.y) * p.x * values[5] +
                    p.z * p.y * (1 - p.x) * values[6] +
                    p.z * p.y * p.x * values[7]) / 8.0f;
        }

        /// <summary>
        /// http://www.grc.nasa.gov/WWW/winddocs/utilities/b4wind_guide/trilinear.html
        /// Precondition: the corner points form a unit cube.
        /// p[i] is [-1,1] for i=0,1,2
        /// v[0] = -1 -1 -1
        /// v[1] =  1 -1 -1
        /// v[2] = -1  1 -1
        /// v[3] =  1  1 -1
        /// v[4] = -1 -1  1
        /// v[5] =  1 -1  1
        /// v[6] = -1  1  1
        /// v[7] =  1  1  1
        /// </summary>
        /// <param name="values">values of eight corner points</param>
        /// <param name="p">point inside cube</param>
        /// <returns>interpolated value</returns>
        public double Interpolate(double[] values, Point p)
        {
            return ((1 - p.z) * (1 - p.y) * (1 - p.x) * values[0] +
              (1 - p.z) * (1 - p.y) * (1 + p.x) * values[1] +
              (1 - p.z) * (1 + p.y) * (1 - p.x) * values[2] +
              (1 - p.z) * (1 + p.y) * (1 + p.x) * values[3] +
              (1 + p.z) * (1 - p.y) * (1 - p.x) * values[4] +
              (1 + p.z) * (1 - p.y) * (1 + p.x) * values[5] +
              (1 + p.z) * (1 + p.y) * (1 - p.x) * values[6] +
              (1 + p.z) * (1 + p.y) * (1 + p.x) * values[7]) / 8;
        }

        /// <summary>
        /// Interpolates an arbitrary pixel inside a image.
        /// Precondition: Point p should be inside image dimensions
        ///               Voxel size is 1x1x1.
        /// </summary>
        /// <param name="image">input image</param>
        /// <param name="p">arbitrary point inside image</param>
        /// <returns>interpolated value at point p</returns>
        public float Interpolate(Image image, Point p)
        {
            // input point ip 
            IntPoint ip = new IntPoint();
            ip.x = (int)p.x;
            ip.y = (int)p.y;
            ip.z = (int)p.z;

            // coordinates inside voxel's unit cube.
            Point pc = new Point();
            pc.x = p.x - (double)ip.x;
            pc.y = p.y - (double)ip.y;
            pc.z = p.z - (double)ip.z;

            double[] value = new double[8];

            value[0] = image.getValue(ip); // 0 0 0
            ip.x += 1;
            value[1] = image.getValue(ip); // 1 0 0
            ip.z += 1;
            value[5] = image.getValue(ip); // 1 0 1
            ip.x -= 1;
            value[4] = image.getValue(ip); // 0 0 1
            ip.y += 1;
            value[6] = image.getValue(ip); // 0 1 1
            ip.z -= 1;
            value[2] = image.getValue(ip); // 0 1 0
            ip.x += 1;
            value[3] = image.getValue(ip); // 1 1 0
            ip.z += 1;
            value[7] = image.getValue(ip); // 1 1 1

            return (float)InterpolateUnitCube(value, pc);
        }

        /// <summary>
        /// Transformation matrix in wikipedia:
        /// </summary>
        /// <param name="image">source image</param>
        /// <param name="v">voxel size</param>
        /// <param name="target_dim">target dimensions</param>
        /// <param name="target_v">target voxel size</param>
        /// <param name="tranformation_matrix">4x4 tranformation matrix</param>
        /// <returns>interpolated values</returns>
        public Image Reslice(Image image, Voxel v, IntLimits target_dim,
            Voxel target_v, Matrix tranformation_matrix)
        {
            if (tranformation_matrix.Columns != 4 || tranformation_matrix.Rows != 4)
                throw new TPCTrilinearInterpolationException(
                    "Transformation matrix is not 4x4.");

            Matrix inv = Transformation.Inverse4x4(tranformation_matrix);

            Image result = new Image(target_dim);

            // output pixel location
            IntPoint op = new IntPoint(0, 0, 0);

            // input pixel location
            IntPoint ip = new IntPoint(0, 0, 0);

            Point p = new Point();

            // eight corner values
            double[] value = new double[8];

            // Iterates over every target pixel.
            for (op.x = 0; op.x < result.dimx; op.x++)
            {
                for (op.y = 0; op.y < result.dimy; op.y++)
                {
                    for (op.z = 0; op.z < result.dimz; op.z++)
                    {
                        p.x = inv[0, 0] * op.x + inv[1, 0] * op.y + inv[2, 0] * op.z;
                        p.y = inv[0, 1] * op.x + inv[1, 1] * op.y + inv[2, 1] * op.z;
                        p.z = inv[0, 2] * op.x + inv[1, 2] * op.y + inv[2, 2] * op.z;

                        ip.x = (int)p.x;
                        ip.y = (int)p.y;
                        ip.z = (int)p.z;

                        // scale inside a cube: p.i in [0,1] for all i=x,y,z
                        p.x = p.x - ip.x;
                        p.y = p.y - ip.y;
                        p.z = p.z - ip.z;

                        if (ip.x + 1 >= image.dimx) continue;
                        if (ip.x < 0) continue;
                        if (ip.y + 1 >= image.dimy) continue;
                        if (ip.y < 0) continue;
                        if (ip.z + 1 >= image.dimz) continue;
                        if (ip.z < 0) continue;

                        value[0] = image.getValue(ip); // 0 0 0
                        ip.x += 1;
                        value[1] = image.getValue(ip); // 1 0 0
                        ip.z += 1;
                        value[5] = image.getValue(ip); // 1 0 1
                        ip.x -= 1;
                        value[4] = image.getValue(ip); // 0 0 1
                        ip.y += 1;
                        value[6] = image.getValue(ip); // 0 1 1
                        ip.z -= 1;
                        value[2] = image.getValue(ip); // 0 1 0
                        ip.x += 1;
                        value[3] = image.getValue(ip); // 1 1 0
                        ip.z += 1;
                        value[7] = image.getValue(ip); // 1 1 1

                        result.setValue(op, (float)InterpolateUnitCube(value, p));
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="img"></param>
        /// <param name="vx"></param>
        /// <param name="target_dim"></param>
        /// <param name="target_v"></param>
        /// <returns></returns>
        public Image Interpolate(Image img, Voxel vx, IntLimits target_dim, Voxel target_v)
        {
            Image result = new Image(target_dim);

            // new voxel size
            target_v.sizex = vx.sizex * target_dim[0] / img.dim[0];
            target_v.sizey = vx.sizey * target_dim[1] / img.dim[1];
            target_v.sizez = vx.sizez * target_dim[2] / img.dim[2];

            // image scale ratio, -1.0f is for preventing image going outside boundaries.
            float ratex = (float)(img.dim[0] - 1.0f) / target_dim[0];
            float ratey = (float)(img.dim[1] - 1.0f) / target_dim[1];
            float ratez = (float)(img.dim[2] - 1.0f) / target_dim[2];

            // output pixel location
            IntPoint op = new IntPoint(0,0,0);

            // input pixel location
            IntPoint ip = new IntPoint(0,0,0); 

            // cubic point: three coordinates inside a regular cube.
            Point p = new Point();

            // eight corner values
            double[] value = new double[8];

            // Iterates over every target pixel.
            for (op.x = 0; op.x < result.dimx; op.x++)
            {
                for (op.y = 0; op.y < result.dimy; op.y++)
                {
                    for (op.z = 0; op.z < result.dimz; op.z++)
                    {
                        // the floor of new coordinates.
                        ip.x = (int)(ratex * op.x);
                        ip.y = (int)(ratey * op.y);
                        ip.z = (int)(ratez * op.z);

                        // pixel location in regular cube
                        p.x = ratex * op.x - ip.x;
                        p.y = ratey * op.y - ip.y;
                        p.z = ratez * op.z - ip.z;

                        if (ip.x + 1 >= img.dimx)
                            throw new TPCTrilinearInterpolationException(
                                "trying to read outside image region");
                        if (ip.y + 1 >= img.dimy)
                            throw new TPCTrilinearInterpolationException(
                                "trying to read outside image region");
                        if (ip.z + 1 >= img.dimz)
                            throw new TPCTrilinearInterpolationException(
                                "trying to read outside image region");

                        value[0] = img.getValue(ip); // 0 0 0
                        ip.x += 1;           
                        value[1] = img.getValue(ip); // 1 0 0
                        ip.z += 1;
                        value[5] = img.getValue(ip); // 1 0 1
                        ip.x -= 1;
                        value[4] = img.getValue(ip); // 0 0 1
                        ip.y += 1;
                        value[6] = img.getValue(ip); // 0 1 1
                        ip.z -= 1;
                        value[2] = img.getValue(ip); // 0 1 0
                        ip.x += 1;
                        value[3] = img.getValue(ip); // 1 1 0
                        ip.z += 1;
                        value[7] = img.getValue(ip); // 1 1 1
                        
                        result.setValue(op, (float)InterpolateUnitCube(value, p));
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// Indicates that there was exception while interpolating.
        /// </summary>
        public class TPCTrilinearInterpolationException : TPCException
        {
            /// <summary>
            /// Constructs TPC exception executed inside Trilinear Interpolation method.
            /// </summary>
            /// <param name="m">message string</param>
            public TPCTrilinearInterpolationException(string m)
                : base(m)
            {
            }
        }
    }
}
