﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections.Generic;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib
{
    /// <summary>
    /// Represents dimension object that consists of integer coodinate index values. 
    /// <remarks>low coordinate is lowest index that is included in coordinates (usually 0)
    ///  while high value is not included in coordinates (256 means that highest index is 255)</remarks>
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public class IntLimits : Limits
    {
        /// <summary>
        /// Read-only width
        /// </summary>
        public new int dimx
        {
            get
            {
                return (int)width;
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public new int dimy
        {
            get
            {
                return (int)height;
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public new int dimz
        {
            get
            {
                return (int)planes;
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public new int dimt
        {
            get
            {
                return (int)frames;
            }
        }
        /// <summary>
        /// Read-only width
        /// </summary>
        public new int width
        {
            get
            {
                if (hi_dim.Length <= WIDTH)
                    throw new TPCException("Dimension object does not contain width");
                return (int)(hi_dim[WIDTH] - lo_dim[WIDTH]);
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public new int height
        {
            get
            {
                if (hi_dim.Length <= HEIGHT)
                    throw new TPCException("Dimension object does not contain height");
                return (int)(hi_dim[HEIGHT] - lo_dim[HEIGHT]);
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public new int planes
        {
            get
            {
                if (hi_dim.Length <= PLANES)
                    throw new TPCException("Dimension object does not contain planes");
                return (int)(hi_dim[PLANES] - lo_dim[PLANES]);
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public new int frames
        {
            get
            {
                if (hi_dim.Length <= FRAMES)
                    throw new TPCException("Dimension object does not contain frames");
                return (int)(hi_dim[FRAMES] - lo_dim[FRAMES]);
            }
        }
        /// <summary>
        /// Read-only gates
        /// </summary>
        public new int gates
        {
            get
            {
                if (hi_dim.Length <= GATES)
                    throw new TPCException("Dimension object does not contain gates");
                return (int)(hi_dim[GATES] - lo_dim[GATES]);
            }
        }
        /// <summary>
        /// Read-only beds
        /// </summary>
        public new int beds
        {
            get
            {
                if (hi_dim.Length <= BEDS)
                    throw new TPCException("Dimension object does not contain beds");
                return (int)(hi_dim[BEDS] - lo_dim[BEDS]);
            }
        }
        /// <summary>
        /// Read-only detectorvectors
        /// </summary>
        public new int detectorvectors
        {
            get
            {
                if (hi_dim.Length <= DETECTORVECTOR)
                    throw new TPCException("Dimension object does not contain detectorvectors");
                return (int)(hi_dim[DETECTORVECTOR] - lo_dim[DETECTORVECTOR]);
            }
        }
        /// <summary>
        /// Read-only energywindows
        /// </summary>
        public new int energywindows
        {
            get
            {
                if (hi_dim.Length <= ENERGYWINDOW)
                    throw new TPCException("Dimension object does not contain energywindows");
                return (int)(hi_dim[ENERGYWINDOW] - lo_dim[ENERGYWINDOW]);
            }
        }
        /// <summary>  
        /// Constructs dimension object without dimensions.
        /// </summary> 
        public IntLimits()
            : base()
        {
        }
        /// <summary>  
        /// Constructs dimension from another dimension.
        /// </summary> 
        /// <param name="l">copied dimension</param>
        public IntLimits(IntLimits l)
            : base(l)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(params long[] dimensions)
            : base(dimensions)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(params uint[] dimensions)
            : base(dimensions)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(params int[] dimensions)
            : base(dimensions)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="lo_limits">low limits</param>
        /// <param name="hi_limits">high limits</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(int[] lo_limits, int[] hi_limits)
        {
            for (int i = 0; i < lo_limits.Length; i++)
            {
                if (lo_limits[i] < 0)
                    throw new TPCInvalidArgumentsException("Low limit (" + i + ") is below zero:" + lo_limits[i] + ".");
                if (hi_limits[i] < lo_limits[i])
                    throw new TPCInvalidArgumentsException("At index " + i + " high limit:" + hi_limits[i] + " is below low limit:" + lo_limits[i] + ".");
            }
            lo_dim = new double[lo_limits.Length];
            hi_dim = new double[lo_limits.Length];
            for (int i = 0; i < lo_limits.Length; i++)
            {
                lo_dim[i] = (double)lo_limits[i];
                hi_dim[i] = (double)hi_limits[i];
            }
        }
        /// <summary>
        /// Returns desired dimension value
        /// </summary>
        /// <param name="i">dimension number (0-based)</param>
        /// <returns>dimension as double value. </returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new int GetDimension(int i)
        {
            return (int)base.GetDimension(i);
        }
        /// <summary>
        /// Returns all dimensions as new limits object (all low limits to zero).
        /// </summary>
        /// <returns>dimension as double value. All dimensions greater than number of actual defined dimensions are regarded as 1.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new IntLimits GetDimensions()
        {
            IntLimits r = new IntLimits();
            for (int i = 0; i < Length; i++)
                r.AddLimit(0, GetDimension(i));
            return r;
        }
        /// <summary>
        /// Returns desired limit value.
        /// </summary>
        /// <param name="i">dimension number [1..number of dimensions]</param>
        /// <param name="limit">limit</param>
        /// <returns>limit as double value.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new int GetLimit(int i, Limit limit)
        {
            return (int)base.GetLimit(i, limit);
        }
        /// <summary>
        /// Sets desired limit value.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="limit">limit</param>
        /// <param name="value">value that is set</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimit(int i, Limit limit, int value)
        {
            base.SetLimit(i, limit, value);
        }
        /// <summary>
        /// Sets desired limit values.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="low">low limit</param>
        /// <param name="high">high limit</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimits(int i, int low, int high)
        {
            base.SetLimits(i, low, high);
        }
        /// <summary>
        /// Calculates n first dimension values multiplied together.
        /// </summary>
        /// <param name="dimensions">last dimension that is multiplied [0..No dimensions-1]</param>
        /// <returns>product of dimensions</returns>
        public new double GetProduct(int dimensions)
        {
            return (int)base.GetProduct(dimensions);
        }
        /// <summary>
        /// Calculates dimension values multiplied together.
        /// </summary>
        /// <returns>product of dimensions</returns>
        public new int GetProduct()
        {
            return (int)base.GetProduct();
        }
        /// <summary>
        /// Converts limits to dimension array. Array has difference between 
        /// low and high values in is indexes.
        /// </summary>
        /// <returns>array if indexes in limits object</returns>
        new public int[] ToArray()
        {
            int[] r = new int[lo_dim.Length];
            for (int i = 0; i < r.Length; i++)
            {
                r[i] = (int)(hi_dim[i] - lo_dim[i]);
            }
            return r;
        }
        /// <summary>
        /// Gest dimension size as integer.
        /// </summary>
        /// <param name="i">0-based index of dimension number</param>
        /// <returns>size of dimension at index</returns>
        public int this[int i]
        {
            get
            {
                if (i < 0 || i >= lo_dim.Length) throw new TPCException("Index out of bounds [0.." + (lo_dim.Length - 1) + "]");
                return (int)(hi_dim[i] - lo_dim[i]);
            }
        }
        /// <summary>
        /// Removes singleton dimensions (length == 1) from dimension object.
        /// Note that 0-length dimensions are not removed, since they should not exist.
        /// </summary>
        public IntLimits RemoveSingletons()
        {
            List<double> lo_dim_tmp = new List<double>();
            List<double> hi_dim_tmp = new List<double>();
            for (int i = 0; i < lo_dim.Length; i++)
            {
                if ((int)(hi_dim[i] - lo_dim[i]) != 1)
                {
                    lo_dim_tmp.Add((double)lo_dim[i]);
                    hi_dim_tmp.Add((double)hi_dim[i]);
                }
            }
            lo_dim = lo_dim_tmp.ToArray();
            hi_dim = hi_dim_tmp.ToArray();
            return new IntLimits(this);
        }
        /// <summary>
        /// Resolves limits for subregion.
        /// </summary>
        /// <see cref="IntLimits"/>
        /// <param name="index">index (0-based)</param>
        /// <param name="dimension">dimension number</param>
        /// <returns>limits object that represent index in dimension</returns>
        /// <exception cref="TPCInvalidArgumentsException">in case of invalid parameters</exception>
        public IntLimits GetDimensionLimits(int index, int dimension)
        {
            if (dimension < 0) throw new TPCInvalidArgumentsException("Dimension argument is negative");
            if (dimension > this.Length)
                throw new TPCInvalidArgumentsException("Dimension argument " + dimension + " is larger than image dimensions " + this.Length);
            if (index < 0 || index >= this.GetDimension(dimension))
                throw new TPCInvalidArgumentsException("Index number " + index + " is out of bounds [0.." + (this.GetDimension(dimension) - 1) + "]");
            IntLimits r = new IntLimits();
            int i = 0;
            while (r.Length < dimension)
                r.AddLimit(0, this.GetDimension(i++));
            r.AddLimit(index, index + 1);
            while (r.Length < this.Length)
                r.AddLimit(0, 1);
            return r;
        }
    }
}