/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices.ComTypes;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib
{
    /// <summary>
    /// Represents limits of N-dimensional matrix or its subregion. Indexes of image are considered to start from zero.
    /// Low limits of dimension are set to zero by default. Hight limits can be at, but not below low limits.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public class Limits {
        /// <summary>
        /// Limit of dimension. Either low limit or high limit.
        /// </summary>
        public enum Limit
        {
            /// <summary>
            /// High limit.
            /// </summary>
            HIGH,
            /// <summary>
            /// Low limit.
            /// </summary>
            LOW
        }
        /// <summary>
        /// Width dimension (WIDTH == 0)
        /// </summary>
        public const int WIDTH = 0;
        /// <summary>
        /// Height dimension (HEIGHT == 1)
        /// </summary>
        public const int HEIGHT = 1;
        /// <summary>
        /// Planes dimension (PLANES == 2)
        /// </summary>
        public const int PLANES = 2;
        /// <summary>
        /// Frames dimension (FRAMES == 3)
        /// </summary>
        public const int FRAMES = 3;
        /// <summary>
        /// Gates dimension (GATES == 4)
        /// </summary>
        public const int GATES = 4;
        /// <summary>
        /// Bed position dimension (BEDS == 5)
        /// </summary>
        public const int BEDS = 5;
        /// <summary>
        /// Detector vector dimension (DETECTORVECTOR == 6)
        /// </summary>
        public const int DETECTORVECTOR = 6;
        /// <summary>
        /// Energy window dimension (ENERGYWINDOW == 7)
        /// </summary>
        public const int ENERGYWINDOW = 7;
        /// <summary>
        /// Read-only width
        /// </summary>
        public double dimx
        {
            get
            {
                return width;
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public double dimy
        {
            get
            {
                return height;
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public double dimz
        {
            get
            {
                return planes;
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public double dimt
        {
            get
            {
                return frames;
            }
        }
        /// <summary>
        /// Read-only width
        /// </summary>
        public double width {
            get {
                if (hi_dim.Length <= WIDTH)
                    throw new TPCException("Dimension object does not contain width");
                return hi_dim[WIDTH] - lo_dim[WIDTH];
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public double height
        {
            get
            {
                if (hi_dim.Length <= HEIGHT)
                    throw new TPCException("Dimension object does not contain height");
                return hi_dim[HEIGHT] - lo_dim[HEIGHT];
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public double planes
        {
            get
            {
                if (hi_dim.Length <= PLANES)
                    throw new TPCException("Dimension object does not contain planes");
                return hi_dim[PLANES] - lo_dim[PLANES];
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public double frames
        {
            get
            {
                if (hi_dim.Length <= FRAMES)
                    throw new TPCException("Dimension object does not contain frames");
                return hi_dim[FRAMES] - lo_dim[FRAMES];
            }
        }
        /// <summary>
        /// Read-only gates
        /// </summary>
        public double gates
        {
            get
            {
                if (hi_dim.Length <= GATES)
                    throw new TPCException("Dimension object does not contain gates");
                return hi_dim[GATES] - lo_dim[GATES];
            }
        }
        /// <summary>
        /// Read-only beds
        /// </summary>
        public double beds
        {
            get
            {
                if (hi_dim.Length <= BEDS)
                    throw new TPCException("Dimension object does not contain beds");
                return hi_dim[BEDS] - lo_dim[BEDS];
            }
        }
        /// <summary>
        /// Read-only detectorvectors
        /// </summary>
        public double detectorvectors
        {
            get
            {
                if (hi_dim.Length <= DETECTORVECTOR)
                    throw new TPCException("Dimension object does not contain detectorvectors");
                return hi_dim[DETECTORVECTOR] - lo_dim[DETECTORVECTOR];
            }
        }
        /// <summary>
        /// Read-only energywindows
        /// </summary>
        public double energywindows
        {
            get
            {
                if (hi_dim.Length <= ENERGYWINDOW)
                    throw new TPCException("Dimension object does not contain energywindows");
                return hi_dim[ENERGYWINDOW] - lo_dim[ENERGYWINDOW];
            }
        }
        /// <summary>
        /// Lowest values that belong to dimension.
        /// </summary>
        protected double[] lo_dim;
        /// <summary>
        /// Highest values that belong to dimension.
        /// </summary>
        protected double[] hi_dim;
        /// <summary>
        /// Number of dimensions in this dimension object.
        /// </summary>
        public int Length { get { return lo_dim.Length; } }
        /// <summary>  
        /// Constructs dimension from another dimension.
        /// </summary> 
        /// <param name="l">copied dimension</param>
        public Limits(Limits l)
        {
            lo_dim = new double[l.Length];
            hi_dim = new double[l.Length];
            for (int i = 0; i < l.Length; i++)
            {
                hi_dim[i] = (double)l.GetLimit(i, Limit.HIGH);
                lo_dim[i] = (double)l.GetLimit(i, Limit.LOW);
            }
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="dimensions">high dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public Limits(params long[] dimensions)
        {
            lo_dim = new double[dimensions.Length];
            hi_dim = new double[dimensions.Length];
            for (int i = 0; i < dimensions.Length; i++)
            {
                if (dimensions[i] < 0) throw new TPCInvalidArgumentsException("Tried to create dimension that is below zero.");
                hi_dim[i] = (double)dimensions[i];
            }
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public Limits(params uint[] dimensions)
        {
            lo_dim = new double[dimensions.Length];
            hi_dim = new double[dimensions.Length];
            for (int i = 0; i < dimensions.Length; i++)
            {
                if (dimensions[i] <= 0) throw new TPCInvalidArgumentsException("Tried to create dimension that is not above zero.");
                hi_dim[i] = (double)dimensions[i];
            }
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public Limits(params int[] dimensions) {
            lo_dim = new double[dimensions.Length];
            hi_dim = new double[dimensions.Length];
            for (int i = 0; i < dimensions.Length; i++)
            {
                if (dimensions[i] < 0) throw new TPCInvalidArgumentsException("Tried to create dimension that is below zero.");
                hi_dim[i] = (double)dimensions[i];
            }
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public Limits(params double[] dimensions) {
            lo_dim = new double[dimensions.Length];
            for (int i = 0; i < dimensions.Length; i++)
            {
                if (dimensions[i] < 0) throw new TPCInvalidArgumentsException("Tried to create dimension that is below zero.");
                
            }
            hi_dim = dimensions;
        }
        /// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="lo_limits">low limits</param>
        /// <param name="hi_limits">high limits</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public Limits(double[] lo_limits, double[] hi_limits)
        {
            for (int i = 0; i < lo_limits.Length; i++)
            {
                if (lo_limits[i] <= 0) throw new TPCInvalidArgumentsException("Low limit (" + i + ") is not above zero.");
                if (hi_limits[i] <= lo_limits[i]) throw new TPCInvalidArgumentsException("High limit ("+i+") is not above low limit.");
            }
            lo_dim = lo_limits;
            hi_dim = hi_limits;
        }
        /// <summary>
        /// Cosntructs dimension object that has zero dimensions.
        /// </summary>
        public Limits() {
            lo_dim = new double[0];
            hi_dim = new double[0];
        }
        /// <summary>
        /// Returns desired dimension value.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <returns>dimension as double value. All dimensions greater than number of actual defined dimensions are regarded as 1.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public double GetDimension(int i)
        {
            if (i < 0) throw new TPCInvalidArgumentsException("Tried to get dimension below zero:"+i+".");
            if (i >= hi_dim.Length) throw new TPCInvalidArgumentsException("Tried to get undefined dimension:"+i+".");
            else return hi_dim[i] - lo_dim[i];
        }
        /// <summary>
        /// Returns all dimensions as new limits object.
        /// </summary>
        /// <returns>dimension as double value. All dimensions greater than number of actual defined dimensions are regarded as 1.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public Limits GetDimensions()
        {
            Limits r = new Limits();
            for(int i = 0; i < Length; i++)
                r.AddLimit(0, GetDimension(i));
            return r;
        }
        /// <summary>
        /// Returns desired limit value.
        /// </summary>
        /// <param name="i">dimension number [1..number of dimensions]</param>
        /// <param name="limit">limit</param>
        /// <returns>limit as double value. All dimensions greater than number of actual defined dimensions are regarded as 1.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public double GetLimit(int i, Limit limit)
        {
            if (i < 0) throw new TPCInvalidArgumentsException("Tried to get limit below zero.");
            if (i >= lo_dim.Length) throw new TPCInvalidArgumentsException("Tried to get "+i+"'s limit above number of dimensions "+(lo_dim.Length-1));
            if (limit == Limit.HIGH) {  return hi_dim[i];  }
            else { return lo_dim[i]; }
        }
        /// <summary>
        /// Sets desired limit value.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="limit">limit</param>
        /// <param name="value">value that is set</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimit(int i, Limit limit, double value)
        {
            if (i < 0) throw new TPCInvalidArgumentsException("Tried to set limit below zero.");
            if (i >= lo_dim.Length) throw new TPCInvalidArgumentsException("Tried to set limit " + i + " above number of dimensions " + (lo_dim.Length-1));
            if (limit == Limit.HIGH) {
                if (value < lo_dim[i]) throw new TPCInvalidArgumentsException("High limit " + i + " is below low limit " + lo_dim[i]);
                hi_dim[i] = value; 
            }
            else {
                if (value < 0) throw new TPCInvalidArgumentsException("Low limit " + value + " is below zero.");
                if (value > hi_dim[i]) throw new TPCInvalidArgumentsException("Low limit " + value + " is above high limit "+hi_dim[i]);
                lo_dim[i] = value;
            }
        }
        /// <summary>
        /// Sets desired limit value.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="low">low limit</param>
        /// <param name="high">high limit</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimits(int i, double low, double high)
        {
            if (i < 0) throw new TPCInvalidArgumentsException("Tried to set limits at negative dimension.");
            if (i >= lo_dim.Length) throw new TPCInvalidArgumentsException("Tried to set limits " + i + " above number of dimensions " + (lo_dim.Length - 1));
            if (low > high) throw new TPCInvalidArgumentsException("Low limit " + low + " is above high limit " + high);
            lo_dim[i] = low;
            hi_dim[i] = high;
        }
        /// <summary>
        /// Adds desired limit value. Limits will be set to next new dimension.
        /// </summary>
        /// <param name="low">value that is set</param>
        /// <param name="high">value that is set</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void AddLimit(double low, double high)
        {
            if (high < low) throw new TPCInvalidArgumentsException("High limit "+high+" is below low limit "+low+".");
            Array.Resize<double>(ref lo_dim, lo_dim.Length + 1);
            Array.Resize<double>(ref hi_dim, hi_dim.Length + 1);
            lo_dim[lo_dim.Length-1] = low;
            hi_dim[hi_dim.Length - 1] = high;
        }
        /// <summary>
        /// Removes limit value. 
        /// </summary>
        /// <param name="dimension">index of dimension that is removed</param>
        /// <exception cref="TPCInvalidArgumentsException">domension not found</exception>
        public void RemoveLimit(int dimension)
        {
            if (dimension < 0 || dimension > Length) 
                throw new TPCInvalidArgumentsException("Dimension "+dimension+" does not exist in limits.");
            double[] tlo_dim = new double[Length-1];
            double[] thi_dim = new double[Length - 1];
            int k = 0;
            for (int i = 0; i < Length; i++)
            {
                tlo_dim[k] = lo_dim[i];
                thi_dim[k] = hi_dim[i];
                if(i != dimension) k++;
            }
            lo_dim = tlo_dim;
            hi_dim = thi_dim;
        }
        /// <summary>
        /// Calculates n first dimension values multiplied together.
        /// </summary>
        /// <param name="dimensions">last dimension that is multiplied [0..No dimensions-1]</param>
        /// <returns>product of dimensions</returns>
        public double GetProduct(int dimensions)
        {
            if (dimensions > hi_dim.Length - 1)
                throw new TPCException("Dimensions for product calculation "+dimensions+" is out of bounds [0.."+(hi_dim.Length-1)+"]");
            double r = 1.0;
            for (int i = 0; i <= dimensions; i++)
                r *= hi_dim[i] - lo_dim[i];
            return r;
        }
        /// <summary>
        /// Calculates dimension values multiplied together.
        /// </summary>
        /// <returns>product of dimensions</returns>
        public double GetProduct() {
            return GetProduct(hi_dim.Length-1);
        }
        /// <summary>
        /// Override of system function.
        /// </summary>
        /// <param name="obj">instance of Limits class</param>
        /// <returns>true if obj is equal to this object</returns>
        public override bool Equals(object obj)
        {
            if(!(obj is Limits)) return false;
            if((obj as Limits).Length != Length) return false;
            for (int i = 0; i < Length; i++)
                if ((obj as Limits).GetLimit(i,Limit.LOW) != lo_dim[i]) return false;
            for (int i = 0; i < Length; i++)
                if ((obj as Limits).GetDimension(i) != GetDimension(i)) return false;
            return true;
        }
        /// <summary>
        /// Rerturns limits as string
        /// </summary>
        /// <returns>string representation</returns>
        public override string ToString()
        {
            string r = "Limits[";
            for (int i = 0; i < lo_dim.Length; i++)
            {
                if (i > 0) r += ",";
                r += "[" + lo_dim[i] + ".." + hi_dim[i] + "]";
            }
            return r+"]";
        }
        /// <summary>
        /// Compares equality of two limits objects.
        /// </summary>
        /// <param name="l1">1st object</param>
        /// <param name="l2">2nd object</param>
        /// <returns>true if objects are equal</returns>
        public static bool operator ==(Limits l1, Limits l2)
        {
            return !(l1 != l2);
        }
        /// <summary>
        /// Compares inequality of two limits objects.
        /// </summary>
        /// <param name="l1">1st object</param>
        /// <param name="l2">2nd object</param>
        /// <returns>true if objects are inequal</returns>
        public static bool operator !=(Limits l1, Limits l2)
        {
            if (l1.Length != l2.Length) return true;
            for (int i = 0; i < l1.Length; i++)
            {
                if (l1.GetLimit(i, Limit.HIGH) != l2.GetLimit(i, Limit.HIGH)) return true;
                if (l1.GetLimit(i, Limit.LOW) != l2.GetLimit(i, Limit.LOW)) return true;
            }
            return false;
        }
        /// <summary>
        /// Converts limits to dimension array. Array has difference between 
        /// low and high values in is indexes.
        /// </summary>
        /// <returns>array if indexes in limits object</returns>
        public double[] ToArray()
        {
            double[] r = new double[lo_dim.Length];
            for (int i = 0; i < r.Length; i++) {
                r[i] = hi_dim[i] - lo_dim[i];
            }
            return r;
        }
        /// <summary>
        /// Implemented because == and != operators require this. Just calls 
        /// parent object's method.
        /// </summary>
        /// <returns>hashcode for object</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
