﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Text;

namespace TPClib
{
    /// <summary>
    /// Data abstraction class of M x N matrix,
    /// where M is rows and N is columns.
    /// </summary>
    public class Matrix
    {
        /// <summary>
        /// Array to hold matrix elements
        /// </summary>
        private double[,] values;

        /// <summary>
        /// Default constructor
        /// </summary>
        public Matrix()
        {
            values = new double[0, 0];
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="m">the number of rows</param>
        /// <param name="n">the number of columns</param>
        public Matrix(int m, int n)
        {
            values = new double[m, n];
        }

        /// <summary>
        /// Square matrix constructor
        /// </summary>
        /// <param name="n">Matrix dimension</param>
        public Matrix(int n)
            : this(n, n) { }

        /// <summary>
        /// Construct and copy from other matrix.
        /// </summary>
        /// <param name="a">Matrix a to copy</param>
        public Matrix(Matrix a)
            : this(a.Rows, a.Columns)
        {
            for (int i = 0; i < Rows; i++)
            {
                for (int j = 0; j < Columns; j++)
                {
                    this[i, j] = a[i, j];
                }
            }
        }

        /// <summary>
        /// Two dimensional index operator
        /// </summary>
        /// <param name="n">Row index</param>
        /// <param name="m">Column index</param>
        /// <returns>Element at [m,n]</returns>
        public virtual double this[int n, int m]
        {
            get { 
                return values[n, m]; 
            }
            set { 
                values[n, m] = value; 
            }
        }

        /// <summary>
        /// Matrix dimensions
        /// </summary>
        public IntLimits Dim {
            get { return new IntLimits(values.GetLength(0), values.GetLength(1)); }
        }

        /// <summary>
        /// The number of rows
        /// </summary>
        public virtual int Rows
        {
            get { return values.GetLength(0); }
        }

        /// <summary>
        /// The number of columns
        /// </summary>
        public virtual int Columns
        {
            get { return values.GetLength(1); }
        }

        /// <summary>
        /// Is this a square matrix of dimension n?
        /// </summary>
        /// <param name="n">Dimension</param>
        /// <returns>True, if matrix is square and of dim n</returns>
        public bool IsSquare(int n)
        {
            return Columns == Rows && n == Columns;
        }

        /// <summary>
        /// Set all matrix elements to a constant c
        /// </summary>
        /// <param name="c">Value the elements are set to</param>
        public void Fill(double c)
        {
            for (int j = 0; j < this.Columns; j++)
            {
                for (int i = 0; i < this.Rows; i++)
                {
                    this[i, j] = c;
                }
            }
        }

        /// <summary>
        /// Matrix transpose
        /// </summary>
        /// <returns>Transpose of this matrix</returns>
        public Matrix Transpose()
        {
            Matrix m = new Matrix(this.Columns, this.Rows);

            for (int i = 0; i < this.Rows; i++)
            {
                for (int j = 0; j < this.Columns; j++)
                {
                    m[j, i] = this[i, j];
                }
            }

            return m;
        }

        /// <summary>
        /// Matrix addition A+B
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Matrix B</param>
        /// <returns>Sum matrix A+B</returns>
        public static Matrix operator +(Matrix a, Matrix b)
        {
            int rows = a.values.GetLength(0);
            int cols = a.values.GetLength(1);

            if (b.values.GetLength(0) != rows || b.values.GetLength(1) != cols)
            {
                throw new ArgumentException();
            }

            Matrix m = new Matrix(rows, cols);

            for (int i = 0; i < rows; i++)
            {
                for (int j = 0; j < cols; j++)
                {
                    m[i, j] = a[i, j] + b[i, j];
                }
            }

            return m;
        }

        /// <summary>
        /// Matrix subtraction A-B
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Matrix B</param>
        /// <returns>Difference matrix A-B</returns>
        public static Matrix operator -(Matrix a, Matrix b)
        {
            int rows = a.values.GetLength(0);
            int cols = a.values.GetLength(1);

            if (b.values.GetLength(0) != rows || b.values.GetLength(1) != cols)
            {
                throw new ArgumentException("Matrix sizes do not match");
            }

            Matrix m = new Matrix(rows, cols);

            for (int i = 0; i < rows; i++)
            {
                for (int j = 0; j < cols; j++)
                {
                    m[i, j] = a[i, j] - b[i, j];
                }
            }

            return m;
        }

        /// <summary>
        /// Naive matrix product AB=C
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Matrix B</param>
        /// <returns>Result matrix C</returns>
        public static Matrix operator *(Matrix a, Matrix b)
        {
            if (b.Rows != a.Columns)
            {
                throw new ArgumentException("Dimension mismatch");
            }

            Matrix c = new Matrix(a.Rows, b.Columns);

            for (int i = 0; i < a.Rows; i++)
            {
                for (int j = 0; j < b.Columns; j++)
                {
                    c[i, j] = 0.0;
                    for (int k = 0; k < a.Columns; k++)
                    {
                        c[i, j] += a[i, k] * b[k, j];
                    }
                }
            }
            return c;
        }

        /// <summary>
        /// Calculates the product vA, where v is vector of R^n and A is a n x m matrix
        /// </summary>
        /// <param name="v">Vector</param>
        /// <param name="a">Matrix</param>
        /// <returns>Product v*A</returns>
        public static Vector operator *(Vector v, Matrix a)
        {
            if (v.Dim != a.Rows)
                throw new ArgumentException("Dimension mismatch");
            
            Vector res = new Vector(a.Columns);
            
            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = Vector.Dot(v, a.GetColumn(i));
            }

            return res;
        }

        /// <summary>
        /// Calculates the product Av, where v is vector of R^m and A is a n x m matrix
        /// </summary>
        /// <param name="v">Vector</param>
        /// <param name="a">Matrix</param>
        /// <returns>Product A*v</returns>
        public static Vector operator *(Matrix a, Vector v)
        {
            if (v.Dim != a.Columns)
                throw new ArgumentException("Dimension mismatch");

            Vector res = new Vector(a.Rows);

            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = Vector.Dot(v, a.GetRow(i));
            }

            return res;
        }

        /// <summary>
        /// Scalar multiplication A*b
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Scalar b</param>
        /// <returns>Matrix A*b</returns>
        public static Matrix operator *(Matrix a, double b)
        {
            int rows = a.values.GetLength(0);
            int cols = a.values.GetLength(1);

            Matrix m = new Matrix(rows, cols);

            for (int i = 0; i < rows; i++)
            {
                for (int j = 0; j < cols; j++)
                {
                    m[i, j] = a[i, j] * b;
                }
            }

            return m;
        }

        /// <summary>
        /// Scalar multiplication b*A
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Scalar b</param>
        /// <returns>Matrix b*A</returns>
        public static Matrix operator *(double b, Matrix a)
        {
            return a * b;
        }

        /// <summary>
        /// Division by scalar A/b.
        /// </summary>
        /// <param name="a">Matrix A</param>
        /// <param name="b">Scalar b</param>
        /// <returns>Matrix A/b</returns>
        public static Matrix operator /(Matrix a, double b)
        {
            if (b == 0.0) throw new DivideByZeroException();
            return a * (1 / b);
        }

        /// <summary>
        /// Cast double array to a m x 1 matrix
        /// </summary>
        /// <param name="v">double array</param>
        /// <returns>Matrix</returns>
        public static implicit operator Matrix(double[] v)
        {
            Matrix m = new Matrix(1, v.Length);
            for (int i = 0; i < v.Length; i++)
            {
                m[0, i] = v[i];
            }
            return m;
        }

        /// <summary>
        /// Cast vector to a m x 1 matrix
        /// </summary>
        /// <param name="v">Vector</param>
        /// <returns>Matrix</returns>
        public static implicit operator Matrix(Vector v)
        {
            Matrix m = new Matrix(1, v.Dim);
            for (int i = 0; i < v.Dim; i++)
            {
                m[0, i] = v[i];
            }
            return m;
        }

        /// <summary>
        /// String representation of this matrix.
        /// </summary>
        /// <returns>String representation</returns>
        public string MakeString()
        {
            StringBuilder sb = new StringBuilder();

            for (int i = 0; i < this.Rows; i++)
            {
                sb.Append("[");
                for (int j = 0; j < this.Columns; j++)
                {
                    sb.AppendFormat(" {0} ", this[i, j]);
                }
                sb.AppendLine("]");
            }
            return sb.ToString();
        }

        /// <summary>
        /// Set a column of this matrix
        /// </summary>
        /// <param name="c">Column index</param>
        /// <returns>Vector holding elements of column c</returns>
        public void SetColumn(int c, Vector v)
        {
            if (c >= Columns || c < 0)
                throw new Exception("Column index " + c + " out of bounds [0," + (Columns - 1) + "]");
            if (v.Length != Rows)
                throw new Exception("Vector length  " + v.Length + " must be equal to number of rows " + Rows);

            // loop the column
            for (int i = 0; i < v.Length; i++)
            {
                this[i, c] = v[i];
            }
        }

        /// <summary>
        /// Get a column from this matrix
        /// </summary>
        /// <param name="c">Column index</param>
        /// <returns>Vector holding elements of column c</returns>
        public Vector GetColumn(int c)
        {
            if (c >= Columns || c < 0)
                throw new Exception("Column index " + c + " out of bounds [0," + (Columns - 1) + "]");
            Vector v = new Vector(Rows);

            // loop the column
            for (int i = 0; i < v.Length; i++)
            {
                v[i] = this[i, c];
            }
            return v;
        }

        /// <summary>
        /// Set a row of this matrix
        /// </summary>
        /// <param name="r">Row index</param>
        /// <returns>Vector holding elements of row r</returns>
        public void SetRow(int r, Vector v)
        {
            if (r >= Rows || r < 0)
                throw new Exception("Row index " + r + " out of bounds [0," + (Rows - 1) + "]");
            if (v.Length != Columns)
                throw new Exception("Vector length  " + v.Length + " must be equal to number of columns " + Columns);

            // loop the column
            for (int i = 0; i < v.Length; i++)
            {
                this[r, i] = v[i];
            }
        }

        /// <summary>
        /// Get a row from this matrix
        /// </summary>
        /// <param name="r">Row index</param>
        /// <returns>Vector holding elements of row c</returns>
        public Vector GetRow(int r)
        {
            if (r >= Rows || r < 0)
                throw new Exception("Row index " + r + " out of bounds [0," + (Rows - 1) + "]");

            Vector v = new Vector(Columns);

            // the row is constant and we
            // loop the column
            for (int i = 0; i < v.Length; i++)
            {
                v[i] = this[r, i];
            }
            return v;
        }

        /// <summary>
        /// String representation
        /// </summary>
        /// <returns>String representation of this matrix</returns>
        public override string ToString()
        {
            return ToString("");
        }

        /// <summary>
        /// Prints contents of this matrix as string
        /// </summary>
        /// <param name="writer">writer where data is is written</param>
        public void PrintAsString(System.IO.TextWriter writer)
        {
            for (int j = 0; j < this.Rows; j++)
            {
                writer.Write("[");
                if (this.Columns > 0)
                    writer.Write(values[j, 0]);
                for (int i = 1; i < this.Columns; i++)
                {
                    writer.Write(" " + values[j, i]);
                }
                writer.Write("]\n");
            }
        }

        /// <summary>
        /// String representation with formatted numeric output
        /// </summary>
        /// <param name="format">Format string</param>
        /// <returns>String representation of this matrix</returns>
        public string ToString(string format)
        {
            string ret = "";
            for (int i = 0; i < this.Rows; i++)
            {
                Vector row = this.GetRow(i);
                ret += row.ToString(format) + "\n";
            }
            return ret;
        }

        /// <summary>
        /// Copy elements of this matrix from matrix A
        /// </summary>
        /// <param name="a">Marix A</param>
        public void Copy(Matrix a)
        {
            values = new double[a.Rows, a.Columns];
            
            for (int i = 0; i < a.Rows; i++)
            {
                for (int j = 0; j < a.Columns; j++)
                {
                    values[i, j] = a[i, j];
                }
            }
        }

        /// <summary>
        /// Gets a submatrix of matrix 'a'
        /// </summary>
        /// <param name="a">Source matrix</param>
        /// <param name="sr">Row index of upperleftmost element</param>
        /// <param name="sc">Column index of upperleftmost element</param>
        /// <param name="er">Row index of lowerrightmost element</param>
        /// <param name="ec">Column index of lowerrightmost element</param>
        /// <returns>Submatrix</returns>
        public static Matrix SubMatrix(Matrix a, int sr, int sc, int er, int ec)
        {
            if ((sr > er) || (sc > ec) || (sr < 0) || (sc < 0) || (ec > a.Columns) || (er > a.Rows))
                throw new ArgumentException("Index out of bounds");

            // Submatrix dimensions
            int srows = er - sr + 1;
            int scols = ec - sc + 1;

            Matrix sub = new Matrix(srows, scols);

            for (int i = 0; i < sub.Rows; i++)
            {
                for (int j = 0; j < sub.Columns; j++)
                {
                    sub[i, j] = a[sr + i, sc + j];
                }
            }
            return sub;
        }

        /// <summary>
        /// N x n identity matrix
        /// </summary>
        /// <param name="n">Dimension</param>
        /// <returns>Identity matrix</returns>
        public static Matrix Identity(int n)
        {
            Matrix m = new Matrix(n);
            for (int i = 0; i < n; i++)
            {
                m[i, i] = 1.0;
            }
            return m;
        }

        /// <summary>
        /// Expand the matrix to upper left.
        /// MAtrix elements are set to 0, except diagonal elements are set to 1.
        /// </summary>
        /// <param name="a">Matrix to expand</param>
        /// <param name="m">Number of rows to add</param>
        /// <param name="n">Number of columns to add</param>
        /// <returns>Expanded matrix</returns>
        public static Matrix Expand(Matrix a, int m, int n)
        {
            Matrix b = new Matrix(a.Rows + m, a.Columns + n);
            for (int i = 0; i < b.Rows; i++)
            {
                for (int j = 0; j < b.Columns; j++)
                {
                    if (i < m || j < n)
                    {
                        b[i, j] = (i == j) ? 1 : 0;
                    }
                    else
                    {
                        b[i, j] = a[i - m, j - n];
                    }
                }
            }
            return b;
        }

        /// <summary>
        /// Expand in storage the covariance matrix covar, 
        /// so as to take into account parameters that
        /// are being held fixed. (For the latter, return zero covariances.)
        /// </summary>
        /// <param name="covar">the covariance matrix</param>
        /// <param name="ma"></param>
        /// <param name="isFitted">the array of fitting parameters</param>
        public static void CovarianceSort(Matrix covar, int ma, bool[] isFitted)
        {
            int mfit = 0;
            foreach (bool b in isFitted) if (b==true) mfit++;

            if (!(0 <= mfit && mfit <= ma))
                throw new Exception("!(0 <= mfit && mfit <= ma)");

            if (covar.IsSquare(isFitted.Length-1))
                throw new Exception("");

            for (int i = mfit; i < ma; i++)
            {
                for (int j = 0; j < i; j++)
                {
                    covar[i, j] = covar[j, i] = 0.0;
                }
            }

            int k = mfit - 1; // the last index
            double swap;

            for (int j = ma - 1; // start the last index
                j >= 0; // until the first index
                j--)
            {
                if (isFitted[j])
                {
                    for (int i = 0; i < ma; i++)
                    {
                        swap = covar[i, k];
                        covar[i, k] = covar[i, j];
                        covar[i, j] = swap;
                    }
                    for (int i = 0; i < ma; i++)
                    {
                        swap = covar[k, i];
                        covar[k, i] = covar[j, i];
                        covar[j, i] = swap;
                    }
                    k--;
                }
            }
        }
        /// <summary>
        /// Inverts this matrix
        /// ftp://garbo.uwasa.fi/pc/c-lang/matrx042.zip
        /// </summary>
        public void Invert() {
            Matrix A = Inverse(this);
            this.Copy(A);
        }

        /// <summary>
        /// in-place LU decomposition with partial pivoting
        /// <remarks>the LU decomposed may NOT be equal to the LU of
        ///		the orignal matrix a. But equal to the LU of the
        ///		rows interchanged matrix.</remarks>
        /// ftp://garbo.uwasa.fi/pc/c-lang/matrx042.zip
        /// </summary>
        /// <param name="A">square matrix (n x n), A will be overwritten to be a LU-composite matrix</param>
        /// <param name="P">permutation vector (n x 1)</param>
        /// <returns>number of permutation performed -1 means suspected singular matrix</returns>
        public static int LUDecomposition(ref Matrix A, ref Matrix P)
        {
	        int	i, j, k, n;
	        int	maxi;
            double tmp;
	        double c, c1;
	        int	p;

	        n = A.Columns;

	        for (p=0,i=0; i<n; i++) {
		        P[i,0] = i;
		    }

	        for (k=0; k<n; k++) {
	            // partial pivoting ---
	            for (i=k, maxi=k, c=0.0; i<n; i++) {
                    c1 = Math.Abs( A[(int)P[i,0],k] );
		            if (c1 > c) {
		                c = c1;
		                maxi = i;
		            }
		        }

	            // row exchange, update permutation vector
	            if (k != maxi) {
		            p++;
		            tmp = P[k,0];
		            P[k,0] = P[maxi,0];
		            P[maxi,0] = tmp;
		        }

	            // suspected singular matrix
	            if ( A[(int)P[k,0],k] == 0.0 )
		            return (-1);

	            for (i=k+1; i<n; i++) {
		            // calculate m(i,j) 
		            A[(int)P[i,0],k] = A[(int)P[i,0],k] / A[(int)P[k,0],k];

		            // elimination 
		            for (j=k+1; j<n; j++) {
                        A[(int)P[i,0],j] -= A[(int)P[i,0],k] * A[(int)P[k,0],j];
			        }
		        }
	        }
	        return (p);
        }

        /// <summary>
        /// back substitution
        /// ftp://garbo.uwasa.fi/pc/c-lang/matrx042.zip
        /// </summary>
        /// <param name="A">square matrix A (LU composite)</param>
        /// <param name="B">column matrix B, B will be overwritten</param>
        /// <param name="X">place to put the result of X</param>
        /// <param name="P">Permutation vector (after calling LUDecomposition)</param>
        /// <param name="xcol">column of x to put the result</param>
        /// <returns>column matrix X (of AX = B)</returns>
        protected static Matrix Backsubs(Matrix A, ref Matrix B, out Matrix X, Matrix P, int xcol)
        {
	        int	i, j, k, n;
	        double	sum;
            X = new Matrix(A.Rows, A.Columns);
	        n = A.Columns;

	        for (k=0; k<n; k++) {
	            for (i=k+1; i<n; i++)
                    B[(int)P[i,0],0] -= A[(int)P[i,0],k] * B[(int)P[k,0],0];
	        }

            X[n-1,xcol] = B[(int)P[n-1,0],0] / A[(int)P[n-1,0],n-1];
	        for (k=n-2; k>=0; k--) {
		        sum = 0.0;
		        for (j=k+1; j<n; j++)
		        {
                    sum += A[(int)P[k,0],j] * X[j,xcol];
		        }
                X[k,xcol] = (B[(int)P[k,0],0] - sum) / A[(int)P[k,0],k];
		    }
	        return (X);
        }
        /// <summary>
        /// Inverts matrix
        /// ftp://garbo.uwasa.fi/pc/c-lang/matrx042.zip
        /// </summary>
        /// <param name="M">matrix to be inverted</param>
        /// <returns>inverted matrix</returns>
        public static Matrix Inverse(Matrix M) {
            Matrix A = new Matrix(M.Columns);
            Matrix B, C, P;
            int i;
            
            int n = M.Columns;
            A.Copy(M);
            B = new Matrix(n, 1);
            C = new Matrix(n, n);
            //pivot vector
            P = new Matrix(n, 1);

            //- LU-decomposition - also check for singular matrix
            if (Matrix.LUDecomposition(ref A, ref P) == -1)
            {
                throw new TPCException("Probable singular matrix - cannot do inversion");
            }

            for (i = 0; i < n; i++)
            {
                B.Fill(0.0);
                B[i,0] = 1.0;
                Matrix.Backsubs(A, ref B, out C, P, i);
            }
            return (C);
        }

        /// <summary>
        /// Inverts the first 3x3 block of a matrix.
        /// </summary>
        /// <param name="M">3x3 matrix of bigger</param>
        /// <returns>Matrix of which the first 3x3 block is inverted.</returns>
        public static Matrix Inverse3x3(Matrix M)
        {
            if (M.Columns < 3 || M.Rows < 3)
                throw new TPCException("Matrix is smaller than 3x3.");

            // 1. Find det(M)
            double det = M[0, 0] * (M[1, 1] * M[2, 2] - M[1, 2] * M[2, 1])
                       - M[0, 1] * (M[1, 0] * M[2, 2] - M[1, 2] * M[2, 0])
                       + M[0, 2] * (M[1, 0] * M[2, 1] - M[1, 1] * M[2, 0]);

            // 2. Transpose M to obtain M^T
            Matrix t = M.Transpose();

            // 3. Find the matrix of cofactors by first calculating the matrix of minors
            Matrix m = new Matrix(3);

            m[0, 0] = t[1, 1] * t[2, 2] - t[1, 2] * t[2, 1];
            m[0, 1] = t[1, 0] * t[2, 2] - t[1, 2] * t[2, 0];
            m[0, 2] = t[1, 0] * t[2, 1] - t[1, 1] * t[2, 0];

            m[1, 0] = t[0, 1] * t[2, 2] - t[0, 2] * t[2, 1];
            m[1, 1] = t[0, 0] * t[2, 2] - t[0, 2] * t[2, 0];
            m[1, 2] = t[0, 0] * t[2, 1] - t[0, 1] * t[2, 0];

            m[2, 0] = t[0, 1] * t[1, 2] - t[0, 2] * t[1, 1];
            m[2, 1] = t[0, 0] * t[1, 2] - t[0, 2] * t[1, 0];
            m[2, 2] = t[0, 0] * t[1, 1] - t[0, 1] * t[1, 0];

            // find the adjacent
            m[0, 1] = -m[0, 1];
            m[1, 0] = -m[1, 0];
            m[2, 1] = -m[2, 1];
            m[1, 2] = -m[1, 2];

            // 4. Find the inverse from the rule: M^-1 = 1/det(M)*Adj(M)
            Matrix result = 1 / det * m;

            return result;
        }

        private const double MATRIX_INVERSE_EPSILON = 0.0000001;

        /// <summary>
        /// Inverts the first 4x4 block of a matrix.
        /// Adapted from Id Software's Doom 3 GPL open source code
        /// </summary>
        /// <remarks>104 (84+4+16) multiplications and 1 division</remarks>
        /// <param name="mat">4x4 matrix or bigger</param>
        /// <returns>Matrix of which the first 4x4 block is inverted.</returns>
        public static Matrix Inverse4x4(Matrix mat)
        {
            double det, invDet;

            // 2x2 sub-determinants required to calculate 4x4 determinant
            double det2_01_01 = mat[0, 0] * mat[1, 1] - mat[0, 1] * mat[1, 0];
            double det2_01_02 = mat[0, 0] * mat[1, 2] - mat[0, 2] * mat[1, 0];
            double det2_01_03 = mat[0, 0] * mat[1, 3] - mat[0, 3] * mat[1, 0];
            double det2_01_12 = mat[0, 1] * mat[1, 2] - mat[0, 2] * mat[1, 1];
            double det2_01_13 = mat[0, 1] * mat[1, 3] - mat[0, 3] * mat[1, 1];
            double det2_01_23 = mat[0, 2] * mat[1, 3] - mat[0, 3] * mat[1, 2];

            // 3x3 sub-determinants required to calculate 4x4 determinant
            double det3_201_012 = mat[2, 0] * det2_01_12 - mat[2, 1] * det2_01_02 + mat[2, 2] * det2_01_01;
            double det3_201_013 = mat[2, 0] * det2_01_13 - mat[2, 1] * det2_01_03 + mat[2, 3] * det2_01_01;
            double det3_201_023 = mat[2, 0] * det2_01_23 - mat[2, 2] * det2_01_03 + mat[2, 3] * det2_01_02;
            double det3_201_123 = mat[2, 1] * det2_01_23 - mat[2, 2] * det2_01_13 + mat[2, 3] * det2_01_12;

            det = (-det3_201_123 * mat[3, 0] + det3_201_023 * mat[3, 1] - det3_201_013 * mat[3, 2] + det3_201_012 * mat[3, 3]);

            if (Math.Abs(det) < MATRIX_INVERSE_EPSILON)
            {
                throw new TPCException("determinant is zero");
            }

            invDet = 1.0f / det;

            // remaining 2x2 sub-determinants
            double det2_03_01 = mat[0, 0] * mat[3, 1] - mat[0, 1] * mat[3, 0];
            double det2_03_02 = mat[0, 0] * mat[3, 2] - mat[0, 2] * mat[3, 0];
            double det2_03_03 = mat[0, 0] * mat[3, 3] - mat[0, 3] * mat[3, 0];
            double det2_03_12 = mat[0, 1] * mat[3, 2] - mat[0, 2] * mat[3, 1];
            double det2_03_13 = mat[0, 1] * mat[3, 3] - mat[0, 3] * mat[3, 1];
            double det2_03_23 = mat[0, 2] * mat[3, 3] - mat[0, 3] * mat[3, 2];

            double det2_13_01 = mat[1, 0] * mat[3, 1] - mat[1, 1] * mat[3, 0];
            double det2_13_02 = mat[1, 0] * mat[3, 2] - mat[1, 2] * mat[3, 0];
            double det2_13_03 = mat[1, 0] * mat[3, 3] - mat[1, 3] * mat[3, 0];
            double det2_13_12 = mat[1, 1] * mat[3, 2] - mat[1, 2] * mat[3, 1];
            double det2_13_13 = mat[1, 1] * mat[3, 3] - mat[1, 3] * mat[3, 1];
            double det2_13_23 = mat[1, 2] * mat[3, 3] - mat[1, 3] * mat[3, 2];

            // remaining 3x3 sub-determinants
            double det3_203_012 = mat[2, 0] * det2_03_12 - mat[2, 1] * det2_03_02 + mat[2, 2] * det2_03_01;
            double det3_203_013 = mat[2, 0] * det2_03_13 - mat[2, 1] * det2_03_03 + mat[2, 3] * det2_03_01;
            double det3_203_023 = mat[2, 0] * det2_03_23 - mat[2, 2] * det2_03_03 + mat[2, 3] * det2_03_02;
            double det3_203_123 = mat[2, 1] * det2_03_23 - mat[2, 2] * det2_03_13 + mat[2, 3] * det2_03_12;

            double det3_213_012 = mat[2, 0] * det2_13_12 - mat[2, 1] * det2_13_02 + mat[2, 2] * det2_13_01;
            double det3_213_013 = mat[2, 0] * det2_13_13 - mat[2, 1] * det2_13_03 + mat[2, 3] * det2_13_01;
            double det3_213_023 = mat[2, 0] * det2_13_23 - mat[2, 2] * det2_13_03 + mat[2, 3] * det2_13_02;
            double det3_213_123 = mat[2, 1] * det2_13_23 - mat[2, 2] * det2_13_13 + mat[2, 3] * det2_13_12;

            double det3_301_012 = mat[3, 0] * det2_01_12 - mat[3, 1] * det2_01_02 + mat[3, 2] * det2_01_01;
            double det3_301_013 = mat[3, 0] * det2_01_13 - mat[3, 1] * det2_01_03 + mat[3, 3] * det2_01_01;
            double det3_301_023 = mat[3, 0] * det2_01_23 - mat[3, 2] * det2_01_03 + mat[3, 3] * det2_01_02;
            double det3_301_123 = mat[3, 1] * det2_01_23 - mat[3, 2] * det2_01_13 + mat[3, 3] * det2_01_12;

            mat[0, 0] = -det3_213_123 * invDet;
            mat[1, 0] = +det3_213_023 * invDet;
            mat[2, 0] = -det3_213_013 * invDet;
            mat[3, 0] = +det3_213_012 * invDet;

            mat[0, 1] = +det3_203_123 * invDet;
            mat[1, 1] = -det3_203_023 * invDet;
            mat[2, 1] = +det3_203_013 * invDet;
            mat[3, 1] = -det3_203_012 * invDet;

            mat[0, 2] = +det3_301_123 * invDet;
            mat[1, 2] = -det3_301_023 * invDet;
            mat[2, 2] = +det3_301_013 * invDet;
            mat[3, 2] = -det3_301_012 * invDet;

            mat[0, 3] = -det3_201_123 * invDet;
            mat[1, 3] = +det3_201_023 * invDet;
            mat[2, 3] = -det3_201_013 * invDet;
            mat[3, 3] = +det3_201_012 * invDet;

            return mat;
        }

        /// <summary>
        /// Compares two matrices.
        /// </summary>
        /// <param name="other">Compared matrix</param>
        /// <param name="epsilon">Maximum absolut error in every element.</param>
        /// <returns>true, if every matrix element equals with every other matrix element.</returns>
        public bool Equals(Matrix other, double epsilon)
        {
            if (this == other)
                return true;
            if (this.Columns != other.Columns)
                return false;
            if (this.Rows != other.Rows)
                return false;

            for (int i = 0; i < other.Rows; i++)
            {
                for (int j = 0; j < other.Columns; j++)
                {
                    if (Math.Abs(other[i, j] - this[i, j]) > epsilon)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// Implicit cast from multidimensional array
        /// </summary>
        /// <param name="a">casted array, must have cubic size</param>
        /// <returns>matrix</returns>
        public static implicit operator Matrix(double[,] a)
        {
            Matrix m = new Matrix(a.GetLength(0), a.GetLength(1));
            for (int i = 0; i < a.GetLength(0); i++)
                for (int j = 0; j < a.GetLength(1); j++)
                    m[i, j] = a[i, j];
            return m;
        }

        /// <summary>
        /// Explicit cast to vector
        /// </summary>
        /// <param name="v">Matrix to cast that has only one row</param>
        /// <returns>Vector holding the matrix's 1st rows elements</returns>
        public static explicit operator Vector(Matrix m)
        {
            if (m.Rows > 1) throw new TPCException("Cannot cast multirow matrix to vector.");
            return m.GetRow(0);
        }

        /// <summary>
        /// Implicit cast to multidimensional array
        /// </summary>
        /// <param name="m">casted matrix</param>
        /// <returns>multidimensional array</returns>
        public static implicit operator double[,](Matrix m) {
            double[,] r = new double[m.Rows, m.Columns];
            for(int i = 0; i < m.Rows; i++)
                for (int j = 0; j < m.Columns; j++)
                    r[i, j] = m[i, j];
            return r;
        }

        /// <summary>
        /// Implicit cast to jagged array
        /// </summary>
        /// <param name="m">casted matrix</param>
        /// <returns>multidimensional array</returns>
        public static implicit operator double[][](Matrix m)
        {
            double[][] r = new double[m.Rows][];
            for (int i = 0; i < m.Rows; i++)
            {
                r[i] = new double[m.Columns];
                for (int j = 0; j < m.Columns; j++)
                    r[j][i] = m[j, i];
            }
            return r;
        }

        /// <summary>
        /// Implicit cast to 1-dimensional array
        /// </summary>
        /// <param name="m">casted matrix</param>
        /// <returns>1-dimensional array</returns>
        public static implicit operator double[](Matrix m)
        {
            double[] r = new double[m.Columns*m.Rows];
            int k = 0;
            for (int i = 0; i < m.Rows; i++)
                for (int j = 0; j < m.Columns; j++)
                    r[k++] = m[i, j];
            return r;
        }
    }
}
