﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// 
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Brent
    {
        private static double CGOLD = 0.3819660;
        private static int MaxBrentIter = 100;
        private static double ZEPS = 1.0e-10;
        private static double GOLD = 1.618034d;
        private static double GLIMIT = 100.0d;
        // tiny is used to prevent any possible division by zero
        private static double TINY = 1.0e-20;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <returns></returns>
        private static double Sign(double a, double b)
        {
            return ((b >= 0.0) ? Math.Abs(a) : -Math.Abs(a));
        }

        /// <summary>
        /// Parabolic interpolation which isolates the minimum to a
        /// fractional precision of about tol using Brent's method.
        /// </summary>
        /// <param name="ax"></param>
        /// <param name="bx"></param>
        /// <param name="cx"></param>
        /// <param name="tol"></param>
        /// <param name="func"></param>
        /// <param name="xmin">The abscissa of the minimum</param>
        /// <returns>the minimum function value between [ax,cx]</returns>
        public double Minimize(double ax, double bx, double cx, double tol, Function func, out double xmin)
        {
            #region Preconditions
            if ((ax <= bx && bx <= cx) || (cx <= bx && bx <= ax))
            {
                if (tol < Math.Sqrt(Double.Epsilon))
                    throw new OptimizationException("Tolerance is too small.");
            }
            else
            {
                throw new OptimizationException("ax=" + ax + " bx=" + bx + " cx=" + cx);
            }
            #endregion

            // Calculates the function values at ax, bx, cx.
            double fax = func(ax);
            double fbx = func(bx);
            double fcx = func(cx);

            if (fax < fbx || fcx < fbx)
                throw new OptimizationException(
                    "fax=" + fax + " fbx=" + fbx + " fcx=" + fcx);

            // a and b - the minimum is bracketed between a and b
            double a, b;
            // x is the point with the recent lowest so far.
            // w is the second lowest.
            // v is the previous value of w
            double x, w, v;
            // u is the point where functio was last evaluated.
            // xm is the midpoint between a and b (not evaluated.
            double u, xm;

            double d = 0.0, etemp, fu, fv, fw, fx, p;
            double q, r, tol1, tol2; ;
            // This will be the distance moved on the step before last.
            double e = 0.0;

            // a and b must be in ascending order,
            // but input abscissas need not be.
            a = (ax < cx ? ax : cx);
            b = (ax > cx ? ax : cx);

            // Initializations...
            x = w = v = bx;
            fw = fv = fx = func(x);

            // Main program loop.
            int iter;
            for (iter = 1; iter <= MaxBrentIter; iter++)
            {
                xm = 0.5 * (a + b);
                tol1 = tol * Math.Abs(x) + ZEPS;
                tol2 = 2.0 * tol1;

                // Tests for done here.
                if (Math.Abs(x - xm) <= (tol2 - 0.5 * (b - a)))
                {
                    xmin = x;
                    return fx;
                }

                if (Math.Abs(e) > tol1)
                {
                    r = (x - w) * (fx - fv);
                    q = (x - v) * (fx - fw);
                    p = (x - v) * q - (x - w) * r;
                    q = 2.0 * (q - r);
                    if (q > 0.0)
                    {
                        p = -p;
                    }
                    q = Math.Abs(q);
                    etemp = e;
                    e = d;

                    if (Math.Abs(p) >= Math.Abs(0.5 * q * etemp)
                        || p <= q * (a - x) || p >= q * (b - x))
                    {
                        e = (x >= xm ? a - x : b - x);
                        d = CGOLD * e;
                    }
                    // The above conditions determine the acceptability
                    // of the parabolic fit. Here we take the golden
                    // section step into the larger of the two segments.
                    else
                    {
                        // Take the parabolic step
                        d = p / q;
                        u = x + d;
                        if (u - a < tol2 || b - u < tol2)
                        {
                            d = Sign(tol1, xm - x);
                        }
                    }
                }
                else
                {
                    e = (x >= xm ? a - x : b - x);
                    d = CGOLD * e;
                }

                u = (Math.Abs(d) >= tol1 ? x + d : x + Sign(tol1, d));
                fu = func(u);

                // This is the one function evaluation per iteration.
                if (fu <= fx)
                {
                    // Now decidde what to do with or function
                    // evaluation. 
                    if (u >= x)
                        a = x;
                    else
                        b = x;

                    // Housekeeping follows.
                    v = w;
                    w = x;
                    x = u;

                    fv = fw;
                    fw = fx;
                    fx = fu;
                }
                else
                {
                    if (u < x)
                        a = u;
                    else
                        b = u;

                    if (fu <= fw || w == x)
                    {
                        v = w;
                        w = u;
                        fv = fw;
                        fw = fu;
                    }
                    else if (fu <= fv || v == x || v == w)
                    {
                        v = u;
                        fv = fu;
                    }
                }

                // Done with housekeeping. Back for another iteration.
            }

            // Never get here.
            if (iter >= MaxBrentIter)
            {
                throw new Exception("Too many iterations in brent.");
            }
            xmin = x;
            return fx;
        }

        /// <summary>
        /// Routine for initially bracketinga a minimum.
        /// </summary>
        public void Bracket(double ax, double bx, Function func, out double axx, out double bxx, out double cx,
            out double fa, out double fb, out double fc)
        {
            axx = ax;
            bxx = bx;
            fa = func(ax);
            fb = func(bx);

            // Switch roles of a and b so that we can go
            // downhill in the direction from a to b.
            if (fb > fa)
            {
                double dum = ax;
                axx = ax = bx;
                bxx = bx = dum;

                dum = fb;
                fb = fa;
                fa = dum;
            }

            if (fb > fa)
                throw new Exception("");

            // This is first guess for c.
            cx = bx + GOLD * (bx - ax);
            fc = func(cx);

            // Keeps returning here until we bracket.
            while (fb > fc)
            {
                // Computes u by parabolic extrapolation from a,b,c.
                double r = (bx - ax) * (fb - fc);
                double q = (bx - cx) * (fb - fa);
                double u = bx - ((bx - cx) * q - (bx - ax) * r) /
                    (2.0 * Sign(Math.Max(Math.Abs(q - r), TINY), q - r));

                // We won't go farther than this. 
                double ulim = bx + GLIMIT * (cx - bx);

                //
                double fu;

                // Next, Tests various possibilities.

                // parabolic u is between b and c: try it.
                if ((bx - u) * (u - cx) > 0.0)
                {
                    fu = func(u);

                    // go a minimum between b and c
                    if (fu < fc)
                    {
                        axx = ax = bx;
                        bxx = bx = u;
                        fa = fb;
                        fb = fu;
                        return;
                    }
                    // Got a minimum between a and u
                    else if (fu > fb)
                    {
                        cx = u;
                        fc = fu;
                        return;
                    }

                    // parabolic fit was no use. Use default magnification.
                    u = cx + GOLD * (cx - bx);
                    fu = func(u);

                }
                // Parabolic fit is between c and its allowed limit.
                else if ((cx - u) * (u - ulim) > 0.0)
                {
                    fu = func(u);

                    if (fu < fc)
                    {
                        bx = cx;
                        cx = u;
                        u = cx + GOLD * (cx - bx);

                        fb = fc;
                        fc = fu;
                        fu = func(u);
                    }
                }
                // Limit parabolic u to maximum allowed value.
                else if ((u - ulim) * (ulim - cx) >= 0.0)
                {
                    u = ulim;
                    fu = func(u);
                }
                // Reject parabolic u, use default magnification.
                else
                {
                    u = cx + GOLD * (cx - bx);
                    fu = func(u);
                }

                // Eliminates oldest point and continue while loop.
                axx = ax = bx;
                bxx = bx = cx;
                cx = u;

                fa = fb;
                fb = fc;
                fc = fu;
            }

            axx = ax;
            bxx = bx;
        }
    }
}