﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{

    /// <summary>
    /// By Chunlei Han. May 10, 2009
    /// 3-tissue compartmental model in parallel
    ///     K1       k3     
    /// Ca<---->C1<----->C2
    ///     k2   ^   k4
    ///          |
    ///       k5 | k6
    ///          |
    ///          v
    ///          C3
    ///          
    /// Reference: http://www.turkupetcentre.net/reports/tpcmod0001.pdf
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class CM3TParallel : Model
    {
        /// <summary>
        /// Vectors for storing time intervals and reference values
        /// </summary>
        private Vector deltaTime;
        private Vector C1;
        private Vector C2;
        private Vector C3;

        private Vector inputFunction;
        private Vector iInputFuntion;

        /// <summary>
        /// Number of samples simulated
        /// </summary>
        public override int Samples
        {
            get { return deltaTime.Length; }
        }


        /// <summary>
        /// Return info of Model
        /// </summary>
        /// <returns>Info of model</returns>
        protected override ModelInfo GetInfo()
        {
            ModelInfo info = new ModelInfo(
                "CM3TParallel",
                "This is CM3TParallel description...", "deltaTime", "C1?", "C2?", "C3?");

            List<ModelParameter> parameters = new List<ModelParameter>();
            parameters.Add(new ModelParameter("V0", 0, 1, 0.5));
            parameters.Add(new ModelParameter("k1", 0, 0.01, 0.01));
            parameters.Add(new ModelParameter("k2", 0, 0.001, 0.01));
            parameters.Add(new ModelParameter("k3", 0, 0.01, 0.01));
            parameters.Add(new ModelParameter("k4", 0, 0.001, 0.01));
            parameters.Add(new ModelParameter("k5", 0, 0.01, 0.01));
            parameters.Add(new ModelParameter("k6", 0, 0.001, 0.01));
            info.Parameters = parameters.ToArray();
            info.ClassType = this.GetType();

            return info;
        }


        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Reference value vector</param>
        public override void Create(Vector t, params Vector[] inputf)
        {
            if (inputf.Length != t.Length) throw new ModelException("Different number of sample times and values");

            model_times = t;

            deltaTime = HalfTimeDifs(t);
            if (inputf.Length > 0) inputFunction = inputf[0];
            iInputFuntion = CalcRefIntegral(inputf[0], deltaTime);
            C1 = new Vector(Samples);
            C2 = new Vector(Samples);
            C3 = new Vector(Samples);
        }

        /// <summary>
        /// Constructor
        /// </summary>
        public CM3TParallel() { Info = GetInfo(); }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Input function value vector</param>
        public CM3TParallel(Vector t, Vector inputf)
        {
            Create(t, inputf);
        }

        /// <summary>
        /// Wrapper function. Overrides Model.Simulate.
        /// </summary>
        /// <param name="par">Model parameters</param>
        /// <returns>Vector of simulated values</returns>
        public override Vector Simulate(params double[] par)
        {
            if (par.Length != 7) throw new ModelException("Wrong number of parameters");
            return Simulate(par[0], par[1], par[2], par[3], par[4], par[5], par[6]);
        }


        public Vector Simulate(double V0, double k1, double k2, double k3, double k4, double k5, double k6)
        {
            Vector simval = new Vector(Samples);
            //Cumulative integral values;
            double iC1 = 0;
            double iC2 = 0;
            double iC3 = 0;
            double iC1_last = 0;
            double iC2_last = 0;
            double iC3_last = 0;
            double iInput = 0;

            double C1_last = 0;
            double C2_last = 0;
            double C3_last = 0;

            double dt2 = 0;

            double ca, c1f, c1s, c2f, c2s, c3f, c3s, c1base;

            for (int i = 0; i < Samples; i++)
            {
                dt2 = deltaTime[i];
                //input = inputFunction[i];
                iInput = iInputFuntion[i];

                //calculate C1;
                //Ca term in upper part 
                ca = k1 * iInput;
                //C1 term in upper part 
                c1f = k2 + k3 / (1 + dt2 * k4) + k5 / (1 + dt2 * k6);
                c1s = iC1_last + dt2 * C1_last;
                //C2 term
                c2f = k4 / (1 + dt2 * k4);
                c2s = iC2_last + dt2 * C2_last;
                //C3 term
                c3f = k6 / (1 + dt2 * k6);
                c3s = iC3_last + dt2 * C3_last;
                //
                c1base = 1 + dt2 * c1f;

                //new c1
                C1[i] = (ca - c1f * c1s + c2f * c2s + c3f * c3s) / c1base;
                iC1 = iC1_last + dt2 * (C1_last + C1[i]);

                //calculate c2
                C2[i] = (k3 * iC1 - k4 * c2s) / (1 + dt2 * k4);
                iC2 = iC2_last + dt2 * (C2_last + C2[i]);

                C3[i] = (k5 * iC1 - k6 * c3s) / (1 + dt2 * k6);
                iC3 = iC3_last + dt2 * (C3_last + C3[i]);


                //non-negative constrain
                if (C1[i] < 0) C1[i] = 0;
                if (C2[i] < 0) C2[i] = 0;
                if (C3[i] < 0) C3[i] = 0;

                simval[i] = V0 * inputFunction[i] + C1[i] + C2[i] + C3[i];

                C1_last = C1[i];
                C2_last = C2[i];
                C3_last = C3[i];

                iC1_last = iC1;
                iC2_last = iC2;
                iC3_last = iC3;

            }
            return simval;
        }
    }
}

