﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// K-means clustering. Clusters set of points around k mean points.
    /// The metric used to calculate the distance between points can be freely
    /// chosen (default is the Euclidean distance).
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Cluster
    {
        /// <summary>
        /// The mean of this cluster
        /// </summary>
        public Vector mean;

        /// <summary>
        /// Set of points in this cluster
        /// </summary>
        public List<Vector> points;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="m">Cluster mean</param>
        private Cluster(Vector m)
        {
            mean = m;
            points = new List<Vector>();
        }

        /// <summary>
        /// Add a point to this cluster
        /// </summary>
        /// <param name="point">Point to add</param>
        private void Add(Vector point)
        {
            points.Add(point);
        }

        /// <summary>
        /// Remove all points
        /// </summary>
        private void Clear()
        {
            points.Clear();
        }

        /// <summary>
        /// Calculate new mean
        /// </summary>
        /// <returns>True, if this clusters mean changed</returns>
        private bool Recalculate()
        {
            bool changed = false;

            if (points.Count > 0)
            {
                Vector newmean = Vector.Sum(points.ToArray()) / points.Count;
                if (newmean != mean)
                {
                    changed = true;
                    mean = newmean;
                }
            }

            return changed;
        }

        /// <summary>
        /// Form k cluster from a set of points using a specified metric
        /// </summary>
        /// <param name="points">Points to cluster</param>
        /// <param name="k">Number of clusers to form</param>
        /// <param name="Metric">Metric used to cluster points</param>
        /// <returns>Array of clusters</returns>
        public static Cluster[] MakeClusters(Vector[] points, int k, RealFunction Metric)
        {
            if (points == null || Metric == null || k <= 0)
                throw new ClusterException("Invalid argument");

            int count = points.Length;
            Cluster[] clusters = new Cluster[k];
            bool changed = true;

            Random rand = new Random();

            // Assign random points as cluster means
            for (int i = 0; i < clusters.Length; i++)
            {
                int n = rand.Next(count);
                clusters[i] = new Cluster(points[n]);
            }

            // Iterate until clusters stabilize
            // (i.e. the cluster means do not change)
            while (changed)
            {
                changed = false;

                Vector point;
                Cluster closest;

                foreach (Cluster c in clusters)
                {
                    c.Clear();
                }

                for (int i = 0; i < count; i++)
                {
                    point = points[i];
                    closest = clusters[NearestMean(clusters, point, Metric)];
                    closest.Add(point);
                }

                foreach (Cluster c in clusters)
                {
                    bool cluster_change = c.Recalculate();
                    changed = (changed || cluster_change);
                }
            }
            return clusters;
        }

        /// <summary>
        /// Form k cluster from a set of points using Euclidean distance as a metric
        /// </summary>
        /// <param name="points">Set of points to cluster</param>
        /// <param name="k">Number of clusters</param>
        /// <returns>Array of clusters</returns>
        public static Cluster[] MakeClusters(Vector[] points, int k)
        {
            return MakeClusters(points, k, Vector.Norm);
        }

        /// <summary>
        /// Calculate the nearest cluster mean for a point
        /// </summary>
        /// <param name="clusters">Set of clusters</param>
        /// <param name="v">Point</param>
        /// <param name="Metric">Metric used</param>
        /// <returns>Index of the nearest cluster</returns>
        private static int NearestMean(Cluster[] clusters, Vector v, RealFunction Metric)
        {
            double min = Double.MaxValue;
            double dist;
            int index = 0;

            for (int i = 0; i < clusters.Length; i++)
            {
                Vector m = clusters[i].mean;
                dist = Metric(m - v);

                if (dist < min)
                {
                    min = dist;
                    index = i;
                }
            }
            return index;
        }
    }
}
