﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class CpBloodInputModel : Model
    {
        private const int valuesBetween = 10;
        private const double kg = 0.5 / 60.0;

        Vector simulated;
        double[] times;
        double[] measureTimes;

        int samples;

        protected override ModelInfo GetInfo()
        {
            ModelInfo info = new ModelInfo("CpBloodInputModel", "This is RTCMModel description...", false);

            List<ModelParameter> parameters = new List<ModelParameter>();
            parameters.Add(new ModelParameter("A", 0, 400000, 1));
            parameters.Add(new ModelParameter("ka", 2, 20.4, 4.99));
            parameters.Add(new ModelParameter("t1", -10, 50, 0));
            parameters.Add(new ModelParameter("t2", 10, 120, 20));
            info.Parameters = parameters.ToArray();
            info.ClassType = this.GetType();

            return info;
        }

        public override void Create(Vector times, params Vector[] Reference)
        {          
            this.times = times;
            simulated = new Vector(times.Length);
            samples = times.Length;

            this.measureTimes = new double[times.Length * valuesBetween];
            int num = 0;
            double oldTime = times[0];

            for (int i = 0; i < samples; i++)
            {
                double step = (times[i] - oldTime) / (valuesBetween);
                for (int w = 0; w < valuesBetween; w++)
                {
                    measureTimes[num] = oldTime + step * (double)w;
                    num++;
                }
                oldTime = times[i];
            }
        }

        public override int Samples
        {
            get { return samples; }
        }

        public CpBloodInputModel()
        {
            Info = GetInfo();
        }

    

        public override Vector Simulate(double[] p)
        {
            double A = p[0];
            double ka = p[1] / 60.0;
            double t1 = p[2];
            double t2 = p[3];


            double last_time = times[0];
            double last_value = CA_Input(times[0], A, ka, t1, t2);
            double value = 0;

            double CA_integ = 0;
            double CT = 0;
            double CT_integ = 0;

            double dt = 0;
            int mnum = 0;
            simulated[0] = 0;
            for (int i = 0; i < samples; i++)
            {
                for (int i2 = 0; i2 < valuesBetween; i2++)
                {
                    dt = measureTimes[mnum] - last_time;

                    value = CA_Input(measureTimes[mnum], A, ka, t1, t2);
                    CA_integ += value * dt * 0.5 + last_value * dt * 0.5;

                    CT = (kg * CA_integ - kg * (CT_integ + CT * dt * 0.5)) / (1.0 + kg * dt * 0.5);
                    CT_integ += CT;

                    last_value = value;
                    last_time = measureTimes[mnum];
                    mnum++;
                }

                simulated[i] = CT;
            }

            return simulated;
        }

        /********************************************************************
        Input Model Function for FDG, Water
        *************************************************************/
        protected double CA_Input(double x, double A, double ka, double t1, double t2)
        {
            double y;

            if (x < t1) y = 0.0d;
            else if (x <= t2) y = 1.0d - Math.Exp(ka * (t1 - x));
            else y = Math.Exp(ka * (t1 - t2)) + Math.Exp(ka * (t2 - x)) - 2.0d * Math.Exp(ka * (t1 - x));

            return (y * A);
        }

    }
}
