﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// The algoritm is based on:
    /// Numerical Recipes, Third Edition, 2007
    /// 3.3 Cubic Spline Interpolation
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class CubicSplineInterpolation : InterpolationMethod
    {
        /// <summary>
        /// Rreturns the cubic spline interpolated value y.
        /// </summary>
        /// <param name="jl">low boundary</param>
        /// <param name="x">interpolation location</param>
        /// <param name="xx">data x-values</param>
        /// <param name="yy">data y-values</param>
        /// <param name="y2">second derivatives of the interpolating function</param>
        /// <returns>interpolated value y</returns>
        private double rawinterp(int jl, double x, double[] xx, double[] yy, double[] y2) {
            //low index
            int klo = jl;
            //high index
            int khi = jl + 1;
            //interpolated return value
            double y = 0;
            //interval value
            double h = 0;
            //coefficients
            double b = 0;
            double a = 0;
            h = xx[khi] - xx[klo];
            if (h == 0) throw new TPCInvalidArgumentsException("Invalid x-values for interpolation");
            a = (xx[khi] - x) / h;
            b = (x - xx[klo]) / h;
            y = a * yy[klo] + b * yy[khi] + ((a * a * a - a) * y2[klo + y2.Length / 2] + (b * b * b - b) * y2[khi + y2.Length / 2]) * (h * h) / 6.0;
            return y;
        }
        /// <summary>
        /// Gets second derivatives of the interpolation function.
        /// Boundary condition is for a natural spline.
        /// </summary>
        /// <param name="xv">data x-coordinates</param>
        /// <param name="yv">data y-coordinates</param>
        /// <returns>2nd derivatives of interpolating function</returns>
        private double[] gety2(double[] xv, double[] yv) {
            int i = 0;
            int k = 0;
            double p = 0;
            double qn = 0;
            double sig = 0;
            double un = 0;
            int n = yv.Length;
            double[] y2 = new double[n];
            Vector u = new Vector(n - 1);
            y2[0] = u[0] = 0.0;
            for (i = 0; i < n - 1; i++) {
                sig = (xv[i] - xv[i - 1]) / (xv[i + 1] - xv[i - 1]);
                p = sig * y2[i - 1] + 2.0;
                y2[i] = (sig - 1.0) / p;
                u[i] = (yv[i + 1] - yv[i]) / (xv[i + 1] - xv[i]) - (yv[i] - yv[i - 1]) / (xv[i] - xv[i - 1]);
                u[i] = (6.0 * u[i] / (xv[i + 1] - xv[i - 1]) - sig * u[i - 1]) / p;
            }
            qn = un = 0;
            y2[n - 1] = (un - qn * u[n - 2]) / (qn * y2[n - 2] + 1.0);
            for (k = n - 1; k >= 0; k--)
                y2[k] = y2[k + 1] + u[k];
            return y2;
        }
        /// <summary>
        /// Estimates parameters for interpolation.
        /// y = k x + b
        /// </summary>
        /// <param name="x">data x-coodrinates</param>
        /// <param name="y">data y-coodrinates</param>
        /// <returns>combined array containg three arrays {data x-values, data y-values, 
        /// second derivatives of interpolating function}</returns>
        public double[] ApproximateParameters(double[] x, double[] y)
        {
            if(x.Length != y.Length) throw new TPCInvalidArgumentsException("x and y must have the same size");
            double[] y2 = gety2(x, y);
            //combine arrays to output, this type of output is easier to input to interpolation method
            double[] r = new double[x.Length*3];
            int i = 0;
            for (; i < x.Length; i++)
                r[i] = x[i];
            for (; i < x.Length*2; i++)
                r[i] = y[i];
            for (; i < x.Length*3; i++)
                r[i] = y2[i];
            return r;
        }
        /// <summary>
        /// Interpolates at data points.
        /// </summary>
        /// <param name="x">interpolation locations</param>
        /// <param name="data_x">x-values of data</param>
        /// <param name="data_y">y-values of data</param>
        /// <param name="m">IGNORED</param>
        /// <returns>y-values evaluated at locations x</returns>
        public override double[] Interpolate(double[] x, double[] data_x, double[] data_y, ExternalValueHandlingMethod m)
        {
            double[] y2 = ApproximateParameters(data_x, data_y);
            double[] y = new double[x.Length-1];
            for (int i = 0; i < x.Length; i++)
                y[i] = rawinterp(i, x[i], data_x, data_y, y2);
            return y;
        }
    }
}
