﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using MathNet.SignalProcessing.Filter.FIR;

namespace TPClib.Model
{
    /// <summary>
    /// Decimate Wavelet Transform in 1-D and 2-D.
    /// 
    /// Original Matlab help text:
    ///     function varargout = wavelet(WaveletName,Level,X,Ext,Dim)
    ///    WAVELET  Discrete wavelet transform.
    ///    Y = WAVELET(W,L,X) computes the L-stage discrete wavelet transform
    ///    (DWT) of signal X using wavelet W.  The length of X must be
    ///    divisible by 2^L.  For the inverse transform, WAVELET(W,-L,X)
    ///    inverts L stages.  Choices for W are
    ///      'Haar'                                      Haar
    ///      'D1','D2','D3','D4','D5','D6'               Daubechies'
    ///      'Sym1','Sym2','Sym3','Sym4','Sym5','Sym6'   Symlets
    ///      'Coif1','Coif2'                             Coiflets
    ///      'BCoif1'                                    Coiflet-like [2]
    ///      'Spline Nr.Nd' (or 'bior Nr.Nd') for        Splines
    ///        Nr = 0,  Nd = 0,1,2,3,4,5,6,7, or 8
    ///        Nr = 1,  Nd = 0,1,3,5, or 7
    ///        Nr = 2,  Nd = 0,1,2,4,6, or 8
    ///        Nr = 3,  Nd = 0,1,3,5, or 7
    ///        Nr = 4,  Nd = 0,1,2,4,6, or 8
    ///        Nr = 5,  Nd = 0,1,3, or 5
    ///      'RSpline Nr.Nd' for the same Nr.Nd pairs    Reverse splines
    ///      'S+P (2,2)','S+P (4,2)','S+P (6,2)',        S+P wavelets [3]
    ///      'S+P (4,4)','S+P (2+2,2)'
    ///      'TT'                                        "Two-Ten" [5]
    ///      'LC 5/3','LC 2/6','LC 9/7-M','LC 2/10',     Low Complexity [1]
    ///      'LC 5/11-C','LC 5/11-A','LC 6/14',
    ///      'LC 13/7-T','LC 13/7-C'
    ///      'Le Gall 5/3','CDF 9/7'                     JPEG2000 [7]
    ///      'V9/3'                                      Visual [8]
    ///      'Lazy'                                      Lazy wavelet
    ///    Case and spaces are ignored in wavelet names, for example, 'Sym4'
    ///    may also be written as 'sym 4'.  Some wavelets have multiple names,
    ///    'D1', 'Sym1', and 'Spline 1.1' are aliases of the Haar wavelet.
    /// 
    ///    WAVELET(W) displays information about wavelet W and plots the
    ///    primal and dual scaling and wavelet functions.
    /// 
    ///    For 2D transforms, prefix W with '2D'.  For example, '2D S+P (2,2)'
    ///    specifies a 2D (tensor) transform with the S+P (2,2) wavelet.
    ///    2D transforms require that X is either MxN or MxNxP where M and N
    ///    are divisible by 2^L.
    /// 
    ///    WAVELET(W,L,X,EXT) specifies boundary handling EXT.  Choices are
    ///      'sym'      Symmetric extension (same as 'wsws')
    ///      'asym'     Antisymmetric extension, whole-point antisymmetry
    ///      'zpd'      Zero-padding
    ///      'per'      Periodic extension
    ///      'sp0'      Constant extrapolation
    /// 
    ///    Various symmetric extensions are supported:
    ///      'wsws'     Whole-point symmetry (WS) on both boundaries
    ///      'hshs'     Half-point symmetry (HS) on both boundaries
    ///      'wshs'     WS left boundary, HS right boundary
    ///      'hsws'     HS left boundary, WS right boundary
    /// 
    ///    Antisymmetric boundary handling is used by default, EXT = 'asym'.
    /// 
    ///    WAVELET(...,DIM) operates along dimension DIM.
    /// 
    ///    [H1,G1,H2,G2] = WAVELET(W,'filters') returns the filters
    ///    associated with wavelet transform W.  Each filter is represented
    ///    by a cell array where the first cell contains an array of
    ///    coefficients and the second cell contains a scalar of the leading
    ///    Z-power.
    /// 
    ///    [X,PHI1] = WAVELET(W,'phi1') returns an approximation of the
    ///    scaling function associated with wavelet transform W.
    ///    [X,PHI1] = WAVELET(W,'phi1',N) approximates the scaling function
    ///    with resolution 2^-N.  Similarly,
    ///    [X,PSI1] = WAVELET(W,'psi1',...),
    ///    [X,PHI2] = WAVELET(W,'phi2',...),
    ///    and [X,PSI2] = WAVELET(W,'psi2',...) return approximations of the
    ///    wavelet function, dual scaling function, and dual wavelet function.
    /// 
    ///    Wavelet transforms are implemented using the lifting scheme [4].
    ///    For general background on wavelets, see for example [6].
    /// 
    /// 
    ///    Examples:
    ///    % Display information about the S+P (4,4) wavelet
    ///    wavelet('S+P (4,4)');
    /// 
    ///    % Plot a wavelet decomposition
    ///    t = linspace(0,1,256);
    ///    X = exp(-t) + Math.Sqrt(t - 0.3).*(t > 0.3) - 0.2*(t > 0.6);
    ///    wavelet('RSpline 3.1',3,X);    ///  Plot the decomposition of X
    /// 
    ///    % Sym4 with periodic boundaries
    ///    Y = wavelet('Sym4',5,X,'per');///  Forward transform with 5 stages
    ///    R = wavelet('Sym4',-5,Y,'per');   % Invert 5 stages
    /// 
    ///    % 2D transform on an image
    ///    t = linspace(-1,1,128); [x,y] = meshgrid(t,t);
    ///    X = ((x+1).*(x-1) - (y+1).*(y-1)) + real(Math.Sqrt(0.4 - x.^2 - y.^2));
    ///    Y = wavelet('2D CDF 9/7',2,X);///  2D wavelet transform
    ///    R = wavelet('2D CDF 9/7',-2,Y);   % Recover X from Y
    ///    imagesc(abs(Y).^0.2); colormap(gray); axis image;
    /// 
    ///    % Plot the Daubechies 2 scaling function
    ///    [x,phi] = wavelet('D2','phi');
    ///    plot(x,phi);
    ///  Pascal Getreuer 2005-2006
    /// </summary>
    public class DWTWaveletTransform
    {
        /// <summary>
        /// Wavelet lifting scheme instance
        /// </summary>
        public class WaveletScheme : List<List<double[]>>
        {
            /// <summary>
            /// Liftingscheme data
            /// </summary>
            List<List<double[]>> data;
            /// <summary>
            /// Data length
            /// </summary>
            public int Length {
                get {
                    return data.Count;
                }
            }
            /// <summary>
            /// Default constructor
            /// </summary>
            public WaveletScheme()
            {
                data = new List<List<double[]>>();
            }
            /// <summary>
            /// Copy constructor
            /// </summary>
            /// <param name="w"></param>
            public WaveletScheme(WaveletScheme w) {
                data = new List<List<double[]>>();
                data.AddRange(w.data);
            }
            /// <summary>
            /// Constructor with size defined
            /// </summary>
            /// <param name="rows">number of rows</param>
            /// <param name="columns">number of columns</param>
            public WaveletScheme(int rows, int columns)
            {
                data = new List<List<double[]>>();
                for (int i = 0; i < rows; i++)
                {
                    data.Add(new List<double[]>());
                    for (int j = 0; j < columns; j++)
                    {
                        data[data.Count - 1].Add(new double[0]);
                    }
                }
            }
            /// <summary>
            /// Constructor with size defined
            /// </summary>
            /// <param name="rows">number of rows</param>
            /// <param name="columns">number of columns</param>
            /// <param name="depth">number of double values in each cell</param>
            public WaveletScheme(int rows, int columns, int depth)
            {
                data = new List<List<double[]>>();
                for (int i = 0; i < rows; i++)
                {
                    data.Add(new List<double[]>());
                    for (int j = 0; j < columns; j++)
                    {
                        data[data.Count - 1].Add(new double[depth]);
                    }
                }           
            }
            /// <summary>
            /// Adds new values into end of the current data
            /// </summary>
            /// <param name="d1">1st item at row</param>
            /// <param name="d2">2nd item at row</param>
            public void Add(double d1, double d2)
            {
                data.Add(new List<double[]>() { new double[] { d1 }, new double[] { d2 } });
            }
            /// <summary>
            /// Adds new values into end of the current data
            /// </summary>
            /// <param name="d1">1st item at row</param>
            /// <param name="d2">2nd item at row</param>
            public void Add(double[] d1, double d2)
            {
                data.Add(new List<double[]>() { d1, new double[] { d2 } });
            }
            /// <summary>
            /// Adds new values into end of the current data
            /// </summary>
            /// <param name="d1">1st item at row</param>
            /// <param name="d2">2nd item at row</param>
            public void Add(double d1, double[] d2)
            {
                data.Add(new List<double[]>() { new double[]{d1}, d2});
            }
            /// <summary>
            /// Adds new values into end of the current data
            /// </summary>
            /// <param name="d1">1st item at row</param>
            /// <param name="d2">2nd item at row</param>
            public void Add(double[] d1, double[] d2)
            {
                data.Add(new List<double[]>() { d1, d2 });
            }
            /// <summary>
            /// Inserts new values into specified index in the current data
            /// </summary>
            /// <param name="i">zero-base index where data is inserted</param>
            /// <param name="d1">1st item at row</param>
            /// <param name="d2">2nd item at row</param>
            public void Insert(int i, double[] d1, double[] d2)
            {
                data.Insert(i, new List<double[]>() { d1, d2 });
            }
            /// <summary>
            /// Row of data
            /// </summary>
            /// <param name="x">row index</param>
            /// <returns></returns>
            public new List<double[]> this[int x]
            {
                get
                {
                    return data[x];
                }
                set
                {
                    data[x] = value;
                }
            }
            /// <summary>
            /// Cell of data
            /// </summary>
            /// <param name="x">row index of data</param>
            /// <param name="y">column index of data</param>
            /// <returns>array of data</returns>
            public double[] this[int x, int y]
            {
                get
                {
                    return data[x][y];
                }
                set
                {
                    data[x][y] = value;
                }
            }
            /// <summary>
            /// Double value inside cell of data
            /// </summary>
            /// <param name="x">row index of data</param>
            /// <param name="y">column index of data</param>
            /// <param name="z">index inside cell of data</param>
            /// <returns>data value</returns>
            public double this[int x, int y, int z] {
                get {
                    return data[x][y][z];
                }
                set {
                    data[x][y][z] = value;
                }
            }
            /// <summary>
            /// Removes the element at the specified index
            /// </summary>
            /// <param name="index">The zero-based index of the element to remove.</param>
            public new void RemoveAt(int index) {
                data.RemoveAt(index);
            }
            /// <summary>
            /// Prints contents of this sequence as string
            /// </summary>
            /// <returns></returns>
            public void PrintAsString(System.IO.TextWriter writer)
            {
                for (int i = 0; i < data.Count; i++)
                {
                    for (int j = 0; j < data[i].Count; j++)
                    {
                        writer.Write("[");
                        for (int k = 0; k < data[i][j].Length; k++)
                        {
                            writer.Write(data[i][j][k] + " ");
                        }
                        writer.Write("]\t");
                    }
                    writer.WriteLine();
                }
            }
        }
        /// <summary>
        /// Wavelet 
        /// </summary>
        public struct Wavelet {
            /// <summary>
            /// Lifting scheme
            /// </summary>
            public WaveletScheme Seq;
            /// <summary>
            /// S-scale
            /// </summary>
            public double ScaleS;
            /// <summary>
            /// D-scale
            /// </summary>
            public double ScaleD;
            /// <summary>
            /// Wavelet type
            /// </summary>
            public string Family;
        }
        /// <summary>
        /// Boundary hanling
        /// </summary>
        public enum BoundaryHandling {
            /// <summary>
            /// Symmetric extension (same as 'wsws')
            /// </summary>
            sym,
            /// <summary>
            /// Antisymmetric extension, whole-point antisymmetry
            /// </summary>
            asym,
            /// <summary>
            /// Zero-padding
            /// </summary>
            zpd,
            /// <summary>
            /// Periodic extension
            /// </summary>
            per,
            /// <summary>
            /// Constant extrapolation
            /// </summary>
            sp0,
            /// <summary>
            /// Whole-point symmetry (WS) on both boundaries
            /// </summary>
            wsws,
            /// <summary>
            /// Half-point symmetry (HS) on both boundaries
            /// </summary>
            hshs,
            /// <summary>
            /// WS left boundary, HS right boundary
            /// </summary>
            wshs,
            /// <summary>
            /// HS left boundary, WS right boundary
            /// </summary>
            hsws
        }
        /// <summary>
        /// Converts lifting scheme to its dual
        /// </summary>
        /// <param name="Seq">lifting scheme to be converted</param>
        /// <param name="ScaleS">S-scale</param>
        /// <param name="ScaleD">D-scale</param>
        protected static void SeqDual(ref WaveletScheme Seq, ref double ScaleS, ref double ScaleD)
        {
            // Dual of a lifting sequence
            int L = Seq.Length;
            int k = 0;
            for(; k < L; k++) {
                // f'(z) = -f(z^-1)
                for (int j = 0; j < Seq[k, 1].Length; j++)
                    Seq[k, 1, j] = -(Seq[k, 1, j] - Seq[k, 0].Length + 1);
                for (int j = 0; j < Seq[k, 0].Length; j++)
                    Seq[k, 0, j] *= -1;
                Array.Reverse(Seq[k, 0]);
            }
            for (k = 0; k < Seq[0, 0].Length; k++)
                if (Seq[0, 0][k] != 0) break;
            //add zero row at the start
            if (k == Seq[0, 0].Length)
            {
                Seq.RemoveAt(0);
            } else {
                Seq.Insert(0, new double[] { 0 }, new double[] { 0 });
            }
            ScaleS = 1/ScaleS;
            ScaleD = 1/ScaleD;        
        }
        /// <summary>
        /// Calculates discrete convolution of two vectors 
        /// </summary>
        /// <param name="f">1st vector</param>
        /// <param name="g">2nd vector</param>
        /// <returns>(f*g)</returns>
        public static double[] Convolution(double[] f, double[] g) {
            double[] r = new double[f.Length+g.Length-1];
            for(int n = 0; n < r.Length; n++) {
                for(int m = 0; m < f.Length; m++) {
                    if((n+0)-m >= g.Length) continue;
                    if(m > (n+0)) break;
                    r[n] += f[m]*g[(n+0)-m];
                }
            }
            return r;
        }
        /// <summary>
        /// Calculates discrete deconvolution of two vectors 
        /// </summary>
        /// <param name="f">1st vector</param>
        /// <param name="g">2nd vector</param>
        /// <returns>(f*g)</returns>
        public static double[] DeConvolution(double[] f, double[] g) {
            double[] r = new double[f.Length+g.Length-1];
            for(int n = 0; n < r.Length; n++) {
                for(int m = 0; m < f.Length; m++) {
                    if((n+1)-m >= g.Length) break;
                    if(m > (n+1)) break;
                    r[n] += f[m]*g[(n+1)-m];
                }
            }
            return r;
        }
        /// <summary>
        /// Lifting a(z) = a(z) + b(z) c(z^2)
        /// </summary>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <param name="c"></param>
        /// <returns></returns>
        protected static List<double[]> Lp_Lift(List<double[]> a, List<double[]> b, List<double[]> c)
        {
            double[] d = new double[c[0].Length*2-1];
            for(int k = 0; k < c[0].Length; k++)
                d[k*2] = c[0][k];
            d = Convolution(b[0],d);
            double z = b[1][0] + c[1][0] * 2;
            double zmax = Math.Max(a[1][0], z);
            int ending_length = (int)(a[1][0] - a[0].Length - z + d.Length);
            if (ending_length < 0) ending_length = 0;
            double[] f = new double[(int)(zmax - a[1][0]) + a[0].Length + ending_length];
            for(int j = 0; j< a[0].Length; j++)
                f[(int)(zmax - a[1][0])+j] = a[0][j];
            int[] i = new int[d.Length];
            for (int k = 0; k < d.Length; k++)
                i[k] = (int)(zmax - z + k);
            for (int k = 0; k < i.Length; k++)
                f[i[k]] += d[k];
            double max_abs_f = double.MinValue;
            double max_f = double.MinValue;
            for (int k = 0; k < f.Length; k++)
            {
                if(f[k] > max_f)
                    max_f = f[k];
                if (Math.Abs(f[k]) > max_abs_f)
                    max_abs_f = Math.Abs(f[k]);
            }
            if(max_abs_f < 1e-12) {
                a = new List<double[]> { new double[] { 0 }, new double[] { 0 } };
            } else {
                int i1 = -1;
                int i2 = -1;
                for (int k = 0; k < f.Length; k++) {
                    if (Math.Abs(f[k]) / max_abs_f > 1e-10)
                    {
                        if (i1 == -1)
                        {
                            i1 = k;
                            if (i2 == -1)
                                i2 = k;
                        }
                        else i2 = k;
                    }
                }
                a = new List<double[]>{new double[i2-i1+1], new double[]{zmax-(i1+1)+1}};
                for(int k = i1; k <= i2; k++) {
                    a[0][k-i1] = f[k];
                }
            }
            return a;
        }
        /// <summary>
        /// Find wavelet filters from lifting sequence
        /// </summary>
        /// <param name="Seq"></param>
        /// <param name="ScaleS"></param>
        /// <param name="ScaleD"></param>
        /// <param name="Dual">true for dual lifting scheme</param>
        /// <param name="h"></param>
        /// <param name="g"></param>
        /// <returns></returns>
        public static void Seq2hg(WaveletScheme Seq, double ScaleS, double ScaleD, bool Dual, ref List<double[]> h, ref List<double[]> g) {
            // Find wavelet filters from lifting sequence
            WaveletScheme scheme = null;
            if (Dual)
            {
                SeqDual(ref Seq, ref ScaleS, ref ScaleD);
            }
            scheme = new WaveletScheme(Seq);
            if (Seq.Length % 2 != 0)
            {
                scheme.Add(new double[] { 0 }, new double[] { 0 });
            }
            
            h = new List<double[]>{ new double[]{1},new double[]{0}};
            g = new List<double[]>{ new double[]{1},new double[]{1}};

            for (int k = 0; k < scheme.Length; k += 2)
            {
                h = Lp_Lift(h, g, scheme[k]);
                g = Lp_Lift(g, h, scheme[k + 1]);
            }

            for (int k = 0; k < h[0].Length; k ++)  
                h[0][k] *= ScaleS;
            for (int k = 0; k < g[0].Length; k++)   
                g[0][k] *= ScaleD;

            if(Dual) {
                for (int k = 0; k < h[1].Length; k++)
                    h[1][k] = -(h[1][k] - h[0].Length + 1);
                Array.Reverse(h[0]);

                for (int k = 0; k < g[1].Length; k++)
                    g[1][k] = -(g[1][k] - g[0].Length + 1);
                Array.Reverse(g[0]);
            }
        }
        /// <summary>
        /// Returns wavelet filters for display.
        /// </summary>
        /// <param name="WaveletName">name of wavelet</param>
        /// <returns>wavelet filters {h1, g1, h2, g2} as curves</returns>
        public static List<double[]>[] GetWaveletFilters(string WaveletName) {
            Wavelet wavelet;
            wavelet = GetWavelet(WaveletName);
            List<double[]> h1 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            List<double[]> h2 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            List<double[]> g1 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            List<double[]> g2 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            Seq2hg(wavelet.Seq, wavelet.ScaleS, wavelet.ScaleD, false, ref h1, ref g1);
            Seq2hg(wavelet.Seq, wavelet.ScaleS, wavelet.ScaleD, true, ref h2, ref g2);
            return new List<double[]>[] { h1, g1, h2, g2 };
        }
        /// <summary>
        /// Get wavelet lifting scheme sequence.
        /// Ported from Matlab-code by Pascal Getreuer 2005-2006
        /// </summary>
        /// <param name="WaveletName">wavelet name</param>
        /// <returns>wavelet lifting scheme</returns>
        public static Wavelet GetWavelet(string WaveletName) {
            Wavelet wavelet;
            WaveletName = WaveletName.Replace("bior","spline");
            wavelet.Seq = new WaveletScheme();
            wavelet.ScaleS = 1/Math.Sqrt(2.0);
            wavelet.ScaleD = 1/Math.Sqrt(2.0);
            wavelet.Family = "Spline";

            WaveletName = WaveletName.ToLowerInvariant().Replace("2d","").Replace(" ","");
            switch(WaveletName) {
                case "haar":
                case "d1":
                case "db1":
                case "sym1":
                case "spline1.1":
                case "rspline1.1":
                    wavelet.Seq.Add(1, 0);
                    wavelet.Seq.Add(-0.5, 0);
                    wavelet.ScaleD = -Math.Sqrt(2);
                    wavelet.Family = "Haar";
                    break;
                case "d2":
                case "db2":
                case "sym2":
                    wavelet.Seq.Add(Math.Sqrt(3),0);
                    wavelet.Seq.Add(new double[]{-Math.Sqrt(3)/4.0,(2-Math.Sqrt(3))/4.0},0);
                    wavelet.Seq.Add(-1,1);
                    wavelet.ScaleS = (Math.Sqrt(3) - 1) / Math.Sqrt(2);
                    wavelet.ScaleD = (Math.Sqrt(3) + 1) / Math.Sqrt(2);
                    wavelet.Family = "Daubechies";
                    break;
                case "d3":
                case "db3":
                case "sym3":
                    wavelet.Seq.Add(2.4254972439123361,0);
                    wavelet.Seq.Add(new double[]{-0.3523876576801823,0.0793394561587384},0);
                    wavelet.Seq.Add(new double[]{0.5614149091879961,-2.8953474543648969},2);
                    wavelet.Seq.Add(-0.0197505292372931,-2);
                    wavelet.ScaleS = 0.4318799914853075;
                    wavelet.ScaleD = 2.3154580432421348;
                    wavelet.Family = "Daubechies";
                    break;
                case "d4":
                case "db4":
                    wavelet.Seq.Add(0.3222758879971411,-1);
                    wavelet.Seq.Add(new double[]{0.3001422587485443,1.1171236051605939},1);
                    wavelet.Seq.Add(new double[]{-0.1176480867984784,0.0188083527262439},-1);
                    wavelet.Seq.Add(new double[]{-0.6364282711906594,-2.1318167127552199},1);
                    wavelet.Seq.Add(new double[]{0.0247912381571950,-0.1400392377326117,0.4690834789110281},2);
                    wavelet.ScaleS = 1.3621667200737697;
                    wavelet.ScaleD = 0.7341245276832514;
                    wavelet.Family = "Daubechies";
                    break;
                case "d5":
                case "db5":
                    wavelet.Seq.Add(.2651451428113514,-1);
                    wavelet.Seq.Add(new double[]{-0.2477292913288009,-0.9940591341382633},1);
                    wavelet.Seq.Add(new double[]{-0.2132742982207803,0.5341246460905558},1);
                    wavelet.Seq.Add(new double[]{0.7168557197126235,-0.2247352231444452},-1);
                    wavelet.Seq.Add(new double[]{-0.0121321866213973,0.0775533344610336},3);
                    wavelet.Seq.Add(0.035764924629411,-3);
                    wavelet.ScaleS = 1.3101844387211246;
                    wavelet.ScaleD = 0.7632513182465389;
                    wavelet.Family = "Daubechies";
                    break;
                case "d6":
                case "db6":
                    wavelet.Seq.Add(4.4344683000391223,0);
                    wavelet.Seq.Add(new double[]{-0.214593449940913,0.0633131925095066},0);
                    wavelet.Seq.Add(new double[]{4.4931131753641633,-9.970015617571832},2);
                    wavelet.Seq.Add(new double[]{-0.0574139367993266,0.0236634936395882},-2);
                    wavelet.Seq.Add(new double[]{0.6787843541162683,-2.3564970162896977},4);
                    wavelet.Seq.Add(new double[]{-0.0071835631074942,0.0009911655293238},-4);
                     wavelet.Seq.Add(-0.0941066741175849,5);
                    wavelet.ScaleS = 0.3203624223883869;
                    wavelet.ScaleD = 3.1214647228121661;
                    wavelet.Family = "Daubechies";
                    break;
                case "sym4":
                    wavelet.Seq.Add(-0.3911469419700402,0);
                    wavelet.Seq.Add(new double[]{0.3392439918649451,0.1243902829333865},0);
                    wavelet.Seq.Add(new double[]{-0.1620314520393038,1.4195148522334731},1);
                    wavelet.Seq.Add(new double[]{-0.1459830772565225,-0.4312834159749964},1);
                    wavelet.Seq.Add(1.049255198049293,-1);
                    wavelet.ScaleS = 0.6366587855802818;
                    wavelet.ScaleD = 1.5707000714496564;
                    wavelet.Family = "Symlet";
                    break;
                case "sym5":
                    wavelet.Seq.Add(0.9259329171294208,0);
                    wavelet.Seq.Add(new double[]{-0.1319230270282341,-0.4985231842281166},1);
                    wavelet.Seq.Add(new double[]{1.452118924420613,0.4293261204657586},0);
                    wavelet.Seq.Add(new double[]{-0.2804023843755281,0.0948300395515551},0);
                    wavelet.Seq.Add(new double[]{-0.7680659387165244,-1.9589167118877153},1);
                    wavelet.Seq.Add(0.1726400850543451,0);
                    wavelet.ScaleS = 0.4914339446751972;
                    wavelet.ScaleD = 2.0348614718930915;
                    wavelet.Family = "Symlet";
                    break;
                case "sym6":
                    wavelet.Seq.Add(-0.2266091476053614,0);
                    wavelet.Seq.Add(new double[]{0.2155407618197651,-1.2670686037583443},0);
                    wavelet.Seq.Add(new double[]{-4.2551584226048398,0.5047757263881194},2);
                    wavelet.Seq.Add(new double[]{0.2331599353469357,0.0447459687134724},-2);
                    wavelet.Seq.Add(new double[]{6.6244572505007815,-18.389000853969371},4);
                    wavelet.Seq.Add(new double[]{-0.0567684937266291,0.1443950619899142},-4);
                    wavelet.Seq.Add(new double[]{-5.5119344180654508},5);
                    wavelet.ScaleS = -0.5985483742581210;
                    wavelet.ScaleD = -1.6707087396895259;
                    wavelet.Family = "Symlet";
                    break;
                case "coif1":
                    wavelet.Seq.Add(-4.6457513110481772,0);
                    wavelet.Seq.Add(new double[]{0.205718913884,0.1171567416519999},0);
                    wavelet.Seq.Add(new double[]{0.6076252184992341,-7.468626966435207},2);
                    wavelet.Seq.Add(-0.0728756555332089, -2);
                    wavelet.ScaleS = -0.5818609561112537;
                    wavelet.ScaleD = -1.7186236496830642;
                    wavelet.Family = "Coiflet";
                    break;
                case "coif2":
                    wavelet.Seq.Add(-2.5303036209828274,0);
                    wavelet.Seq.Add(new double[]{0.3418203790296641,-0.2401406244344829},0);
                    wavelet.Seq.Add(new double[]{15.268378737252995,3.1631993897610227},2);
                    wavelet.Seq.Add(new double[]{-0.0646171619180252,0.005717132970962},-2);
                    wavelet.Seq.Add(new double[]{13.59117256930759,-63.95104824798802},4);
                    wavelet.Seq.Add(new double[]{-0.0018667030862775,0.0005087264425263},-4);
                    wavelet.Seq.Add(-3.7930423341992774,5);
                    wavelet.ScaleS = 0.1076673102965570;
                    wavelet.ScaleD = 9.2878701738310099;
                    wavelet.Family = "Coiflet";
                    break;
                case "bcoif1":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/5.0,-1.0/5.0},1);
                    wavelet.Seq.Add(new double[]{5.0/14.0,5.0/14.0},0);
                    wavelet.Seq.Add(new double[]{-21.0/100.0,-21.0/100.0},1);
                    wavelet.ScaleS = Math.Sqrt(2) * 7.0 / 10.0;
                    wavelet.ScaleD = Math.Sqrt(2) * 5.0 / 7.0;
                    wavelet.Family = "Nearly orthonormal Coiflet-like";
                    break;
                case "lazy":
                case "spline0.0":
                case "rspline0.0":
                case "d0":
                    wavelet.Seq.Add(0, 0);
                    wavelet.ScaleS = 1;
                    wavelet.ScaleD = 1;
                    wavelet.Family = "Lazy";
                    break;
                case "spline0.1":
                case "rspline0.1":
                    wavelet.Seq.Add(1, -1);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.2":
                case "rspline0.2":
                    wavelet.Seq.Add( new double[] { 1 / 2, 1 / 2 }, 0);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.3":
                case "rspline0.3":
                    wavelet.Seq.Add( new double[] { -1 / 8, 6 / 8, 3 / 8 }, 1);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.4":
                case "rspline0.4":
                    wavelet.Seq.Add( new double[] { -1 / 16, 9 / 16, 9 / 16, -1 / 16 }, 1);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.5":
                case "rspline0.5":
                    wavelet.Seq.Add( new double[] { 3 / 128, -20 / 128, 90 / 128, 60 / 128, -5 / 128 }, 2);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.6":
                case "rspline0.6":
                    wavelet.Seq.Add( new double[] { 3 / 256, -25 / 256, 150 / 256, 150 / 256, -25 / 256, 3 / 256 }, 2);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.7":
                case "rspline0.7":
                    wavelet.Seq.Add( new double[] { -5 / 1024, 42 / 1024, -175 / 1024, 700 / 1024, 525 / 1024, -70 / 1024, 7 / 1024 }, 3);
                    wavelet.ScaleD = 1;
                    break;
                case "spline0.8":
                case "rspline0.8":
                    wavelet.Seq.Add(new double[] { -5 / 2048, 49 / 2048, -245 / 2048, 1225 / 2048, 1225 / 2048, -245 / 2048, 49 / 2048, -5 / 2048 }, 3);
                    wavelet.ScaleD = 1;
                    break;
                case "spline1.0":
                case "rspline1.0":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(-1,0);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);   
                    break;
                case "spline1.3":
                case "rspline1.3":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(-1,0);
                    wavelet.Seq.Add(new double[]{-1/16,8/16,1/16},1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    break;
                case "spline1.5":
                case "rspline1.5":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(-1,0);
                    wavelet.Seq.Add(new double[]{3/256,-22/256,128/256,22/256,-3/256},2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    break;
                case "spline1.7":
                case "rspline1.7":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(-1,0);
                    wavelet.Seq.Add(new double[]{-5/2048,44/2048,-201/2048,1024/2048,201/2048,-44/2048,5/2048},3);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    break;
                case "spline2.0":
                case "rspline2.0":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/2,-1/2},1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = 1;
                    break;
                case "spline2.1":
                case "rspline2.1":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/2,-1/2},1);
                    wavelet.Seq.Add(new double[]{0.5},0);
                    wavelet.ScaleS = Math.Sqrt(2);
                    break;
                case "spline2.2":
                case "rspline2.2":
                case "cdf5/3":
                case "legall5/3":
                case "s+p(2,2)":
                case "lc5/3":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/2.0,-1.0/2.0},1);
                    wavelet.Seq.Add(new double[]{1.0/4.0,1.0/4.0},0);
                    wavelet.ScaleS = Math.Sqrt(2);
                    break;
                case "spline2.4":
                case "rspline2.4":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/2.0,-1.0/2.0},1);
                    wavelet.Seq.Add(new double[]{-3.0/64.0,19.0/64.0,19.0/64.0,-3.0/64.0},1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    break;
                case "spline2.6":
                case "rspline2.6":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/2.0,-1.0/2.0},1);
                    wavelet.Seq.Add(new double[]{5.0/512.0,-39.0/512.0,162.0/512.0,162.0/512.0,-39.0/512.0,5.0/512.0},2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    break;
                case "spline2.8":
                case "rspline2.8":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/2.0,-1.0/2.0},1);
                    wavelet.Seq.Add(new double[]{-35.0/16384.0,335.0/16384.0,-1563.0/16384.0,5359.0/16384.0,5359.0/16384.0,-1563.0/16384.0,335.0/16384.0,-35.0/16384.0},3);
                    wavelet.ScaleS = Math.Sqrt(2);
                    break;
                case "spline3.0":
                case "rspline3.0":
                    wavelet.Seq.Add(-1.0/3.0,-1);
                    wavelet.Seq.Add(new double[]{-3.0/8.0,-9.0/8.0},1);
                    wavelet.ScaleS = 3.0 / Math.Sqrt(2);
                    wavelet.ScaleD = 2.0 / 3.0;
                    break;
                case "spline3.1":
                case "rspline3.1":
                    wavelet.Seq.Add(-1/3,-1);
                    wavelet.Seq.Add(new double[]{-3/8,-9/8},1);
                    wavelet.Seq.Add(new double[]{4/9},0);
                    wavelet.ScaleS = 3 / Math.Sqrt(2);
                    wavelet.ScaleD = -2 / 3;
                    break;
                case "spline3.3":
                case "rspline3.3":
                    wavelet.Seq.Add(-1/3,-1);
                    wavelet.Seq.Add(new double[]{-3/8,-9/8},1);
                    wavelet.Seq.Add(new double[]{-3/36,16/36,3/36},1);
                    wavelet.ScaleS = 3 / Math.Sqrt(2);
                    wavelet.ScaleD = -2 / 3;
                    break;
                case "spline3.5":
                case "rspline3.5":
                    wavelet.Seq.Add(-1/3,-1);
                    wavelet.Seq.Add(new double[]{-3/8,-9/8},1);
                    wavelet.Seq.Add(new double[]{5/288,-34/288,128/288,34/288,-5/288},2);
                    wavelet.ScaleS = 3 / Math.Sqrt(2);
                    wavelet.ScaleD = -2 / 3;
                    break;
                case "spline3.7":
                case "rspline3.7":
                    wavelet.Seq.Add(-1/3,-1);
                    wavelet.Seq.Add(new double[]{-3/8,-9/8},1);
                    wavelet.Seq.Add(new double[]{-35/9216,300/9216,-1263/9216,4096/9216,1263/9216,-300/9216,35/9216},3);
                    wavelet.ScaleS = 3 / Math.Sqrt(2);
                    wavelet.ScaleD = -2 / 3;
                    break;
                case "spline4.0":
                case "rspline4.0":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / Math.Sqrt(2);
                    break;
                case "spline4.1":
                case "rspline4.1":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.Seq.Add(new double[]{6/16},0);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / 2;
                    break;
                case "spline4.2":
                case "rspline4.2":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.Seq.Add(new double[]{3/16,3/16},0);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / 2;
                    break;
                case "spline4.4":
                case "rspline4.4":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.Seq.Add(new double[]{-5/128,29/128,29/128,-5/128},1);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / 2;
                    break;
                case "spline4.6":
                case "rspline4.6":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.Seq.Add(new double[]{35/4096,-265/4096,998/4096,998/4096,-265/4096,35/4096},2);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / 2;
                    break;
                case "spline4.8":
                case "rspline4.8":
                    wavelet.Seq.Add(new double[]{-1/4,-1/4},0);
                    wavelet.Seq.Add(new double[]{-1,-1},1);
                    wavelet.Seq.Add(new double[]{-63/32768,595/32768,-2687/32768,8299/32768,8299/32768,-2687/32768,595/32768,-63/32768},3);
                    wavelet.ScaleS = 4 / Math.Sqrt(2);
                    wavelet.ScaleD = 1 / 2;
                    break;
                case "spline5.0":
                case "rspline5.0":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/5},0);
                    wavelet.Seq.Add(new double[]{-5/24,-15/24},0);
                    wavelet.Seq.Add(new double[]{-9/10,-15/10},1);
                    wavelet.ScaleS = 3 * Math.Sqrt(2);
                    wavelet.ScaleD = Math.Sqrt(2) / 6;
                    break;
                case "spline5.1":
                case "rspline5.1":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/5},0);
                    wavelet.Seq.Add(new double[]{-5/24,-15/24},0);
                    wavelet.Seq.Add(new double[]{-9/10,-15/10},1);
                    wavelet.Seq.Add(1/3,0);
                    wavelet.ScaleS = 3 * Math.Sqrt(2);
                    wavelet.ScaleD = Math.Sqrt(2) / 6;
                    break;
                case "spline5.3":
                case "rspline5.3":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/5},0);
                    wavelet.Seq.Add(new double[]{-5/24,-15/24},0);
                    wavelet.Seq.Add(new double[]{-9/10,-15/10},1);
                    wavelet.Seq.Add(new double[]{-5/72,24/72,5/72},1);
                    wavelet.ScaleS = 3 * Math.Sqrt(2);
                    wavelet.ScaleD = Math.Sqrt(2) / 6;
                    break;
                case "spline5.5":
                case "rspline5.5":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1/5},0);
                    wavelet.Seq.Add(new double[]{-5/24,-15/24},0);
                    wavelet.Seq.Add(new double[]{-9/10,-15/10},1);
                    wavelet.Seq.Add(new double[]{35/2304,-230/2304,768/2304,230/2304,-35/2304},2);
                    wavelet.ScaleS = 3 * Math.Sqrt(2);
                    wavelet.ScaleD = Math.Sqrt(2) / 6;
                    break;
                case "cdf9/7":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.5861343420693648,-1.5861343420693648},1);
                    wavelet.Seq.Add(new double[]{-0.0529801185718856,-0.0529801185718856},0);
                    wavelet.Seq.Add(new double[]{0.8829110755411875,0.8829110755411875},1);
                    wavelet.Seq.Add(new double[]{0.4435068520511142,0.4435068520511142},0);
                    wavelet.ScaleS = 1.1496043988602418;
                    wavelet.ScaleD = 1 / wavelet.ScaleS;
                    wavelet.Family = "Cohen-Daubechies-Feauveau";
                    break;
                case "v9/3":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1.0/2.0,-1.0/2.0},1);
                    wavelet.Seq.Add(new double[]{1.0/80.0,19.0/80.0,19.0/80.0,1/80.0},1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.Family = "HSV design";
                    break;
                case "s+p(4,2)":
                case "lc9/7-m":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{1.0/16.0,-9.0/16.0,-9.0/16.0,1.0/16.0},2);
                    wavelet.Seq.Add(new double[]{1.0/4.0,1.0/4.0},0);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.Family = "S+P";
                    break;
                case "s+p(6,2)":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-3.0/256.0,25.0/256.0,-150.0/256.0,-150.0/256.0,25.0/256.0,-3.0/256.0},3);
                    wavelet.Seq.Add(new double[]{1.0/4.0,1.0/4.0},0);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.Family = "S+P";
                    break;
                case "s+p(4,4)":
                case "lc13/7-t":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { 1.0 / 16.0, -9.0 / 16.0, -9.0 / 16.0, 1.0 / 16.0 }, 2);
                    wavelet.Seq.Add(new double[] { -1.0 / 32.0, 9.0 / 32.0, 9.0 / 32.0, -1.0 / 32.0 }, 1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.Family = "S+P";
                    break;
                case "s+p(2+2,2)":
                case "lc5/11-c":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { -1.0 / 2.0, -1.0 / 2.0 }, 1);
                    wavelet.Seq.Add(new double[] { 1.0 / 4.0, 1.0 / 4.0 }, 0);
                    wavelet.Seq.Add(new double[] { 1.0 / 16.0, -1.0 / 16.0, -1.0 / 16.0, 1.0 / 16.0 }, 2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.Family = "S+P";
                    break;
                case "tt":
                    wavelet.Seq.Add(1,0);
                    wavelet.Seq.Add(new double[] { 3.0 / 256.0, -22.0 / 256.0, -128.0 / 256.0, 22.0 / 256.0, -3.0 / 256.0 }, 2);
                    wavelet.ScaleD = Math.Sqrt(2);
                    wavelet.Family = "Le Gall-Tabatabai polynomial";
                    break;
                case "lc2/6":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { -1.0 }, 0);
                    wavelet.Seq.Add(new double[] { 1.0 / 2.0 }, 0);
                    wavelet.Seq.Add(new double[] { -1.0 / 4.0, 0.0 / 4.0, 1.0 / 4.0 }, 1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    wavelet.Family = "Reverse spline";
                    break;
                case "lc2/10":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { -1.0 }, 0);
                    wavelet.Seq.Add(new double[] { 1.0 / 2.0 }, 0);
                    wavelet.Seq.Add(new double[] { 3.0 / 64.0, -22.0 / 64.0, 0.0 / 64.0, 22.0 / 64.0, -3.0 / 64.0 }, 2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    wavelet.Family = "Reverse spline";
                    break;
                case "lc5/11-a":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { -1.0 / 2.0, -1.0 / 2.0 }, 1);
                    wavelet.Seq.Add(new double[] { 1.0 / 4.0, 1.0 / 4.0 }, 0);
                    wavelet.Seq.Add(new double[] { 1.0 / 32.0, -1.0 / 32.0, -1.0 / 32.0, 1.0 / 32.0 }, 2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    wavelet.Family = "Low complexity";
                    break;
                case "lc6/14":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{-1},0);
                    wavelet.Seq.Add(new double[] { -1.0 / 16.0, 8.0 / 16.0, 1.0 / 16.0 }, 1);
                    wavelet.Seq.Add(new double[] { 1.0 / 16.0, -6.0 / 16.0, 0.0 / 16.0, 6.0 / 16.0, -1.0 / 16.0 }, 2);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    wavelet.Family = "Low complexity";
                    break;
                case "lc13/7-c":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[] { 1.0 / 16.0, -9.0 / 16.0, -9.0 / 16.0, 1.0 / 16.0 }, 2);
                    wavelet.Seq.Add(new double[] { -1.0 / 16.0, 5.0 / 16.0, 5.0 / 16.0, -1.0 / 16.0 }, 1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1 / Math.Sqrt(2);
                    wavelet.Family = "Low complexity";
                    break;
                case "e-spline":
                    wavelet.Seq.Add(0,0);
                    wavelet.Seq.Add(new double[]{1.0/16.0,-9.0/16.0,-9.0/16.0,1.0/16.0},2);
                    wavelet.Seq.Add(new double[]{-1.0/16.0,5.0/16.0,5.0/16.0,-1/16.0},1);
                    wavelet.ScaleS = Math.Sqrt(2);
                    wavelet.ScaleD = -1.0 / Math.Sqrt(2);
                    wavelet.Family = "Low complexity";
                    break;
                default:
                    throw new WaveletTransformException("Unknown wavelet ["+WaveletName+"]");
            }
            return wavelet;
        }
        /// <summary>
        /// Wavelet cascade algorithm
        /// </summary>
        protected static void Cascade(List<double[]> h, List<double[]> g, double[] dx,
                                      out double[] x1, out double[] phi)
        {
            double[] x2;
            double[] psi;
            Cascade(h, g, dx, out x1, out phi, out x2, out psi);
        }
        /// <summary>
        /// Wavelet cascade algorithm
        /// </summary>
        protected static void Cascade(List<double[]> h, List<double[]> g, double[] dx,
                                      out double[] x1, out double[] phi, out double[] x2, out double[] psi)
        {
            double[] c = new double[h[0].Length];
            double sum_h = 0.0;
            int i = 0;
            for(; i < h[0].Length; i++) 
                sum_h += h[0][i];
            for(i = 0; i < c.Length; i++) 
                c[i] = (h[1][i]*2)/sum_h;
            double[] x = new double[(int)(c.Length/dx[0])];
            for (i = 0; i < x.Length; i++)
                x[i] = i*dx[0] - 1;
            x1 = new double[x.Length];
            for (i = 0; i < x1.Length; i++)
                x1[i] = x[i] - h[1][0];
            double linspace_step = 2.0/(x.Length-1);
            double[] phi0 = new double[x.Length];
            for (i = 0; -1.0 + linspace_step * i < x.Length; i++)
                phi0[i] = 1 - Math.Abs(-1.0+linspace_step*i);

            List<int> ii = new List<int>();
            List<int> jj = new List<int>();
            List<double> s = new List<double>();
            for (int k = 0; k < c.Length; k++)
            {
                double[] xk = new double[x.Length];
                for(int l = 1; l < xk.Length; l++)
                    xk[l] = 2*x[l] - k;
                List<int> indexes = new List<int>();
                for(int j = 0; j < c.Length; j++) {
                    if (xk[j] >= 0 && xk[j] <= c.Length - 1)
                    {
                        indexes.Add(j);
                    }
                }
                ii.AddRange(indexes);
                for(int j = 0; j < c.Length; j++)
                    jj.Add((int)Math.Floor(xk[indexes[j]]/dx[0])+1);
                for (int j = 0; j < c.Length; j++)
                    for (int l = 0; l < indexes.Count; l++)
                        s.Add(c[k]);
            }
            throw new NotImplementedException();
            /*
            //Construct a sparse linear operator that iterates the dilation equation
            Dilation = sparse(ii,jj,s,length(x),length(x));

            for(N = 1; N <= 30; N++) {
                phi = Dilation*phi0;
                if(norm(phi - phi0,inf) < 1e-5) {
                    break;
                }
                phi0 = phi;
            }

            if(norm(phi) == 0) {
                //Special case for Haar scaling function
                phi = ones(size(phi))*Math.Sqrt(2);   
            } else {
                //Rescale result
                phi = phi/(norm(phi)*Math.Sqrt(dx));  
            }

            if(nargout > 2) {
                //phi2 is approximately phi(2x)
                for(int i = 0; i < phi.Length; i+=2)
                    phi2 = phi[i];  
                if(length(c) == 2) {
                    L = length(phi2);
                } else {
                    L = ceil(0.5/dx);
                }

                //Construct psi from translates of phi2
                c = g[1];
                psi = new double[phi2.Length+L*c.Length-1];
                double[] x2 = new double[psi.Length-1];
                for(int i = 0; i < x2.Length; i++) {
                    x2[i] = i*dx[i] - g[1] - 0*h[1,i]/2;
                }

                for(int k = 0; k < c.Length; k++) {
                    for(int j = L*(k-1); j < phi2.Length; j++) {
                        psi[i] = psi[i] + c[k]*phi2;
                    }
                }
            }
            return;
            */
        }

        /// <summary>
        /// Wavelet information
        /// </summary>
        /// <param name="WaveletName">used wavelet</param>
        /// <param name="Level">number of stages</param>
        /// <param name="X">data to be transformed</param>
        /// <param name="Ext">boundary handling convention</param>
        /// <param name="Dim">dimension that is operated</param>
        /// <returns></returns>
        public static List<double[]> WaveletInformation(string WaveletName, 
                                                       WaveletScheme Seq, 
                                                       double ScaleS, double ScaleD, 
                                                       string Level,
                                                       double[] X)
        {
            ///  Get a lifting scheme sequence for the specified wavelet
            bool Flag1D = !WaveletName.ToLowerInvariant().Contains("2d");
            Wavelet wavelet = GetWavelet(WaveletName);

            List<double[]> h1 = new List<double[]>() { new double[]{0}, new double[]{0}};
            List<double[]> h2 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            List<double[]> g1 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            List<double[]> g2 = new List<double[]>() { new double[] { 0 }, new double[] { 0 } };
            Seq2hg(Seq, ScaleS, ScaleD, false, ref h1, ref g1);
            Seq2hg(Seq, ScaleS, ScaleD, true, ref h2, ref g2);
            double[] x1;
            double[] phi;
            double[] x2;
            double[] psi;
            for (int i = 0; i < X.Length; i++)
                X[i] = Math.Pow(2.0, -X[i]);

            if(Level.ToLowerInvariant().CompareTo("filters") == 0) {
                return new List<double[]>() { h1[0], h1[2], g1[0], g1[1], h2[0], h2[1], g2[0], g2[1] };
            } else {
                switch(Level.ToLowerInvariant()) {
                    case "phi1":
                    case "phi":
                        Cascade(h1,g1,X, out x1, out phi);
                        return new List<double[]>() {x1,phi};
                    case "psi1":
                    case "psi":
                        Cascade(h1, g1, X, out x1, out phi, out x2, out psi);
                        return new List<double[]>() { x2, psi };
                    case "phi2":
                        Cascade(h2, g2, X, out x1, out phi);
                        return new List<double[]>() { x1, phi };
                    case "psi2":
                        Cascade(h2, g2, X, out x1, out phi, out x2, out psi);
                        return new List<double[]>() { x2, psi };
                    case "":
                        Console.WriteLine("\nwavelet \"%s\" ",WaveletName);                
                        Console.WriteLine("Filter lengths: %d/%d-tap\n",
                        (((double[])h1[0]).Length),((double[])g1[0]).Length);         

                         Cascade(h1, g1, X, out x1, out phi, out x2, out psi);
                         return new List<double[]>() { x1, phi, x2, psi };
              default:
                    throw new Exception("Invalid parameter");
                }
            }
        }
        /// <summary>
        /// XFIR  Noncausal FIR filtering with boundary handling.
        /// Y = XFIR(B,Z,X,EXT) filters X with FIR filter B with leading
        /// delay -Z along the columns of X.  EXT specifies the  boundary
        /// handling.  Special handling  is done for one and two-tap filters.
        /// 
        /// Ported from Matlab-code by Pascal Getreuer 2005-2006
        /// </summary>
        /// <param name="B">filter</param>
        /// <param name="Z">leading delay</param>
        /// <param name="X">filtered data</param>
        /// <param name="Ext">boundary handling</param>
        /// <returns></returns>
        protected static Matrix Xfir(double[] B, int Z, Matrix X, BoundaryHandling Ext)
        {
            //Console.Write("B=");
            //for (int i = 0; i < B.Length; i++)
            //    Console.Write(" "+B[i]);
            //Console.WriteLine();
            //Console.WriteLine("Z="+Z);
            //Console.WriteLine("Ext=" + Ext.ToString());
            //Console.WriteLine("N=" + X.Dim);
            List<double> list = new List<double>();
                    
            // Special handling for short filters
            if(B.Length == 1 && Z == 0) {
                if (B[0] == 0)
                {
                    return new Matrix(X.Rows, X.Columns);
                }
                else if (B[0] != 1)
                {
                    return X*B[0];
                }
                else {
                    return new Matrix(X);
                }
            }
            
            OnlineFirFilter FIRfilter;
            Matrix m = new Matrix(X.Rows, X.Columns);
            // Padding on the left end
            int pl = (int)Math.Max(B.Length - 1 - Z, 0);
            // Padding on the right end
            int pr = (int)Math.Max(Z, 0);
            double[] i1;
            //Console.WriteLine("pl="+pl+" pr="+pr);
            switch (Ext)
            {
                case BoundaryHandling.asym:
                    list.Clear();
                    for (int i = 0; i < pl; i++) list.Add(1);
                    for (int i = 1; i <= X.Rows; i++) list.Add(i);
                    for (int i = 0; i < pr; i++) list.Add(X.Rows);
                    i1 = list.ToArray();
                    //Console.Write("i1="); ((Vector)i1).PrintAsString(Console.Out);
                    double[] i2;
                    if (pl < X.Rows && pr < X.Rows)
                    {
                        list.Clear();
                        for (int i = pl + 1; i >= 2; i--) list.Add(i);
                        for (int i = 1; i <= X.Rows; i++) list.Add(i);
                        for (int i = X.Rows - 1; i >= X.Rows - pr; i--) list.Add(i);
                        i2 = list.ToArray();
                        //Console.Write("1 i2="); ((Vector)i2).PrintAsString(Console.Out);
                    }
                    else {
                        list.Clear();
                        //Console.WriteLine("X.Length=" + X.Length);
                        for (int i = 1; i <= X.Rows; i++) list.Add(i);
                        for (int i = X.Rows - 1; i >= 2; i--) list.Add(i);
                        i2 = list.ToArray();
                        int Ns = 2 * X.Rows - 2 + (X.Rows == 1 ? 1 : 0);
                        //Console.WriteLine("Ns="+Ns);
                        List<int> i2_indexes = new List<int>();
                        int remainder = 0;
                        for (int i = pl * (Ns - 1); i <= pl * Ns - 1; i++)
                        {
                            Math.DivRem(i, Ns, out remainder); 
                            i2_indexes.Add(remainder + 1);
                        }
                        for (int i = 1; i <= X.Rows; i++) i2_indexes.Add(i);
                        for (int i = X.Rows; i <= X.Rows + pr - 1; i++) {
                            Math.DivRem(i, Ns, out remainder);
                            i2_indexes.Add(remainder + 1);
                        }
                        //Console.Write("i2="); ((Vector)i2).PrintAsString(Console.Out);
                        //Console.Write("i2_indexes="); ((Vector)i2_indexes.ToArray()).PrintAsString(Console.Out);
                        list.Clear();
                        for (int i = 0; i < i2_indexes.Count; i++) list.Add(i2[i2_indexes[i]-1]);
                        i2 = list.ToArray();
                        //Console.Write("2 i2="); ((Vector)i2).PrintAsString(Console.Out);
                    }
                    Matrix filtered = new Matrix(i1.Length,X.Columns);
                    //Console.Write("Xf="); ((Matrix)X).PrintAsString(Console.Out);
                    for (int i = 0; i < i1.Length; i++)
                    {
                        filtered.SetRow(i, 2 * X.GetRow((int)i1[i] - 1) - X.GetRow((int)i2[i] - 1));
                    }
                    //Console.Write("Xf_2="); ((Matrix)(filtered)).PrintAsString(Console.Out);
                    for (int c = 0; c < X.Columns; c++)
                    {
                        FIRfilter = new OnlineFirFilter(B);
                        filtered.SetColumn(c, FIRfilter.ProcessSamples(filtered.GetColumn(c)));
                    }
                    //Console.Write("X="); ((Matrix)(filtered)).PrintAsString(Console.Out);
                    for (int i = Z + pl + 1; i <= Z + pl + X.Rows; i++)
                    {
                        m.SetRow(i - (Z + pl + 1), filtered.GetRow(i - 1));
                    }               
                    break;
                default:
                    for(int row_i = 0; row_i < X.Rows; row_i++) {
                        FIRfilter = new OnlineFirFilter(B);
                        m.SetRow(row_i, FIRfilter.ProcessSamples(X.GetRow(row_i)));
                    }
                    break;
            }   
            return m;
        }
        /// <summary>
        /// 1D-Wavelet transform
        /// </summary>
        /// <param name="WaveletName">used wavelet</param>
        /// <param name="Level">number of stages, positive values for forward transform, negative values for backward transform</param>
        /// <param name="X">data to be transformed</param>
        /// <param name="Ext">boundary handling convention</param>
        /// <param name="Dim">dimension that is operated</param>
        /// <returns>transformed data</returns>
        public static double[] Wavelet1D(string WaveletName, int Level, double[] X, BoundaryHandling Ext, int Dim) {
            
            Wavelet wavelet = GetWavelet(WaveletName);

            int NumStages = wavelet.Seq.Length;
            int EvenStages = (NumStages % 2 == 0 ? 1 : 0);

            // Convert N-D array to a 2-D array with dimension Dim along the columns %%%
            ///  Save original dimensions
            int N = X.Length;
            int N2;
            double[] Xfiltered;

            if (Level == 0)
                throw new Exception("Level must be non-zero for transform.");
            if (N % Math.Pow(2, Level) != 0)
                throw new Exception("Signal length " + N + " must be divisible by 2^" + Level + "=" + Math.Pow(2, Level) + " rem=" + (N % Math.Pow(2, Level)) + ".");
            if (N < Math.Pow(2, Level))
                throw new Exception("Signal length too small for transform level.");
            ///  Forward transform
            if(Level >= 0) {
                for(int i = 0; i < Level; i++) {
                    double[] Xo = new double[(int)(N / 2.0)];
                    double[] Xe = new double[(int)(N / 2.0)];
                    wavelet.Seq.PrintAsString(Console.Out);

                    for(int j = 1; j < N; j+=2)
                    {
                        Xo[j/2] = X[j];
                    }
                    //Console.Write("Xo=\n"); for (int j = 0; j < Xo.Length; j++) Console.Write(Xo[j] + "\n"); Console.WriteLine();
                    Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                    //Console.Write("Xfiltered=\n"); for (int j = 0; j < Xfiltered.Length; j++) Console.Write(Xfiltered[j] + "\n"); Console.WriteLine();
                    for (int j = 1; j < N; j += 2)
                    {
                        Xe[j/2] = X[j - 1] + Xfiltered[(int)Math.Floor(j/2.0)];
                    }

                    //Console.Write("Xe=\n"); for (int j = 0; j < Xe.Length; j++) Console.Write(Xe[j] + "\n"); Console.WriteLine();
                    
                    for (int k = 3; k <= NumStages; k += 2)
                    {
                        //Console.WriteLine("stage="+k);
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        //Console.Write("Xfiltered=\n"); for (int j = 0; j < Xfiltered.Length; j++) Console.Write(Xfiltered[j] + "\n"); Console.WriteLine();
                        for(int j = 0; j < Xo.Length; j++)
                            Xo[j] += Xfiltered[j];
                        //Console.Write("Xo=\n"); for (int j = 0; j < Xo.Length; j++) Console.Write(Xo[j] + "\n"); Console.WriteLine();
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        //Console.Write("Xfiltered=\n"); for (int j = 0; j < Xfiltered.Length; j++) Console.Write(Xfiltered[j] + "\n"); Console.WriteLine();
                        for (int j = 0; j < Xe.Length; j++)
                            Xe[j] += Xfiltered[j];
                        //Console.Write("Xe=\n"); for (int j = 0; j < Xe.Length; j++) Console.Write(Xe[j] + "\n"); Console.WriteLine();
                    }

                    //Console.Write("Xo=\n"); for (int j = 0; j < Xo.Length; j++) Console.Write(Xo[j] + "\n"); Console.WriteLine();
                    //Console.Write("Xe=\n"); for (int j = 0; j < Xe.Length; j++) Console.Write(Xe[j] + "\n"); Console.WriteLine();
                    
                    if(EvenStages == 1) {
                        Xfiltered = Xfir(wavelet.Seq[NumStages-1, 0], (int)wavelet.Seq[NumStages-1, 1, 0], Xe, Ext);
                        for (int j = 0; j < Xe.Length; j++)
                            Xo[j] += Xfiltered[j]; 
                    }
                    for (int k = 0; k < Xe.Length; k++)
                        X[k] = Xe[k] * wavelet.ScaleS;
                    for (int k = 0; k < Xo.Length; k++)
                        X[Xe.Length+k] = Xo[k] * wavelet.ScaleD;
                    N = (int)(0.5*N);

                    //Console.Write("Level="+i+" X=\n"); for (int j = 0; j < X.Length; j++) Console.Write(X[j] + "\n"); Console.WriteLine();
                }
            ///  Inverse transform
            } else {                 
                N = (int)(N*Math.Pow(2.0, Level));
                //Console.Write("X.Length=" + X.Length + " N=" + N + " wavelet.ScaleS=" + wavelet.ScaleS + " " + " wavelet.ScaleD=" + wavelet.ScaleD + "\n");
                for(int i = 1; i <= -Level; i++) {
                    N2 = 2*N;
                    double[] Xo = new double[N];
                    double[] Xe = new double[N];
                    for(int j = 0; j < N; j++)
                        Xe[j] = X[j] / wavelet.ScaleS;
                    for (int j = N; j < N2; j++)
                        Xo[j - N] = X[j] / wavelet.ScaleD;

                    //Console.Write("Xo=\n"); for (int j = 0; j < Xo.Length; j++) Console.Write(Xo[j] + "\n"); Console.WriteLine();
                    //Console.Write("Xe=\n"); for (int j = 0; j < Xe.Length; j++) Console.Write(Xe[j] + "\n"); Console.WriteLine();

                    if(EvenStages == 1) {
                        Xfiltered = Xfir(wavelet.Seq[NumStages-1, 0], (int)wavelet.Seq[NumStages-1, 1, 0], Xe, Ext);
                        for (int j = 0; j < Xo.Length; j++)
                            Xo[j] -= Xfiltered[j];
                    }

                    //Console.Write("Xo=\n"); for (int j = 0; j < Xo.Length; j++) Console.Write(Xo[j] + "\n"); Console.WriteLine();
                    
                    for(int k = NumStages - EvenStages; k >= 2; k-=2) {
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        for (int j = 0; j < Xe.Length; j++)
                            Xe[j] -= Xfiltered[j];
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        for (int j = 0; j < Xo.Length; j++)
                            Xo[j] -= Xfiltered[j];
                    }
                    for(int k = 0; k <= N2; k+=2) {
                        Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                        for(int j = 0; j < N; j++) {
                            X[2*j] = Xe[j] - Xfiltered[j];
                            X[2*j + 1] = Xo[j];
                        }
                    }
                    N = N2;
                }
            }
            return X;
        }
        /// <summary>
        /// 2D-Wavelet transform
        /// </summary>
        /// <param name="WaveletName">used wavelet</param>
        /// <param name="Level">number of stages</param>
        /// <param name="X">data to be transformed</param>
        /// <param name="Ext">boundary handling convention</param>
        /// <param name="Dim">dimension that is operated</param>
        /// <returns>transformed data</returns>
        public static Matrix Wavelet2D(string WaveletName, int Level, Matrix X, BoundaryHandling Ext, int Dim)
        {
            Wavelet wavelet = GetWavelet(WaveletName);

            int NumStages = wavelet.Seq.Length;
            int EvenStages = (NumStages % 2 == 0 ? 1 : 0);

            // Convert N-D array to a 2-D array with dimension Dim along the columns %%%
            ///  Save original dimensions
            int[] N = new int[] { X.Rows, X.Columns };
            Console.WriteLine("N=[" + N[0] + "," + N[1] + "]");
            int[] N2 = new int[2];
            double[,] Xfiltered;

            if (Level == 0)
                throw new Exception("Level must be non-zero for transform.");
            if (N[0] % Math.Pow(2, Level) != 0)
                throw new Exception("Signal length " + N[0] + "xM must be divisible by 2^" + Level + "=" + Math.Pow(2, Level) + " rem=" + (N[0] % Math.Pow(2, Level)) + ".");
            if (N[1] % Math.Pow(2, Level) != 0)
                throw new Exception("Signal length Nx" + N[1] + " must be divisible by 2^" + Level + "=" + Math.Pow(2, Level) + " rem=" + (N[1] % Math.Pow(2, Level)) + ".");
            if (N[0] < Math.Pow(2, Level))
                throw new Exception("Signal length " + N[0] + "xM too small for transform level.");
            if (N[1] < Math.Pow(2, Level))
                throw new Exception("Signal length Nx" + N[1] + " too small for transform level.");
            ///  Forward transform
            if (Level >= 0)
            {
                //Console.Write("X="); ((Matrix)X).PrintAsString(Console.Out);

                wavelet.Seq.PrintAsString(Console.Out);
                for (int i = 0; i < Level; i++)
                {
                    double[,] Xo = new double[N[0] / 2, N[1]];
                    double[,] Xe = new double[N[0] / 2, N[1]];
                    //initialize and transform each column
                    #region initialize_rows
                    for (int r = 0; r < N[0]/2; r ++)
                        for (int j = 0; j < N[1]; j++)
                            Xo[r, j] = X[2*r+1, j];
                    Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                    //Console.Write("Xfiltered="); ((Matrix)Xfiltered).PrintAsString(Console.Out);
                    for (int c = 0; c < N[0]/2; c++)
                        for (int j = 0; j < N[1]; j++)
                            Xe[c, j] = X[2*c, j] + Xfiltered[c, j];
                    #endregion

                    //Console.Write("(after init) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after init) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region apply_filter_stages
                    for (int k = 3; k <= NumStages; k += 2)
                    {
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        Console.Write("Xfiltered="); ((Matrix)Xfiltered).PrintAsString(Console.Out);
                        for (int c = 0; c < N[0] / 2; c++)
                        for (int j = 0; j < N[1]; j++)
                            Xo[c, j] += Xfiltered[c, j];
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        for (int c = 0; c < N[0] / 2; c++)
                        for (int j = 0; j < N[1]; j++)
                            Xe[c, j] += Xfiltered[c, j];
                    }
                    #endregion

                    //Console.Write("(after filter) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after filter) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region handle_even_stage_filters
                    if (EvenStages == 1)
                    {
                        Xfiltered = Xfir(wavelet.Seq[NumStages - 1, 0], (int)wavelet.Seq[NumStages - 1, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[0] / 2; c++)
                        for (int j = 0; j < N[1]; j++)
                            Xo[c, j] += Xfiltered[c, j];
                    }
                    #endregion

                    #region apply_scaling_and_save_into_X
                    for (int c = 0; c < N[0] / 2; c++)
                    {
                        for (int k = 0; k < N[1]; k++)
                        {
                            X[c, k] = Xe[c, k] * wavelet.ScaleS;
                            X[N[0]/2 + c, k] = Xo[c, k] * wavelet.ScaleD;
                        }
                    }
                    #endregion

                    //Console.Write("(after save) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after save) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);
                    //Console.Write("(after save) X="); ((Matrix)X).PrintAsString(Console.Out);
                    Xo = new double[N[1] / 2, N[0]];
                    Xe = new double[N[1] / 2, N[0]];
                    //transform each column
                    #region initialize_columns
                    for (int r = 0; r < N[1]/2; r++)
                        for (int j = 0; j < N[0]; j++)
                        {
                            Xo[r, j] = X[j, 2*r+1];
                            Xe[r, j] = X[j, 2*r];
                        }
                    Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                    for (int c = 0; c < N[1] / 2; c++)
                        for (int j = 0; j < N[0]; j++)
                            Xe[c, j] += Xfiltered[c, j];
                    #endregion

                    //Console.Write("(after col init) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after col init) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region apply_filter_stages
                    for (int k = 3; k <= NumStages; k += 2)
                    {
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[1] / 2; c++)
                            for (int j = 0; j < N[0]; j++)
                                Xo[c, j] += Xfiltered[c, j];
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        for (int c = 0; c < N[1] / 2; c++)
                            for (int j = 0; j < N[0]; j++)
                                Xe[c, j] += Xfiltered[c, j];
                    }
                    #endregion

                    //Console.Write("(after filter) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after filter) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region handle_even_stage_filters
                    if (EvenStages == 1)
                    {
                        Xfiltered = Xfir(wavelet.Seq[NumStages - 1, 0], (int)wavelet.Seq[NumStages - 1, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[1]/ 2; c++)
                            for (int j = 0; j < N[0]; j++)
                                Xo[c, j] += Xfiltered[c, j];
                    }
                    #endregion

                    #region apply_scaling_and_save_into_X
                    for (int c = 0; c < N[0]; c++)
                    {
                        for (int k = 0; k < N[1]/2; k++)
                        {
                            X[c, k] = Xe[k, c] * wavelet.ScaleS;
                            X[c, N[1] / 2 + k] = Xo[k, c] * wavelet.ScaleD;
                        }
                    }
                    #endregion

                    N[0] /= 2;
                    N[1] /= 2;
                }
            }
            ///  Inverse transform
            else
            {
                N[0] = (int)(Math.Pow(2.0, Level)*N[0]);
                N[1] = (int)(Math.Pow(2.0, Level)*N[1]);
                for (int i = 1; i <= -Level; i++)
                {
                    Console.WriteLine("Level="+i+" N=["+N[0]+","+N[1]+"]");
                    N2[0] = 2 * N[0];
                    N2[1] = 2 * N[1];
                    double[,] Xo = new double[N[1], N2[0]];
                    double[,] Xe = new double[N[1], N2[0]];
                    //initialize and transform each row
                    #region initialize_rows
                    for (int r = 0; r < N[1]; r++)
                        for (int j = 0; j < N2[0]; j++)
                        {
                            Xe[r, j] = X[j, r] / wavelet.ScaleS;
                            Xo[r, j] = X[j, N[1] + r] / wavelet.ScaleD;
                        }
                    #endregion

                    //Console.Write("(after init) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after init) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region handle_even_stage_filters
                    if (EvenStages == 1)
                    {
                        Xfiltered = Xfir(wavelet.Seq[NumStages - 1, 0], (int)wavelet.Seq[NumStages - 1, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[1]; c++)
                            for (int j = 0; j < N2[0]; j++)
                                Xo[c, j] -= Xfiltered[c, j];
                    }
                    #endregion

                    #region apply_filter_stages
                    for (int k = NumStages-EvenStages; k >= 3; k -= 2)
                    {
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        for (int c = 0; c < N[1]; c++)
                            for (int j = 0; j < N2[0]; j++)
                                Xe[c, j] -= Xfiltered[c, j];
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[1]; c++)
                            for (int j = 0; j < N2[0]; j++)
                                Xo[c, j] -= Xfiltered[c, j];
                    }
                    #endregion

                    //Console.Write("(after filter) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after filter) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region save_into_X
                    Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                    for (int c = 0; c < N2[0]; c++)
                    {
                        for (int k = 0; k < N2[1] / 2; k++)
                        {
                            X[c, 2*k] = Xe[k, c] - Xfiltered[k, c];
                            X[c, 2*k+1] = Xo[k, c];
                        }
                    }
                    #endregion

                    //.Write("(after save) X="); ((Matrix)X).PrintAsString(Console.Out);

                    Xe = new double[N[0], N2[1]];
                    Xo = new double[N[0], N2[1]];
                    
                    #region initialize_columns
                    for (int r = 0; r < N[0]; r++)
                    {
                        for (int j = 0; j < N2[1]; j++)
                        {
                            Xe[r, j] = X[r, j] / wavelet.ScaleS;
                            Xo[r, j] = X[N[0] + r, j] / wavelet.ScaleD;
                        }
                    }
                    #endregion

                    //Console.Write("(after init) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after init) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region handle_even_stage_filters
                    if (EvenStages == 1)
                    {
                        Xfiltered = Xfir(wavelet.Seq[NumStages - 1, 0], (int)wavelet.Seq[NumStages - 1, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[0]; c++)
                            for (int j = 0; j < N2[1]; j++)
                                Xo[c, j] -= Xfiltered[c, j];
                    }
                    #endregion

                    #region apply_filter_stages
                    for (int k = NumStages - EvenStages; k >= 3; k -= 2)
                    {
                        Xfiltered = Xfir(wavelet.Seq[k - 1, 0], (int)wavelet.Seq[k - 1, 1, 0], Xo, Ext);
                        for (int c = 0; c < N[0]; c++)
                            for (int j = 0; j < N2[1]; j++)
                                Xe[c, j] -= Xfiltered[c, j];
                        Xfiltered = Xfir(wavelet.Seq[k - 2, 0], (int)wavelet.Seq[k - 2, 1, 0], Xe, Ext);
                        for (int c = 0; c < N[0]; c++)
                            for (int j = 0; j < N2[1]; j++)
                                Xo[c, j] -= Xfiltered[c, j];
                    }
                    #endregion

                    //Console.Write("(after filter) Xo="); ((Matrix)Xo).PrintAsString(Console.Out);
                    //Console.Write("(after filter) Xe="); ((Matrix)Xe).PrintAsString(Console.Out);

                    #region save_into_X
                    Xfiltered = Xfir(wavelet.Seq[0, 0], (int)wavelet.Seq[0, 1, 0], Xo, Ext);
                    for (int c = 0; c < N2[0]/2; c++)
                    {
                        for (int k = 0; k < N2[1]; k++)
                        {
                            X[2*c, k] = Xe[c, k] - Xfiltered[c, k];
                            X[2*c+1, k] = Xo[c, k];
                        }
                    }
                    #endregion

                    N[0] = N2[0];
                    N[1] = N2[1];
                }
            }
            return X;
        }
    }
}
