﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Fit is used to calculate different kinds of estimators
    /// for simulated data.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Fit
    {
        /// <summary>
        /// Model used to simulate data
        /// </summary>
        private Model model;

        /// <summary>
        /// Measured data
        /// </summary>
        private Vector data;

        /// <summary>
        /// Weights given to data samples
        /// </summary>
        private Vector weights;

        /// <summary>
        /// Constructor. Uses default weights of 1.0
        /// </summary>
        /// <param name="m">Model</param>
        /// <param name="d">Data</param>
        public Fit(Model m, Vector d)
        {
            if (m.Samples != d.Dim) throw new FitException("Data and model sample counts inequal");

            model = m;
            data = new Vector(d);
            weights = Vector.Fill(d.Dim, 1.0);
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="m">Model</param>
        /// <param name="d">Data</param>
        /// <param name="w">
        /// Weights given to different data points. By default,
        /// this is 1.0 for all points.
        /// </param>
        public Fit(Model m, Vector d, Vector w)
        {
            if ((m.Samples != d.Dim) || (w.Dim != d.Dim))
                throw new FitException("Data, weight and model sample counts inequal");

            model = m;
            data = new Vector(d);
            weights = new Vector(w);
        }

        /// <summary>
        /// Sum of squares of difference between the data simulated using given parameters
        /// and measured data.
        /// </summary>
        /// <param name="parameters">Model parameters</param>
        /// <returns>Sum of squares</returns>
        public double SumOfSq(Vector parameters)
        {
            Vector simdata = model.Simulate(parameters);

            Vector dif = simdata - data;
            return Vector.Dot(dif, dif);
        }

        /// <summary>
        /// Weighted sum of squares of difference between the data simulated using 
        /// given parameters and measured data.
        /// </summary>
        /// <param name="parameters">Model parameters</param>
        /// <returns>Weighted sum of squares</returns>
        public double WSumOfSq(Vector parameters)
        {
            Vector simdata = model.Simulate(parameters);

            Vector dif = (simdata - data) * weights;
            return Vector.SqNorm(dif);
        }

    }
}
