﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// (Iterative) Topographical Global Optimization. Finds a global minimum
    /// of a function by clustering random samples around local minimums. This method
    /// also uses the Powell-Brent method for local optimization.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class ITGO : Optimization
    {
        // Number of Powell-Brent iterations run from topographical minima
        private const int LOCAL_ITERATIONS = 1;

        private List<Vector> minima;
        private int clustersize;
        private int samplecount;

        private Vector floor, ceiling;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="tf">Target function to minimize</param>
        /// <param name="csize">Clustersize i.e. the number of closest neighbours considered. At most smp-1.</param>
        /// <param name="smp">Initial number of samples generated</param>
        /// <param name="c">Upper limits for parameters</param>
        /// <param name="f">Lower limits for parameters</param>
        public ITGO(RealFunction tf, int csize, int smp, Vector c, Vector f)
        {
            SetTargetFunction(tf);
            floor = f;
            ceiling = c;

            // Put something into InitialParams, so that it's not null
            InitialParams = floor;

            // Clustersize is at most samples - 1
            clustersize = (csize < smp) ? csize : (smp - 1);
            samplecount = smp;

            minima = new List<Vector>();
        }

        /// <summary>
        /// Constructor. InitMethod must be called if this is used
        /// </summary>
        public ITGO() { }


        /// <summary>
        /// Gets info of Method and its parameters
        /// </summary>
        /// <returns>Info of method</returns>
        public static OptimizationInfo GetInfo()
        {
            OptimizationInfo info = new OptimizationInfo();
            info.MethodName = "ITGO";
            info.ParameterName = new System.Collections.Generic.List<string>();
            info.ParameterValue = new System.Collections.Generic.List<double>();

            info.ParameterName.Add("ITGOSamples");
            info.ParameterName.Add("SampleSize");
            info.ParameterValue.Add(20);
            info.ParameterValue.Add(1);

            return info;
        }

        /// <summary>
        /// Inits PowellBrent. This method is called from Parent class (optimization)
        /// Create() -method. 
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="info">Info of method (Not used in PowellBrent)</param>
        /// <param name="c">parameter ceiling values</param>
        /// <param name="f">parameter floor values</param>
        /// <param name="initial">initial parameter values</param>
        protected override void InitMethod(
            RealFunction function,
            OptimizationInfo info,
            Vector c,
            Vector f,
            Vector initial)
        {
            // Powell is not using min and max values. Values must be converted to
            // deltas
            int length = c.Length;

            SetTargetFunction(function);
            floor = f;
            ceiling = c;

            // Put something into InitialParams, so that it's not null
            InitialParams = initial;
            int smp = (int)info.ParameterValue[0];
            int csize = (int)info.ParameterValue[1];

            // Clustersize is at most samples - 1
            clustersize = (csize < smp) ? csize : (smp - 1);
            samplecount = smp;

            minima = new List<Vector>();
        }

        /// <summary>
        /// Run a single iteration of ITGO
        /// </summary>
        /// <returns>Smallest topographical minimum found</returns>
        protected override Vector Step()
        {
            // Create set of samples
            Vector[] samples = MakeSamples();

            // Find the topographical minima
            minima = GetMinima(samples);

            double globalminval = Double.MaxValue;
            Vector globalmin = new Vector();
            
            // Run Powell optimization from each minimum and find the best one
            for (int i = 0; i < minima.Count; i++)
            {
                Optimization opt = new PowellBrent(TargetFunction, minima[i]);
                opt.SetRange(floor, ceiling);
                opt.Iterate(LOCAL_ITERATIONS);

                minima[i] = opt.GetMinimum();

                double localminval = opt.AtMinimum();

                if (localminval < globalminval)
                {
                    globalmin = opt.GetMinimum();
                    globalminval = localminval;
                }
            
            }            
            return globalmin;
        }

        /// <summary>
        /// Generates a set of samples and finds the topographical minima
        /// </summary>
        /// <param name="samples">Number of samples to generate</param>
        /// <returns>List of topographical minima</returns>
        private List<Vector> GetMinima(Vector[] samples)
        {
            List<Vector> clusterminima = new List<Vector>();

            DistanceMatrix dm = new DistanceMatrix(samples);

            double[] fvals = new double[samples.Length];
            double[] dists = new double[samples.Length];
            int[] ind = new int[samples.Length];

            // Calculate function values
            for (int i = 0; i < samples.Length; i++)
            {
                fvals[i] = TargetFunction(samples[i]);
            }

            for (int i = 0; i < samples.Length; i++)
            {
                for (int j = 0; j < samples.Length; j++)
                {
                    dists[j] = dm[i, j];
                    ind[j] = j;
                }

                Array.Sort(dists, ind);

                bool isMinima = true;
                for (int k = 0; k < clustersize; k++)
                {
                    if (fvals[ind[k]] < fvals[i])
                    {
                        // Sample i is not a topological minimum
                        isMinima = false;
                        break;
                    }
                }
                // If sample i is smaller than its neighbours, add to minima
                if (isMinima)
                {
                    clusterminima.Add(samples[i]);
                }
            }

            return clusterminima;
        }

        /// <summary>
        /// Generates a new set of samples that includes the minima from previous iteration
        /// </summary>
        /// <returns>Array of samples</returns>
        private Vector[] MakeSamples()
        {
            Vector[] samples = new Vector[samplecount];

            // Include the minima of previous iteratiot in the samples
            minima.CopyTo(samples);

            for ( int i = minima.Count; i < samplecount; i++)
            {
                samples[i] = Vector.RandomPoint(ceiling, floor, i * samplecount * 2);
            }

            return samples;
        }

        /// <summary>
        /// Stores distances between samples in a jagged matrix
        /// </summary>
        private class DistanceMatrix
        {
            double[][] distances;

            public DistanceMatrix(Vector[] samples)
            {
                int n = samples.Length;
                distances = new double[n][];

                for (int i = 0; i < n; i++)
                {
                    distances[i] = new double[i];

                    for (int j = 0; j < i; j++)
                    {
                        // The actual distance is not needed, so use square of the
                        // euclidean norm to avoid square root calculation
                        distances[i][j] = Vector.SqNorm(samples[i] - samples[j]);
                    }
                }
            }

            /// <summary>
            /// Indexer returns the distance between n:th and m:th sample.
            /// Distance between sample and itself is defined as Double.MaxValue
            /// </summary>
            /// <param name="n">Sample number</param>
            /// <param name="m">Sample number</param>
            /// <returns>Distance between samples</returns>
            public double this[int n, int m]
            {
                get
                {
                    if (m == n) return Double.MaxValue;
                    return (n > m) ? distances[n][m] : distances[m][n];
                }
                set
                {
                    if (m != n)
                    {
                        if (n > m) distances[n][m] = value;
                        else distances[m][n] = value;
                    }
                }
            }
        }
    }
}
