/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Methods for linear interpolation and integration of PET and blood/plasma TACs.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class LinearIntegration
    {
        private bool IsAscending(double[] a)
        {
            for (int i = 1; i < a.Length; i++)
            {
                if (a[i - 1] > a[i]) return false;
            }
            return true;
        }

        /// <summary>
        /// Linear integration from time 0 to x[x.Lenght-1].
        /// </summary>
        /// <remarks>
        /// PRE: 
        /// The x values must be in ascending order.
        /// x.Length == y.Length and x != null and y != null
        /// Duplicates are not allowed, all x[i] must be >=0.
        /// POST:
        /// If x[0] is greater than 0 and x[0] is less or equal to x[1]-x[0], 
        /// the beginning is interpolated from it to (0,0).
        /// Tables x and y are not changed.
        /// RESULT.Length == x.Length
        /// </remarks>
        /// <param name="x">original x values</param>
        /// <param name="y">original y values</param>
        /// <returns>Integral values in each time point.</returns>
        public double[] Integrate(double[] x, double[] y)
        {
            if (x == null || y == null)
                throw new TPCException("Null values not accepted.");
            if (x.Length != y.Length)
                throw new TPCException("The length of input arrays are not same.");
            if (!IsAscending(x))
                throw new TPCException("The data is not ascending");

            // initializes the return value.
            double[] yi = new double[x.Length];

            // index 0
            // If the gap in the beginning is longer than time 
            // between first and second time point 
            if (x.Length == 1 || x[0] <= (x[1] - x[0]))
            {
                yi[0] = 0.0 + 0.5 * (y[0] + 0)* (x[0] - 0);
            }
            else // the gap is narrower
            {
                yi[0] = 0;
            }

            // index 1 to x.Length
            for (int i = 1; i < x.Length; i++)
            {
                yi[i] = yi[i - 1] + 0.5 * (y[i] + y[i - 1]) * (x[i] - x[i - 1]);
            }

            return yi;
        }

        /// <summary>
        /// Calculates integrals of PET data at frame end times.
        /// </summary>
        /// <remarks>
        /// PRE:
        /// Data does not have to be continuous, but it must be in ascending order.
        /// POST:
        /// If x1[0] > 0 and less or equal to(x2[0]-x1[0]), the 
        /// beginning is interpolated from (0, 0) to (x1[0], y1[0]*x1[0]/x) 
        /// where x is midtime of the first frame. 
        /// Returns 
        /// </remarks>
        /// <param name="x1">Array of frame start times</param>
        /// <param name="x2">Array of frame end times</param>
        /// <param name="y">Array of y values (avg during each frame)</param>
        /// <param name="integral1st">Output: integral values at frame end times, or null</param>
        /// <param name="integral2nd">Output: 2nd integral values at frame end times, or null</param>
        public static void IntegrateToFrameEndTimes(
            double[] x1, double[] x2,
            double[] y, out double[] integral1st, out double[] integral2nd)
        {
            integral1st = new double[y.Length];
            integral2nd = new double[y.Length];

            if (x1.Length != x2.Length)
                throw new TPCException("Input arrays have not the same length.");

            if (integral1st != null && x1.Length != integral1st.Length)
                throw new TPCException("Output array is wrong sized.");

            if (integral2nd != null && x1.Length != integral2nd.Length)
                throw new TPCException("Output array is wrong sized.");

            if (x1.Length < 1 || x1[0] < 0)
                throw new TPCException("No negative x values allowed.");

            for (int i = 0; i < x1.Length; i++)
            {
                if (x2[i] < x1[i])
                    throw new TPCException("the data is not increasing in time.");
            }

            for (int i = 1; i < x1.Length; i++)
            {
                if (x1[i] <= x1[i - 1])
                    throw new TPCException("The data is too much overlapping");
            }

            double x, a;

            // Allocate memory for temp data, if necessary
            if (integral1st == null)
            {
                integral1st = new double[x1.Length];
            }

            // Calculate the integral at the end of first frame
            integral1st[0] = (x2[0] - x1[0]) * y[0];
            
            // If frame does not start at 0 time,
            // add the area in the beginning
            if (x1[0] > 0)
            {
                // But only if the gap in the beginning is
                // less than the lenght of the first frame
                if (x1[0] <= x2[0] - x1[0])
                {
                    x = (x1[0] + x2[0]) / 2.0;
                    a = (x1[0] * (y[0] / x) * x1[0]) / 2.0;
                    integral1st[0] += a;
                }
            }

            // Calculate integrals at the ends of following frames
            for (int i = 1; i < x1.Length; i++)
            {
                // Add the area of this frame to the previous integral
                a = (x2[i] - x1[i]) * y[i];
                integral1st[i] = integral1st[i - 1] + a;
                
                // Check whether frames are continuous
                if (x1[i] == x2[i - 1])
                    continue;

                // When not, calculate the area of an imaginary frame
                x = (x1[i] + x2[i - 1]) / 2.0;
                a = (x1[i] - x2[i - 1]) *
                  (y[i] - (y[i] - y[i - 1]) * (x2[i] + x1[i] -
                  2.0 * x) / (x2[i] + x1[i] - x2[i - 1] - x1[i - 1]));

                integral1st[i] += a;
            }

            // Calculate 2nd integrals if required
            if (integral2nd != null)
            {
                integral2nd[0] = x2[0] * integral1st[0] / 2.0;
                
                for (int i = 1; i < x1.Length; i++)
                {
                    integral2nd[i] = integral2nd[i - 1] + (x2[i] - x2[i - 1]) * (integral1st[i - 1] + integral1st[i]) / 2;
                }
            }
        }

        /// <summary>
        /// Integrate PET TAC data to frame mid times.
        /// </summary>
        /// <remarks>
        /// PRE:
        /// Frames must be in ascending time order. Gaps and small overlap are allowed. 
        /// Any of output arrays may be set to null if that is not needed.
        /// The first and second integral must be allocated.
        /// POST:
        /// If x1[0] greater than 0 and x1[0] is less or equal to x2[0]-x1[0], 
        /// an imaginary line is drawn from (0,0) to (x[0],y[0]). 
        /// </remarks>
        /// <param name="x1">frame start times</param>
        /// <param name="x2">frame end times </param>
        /// <param name="y">avg value during frame </param>
        /// <param name="out ie">integral values at frame mid times</param>
        /// <param name="out iie">second integral values at frame mid times</param>
        public static void IntegrateToFrameMidTimes(double[] x1, double[] x2,
            double[] y, out double[] ie, out double[] iie)
        {
            ie = new double[y.Length];
            iie = new double[y.Length];
            
            // Check for data
            if (x1.Length < 1 || x2.Length < 1)
                throw new TPCException("1");

            if (ie == null || iie == null)
                throw new TPCException("output must've been allocated");

            // for looping history
            double last_x = 0;
            double last_x2 = 0;
            double last_y = 0;
            double last_integral = 0;
            double box_integral = 0;
            double integral1st = 0;
            double integral2nd = 0;

            for (int i = 0; i < x1.Length; i++)
            {
                double frame_len = x2[i] - x1[i];
                if (frame_len < 0.0)
                    throw new TPCException("the frame length was negative.");

                double x = 0.5 * (x1[i] + x2[i]);
                double xdist = x - last_x;
                if (last_x > 0.0 && xdist <= 0.0)
                    throw new TPCException("Frames must be in ascending order.");

                if (x < 0)
                {
                    if (ie != null) ie[i] = integral1st;
                    if (iie != null) iie[i] = integral2nd;
                    continue;
                }

                // slope
                double s = (y[i] - last_y) / xdist;

                // If there is a big gap in the beginning it is eliminated
                if (i == 0)
                {
                    if (x1[0] > x2[0] - x1[0])
                    {
                        last_x2 = last_x = x1[0];
                    }
                }

                // Integral of a possible gap between frames
                double gap_integral = (x1[i] - last_x2) * (last_y + s * ((last_x2 + x1[i]) / 2.0 - last_x));

                // Integral from the beginning of the frame to the middle of the frame 
                double half_integral = (x - x1[i]) * (last_y + s * ((x1[i] + x) / 2.0 - last_x));

                // sum up
                integral1st = box_integral + gap_integral + half_integral;

                // half_integral is not added to box because it is
                // more accurate to  increment integrals of full frames
                box_integral += gap_integral + frame_len * y[i];
                integral2nd += xdist * (integral1st + last_integral) * 0.5;

                if (ie != null)
                    ie[i] = integral1st;
                if (iie != null)
                    iie[i] = integral2nd;

                last_x = x;
                last_x2 = x2[i];
                last_y = y[i];
                last_integral = integral1st;
            }
        }
    }
}
