﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Linear interpolation functions
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class LinearInterpolation : InterpolationMethod
    {
        
        /// <summary>
        /// Interpolates at data points.
        /// </summary>
        /// <param name="x">interpolation locations</param>
        /// <param name="data_x">x-values of data</param>
        /// <param name="data_y">y-values of data</param>
        /// <returns>y-values evaluated at locations x</returns>
        public override double[] Interpolate(double[] x, double[] data_x, double[] data_y, ExternalValueHandlingMethod m)
        {
            /// <summary>
            /// Y-values evaluated at locations x
            /// </summary>
            double[] result = new double[x.Length];
            /// <summary>
            /// slopes between two data value points
            /// </summary>
            double[] slopes = new double[data_x.Length - 1];

            // Counting slopes 
            double x1;
            double y1;
            double x2;
            double y2;
            for (int k = 0; k < data_x.Length - 2; k++)
            {
                x1 = data_x[k];
                y1 = data_y[k];
                x2 = data_x[k + 1];
                y2 = data_y[k + 1];
                slopes[k] = (y2 - y1) / (x2 - x1);
            }
            int i, j;
            
            
            switch (m)
            {
                case ExternalValueHandlingMethod.EXTRAPOLATION:
                    // evaluation of y-values
                    i = 0;
                    j = 0;
                    // x-values start before data_x-values
                    while (x[i] <= data_x[j])
                    {
                        result[i] = slopes[j] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }
                    j++;
                    // x-values are between data_x[0] - data_x[data_x.Length]
                    while (i < x.Length - 1 && j < data_x.Length)
                    {
                        if (x[i] <= data_x[j])
                        {
                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;
                        }
                        else
                        {
                            while (x[i] > data_x[j] && j < data_x.Length) j++;

                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;

                        }
                    }
                    // data_x-values end before x-values end
                    while (i < x.Length)
                    {
                        result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }    
                break;                
                case ExternalValueHandlingMethod.ZERO_VALUES:
                    // evaluation of y-values
                    i = 0;
                    j = 0;
                    // Turning result values, which x-coordinates are located before data_x[0], to zero.
                    while (x[i] <= data_x[j])
                    {
                        result[i] = 0;
                        i++;
                    }
                    j++;
                    // x-values are between data_x[0] - data_x[data_x.Length]
                    while (i < x.Length - 1 && j < data_x.Length)
                    {
                        if (x[i] <= data_x[j])
                        {
                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;
                        }
                    else
                        {
                        while (x[i] > data_x[j] && j < data_x.Length) j++;

                        result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                        i++;

                        }
                    }
                    // Turning result values, which x-coordinates are located after data_x[data_x.Length], to zero.
                    while (i < x.Length)
                    {
                        result[i] = 0;
                        i++;
                    }  
                break;
                case ExternalValueHandlingMethod.PATLAK:
                // evaluation of y-values
                i = 0;
                j = 0;
                // Turning result values, which x-coordinates are located before data_x[0], to zero.
                if (data_x[1] - data_x[0] < data_x[0] - x[0])
                {
                    while (x[i] <= data_x[j])
                    {
                        result[i] = 0;
                        i++;
                    }
                }
                j++;
                // x-values are between data_x[0] - data_x[data_x.Length]
                while (i < x.Length - 1 && j < data_x.Length)
                {
                    if (x[i] <= data_x[j])
                    {
                        result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }
                    else
                    {
                        while (x[i] > data_x[j] && j < data_x.Length) j++;

                        result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                        i++;

                    }
                }
                // data_x-values end before x-values end
                while (i < x.Length)
                {
                    result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                    i++;
                }  
                break;
                default: break;
            } 
            return result;
        }
        
        /// <summary>
        /// Linear interpolation and integration.
        /// </summary>
        /// <remarks>
        /// Both original and new interpolated data represent the actual values
        /// at specified time points, not from framed data.
        /// Then output can be integrated dot-by-dot.
        /// PRE: 
        /// Original data and new x values must be sorted by ascending x.
        /// Subsequent x values can have equal values, enabling the use of step functions.
        /// Negative x (time) values can be processed.
        /// The output data must've been allocated.
        /// POST:
        /// If output y is null, this doesn't calculate the first and second integral values.
        /// If necessary, the data is extrapolated assuming that: 
        ///     1) y[infinity]=y[y.Length-1] and 
        ///     2) if x[0] greater than 0, y[0] greater than 0 and x[0] is less or equal to x[1]-x[0], 
        ///        an imaginary line is drawn from (0,0) to (x[0],y[0]). 
        /// </remarks>
        /// <param name="x">input x-values</param>
        /// <param name="y">input y-values</param>
        /// <param name="newx">interpolation x-locations</param>
        /// <param name="newy">interpolated y-values; null can be given</param>
        /// <param name="int1st">Integral; null can be given</param>
        /// <param name="int2nd">Second Integral; null can be given</param>
        public static void Interpolate(
            double[] x, double[] y,
            double[] newx, out double[] newy,
            out double[] int1st, out double[] int2nd)
        {
            newy = new double[newx.Length];
            int1st = new double[newx.Length];
            int2nd = new double[newx.Length];

            // Check for data
            if (newx.Length != newy.Length)
                throw new TPCException("the x and y input data sizes don't match.");

            if (x.Length != y.Length)
                throw new TPCException("the x and y input data sizes don't match.");

            if (newy == null && int1st == null && int2nd == null)
                throw new TPCException("The output data must've been allocated.");

            double ty, tyi, tyii, ly, li, lii, lil;

            // Set output values before the input data has started
            // If newx is also negative, set values to zero
            int j;
            for (j = 0; j < newx.Length; j++)
            {
                if (newx[j] >= x[0] || newx[j] >= 0.0)
                    break;

                ty = tyi = tyii = 0.0;

                if (newy != null)
                    newy[j] = ty;
                if (int1st != null)
                    int1st[j] = tyi;
                if (int2nd != null)
                    int2nd[j] = tyii;
            }
            // Now newx[j]>=x[0] or newx[j]>=0 
            // If newx is still smaller than x[0],
            // but positive, then interpolate
            // between (0,0) and (x[0], y[0]) 
            if (newx[j] >= 0.0)
            {
                for (; j < newx.Length; j++)
                {
                    if (newx[j] > x[0]) break;

                    // Second condition was added 14.6.
                    if (x[0] == 0.0 || (x.Length > 1 && x[0] > x[1] - x[0]))
                    {
                        ty = tyi = tyii = 0.0;
                        if (newx[j] == x[0])
                            ty = y[0];
                    }
                    else
                    {
                        ty = newx[j] * y[0] / x[0];
                        tyi = 0.5 * ty * newx[j];
                        tyii = 0.5 * tyi * newx[j];
                    }

                    if (newy != null)
                        newy[j] = ty;
                    if (int1st != null)
                        int1st[j] = tyi;
                    if (int2nd != null)
                        int2nd[j] = tyii;
                }
            }

            if (j == newx.Length)
            {
                // Normal Return
                return;
            }

            // Now newx[j]>=x[0]
            if (newx[j] < x[0])
                throw new TPCException("3");

            // Calculate input data values at x[0]
            ly = y[0];

            if (x[0] <= 0.0 || x[0] > x[1] - x[0])
            {
                li = lii = lil = 0.0;
            }
            else
            {
                lil = li = 0.5 * x[0] * ly;
                lii = 0.5 * x[0] * li;
            }

            // Calculate input data values at x[i]
            int i;
            for (i = 1; i < x.Length && j < newx.Length; i++)
            {
                // calculate output values between x[i-1] and x[i]
                for (; j < newx.Length; j++)
                {
                    if (newx[j] > x[i])
                        break;

                    ty = ((y[i] - y[i - 1]) / (x[i] - x[i - 1]))
                        * (newx[j] - x[i - 1]) + y[i - 1];

                    if (newy != null)
                        newy[j] = ty;

                    tyi = li + 0.5 * (ty + y[i - 1]) * (newx[j] - x[i - 1]);

                    if (int1st != null)
                        int1st[j] = tyi;
                    if (int2nd != null)
                        int2nd[j] = tyii = lii + 0.5 * (tyi + li) * (newx[j]
                            - x[i - 1]);
                }
                // calculate integrals of original data at x[i]
                li += 0.5 * (y[i] + y[i - 1]) * (x[i] - x[i - 1]);
                lii += 0.5 * (li + lil) * (x[i] - x[i - 1]);
                lil = li;
            }

            // new values after the last x
            for (; j < newx.Length; j++)
            {
                ty = y[x.Length - 1];
                if (newy != null)
                    newy[j] = ty;
                tyi = li + ty * (newx[j] - x[i - 1]);
                if (int1st != null)
                    int1st[j] = tyi;
                if (int2nd != null)
                    int2nd[j] = tyii = lii + 0.5 * (tyi + li) * (newx[j] - x[i - 1]);
            }
        }

        /// <summary>
        /// Interpolate and integrate PET TAC data to frame end times.
        /// </summary>
        /// <remarks>
        /// PRE: Any of output arrays may be set to null if not needed. 
        /// Frames must be in ascending time order.
        /// Gaps and small overlap are allowed.
        /// The output must've been allocated
        /// POST:
        /// </remarks>
        /// <param name="x1">frame start times</param>
        /// <param name="x2">frame end times</param>
        /// <param name="y">avg value during frame</param>
        /// <param name="e">number of frames</param>
        /// <param name="ie">integrals at frame end time</param>
        /// <param name="iie">2nd integrals at frame end time</param>
        public void InterpolateToFrameEndTimes(
            double[] x1, double[] x2,
            double[] y, double[] e, double[] ie, double[] iie)
        {
            if (e == null && ie == null && iie == null)
                throw new TPCException("The output must've been allocated");

            int nr = x1.Length;
            double x;
            int i;
            double last_x = 0.0, last_x2 = 0.0, last_y = 0.0,
                last_integral = 0.0, value = 0.0, integral = 0.0,
                integral2 = 0.0, frame_len = 0.0, xdist = 0.0,
                s = 0.0;

            for (i = 0; i < nr; i++)
            {
                frame_len = x2[i] - x1[i];
                if (frame_len < 0.0)
                    throw new TPCException("5");

                x = 0.5 * (x1[i] + x2[i]);
                xdist = x - last_x;
                if (last_x > 0.0 && xdist <= 0.0)
                    throw new TPCException("6");

                if (x < 0)
                {
                    if (e != null)
                        e[i] = value;
                    if (ie != null)
                        ie[i] = integral;
                    if (iie != null)
                        iie[i] = integral2;
                    continue;
                }

                // slope between x[i-1] and x[i]
                s = (y[i] - last_y) / xdist;
                // If there is a big gap in the
                // beginning, it is eliminated
                if (i == 0)
                {
                    if (x1[0] > x2[0] - x1[0])
                    {
                        last_x2 = last_x = x1[0];
                    }
                }

                // gap
                integral += (x1[i] - last_x2) * (last_y +
                    s * ((last_x2 + x1[i]) / 2.0 - last_x));
                integral += frame_len * y[i];
                integral2 += (x2[i] - last_x2) *
                    (integral + last_integral) * 0.5;

                if (e != null && i > 0)
                {
                    // value at previous frame end
                    value = last_y + s * (last_x2 - last_x);
                    e[i - 1] = value;
                }
                if (ie != null)
                    ie[i] = integral;
                if (iie != null)
                    iie[i] = integral2;
                last_x = x;
                last_x2 = x2[i];
                last_y = y[i];
                last_integral = integral;
            }

            if (e != null)
            {
                // Value for the last frame
                value = last_y + s * (last_x2 - last_x);
                e[i - 1] = value;
            }
        }

        /// <summary>
        /// Interpolate and integrate TAC to PET frames.
        /// Both original and new interpolated data represent the actual values
        /// at specified time points, not from framed data.
        /// Then output can be integrated dot-by-dot.
        /// </summary>
        /// <remarks>
        /// PRE:
        /// Original data and new x values must be sorted by ascending x.
        /// Subsequent x values can have equal values, enabling the use of step functions.
        /// The OutputX2values must always be higher or equal than OutputX1values.
        /// Frames must not overlap. Negative x (time) values  can be processed.
        /// The  output data must've been allocated.
        /// POST:
        /// If output y is null, the first and second integral values are not calculated.
        /// If necessary, the data is extrapolated assuming that 
        ///     1) y[inf]=y[nr-1] and 
        ///     2) if x[0]>0 and y[0]>0, an imaginary line is drawn from (0,0) to (x[0],y[0]). 
        /// </remarks>
        /// <param name="x">Times of original data</param>
        /// <param name="y">Values of original data</param>
        /// <param name="pet_frame_start">PET frame start times</param>
        /// <param name="pet_frame_end">PET frame end times</param>
        /// <param name="interpolated_y">Interpolated mean values in PET frames</param>
        /// <param name="integral1st">Integral at frame mid time</param>
        /// <param name="integral2nd">2nd integral at frame mid time</param>
        public static void InterpolateToPETFrames(double[] x, double[] y,
            double[] pet_frame_start, double[] pet_frame_end,
            out double[] interpolated_y, out double[] integral1st, out double[] integral2nd)
        {
            interpolated_y = new double[pet_frame_start.Length];
            integral1st = new double[pet_frame_start.Length];
            integral2nd = new double[pet_frame_start.Length];

            // The  output data must've been allocated.
            if (interpolated_y == null)
                throw new TPCException("The  output data must've been allocated.");

            // Check that PET input data is not totally outside the original input data
            if (pet_frame_end[pet_frame_start.Length - 1] <= x[0] || pet_frame_start[0] >= x[x.Length - 1])
                throw new TPCException("PET times are totally outside the original input data");

            // Variables in the function:
            int indexOrig = 0; // the index of original data curve
            int indexNew; // the index of new data curve
            double ox1;//??
            double ox2;//??
            double oy1; // y value of original data at last index on-1 (at ox1)
            double oy2; // y value of original data at current index on (at ox2)
            double oi1 = 0.0; // integral of original data at last index on-1
            double oi2 = 0.0; // integral of original data at current index on
            double oii1 = 0.0; // 2nd integral of original data at last index on-1
            double oii2 = 0.0; // 2nd integral of original data at current index on
            double ny; // curve value to be interpolated at nx
            double nyi; // integral value at nx

            // One output frame at a time
            // Set output values to 0 as long
            // as input data has not started 
            for (indexNew = 0; indexNew < pet_frame_start.Length; indexNew++)
            {
                if (pet_frame_end[indexNew] <= x[0] && pet_frame_end[indexNew] <= 0.0)
                {
                    interpolated_y[indexNew] = 0.0;
                    integral1st[indexNew] = 0.0;
                    integral2nd[indexNew] = 0.0;
                }
                else
                {
                    break /*for*/;
                }
            }

            if (x[0] > 0.0)
            {
                ox1 = oy1 = ox2 = oy2 = 0.0;
            }
            else
            {
                ox1 = ox2 = x[0];
                oy1 = oy2 = y[0];
            }

            // Now there must be some input data inside the current time frame
            
            for (; indexNew < pet_frame_start.Length; indexNew++)
            {
                // Calculate and check frame length 
                double frameLength = pet_frame_end[indexNew] - pet_frame_start[indexNew];

                if (frameLength < 0.0)
                    throw new TPCException("The frame length was negative.");

                if (indexNew > 0 && pet_frame_start[indexNew] < pet_frame_end[indexNew - 1])
                    throw new TPCException("The output frames overlap");

                // frame start time, where ny is to be calculated
                double nx_start = pet_frame_start[indexNew];

                // go through original data until frame start is reached
                
                while (indexOrig < x.Length && x[indexOrig] <= nx_start) // integrate original data
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    double oLength = ox2 - ox1;

                    if (oLength > 0.0)
                    {
                        oi2 = oi1 + 0.5 * (oy2 + oy1) * (ox2 - ox1);
                        oii2 = oii1 + 0.5 * (oi2 + oi1) * (ox2 - ox1);
                    }
                    else
                    {
                        oi2 = oi1;
                        oii2 = oii1;
                    }
                    ox1 = ox2;
                    oy1 = oy2;
                    oi1 = oi2;
                    oii1 = oii2;
                    indexOrig++;
                }

                //modified
                if (indexOrig == 0 && x[0] > pet_frame_start[0])
                {
                    ox1 = 0;
                    oy1 = 0;
                }
                else
                {
                    ox1 = x[indexOrig - 1];
                    oy1 = y[indexOrig - 1];
                }
                //Original value;
                //ox1 = x[indexOrig - 1];
                //oy1 = y[indexOrig - 1];
                

                if (indexOrig < x.Length)
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    ny = oy1 + (nx_start - ox1) * (oy2 - oy1) / (ox2 - ox1);
                }
                else
                {
                    ox2 = nx_start;
                    oy2 = y[indexOrig - 1];
                    ny = oy1;
                }

                // integral of new data at frame start time
                double ni1 = oi1 + 0.5 * (oy1 + ny) * (nx_start - ox1);

                // frame mid time, where ny is to be calculated
                double nx_mid = 0.5 * (pet_frame_start[indexNew] + pet_frame_end[indexNew]);

                // go through original data until frame mid is reached
                while (indexOrig < x.Length && x[indexOrig] <= nx_mid) // integrate original data
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    double oLength = ox2 - ox1;

                    if (oLength > 0.0)
                    {
                        oi2 = oi1 + 0.5 * (oy2 + oy1) * (ox2 - ox1);
                        oii2 = oii1 + 0.5 * (oi2 + oi1) * (ox2 - ox1);
                    }
                    else
                    {
                        oi2 = oi1;
                        oii2 = oii1;
                    }

                    ox1 = ox2;
                    oy1 = oy2;
                    oi1 = oi2;
                    oii1 = oii2;
                    indexOrig++;
                }

                ox1 = x[indexOrig - 1];
                oy1 = y[indexOrig - 1];

                if (indexOrig < x.Length)
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    ny = oy1 + (nx_mid - ox1) * (oy2 - oy1) / (ox2 - ox1);
                }
                else
                {
                    ox2 = nx_mid;
                    oy2 = y[indexOrig - 1];
                    ny = oy1;
                }

                nyi = oi1 + 0.5 * (oy1 + ny) * (nx_mid - ox1);
                if (integral1st != null)
                    integral1st[indexNew] = nyi;

                if (integral2nd != null)
                    integral2nd[indexNew] = oii1 + 0.5 * (oi1 + nyi) * (nx_mid - ox1);

                // frame end time, where ny is to be calculated
                double nx_end = pet_frame_end[indexNew];

                // go through original data until frame end is reached
                while (indexOrig < x.Length && x[indexOrig] <= nx_end) // integrate original data 
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    double oLength = ox2 - ox1;

                    if (oLength > 0.0)
                    {
                        oi2 = oi1 + 0.5 * (oy2 + oy1) * (ox2 - ox1);
                        oii2 = oii1 + 0.5 * (oi2 + oi1) * (ox2 - ox1);
                    }
                    else
                    {
                        oi2 = oi1;
                        oii2 = oii1;
                    }

                    ox1 = ox2;
                    oy1 = oy2;
                    oi1 = oi2;
                    oii1 = oii2;
                    indexOrig++;
                }

                ox1 = x[indexOrig - 1];
                oy1 = y[indexOrig - 1];

                if (indexOrig < x.Length)
                {
                    ox2 = x[indexOrig];
                    oy2 = y[indexOrig];
                    ny = oy1 + (nx_end - ox1) * (oy2 - oy1) / (ox2 - ox1);
                }
                else
                {
                    ox2 = nx_end;
                    oy2 = y[indexOrig - 1];
                    ny = oy1;
                }

                // integral of new data at frame end time
                double ni2 = oi1 + 0.5 * (oy1 + ny) * (nx_end - ox1);

                if (interpolated_y != null)
                {
                    if (frameLength > 0.0)
                        interpolated_y[indexNew] = (ni2 - ni1) / frameLength;
                    else
                        interpolated_y[indexNew] = ny;
                }
            } // for

        } // void 
    }
}
