﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// The algoritm is based on:
    /// Numerical Recipes, Third Edition, 2007
    /// 15.2 Fitting Data to a Straight Line (page 780)
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class LinearRegression
    {
        /// <summary>
        /// Estimates parameters for interpolation.
        /// y = k x + b
        /// </summary>
        /// <param name="x">data x-coodrinates</param>
        /// <param name="y">data y-coodrinates</param>
        /// <returns>parameters {k,b}</returns>
        public double[] ApproximateParameters(double[] x, double[] y) {
            if (x.Length != y.Length)
                throw new TPCInvalidArgumentsException("x and y length must be the same");
            int ndata = x.Length;

            int i;
            double sx = 0.0, sy = 0.0, st2 = 0.0, t, sxoss;
            double k = 0;
            for (i = 0; i < ndata; i++)
            {
                sx += x[i];
                sy += y[i];
            }
            sxoss = sx / ndata;
            for (i = 0; i < ndata; i++)
            {
                t = x[i] - sxoss;
                st2 += t * t;
                k += t * y[i];
            }
            k = k / st2;
            double b = (sy - sx * k) / ndata;
            return new double[]{k, b};
        }

        /// <summary>
        /// Interpolates at data points.
        /// </summary>
        /// <param name="x">data points</param>
        /// <param name="p">parameters {k,a}</param>
        /// <returns>y-values evaluated at locations x</returns>
        public double[] Interpolate(double[] x, params double[] p)
        {
            if (p.Length != 2) throw new TPCInvalidArgumentsException("Parameters for line must be {k,a}");
            double[] y = new double[x.Length];
            for (int i = 0; i < x.Length; i++) {
                y[i] = p[0] * x[i] + p[1];
            }
            return y;
        }
        /// <summary>
        /// Calculate slope and intercept of a line and Pearson's correlation coefficient
        /// </summary>
        /// <param name="x">data x-coordinates</param>
        /// <param name="y">data y-coordinates</param>
        /// <returns>parameters {k, kSD, b, bSD, r, ySD}</returns>
        public double[] pearson(double[] x, double[] y)
        {
            double e, f, g, meanx, meany, kk, bb, ke, be, rr, sumsdcy = 0.0;
            double sumx = 0.0, sumy = 0.0, sumsx = 0.0, sumsdx = 0.0, sumsdy = 0.0, sumdxdy = 0.0;
            double sumxy = 0.0, sumsy = 0.0;
            double res_k, res_b, res_r, res_kSD, res_bSD, res_ySD;

            //Check that there is some data
            if (x == null || y == null || x.Length < 2) throw new TPCException("There must be at least 2 values in x and y values");

            if (x.Length == 2) {
                f = x[1] - x[0];
                if (Math.Abs(f) < 1.0E-50) throw new TPCException("Error using 2 datapoints");
                res_k = (y[1] - y[0]) / f;
                res_b = y[0] - res_k * x[0];
                res_kSD = res_bSD = res_ySD = 0;
                res_r = 1;
                return new double[] { res_k, res_kSD, res_b, res_bSD, res_r, res_ySD };
            }
            //Calculate (x,y) sums and means
            for (int i = 0; i < x.Length; i++)
            {
                sumx += x[i];
                sumy += y[i];
                sumsx += x[i] * x[i];
                sumsy += y[i] * y[i];
                sumxy += x[i] * y[i];
            }
            meanx = sumx / (double)x.Length;
            meany = sumy / (double)x.Length;
            //and then based on means
            for (int i = 0; i < x.Length; i++)
            {
                f = x[i] - meanx;
                sumsdx += f * f;
                g = y[i] - meany;
                sumsdy += g * g;
                sumdxdy += f * g;
            }
            if (sumsdx < 1.0e-50 || sumsdy < 1.0e-50) throw new TPCException("pearson failed");
            //Regression coefficient
            kk = sumdxdy / sumsdx;
            res_k = kk;
            //Intercept with y axis
            bb = (sumsdx * sumy - sumx * sumdxdy) / ((double)x.Length * sumsdx);
            res_b = bb;
            //Errors
            for (int i = 0; i < x.Length; i++)
            {
                f = kk * x[i] + bb - y[i];
                sumsdcy += f * f;
            }
            //Deviation of y-values
            if (sumsdcy <= 1.0e-12) e = 0.0;
            else e = Math.Sqrt(sumsdcy / (double)(x.Length - 2));
            res_ySD = e;
            //SD of slope and intercept
            ke = e / Math.Sqrt(sumsdx);
            be = e / Math.Sqrt((double)x.Length - sumx * sumx / sumsx);
            res_kSD = ke;
            res_bSD = be;
            be = Math.Sqrt(sumsdcy / (double)(x.Length - 2)) / Math.Sqrt((double)x.Length - (sumx * sumx) / sumsx);
            //Pearson's correlation coefficient
            rr = (sumxy - ((sumx * sumy) / (double)x.Length)) / Math.Sqrt((sumsx - sumx * sumx / (double)x.Length) * (sumsy - sumy * sumy / (double)x.Length));
            //Correct for small sample size
            if (x.Length > 4) rr *= 1.0 + (1.0 - rr * rr) / (double)(2 * (x.Length - 4));
            res_r = rr;
            return new double[] { res_k, res_kSD, res_b, res_bSD, res_r, res_ySD };

        } //pearson

        /// <summary>
        /// Calculate slope and intercept of a line and Pearson's correlation coefficient.
        /// Data points may contain NA's.
        /// </summary>
        /// <param name="x">data x-coordinates</param>
        /// <param name="y">data y-coordinates</param>
        /// <returns>parameters {k, kSD, b, bSD, r, ySD}</returns>
        public double[] pearson3(double[] x, double[] y)
        {
            List<double> new_x = new List<double>();
            List<double> new_y = new List<double>();
            
            //Copy data to new arrays
            int j = 0;
            for (int i = 0; i < x.Length; i++)
            {
                if (x[i] != double.MinValue && y[i] != double.MinValue) {
                    new_x.Add(x[i]);
                    new_y.Add(y[i]);
                    j++;
                }
            }
            double[] nx = new_x.ToArray();
            double[] ny = new_y.ToArray();
            //Use pearson()
            return pearson(nx, ny);

        }
    }
}

