﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Abstract base class for PET Centre models.
    /// Each model is generated using some reference data.
    /// Simulated values are generally available (without interpolation)
    /// only at the same points as in the reference data.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public abstract class Model
    {
        /// <summary>
        /// Number of samples simulated by this model
        /// </summary>
        public abstract int Samples
        {
            get;
        }

        /// <summary>
        /// Times of the model
        /// </summary>
        public Vector model_times;
        

        /// <summary>
        /// Default constructor
        /// </summary>
        public Model() { }

        /// <summary>
        /// Simulate data
        /// </summary>
        /// <param name="par">Parameters used to simulate data</param>
        /// <returns>Vector of simulated values</returns>
        public abstract Vector Simulate(params double[] par);

        /// <summary>
        /// Model Info
        /// </summary>
        public ModelInfo Info;

        /// <summary>
        /// Creates Model with time values and reference( input function ) data.
        /// </summary>
        /// <param name="times">Times</param>
        /// <param name="Reference">Reference values(input function) Not needed in all Models</param>
        public abstract void Create(Vector times, params Vector[] Reference);

        /// <summary>
        /// Creates Model with given parameters
        /// </summary>
        /// <param name="ModelType">type of the Model</param>
        /// <param name="times">time values</param>
        /// <param name="inputFunctions">input functions at the time values</param>
        /// <returns>Instance of created model</returns>
        public static Model CreateModel(Type ModelType, Vector times, params Vector[] inputFunctions)
        {
            Model model = null;
            // We try to create instance of given object:
            try
            {
                model = (Model)System.Activator.CreateInstance(ModelType);
            }
            catch (Exception e)
            {
                throw new ModelException(e.Message);
            }
            model.Create(times, inputFunctions);
            return model;
        }


        /// <summary>
        /// Returns info of Model
        /// </summary>
        /// <returns></returns>
        protected abstract ModelInfo GetInfo();
      

        /// <summary>
        /// Gets information about all models
        /// </summary>
        /// <returns>Array of ModelInfo</returns>
        public static ModelInfo[] GetInfoOfAllModels()
        {
            System.Reflection.Assembly assembly = System.Reflection.Assembly.GetAssembly(typeof(Model));
            Type[] all_types = assembly.GetTypes();

            System.Collections.Generic.List<ModelInfo> infos =
                new System.Collections.Generic.List<ModelInfo>();

            foreach (Type t in all_types)
            {
                if (t.IsSubclassOf(typeof(Model)))
                {
                    try
                    {
                        Model model = (Model)System.Activator.CreateInstance(t);
                        infos.Add( model.Info );
                    }
                    catch (Exception e)
                    {
                        throw new ModelException(e.Message);
                    }                    
                }
            }
            return infos.ToArray();
        }

        /// <summary>
        /// Event handler for I/O progress notifications.
        /// </summary>
        /// <param name="sender">object that sent the event</param>
        /// <param name="e">event arguments</param>
        public delegate void FittingProcessEventHandler(object sender, OptimizationProgressEventArgs e);

        /// <summary>
        /// Calculates differences between sample times and divides them by 2
        /// </summary>
        /// <param name="t">Sample times</param>
        /// <returns>Vector of time differences</returns>
        protected Vector HalfTimeDifs(Vector t)
        {
            Vector difs = new Vector(t.Length);

            for (int i = 1; i < t.Length; i++)
            {
                difs[i] = (t[i] - t[i - 1]) * 0.5;
            }

            // If first sample time is positive, assume starting point is 0.0;
            // if it's negative, start is set at first sample time
            difs[0] = (t[0] > 0) ? (t[0] * 0.5) : 0.0;

            return difs;
        }

        /// <summary>
        /// Calculates the reference integral using trapezium rule
        /// </summary>
        /// <param name="refvals">reference values of curve</param>
        /// <param name="times">corresponding time delta values divided by 2</param>
        /// <returns>Vector of cumulative integral values</returns>
        protected Vector CalcRefIntegral(Vector refvals, Vector times)
        {
            int samples = refvals.Length;

            if (times.Length != samples) throw new ModelException("Frame count and data length don't match");

            Vector refint = new Vector(samples);

            // Cumulative integral value
            double cri;

            // Calculate the integral
            cri = times[0] * refvals[0];
            refint[0] = cri;

            for (int i = 1; i < samples; i++)
            {
                cri += (refvals[i] + refvals[i - 1]) * times[i];
                refint[i] = cri;
            }

            return refint;
        }
    }
}
