﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using NUnit.Framework;

namespace TPClib.Model
{
    /// <summary>
    /// Unit testing for TPClib.Model.Cluster
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Cluster
    {
        /// <summary>
        /// 1.0 Every point is in some cluster
        /// </summary>
        [Test]
        public void Test1_0()
        {
            // how many clusters we make
            int testClusters = 3;
            // how many points there are in cluster
            int points = 10;
            //
            int total = testClusters*points;
            //
            Vector[] table = new Vector[total];

            Random r = new Random();

            // Creates easy clustering
            int i = 0;
            for (int cluster = 0; cluster < testClusters; cluster++)
            {
                for (int p = 0; p < points; p++)
                {
                    double x = cluster + 0.01 * r.NextDouble();
                    double y = cluster + 0.01 * r.NextDouble();
                    table[i] = new double[] { x, y };
                    Console.WriteLine(table[i]);
                    i++;
                }
            }

            Cluster[] clr = Cluster.MakeClusters(table, testClusters);

            Assert.AreEqual(testClusters, clr.Length);

            // Assert that every point is in one and only one cluster.
            foreach( Vector ve in table )
            {
                int count = 0;

                foreach( Cluster c in clr ) {
                    if (c.points.Contains(ve))
                        count++;
                }

                Assert.AreEqual(1, count);
            }

            foreach (Cluster c in clr)
            {
                Console.WriteLine("cluster");

                foreach( Vector v in c.points)
                    Console.WriteLine(v);
            }

        }

        /// <summary>
        /// Checks that all points are located in the cluster with the nearest mean
        /// </summary>
        /// <param name="clusters">Clusters to check</param>
        /// <param name="metric">Used metric</param>
        private void CheckValidity(Cluster[] clusters, RealFunction metric)
        {
            foreach (Cluster c in clusters)
            {
                Console.WriteLine("cluster length: " + c.points.Count);
                Console.WriteLine("cluster mean: " + c.mean);

                Console.WriteLine("Point\tDistances");
                foreach (Vector v in c.points)
                {
                    Console.Write(v + "\t");
                    foreach (Cluster k in clusters)
                    {
                        Assert.IsTrue(metric(v - c.mean) <= metric(v - k.mean));
                        Console.Write(metric(v - k.mean).ToString("F2") + "\t");
                    }
                    Console.WriteLine();
                }
            }
        }

        /// <summary>
        /// 1.2 Divide even distribution of points on a line to clusters
        /// and check that each point is in the cluster with the closest mean.
        /// </summary>
        [Test]
        public void Test1_2_Regular_line()
        {
            Vector[] table = new Vector[100];

            for (int i = 0; i < 100; i++)
            {
                    table[i] = new double[] { i };
            }

            Cluster[] clusters = Cluster.MakeClusters(table, 5);

            CheckValidity(clusters, Vector.Norm);
        }

        /// <summary>
        /// 1.3 Divide even distribution of points on a plane to clusters
        /// and check that each point is in the cluster with the closest mean.
        /// </summary>
        [Test]
        public void Test1_3_Regular_plane()
        {
            Vector[] table = new Vector[100];

            for (int i = 0; i < 10; i++)
            {
                for (int j = 0; j < 10; j++)
                {
                    table[10*i+j] = new double[] { i, j };
                }
            }

            Cluster[] clusters = Cluster.MakeClusters(table, 5);

            // All points are located in the cluster with the nearest mean
            CheckValidity(clusters, Vector.Norm);
        }

        /// <summary>
        /// 1.4 Clustering using Manhattan distance
        /// </summary>
        [Test]
        public void Test1_4()
        {
            Vector[] table = new Vector[100];

            for (int i = 0; i < 10; i++)
            {
                for (int j = 0; j < 10; j++)
                {
                    table[10 * i + j] = new double[] { i, j };
                }
            }

            RealFunction Manhattan = delegate(Vector p) {
                double x = p[0], y = p[1];
                return Math.Abs(x) + Math.Abs(y);
            };

            Cluster[] clusters = Cluster.MakeClusters(table, 5, Manhattan);
            // All points are located in the cluster with the nearest mean
            CheckValidity(clusters, Manhattan);

        }
    }
}