﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using TPClib.Curve;

namespace TPClib.Model
{
    /// <summary>
    /// Test class for Fit
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Fit
    {
        private const string PATH = @"p:\data\TPClib_Model\";
        private const double TOLERANCE = 0.01;
        private const double MIN_TOLERANCE = 1e-5;

        private double[] AvgTimes(FrameTimeCell[] frames)
        {
            double[] times = new double[frames.Length];
            for (int i = 0; i < times.Length; i++)
            {
                times[i] = (frames[i].start_time + frames[i].end_time) / 2.0;
            }
            return times;
        }

        /// <summary>
        /// Maximum absolut value of elements in a-b
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Maximum difference between elements</returns>
        private double MaxDiff(Vector a, Vector b)
        {
            Vector c = a - b;
            return Math.Max(Vector.Max(c), Math.Abs(Vector.Min(c)));
        }

        /// <summary>
        /// 1 Find optimal model parameters and simulate curves with RTCM model.
        /// Compare to original data (generated with PET Centre C library functions)
        /// </summary>
        [Test]
        public void Test1()
        {
            // Input and output files
            string ref_fname = PATH + @"rtcm.dft";
            string data_fname = PATH + @"test_rtcm.dft";
            string result_fname = PATH + @"tpclib_simulation_rtcm.dft";

            // Initial parameters and correct model parameters
            double[] initial = { 1.0, 0.05, 0.1, 20.0 };
            double[] params1 = { 1.390909, 0.489, 0.303, 3.272138 };

            // Prepare the files and read input file
            DFTRegionalCurveFile refdft = new DFTRegionalCurveFile(ref_fname);
            DFTRegionalCurveFile datadft = new DFTRegionalCurveFile(data_fname);
            DFTRegionalCurveFile resdft = new DFTRegionalCurveFile(result_fname);
            datadft.ReadFile();
            refdft.ReadFile();

            // Read time data, reference data and two data columns from input file
            // Time values must be converted to minutes and time frame midpoints
            // are used for simulation
            Vector times = (Vector)(AvgTimes(refdft.Curves.GetTimeCells())) / 60.0;
            Vector refdata = (double[])(refdft.Curves.GetColumn(0));
            Vector data1 = (double[])(datadft.Curves.GetColumn(0));

            // Write time values and reference data to result file
            resdft.TimeConversion(TimeUnit.minute);
            resdft.Curves.SetTimeCells(refdft.Curves.GetTimeCells());
            resdft.Curves.curveheader = refdft.Curves.curveheader;
            resdft.Curves.AddColumn(refdata);

            // RTCM model
            Model m = new RTCMModel(times, refdata);

            // Fit the model to data columns
            Fit fit1 = new Fit(m, data1);

            // Real minimum values
            double min1 = fit1.SumOfSq(params1);

            // Find optimal parameters for fits
            Optimization opt1 = new PowellBrent(fit1.SumOfSq, initial);

            // Run optimization until minimum found
            opt1.SetTolerance(min1 + MIN_TOLERANCE);
            //opt1.Iterate(100);
            opt1.IterateUntilStop();
            Vector optmin1 = opt1.GetMinimum();

            // Simulate data with the found parameters
            Vector sim1 = m.Simulate(optmin1);

            // Add simulated data to result file
            resdft.Curves.AddColumn(sim1);

            // Write file
            resdft.WriteFile();

            // Check the results:
            // parameters must be within error limit from the real ones
            Assert.AreEqual(0.0, MaxDiff(params1, optmin1), TOLERANCE);
        }

        /// <summary>
        /// 2 Find optimal model parameters and simulate curves with TRTM model.
        /// Compare to original data (generated with PET Centre C library functions)
        /// </summary>
        [Test]
        public void Test2()
        {
            // Parameter limits
            Vector par_max = new Vector(10.0, 10.0, 60.0);
            Vector par_min = new Vector(0.001, 0.000001, 0.0);

            // Input and output files
            string ref_fname = PATH + @"trtm.dft";
            string data_fname = PATH + @"test_trtm.dft";
            string res_fname = PATH + @"tpclib_simulation_trtm.dft";

            // Correct model parameters
            double[] params1 = { 0.8, 0.1125, 0.07 };

            // Prepare files and read input files
            DFTRegionalCurveFile ref_dft = new DFTRegionalCurveFile(ref_fname);
            DFTRegionalCurveFile data_dft = new DFTRegionalCurveFile(data_fname);
            DFTRegionalCurveFile res_dft = new DFTRegionalCurveFile(res_fname);
            ref_dft.ReadFile();
            data_dft.ReadFile();

            // Read time data, reference data and two data columns from input file
            // Time values must be converted to minutes and time frame midpoints
            // are used for simulation
            Vector times = (Vector)(AvgTimes(ref_dft.Curves.GetTimeCells())) / 60.0;
            Vector refdata = (double[])(ref_dft.Curves.GetColumn(0));
            Vector data1 = (double[])(data_dft.Curves.GetColumn(0));

            // Write time values and reference data to result file
            res_dft.TimeConversion(TimeUnit.minute);
            res_dft.Curves.SetTimeCells(ref_dft.Curves.GetTimeCells());
            res_dft.Curves.AddColumn(refdata);

            // TRTM model
            Model m = new TRTMModel(times, refdata);

            // Fit the model to data columns
            Fit fit1 = new Fit(m, data1);

            // Real minimum values
            double min1 = fit1.SumOfSq(params1);

            // Find optimal parameters for fit
            Optimization opt1 = new ITGO(fit1.SumOfSq, 5, 100, par_max, par_min);

            // Run optimization until minimum found
            opt1.SetTolerance(min1 + MIN_TOLERANCE);
            opt1.IterateUntilStop();
            Vector optmin1 = opt1.GetMinimum();

            // Simulate data with the found parameters
            Vector sim1 = m.Simulate(opt1.GetMinimum());

            // Add simulated data to result file
            res_dft.Curves.AddColumn(sim1);

            // Write file
            res_dft.WriteFile();

            // Check the results:
            // parameters must be within error limit from the real ones
            Assert.AreEqual(0.0, MaxDiff(params1, optmin1), TOLERANCE);
        }

        /// <summary>
        /// 3 Find optimal model parameters and simulate curves with SRTM model.
        /// Compare to original data (generated with PET Centre C library functions)
        /// </summary>
        [Test]
        public void Test3()
        {
            // Parameter limits
            Vector par_max = new Vector(10.0, 10.0, 60.0);
            Vector par_min = new Vector(0.001, 0.000001, 0.0);

            // Input and output files
            string ref_fname = PATH + @"srtm.dft";
            string data_fname = PATH + @"test_srtm.dft";
            string res_fname = PATH + @"tpclib_simulation_srtm.dft";

            // Correct model parameters
            double[] params1 = { 1.390909, 0.489, 3.272138 };

            // Prepare files and read input files
            DFTRegionalCurveFile ref_dft = new DFTRegionalCurveFile(ref_fname);
            DFTRegionalCurveFile data_dft = new DFTRegionalCurveFile(data_fname);
            DFTRegionalCurveFile res_dft = new DFTRegionalCurveFile(res_fname);
            ref_dft.ReadFile();
            data_dft.ReadFile();

            // Read time data, reference data and two data columns from input file
            // Time values must be converted to minutes and time frame midpoints
            // are used for simulation
            Vector times = (Vector)(AvgTimes(ref_dft.Curves.GetTimeCells())) / 60.0;
            Vector refdata = (double[])(ref_dft.Curves.GetColumn(0));
            Vector data1 = (double[])(data_dft.Curves.GetColumn(0));

            // Write time values and reference data to result file
            res_dft.TimeConversion(TimeUnit.minute);
            res_dft.Curves.SetTimeCells(ref_dft.Curves.GetTimeCells());
            res_dft.Curves.AddColumn(refdata);

            // SRTM model
            Model m = new SRTMModel(times, refdata);

            // Fit the model to data columns
            Fit fit1 = new Fit(m, data1);

            // Real minimum values
            double min1 = fit1.SumOfSq(params1);

            // Find optimal parameters for fit
            Optimization opt1 = new ITGO(fit1.SumOfSq, 5, 100, par_max, par_min);

            // Run optimization until minimum found
            opt1.SetTolerance(min1 + MIN_TOLERANCE);
            opt1.IterateUntilStop();
            Vector optmin1 = opt1.GetMinimum();

            // Simulate data with the found parameters
            Vector sim1 = m.Simulate(opt1.GetMinimum());

            // Add simulated data to result file
            res_dft.Curves.AddColumn(sim1);

            // Write file
            res_dft.WriteFile();

            // Check the results:
            // parameters must be within error limit from the real ones
            Assert.AreEqual(0.0, MaxDiff(params1, optmin1), TOLERANCE);
        }
    }
}
