﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.Model
{
    /// <summary>
    /// The Vector class How-To Documentation.
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_FitLMA
    {
        private Random rand = new Random();

        private double GaussFunction(double x, Vector a, Vector dyda)
        {
            if ((a.Length % 3) != 0)
                throw new Exception("The Number of components don't divide three.");

            if (dyda.Length != a.Length)
                throw new Exception("");

            double y = 0.0;

            // Mathematical sum from 0 to K
            for (int i = 0; i < a.Length - 1; i += 3)
            {
                double arg = (x - a[i + 1]) / a[i + 2];
                double ex = Math.Exp(-arg * arg);
                y += a[i] * ex;

                // Differentials
                double fac = a[i] * ex * 2.0 * arg;
                dyda[i] = ex;                   // B_k
                dyda[i + 1] = fac / a[i + 2];         // E_k
                dyda[i + 2] = fac * arg / a[i + 2];     // G_k
            }
            return y;
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_1_Fgauss()
        {
            double[] x = new double[] { 0.0, 0.1, 0.2 };
            double[] y = new double[] { 1.0, 1.1, 1.2 };
            double[] init = new double[] { 1, 2, 3 };

            FitLMA m = new FitLMA(x, y, init, GaussFunction);
            m.Iterate();

            Console.WriteLine(m.AtMinimum());
            Console.WriteLine("Minimum parameters: " + m.GetMinimum());
        }

        /// <summary>        
        /// f(x) = A * exp(-B*x)
        /// </summary>
        [Test]
        public void Test1_2_PetFunction()
        {
            // analytical partial derivates
            PartialDerivate fu1 = delegate(double t, Vector a, Vector dyda)
            {
                dyda[0] = Math.Exp(-a[1] * t);
                dyda[1] = -a[0] * Math.Exp(-a[1] * t) * t;
                return a[0] * Math.Exp(-a[1] * t);
            };

            // only target function
            ParameterFunction fu2 = delegate(double t, Vector a)
            {
                return a[0] * Math.Exp(-a[1] * t);
            };

            // an initial guess
            Vector initialParams = new double[] { 1, 0.08 };
            // the data
            double[] x = { 0, 2, 5, 10, 15, 20, 30, 40 };
            double[] y = { 1, 0.8, 0.4, 0.3, 0.27, 0.22, 0.15, 0.05 };
            Optimization m = new FitLMA(x, y, initialParams, fu2);
            m.IterationNotify += delegate(object sender, IterationEventArgs e)
            {
                Console.WriteLine("Iter " + m.Iterations + ", MeritFunction := \t" + m.AtMinimum());
            };
            m.Iterate(50);

            Assert.AreEqual(1.00, initialParams[0], 0.001);
            Assert.AreEqual(0.08, initialParams[1], 0.001);

            Vector result = m.GetMinimum();
            Assert.AreEqual(0.903, result[0], 0.04);
            Assert.AreEqual(0.075, result[1], 0.04);

            Console.WriteLine("result: " + result);
        }

        /// <summary>        
        /// Periodic f(x) = -A Sin[B x]
        /// Right answer:
        ///     A = 2.5;
        ///     B = 0.9;
        /// </summary>
        [Test]
        public void Test1_3_Sin()
        {
            // the data
            double[] datax = { 0.0, 0.5, 1.0, 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0 };
            double[] datay =   {0.0, 1.08741, 1.95832, 2.43931, 2.43462, 1.94518, 1.06845, 
                                   -0.0210181, -1.1063, -1.97131, -2.44383};

            // the fitted function
            ParameterFunction yFit = delegate(double x, Vector a)
            {
                return a[0] * Math.Sin(a[1] * x);
            };

            PartialDerivate yFitAnalytic = delegate(double x, Vector a, Vector dyda)
            {
                double A = a[0], B = a[1];

                // differentials
                // 1. parameter Sin[B x]
                dyda[0] = Math.Sin(B * x);
                // 2. parameter A x Cos[B x]   
                dyda[1] = A * x * Math.Cos(B * x);

                // value with current params
                return yFit(x, a);
            };

            double min = Double.MaxValue;
            Vector vmin = new Vector(2);
            double[] init = new double[] { 2.0, 0.4 };
            double radius = 1.0;
            Random rand = new Random();

            for (int i = 0; i < 200; i++)
            {
                // an initial guess
                double x = init[0] + radius - 2 * radius * rand.NextDouble();
                double y = init[0] + radius - 2 * radius * rand.NextDouble();
                Vector guess = new double[] { x, y };
                Optimization sut = new FitLMA(datax, datay, guess, yFitAnalytic);
                sut.Iterate(2000);

                if (sut.AtMinimum() < min)
                {
                    min = sut.AtMinimum();
                    vmin = sut.GetMinimum();
                    Console.WriteLine("iter " + i + " found minimum: " + min + " par " + vmin);
                }
            }

            Console.WriteLine("Minimum target function: " + min);
            Console.WriteLine("Minimum parameters:      " + vmin);
            Assert.AreEqual(0.0, min, 0.001);
        }

        /// <summary>        
        /// Polynom f1[x_] := A x^3 - x^2 + B x;
        /// </summary>
        [Test]
        public void Test1_4a_Polynom()
        {
            Console.WriteLine("Right answer: " + "0.4, 0.5");

            // the data
            double[] datax = { 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0,
                         1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0,};
            double[] datay =   {0.0404, 0.0632, 0.0708, 0.0656, 0.05, 0.0264, -0.0028, -0.0352,
                               -0.0684, -0.1, -0.1276, -0.1488, -0.1612, -0.1624,
                               -0.15, -0.1216, -0.0748, -0.0072, 0.0836, 0.2};

            ParameterFunction target = delegate(double x, Vector p)
            {
                return p[0] * x * x * x - x * x + p[1] * x;
            };

            double min = Double.MaxValue;
            Vector vmin = new Vector(2);
            double[] init = new double[] { 0.4, 0.5 };
            double radius = 2.0;
            Random rand = new Random();

            for (int i = 0; i < 100; i++)
            {
                // an initial guess
                double x = init[0] + radius - 2 * radius * rand.NextDouble();
                double y = init[0] + radius - 2 * radius * rand.NextDouble();
                Vector guess = new double[] { x, y };
                Optimization m = new FitLMA(datax, datay, guess, target);
                m.Iterate(1000);

                if (m.AtMinimum() < min)
                {
                    min = m.AtMinimum();
                    vmin = m.GetMinimum();
                }
            }

            Console.WriteLine("Chi^2: \t" + min);
            Console.WriteLine("Minimum parameters: " + vmin);
            Assert.AreEqual(0.0, min, 0.1);
        }

        /// <summary>        
        /// Analytical Polynom f1[x_] := A x^3 - x^2 + B x;
        /// </summary>
        [Test]
        public void Test1_4b_Polynom()
        {
            Console.WriteLine("Right answer: " + "0.4, 0.5");

            // the data
            double[] datax = { 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0,
                         1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0,};
            double[] datay =   {0.0404, 0.0632, 0.0708, 0.0656, 0.05, 0.0264, -0.0028, -0.0352,
                               -0.0684, -0.1, -0.1276, -0.1488, -0.1612, -0.1624,
                               -0.15, -0.1216, -0.0748, -0.0072, 0.0836, 0.2};

            // Function is f1[x_] := A x^3 - x^2 + B x;
            PartialDerivate fu = delegate(double x, Vector a, Vector dyda)
            {
                // value with current params
                double y = a[0] * x * x * x - x * x + a[1] * x; ;
                // differentials
                dyda[0] = x * x * x;
                dyda[1] = x;
                return y;
            };

            double min = Double.MaxValue;
            Vector vmin = new Vector(2);
            double[] init = new double[] { 0.4, 0.5 };
            double radius = 2.0;
            Random rand = new Random();

            for (int i = 0; i < 100; i++)
            {
                // an initial guess
                double x = init[0] + radius - 2 * radius * rand.NextDouble();
                double y = init[0] + radius - 2 * radius * rand.NextDouble();
                Vector guess = new double[] { x, y };
                Optimization m = new FitLMA(datax, datay, guess, fu);
                m.Iterate(1000);

                if (m.AtMinimum() < min)
                {
                    min = m.AtMinimum();
                    vmin = m.GetMinimum();
                }
            }

            Console.WriteLine("Chi^2: \t" + min);
            Console.WriteLine("Minimum parameters: " + vmin);
            Assert.AreEqual(0.0, min, 0.01);
        }

        /// <summary>
        /// Try to fit hard non-linear pet-function.
        /// </summary>
        [Test]
        public void Test1_5_ComplexModel()
        {
            double[] datax = new double[] {0.001, 0.051, 0.101, 0.151, 0.201, 0.251, 0.301, 0.351, 0.401, 
            0.451, 0.501, 0.551, 0.601, 0.651, 0.701, 0.751, 0.801, 0.851, 0.901, 
            0.951, 1.001, 1.051, 1.101, 1.151, 1.201, 1.251, 1.301, 1.351, 1.401, 
            1.451, 1.501, 1.551, 1.601, 1.651, 1.701, 1.751, 1.801, 1.851, 1.901, 
            1.951, 2.001, 2.051, 2.101, 2.151, 2.201, 2.251, 2.301, 2.351, 2.401, 
            2.451, 2.501, 2.551, 2.601, 2.651, 2.701, 2.751, 2.801, 2.851, 2.901, 
            2.951, 3.001, 3.051, 3.101, 3.151, 3.201, 3.251, 3.301, 3.351, 3.401, 
            3.451, 3.501, 3.551, 3.601, 3.651, 3.701, 3.751, 3.801, 3.851, 3.901, 
            3.951, 4.001, 4.051, 4.101, 4.151, 4.201, 4.251, 4.301, 4.351, 4.401, 
            4.451, 4.501, 4.551, 4.601, 4.651, 4.701, 4.751, 4.801, 4.851, 4.901, 
            4.951, 5.001, 5.051, 5.101, 5.151, 5.201, 5.251, 5.301, 5.351, 5.401, 
            5.451, 5.501, 5.551, 5.601, 5.651, 5.701, 5.751, 5.801, 5.851, 5.901, 
            5.951, 6.001, 6.051, 6.101, 6.151, 6.201, 6.251, 6.301, 6.351, 6.401, 
            6.451, 6.501, 6.551, 6.601, 6.651, 6.701, 6.751, 6.801, 6.851, 6.901, 
            6.951, 7.001, 7.051, 7.101, 7.151, 7.201, 7.251, 7.301, 7.351, 7.401, 
            7.451, 7.501, 7.551, 7.601, 7.651, 7.701, 7.751, 7.801, 7.851, 7.901, 
            7.951, 8.001, 8.051, 8.101, 8.151, 8.201, 8.251, 8.301, 8.351, 8.401, 
            8.451, 8.501, 8.551, 8.601, 8.651, 8.701, 8.751, 8.801, 8.851, 8.901, 
            8.951, 9.001, 9.051, 9.101, 9.151, 9.201, 9.251, 9.301, 9.351, 9.401, 
            9.451, 9.501, 9.551, 9.601, 9.651, 9.701, 9.751, 9.801, 9.851, 9.901, 
            9.951};

            double[] datay = new double[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.00339605, 0.163442, 0.305506, 
            0.431186, 0.541946, 0.639126, 0.723955, 0.797556, 0.860959, 0.915107, 
            0.960865, 0.999022, 1.0303, 1.05537, 1.07483, 1.08923, 1.0991, 
            1.10488, 1.10701, 1.10587, 1.10181, 1.09517, 1.08624, 1.07528, 
            1.06255, 1.04826, 1.03263, 1.01583, 0.998043, 0.979414, 0.960084, 
            0.94018, 0.919816, 0.899096, 0.878112, 0.85695, 0.835684, 0.814383, 
            0.793108, 0.771912, 0.750844, 0.729948, 0.70926, 0.688815, 0.668642, 
            0.648766, 0.629209, 0.609991, 0.591126, 0.57263, 0.554512, 0.536782, 
            0.519446, 0.50251, 0.485977, 0.46985, 0.45413, 0.438815, 0.423906, 
            0.409399, 0.395292, 0.38158, 0.36826, 0.355326, 0.342773, 0.330595, 
            0.318786, 0.307338, 0.296246, 0.285502, 0.275098, 0.265028, 0.255284, 
            0.245857, 0.236741, 0.227928, 0.21941, 0.21118, 0.203229, 0.19555, 
            0.188136, 0.180979, 0.174072, 0.167408, 0.160979, 0.154778, 0.148798, 
            0.143034, 0.137477, 0.132121, 0.126961, 0.121989, 0.1172, 0.112588, 
            0.108147, 0.10387, 0.0997538, 0.0957916, 0.0919784, 0.0883092, 
            0.0847789, 0.0813829, 0.0781163, 0.0749747, 0.0719536, 0.0690488, 
            0.0662561, 0.0635715, 0.0609911, 0.0585112, 0.056128, 0.0538381, 
            0.0516379, 0.0495243, 0.047494, 0.045544, 0.0436711, 0.0418725, 
            0.0401455, 0.0384874, 0.0368954, 0.0353672, 0.0339003, 0.0324923,
            0.031141, 0.0298443, 0.0285999, 0.0274059, 0.0262604, 0.0251613, 
            0.0241071, 0.0230958, 0.0221258, 0.0211954, 0.0203032, 0.0194476, 
            0.0186272, 0.0178405, 0.0170862, 0.0163631, 0.0156699, 0.0150053, 
            0.0143684, 0.0137578, 0.0131726, 0.0126118, 0.0120744, 0.0115593, 
            0.0110658, 0.010593, 0.0101399, 0.00970582, 0.00928997, 0.00889159, 
            0.00850998, 0.00814444, 0.00779431, 0.00745897, 0.00713779, 
            0.0068302, 0.00653563, 0.00625355, 0.00598344, 0.0057248, 0.00547715, 
            0.00524005, 0.00501304, 0.00479572, 0.00458766, 0.0043885, 
            0.00419784, 0.00401535, 0.00384067, 0.00367348, 0.00351346, 
            0.00336032, 0.00321375, 0.00307349, 0.00293926, 0.00281082, 
            0.00268791, 0.00257031, 0.00245778, 0.00235012, 0.00224711, 
            0.00214856, 0.00205428, 0.00196408};

            ParameterFunction yFit = delegate(double t, Vector p)
            {
                double tau = p[0], A1 = p[1], A2 = p[2], A3 = p[3], L1 = p[4], L2 = p[5], L3 = p[6];
                double d = (t - tau);
                if (t < tau)
                    return 0;
                else
                {
                    double ex1 = (A1 * d - A2 - A3) * Math.Exp(L1 * d);
                    double ex2 = A2 * Math.Exp(L2 * d);
                    double ex3 = A3 * Math.Exp(L3 * d);
                    return ex1 + ex2 + ex3;
                }
            };

            int lessThanOne = 0;
            int iters = 30;

            for (int k = 0; k < iters; k++)
            {

                // the right parameters
                Vector right = new double[] { 0.6, 2.3, -1.1, -1, -1, -2, -1 };
                double radius = 0.75;

                // initial parameters
                Vector initial = new Vector(right.Length);
                for (int i = 0; i < right.Length; i++)
                {
                    initial[i] = right[i] + radius - 2 * radius * rand.NextDouble();
                }

                // the optimized parameters
                Optimization sut = new FitLMA(datax, datay, initial, yFit);
                sut.Iterate(100);
                
                double min = sut.AtMinimum();
                
                if (min < 1.0) {
                    lessThanOne++;
                }
                
                //Console.WriteLine("initial answer: \t" + initial);
                //Console.WriteLine("found answer: \t" + sut.GetMinimum());
                //Console.WriteLine("right answer: \t" + right);
                Console.WriteLine("min: \t" + min);
            }

            double per = 100*(lessThanOne+0.0)/iters;
            Console.WriteLine("" + per + " percent of the answers were below one." );
            Assert.Greater(per, 5);
        }

        /// <summary>
        /// Exception
        /// </summary>
        [Test]
        public void Test1_6_Exception()
        {
            double[] temp = new double[] { 1, 2, 3 };
            double[] temp2 = new double[] { 1, 2, 3, 4 };
            PartialDerivate p = delegate(double x, Vector a, Vector dyda)
            {
                return 0.0;
            };

            try
            {
                FitLMA sut = new FitLMA(null, temp, temp, p);
                Assert.Fail();
            }
            catch (OptimizationException) { }
            try
            {
                FitLMA sut = new FitLMA(temp, temp2, temp, p);
                Assert.Fail();
            }
            catch (OptimizationException) { }
        }

        /// <summary>
        /// Approximate differential.
        /// Input data is analytical correct values.
        /// </summary>
        [Test]
        public void Test1_7_Derivate()
        {
            ApproximateDerivate app = delegate(ParameterFunction f, double x, Vector a, int i)
            {
                Vector b = new Vector(a);
                b[i] = b[i] + 1e-10;
                return (f(x, b) - f(x, a)) / 1e-10;
            };

            ParameterFunction pf = delegate(double x, Vector p)
            {
                double a = p[0], b = p[1];
                return a * x * x * x - x * x + b * x;
            };

            PartialDerivate parAn = delegate(double x, Vector p, Vector dyda)
            {
                double a = p[0], b = p[1];
                dyda[0] = x * x * x;
                dyda[1] = x;
                return pf(x, p);
            };

            PartialDerivate parAp = delegate(double x, Vector a, Vector dyda)
            {
                for (int i = 0; i < a.Length; i++)
                {
                    dyda[i] = app(pf, x, a, i);
                }
                return pf(x, a);
            };

            double[] param = new double[] { 9, 3 };

            Vector dyda1 = new Vector(2);
            parAn(3, param, dyda1);
            Vector dyda2 = new Vector(2);
            parAp(3, param, dyda2);

            for (int i = 0; i < dyda1.Length; i++)
                Assert.AreEqual(dyda1[i], dyda2[i], 0.001, "i := " + i);
        }


    }
}