﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.Model
{
    /// <summary>
    /// Tests for Linear Integration class
    /// This testbench implements and extends method ‘test_integr’ in former C-library.
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Integral
    {
        // Help routine.
        private void Print(double[] s)
        {
            foreach (double x in s)
            {
                Console.Write(x + "  ");
            }
            Console.WriteLine();
        }

        /// <summary>
        /// Integrate gap a and b
        /// </summary>
        [Test]
        public void Test1_1_Integrate()
        {
            // the system under test
            LinearIntegration sut = new LinearIntegration();

            double[] xin = new double[] { 1, 1.5 };
            double[] yin = new double[] { 1, 1 };
            double[] yout = sut.Integrate(xin, yin);
            // now gap is x[0] < (x[1] - x[0])
            Assert.AreEqual(0.0, yout[0]);
            xin[1] = 2;
            // now gap is x[0] >= (x[1] - x[0])
            yout = sut.Integrate(xin, yin);
            Assert.AreEqual(0.5, yout[0]);

            // negative values
            double[] x = new double[] { 2, 3, 4, 5, 6 };
            double[] y = new double[] { 1, 1, 1, 1, 1 };
            yout = sut.Integrate(x, y);
            Assert.AreEqual(0.0, yout[0]);
            x = new double[] { -2, -1, 0, 1, 2 };
            yout = sut.Integrate(x, y);
            Assert.AreEqual(-1.0, yout[0]);

            try
            {
                // the data is not ascending.
                x = new double[] { 6, 5, 4, 3, 2 };
                yout = sut.Integrate(x, y);
                Assert.Fail();
            }
            catch (Exception)
            {
            }
        }

        /// <summary>
        /// IntegrateToFrameEndTimes
        /// Ten timepoints (1,10), all of value 1.0 
        /// </summary>
        [Test]
        public void Test1_2_IntegrateToFrameEndTimes()
        {
            LinearIntegration ic = new LinearIntegration();

            // the number of test points
            int length = 10;

            double[] x1in = new double[length];
            double[] x2in = new double[length];
            double[] yin = new double[length];
            double[] xnew1 = new double[length];

            // initialize test data
            for (int i = 0; i < length; i++)
            {
                x1in[i] = i + 1; // start time
                x2in[i] = i + 2; // end time
                yin[i] = 1.0; // value
            }

            double[] yout = ic.Integrate(x1in, yin);
            double[] not_needed;
            Assert.AreEqual(10, yout.Length);
            Assert.AreEqual(0.5, yout[0]);
            Assert.AreEqual(1.5, yout[1]);
            Assert.AreEqual(9.5, yout[9]);

            LinearIntegration.IntegrateToFrameEndTimes(x1in, x2in, yin, out yout, out not_needed);
            Assert.AreEqual(10.3333, yout[9], 0.0001);
        }
        
        /// <summary>
        /// IntegrateToFrameEndTimes
        /// Ten timepoints (3,13), all of value 1.0
        /// </summary>
        [Test]
        public void Test1_3_IntegrateToFrameEndTimes()
        {
            LinearIntegration ic = new LinearIntegration();

            // the number of test points
            int length = 10;

            double[] x1in = new double[length];
            double[] x2in = new double[length];
            double[] yin = new double[length];
            double[] xnew1 = new double[length];
            double[] _2nd = new double[length];

            for (int i = 0; i < 10; i++)
            {
                x1in[i] = i + 3;
                x2in[i] = i + 4;
                yin[i] = 1.0;
            }

            Console.WriteLine("input");
            Print(x1in);
            Print(x2in);
            Print(yin);
            double[] yout = ic.Integrate(x1in, yin);
            Assert.AreEqual(0.0, yout[0]);
            Assert.AreEqual(9.0, yout[9]);
            
            Console.WriteLine("output");
            Print(yout);

            LinearIntegration.IntegrateToFrameEndTimes(x1in, x2in, yin, out yout, out _2nd);
            Assert.AreEqual(10.0, yout[9]);
            Console.WriteLine("1st integral");
            Print(yout);
            Console.WriteLine("2nd integral");
            Print(_2nd);
        }

        /// <summary>
        /// IntegrateToFrameMidTimes
        /// </summary>
        [Test]
        public void Test1_4_IntegrateToFrameMidTimes()
        {
            LinearIntegration li = new LinearIntegration();

            int len = 5;
            double[] x1 = { 1, 2, 3, 4, 5 };
            double[] x2 = { 2, 3, 4, 5, 6 };
            double[] y = { 1, 2, 3, 4, 5 };
            double[] _1st = new double[len];
            double[] _2nd = new double[len];
            double[] not_needed;
            double[] not_needed2;

            try
            {
                LinearIntegration.IntegrateToFrameMidTimes(x1, x2, y, out not_needed, out not_needed2);
                Assert.Fail("");
            }
            catch (Exception)
            {
                LinearIntegration.IntegrateToFrameMidTimes(x1, x2, y, out _1st, out _2nd);
            }

            Print(x1);
            Print(x2);
            Print(y);
            Console.WriteLine("output");
            Print(_1st);
            Print(_2nd);

            Assert.AreEqual(0.75, _1st[0], 0.001);
            Assert.AreEqual(2.208, _1st[1], 0.001);
            Assert.AreEqual(4.708, _1st[2], 0.001);
            Assert.AreEqual(8.208, _1st[3], 0.001);
            Assert.AreEqual(12.708, _1st[4], 0.001);

            Assert.AreEqual(0.5625, _2nd[0], 0.001);
            Assert.AreEqual(2.042, _2nd[1], 0.001);
            Assert.AreEqual(5.5, _2nd[2], 0.001);
            Assert.AreEqual(11.958, _2nd[3], 0.001);
            Assert.AreEqual(22.417, _2nd[4], 0.001);
        }

        private delegate void DoIt();
        
        /// <summary>
        /// Test for thrown expections.
        /// </summary>
        [Test]
        public void Test3_1()
        {
            double[] size3 = new double[] { 0, 1, 2 };
            double[] size4 = new double[] { 0, 1, 2, 4 };
            double[] notAscending = new double[] { 3, 2, 1 };
            LinearIntegration sut = new LinearIntegration();

            List<DoIt> list = new List<DoIt>();
            list.Add(delegate() { sut.Integrate(null, null); });
            list.Add(delegate() { sut.Integrate(size3, size4); });
            list.Add(delegate() { sut.Integrate(notAscending, notAscending); });
            
            foreach (DoIt test in list)
            {
                try
                {
                    test();
                    Assert.Fail("No exception thrown!");
                }
                catch (TPCException e)
                {
                    Console.WriteLine(e.Message);
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                }
            }
        }
    }
}