﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;

namespace TPClib.Model
{
    /// <summary>
    /// Unit testing class for TPClib.Model.Linear
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Linear
    {
        private const int MAX_DIM = 40;
        private const double MAX_VAL = 1e5;
        private const double TOLERANCE = 1e-5;
        private const double SPARSE = 0.30;
        private const double ULTRASPARSE = 0.15;

        private Matrix dense, sparse, vsparse;
        private Vector solution, bdense, bsparse, bvsparse;

        private Random rand;

        /// <summary>
        /// Set up test cases
        /// </summary>
        [TestFixtureSetUp]
        public void Init()
        {
            rand = new Random();
            int m = MAX_DIM;
            int n = rand.Next(m) + 1;

            dense = new Matrix(m, n);
            sparse = new Matrix(m, n);
            vsparse = new Matrix(m, n);
            solution = new Vector(n);

            for (int i = 0; i < n; i++)
            {
                solution[i] = MAX_VAL * (rand.NextDouble() - 0.5);

                for (int j = 0; j < m; j++)
                {
                    dense[j, i] = MAX_VAL * (rand.NextDouble() - 0.5);
                    if (rand.NextDouble() < SPARSE) sparse[j, i] = MAX_VAL * (rand.NextDouble() - 0.5);
                    if (rand.NextDouble() < ULTRASPARSE) vsparse[j, i] = MAX_VAL * (rand.NextDouble() - 0.5);
                }
            }
            bdense = dense * solution;
            bsparse = sparse * solution;
            bvsparse = vsparse * solution;
            Console.WriteLine("Testing with");
            Console.WriteLine("Dense {0}x{1} matrix\n" + dense.ToString("F2"), dense.Rows, dense.Columns);
            Console.WriteLine("Sparse {0}x{1} matrix\n" + sparse.ToString("F2"), sparse.Rows, sparse.Columns);
            Console.WriteLine("Very {0}x{1} sparse matrix\n" + vsparse.ToString("F2"), vsparse.Rows, vsparse.Columns);
            Console.WriteLine("Solution vector " + solution.ToString("F2"));
        }

        /// <summary>
        /// 1 Solve three linear systems (with dense, sparse, and very sparse matrices) using QRSolve.
        /// </summary>
        [Test]
        public void Test1()
        {
            Vector qr1 = LinearEquations.QRSolve(dense, bdense);
            Vector qr2 = LinearEquations.QRSolve(sparse, bsparse);
            Vector qr3 = LinearEquations.QRSolve(vsparse, bvsparse);

            Vector dif1 = qr1 - solution;
            Vector dif2 = qr2 - solution;
            Vector dif3 = qr3 - solution;

            Assert.AreEqual(0.0, Vector.Max(dif1 * dif1), TOLERANCE, "Dense matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif2 * dif2), TOLERANCE, "Sparse matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif3 * dif3), TOLERANCE, "Very sparse matrix failed");
        }

        /// <summary>
        /// 2 Solve three linear systems (with dense, sparse, and very sparse matrices) using CholeskySolve.
        /// </summary>
        [Test]
        public void Test2()
        {
            Vector chol1 = LinearEquations.CholeskySolveGen(dense, bdense);
            Vector chol2 = LinearEquations.CholeskySolveGen(sparse, bsparse);
            Vector chol3 = LinearEquations.CholeskySolveGen(vsparse, bvsparse);

            Vector dif1 = chol1 - solution;
            Vector dif2 = chol2 - solution;
            Vector dif3 = chol3 - solution;

            Assert.AreEqual(0.0, Vector.Max(dif1 * dif1), TOLERANCE, "Dense matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif2 * dif2), TOLERANCE, "Sparse matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif3 * dif3), TOLERANCE, "Very sparse matrix failed");
        }

        /// <summary>
        /// 3 Solve three linear systems (with dense, sparse, and very sparse matrices) Gauss-Jordan.
        /// </summary>
        [Test]
        public void Test3()
        {
            Vector gj1 = LinearEquations.GaussJordan(dense, bdense);
            Vector gj22 = LinearEquations.GaussJordan(sparse, bsparse);
            Vector gj3 = LinearEquations.GaussJordan(vsparse, bvsparse);

            Vector dif1 = gj1 - solution;
            Vector dif2 = gj22 - solution;
            Vector dif3 = gj3 - solution;

            Assert.AreEqual(0.0, Vector.Max(dif1 * dif1), TOLERANCE, "Dense matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif2 * dif2), TOLERANCE, "Sparse matrix failed");
            Assert.AreEqual(0.0, Vector.Max(dif3 * dif3), TOLERANCE, "Very sparse matrix failed");
        }

        /// <summary>
        /// 4 Try to solve non-symmetric system with Cholesky
        /// </summary>
        [Test]
        [ExpectedException]
        public void Test4()
        {
            LinearEquations.CholeskySolve(new Matrix(3, 2), new Vector(2));
        }

        /// <summary>
        /// 5 Try to solve non-positive definitive system with Cholesky
        /// </summary>
        [Test]
        [ExpectedException]
        public void Test5()
        {
            // Linearly dependent matrix should do the trick...
            Matrix a = new Matrix(2);
            a[0, 0] = a[1, 0] = 1.0;
            LinearEquations.CholeskySolve(a, new Vector(2));
        }

        /// <summary>
        /// 6 Try to solve underdetermined system with QR.
        /// </summary>
        [Test]
        public void Test6()
        {
            // Use the transpose of dense matrix;
            Matrix a1 = dense.Transpose();

            Vector qr1 = LinearEquations.QRSolve(a1, solution);

            Vector dif1 = a1 * qr1 - solution;

            Assert.AreEqual(0.0, Vector.Max(dif1 * dif1), TOLERANCE, "Dense matrix failed");
        }

        /// <summary>
        /// 7 Form a QR decomposition of a matrix (randomly generated)
        /// </summary>
        [Test]
        public void Test7()
        {
            Matrix q = LinearEquations.QR(dense);
            int n = q.Rows;
            Vector v = Vector.Fill(n, 1.0);
            Matrix i = q.Transpose() * q;
            Vector err = i * v - v;
            Assert.AreEqual(0.0, Vector.Max(err * err), TOLERANCE, "QQ' != I");
        }

        /// <summary>
        /// 8 Form a Cholesky decomposition of a matrix
        /// </summary>
        [Test]
        public void Test8()
        {
            Matrix a = dense.Transpose() * dense;
            Matrix ch = LinearEquations.Cholesky(a);

            int n = ch.Rows;
            Vector v = Vector.Fill(n, 1.0);
            Matrix zero = ch * ch.Transpose() - a;
            Vector err = zero * v;

            Assert.AreEqual(0.0, Vector.Max(err * err), TOLERANCE, "LL' != A");
        }

        /// <summary>
        /// 9 Form a Householder transformation matrix with randomly generated normal vector
        /// </summary>
        [Test]
        public void Test9()
        {
            Matrix hh = LinearEquations.Householder(solution);

            Vector zero = hh * solution + solution;

            Assert.AreEqual(0.0, Vector.Norm(zero), TOLERANCE, "Vector not reflected correctly!");
        }

    /*  NNLS testing
        /// <summary>
        /// NNLS
        /// </summary>
        [Test]
        public void Test10()
        {
            Vector solution = new Vector(0, 0.5);
            Matrix a = new Matrix(3, 2);
            a[0, 0] = 1; a[0, 1] = 0;
            a[1, 0] = 0; a[1, 1] = 1;
            a[2, 0] = 2; a[2, 1] = 1;
            Vector b = new Vector(1, 3, -2);

            Vector x = Linear.NNLS(a, b);
            for (int i = 0; i < x.Length; i++)
            {
                Assert.AreEqual(solution[i], x[i], TOLERANCE);
            }
        }

        /// <summary>
        /// NNLS
        /// </summary>
        [Test]
        public void Test11()
        {
            Vector solution = new Vector(1, 0);
            Matrix a = new Matrix(4, 2);
            a[0, 0] = 0; a[0, 1] = 1;
            a[1, 0] = 1; a[1, 1] = 1;
            a[2, 0] = 2; a[2, 1] = 1;
            a[3, 0] = 3; a[3, 1] = 1;
            Vector b = new Vector(-1, 1, 2, 3);
            Vector x = Linear.NNLS(a, b);

            Console.WriteLine("x=" + x);
            for (int i = 0; i < x.Length; i++)
            {
                Assert.AreEqual(solution[i], x[i], TOLERANCE);
            }
        }

        class LinearModel : Model
        {
            private Matrix m;
            public override int Samples
            {
                get { return m.Rows; }
            }
            public LinearModel(Matrix matrix) { m = matrix; }

            public override Vector Simulate(params double[] par)
            {
                return m * par;
            }
        }

        /// <summary>
        /// NNLS
        /// </summary>
        [Test]
        public void Test12()
        {
            Model m = new LinearModel(sparse);
            Fit f = new Fit(m, bsparse);
            Optimization opt = new PowellBrent(f.SumOfSq, new Vector(solution.Length));
            opt.SetFloor(new Vector(solution.Length));
            opt.Iterate(200);

            Console.WriteLine("Powell ready");
            Console.WriteLine(opt.GetMinimum());
            Console.WriteLine(f.SumOfSq(opt.GetMinimum()));
            Vector nnls = Linear.NNLS(sparse, bsparse);
            Console.WriteLine(nnls);
            Console.WriteLine(f.SumOfSq(nnls));
        }
*/
    }
}
