﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using NUnit.Framework;
using TPClib.Curve;

namespace TPClib.Model
{
    /// <summary>
    /// Unit testing for TPClib.Model.Model
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Model
    {
        private string path = @"P:\data\TPClib_Model\";
        private const double TOLERANCE = 1e-3;

        private Vector TimeData(string file)
        {
            DFTRegionalCurveFile f = new DFTRegionalCurveFile(file);
            f.ReadFile();
            RegionalTACTable rtt = f.Curves;
            FrameTimeCell[] ftc = rtt.GetTimeCells();
            Vector times = new Vector(ftc.Length);
            
            for (int i = 0; i < ftc.Length; i++)
            {
                // average of start and end times
                double avg = (ftc[i].start_time + ftc[i].end_time) / 2;
                
                // convert to minutes
                times[i] = avg / 60.0;
            }

            return times;
        }

        private Vector RefData(string file)
        {
            DFTRegionalCurveFile f = new DFTRegionalCurveFile(file);
            f.ReadFile();
            RegionalTACTable rtt = f.Curves;
            return (double[])(rtt.GetColumn(0));
        }
        
        private Vector TestData(string file)
        {
            DFTRegionalCurveFile t = new DFTRegionalCurveFile(file);
            t.ReadFile();
            RegionalTACTable tac = t.Curves;
            return (double[])(tac.GetColumn(0));
        }

        /// <summary>
        /// Real data desc..
        /// </summary>
        [Test]
        public void Test1_1_SRTM()
        {
            Vector refdata = RefData(path + "srtm.dft");
            Vector times = TimeData(path + "srtm.dft");
            Vector test = TestData(path + "test_srtm.dft");

            SRTMModel srt = new SRTMModel(times, refdata);
            double R1 = 1.390909;
            double k2 = 0.489;
            double BP = 3.272138;
            Vector res = srt.Simulate(R1, k2, BP);

            double error = 0;
            Console.WriteLine("times\tanswer\toutput");
            for (int i = 0; i < res.Length; i++)
            {
                double sim = res[i];
                double cor = test[i];
                Console.WriteLine(times[i] + "\t" + cor + "\t " + sim);
                
                if (sim != 0)
                    error = Math.Max(error, Math.Abs((sim - cor) / sim));
            }

            Console.WriteLine("Maximum error was " + error * 100 + " % ");
            Assert.AreEqual(error, 0, TOLERANCE, "Error too big");
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_2_TRTM()
        {
            Vector refdata = RefData(path + "trtm.dft");
            Vector times = TimeData(path + "trtm.dft");
            Vector tests = TestData(path + "test_trtm.dft");

            TRTMModel trt = new TRTMModel(times, refdata);
            double R1 = 0.8;
            double rk2 = 0.1125;
            double k3 = 0.07;
            Vector res = trt.Simulate(R1,rk2,k3);

            double error = 0;
            Console.WriteLine("times\tanswer\toutput");
            for (int i = 0; i < res.Length; i++)
            {
                double sim = res[i];
                double cor = tests[i];
                Console.WriteLine(times[i] + "\t" + cor + "\t " + sim);

                if(sim != 0)
                    error = Math.Max(error, Math.Abs((sim - cor) / sim));
            }

            Assert.AreEqual(error, 0, TOLERANCE, "Error too big");
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void Test1_3_RTCM()
        {
            Vector refdata = RefData(path + "rtcm.dft");
            Vector times = TimeData(path + "rtcm.dft");
            Vector test = TestData(path + "test_rtcm.dft");

            RTCMModel rtcm = new RTCMModel(times, refdata);
            double R1 = 1.390909;
            double k2 = 0.489;
            double k3 = 0.303;
            double BP = 3.272138;
            Vector res = rtcm.Simulate(R1, k2, k3, BP);

            double error = 0;
            Console.WriteLine("times\tanswer\toutput");
            for (int i = 0; i < res.Length; i++)
            {
                double sim = res[i];
                double cor = test[i];
                
                if (sim != 0)
                {
                    error = Math.Max(error, Math.Abs((sim - cor) / sim));
                }

                Console.WriteLine(times[i] + "\t" + cor + "\t " + sim + "\t" + error);
            }

            Console.WriteLine("Maximum error was " + error * 100 + " % ");
            Assert.AreEqual(error, 0, TOLERANCE, "Error too big " + 100*error + " % ");
        }
    }
}
