﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;

namespace TPClib.Model
{
    /// <summary>
    /// d y[t] / dt = 2 * y[t] * (1 - y[t])
    /// </summary>
    class ODEDim1 : ODE
    {
        double[] y;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="y">initial value</param>
        public ODEDim1(double y)
        {
            this.y = new double[] { y };
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public double[] GetVars()
        {
            return y;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="y"></param>
        /// <param name="change"></param>
        public void Evaluate(double[] y, double[] change)
        {
            change[0] = 2 * y[0] * (1 - y[0]);
        }
    }

    /// <summary>
    /// Compartmental Model for Ammonia
    /// Eq. 1 page 1175 
    /// The Journal of Nuclear Medicine Vol 42 No 8 August 2001
    /// </summary>
    class ODEDim2 : ODE
    {
        // concentrations
        double[] c;

        // Constants
        double K1, ca, k2, k3;

        public ODEDim2(double c1, double c2, double K1, double ca, double k2, double k3)
        {
            c = new double[] { c1, c2 };
            this.K1 = K1;
            this.ca = ca;
            this.k2 = k2;
            this.k3 = k3;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public double[] GetVars()
        {
            return c;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="c"></param>
        /// <param name="change"></param>
        public void Evaluate(double[] c, double[] change)
        {
            change[0] = K1 * ca - (k2 + k3) * c[0];
            change[1] = k3 * c[0];
        }
    }

    /// <summary>
    /// Test cases for Differential Equation
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_ODESolver
    {
        double error_margin = 0.00001;

        /// <summary>
        /// 1.1 In Mathematica: 
        ///     eqs = {y'[t] == 2 y[t] (1 - y[t]), y[0] == 0.5};
        ///     s = NDSolve[eqs, y, {t, 0, 1}];
        ///     Evaluate[y[t] /. s] /. t -> 1.0
        /// Output:
        ///     0.880797
        /// </summary>
        [Test]
        public void Test1_1_OneDimensionalODE()
        {
            ODE ode = new ODEDim1(0.5);
            ODESolver sut = new RungeKutta(ode);
            double step = 0.1;

            int steps = (int)(1.0 / step);
            for (int i = 0; i < steps; i++)
            {
                sut.Step(step);
            }

            double[] y = ode.GetVars();
            Assert.AreEqual(0.880797, y[0], error_margin);
        }

        /// <summary>
        /// 1.2 The results are compared with the same model in Mathematica
        /// Input:
        ///     K1 = 0.76; ca = 0.4; k2 = 0.4; k3 = 0.2;
        ///     eqs = {c1'[t] == K1 ca - (k2 + k3) c1[t], c2'[t] == k3 c1[t], c1[0] == 0.5, c2[0] == 0.6};
        ///     s = NDSolve[eqs, {c1, c2}, {t, 0, 5}];
        ///     Evaluate[{c1[t], c2[t]} /. s] /. t -> 5.0
        /// Output:
        ///     {{0.506335, 1.10456}}
        /// </summary>
        [Test]
        public void Test1_2_TwoDimensionalODE()
        {
            ODE ode = new ODEDim2(0.5, 0.6, 0.76, 0.4, 0.4, 0.2);
            // system under test = sut
            ODESolver sut = new RungeKutta(ode);
            double step = 0.1;

            // calculate the needed steps to minimize floating point rounding errors.
            int steps = (int)(5.0 / step);
            for (int i = 0; i < steps; i++)
            {
                sut.Step(step);
            }

            double[] c = ode.GetVars();
            Assert.AreEqual(0.5063347264025634, c[0], error_margin);
            Assert.AreEqual(1.104555091199146, c[1], error_margin);
        }
    }
}
