﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.Model
{

    /// <summary>
    /// Optimization methods test fixture
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Optimization
    {
        private const int MAX_DIM = 10;
        private const double MAX_VAL = 1e5;
        private const double TOLERANCE = 1e-5;

        private readonly Vector Rosenbrockmin = new Vector(1.0, 1.0);

        /// <summary>
        /// Current test data
        /// </summary>
        private Vector data;

        private Random rand;

        // parabola with minimum at 1.0
        private double Parabola(Vector pars)
        {
            double x = pars[0] - 1.0;
            return x * x;
        }

        // Rosenbrock function, global minimum at (1.0, 1.0)
        private double Rosenbrock(Vector pars)
        {
            double x = pars[0];
            double y = pars[1];

            double a = (1 - x);
            double b = (y - x * x);
            return a * a + 100 * y * y;
        }

        // Himmelblau's function, with four global minimums (function value 0.0) at
        // (3.0, 2.0), (-3.78, -3.28), (-2.81, 3.13) and (3.58, -1.84)
        private double Himmelblau(Vector pars)
        {
            double x = pars[0];
            double y = pars[1];
            double a = x * x + y - 11;
            double b = x + y * y - 7;
            return a * a + b * b;
        }

        // General polynom function
        private double Polynom(double x, Vector coefs)
        {
            double sum = 0.0;

            for (int i = coefs.Length - 1; i >= 0; i--)
            {
                sum = sum * x + coefs[i];
            }

            return sum;
        }

        /// <summary>
        /// Calculates the sum of squares of a
        /// n-1 degree polynom to a set of n data
        /// points. The minimum of this function
        /// is always 0.0, regardless of dimension.
        /// </summary>
        /// <param name="pars">polynom coefficients</param>
        /// <returns>sum of squares between polynom</returns>
        private double PolSumOfSq(Vector pars)
        {
            double sum = 0.0;
            double dif;
            for (int i = 0; i < data.Length; i++)
            {
                dif = data[i] - Polynom(i, pars);
                sum += dif * dif;
            }
            return sum;
        }

        /// <summary>
        /// Calculates distance between data and parameters
        /// </summary>
        /// <param name="pars">parameters</param>
        /// <returns>sum of squares between data and parameters</returns>
        private double SumOfSquares(Vector pars)
        {
            Vector dif = data - pars;
            return Vector.Dot(dif, dif);
        }

        /// <summary>
        /// This will be excuted once before running any tests
        /// </summary>
        [TestFixtureSetUp]
        public void Init()
        {
            // Random data vector with values in range [-MAX_VAL, MAX_VAL]
            data = Vector.RandomPoint(
                Vector.Fill(MAX_DIM, MAX_VAL),
                Vector.Fill(MAX_DIM, -MAX_VAL),
                DateTime.Now.Millisecond);

            rand = new Random();
        }

        /// <summary>
        /// Optimize one dimensional parabola function with Nelder-Mead
        /// </summary>
        [Test]
        public void Test3_1()
        {
            Vector initial = new Vector(rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new NelderMead(Parabola, initial);
            opt1.SetTolerance(Parabola(1.0 + TOLERANCE));
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(1.0, min[0], TOLERANCE, "Nelder-Mead failed");
        }

        /// <summary>
        /// Optimize one dimensional parabola function with Powell-Brent
        /// </summary>
        [Test]
        public void Test3_2()
        {
            Vector initial = new Vector(rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new PowellBrent(Parabola, initial);
            opt1.SetTolerance(Parabola(1.0 + TOLERANCE));
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(1.0, min[0], TOLERANCE, "Powell-Brent failed");
        }

        /// <summary>
        /// Optimize one dimensional parabola function with Random directions
        /// </summary>
        [Test]
        public void Test3_3()
        {
            Vector initial = new Vector(rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new RandomDirections(Parabola, initial);
            opt1.SetTolerance(Parabola(1.0 + TOLERANCE));
            opt1.SetMaxIterations(1000);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(1.0, min[0], TOLERANCE, "Random directions failed");
        }

        /// <summary>
        /// Optimize one dimensional parabola function with ITGO
        /// </summary>
        [Test]
        public void Test3_4()
        {
            Vector low = new Vector(-MAX_VAL);
            Vector high = new Vector(MAX_VAL);
            Optimization opt1 = new ITGO(Parabola, 10, 100, high, low);
            opt1.SetTolerance(Parabola(1.0 + TOLERANCE));
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(1.0, min[0], TOLERANCE, "ITGO failed");
        }


        /// <summary>
        /// Optimize the Rosenbrock function with Nelder-Mead
        /// </summary>
        [Test]
        public void Test4_1()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new NelderMead(Rosenbrock, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Rosenbrock(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Nelder-Mead failed");
        }

        /// <summary>
        /// Optimize the Rosenbrock function with Powell-Brent
        /// /// </summary>
        [Test]
        public void Test4_2()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new PowellBrent(Rosenbrock, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Rosenbrock(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Powell-Brent failed");
        }

        /// <summary>
        /// Optimize the Rosenbrock function with Random directions
        /// /// </summary>
        [Test]
        public void Test4_3()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new RandomDirections(Rosenbrock, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Rosenbrock(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Random directions failed");
        }

        /// <summary>
        /// Optimize the Rosenbrock function with ITGO
        /// </summary>
        [Test]
        public void Test4_4()
        {
            Vector low = new Vector(-MAX_VAL, -MAX_VAL);
            Vector high = new Vector(MAX_VAL, MAX_VAL);
            Optimization opt1 = new ITGO(Rosenbrock, 10, 100, high, low);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Rosenbrock(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "ITGO failed");
        }

        /// <summary>
        /// Optimize Himmelblau's function with Nelder-Mead
        /// </summary>
        [Test]
        public void Test5_1()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new NelderMead(Himmelblau, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Himmelblau(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Nelder-Mead failed");
        }

        /// <summary>
        /// Optimize Himmelblau's function with Powell-Brent
        /// </summary>
        [Test]
        public void Test5_2()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new PowellBrent(Himmelblau, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Himmelblau(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Powell-Brent failed");
        }

        /// <summary>
        /// Optimize Himmelblau's function with Random directions
        /// </summary>
        [Test]
        public void Test5_3()
        {
            Vector initial = Vector.Fill(2, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new RandomDirections(Himmelblau, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Himmelblau(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Random directions failed");
        }

        /// <summary>
        /// Optimize Himmelblau's function with ITGO
        /// </summary>
        [Test]
        public void Test5_4()
        {
            Vector low = new Vector(-MAX_VAL, -MAX_VAL);
            Vector high = new Vector(MAX_VAL, MAX_VAL);
            Optimization opt1 = new ITGO(Himmelblau, 10, 100, high, low);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = Himmelblau(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "ITGO failed");
        }

        /// <summary>
        /// Optimize sum of squares function with Nelder-Mead
        /// Note: This will take a *lot* of time, due to methods
        /// rapid decrease in efficiency when the problems dimension
        /// increases.
        /// </summary>
        [Test]
        public void Test6_1()
        {
            Vector initial = Vector.Fill(MAX_DIM, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new NelderMead(SumOfSquares, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = SumOfSquares(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Nelder-Mead failed");
        }

        /// <summary>
        /// Optimize sum of squares function with Powell-Brent
        /// </summary>
        [Test]
        public void Test6_2()
        {
            Vector initial = Vector.Fill(MAX_DIM, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new PowellBrent(SumOfSquares, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = SumOfSquares(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Powell-Brent failed");
        }

        /// <summary>
        /// Optimize sum of squares function with Random directions
        /// </summary>
        [Test]
        public void Test6_3()
        {
            Vector initial = Vector.Fill(MAX_DIM, rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new RandomDirections(SumOfSquares, initial);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = SumOfSquares(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "Random directions failed");
        }

        /// <summary>
        /// Optimize sum of squares function with ITGO
        /// </summary>
        [Test]
        public void Test6_4()
        {
            Vector low = Vector.Fill(MAX_DIM, -MAX_VAL);
            Vector high = Vector.Fill(MAX_DIM, MAX_VAL);
            Optimization opt1 = new ITGO(SumOfSquares, 10, 100, high, low);
            opt1.SetTolerance(TOLERANCE);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            double minval = SumOfSquares(min);
            Assert.AreEqual(0.0, minval, TOLERANCE, "ITGO failed");
        }

        /// <summary>
        /// 7.4 Test constraints
        /// Optimize one dimensional parabola function with Nelder-Mead, with maximum value set at 0.
        /// </summary>
        [Test]
        public void Test7_1()
        {
            Vector initial = new Vector(-rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new NelderMead(Parabola, initial);
            // Stop, when near enough of zero
            opt1.AddStop(delegate(Vector v) { return Vector.Norm(opt1.GetMinimum()) < TOLERANCE; });
            opt1.SetCeiling(0.0);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(0.0, min[0], TOLERANCE, "Nelder-Mead failed");
        }

        /// <summary>
        /// 7.4 Test constraints
        /// Optimize one dimensional parabola function with Powell-Brent, with maximum value set at 0.
        /// </summary>
        [Test]
        public void Test7_2()
        {
            Vector initial = new Vector(-rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new PowellBrent(Parabola, initial);
            // Stop, when near enough of zero
            opt1.AddStop(delegate(Vector v) { return Vector.Norm(opt1.GetMinimum()) < TOLERANCE; });
            opt1.SetCeiling(0.0);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(0.0, min[0], TOLERANCE, "Powell-Brent failed");
        }

        /// <summary>
        /// 7.4 Test constraints
        /// Optimize one dimensional parabola function with Random Directions, with maximum value set at 0.
        /// </summary>
        [Test]
        public void Test7_3()
        {
            Vector initial = new Vector(-rand.NextDouble() * MAX_VAL);
            Optimization opt1 = new RandomDirections(Parabola, initial);
            // Stop, when near enough of zero
            opt1.AddStop(delegate(Vector v) { return Vector.Norm(opt1.GetMinimum()) < TOLERANCE; });
            opt1.SetCeiling(0.0);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(0.0, min[0], TOLERANCE, "Random directions failed");
        }

        /// <summary>
        /// 7.4 Test constraints
        /// Optimize one dimensional parabola function with ITGO, with maximum value set at 0.
        /// </summary>
        [Test]
        public void Test7_4()
        {
            Vector low = new Vector(-MAX_VAL);
            Vector high = new Vector(0.0);
            Optimization opt1 = new ITGO(Parabola, 10, 100, high, low);
            // Stop, when near enough of zero
            opt1.AddStop(delegate(Vector v) { return Vector.Norm(opt1.GetMinimum()) < TOLERANCE; });
            opt1.SetCeiling(0.0);
            opt1.IterateUntilStop();

            Vector min = opt1.GetMinimum();
            Assert.AreEqual(0.0, min[0], TOLERANCE, "ITGO failed");
        }
    }
}
