﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.Model
{
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Powell
    {
        /// <summary>
        /// Bracketing
        /// </summary>
        [Test]
        public void Test1_1()
        {
            Function func1 = delegate(double x)
            {
                return x * x - x;
            };

            Function func2 = delegate(double x)
            {
                return Math.Cos(x) * Math.Sin(x) * Math.Sin(x);
            };

            double error = 1e-16;
            Brent brent = new Brent();
            for (int i = 0; i < 2; i++)
            {
                double ax = 0, bx = 0, cx = 0, fa = 0, fb = 0, fc = 0;

                if(i == 0)
                    brent.Bracket(0.2, 0.9, func1, out ax, out bx, out cx, out fa, out fb, out fc);
                
                if(i == 1)
                    brent.Bracket(1, 3, func2, out ax, out bx, out cx, out fa, out fb, out fc);

                Console.WriteLine("ax=" + ax + " bx=" + bx + " cx=" + cx);
                Console.WriteLine("fa=" + fa + " fb=" + fb + " fc=" + fc);

                Assert.IsTrue((ax <= bx && bx <= cx) || (cx <= bx && bx <= ax));

                // fb on minimi
                Assert.LessOrEqual(fb, fa + error);
                Assert.LessOrEqual(fb, fc + error);
            }
        }

        /// <summary>
        /// LineMinimum
        /// </summary>
        [Test]
        public void Test1_2()
        {
            Random rand = new Random();

            RealFunction multiDim = delegate(Vector p)
            {
                return Math.Cos(p[0]) * Math.Sin(p[1]);
            };

            Vector point = Vector.Fill(2, 0), dir = Vector.Fill(2, 0);

            Function oneDim = delegate(double t)
            {
                return multiDim(point + t * dir);
            };

            for (int i = 0; i < 100; i++)
            {
                point = new double[] { 3 - 6 * rand.NextDouble(), 3 - 6 * rand.NextDouble() };
                dir = new double[] { 1-2*rand.NextDouble(), 1-2*rand.NextDouble() };

                Console.WriteLine(point);
                Console.WriteLine(dir);

                Linemethod sut = new Linemethod(oneDim, 2e-10);
                double before = oneDim(0);
                sut.LineMinimum(ref point, ref dir);
                double after = oneDim(0);

                Assert.IsTrue(before > after, "Couldn't find better minimum.");
            }
        }

        /// <summary>
        /// Test PowellBrent when the function gets
        /// maximum double values.
        /// </summary>
        [Test]
        public void Test2_1_Wallfunction()
        {
            RealFunction wallFunction = delegate(Vector p)
            {
                double x = p[0], y = p[1];
                return x * x + y * y * y * y;
            };

            Optimization.IterationEventHandler h = delegate(object sender, IterationEventArgs e)
            {
                Optimization o = sender as Optimization;
                Console.WriteLine("iteration " + o.Iterations + ": " + o.GetMinimum());;
            };

            // start iterate 
            Vector initial = new double[] { -1, 2 };

            Optimization sut = new PowellBrent(wallFunction, initial);
            sut.SetTolerance(1e-10);
            sut.IterationNotify += h;
            int Xindex = 0;
            double downLimit = 1.0;
            sut.SetFloor(Xindex, downLimit);
            sut.Iterate(2000);
            
            // the optimum is found
            Vector min = sut.GetMinimum();
            Assert.AreEqual(1.0, min[0], 0.001);
            Assert.AreEqual(0.0, min[1], 0.001);
        }

        /// <summary>
        /// Test PowellBrent when the function gets
        /// maximum double values.
        /// </summary>
        [Test]
        public void Test2_2_Wallfunction()
        {
            Vector initial = new double[] { -1, 2 };

            RealFunction wallFunction = delegate(Vector p)
            {
                double x = p[0], y = p[1];
                return (x > 1) ? x * x + y * y * y * y : Double.MaxValue;
            };
            
            Optimization.IterationEventHandler h = delegate(object sender, IterationEventArgs e)
            {
                Optimization o = sender as Optimization;
                Console.WriteLine("iteration " + o.Iterations + ": " + o.GetMinimum()); ;
            };

            Optimization sut = new PowellBrent(wallFunction, initial);
            sut.SetTolerance(1e-10);
            sut.IterationNotify += h;
            sut.Iterate(2000);
            Vector min = sut.GetMinimum();

            // the solution is not feasible.
            Assert.AreEqual(1.0, min[0], 0.001);
            Assert.AreEqual(0.0, min[1], 0.001);
        }
    }
}
