﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;
using TPClib;
using TPClib.Forms;
using System.Windows.Forms;

namespace TPClib.Model
{
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_WaveletTransform
    {
        //names of supported wavelet filters
        string[] names = new string[] {"E-spline","Haar","D1","D2","D3","D4","D5","D6", "Sym1","Sym2","Sym3","Sym4","Sym5","Sym6",
                                           "Coif1","Coif2","BCoif1","S+P (2,2)","S+P (4,2)","S+P (6,2)","S+P (4,4)","S+P (2+2,2)", 
                                           "TT" ,"LC 5/3","LC 2/6","LC 9/7-M","LC 2/10","LC 5/11-C","LC 5/11-A","LC 6/14",
                                           "LC 13/7-T","LC 13/7-C","Le Gall 5/3","CDF 9/7","V9/3","Lazy"};
        /// <summary>
        /// Test wavelet transform filters against original Matlab code.
        /// </summary>
        [Test]
        //[Ignore("Disabled because not required")]
        public void Test_Filters()
        {
            //error to Matlab filter value
            double error_tolerance = 0.00001;
            List<List<double[]>[]> ref_filters = new List<List<double[]>[]>();
            List<double[]> g1;
            List<double[]> h1;
            List<double[]> g2;
            List<double[]> h2;
            #region define_reference_Matlab_filters
            g1 = new List<double[]> { new double[] { -5.524272e-003, 0, 7.733980e-002, -8.838835e-002, -1.712524e-001, 4.419417e-001, 9.059806e-001, 4.419417e-001, -1.712524e-001, -8.838835e-002, 7.733980e-002, 0, -5.524272e-003 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, -7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { 5.524272e-003, 0, -7.733980e-002, -8.838835e-002, 1.712524e-001, 4.419417e-001, -9.059806e-001, 4.419417e-001, 1.712524e-001, -8.838835e-002, -7.733980e-002, 0, 5.524272e-003 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { -7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 7.071068e-001, -7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { -7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 7.071068e-001, -7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.294095e-001, 2.241439e-001, 8.365163e-001, 4.829629e-001 }, new double[] { 3 } };
            h1 = new List<double[]> { new double[] { 4.829629e-001, -8.365163e-001, 2.241439e-001, 1.294095e-001 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 4.829629e-001, 8.365163e-001, 2.241439e-001, -1.294095e-001 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 1.294095e-001, 2.241439e-001, -8.365163e-001, 4.829629e-001 }, new double[] { 2 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 3.522629e-002, -8.544127e-002, -1.350110e-001, 4.598775e-001, 8.068915e-001, 3.326706e-001 }, new double[] { 5 } };
            h1 = new List<double[]> { new double[] { 3.326706e-001, -8.068915e-001, 4.598775e-001, 1.350110e-001, -8.544127e-002, -3.522629e-002 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 3.326706e-001, 8.068915e-001, 4.598775e-001, -1.350110e-001, -8.544127e-002, 3.522629e-002 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { -3.522629e-002, -8.544127e-002, 1.350110e-001, 4.598775e-001, -8.068915e-001, 3.326706e-001 }, new double[] { 4 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.059740e-002, 3.288301e-002, 3.084138e-002, -1.870348e-001, -2.798377e-002, 6.308808e-001, 7.148466e-001, 2.303778e-001 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { -2.303778e-001, 7.148466e-001, -6.308808e-001, -2.798377e-002, 1.870348e-001, 3.084138e-002, -3.288301e-002, -1.059740e-002 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 2.303778e-001, 7.148466e-001, 6.308808e-001, -2.798377e-002, -1.870348e-001, 3.084138e-002, 3.288301e-002, -1.059740e-002 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { -1.059740e-002, -3.288301e-002, 3.084138e-002, 1.870348e-001, -2.798377e-002, -6.308808e-001, 7.148466e-001, -2.303778e-001 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 3.335725e-003, -1.258075e-002, -6.241490e-003, 7.757149e-002, -3.224487e-002, -2.422949e-001, 1.384281e-001, 7.243085e-001, 6.038293e-001, 1.601024e-001 }, new double[] { 8 } };
            h1 = new List<double[]> { new double[] { -1.601024e-001, 6.038293e-001, -7.243085e-001, 1.384281e-001, 2.422949e-001, -3.224487e-002, -7.757149e-002, -6.241490e-003, 1.258075e-002, 3.335725e-003 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 1.601024e-001, 6.038293e-001, 7.243085e-001, 1.384281e-001, -2.422949e-001, -3.224487e-002, 7.757149e-002, -6.241490e-003, -1.258075e-002, 3.335725e-003 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { 3.335725e-003, 1.258075e-002, -6.241490e-003, -7.757149e-002, -3.224487e-002, 2.422949e-001, 1.384281e-001, -7.243085e-001, 6.038293e-001, -1.601024e-001 }, new double[] { 7 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.077301e-003, 4.777258e-003, 5.538423e-004, -3.158204e-002, 2.752287e-002, 9.750160e-002, -1.297669e-001, -2.262647e-001, 3.152504e-001, 7.511339e-001, 4.946239e-001, 1.115407e-001 }, new double[] { 11 } };
            h1 = new List<double[]> { new double[] { 1.115407e-001, -4.946239e-001, 7.511339e-001, -3.152504e-001, -2.262647e-001, 1.297669e-001, 9.750161e-002, -2.752287e-002, -3.158204e-002, -5.538422e-004, 4.777258e-003, 1.077301e-003 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 1.115407e-001, 4.946239e-001, 7.511339e-001, 3.152504e-001, -2.262647e-001, -1.297669e-001, 9.750161e-002, 2.752287e-002, -3.158204e-002, 5.538422e-004, 4.777258e-003, -1.077301e-003 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 1.077301e-003, 4.777258e-003, -5.538423e-004, -3.158204e-002, -2.752287e-002, 9.750160e-002, 1.297669e-001, -2.262647e-001, -3.152504e-001, 7.511339e-001, -4.946239e-001, 1.115407e-001 }, new double[] { 10 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { -7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 7.071068e-001, -7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.294095e-001, 2.241439e-001, 8.365163e-001, 4.829629e-001 }, new double[] { 3 } };
            h1 = new List<double[]> { new double[] { 4.829629e-001, -8.365163e-001, 2.241439e-001, 1.294095e-001 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 4.829629e-001, 8.365163e-001, 2.241439e-001, -1.294095e-001 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 1.294095e-001, 2.241439e-001, -8.365163e-001, 4.829629e-001 }, new double[] { 2 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 3.522629e-002, -8.544127e-002, -1.350110e-001, 4.598775e-001, 8.068915e-001, 3.326706e-001 }, new double[] { 5 } };
            h1 = new List<double[]> { new double[] { 3.326706e-001, -8.068915e-001, 4.598775e-001, 1.350110e-001, -8.544127e-002, -3.522629e-002 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 3.326706e-001, 8.068915e-001, 4.598775e-001, -1.350110e-001, -8.544127e-002, 3.522629e-002 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { -3.522629e-002, -8.544127e-002, 1.350110e-001, 4.598775e-001, -8.068915e-001, 3.326706e-001 }, new double[] { 4 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -7.576571e-002, -2.963553e-002, 4.976187e-001, 8.037388e-001, 2.978578e-001, -9.921954e-002, -1.260397e-002, 3.222310e-002 }, new double[] { 3 } };
            h1 = new List<double[]> { new double[] { 3.222310e-002, 1.260397e-002, -9.921954e-002, -2.978578e-001, 8.037388e-001, -4.976187e-001, -2.963553e-002, 7.576571e-002 }, new double[] { 5 } };
            g2 = new List<double[]> { new double[] { 3.222310e-002, -1.260397e-002, -9.921954e-002, 2.978578e-001, 8.037388e-001, 4.976187e-001, -2.963553e-002, -7.576571e-002 }, new double[] { 4 } };
            h2 = new List<double[]> { new double[] { 7.576571e-002, -2.963553e-002, -4.976187e-001, 8.037388e-001, -2.978578e-001, -9.921954e-002, 1.260397e-002, 3.222310e-002 }, new double[] { 2 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 2.733307e-002, 2.951949e-002, -3.913425e-002, 1.993975e-001, 7.234077e-001, 6.339790e-001, 1.660211e-002, -1.753281e-001, -2.110183e-002, 1.953888e-002 }, new double[] { 5 } };
            h1 = new List<double[]> { new double[] { 1.953888e-002, 2.110183e-002, -1.753281e-001, -1.660211e-002, 6.339790e-001, -7.234077e-001, 1.993975e-001, 3.913425e-002, 2.951949e-002, -2.733307e-002 }, new double[] { 5 } };
            g2 = new List<double[]> { new double[] { 1.953888e-002, -2.110183e-002, -1.753281e-001, 1.660211e-002, 6.339790e-001, 7.234077e-001, 1.993975e-001, -3.913425e-002, 2.951949e-002, 2.733307e-002 }, new double[] { 4 } };
            h2 = new List<double[]> { new double[] { -2.733307e-002, 2.951949e-002, 3.913425e-002, 1.993975e-001, -7.234077e-001, 6.339790e-001, -1.660211e-002, -1.753281e-001, 2.110183e-002, 1.953888e-002 }, new double[] { 4 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 1.540411e-002, 3.490712e-003, -1.179901e-001, -4.831174e-002, 4.910559e-001, 7.876411e-001, 3.379294e-001, -7.263752e-002, -2.106029e-002, 4.472490e-002, 1.767712e-003, -7.800708e-003 }, new double[] { 11 } };
            h1 = new List<double[]> { new double[] { -7.800708e-003, -1.767712e-003, 4.472490e-002, 2.106029e-002, -7.263752e-002, -3.379294e-001, 7.876411e-001, -4.910559e-001, -4.831174e-002, 1.179901e-001, 3.490712e-003, -1.540411e-002 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { -7.800708e-003, 1.767712e-003, 4.472490e-002, -2.106029e-002, -7.263752e-002, 3.379294e-001, 7.876411e-001, 4.910559e-001, -4.831174e-002, -1.179901e-001, 3.490712e-003, 1.540411e-002 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { -1.540411e-002, 3.490712e-003, 1.179901e-001, -4.831174e-002, -4.910559e-001, 7.876411e-001, -3.379294e-001, -7.263752e-002, 2.106029e-002, 4.472490e-002, -1.767712e-003, -7.800708e-003 }, new double[] { 10 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.565573e-002, -7.273262e-002, 3.848648e-001, 8.525720e-001, 3.378977e-001, -7.273262e-002 }, new double[] { 5 } };
            h1 = new List<double[]> { new double[] { -7.273262e-002, -3.378977e-001, 8.525720e-001, -3.848648e-001, -7.273262e-002, 1.565573e-002 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { -7.273262e-002, 3.378977e-001, 8.525720e-001, 3.848648e-001, -7.273262e-002, -1.565573e-002 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 1.565573e-002, -7.273262e-002, -3.848648e-001, 8.525720e-001, -3.378977e-001, -7.273262e-002 }, new double[] { 4 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -7.205494e-004, -1.823209e-003, 5.611435e-003, 2.368017e-002, -5.943442e-002, -7.648860e-002, 4.170052e-001, 8.127236e-001, 3.861101e-001, -6.737255e-002, -4.146494e-002, 1.638734e-002 }, new double[] { 11 } };
            h1 = new List<double[]> { new double[] { 1.638734e-002, 4.146494e-002, -6.737255e-002, -3.861101e-001, 8.127236e-001, -4.170052e-001, -7.648860e-002, 5.943442e-002, 2.368017e-002, -5.611435e-003, -1.823209e-003, 7.205494e-004 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 1.638734e-002, -4.146494e-002, -6.737255e-002, 3.861101e-001, 8.127236e-001, 4.170052e-001, -7.648860e-002, -5.943442e-002, 2.368017e-002, 5.611435e-003, -1.823209e-003, -7.205494e-004 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 7.205494e-004, -1.823209e-003, -5.611435e-003, 2.368017e-002, 5.943442e-002, -7.648860e-002, -4.170052e-001, 8.127236e-001, -3.861101e-001, -6.737255e-002, 4.146494e-002, 1.638734e-002 }, new double[] { 10 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -7.071068e-002, 3.535534e-001, 8.485281e-001, 3.535534e-001, -7.071068e-002 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { 1.515229e-002, -7.576144e-002, -3.687057e-001, 8.586297e-001, -3.687057e-001, -7.576144e-002, 1.515229e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -1.515229e-002, -7.576144e-002, 3.687057e-001, 8.586297e-001, 3.687057e-001, -7.576144e-002, -1.515229e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { -7.071068e-002, -3.535534e-001, 8.485281e-001, -3.535534e-001, -7.071068e-002 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { -3.535534e-001, 7.071068e-001, -3.535534e-001 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 3.535534e-001, 7.071068e-001, 3.535534e-001 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { -1.767767e-001, -3.535534e-001, 1.060660e+000, -3.535534e-001, -1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 2.209709e-002, 0, -1.767767e-001, 3.535534e-001, 1.016466e+000, 3.535534e-001, -1.767767e-001, 0, 2.209709e-002 }, new double[] { 4 } };
            h1 = new List<double[]> { new double[] { 4.419417e-002, 0, -3.977476e-001, 7.071068e-001, -3.977476e-001, 0, 4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { 2.209709e-002, 0, -1.767767e-001, -3.535534e-001, 1.016466e+000, -3.535534e-001, -1.767767e-001, 0, 2.209709e-002 }, new double[] { 3 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -4.143204e-003, 0, 3.038349e-002, 0, -1.726335e-001, 3.535534e-001, 9.998932e-001, 3.535534e-001, -1.726335e-001, 0, 3.038349e-002, 0, -4.143204e-003 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { -8.286408e-003, 0, 6.905340e-002, 0, -4.143204e-001, 7.071068e-001, -4.143204e-001, 0, 6.905340e-002, 0, -8.286408e-003 }, new double[] { 6 } };
            g2 = new List<double[]> { new double[] { 8.286408e-003, 0, -6.905340e-002, 0, 4.143204e-001, 7.071068e-001, 4.143204e-001, 0, -6.905340e-002, 0, 8.286408e-003 }, new double[] { 5 } };
            h2 = new List<double[]> { new double[] { -4.143204e-003, 0, 3.038349e-002, 0, -1.726335e-001, -3.535534e-001, 9.998932e-001, -3.535534e-001, -1.726335e-001, 0, 3.038349e-002, 0, -4.143204e-003 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -2.762136e-003, 0, 4.971845e-002, -4.419417e-002, -1.740146e-001, 3.977476e-001, 9.612233e-001, 3.977476e-001, -1.740146e-001, -4.419417e-002, 4.971845e-002, 0, -2.762136e-003 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { 4.419417e-002, 0, -3.977476e-001, 7.071068e-001, -3.977476e-001, 0, 4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { -2.762136e-003, 0, 4.971845e-002, 4.419417e-002, -1.740146e-001, -3.977476e-001, 9.612233e-001, -3.977476e-001, -1.740146e-001, 4.419417e-002, 4.971845e-002, 0, -2.762136e-003 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { -5.524272e-003, 1.104854e-002, 3.866990e-002, 0, -3.866990e-001, 6.850097e-001, -3.866990e-001, 0, 3.866990e-002, 1.104854e-002, -5.524272e-003 }, new double[] { 6 } };
            g2 = new List<double[]> { new double[] { 5.524272e-003, 1.104854e-002, -3.866990e-002, 0, 3.866990e-001, 6.850097e-001, 3.866990e-001, 0, -3.866990e-002, 1.104854e-002, 5.524272e-003 }, new double[] { 5 } };
            h2 = new List<double[]> { new double[] { -1.767767e-001, -3.535534e-001, 1.060660e+000, -3.535534e-001, -1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { 1.657282e-002, 1.657282e-002, -1.215340e-001, -1.215340e-001, 7.071068e-001, -7.071068e-001, 1.215340e-001, 1.215340e-001, -1.657282e-002, -1.657282e-002 }, new double[] { 5 } };
            g2 = new List<double[]> { new double[] { 1.657282e-002, -1.657282e-002, -1.215340e-001, 1.215340e-001, 7.071068e-001, 7.071068e-001, 1.215340e-001, -1.215340e-001, -1.657282e-002, 1.657282e-002 }, new double[] { 4 } };
            h2 = new List<double[]> { new double[] { -7.071068e-001, 7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { -3.535534e-001, 7.071068e-001, -3.535534e-001 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 3.535534e-001, 7.071068e-001, 3.535534e-001 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { -1.767767e-001, -3.535534e-001, 1.060660e+000, -3.535534e-001, -1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { 8.838835e-002, 8.838835e-002, -7.071068e-001, 7.071068e-001, -8.838835e-002, -8.838835e-002 }, new double[] { 3 } };
            g2 = new List<double[]> { new double[] { -8.838835e-002, 8.838835e-002, 7.071068e-001, 7.071068e-001, 8.838835e-002, -8.838835e-002 }, new double[] { 2 } };
            h2 = new List<double[]> { new double[] { 7.071068e-001, -7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 2.209709e-002, 0, -1.767767e-001, 3.535534e-001, 1.016466e+000, 3.535534e-001, -1.767767e-001, 0, 2.209709e-002 }, new double[] { 4 } };
            h1 = new List<double[]> { new double[] { 4.419417e-002, 0, -3.977476e-001, 7.071068e-001, -3.977476e-001, 0, 4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { 2.209709e-002, 0, -1.767767e-001, -3.535534e-001, 1.016466e+000, -3.535534e-001, -1.767767e-001, 0, 2.209709e-002 }, new double[] { 3 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 7.071068e-001, 7.071068e-001 }, new double[] { 1 } };
            h1 = new List<double[]> { new double[] { -1.657282e-002, -1.657282e-002, 1.215340e-001, 1.215340e-001, -7.071068e-001, 7.071068e-001, -1.215340e-001, -1.215340e-001, 1.657282e-002, 1.657282e-002 }, new double[] { 5 } };
            g2 = new List<double[]> { new double[] { 1.657282e-002, -1.657282e-002, -1.215340e-001, 1.215340e-001, 7.071068e-001, 7.071068e-001, 1.215340e-001, -1.215340e-001, -1.657282e-002, 1.657282e-002 }, new double[] { 4 } };
            h2 = new List<double[]> { new double[] { 7.071068e-001, -7.071068e-001 }, new double[] { 0 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { -5.524272e-003, 1.104854e-002, 3.866990e-002, 0, -3.866990e-001, 6.850097e-001, -3.866990e-001, 0, 3.866990e-002, 1.104854e-002, -5.524272e-003 }, new double[] { 6 } };
            g2 = new List<double[]> { new double[] { 5.524272e-003, 1.104854e-002, -3.866990e-002, 0, 3.866990e-001, 6.850097e-001, 3.866990e-001, 0, -3.866990e-002, 1.104854e-002, 5.524272e-003 }, new double[] { 5 } };
            h2 = new List<double[]> { new double[] { -1.767767e-001, -3.535534e-001, 1.060660e+000, -3.535534e-001, -1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { 2.762136e-003, -5.524272e-003, -1.933495e-002, 0, 3.701262e-001, -6.960582e-001, 3.701262e-001, 0, -1.933495e-002, -5.524272e-003, 2.762136e-003 }, new double[] { 6 } };
            g2 = new List<double[]> { new double[] { 2.762136e-003, 5.524272e-003, -1.933495e-002, 0, 3.701262e-001, 6.960582e-001, 3.701262e-001, 0, -1.933495e-002, 5.524272e-003, 2.762136e-003 }, new double[] { 5 } };
            h2 = new List<double[]> { new double[] { 1.767767e-001, 3.535534e-001, -1.060660e+000, 3.535534e-001, 1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -8.838835e-002, 8.838835e-002, 7.071068e-001, 7.071068e-001, 8.838835e-002, -8.838835e-002 }, new double[] { 3 } };
            h1 = new List<double[]> { new double[] { 2.762136e-003, -2.762136e-003, -3.866990e-002, -5.524272e-003, 1.298204e-001, 1.353447e-001, -6.739612e-001, 6.739612e-001, -1.353447e-001, -1.298204e-001, 5.524272e-003, 3.866990e-002, 2.762136e-003, -2.762136e-003 }, new double[] { 7 } };
            g2 = new List<double[]> { new double[] { -2.762136e-003, -2.762136e-003, 3.866990e-002, -5.524272e-003, -1.298204e-001, 1.353447e-001, 6.739612e-001, 6.739612e-001, 1.353447e-001, -1.298204e-001, -5.524272e-003, 3.866990e-002, -2.762136e-003, -2.762136e-003 }, new double[] { 6 } };
            h2 = new List<double[]> { new double[] { -8.838835e-002, -8.838835e-002, 7.071068e-001, -7.071068e-001, 8.838835e-002, 8.838835e-002 }, new double[] { 2 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -2.762136e-003, 0, 4.971845e-002, -4.419417e-002, -1.740146e-001, 3.977476e-001, 9.612233e-001, 3.977476e-001, -1.740146e-001, -4.419417e-002, 4.971845e-002, 0, -2.762136e-003 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { 4.419417e-002, 0, -3.977476e-001, 7.071068e-001, -3.977476e-001, 0, 4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { -2.762136e-003, 0, 4.971845e-002, 4.419417e-002, -1.740146e-001, -3.977476e-001, 9.612233e-001, -3.977476e-001, -1.740146e-001, 4.419417e-002, 4.971845e-002, 0, -2.762136e-003 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -5.524272e-003, 0, 7.733980e-002, -8.838835e-002, -1.712524e-001, 4.419417e-001, 9.059806e-001, 4.419417e-001, -1.712524e-001, -8.838835e-002, 7.733980e-002, 0, -5.524272e-003 }, new double[] { 6 } };
            h1 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, -7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -4.419417e-002, 0, 3.977476e-001, 7.071068e-001, 3.977476e-001, 0, -4.419417e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { 5.524272e-003, 0, -7.733980e-002, -8.838835e-002, 1.712524e-001, 4.419417e-001, -9.059806e-001, 4.419417e-001, 1.712524e-001, -8.838835e-002, -7.733980e-002, 0, 5.524272e-003 }, new double[] { 5 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -1.767767e-001, 3.535534e-001, 1.060660e+000, 3.535534e-001, -1.767767e-001 }, new double[] { 2 } };
            h1 = new List<double[]> { new double[] { -3.535534e-001, 7.071068e-001, -3.535534e-001 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 3.535534e-001, 7.071068e-001, 3.535534e-001 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { -1.767767e-001, -3.535534e-001, 1.060660e+000, -3.535534e-001, -1.767767e-001 }, new double[] { 1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 3.782846e-002, -2.384947e-002, -1.106244e-001, 3.774029e-001, 8.526987e-001, 3.774029e-001, -1.106244e-001, -2.384947e-002, 3.782846e-002 }, new double[] { 4 } };
            h1 = new List<double[]> { new double[] { 6.453888e-002, -4.068942e-002, -4.180923e-001, 7.884856e-001, -4.180923e-001, -4.068942e-002, 6.453888e-002 }, new double[] { 4 } };
            g2 = new List<double[]> { new double[] { -6.453888e-002, -4.068942e-002, 4.180923e-001, 7.884856e-001, 4.180923e-001, -4.068942e-002, -6.453888e-002 }, new double[] { 3 } };
            h2 = new List<double[]> { new double[] { 3.782846e-002, 2.384947e-002, -1.106244e-001, -3.774029e-001, 8.526987e-001, -3.774029e-001, -1.106244e-001, 2.384947e-002, 3.782846e-002 }, new double[] { 3 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { -8.838835e-003, 1.767767e-002, -1.767767e-001, 3.358757e-001, 1.078338e+000, 3.358757e-001, -1.767767e-001, 1.767767e-002, -8.838835e-003 }, new double[] { 4 } };
            h1 = new List<double[]> { new double[] { -3.535534e-001, 7.071068e-001, -3.535534e-001 }, new double[] { 2 } };
            g2 = new List<double[]> { new double[] { 3.535534e-001, 7.071068e-001, 3.535534e-001 }, new double[] { 1 } };
            h2 = new List<double[]> { new double[] { -8.838835e-003, -1.767767e-002, -1.767767e-001, -3.358757e-001, 1.078338e+000, -3.358757e-001, -1.767767e-001, -1.767767e-002, -8.838835e-003 }, new double[] { 3 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            g1 = new List<double[]> { new double[] { 1 }, new double[] { 0 } };
            h1 = new List<double[]> { new double[] { 1 }, new double[] { 1 } };
            g2 = new List<double[]> { new double[] { 1 }, new double[] { 0 } };
            h2 = new List<double[]> { new double[] { 1 }, new double[] { -1 } };
            ref_filters.Add(new List<double[]>[] { g1, h1, g2, h2 });
            #endregion


            List<double[]>[] filter;
            for (int f = 0; f < names.Length; f++)
            {
                Console.WriteLine("Wavelet[" + names[f] + "]");
            
                filter = WaveletTransform.GetWaveletFilters(names[f]);

                for (int j = 0; j < filter[0].Count; j++)
                {
                    Console.Write("g1[" + j + "] ");
                    for (int k = 0; k < filter[0][j].Length; k++)
                    {
                        Console.Write(filter[0][j][k] + " ");
                        Assert.AreEqual(ref_filters[f][0][j][k], filter[0][j][k], error_tolerance, "[" + names[f] + "]");
                    }
                }
                Console.WriteLine();
                for (int j = 0; j < filter[1].Count; j++)
                {
                    Console.Write("h1[" + j + "] ");
                    for (int k = 0; k < filter[1][j].Length; k++)
                    {
                        Console.Write(filter[1][j][k] + " ");
                        Assert.AreEqual(ref_filters[f][1][j][k], filter[1][j][k], error_tolerance, "[" + names[f] + "]");
                    }
                }
                Console.WriteLine();
                for (int j = 0; j < filter[2].Count; j++)
                {
                    Console.Write("g2[" + j + "] ");
                    for (int k = 0; k < filter[2][j].Length; k++)
                    {
                        Console.Write(filter[2][j][k] + " ");
                        Assert.AreEqual(ref_filters[f][2][j][k], filter[2][j][k], error_tolerance, "[" + names[f] + "]");
                    }
                }
                Console.WriteLine();
                for (int j = 0; j < filter[3].Count; j++)
                {
                    Console.Write("h2[" + j + "] ");
                    for (int k = 0; k < filter[3][j].Length; k++)
                    {
                        Console.Write(filter[3][j][k] + " ");
                        Assert.AreEqual(ref_filters[f][3][j][k], filter[3][j][k], error_tolerance, "[" + names[f] + "]");
                    }
                }
                Console.WriteLine();
                //Do not show filters in window, filters are tested numerically
                //new System.Threading.Thread(new System.Threading.ThreadStart(delegate()
                //{
                //    PaintCurve(names[f], filter);
                //}));
            }
        }
        /// <summary>
        /// Test wavelet 1D transform to wavelet space and back
        /// </summary>
        [Test]
        public void Test_1DTransform() {
            //error to original data
            double error_tolerance = 0.00001;
            //length of test data
            int test_data_length = 32;
            //create allways same random data
            Random rand = new Random(0);
            List<double> randvalues = new List<double>();
            for (int i = 0; i < test_data_length; i++)
            {
                randvalues.Add(rand.NextDouble());
            }
            double[] X = randvalues.ToArray();
            double[] Xwavelet;
            double[] Xwavelet_back = X;
            for (int f = 7; f < names.Length; f++)
            {
                //test stage 3
                for (int l = 3; l <= 3; l++)
                {
                    Console.WriteLine("---FORW TRANSFORM---[" + names[f] + "] stage=" + l);
                    Xwavelet = WaveletTransform.Wavelet1D(names[f], l, X, WaveletTransform.BoundaryHandling.asym, 1);
                    Assert.AreEqual(X.Length, Xwavelet.Length, "Transformed length is not equal to original length");
                    Console.WriteLine("---BACK TRANSFORM---[" + names[f] + "] stage=" + l);
                    Xwavelet_back = WaveletTransform.Wavelet1D(names[f], -l, Xwavelet, WaveletTransform.BoundaryHandling.asym, 1);
                    Assert.AreEqual(Xwavelet_back.Length, Xwavelet.Length, "Transformed length is not equal to original length");
                    for (int i = 0; i < X.Length; i++)
                        Console.WriteLine("[" + X[i] + "]\t[" + Xwavelet[i] + "]\t[" + Xwavelet_back[i] + "]");
                    for (int i = 0; i < X.Length; i++)
                    {
                        Assert.AreEqual(X[i], Xwavelet_back[i], error_tolerance, "Forward-backward transformed data was not the same as original");
                    }
                }
            }
        }
        /// <summary>
        /// Test wavelet 2D transform to wavelet space and back
        /// </summary>
        [Test]
        public void Test_2DTransform()
        {
            //error to original data
            double error_tolerance = 0.00001;
            //length of test data
            int test_data_length = 8;
            //create allways same random data
            Random rand = new Random(0);
            double[,] X = new double[test_data_length,test_data_length];
            double[,] Xwavelet = new double[test_data_length, test_data_length];
            double[,] Xwavelet_back = new double[test_data_length, test_data_length];
            for (int i = 0; i < test_data_length; i++)
            {
                for (int j = 0; j < test_data_length; j++)
                {
                    X[j,i] = rand.NextDouble();
                }
            }

            for (int f = 0; f < names.Length; f++)
            {
                //test stage 3
                for (int l = 3; l <= 3; l++)
                {
                    Console.WriteLine("ORIGINAL");
                    ((Matrix)X).PrintAsString(Console.Out);
                    Console.WriteLine("---FORW TRANSFORM---[" + names[f] + "] stage=" + l);
                    Xwavelet = WaveletTransform.Wavelet2D(names[f], l, (Matrix)X, WaveletTransform.BoundaryHandling.asym, 1);
                    Assert.AreEqual(X.Length, Xwavelet.Length, "Transformed length is not equal to original length");
                    Console.WriteLine("TRANSFORMED");
                    ((Matrix)Xwavelet).PrintAsString(Console.Out);
                    Console.WriteLine("---BACK TRANSFORM---[" + names[f] + "] stage=" + l);
                    Xwavelet_back = WaveletTransform.Wavelet2D(names[f], -l, (Matrix)Xwavelet, WaveletTransform.BoundaryHandling.asym, 1);
                    Assert.AreEqual(Xwavelet_back.Length, Xwavelet.Length, "Transformed length is not equal to original length");
                    Console.WriteLine("BACK TRANSFORMED");
                    ((Matrix)Xwavelet_back).PrintAsString(Console.Out);
                    
                    for (int i = 0; i < test_data_length; i++)
                    {
                        for (int j = 0; j < test_data_length; j++)
                        {
                            Assert.AreEqual(X[i,j], Xwavelet_back[i,j], error_tolerance, "Forward-backward transformed data was not the same as original at ["+i+","+j+"]");
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Paints wavelet filter curves on screen
        /// </summary>
        /// <param name="name">wavelet name</param>
        /// <param name="filters">filters</param>
        private void PaintCurve(string name, List<double[]>[] filters)
        {
            int plotsize = 400;
            Form form = new Form();
            form.Width = plotsize*2;
            form.Height = plotsize * (int)Math.Ceiling(filters.Length / 2.0);
            form.Dock = System.Windows.Forms.DockStyle.Fill;
            form.Controls.Clear();
            form.Text = name;
            int x = 0;
            int y = -plotsize;
            string title = "";
            string[] names = { "g1", "h1", "g2", "h2" };
            for (int i = 0; i < filters.Length; i++)
            {
                if (i % 2 == 0)
                {
                    x = 0;
                    y += plotsize;
                }
                else
                {
                    x = plotsize;
                }
                if (i < 4) title = names[i];
                else title = "";
                form.Controls.Add(CreateCurvePanel(title, filters[i][0], x, y, plotsize, plotsize));
            }
            form.StartPosition = FormStartPosition.Manual;
            form.Location = new System.Drawing.Point();
            form.FormBorderStyle = FormBorderStyle.FixedSingle;
            form.ShowDialog();
        }
        /// <summary>
        /// Paints wavelet filter curves on screen
        /// </summary>
        /// <param name="name">wavelet name</param>
        /// <param name="curve">curve</param>
        private void PaintCurve(string name, double[] curve)
        {
            int plotsize = 400;
            Form form = new Form();
            form.Width = plotsize * 2;
            form.Height = plotsize;
            form.Dock = System.Windows.Forms.DockStyle.Fill;
            form.Controls.Clear();
            form.Text = name;
            int x = 0;
            int y = -plotsize;
            form.Controls.Add(CreateCurvePanel(name, curve, x, y, plotsize, plotsize));
            form.StartPosition = FormStartPosition.Manual;
            form.Location = new System.Drawing.Point();
            form.FormBorderStyle = FormBorderStyle.FixedSingle;
            form.ShowDialog();
        }
        /// <summary>
        /// Creates panel which has one curve of filter plotted
        /// </summary>
        /// <param name="title">title of curve</param>
        /// <param name="y">y-values of curve</param>
        /// <param name="x">x-coordinate</param>
        /// <param name="y">y-coordinate</param>
        /// <param name="width">width of panel</param>
        /// <param name="height">height if panel</param>
        /// <returns></returns>
        private Plot2D CreateCurvePanel(string title, double[] ys, int x, int y, int width, int height)
        {
            FloatAxis xaxis = new FloatAxis();
            FloatAxis yaxis = new FloatAxis();
            xaxis.min = 0.0;
            xaxis.max = 10.0;
            yaxis.min = 0.0;
            yaxis.max = 10.0;
            xaxis.style.Color = Color.Black;
            yaxis.style.Color = Color.Black;
            yaxis.Scale = ScaleType.Fit_To_Screen;
            Plot2D viewer = new Plot2D(xaxis, yaxis);
            double[] times = new double[ys.Length];
            for (int i = 0; i < times.Length; i++)
                times[i] = i;
            TPClib.Forms.Curve curve = new TPClib.Forms.Curve(title, times, ys, new LineStyle(false, true, false, 1.0f, Color.Blue));
            viewer.AddCurve(curve);
            viewer.Size = new Size(width, height);
            viewer.MaximumSize = viewer.Size;
            viewer.SetBounds(x, y, width, height);
            viewer.UpdateScales();
            viewer.BackColor = Color.White;
            return viewer;
        }
    }
}
