﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Implementation of Nelder-Mead optimization algorithm
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class NelderMead : Optimization
    {
        /// <summary>
        /// Reflection scale
        /// </summary>
        private double alpha = 1.0;

        /// <summary>
        /// Expansion scale
        /// </summary>
        private double gamma = 2.0;

        /// <summary>
        /// Contraction scale
        /// </summary>
        private double rho = 0.5;

        /// <summary>
        /// Shrink scale
        /// </summary>
        private double theta = 0.5;

        /// <summary>
        /// Polytope
        /// </summary>
        private Polytope ptope;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="target_func">Function to optimize</param>
        /// <param name="p">Initial parameters</param>
        public NelderMead(RealFunction target_func, Vector p)
        {
            InitialParams = p;
            SetTargetFunction(target_func);
            MakePolytope(p, alpha);
        }

        /// <summary>
        /// Constructor. InitMethod must be called if this is used
        /// </summary>
        public NelderMead() { }

        /// <summary>
        /// Gets info of Method and its parameters
        /// </summary>
        /// <returns>Info of method</returns>
        public static OptimizationInfo GetInfo()
        {
            OptimizationInfo info = new OptimizationInfo();
            info.MethodName = "NelderMead";
            info.ParameterName = new System.Collections.Generic.List<string>();
            info.ParameterValue = new System.Collections.Generic.List<double>();
            return info;
        }


        /// <summary>
        /// Inits NelderMead. This method is called from Parent class (optimization)
        /// Create() -method. 
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="info">Info of method (Not needed with NelderMean)</param>
        /// <param name="c">parameter ceiling values</param>
        /// <param name="f">parameter floor values</param>
        /// <param name="initial">initial parameter values</param>
        protected override void InitMethod(
            RealFunction function,
            OptimizationInfo info,
            Vector c,
            Vector f,
            Vector initial)
        {
            InitialParams = initial;
            SetTargetFunction(function);
            MakePolytope(initial, alpha);            
        }

        /// <summary>
        /// Single minimization step; called from Optimization.Iterate()
        /// </summary>
        /// <returns>
        /// </returns>
        protected override Vector Step()
        {
            // Calculate the reflection of the worst point
            Vector worst = ptope.GetMaximum();
            Vector candidate = ptope.GetReflection(worst, alpha);
            double fval = TargetFunction(candidate);


            // Is the reflection better than worst point?
            if (ptope.IsBetter(fval))
            {
                // If the reflection is best point so far...
                if (ptope.IsBest(fval))
                {
                    // ... try to find even better point further along the same line
                    Vector newcandidate = worst + gamma * (candidate - worst);
                    double fval_new = TargetFunction(newcandidate);

                    // Keep looking (exponentially further) as long as better points are found
                    while (fval_new < fval)
                    {
                        candidate = newcandidate;
                        fval = fval_new;

                        newcandidate = worst + gamma * (candidate - worst);
                        fval_new = TargetFunction(newcandidate);
                    }
                }
                // Replace the worst point
                ptope.ReplaceWorst(candidate, fval);
            }

            // If the reflection is worse than the worst point so far,
            // try to find a better one along the same line
            else
            {
                Vector newcandidate = ptope.GetReflection(worst, rho);
                double fval_new = TargetFunction(newcandidate);

                // If it's indeed better, use it
                if (fval_new < fval)
                {
                    // Replace the worst point
                    ptope.ReplaceWorst(newcandidate, fval_new);
                }

                // If it's even worse, try to continue with a new polytope
                else
                {
                    MakePolytope(GetMinimum(), theta);
                }
            }

            return ptope.GetMinimum();
        }

        /// <summary>
        /// Constructs a polytope with 'initial' point as one of the vertices,
        /// other vertices are along the unit vectors, 'scale' units away
        /// </summary>
        /// <param name="initial">Initial point</param>
        /// <param name="scale">
        /// Polytope size; polytope with 'scale==1' has
        /// vertices separated from initial point by unit vectors
        /// </param>
        private void MakePolytope(Vector initial, double scale)
        {
            ptope = new Polytope();
            double funcval = TargetFunction(initial);
            ptope.Add(initial, funcval);

            for (int i = 0; i < initial.Dim; i++)
            {
                Vector vertex = initial + scale * Vector.Unit(initial.Dim, i);
                funcval = TargetFunction(vertex);
                ptope.Add(vertex, funcval);
            }
        }

        /// <summary>
        /// Get the vertices of the current polytope
        /// </summary>
        /// <returns>Array of vectors</returns>
        public Vector[] GetSimplex()
        {
            return ptope.GetVertices();
        }
    }
}
